#!/bin/sh
# Setup syslog.conf for logging from vasm. Analyzes the 
# /etc/syslog.conf file to determine which local logging facility 
# can be used by vasm, and configures syslog to log for
# /var/log/vasm.log.
#
# This script by Jason Pierce <echo -e "moc\x2eliamg\x40nosaj.ecreip" | rev>.
# Licensed as MIT.
logFile="/var/log/vasm.log";

# See if a specific property (aka keyword) is in use in syslog.conf
# This can be used to check for presence in the syslog.conf for logfile
# target vasm.log `in_use vasm.log` or for the logging facility local0
# `in_use local0`. 
in_use() {
	local property;
	property=$1;
	usingProperty=$( grep -I $property /etc/syslog.conf | sed "s/#.*$//" | sed '/^$/d' | wc -l );
	if [ "$usingProperty" -gt "0" ]; then
		# This property is in use.
		echo true;
	fi;
}

# Output the configuration lines to be inserted into syslog.conf.
syslog_payload() {
	local facility;
	facility=$1;
	cat <<EOF

# Vasm centralized logging to facility $facility.
$facility.*					-$logFile
EOF
}
	
# See if syslog.conf is already setup to be using vasm.log
usingVasm=$( in_use "$logFile" );
if [ $usingVasm ]; then
	# Vasm logging already setup
	echo "Logging already configured for $logFile."
	echo "Doing nothing.";
	exit 0;
else
	echo "Logging not configured for $logFile.";
	echo "Configuring in /etc/syslog.conf.";
	# See if we can use logging facility "local0" or if it is already in use.
	usingLocal0=$( in_use "local0" );
	if [ $usingLocal0 ]; then
		# Something else is using 'local0'.
		echo "Logging facility 'local0' already in use."
		echo "Checking for 'local1' now.";
		# See if we can use logging facility "local1" or if it is already in use.
		usingLocal1=$( in_use "local1" );
		if [ $usingLocal1 ]; then
			# Something else is using 'local1'.
			echo "Logging facility 'local1' already in use.";
			echo "Aborting configuration of /etc/syslog.conf.";
			echo "Please configure manually for vasm logging.";
		else
			# We can use 'local1'.
			facility="local1";
		fi;
	else
		# We can use 'local0'.
		facility="local0";
	fi;
	
	# Write our conf lines to syslog config file.
	syslog_payload $facility >> /etc/syslog.conf;
	echo "Configured with facility=$facility logfile=$logFile.";
fi;
