/**************************************************************************
   mashup - arrange images and text in a layout and print

   Copyright 2008 2009 2010 2011  Michael Cornelison
   source URL:  kornelix.squarespace.com
   contact: kornelix2@googlemail.com
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see <http://www.gnu.org/licenses/>.

***************************************************************************/

#include "zfuncs.h"

#define fversion "mashup v.2.9"                                            //  version
#define flicense "Free software - GNU General Public License v.3"
#define fsource "http://kornelix.squarespace.com/mashup"

#define bilinear GDK_INTERP_BILINEAR                                       //  GDK shortcuts
#define nodither GDK_RGB_DITHER_NONE,0,0
#define colorspace GDK_COLORSPACE_RGB
#define pixbuf_scale gdk_pixbuf_scale_simple

#define maxtext 9999                                                       //  max. text cc
#define layoutsize 1600                                                    //  max. layout dimension
#define pack_tolerance 0.02                                                //  alignment tolerance for auto-packing

#define     Npformats 6
const char  *pformats[Npformats] = {                                       //  printer paper formats  v.2.4  
               "letter  21.6 x 27.9 cm" ,
               "A3  29.7 x 42.0 cm" ,                                      //  v.2.5
               "A4  21.0 x 29.7 cm" ,
               "A5  14.8 x 21.0 cm" ,
               "A6  10.5 x 14.8 cm" ,
               "custom  N.N x N.N cm"   };                                 //  custom must be last
int         deformat = 2;                                                  //  default format = [2] = A4
char        format[20];                                                    //  last used paper format
char        orent[20];                                                     //    and orientation
double      paper_wwcm, paper_hhcm;                                        //  paper dimensions in cm

GtkWidget   *mainWin, *drawWin, *mVbox;                                    //  main and drawing window
GdkPixbuf   *pxb_layout = 0;                                               //  print layout pixbuf
GdkPixbuf   *pxb_drawwin = 0;                                              //  drawing window pixbuf
GdkGC       *gdkgc = 0;                                                    //  graphics context
int         wwD = 1000, hhD = 700;                                         //  main window initial size
typedef     uchar *pixel;                                                  //  3 RGB values, 0-255 each

zdialog     *zdtext = 0;                                                   //  active text dialog
zdialog     *zdrotate = 0;                                                 //  active rotate dialog
zdialog     *zdframe = 0;                                                  //  active frame dialog

char        imagefile[maxfcc] = "";                                        //  current image file
char        imagedirk[maxfcc] = "";                                        //  current image file directory

//  page layout data and default settings
int         layout_ww, layout_hh;                                          //  layout pixel dimensions
int         layout_pix = 400;                                              //  default image size in layout
int         layout_select = -1;                                            //  last selected image/text index
double      Rscale = 0.5;                                                  //  layout to drawing window
char        pendfile[maxfcc] = "";                                         //  pending image, add to layout
char        pendtext[maxtext+1] = "";                                      //  pending text, add to layout
char        textfont[60] = "sans 40";                                      //  layout text default font
int         textbw = 'b';                                                  //  default text: black on white
int         transp = 0;                                                    //  text background is transparent
int         framrgb[3] = { 0, 0, 150 };                                    //  default frame color       v.2.2
int         framethick = 0;                                                //  default frame thickness   v.2.2

struct      playout {                                                      //  image and text members of layout
   int         type;                                                       //  1: image, 2: text image
   int         textbw;                                                     //  text color, 'b' or 'w'
   int         transp;                                                     //  background is transparent
   char        *filespec;                                                  //  image filespec
   char        *textspec;                                                  //  text image text content
   GdkPixbuf   *pixbuf1;                                                   //  image pixbuf, full size
   GdkPixbuf   *pixbuf2;                                                   //  image pixbuf, rscale size
   int         ww, hh;                                                     //  full image dimensions
   int         xpos, ypos;                                                 //  position in layout
   double      rscale;                                                     //  scale factor, image to layout
   double      angle;                                                      //  rotation, degrees CW      v.2.0
   int         framrgb[3];                                                 //  frame color               v.2.0
   int         framw;                                                      //  frame thickness           v.2.0
};
playout     layout[200];                                                   //  layout for all images + text
int         maxNF = 200;                                                   //  max. images + text in layout
int         NPL = 0;                                                       //  current layout count

//  GTK functions
int   gtkinitfunc(void *data);                                             //  GTK initz. function
void  mwpaint();                                                           //  window repaint - expose event
void  mwpaint2();                                                          //  window repaint - application call
void  destroy();                                                           //  main window destroy signal
void  mouse_event(GtkWidget *, GdkEventButton *, void *);                  //  mouse event function
int   KBpress(GtkWidget *, GdkEventKey *, void *);                         //  KB key press event
int   KBrelease(GtkWidget *, GdkEventKey *, void *);                       //  KB key release event
void  drag_drop(int x, int y, char *text);                                 //  text drag/drop event   v.2.0.2
void  menufunc(GtkWidget *, const char *menu);                             //  menu function, main window

//  toolbar button functions
void  m_gallery();                                                         //  show image gallery (thumbnail) window
void  m_image();                                                           //  add images to layout
void  m_text();                                                            //  add text to layout
void  m_rotate();                                                          //  rotate image
void  m_frame();                                                           //  edit image frame
void  m_pack();                                                            //  pack images, close gaps
void  m_trim();                                                            //  trim layout to fit images
void  m_save();                                                            //  save layout to file
void  m_print();                                                           //  print finished layout 
void  m_help();                                                            //  show user guide
void  m_quit();                                                            //  quit

void  init_layout();                                                       //  page layout from paper size
void  set_pendfile(char *file);                                            //  set pending file to be added
void  add_layout_image(int mpx, int mpy);                                  //  add pending image to layout
void  add_layout_text(int mpx, int mpy);                                   //  add pending text to layout
void  save_imagedirk();                                                    //  save for next startup
void  load_imagedirk();                                                    //  load saved image directory

GdkPixbuf * make_layout(int final);                                        //  make layout pixbuf
GdkPixbuf * load_pixbuf(const char *file);                                 //  load pixbuf from file, remove alpha
GdkPixbuf * pixbuf_text(GtkWidget *, cchar *text, cchar *font, int bw);    //  create pixbuf containing text
void pixbuf_copy_alpha(GdkPixbuf *pxb1, int x1, int y1, int ww, int hh,    //  copy pixbuf with transparent color 
                       GdkPixbuf *pxb2, int x2, int y2, int rgb, int ff);
GdkPixbuf * pixbuf_add_frame(GdkPixbuf *pixbuf1, int *rgb, int thick);     //  add frame around a pixbuf
int parse_format(const char *, char *, float &, float &, char *);          //  parse a print format string
int parse_frame(const char *, int &thick, int *rgb);                       //  parse a frame data record


/**************************************************************************/

//  main program

int main(int argc, char *argv[])
{
   GtkWidget   *mtbar;
   char        lang[8] = "";

   printf(fversion "\n");                                                  //  print version
   if (argc > 1 && strEqu(argv[1],"-v")) return 0;

   gtk_init(&argc,&argv);                                                  //  GTK command line options

   zinitapp("mashup",null);                                                //  get app directories

   for (int ii = 1; ii < argc; ii++)                                       //  command line options
   {
      if (strEqu(argv[ii],"-l") && argc > ii+1)                            //  -l language code
            strncpy0(lang,argv[++ii],7);
      else if (strEqu(argv[ii],"-i") && argc > ii+1)                       //  -i imageDirectory
            strcpy(imagedirk,argv[++ii]);
   }
   
   ZTXinit(lang);                                                          //  setup translations

   mainWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                          //  create main window
   gtk_window_set_title(GTK_WINDOW(mainWin),fversion);
   gtk_window_set_position(GTK_WINDOW(mainWin),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(mainWin),wwD,hhD);

   mVbox = gtk_vbox_new(0,0);                                              //  add vert. packing box
   gtk_container_add(GTK_CONTAINER(mainWin),mVbox);

   mtbar = create_toolbar(mVbox);                                          //  add toolbar and buttons  
   add_toolbar_button(mtbar,ZTX("gallery"),ZTX("image gallery"),"gallery.png",menufunc);
   add_toolbar_button(mtbar,ZTX("+image"),ZTX("add image"),"image.png",menufunc);
   add_toolbar_button(mtbar,ZTX("+text"),ZTX("add text"),"text.png",menufunc);
   add_toolbar_button(mtbar,ZTX("rotate"),ZTX("rotate image"),"rotate.png",menufunc);
   add_toolbar_button(mtbar,ZTX("frame"),ZTX("edit frame"),"frame.png",menufunc);
   add_toolbar_button(mtbar,ZTX("pack"),ZTX("pack images"),"pack.png",menufunc);
   add_toolbar_button(mtbar,ZTX("trim"),ZTX("trim layout"),"trim.png",menufunc);
   add_toolbar_button(mtbar,ZTX("toolbar::save"),ZTX("save layout to file"),"gtk-save",menufunc);
   add_toolbar_button(mtbar,ZTX("print"),ZTX("setup and print"),"print.png",menufunc);
   add_toolbar_button(mtbar,ZTX("quit"),ZTX("quit mashup"),"gtk-quit",menufunc);
   add_toolbar_button(mtbar,ZTX("help"),ZTX("help"),"gtk-help",menufunc);

   drawWin = gtk_drawing_area_new();                                       //  drawing window
   gtk_container_add(GTK_CONTAINER(mVbox),drawWin);                        //  add to main window

   G_SIGNAL(mainWin,"destroy",destroy,0)                                   //  connect events to main window
   G_SIGNAL(drawWin,"expose-event",mwpaint,0)

   G_SIGNAL(mainWin,"key-press-event",KBpress,0)                           //  connect KB events
   G_SIGNAL(mainWin,"key-release-event",KBrelease,0)

   gtk_widget_add_events(drawWin,GDK_BUTTON_PRESS_MASK);                   //  connect mouse events
   gtk_widget_add_events(drawWin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(drawWin,GDK_BUTTON_MOTION_MASK);
   gtk_widget_add_events(drawWin,GDK_POINTER_MOTION_MASK);
   G_SIGNAL(drawWin,"button-press-event",mouse_event,0)
   G_SIGNAL(drawWin,"button-release-event",mouse_event,0)
   G_SIGNAL(drawWin,"motion-notify-event",mouse_event,0)

   drag_drop_connect(drawWin,drag_drop);                                   //  connect drag-drop event  v.2.0.2
   
   gtk_widget_show_all(mainWin);                                           //  show all widgets
   gdkgc = gdk_gc_new(drawWin->window);                                    //  initz. graphics context
   
   gtk_init_add((GtkFunction) gtkinitfunc,0);                              //  initz. function
   gtk_main();                                                             //  process window events

   return 0;
}


//  initial function called from gtk_main() at startup

int gtkinitfunc(void * data)
{
   char        file[200], *pp, buff[100];
   int         ii, err;
   float       ww = 0, hh = 0;
   FILE        *fid;
   
   *format = *orent = 0;

   snprintf(file,200,"%s/paper-format",get_zuserdir());                    //  get prior paper format   v.2.4
   fid = fopen(file,"r");
   if (fid) {
      pp = fgets_trim(buff,99,fid,1);                                      //  v.2.5
      if (pp) {
         err = parse_format(pp,format,ww,hh,orent);
         if (! err) printf("saved format: %s %.1f %.1f %s \n",format,ww,hh,orent);
      }
      fclose(fid);
   }
   
   if (*format) {
      for (ii = 0; ii < Npformats; ii++)
         if (strEqu(format,pformats[ii])) break;
      if (ii < Npformats-1) {                                              //  one of the predefined formats
         parse_format(pformats[ii],format,ww,hh,orent);
         printf(" substituted: %s %.1f %.1f %s \n", format,ww,hh,orent);
      }
   }
   
   if (ww < 3 || ww > 42 || hh < 3 || hh > 42) {                           //  screen crazy values
      parse_format(pformats[deformat],format,ww,hh,orent);                 //  substitute default format
      printf(" substituted: %s %.1f %.1f %s \n", format,ww,hh,orent);
   }
   
   paper_wwcm = ww;
   paper_hhcm = hh;
   
   if (! strcmpv(orent,"portrait","landscape",0))                          //  bad orientation > landscape
      strcpy(orent,"landscape");

   snprintf(file,200,"%s/frame-data",get_zuserdir());                      //  get prior frame data      v.2.7
   fid = fopen(file,"r");
   if (fid) {
      pp = fgets_trim(buff,99,fid,1);
      if (pp) parse_frame(buff,framethick,framrgb);
      printf("frame data: %d %d %d %d \n",framethick,framrgb[0],framrgb[1],framrgb[2]);
      fclose(fid);
   }

   if (! *imagedirk) load_imagedirk();                                     //  get prior image directory

   init_layout();                                                          //  set up page layout pixbuf
   return 0;
}


//  paint window when created, exposed, resized

void mwpaint()
{
   GdkPixbuf   *pxbM;
   GdkDrawable *Lwin = drawWin->window;
   int         ww, hh;
   double      scalew, scaleh;
   
   if (pxb_layout) g_object_unref(pxb_layout);                             //  (re)make editing layout
   pxb_layout = make_layout(0);

   wwD = drawWin->allocation.width;                                        //  current drawing window size
   hhD = drawWin->allocation.height;

   scalew = 1.0 * wwD / layout_ww;                                         //  scale layout to window
   scaleh = 1.0 * hhD / layout_hh;
   if (scalew < scaleh) Rscale = scalew;
   else Rscale = scaleh;
   ww = int(Rscale * layout_ww + 0.5);
   hh = int(Rscale * layout_hh + 0.5);
   pxbM = pixbuf_scale(pxb_layout,ww,hh,bilinear);
   if (! pxbM) zappcrash("cannot create pixbuf");

   gdk_draw_pixbuf(Lwin,0,pxbM,0,0,0,0,-1,-1,nodither);
   g_object_unref(pxbM);

   return;
}


//  invalidate window - cause repaint for application changes

void mwpaint2()
{
   gdk_window_invalidate_rect(drawWin->window,0,1);                        //  invalidate whole window   v.1.8
   return;
}


//  main window destroy signal

void destroy()
{
   printf("main window destroyed \n");
   m_quit();
   return;
}


//  mouse event function - capture buttons and drag movements

void mouse_event(GtkWidget *, GdkEventButton *event, void *)
{
   static int     bdtime = 0, butime = 0;
   static int     Lmouse = 0, Rmouse = 0;
   static int     Fclick = 0, Fselect = 0, Fcorner = 0;
   static int     elapsed, mpx0 = 0, mpy0 = 0;
   static int     ii, kk, ww, hh;
   int            mpx, mpy, dx = 0, dy = 0;
   int            x1, y1, x2, y2;
   double         rscale;
   playout        templayout;

   mpx = int(event->x);                                                    //  mouse position in window
   mpy = int(event->y);
   mpx = int(mpx / Rscale + 0.5);                                          //  in print layout
   mpy = int(mpy / Rscale + 0.5);

   if (event->type == GDK_BUTTON_PRESS)
   {
      Lmouse = Rmouse = 0;
      if (event->button == 1) Lmouse++;                                    //  left or right mouse button
      if (event->button == 3) Rmouse++;
      bdtime = event->time;
      Fselect = Fcorner = 0;
      layout_select = -1;                                                  //  set no current item    v.2.0

      for (ii = NPL-1; ii >= 0; ii--)                                      //  search from top image down
      {
         rscale = layout[ii].rscale;
         ww = gdk_pixbuf_get_width(layout[ii].pixbuf2);                    //  cursor inside an image?  v.2.0
         hh = gdk_pixbuf_get_height(layout[ii].pixbuf2);
         if (mpx < layout[ii].xpos) continue;
         if (mpx > layout[ii].xpos + ww) continue;
         if (mpy < layout[ii].ypos) continue;
         if (mpy > layout[ii].ypos + hh) continue;
         break;
      }
      if (ii >= 0) {                                                       //  yes, iith image/text selected
         layout_select = ii;                                               //  remember last selected object
         if (layout[ii].type == 1)                                         //  set current image file
            strcpy(imagefile,layout[ii].filespec);                         //  (image gallery window anchor)
         else strcpy(pendtext,layout[ii].textspec);                        //  or current text string
         Fselect = 1;
      }
      
      if (Fselect) 
      {
         mpx0 = mpx;                                                       //  set new drag origin
         mpy0 = mpy;
         
         if (layout[ii].type == 1) {                                       //  mouse in image corner?
            if (mpx < layout[ii].xpos + 0.2 * ww  &&
                mpy < layout[ii].ypos + 0.2 * hh) Fcorner = 1;             //  upper left
            if (mpx > layout[ii].xpos + 0.8 * ww  &&
                mpy < layout[ii].ypos + 0.2 * hh) Fcorner = 2;             //  upper right
            if (mpx > layout[ii].xpos + 0.8 * ww  &&
                mpy > layout[ii].ypos + 0.8 * hh) Fcorner = 3;             //  lower right
            if (mpx < layout[ii].xpos + 0.2 * ww  &&
                mpy > layout[ii].ypos + 0.8 * hh) Fcorner = 4;             //  lower left
         }

         if (layout[ii].type == 2) {                                       //  mouse in text corner?
            if (ww > hh) {                                                 //  (broader range for narrow edge)
               if (mpx < layout[ii].xpos + 0.2 * ww  &&
                   mpy < layout[ii].ypos + 0.5 * hh) Fcorner = 1;          //  upper left
               if (mpx > layout[ii].xpos + 0.8 * ww  &&
                   mpy < layout[ii].ypos + 0.5 * hh) Fcorner = 2;          //  upper right
               if (mpx > layout[ii].xpos + 0.8 * ww  &&
                   mpy > layout[ii].ypos + 0.5 * hh) Fcorner = 3;          //  lower right
               if (mpx < layout[ii].xpos + 0.2 * ww  &&
                   mpy > layout[ii].ypos + 0.5 * hh) Fcorner = 4;          //  lower left
            }

            else  {
               if (mpx < layout[ii].xpos + 0.5 * ww  &&
                   mpy < layout[ii].ypos + 0.2 * hh) Fcorner = 1;          //  upper left
               if (mpx > layout[ii].xpos + 0.5 * ww  &&
                   mpy < layout[ii].ypos + 0.2 * hh) Fcorner = 2;          //  upper right
               if (mpx > layout[ii].xpos + 0.5 * ww  &&
                   mpy > layout[ii].ypos + 0.8 * hh) Fcorner = 3;          //  lower right
               if (mpx < layout[ii].xpos + 0.5 * ww  &&
                   mpy > layout[ii].ypos + 0.8 * hh) Fcorner = 4;          //  lower left
            }
         }
      }
      
      return;
   }
   
   if (event->type == GDK_MOTION_NOTIFY)                                   //  capture mouse movement
   {
      if (! Lmouse) return;                                                //  only if left-mouse

      if (Fselect) {
         dx = mpx - mpx0;                                                  //  capture drag motion
         dy = mpy - mpy0;
         mpx0 = mpx;                                                       //  set new drag origin
         mpy0 = mpy;
      }
      
      if (Fselect && ! Fcorner) {                                          //  image move: add mouse drag
         layout[ii].xpos += dx;                                            //    to image position
         layout[ii].ypos += dy;
      }
      
      if (Fcorner)                                                         //  image resize: add mouse drag
      {                                                                    //    to image size
         rscale = layout[ii].rscale;
         ww = layout[ii].ww;
         hh = layout[ii].hh;
         x1 = layout[ii].xpos;
         y1 = layout[ii].ypos;
         x2 = x1 + int(rscale * ww + 0.5);
         y2 = y1 + int(rscale * hh + 0.5);

         if (Fcorner == 1) {                                               //  drag upper left corner
            if (ww > hh) rscale = (rscale * ww - dx) / ww;
            else  rscale = (rscale * hh - dy) / hh;
            x1 = x2 - int(rscale * ww + 0.5);                              //  leave opposite corner fixed
            y1 = y2 - int(rscale * hh + 0.5);
         }

         if (Fcorner == 2) {                                               //  upper right
            if (ww > hh) rscale = (rscale * ww + dx) / ww;
            else  rscale = (rscale * hh - dy) / hh;
            x2 = x1 + int(rscale * ww + 0.5);
            y1 = y2 - int(rscale * hh + 0.5);
         }

         if (Fcorner == 3) {                                               //  lower right
            if (ww > hh) rscale = (rscale * ww + dx) / ww;
            else  rscale = (rscale * hh + dy) / hh;
            x2 = x1 + int(rscale * ww + 0.5);
            y2 = y1 + int(rscale * hh + 0.5);
         }

         if (Fcorner == 4) {                                               //  lower left
            if (ww > hh) rscale = (rscale * ww - dx) / ww;
            else  rscale = (rscale * hh + dy) / hh;
            x1 = x2 - int(rscale * ww + 0.5);
            y2 = y1 + int(rscale * hh + 0.5);
         }
         
         if (rscale < 0.05 || rscale > 4.0) return;                        //  limit extremes
         
         if (ww > hh) rscale = 1.0 * (x2 - x1) / ww;                       //  make precise
         else  rscale = 1.0 * (y2 - y1) / hh;

         if (layout[ii].pixbuf2) g_object_unref(layout[ii].pixbuf2);       //  discard old scaled pixbuf
         layout[ii].pixbuf2 = 0;

         layout[ii].rscale = rscale;                                       //  will recreate from new data
         layout[ii].xpos = x1;
         layout[ii].ypos = y1;
      }
   }
   
   if (event->type == GDK_BUTTON_RELEASE)
   {
      Fclick = 0;
      butime = event->time;
      elapsed = butime - bdtime;                                           //  button down time, milliseconds
      if (elapsed < 500) Fclick = 1;                                       //  mouse clicked, no drag

      if (Fselect && Fclick && Lmouse) {                                   //  image was left-clicked
         templayout = layout[ii];
         for (kk = ii; kk < NPL-1; kk++)                                   //  make topmost (will occlude others)
         {
            if (layout[kk+1].type > layout[ii].type) break;                //  all text above all images   v.1.6
            layout[kk] = layout[kk+1];
         }
         layout[kk] = templayout;
         ii = layout_select = kk;                                          //  new current image index     v.1.7
      }

      if (Fselect && Fclick && Rmouse) {                                   //  image was right-clicked
         if (layout[ii].filespec) zfree(layout[ii].filespec);
         if (layout[ii].textspec) zfree(layout[ii].textspec);
         if (layout[ii].pixbuf1) g_object_unref(layout[ii].pixbuf1);       //  delete image from layout
         if (layout[ii].pixbuf2) g_object_unref(layout[ii].pixbuf2);
         for (kk = ii; kk < NPL-1; kk++)                                   //  cover the hole
            layout[kk] = layout[kk+1];
         NPL--;
         layout_select = -1;                                               //  no selection active    v.2.0
      }

      if (! Fselect && ! Fcorner && Fclick) {                              //  empty space was clicked
         if (*pendfile) add_layout_image(mpx,mpy);                         //  add pending file image to layout
         else if (*pendtext) add_layout_text(mpx,mpy);                     //  add pending text image to layout
      }
      
      if (layout_select >= 0) {                                            //  if item selected and dialog 
         if (zdtext) zdialog_send_event(zdtext,"select");                  //    active, send select event  
         if (zdrotate) zdialog_send_event(zdrotate,"select");              //      to dialog     v.2.0
         if (zdframe) zdialog_send_event(zdframe,"select");
      }

      Fselect = Fcorner = Fclick = 0;                                      //  stop drag/resize
      Lmouse = Rmouse = 0;
   }

   mwpaint2();                                                             //  repaint window
   return;
}


//  prevent propagation of key-press events to toolbar buttons     v.1.7

int KBpress(GtkWidget *win, GdkEventKey *event, void *)
{
   return 1;
}


//  keyboard event - arrow keys >> 1-pixel movement of current image
//  GDK key symbols: /usr/include/gtk-2.0/gdk/gdkkeysyms.h

int KBrelease(GtkWidget *win, GdkEventKey *event, void *)                  //  process KB arrow keys   v.1.7
{
   int         ii, KBkey;

   if (layout_select < 0) return 1;
   ii = layout_select;

   KBkey = event->keyval;
   if (KBkey == GDK_Left) --layout[ii].xpos;                               //  move image in 1-pixel steps
   if (KBkey == GDK_Right) ++layout[ii].xpos;
   if (KBkey == GDK_Up) --layout[ii].ypos;
   if (KBkey == GDK_Down) ++layout[ii].ypos;
   
   mwpaint2();
   return 1;
}


//  text drag and drop event - get filespec or text to add to layout       v.2.0.2

void drag_drop(int mpx, int mpy, char *text)
{
   mpx = int(mpx / Rscale + 0.5);                                          //  position in print layout
   mpy = int(mpy / Rscale + 0.5);
   
   *pendfile = *pendtext = 0;                                              //  cancel prior pending items
   
   if (*text == '/')                                                       //  text is a filespec
   {
      set_pendfile(text);                                                  //  check if valid image file
      if (*pendfile) add_layout_image(mpx,mpy);                            //  add to layout at position
   }
   
   else                                                                    //  text is just text
   {      
      strncpy0(pendtext,text,maxtext);                                     //  save for layout insertion
      add_layout_text(mpx,mpy);
   }

   zfree(text);
   mwpaint2();
   return;
}


//  process main window toolbar events

void menufunc(GtkWidget *, const char *menu)
{
   if (strEqu(menu,ZTX("gallery"))) { m_gallery(); return; }
   if (strEqu(menu,ZTX("+image"))) { m_image(); return; }
   if (strEqu(menu,ZTX("+text"))) { m_text(); return; }
   if (strEqu(menu,ZTX("rotate"))) { m_rotate(); return; }
   if (strEqu(menu,ZTX("frame"))) { m_frame(); return; }
   if (strEqu(menu,ZTX("pack"))) { m_pack(); return; }
   if (strEqu(menu,ZTX("trim"))) { m_trim(); return; }
   if (strEqu(menu,ZTX("toolbar::save"))) { m_save(); return; }            //  bugfix     v.2.1.2
   if (strEqu(menu,ZTX("print"))) { m_print(); return; }
   if (strEqu(menu,ZTX("quit"))) { m_quit(); return; }
   if (strEqu(menu,ZTX("help"))) { m_help(); return; }
   return;
}


//  display an image gallery (thumbnails) in a separate window

void m_gallery()                                                           //  v.1.3.2
{
   void galleryclick(int Nth);

   int            Nth;
   static char    pimagedirk[maxfcc] = "";
   
   if (strNeq(imagedirk,pimagedirk)) {                                     //  directory changed, new gallery
      image_gallery(imagedirk,"init",0,galleryclick);
      strcpy(pimagedirk,imagedirk);
   }

   if (*imagefile) {
      Nth = image_gallery_position(imagefile,0);                           //  position gallery at curr. file
      image_gallery(imagefile,"paint1",Nth,galleryclick);
   }
   else  image_gallery(0,"paint1");

   return;
}


//  this function receives clicked thumbnails from the gallery window

void galleryclick(int Nth)
{
   char     *file;

   if (Nth < 0) return;
   file = image_gallery(0,"find",Nth);
   if (file) set_pendfile(file);
   if (file) zfree(file);
   return;
}


//  choose image files and add to print layout
//  file-chooser dialog box and response function
//  chosen file is saved for drawing window deposit via mouse-click

void m_image()
{
   int image_compl(GtkDialog *dwin, int arg, void *data);
   void image_preview(GtkFileChooser *, GtkWidget *pvwidget);

   GtkWidget   *dialog, *file_widget;
   GtkWidget   *pvwidget = gtk_image_new();

   dialog = gtk_dialog_new_with_buttons(ZTX("choose files"),GTK_WINDOW(mainWin),
                                         (GtkDialogFlags) 0,ZTX("done"),2,null);
   gtk_window_set_default_size(GTK_WINDOW(dialog),700,500);
   G_SIGNAL(dialog,"response",image_compl,0)

   file_widget = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_OPEN);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox),file_widget);
   gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(file_widget),imagedirk);
   gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER(file_widget),0);

   gtk_file_chooser_set_preview_widget(GTK_FILE_CHOOSER(file_widget),pvwidget);
   G_SIGNAL(file_widget,"update-preview",image_preview,pvwidget);

   gtk_widget_show_all(dialog);
   return;
}

int image_compl(GtkDialog *dwin, int arg, void *data)                      //  done or cancel
{
   gtk_widget_destroy(GTK_WIDGET(dwin));
   return 0;
}

void image_preview(GtkFileChooser *file_widget, GtkWidget *pvwidget)       //  get file preview thumbnail
{
   GdkPixbuf   *thumbnail;
   char        *filename;
   
   filename = gtk_file_chooser_get_preview_filename(file_widget);
   if (! filename) return;

   thumbnail = image_thumbnail(filename,128);                              //  get 128x128 thumbnail   v.1.7
   if (thumbnail) {                                                        //  (cached or created)
      gtk_image_set_from_pixbuf(GTK_IMAGE(pvwidget),thumbnail);
      gtk_file_chooser_set_preview_widget_active(file_widget,1);
      g_object_unref(thumbnail);
      set_pendfile(filename);                                              //  save for layout insertion   v.2.0
   }
   else
      gtk_file_chooser_set_preview_widget_active(file_widget,0);

   g_free(filename);
   return;
}


//  add text to print layout
//  get text dialog box and response function
//  entered text is saved for drawing window deposit via mouse-click

void m_text()
{
   int text_dialog_event(zdialog *, const char *event);

   int         ii;
   const char  *helptext = ZTX(" enter text, select font, \n"
                               " click empty space on layout ");
   if (zdtext) return;

   zdtext = zdialog_new(ZTX("enter text"),mainWin,ZTX("cancel"),null);
   zdialog_add_widget(zdtext,"label","lab1","dialog",helptext,"space=10");
   zdialog_add_widget(zdtext,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdtext,"label","lab2","hb1",ZTX("text"),"space=3");        //  text [_______________]
   zdialog_add_widget(zdtext,"frame","fr1","hb1",0,"expand");
   zdialog_add_widget(zdtext,"edit","text","fr1",0,"space=5");
   zdialog_add_widget(zdtext,"hbox","hb2","dialog");                             //  (o) black   (o) white
   zdialog_add_widget(zdtext,"radio","black","hb2",ZTX("black"),"space=5");
   zdialog_add_widget(zdtext,"radio","white","hb2",ZTX("white"),"space=10");
   zdialog_add_widget(zdtext,"hbox","hb3","dialog");                             //  transparent [_]
   zdialog_add_widget(zdtext,"label","lab3","hb3",ZTX("transparent"),"space=8");
   zdialog_add_widget(zdtext,"check","transp","hb3",0);
   zdialog_add_widget(zdtext,"hbox","hb4","dialog",0,"space=10");                //  [erase text]  [select font] 
   zdialog_add_widget(zdtext,"button","erase","hb4",ZTX("erase text"));
   zdialog_add_widget(zdtext,"button","font","hb4",ZTX("select font"));

   ii = layout_select;                                                     //  last selected text image
   if (ii >= 0 && layout[ii].type == 2) {
      textbw = layout[ii].textbw;
      transp = layout[ii].transp;
      strcpy(pendtext,layout[ii].textspec);
      *pendfile = 0;
   }

   if (textbw == 'b') zdialog_stuff(zdtext,"black",1);                     //  use prior text color and
   else  zdialog_stuff(zdtext,"white",1);                                  //    transparency options
   if (transp) zdialog_stuff(zdtext,"transp",1);
   zdialog_stuff(zdtext,"text",pendtext);                                  //  prior or pending text

   zdialog_run(zdtext,text_dialog_event);                                  //  run dialog, parallel
   return;
}


//  dialog event function

int text_dialog_event(zdialog *zd,const char *event)
{
   GtkWidget   *font_dialog;
   char        *pp, text[1000] = "";
   int         ii, acolor;
   
   if (zd->zstat)
   {
      zdialog_free(zdtext);
      return 0;
   }
   
   if (strEqu(event,"erase")) {                                            //  erase text entry box
      zdialog_stuff(zd,"text","");
      *pendtext = 0;
      return 0;
   }
   
   if (strEqu(event,"font")) {                                             //  select a new font
      font_dialog = gtk_font_selection_dialog_new(ZTX("select font"));
      gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(font_dialog),textfont);
      gtk_dialog_run(GTK_DIALOG(font_dialog));
      pp = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(font_dialog));
      if (pp) strncpy0(textfont,pp,59);
      gtk_widget_destroy(GTK_WIDGET(font_dialog));
   }
   
   if (strEqu(event,"select")) {
      ii = layout_select;                                                  //  last selected text image
      if (ii < 0 || layout[ii].type != 2) return 0;
      zdialog_stuff(zd,"text",layout[ii].textspec);
   }

   zdialog_fetch(zd,"text",text,999);                                      //  capture entered text
   zdialog_fetch(zd,"transp",transp);                                      //  and transparency choice
   zdialog_fetch(zd,"black",acolor);                                       //  and black/white choice
   
   if (acolor) textbw = 'b';                                               //  black or white
   else textbw = 'w';

   if (*text) {
      if (NPL == maxNF-1) {
         zmessageACK(mainWin,ZTX("exceed %d files"),maxNF);
         return 0;
      }
      strncpy0(pendtext,text,maxtext);                                     //  save for layout insertion
      *pendfile = 0;
   }

   return 0;
}


//  rotate last-selected image or text through arbitrary angle             //  new  v.2.0

void m_rotate()
{
   int    rotate_dialog_event(zdialog *, const char * event);

   char        text[20];
   double      angle = 0;
   
   if (zdrotate) return;

   zdrotate = zdialog_new(ZTX("rotate image"),mainWin,ZTX("done"),ZTX("cancel"),null);
   zdialog_add_widget(zdrotate,"label","labdeg","dialog",ZTX("degrees"),"space=5");
   zdialog_add_widget(zdrotate,"hbox","hb1","dialog",0,"homog|space=5");
   zdialog_add_widget(zdrotate,"vbox","vb1","hb1",0,"space=5");
   zdialog_add_widget(zdrotate,"vbox","vb2","hb1",0,"space=5");
   zdialog_add_widget(zdrotate,"vbox","vb3","hb1",0,"space=5");
   zdialog_add_widget(zdrotate,"vbox","vb4","hb1",0,"space=5");
   zdialog_add_widget(zdrotate,"button"," +0.1  ","vb1"," + 0.1 ");        //  button name is increment to use
   zdialog_add_widget(zdrotate,"button"," -0.1  ","vb1"," - 0.1 ");
   zdialog_add_widget(zdrotate,"button"," +1.0  ","vb2"," + 1   ");
   zdialog_add_widget(zdrotate,"button"," -1.0  ","vb2"," - 1   ");
   zdialog_add_widget(zdrotate,"button"," +10.0 ","vb3"," + 10  ");
   zdialog_add_widget(zdrotate,"button"," -10.0 ","vb3"," - 10  ");
   zdialog_add_widget(zdrotate,"button"," +90.0 ","vb4"," + 90  ");
   zdialog_add_widget(zdrotate,"button"," -90.0 ","vb4"," - 90  ");
   zdialog_add_widget(zdrotate,"hbox","hb2","dialog",0,"space=10");

   if (layout_select >= 0) angle = layout[layout_select].angle;
   sprintf(text,ZTX("degrees: %.1f"),angle);                               //  update dialog angle display
   zdialog_stuff(zdrotate,"labdeg",text);

   zdialog_run(zdrotate,rotate_dialog_event);                              //  run dialog - parallel
   return;
}


//  dialog event function

int rotate_dialog_event(zdialog *zd, const char * event)
{
   int         ii, err;
   char        text[20];
   double      angle, incr;
   
   if (zd->zstat)
   {
      zdialog_free(zdrotate);
      if (zd->zstat != 1 && layout_select >= 0) 
         layout[layout_select].angle = 0;
      mwpaint2();
      return 0;
   }

   ii = layout_select;                                                     //  use last selected object
   if (ii < 0) return 0;

   if (strpbrk(event,"+-")) {
      err = convSD(event,incr);                                            //  button name is increment to use
      if (err) return 0;
   }
   else return 0;

   angle = layout[ii].angle + incr;

   sprintf(text,ZTX("degrees: %.1f"),angle);                               //  update dialog angle display
   zdialog_stuff(zd,"labdeg",text);
   
   if (angle >= 360) angle -= 360;
   if (angle <= -360) angle += 360;
   if (fabs(angle) < 0.01) angle = 0;
   
   layout[ii].angle = angle;
   if (layout[ii].pixbuf2) g_object_unref(layout[ii].pixbuf2);
   layout[ii].pixbuf2 = 0;

   mwpaint2();
   return 1;
}


//  edit the frame around an image

void m_frame()
{
   int    frame_dialog_event(zdialog *, const char * event);

   int      ii;
   char     color[20];

   if (zdframe) return;                                                    //  already active

   ii = layout_select;                                                     //  use last selected image
   if (ii < 0) return;
   if (layout[ii].type != 1) return;
   
   framethick = layout[ii].framw;                                          //  set frame data from image
   memcpy(framrgb,layout[ii].framrgb,3*sizeof(int));

   zdframe = zdialog_new(ZTX("edit frame"),mainWin,ZTX("done"),ZTX("cancel"),null);
   zdialog_add_widget(zdframe,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdframe,"label","lab1","hb1",ZTX("thickness"));
   zdialog_add_widget(zdframe,"spin","thick","hb1","0|20|1|8","scc=2");
   zdialog_add_widget(zdframe,"label","space","hb1","","space=10");
   zdialog_add_widget(zdframe,"label","lab2","hb1",ZTX("color"));
   zdialog_add_widget(zdframe,"colorbutt","color","hb1","100|100|100");
   zdialog_add_widget(zdframe,"hbox","hb2","dialog");
   zdialog_add_widget(zdframe,"radio","this","hb2","this image");
   zdialog_add_widget(zdframe,"radio","all","hb2","all images","space=10");
   
   zdialog_stuff(zdframe,"thick",layout[ii].framw);                        //  stuff frame data into dialog
   snprintf(color,19,"%d|%d|%d",framrgb[0],framrgb[1],framrgb[2]);
   zdialog_stuff(zdframe,"color",color);

   zdialog_run(zdframe,frame_dialog_event);                                //  run dialog - parallel
   return;
}


//  dialog event function

int frame_dialog_event(zdialog *zd, const char * event)
{
   int         ii, Fall;
   char        color[20];
   const char  *pp;
   
   if (zd->zstat)
   {
      zdialog_free(zdframe);
      mwpaint2();
      return 0;
   }

   ii = layout_select;                                                     //  use last selected image 
   if (ii < 0) return 0;
   if (layout[ii].type != 1) return 0;

   framethick = layout[ii].framw;                                          //  set frame data from image
   memcpy(framrgb,layout[ii].framrgb,3*sizeof(int));

   if (strEqu(event,"select")) {                                           //  stuff dialog from image frame
      zdialog_stuff(zd,"thick",framethick);
      snprintf(color,19,"%d|%d|%d",framrgb[0],framrgb[1],framrgb[2]);
      zdialog_stuff(zd,"color",color);
      return 1;
   }

   zdialog_fetch(zd,"thick",framethick);                                   //  get frame data from dialog
   zdialog_fetch(zd,"color",color,19);

   pp = strField(color,'|',1);
   if (pp) framrgb[0] = atoi(pp);
   pp = strField(color,'|',2);
   if (pp) framrgb[1] = atoi(pp);
   pp = strField(color,'|',3);
   if (pp) framrgb[2] = atoi(pp);
   
   zdialog_fetch(zd,"all",Fall);

   if (! Fall)                                                             //  update selected image
   {
      layout[ii].framw = framethick;                                       //  set image frame data
      layout[ii].framrgb[0] = framrgb[0];
      layout[ii].framrgb[1] = framrgb[1];
      layout[ii].framrgb[2] = framrgb[2];
      if (layout[ii].pixbuf2) g_object_unref(layout[ii].pixbuf2);          //  refresh current image
      layout[ii].pixbuf2 = 0;
   }
   
   if (Fall)                                                               //  update all images      v.2.6
   {
      for (ii = 0; ii < NPL; ii++)
      {
         if (layout[ii].type != 1) continue;
         layout[ii].framw = framethick;
         layout[ii].framrgb[0] = framrgb[0];
         layout[ii].framrgb[1] = framrgb[1];
         layout[ii].framrgb[2] = framrgb[2];
         if (layout[ii].pixbuf2) g_object_unref(layout[ii].pixbuf2);
         layout[ii].pixbuf2 = 0;
      }
   }

   mwpaint2();
   return 1;
}


//  align and pack the images

void m_pack()                                                              //  overhauled v.2.8.1
{
   #define pnear(x,y) (abs((x)-(y)) < packtolr)                            //  close, within packing tolerance

   int         ii, jj, ii0;
   int         packtolr;
   int         iixp, iiyp, iiww, iihh, iixpww, iiyphh, iifr;
   int         jjxp, jjyp, jjww, jjhh, jjxpww, jjyphh, jjfr;
   double      iirs, jjrs, ww2, hh2;
   
   packtolr = layout_ww;                                                   //  [pack] alignment tolerance,
   if (layout_hh > packtolr) packtolr = layout_hh;                         //    3% of layout scale
   packtolr = packtolr * pack_tolerance;

   for (ii = 0; ii < NPL; ii++)                                            //  find first image after text images
      if (layout[ii].type == 1) break;
   ii0 = ii;

   for (ii = ii0; ii < NPL; ii++)                                          //  if near layout top or left edge,
   {                                                                       //    pack to top or left edge
      if (pnear(layout[ii].xpos,0)) layout[ii].xpos = 0;
      if (pnear(layout[ii].ypos,0)) layout[ii].ypos = 0;
   }

   for (ii = ii0; ii < NPL; ii++)                                          //  pack images (jj) to the left,
   for (jj = ii0; jj < NPL; jj++)                                          //    against image on the left (ii)
   {
      if (layout[jj].xpos <= layout[ii].xpos) continue;                    //  jj is left of ii
      iixpww = layout[ii].xpos + layout[ii].rscale * layout[ii].ww 
                               + layout[ii].framw;
      if (! pnear(layout[jj].xpos,iixpww)) continue;                       //  jj left edge not near ii right edge
      iiyphh = layout[ii].ypos + layout[ii].rscale * layout[ii].hh 
                               + layout[ii].framw;
      if (layout[jj].ypos > iiyphh) continue;                              //  jj is completely below ii
      jjyphh = layout[jj].ypos + layout[jj].rscale * layout[jj].hh 
                               + layout[jj].framw;
      if (jjyphh < layout[ii].ypos) continue;                              //  jj is completely above ii
      layout[jj].xpos = iixpww;                                            //  move jj to right edge of ii
   }

   for (ii = ii0; ii < NPL; ii++)                                          //  pack images (jj) up,
   for (jj = ii0; jj < NPL; jj++)                                          //    against image above (ii)
   {
      if (layout[jj].ypos <= layout[ii].ypos) continue;                    //  jj is above ii
      iiyphh = layout[ii].ypos + layout[ii].rscale * layout[ii].hh 
                               + layout[ii].framw;
      if (! pnear(layout[jj].ypos,iiyphh)) continue;                       //  jj top not near ii bottom
      iixpww = layout[ii].xpos + layout[ii].rscale * layout[ii].ww 
                               + layout[ii].framw;
      if (layout[jj].xpos > iixpww) continue;                              //  jj is completely right of ii
      jjxpww = layout[jj].xpos + layout[jj].rscale * layout[jj].ww 
                               + layout[jj].framw;
      if (jjxpww < layout[ii].xpos) continue;                              //  jj is completely left of ii
      layout[jj].ypos = iiyphh;                                            //  move jj up to bottom of ii
   }
   
   for (ii = ii0; ii < NPL; ii++)                                          //  align jj corners where close to
   for (jj = ii0; jj < NPL; jj++)                                          //    ii corners along close edges
   {
      iixp = layout[ii].xpos;                                              //  left edge in layout
      iiyp = layout[ii].ypos;                                              //  top edge
      iiww = layout[ii].ww;                                                //  image full width
      iihh = layout[ii].hh;                                                //  image full height
      iifr = layout[ii].framw;                                             //  frame size
      iirs = layout[ii].rscale;                                            //  image scale to layout
      iixpww = iixp + iirs * iiww + iifr;                                  //  right edge in layout
      iiyphh = iiyp + iirs * iihh + iifr;                                  //  bottom edge

      jjxp = layout[jj].xpos;
      jjyp = layout[jj].ypos;
      jjww = layout[jj].ww;
      jjhh = layout[jj].hh;
      jjfr = layout[jj].framw;
      jjrs = layout[jj].rscale;
      jjxpww = jjxp + jjrs * jjww + jjfr;
      jjyphh = jjyp + jjrs * jjhh + jjfr;

      if (pnear(jjxp,iixpww) && pnear(jjyp,iiyp)) {                        //  jj top left near ii top right
         jjxp = iixpww;                                                    //  shift jj to meet ii
         jjyp = iiyp;
      }

      if (pnear(jjxp,iixp) && pnear(jjyp,iiyphh)) {                        //  jj top left near ii bottom left
         jjxp = iixp;                                                      //  shift jj to meet ii
         jjyp = iiyphh;
      }

      if (pnear(jjxp,iixpww) && pnear(jjyphh,iiyphh)) {                    //  jj bottom left near ii bottom right
         jjxp = iixpww;
         if (jjyphh != iiyphh) {                                           //  resize jj
            jjyphh = iiyphh;
            hh2 = jjyphh - jjyp - jjfr;
            jjrs = hh2 / jjhh;
            jjxpww = jjxp + jjrs * jjww + jjfr;
            jjyphh = jjyp + jjrs * jjhh + jjfr;
         }
      }

      if (pnear(jjxpww,iixpww) && pnear(jjyp,iiyphh)) {                    //  jj top right near ii bottom right
         jjyp = iiyphh;
         if (jjxpww != iixpww) {                                           //  resize jj
            jjxpww = iixpww;
            ww2 = jjxpww - jjxp - jjfr;
            jjrs = ww2 / jjww;
            jjxpww = jjxp + jjrs * jjww + jjfr;
            jjyphh = jjyp + jjrs * jjhh + jjfr;
         }
      }

      if (iirs != layout[ii].rscale) {                                     //  image ii to be rescaled
         if (layout[ii].pixbuf2) g_object_unref(layout[ii].pixbuf2);
         layout[ii].pixbuf2 = 0;
      }

      if (jjrs != layout[jj].rscale) {                                     //  image jj to be rescaled
         if (layout[jj].pixbuf2) g_object_unref(layout[jj].pixbuf2);
         layout[jj].pixbuf2 = 0;
      }

      layout[ii].rscale = iirs;
      layout[jj].xpos = jjxp;
      layout[jj].ypos = jjyp;
      layout[jj].rscale = jjrs;
   }

   mwpaint2();
}


//  trim the layout to minimum size that fits the contained images

void m_trim()                                                              //  new v.2.6
{
   int         ii, ww, hh, maxww, maxhh;
   
   maxww = maxhh = 0;
   
   for (ii = 0; ii < NPL; ii++)
   {
      ww = layout[ii].xpos + layout[ii].rscale * layout[ii].ww + 2 * layout[ii].framw;
      if (ww > maxww) maxww = ww;
      hh = layout[ii].ypos + layout[ii].rscale * layout[ii].hh + 2 * layout[ii].framw;
      if (hh > maxhh) maxhh = hh;
   }
   
   layout_ww = maxww;
   layout_hh = maxhh;
   mwpaint2();
   return;
}


//  save page layout to a file

void m_save()
{
   GdkPixbuf   *layout2;
   GError      *gerror = 0;                                                //  bugfix   v.2.3.1
   char        printfile[200], command[maxfcc+100];
   char        file1[maxfcc], *file2, *pp;
   int         ignore;
   
   layout2 = make_layout(1);                                               //  make 200% layout
   snprintf(printfile,199,"%s/printfile.jpg",get_zuserdir());              //  jpeg print file      v.2.8
   gdk_pixbuf_save(layout2,printfile,"jpeg",&gerror,"quality","100",null);
   g_object_unref(layout2);
   
   snprintf(file1,maxfcc,"%s/mashup.jpg",imagedirk);                       //  .jpg instead of .jpeg   v.2.8
   
   file2 = zgetfile1(ZTX("save file"),"save",file1);                       //  query user for file
   if (! file2) return;
   
   strncpy0(file1,file2,maxfcc-6);
   zfree(file2);
   pp = strrchr(file1,'/');                                                //  force .jpg extension   v.2.8
   if (pp) pp = strrchr(pp,'.');
   if (! pp) pp = file1 + strlen(file1);
   strcpy(pp,".jpg");

   snprintf(command,maxfcc+100,"cp -f \"%s\" \"%s\" ",printfile,file1);
   ignore = system(command);
   return;
}


//  dialog for print setup and print

int   print_layout_err = 0;

void m_print()
{
   char        paper_format[100];
   int         zstat;
   GdkPixbuf   *layout2;
   GError      *gerror = 0;                                                //  bugfix   v.2.3.1
   char        printfile[200];
   zdialog     *zd;

   int   print_dialog_event(zdialog *zd, const char *event);

   zd = zdialog_new(ZTX("print format"),mainWin,ZTX("OK"),ZTX("print"),null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labpap","hb1",ZTX("paper format"));         //  paper format [_________|v]
   zdialog_add_widget(zd,"comboE","entpap","hb1",0,"scc=20|space=3");         //  (o) portrait  (o) landscape
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(zd,"radio","port","hb2",ZTX("portrait"),"space=10");
   zdialog_add_widget(zd,"radio","land","hb2",ZTX("landscape"),"space=10");

   for (int ii = 0; ii < Npformats; ii++)                                  //  stuff combobox with paper formats
      zdialog_cb_app(zd,"entpap",pformats[ii]);

   snprintf(paper_format,99,"%s  %.1f x %.1f cm",                          //  preload previous format
                    format,paper_wwcm,paper_hhcm);
   zdialog_stuff(zd,"entpap",paper_format);
   if (strEqu(orent,"landscape")) zdialog_stuff(zd,"land",1);              //  and orientation
   else  zdialog_stuff(zd,"port",1);

   zdialog_run(zd,print_dialog_event);                                     //  run dialog
   zstat = zdialog_wait(zd);
   zdialog_free(zd);                                                       //  kill dialog
   if (zstat != 2) return;                                                 //  continue only if [print]
   if (print_layout_err) return;                                           //  and no layout error

   layout2 = make_layout(1);                                               //  make 200% layout

   snprintf(printfile,200,"%s/printfile.jpg",get_zuserdir());              //  layout >> .jpg print file  v.2.5
   gdk_pixbuf_save(layout2,printfile,"jpeg",&gerror,"quality","100",null);
   g_object_unref(layout2);
   
   snprintf(paper_format,99,"%s  %.1f x %.1f cm",format,paper_wwcm,paper_hhcm);
   print_imagefile(printfile,paper_format,orent);                          //  GTK print function
   return;
}


//  print dialog event function
//  dialog fields >> memory parameters >> layout

int print_dialog_event(zdialog *zd, const char *event)
{
   char        paper_format[100];
   float       ww = 0, hh = 0;
   int         port;
   
   zdialog_fetch(zd,"entpap",paper_format,99);                             //  get paper size from format
   parse_format(paper_format,format,ww,hh,orent);

   if (ww < 3 || ww > 42 || hh < 3 || hh > 42) {                           //  screen for reasonableness
      print_layout_err = 1;
      return 0;
   }
   else  print_layout_err = 0;

   paper_wwcm = ww;
   paper_hhcm = hh;

   zdialog_fetch(zd,"port",port);                                          //  get orientation
   if (port) strcpy(orent,"portrait");
   else strcpy(orent,"landscape");
   
   init_layout();                                                          //  update layout to match
   return 0;
}


//  thread functions to display help file

void m_help()
{
   showz_userguide();
   return;
}


//  quit - exit program

void m_quit()
{
   char     file[200];
   FILE     *fid;

   save_imagedirk();

   snprintf(file,200,"%s/paper-format",get_zuserdir());
   fid = fopen(file,"w");                                                  //  update paper format file
   if (fid) {
      fprintf(fid," %s  %.1f x %.1f cm  %s \n",format,paper_wwcm,paper_hhcm,orent);
      fclose(fid);
   }

   snprintf(file,200,"%s/frame-data",get_zuserdir());
   fid = fopen(file,"w");                                                  //  frame-data file
   if (fid) {
      fprintf(fid,"frame-data %d %d %d %d \n",framethick,framrgb[0],framrgb[1],framrgb[2]);
      fclose(fid);
   }

   gtk_main_quit();
   return;
}


//  set up page layout pixbuf based on layout dimensions

void init_layout()
{
   if (strEqu(orent,"landscape")) {                                        //  v.2.4
      layout_ww = layoutsize;
      layout_hh = 1.0 * layout_ww * paper_wwcm / paper_hhcm + 0.5;
   }
   else {
      layout_hh = layoutsize;
      layout_ww = 1.0 * layout_hh * paper_wwcm / paper_hhcm + 0.5;
   }
   
   pxb_layout = gdk_pixbuf_new(colorspace,0,8,layout_ww,layout_hh);        //  pixbuf for page layout
   if (! pxb_layout) 
      zappcrash("cannot create pixbuf %.1f %.1f %d %d",
                  paper_wwcm, paper_hhcm, layout_ww, layout_hh);

   wwD = int(Rscale * layout_ww);                                          //  matching drawing window size
   hhD = int(Rscale * layout_hh + 60);
   gtk_window_resize(GTK_WINDOW(mainWin),wwD,hhD);

   return;
}


//  verify chosen file and make pending add to layout
 
void set_pendfile(char *file)
{
   GdkPixbuf      *pixbuf;
   char           *pp;

   if (NPL == maxNF-1) {
      zmessageACK(mainWin,ZTX("exceed %d files"),maxNF);
      return;
   }
   
   pixbuf = load_pixbuf(file);                                             //  test file is legit. image file
   if (! pixbuf) {
      *pendfile = 0;                                                       //  simply ignore    v.2.0
      return;
   }
   g_object_unref(pixbuf);

   strcpy(pendfile,file);                                                  //  save filespec
   *pendtext = 0;

   strcpy(imagefile,file);                                                 //  set curr. image file
   strcpy(imagedirk,file);                                                 //  and image directory
   pp = (char *) strrchr(imagedirk,'/');
   if (pp) *pp = 0;
   
   gtk_window_present(GTK_WINDOW(mainWin));                                //  bring to foreground
   return;
}


//  add a new image file to the layout at designated position

void add_layout_image(int mpx, int mpy)
{
   GdkPixbuf   *pixbuf;
   int         ii, kk;

   pixbuf = load_pixbuf(pendfile);
   if (! pixbuf) zappcrash("cannot create pixbuf");
   for (ii = 0; ii < NPL; ii++)                                            //  insert at end of images
      if (layout[ii].type != 1) break;                                     //  (topmost file image)
   for (kk = NPL; kk > ii; kk--)
      layout[kk] = layout[kk-1];
   NPL++;
   layout_select = ii;
   layout[ii].type = 1;
   layout[ii].textbw = 0;
   layout[ii].transp = 0;
   layout[ii].filespec = strdupz(pendfile);
   layout[ii].textspec = 0;
   layout[ii].pixbuf1 = pixbuf;
   layout[ii].pixbuf2 = 0;
   layout[ii].ww = gdk_pixbuf_get_width(layout[ii].pixbuf1);
   layout[ii].hh = gdk_pixbuf_get_height(layout[ii].pixbuf1);
   layout[ii].xpos = mpx;
   layout[ii].ypos = mpy;
   layout[ii].rscale = 1.0 * layout_pix / layout[ii].ww;
   layout[ii].angle = 0;                                                   //  v.2.0
   memcpy(layout[ii].framrgb,framrgb,3*sizeof(int));                       //  v.2.0
   layout[ii].framw = framethick;                                          //  v.2.0
   *pendfile = 0;                                                          //  v.2.0
   return;
}


//  add a new text image to the layout at designated position

void add_layout_text(int mpx, int mpy)
{
   GdkPixbuf   *pixbuf;
   int         ii;

   pixbuf = pixbuf_text(drawWin,pendtext,textfont,textbw);
   if (! pixbuf) zappcrash("cannot create text pixbuf");
   ii = NPL++;                                                             //  topmost text image
   layout_select = ii;
   layout[ii].type = 2;
   layout[ii].textbw = textbw;
   layout[ii].transp = transp;
   layout[ii].textspec = strdupz(pendtext);
   layout[ii].filespec = 0;
   layout[ii].pixbuf1 = pixbuf;
   layout[ii].pixbuf2 = 0;
   layout[ii].ww = gdk_pixbuf_get_width(layout[ii].pixbuf1);
   layout[ii].hh = gdk_pixbuf_get_height(layout[ii].pixbuf1);
   layout[ii].xpos = mpx;
   layout[ii].ypos = mpy;
   layout[ii].rscale = 0.5;
   layout[ii].angle = 0;                                                   //  v.2.0
   layout[ii].framw = 0;                                                   //  v.2.0
   *pendtext = 0;                                                          //  v.2.0
   return;
}


//  save current image directory upon exit, reload upon startup            //  v.1.2
//  directory is saved in file $HOME/.mashup/image_directory

void save_imagedirk()
{
   char     command[maxfcc+100];
   int      ignore;
   
   snprintf(command,maxfcc+99,"echo %s > %s/image_directory",              //  save imagedirk to file
                                    imagedirk, get_zuserdir());
   ignore = system(command);
   return;
}

void load_imagedirk()
{
   int            err;
   FILE           *fid;
   struct stat    statdat;
   char           dirbuff[maxfcc], *pp;

   pp = getcwd(imagedirk,maxfcc-1);                                        //  default is current directory

   snprintf(dirbuff,maxfcc-1,"%s/image_directory",get_zuserdir());         //  read saved file
   fid = fopen(dirbuff,"r");
   if (! fid) return;
   pp = fgets_trim(dirbuff,maxfcc-1,fid,1);
   fclose(fid);
   if (! pp) return;
   err = stat(dirbuff,&statdat);                                           //  contains valid directory name?
   if (err) return;
   if (! S_ISDIR(statdat.st_mode)) return;
   strcpy(imagedirk,dirbuff);                                              //  yes, use it
   return;
}


//  make a layout pixbuf containing all scaled images
//  final = 0 = edit pixbuf: 1x size and scaled image pixbufs are saved and reused
//  final = 1 = pixbuf to print or save: 2x size and scaled image pixbufs are discarded

GdkPixbuf * make_layout(int final)
{
   GdkPixbuf   *pixbuf1, *pixbuf2, *pixbuf3;
   int         type, textbw, transp, ftext;
   int         ii, lscale, lww, lhh, iww, ihh;
   int         xpos, ypos, orgx, orgy;
   int         rgb, rcolor, acolor;
   int         *framrgb;
   int         framethick;
   double      angle, rscale;
   
   lscale = 1;
   if (final) lscale = 2;                                                  //  use 2x size for final layout
   
   lww = lscale * layout_ww;                                               //  layout size
   lhh = lscale * layout_hh;

   pixbuf1 = gdk_pixbuf_new(colorspace,0,8,lww,lhh);                       //  layout pixbuf
   if (! pixbuf1) zappcrash("cannot create pixbuf");
   gdk_pixbuf_fill(pixbuf1,(unsigned) 0xffffffff);
   
   for (ii = 0; ii < NPL; ii++)                                            //  copy image pixbufs
   {                                                                       //    into layout pixbuf
      type = layout[ii].type;
      textbw = layout[ii].textbw;
      transp = layout[ii].transp;
      rscale = layout[ii].rscale;
      angle = layout[ii].angle;
      framrgb = layout[ii].framrgb;
      framethick = layout[ii].framw;
      pixbuf2 = layout[ii].pixbuf2;                                        //  image pixbuf
      iww = lscale * int(rscale * layout[ii].ww + 0.5);                    //  image size
      ihh = lscale * int(rscale * layout[ii].hh + 0.5);
      
      if (type == 1) {                                                     //  image                  v.2.0
         if (angle == 0) rcolor = acolor = 0;                              //  no angle, no rotate border
         else rcolor = acolor = 'w';                                       //  border and alpha color = white
         ftext = 0;
      }
      else {                                                               //  text
         if (textbw == 'b') rcolor = acolor = 'w';                         //  rotate border and alpha color
         else  rcolor = acolor = 'b';                                      //    opposite of text color
         if (! transp) acolor = 0;                                         //  opaque background, no alpha color
         ftext = 1;
      }

      if (! pixbuf2 || final) {
         pixbuf2 = pixbuf_scale(layout[ii].pixbuf1,iww,ihh,bilinear);      //  scale image to layout

         if (type == 1) {
            pixbuf3 = pixbuf_add_frame(pixbuf2,framrgb,framethick);        //  add frame around image    v.2.0
            g_object_unref(pixbuf2);
            pixbuf2 = pixbuf3;
         }

         if (angle) {                                                      //  rotate if needed    v.2.0
            if (rcolor == 'b') rgb = 0;
            else rgb = 255;
            pixbuf3 = gdk_pixbuf_rotate(pixbuf2,angle,rgb);
            g_object_unref(pixbuf2);
            pixbuf2 = pixbuf3;
         }

         if (! final) layout[ii].pixbuf2 = pixbuf2;                        //  retain for efficiency
      }

      iww = gdk_pixbuf_get_width(pixbuf2);                                 //  (rotated) image size   v.2.0
      ihh = gdk_pixbuf_get_height(pixbuf2);

      xpos = lscale * layout[ii].xpos;                                     //  image position in layout
      ypos = lscale * layout[ii].ypos;

      orgx = orgy = 0;                                                     //  cut-off beyond layout edge
      if (xpos < 0) {
         orgx = -xpos;
         xpos = 0;
      }
      if (ypos < 0) {
         orgy = -ypos;
         ypos = 0;
      }

      if (xpos + iww > lww) iww = lww - xpos;
      if (ypos + ihh > lhh) ihh = lhh - ypos;

      if (acolor) {                                                        //  paste image into layout
         if (acolor == 'b') rgb = 0;                                       //    with transparency
         else rgb = 255;
         pixbuf_copy_alpha(pixbuf2,orgx,orgy,iww-orgx,ihh-orgy,pixbuf1,xpos,ypos,rgb,ftext);
      }
      else gdk_pixbuf_copy_area(pixbuf2,orgx,orgy,iww-orgx,ihh-orgy,pixbuf1,xpos,ypos);
      
      if (final) g_object_unref(pixbuf2);
   }
   
   return pixbuf1;
}


//  validate an image file and load pixbuf from file                       //  v.1.2
//  if an alpha channel is present, remove it

GdkPixbuf * load_pixbuf(const char *file)
{
   GdkPixbuf   *pxb91 = 0, *pxb92 = 0;
   GError      *gerror = 0;                                                //  bugfix   v.2.3.1
   int         ww91, hh91, rs91, rs92;
   int         nch, nbits, px, py, alfa;
   pixel       ppix91, pix91, ppix92, pix92;

   pxb91 = gdk_pixbuf_new_from_file(file,&gerror);                         //  validate file and load pixbuf
   if (! pxb91) return 0;

   nch = gdk_pixbuf_get_n_channels(pxb91);
   nbits = gdk_pixbuf_get_bits_per_sample(pxb91);
   alfa = gdk_pixbuf_get_has_alpha(pxb91);

   if (nch < 3 || nbits != 8) {                                            //  must be 3 or 4 channels
      g_object_unref(pxb91);                                               //  and 8 bits per channel
      return 0;
   }
   
   if (! alfa) return pxb91;                                               //  no alpha channel, 3 channels

   ww91 = gdk_pixbuf_get_width(pxb91);                                     //  copy without alpha    v.1.2
   hh91 = gdk_pixbuf_get_height(pxb91);
   rs91 = gdk_pixbuf_get_rowstride(pxb91);
   ppix91 = gdk_pixbuf_get_pixels(pxb91);

   pxb92 = gdk_pixbuf_new(colorspace,0,8,ww91,hh91);
   if (! pxb92) zappcrash("cannot create pixbuf");
   rs92 = gdk_pixbuf_get_rowstride(pxb92);
   ppix92 = gdk_pixbuf_get_pixels(pxb92);

   for (py = 0; py < hh91; py++)
   for (px = 0; px < ww91; px++)
   {
      pix91 = ppix91 + rs91 * py + 4 * px;
      pix92 = ppix92 + rs92 * py + 3 * px;
      pix92[0] = pix91[0];
      pix92[1] = pix91[1];
      pix92[2] = pix91[2];
   }

   g_object_unref(pxb91);
   return pxb92;
}


//  create a pixbuf containing text
//  drawWin is from gtk_drawing_area_new()
//  drawWin must be a realized window even though it is not modified
//  bw is 'b'/'w' for black/white text on white/black background

GdkPixbuf * pixbuf_text(GtkWidget *drawWin, cchar *text, cchar *font, int bw)
{
   PangoFontDescription  *pfont;
   static GdkColormap    *colormap = 0;
   static GdkColor       black, white;
   GdkColor              foreground, background;

   PangoLayout    *playout;
   GdkPixmap      *pixmap;
   GdkGC          *gdkgc;
   GdkPixbuf      *pixbuf;
   int            ww, hh;

   if (! colormap) {
      black.red = black.green = black.blue = 0;
      white.red = white.green = white.blue = 0xffff;
      colormap = gtk_widget_get_colormap(drawWin);
      gdk_rgb_find_color(colormap,&black);
      gdk_rgb_find_color(colormap,&white);
   }
   
   if (bw == 'b') {
      foreground = black;
      background = white;
   }
   else {
      foreground = white;
      background = black;
   }

   pfont = pango_font_description_from_string(font);
   playout = gtk_widget_create_pango_layout(drawWin,null);
   pango_layout_set_font_description(playout,pfont);
   pango_layout_set_text(playout,text,-1);
   pango_layout_get_pixel_size(playout,&ww,&hh);
   if (! ww) return 0;

   pixmap = gdk_pixmap_new(drawWin->window,ww,hh,-1);
   if (! pixmap) zappcrash("cannot create pixmap");
   gdkgc = gdk_gc_new(pixmap);
   gdk_gc_set_foreground(gdkgc,&background);
   gdk_draw_rectangle(pixmap,gdkgc,1,0,0,ww,hh);

   gdk_draw_layout_with_colors(pixmap,gdkgc,0,0,playout,&foreground,&background);
   pixbuf = gdk_pixbuf_get_from_drawable(null,pixmap,0,0,0,0,0,ww,hh);
   if (! pixbuf) zappcrash("cannot create text pixbuf");
   
   g_object_unref(playout);
   g_object_unref(pixmap);
   g_object_unref(gdkgc);

   return pixbuf;
}


//  copy pixbuf1 into pixbuf2, treating color "rgb" as transparent
//  rgb = 0 = black, rgb = 255 = white
//  ftext = true to blend anti-aliased text edges (looks better)

void pixbuf_copy_alpha(GdkPixbuf *pxb1, int x1, int y1, int ww, int hh,
                       GdkPixbuf *pxb2, int x2, int y2, int rgb, int ftext)
{
   int         px, py, px1, py1, px2, py2;
   int         ww1, hh1, rs1, ww2, hh2, rs2;
   pixel       ppix1, ppix2, pix1, pix2;
   int         red1, green1, blue1, red2, green2, blue2;
   double      f1, f2;
   
   ww1 = gdk_pixbuf_get_width(pxb1);
   hh1 = gdk_pixbuf_get_height(pxb1);
   rs1 = gdk_pixbuf_get_rowstride(pxb1);
   ppix1 = gdk_pixbuf_get_pixels(pxb1);

   ww2 = gdk_pixbuf_get_width(pxb2);
   hh2 = gdk_pixbuf_get_height(pxb2);
   rs2 = gdk_pixbuf_get_rowstride(pxb2);
   ppix2 = gdk_pixbuf_get_pixels(pxb2);
   
   for (px = 0; px < ww; px++)
   for (py = 0; py < hh; py++)
   {
      px1 = x1 + px;                                                       //  copy-from pixel
      py1 = y1 + py;
      pix1 = ppix1 + rs1 * py1 + 3 * px1;

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      if (red1 == rgb && green1 == rgb && blue1 == rgb)                    //  transparent, do not copy
         continue;
      
      px2 = x2 + px;                                                       //  copy-to pixel
      py2 = y2 + py;
      pix2 = ppix2 + rs2 * py2 + 3 * px2;

      if (ftext)
      {
         red2 = pix2[0];
         green2 = pix2[1];
         blue2 = pix2[2];
         
         f1 = abs(rgb - red1) / 255.0;                                     //  blend pixels (text)
         f2 = 1.0 - f1;
         red2 = int(f1 * red1 + f2 * red2);

         f1 = abs(rgb - green1) / 255.0;
         f2 = 1.0 - f1;
         green2 = int(f1 * green1 + f2 * green2);

         f1 = abs(rgb - blue1) / 255.0;
         f2 = 1.0 - f1;
         blue2 = int(f1 * blue1 + f2 * blue2);
         
         pix2[0] = red2;
         pix2[1] = green2;
         pix2[2] = blue2;
      }

      else                                                                 //  no blending (picture)  v.2.0
      {
         pix2[0] = red1;
         pix2[1] = green1;
         pix2[2] = blue1;
      }
   }

   return;
}


//  add a frame around a pixbuf image                                      //  new v.2.0
//  new pixbuf with added frame is returned
//  rgb: frame color   thick: thickness in pixels

GdkPixbuf * pixbuf_add_frame(GdkPixbuf *pixbuf1, int *rgb, int thick)
{
   GdkPixbuf   *pixbuf2;
   int         px, py;
   int         ww1, hh1, ww2, hh2, rs2;
   pixel       ppix2, pix2;
   
   ww1 = gdk_pixbuf_get_width(pixbuf1);                                    //  input pixbuf dimensions
   hh1 = gdk_pixbuf_get_height(pixbuf1);

   ww2 = ww1 + 2 * thick;                                                  //  new pixbuf with space for frame
   hh2 = hh1 + 2 * thick;
   pixbuf2 = gdk_pixbuf_new(colorspace,0,8,ww2,hh2);
   if (! pixbuf2) zappcrash("cannot create pixbuf");
   rs2 = gdk_pixbuf_get_rowstride(pixbuf2);
   ppix2 = gdk_pixbuf_get_pixels(pixbuf2);
   
   gdk_pixbuf_copy_area(pixbuf1,0,0,ww1,hh1,pixbuf2,thick,thick);          //  copy input pixbuf to output
   
   for (py = 0; py < thick; py++)                                          //  color frame, top
   for (px = 0; px < ww2; px++)
   {
      pix2 = ppix2 + rs2 * py + 3 * px;
      pix2[0] = rgb[0];
      pix2[1] = rgb[1];
      pix2[2] = rgb[2];
   }
   
   for (py = 0; py < hh2; py++)                                            //  right
   for (px = ww2 - thick; px < ww2; px++)
   {
      pix2 = ppix2 + rs2 * py + 3 * px;
      pix2[0] = rgb[0];
      pix2[1] = rgb[1];
      pix2[2] = rgb[2];
   }
   
   for (py = hh2 - thick; py < hh2; py++)                                  //  bottom
   for (px = 0; px < ww2; px++)
   {
      pix2 = ppix2 + rs2 * py + 3 * px;
      pix2[0] = rgb[0];
      pix2[1] = rgb[1];
      pix2[2] = rgb[2];
   }
   
   for (py = 0; py < hh2; py++)                                            //  left
   for (px = 0; px < thick; px++)
   {
      pix2 = ppix2 + rs2 * py + 3 * px;
      pix2[0] = rgb[0];
      pix2[1] = rgb[1];
      pix2[2] = rgb[2];
   }
   
   return pixbuf2;
}


//  parse a print format string independently of the locale                //  v.2.5
//  format string format:  fname  N.NN x N.NN cm  orientation
//  returns 0 if no error

int parse_format(const char *fstring, char *fname, float &ww, float &hh, char *orien)
{
   const char  *pp;
   int         err;
   double      val;
   
   pp = strField(fstring,' ',1);                                           //  A4, letter, custom ...
   if (! pp) return 1;
   strncpy0(fname,pp,20);

   pp = strField(fstring,' ',2);                                           //  width
   if (! pp) return 2;
   err = convSD(pp,val,3,42);
   if (err) return 2;
   ww = val;

   pp = strField(fstring,' ',4);                                           //  height
   if (! pp) return 4;
   err = convSD(pp,val,3,42);
   if (err) return 4;
   hh = val;
   
   pp = strField(fstring,' ',6);                                           //  orientation
   if (! pp) return 6;
   strncpy0(orien,pp,20);

   return 0;
}


//  parse a saved frame data record

int parse_frame(const char *framerec, int &thick, int *rgb)                //  v.2.7
{
   char     recname[12];
   int      ii, jj, rr, gg, bb;

   thick = 0;                                                              //  set frame defaults
   rgb[0] = rgb[1] = rgb[2] = 128;
   
   ii = sscanf(framerec,"%12s %d %d %d %d",recname,&jj,&rr,&gg,&bb);
   if (ii != 5) return 0;
   if (! strEqu(recname,"frame-data")) return 0;
   if (jj < 0 || jj > 20) return 0;
   thick = jj;
   if (rr < 0 || rr > 255) return 0;
   rgb[0] = rr;
   if (gg < 0 || gg > 255) return 0;
   rgb[1] = gg;
   if (bb < 0 || bb > 255) return 0;
   rgb[2] = bb;
   return 1;
}




