/*
    EasyImageSizer - Exif

    EasyImageSizer plugin rename images

    Copyright (C) 2010  FalseCAM

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "exif.h"
#include "ui_exif.h"

Exif::Exif(QWidget *parent) :
        QWidget(parent),
        ui(new Ui::Exif)
{
    loadTranslation();
    ui->setupUi(this);
    loadState();
}

Exif::~Exif()
{
    saveState();
    delete ui;
}

QString Exif::getName(){
    return QString("Exif");
}

QString Exif::getTitle(){
    return QString(tr("Rename by Exifdata"));
}

QString Exif::getVersion(){
    return QString("0.2");
}

QString Exif::getAuthor(){
    return QString("FalseCAM");
}

QString Exif::getDescription(){
    return QString(tr("Plugin to rename Images by Exifdata"));
}

void Exif::convert(QImage &image, QString &fileName, int index, QString origImage){
    if(ui->prependRadio->isChecked()){
        fileName = exifToString(origImage) + fileName;
    }else if(ui->appendRadio->isChecked()){
        fileName.append(exifToString(origImage));
    }else{}
}

QString Exif::exifToString(QString origImage){
    return getDateTime(origImage).toString(ui->lineEdit->text());
}

QDateTime Exif::getDateTime(QString file)
{
    // Command String
    QString commandStr = QString("exiftool -exif:DateTimeOriginal -s -s -s \"").append(file).append("\"");

    // Start new Process that reads exif date and time
    QProcess process;
    process.start(commandStr );
    // Wait until process finished
    process.waitForFinished();
    // read Process string
    QString output = QString::fromLatin1(process.readLine());
    output.remove(QChar('\r'));
    output.remove(QChar('\n'));
    // get Date from Output String
    QDateTime DateTime = QDateTime::fromString(output, "yyyy:MM:dd hh:mm:ss");

    // Check if there is a correct Date
    if(!DateTime.isValid()) return QDateTime( QDate(1900,1,1), QTime(0,0,0) );

    return DateTime;
}

// loads last state of gui
void Exif::loadState(){
    QSettings settings(QCoreApplication::organizationName(), QCoreApplication::applicationName());
    settings.beginGroup("plugin");
    settings.beginGroup(QString(getName()));
    ui->prependRadio->setChecked(settings.value("Prepend", false).toBool());
    ui->appendRadio->setChecked(settings.value("Append", false).toBool());
    ui->lineEdit->setText(settings.value("String", "").toString());
}

// saves state of gui
void Exif::saveState(){
    QSettings settings(QCoreApplication::organizationName(), QCoreApplication::applicationName());
    settings.beginGroup("plugin");
    settings.beginGroup(QString(getName()));
    settings.setValue("Prepend", ui->prependRadio->isChecked());
    settings.setValue("Append", ui->appendRadio->isChecked());
    settings.setValue("String", ui->lineEdit->text());

}

void Exif::loadTranslation(){
    QString filename;
    filename = QString("lang_" + QLocale::system().name());
    QTranslator *translator = new QTranslator;
    QString translationsDir = QString(":/"+ getName().toLower() +"/languages/");
    translator->load(filename, translationsDir );
    QTranslator *translator2 = new QTranslator;
#if defined(Q_OS_WIN)
    translationsDir = qApp->applicationDirPath() + "/languages/";
#elif defined(Q_OS_LINUX)
    translationsDir = QString("/usr/share/"+qApp->applicationName().toLower()+"/languages/");
#endif
    translationsDir.append(getName().toLower()+"/");
    translator2->load(filename, translationsDir );
    qApp->installTranslator(translator);
    qApp->installTranslator(translator2);
}

Q_EXPORT_PLUGIN2(exif, Exif);
