/*
 * Dillo
 *
 * Some code copied from:
 * The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "history.h"
#include "nav.h"
#include "commands.h"
#include "bookmark.h"
#include "interface.h"
#include "menu.h"
#include "prefs.h"

/* CALLBACKS */
/*
 * Called when the user selects "Copy Link Location" in the link menu.
 */
void a_Menu_popup_select_url_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;
   a_Commands_set_selection(URL_STR(a_Menu_popup_get_url(bw)));
}

/*
 * Put the url in the status bar when moving cursor over menuitem.
 */
static void
Menu_popup_history_select_callback(GtkWidget *widget, gpointer data)
{
   gint idx;
   BrowserWindow *bw = data;

   idx = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT (widget), "nav_idx"));
   if (idx >= 0 && idx < a_Nav_stack_size(bw->dd)) {
      a_Interface_msg(bw, "%s", URL_STR_(a_History_get_url(NAV_IDX(bw->dd,idx))));
      bw->status_is_link = 1;
   }
}

/*
 * Remove the url in the status bar when removing cursor from menuitem.
 */
static void
Menu_popup_history_deselect_callback(GtkWidget *widget, gpointer data) {
   BrowserWindow *bw = data;

   a_Interface_msg(bw, "");
}

/* PRIVATE FUNCTIONS */

/*
 * Make a new menu, insert it into the menu bar, and return it.
 */
static GtkWidget *
Menu_new(GtkWidget *menubar, const char *name,
                           gboolean right_justify, BrowserWindow *bw)
{
   GtkWidget *menu;
   GtkWidget *menuitem;
   guint tmp_key;

   menu = gtk_menu_new();
   menuitem = gtk_menu_item_new_with_label((char *) name);
   tmp_key = gtk_label_parse_uline(GTK_LABEL(GTK_BIN(menuitem)->child), name);
   gtk_widget_add_accelerator(menuitem, "activate_item", bw->accel_group,
                              tmp_key, GDK_MOD1_MASK, 0);

   if ( right_justify )
      gtk_menu_item_right_justify(GTK_MENU_ITEM(menuitem));
   gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu);
   gtk_menu_bar_append(GTK_MENU_BAR(menubar), menuitem);
   gtk_widget_show(menuitem);
   gtk_menu_set_accel_group(GTK_MENU(menu), bw->accel_group);
   return menu;
}

/*
 * Add an item to a menu, including the name, an accelerator (not yet
 * implemented), and a callback function for activation.
 */
static GtkWidget *
Menu_add (GtkWidget *menu, const char *name, const char *accel,
           BrowserWindow *bw,
           void (*callback) (GtkWidget *widget, void *data), void *data)
{
   GtkWidget *menuitem;
   GtkAccelGroup *menu_accels;
   GdkModifierType accel_mods;
   guint accel_key;
   guint tmp_key;

   menuitem = gtk_menu_item_new_with_label((char *) name);
   gtk_menu_append(GTK_MENU(menu), menuitem);
   menu_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(menu));
   tmp_key = gtk_label_parse_uline(GTK_LABEL(GTK_BIN(menuitem)->child), name);
   gtk_widget_add_accelerator(menuitem, "activate_item",
                              menu_accels, tmp_key, 0, 0);
   gtk_widget_show(menuitem);

   if (accel != NULL) {
      gtk_accelerator_parse(accel, &accel_key, &accel_mods);
      gtk_widget_add_accelerator(menuitem, "activate", bw->accel_group,
         accel_key, (guint)accel_mods, GTK_ACCEL_VISIBLE);
   }
   if (callback != NULL)
      gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                         (GtkSignalFunc) callback, data);
   return menuitem;
}

/*
 * Add a separator into the menu.
 */
static void Menu_sep(GtkWidget *menu)
{
   GtkWidget *widget;

   widget = gtk_menu_item_new();
   gtk_menu_append(GTK_MENU(menu), widget);
   gtk_widget_set_sensitive(widget, FALSE);
   gtk_widget_show(widget);
}

/* PUBLIC FUNCTIONS */

/*
 * Make up a new menubar for a main browser window. The accelerator table
 * is stored in bw->accel_group.
 * Currently does not deal with dynamic menu items (bookmarks and history).
 * CP: It *seems* to handle dynamic menu items...
 */
GtkWidget *a_Menu_mainbar_new(BrowserWindow *bw, gint tiny)
{
   GtkWidget *menubar;
   GtkWidget *file_menu;
#ifndef DISABLE_TABS
   GtkWidget *view_menu;
#endif /* !DISABLE_TABS */

   /* GtkWidget *bookmarks_menu; */
   /* GtkWidget *help_menu; */

   bw->menubar = menubar = gtk_menu_bar_new();

   /* FILE MENU */
   file_menu = Menu_new(menubar, tiny ? "_F" : "_File", FALSE, bw);
   Menu_add(file_menu, "_New Browser", "<ctrl>N", bw,
            a_Commands_new_callback, bw);
#ifndef DISABLE_TABS
   Menu_add(file_menu, "New _Tab", "<ctrl>T", bw,
            a_Commands_new_tab_callback, bw);
   Menu_sep(file_menu);
#endif /* !DISABLE_TABS */
   Menu_add(file_menu, "_Open File...", "<ctrl>O", bw,
            a_Commands_openfile_callback, bw);
   Menu_add(file_menu, "Open _URL...", "<ctrl>L", bw,
            a_Commands_openurl_callback, bw);
   /*
   Menu_add(file_menu, "_Preferences", "<ctrl>E", bw,
            a_Commands_prefs_callback, bw);
   */
   Menu_add(file_menu, "Close Window", "<ctrl>Q", bw,
            a_Commands_close_callback, bw);
#ifndef DISABLE_TABS
   Menu_add(file_menu, "Close Tab", "<ctrl>W", bw,
            a_Commands_close_current_tab_callback, bw);
#endif /* !DISABLE_TABS */
   Menu_sep(file_menu);
   Menu_add(file_menu, "Exit Dillo", "<alt>Q", bw,
            a_Commands_exit_callback, bw);

#ifndef DISABLE_TABS
   /* VIEW MENU */
   view_menu = Menu_new(menubar, tiny? "_V" : "_View", FALSE, bw);
   Menu_add(view_menu, "Previous tab", "<ctrl>comma", bw,
            a_Commands_tab_prev_callback, bw);
   Menu_add(view_menu, "Next tab", "<ctrl>period", bw,
            a_Commands_tab_next_callback, bw);
#endif /* !DISABLE_TABS */

   /* BOOKMARKS MENU
   bookmarks_menu = Menu_new(menubar, tiny ? "_B" : "_Bookmarks", FALSE, bw);
   bw->bookmarks_menu = bookmarks_menu;
   Menu_add(bookmarks_menu, "_View Bookmarks", NULL, bw,
            a_Commands_viewbm_callback, bw);
   Menu_sep(bookmarks_menu);
   */

   /* HELP MENU
   help_menu = Menu_new(menubar, "_Help", TRUE, bw);
   Menu_add(help_menu, "Dillo _Home", NULL, bw,
            a_Commands_helphome_callback, bw);
   Menu_add(help_menu, "Dillo _Manual", NULL, bw,
            a_Commands_manual_callback, bw);
   */
   return menubar;
}

/*
 * Make a new popup menu and return it.
 */
GtkWidget *a_Menu_popup_op_new(BrowserWindow *bw)
{
   GtkWidget *menu;

   menu = gtk_menu_new();
   Menu_sep(menu);
   Menu_add(menu, "     PAGE OPTIONS", NULL, bw, NULL, NULL);
   Menu_sep(menu);
   if(prefs.show_popup_navigation) {
     bw->back_op_menu_button = Menu_add(menu, "Back", NULL, bw,
					a_Commands_back_callback, bw);
     bw->forw_op_menu_button = Menu_add(menu, "Forward", NULL, bw,
					a_Commands_forw_callback, bw);
     bw->stop_op_menu_button = Menu_add(menu, "Stop", NULL, bw,
					a_Commands_stop_callback, bw);
     Menu_add(menu, "_Reload", "<ctrl>r", bw,
	      a_Commands_reload_callback, bw);
     Menu_sep(menu);
   }
   Menu_add(menu, "View page So_urce", "<ctrl>u", bw,
            a_Commands_viewsource_callback, bw);
   Menu_add(menu, "_Bookmark this Page", "<ctrl>b", bw,
            a_Commands_addbm_callback, bw);
   Menu_sep(menu);
   Menu_add(menu, "Duplicate this Page", "<alt>n", bw,
            a_Commands_duplicate_callback, bw);
#ifndef DISABLE_TABS
   Menu_add(menu, "Duplicate this tab", "<alt>t", bw,
            a_Commands_duplicate_tab_callback, bw);
#endif /* !DISABLE_TABS */
   Menu_sep(menu);
   Menu_add(menu, "_Find Text", "<ctrl>F", bw,
            a_Commands_findtext_callback, bw);

   bw->pagemarks_menuitem = Menu_add(menu, "Jump to...", NULL, bw, NULL, NULL);
#ifndef XHTML_DTD_STRICT
   bw->frame_menuitem = Menu_add(menu, "This Frame...", NULL, bw, NULL, NULL);
   bw->frameset_menuitem = Menu_add(menu, "This Frameset...", NULL, bw, NULL, NULL);
#endif /* !XHTML_DTD_STRICT */

   Menu_sep(menu);
   Menu_add(menu, "Save Page As...", "<ctrl>s", bw,
            a_Commands_save_callback, bw);

   return menu;
}

#ifndef XHTML_DTD_STRICT
/*
 * Make a new frame menu and return it
 */
GtkWidget *a_Menu_popup_of_new(DilloDoc *dd)
{
  GtkWidget *menu;

  menu = gtk_menu_new();
  Menu_sep(menu);
  Menu_add(menu, "    FRAME OPTIONS", NULL, dd->bw, NULL, NULL);
  Menu_sep(menu);
  Menu_add(menu, "View frame Source", NULL, dd->bw,
	   a_Commands_viewsource_doc_callback, dd);
  Menu_add(menu, "Bookmark this Frame", NULL, dd->bw,
	   a_Commands_addbm_doc_callback, dd);
  Menu_add(menu, "Reload this Frame", NULL, dd->bw,
	   a_Commands_reload_doc_callback, dd);
  Menu_sep(menu);
  Menu_add(menu, "Show only this Frame", NULL, dd->bw,
	   a_Commands_showonly_doc_callback, dd);
  Menu_add(menu, "Open Frame in new Window", NULL, dd->bw,
	   a_Commands_open_link_nw_callback, dd->bw);
#ifndef DISABLE_TABS
  Menu_add(menu, "Open Frame in new Tab", NULL, dd->bw,
	   a_Commands_open_link_nw_tab_callback, dd->bw);
#endif /* !DISABLE_TABS */
  Menu_sep(menu);
  Menu_add(menu, "_Find Text in this Frame", NULL, dd->bw,
	   a_Commands_findtext_doc_callback, dd);
  Menu_sep(menu);
  Menu_add(menu, "Save Frame As...", NULL, dd->bw,
	   a_Commands_save_doc_callback, dd);
  
  return menu;
}

/*
 * Make a new frameset menu and return it
 *
 */
GtkWidget *a_Menu_popup_ofs_new(DilloDoc *dd)
{
  GtkWidget *frameset_menu;

  frameset_menu = gtk_menu_new();
  Menu_sep(frameset_menu);
  Menu_add(frameset_menu, "    FRAMESET OPTIONS", NULL, dd->bw, NULL, NULL);
  Menu_sep(frameset_menu);
  Menu_add(frameset_menu, "View Frameset Source", NULL, dd->bw,
	   a_Commands_viewsource_doc_callback, dd->parent);
  Menu_add(frameset_menu, "Bookmark this Frameset", NULL, dd->bw,
	   a_Commands_addbm_doc_callback, dd->parent);
  Menu_add(frameset_menu, "Reload this Frameset", NULL, dd->bw,
	   a_Commands_reload_doc_callback, dd->parent);
  Menu_sep(frameset_menu);
  Menu_add(frameset_menu, "Show only this Frameset", NULL, dd->bw,
	   a_Commands_showonly_doc_callback, dd->parent);
  Menu_sep(frameset_menu);
  Menu_add(frameset_menu, "_Find Text in this Frameset", NULL, dd->bw,
	   a_Commands_findtext_doc_callback, dd->parent);
  Menu_sep(frameset_menu);
  Menu_add(frameset_menu, "Save Frameset As...", NULL, dd->bw,
	   a_Commands_save_doc_callback, dd->parent);

  return frameset_menu;
}

#endif /* !XHTML_DTD_STRICT */
/*
 * Make a new popup menu for when the mouse is over a link.
 */
GtkWidget *a_Menu_popup_ol_new(BrowserWindow *bw)
{
   GtkWidget *menu;
   GtkWidget *copy;

   menu = gtk_menu_new();
   Menu_sep(menu);
   Menu_add(menu, "       LINK OPTIONS", NULL, bw, NULL, NULL);
   Menu_sep(menu);
   Menu_add(menu, "Open Link in New Window", NULL, bw,
            a_Commands_open_link_nw_callback, bw);
#ifndef DISABLE_TABS
   Menu_add(menu, "Open Link in New Tab", NULL, bw,
            a_Commands_open_link_nw_tab_callback, bw);
#endif /* !DISABLE_TABS */
   Menu_add(menu, "Open Link", NULL, bw,
            a_Commands_open_link_callback, bw);
   Menu_sep(menu);
   Menu_add(menu, "Add Bookmark for Link", NULL, bw,
            a_Commands_addbm_link_callback, bw);

   copy = Menu_add(menu, "Copy Link location", NULL, bw,
                   a_Menu_popup_select_url_callback, bw);
   a_Commands_init_selection(copy);
   gtk_signal_connect(GTK_OBJECT(copy), "selection_clear_event",
      GTK_SIGNAL_FUNC(a_Commands_clear_selection_callback), NULL);
   gtk_selection_add_target(copy, GDK_SELECTION_PRIMARY,
      GDK_SELECTION_TYPE_STRING, 1);
   gtk_signal_connect(GTK_OBJECT(copy), "selection_get",
      GTK_SIGNAL_FUNC(a_Commands_give_selection_callback), NULL);

   Menu_sep(menu);
   Menu_add(menu, "Save Link As...", NULL, bw,
            a_Commands_save_link_callback, bw);
   return menu;
}

/*
 * Make a new popup menu for when the mouse is over an image
 */
GtkWidget *a_Menu_popup_oi_new (BrowserWindow *bw)
{
   GtkWidget *menu;
   GtkWidget *copy;

   menu = gtk_menu_new();
   Menu_sep(menu);
   Menu_add(menu, "       IMAGE OPTIONS", NULL, bw, NULL, NULL);
   Menu_sep(menu);
   Menu_add(menu, "Open Image in New Window", NULL, bw,
            a_Commands_open_link_nw_callback, bw);
#ifndef DISABLE_TABS
   Menu_add(menu, "Open Image in New Tab", NULL, bw,
            a_Commands_open_link_nw_tab_callback, bw);
#endif /* !DISABLE_TABS */
   Menu_add(menu, "Open Image", NULL, bw,
           a_Commands_open_link_callback, bw);
   Menu_sep(menu);
   Menu_add(menu, "Bookmark this Image", NULL, bw,
            a_Commands_addbm_link_callback, bw);
   copy = Menu_add(menu, "Copy Image location", NULL, bw,
                   a_Menu_popup_select_url_callback, bw);
   a_Commands_init_selection(copy);
   gtk_signal_connect(GTK_OBJECT(copy), "selection_clear_event",
      GTK_SIGNAL_FUNC(a_Commands_clear_selection_callback), NULL);
   gtk_selection_add_target(copy, GDK_SELECTION_PRIMARY,
      GDK_SELECTION_TYPE_STRING, 1);
   gtk_signal_connect(GTK_OBJECT(copy), "selection_get",
      GTK_SIGNAL_FUNC(a_Commands_give_selection_callback), NULL);

   Menu_sep(menu);
   Menu_add(menu, "Save Image As...", NULL, bw,
            a_Commands_save_link_callback, bw);
   return menu;
}

/*
 * Make a new popup menu for history browsing
 * direction: {backward = -1, forward = 1}
 */
GtkWidget *a_Menu_popup_history_new(BrowserWindow *bw, gint direction)
{
   int i;
   GtkWidget *menu, *menu_item;
   GString *text = g_string_sized_new(128);

   menu = gtk_menu_new();
   Menu_sep(menu);
   if (direction>0) {
      Menu_add(menu, "     FOLLOWING PAGES     ", NULL, bw, NULL, NULL);
      i = a_Nav_stack_ptr(bw->dd) + 1;
   } else {
      Menu_add(menu, "     PREVIOUS PAGES     ", NULL, bw, NULL, NULL);
      i = a_Nav_stack_ptr(bw->dd) - 1;
   }
   Menu_sep(menu);

   for ( ; i >= 0 && i < a_Nav_stack_size(bw->dd); i+=direction ) {
      if (a_History_get_title(NAV_IDX(bw->dd,i))) {
         g_string_assign(text, a_History_get_title(NAV_IDX(bw->dd,i)));
      } else {
         g_string_assign(text, URL_STR(a_History_get_url(NAV_IDX(bw->dd,i))));
      }
      if ( text->len > 64 ) {
         g_string_truncate(text, 64);
         g_string_append(text, "...");
      }
      menu_item = Menu_add(menu, text->str, NULL, bw, NULL, NULL);
      /* attach the nav_stack index to the menu item */
      gtk_object_set_data(GTK_OBJECT (menu_item), "nav_idx",
                          GINT_TO_POINTER(i));

      gtk_signal_connect (
         GTK_OBJECT (menu_item), "select",
         GTK_SIGNAL_FUNC (a_Interface_scroll_popup), NULL);
      gtk_signal_connect (
         GTK_OBJECT (menu_item), "select",
         GTK_SIGNAL_FUNC (Menu_popup_history_select_callback), bw);
      gtk_signal_connect (
         GTK_OBJECT (menu_item), "deselect",
         GTK_SIGNAL_FUNC (Menu_popup_history_deselect_callback), bw);
      gtk_signal_connect (
         GTK_OBJECT (menu_item), "button-press-event",
         GTK_SIGNAL_FUNC (a_Commands_historypress_callback), bw);
   }

   g_string_free(text, TRUE);
   return menu;
}

#ifndef DISABLE_TABS
/*
 * Make a new popup menu for when the mouse is over a tab
 */
GtkWidget *a_Menu_popup_tab_new(DilloDoc *dd)
{
   GtkWidget *menu;

   menu = gtk_menu_new();
   Menu_sep(menu);
   Menu_add(menu, "TAB OPTIONS", NULL, dd->bw, NULL, NULL);
   Menu_sep(menu);
   Menu_add(menu, "Close Tab", NULL, dd->bw,
            a_Commands_close_tab_callback, dd);
   Menu_add(menu, "New Tab", NULL, dd->bw,
            a_Commands_new_tab_callback, dd->bw);
   Menu_add(menu, "Reload Tab", NULL, dd->bw,
            a_Commands_reload_doc_callback, dd);
   return menu;
}
#endif /* !DISABLE_TABS */

/*
 * Set the popup's DilloUrl
 */
void a_Menu_popup_set_url(BrowserWindow *bw, DilloUrl *url)
{
   if (bw->menu_popup.url)
      a_Url_free(bw->menu_popup.url);
   bw->menu_popup.url = a_Url_dup(url);
}

/*
 * Get the popup's DilloUrl
 */
DilloUrl *a_Menu_popup_get_url(BrowserWindow *bw)
{
   return bw->menu_popup.url;
}


