/*
 * File: interface.c
 *
 * Copyright (C) 1997 Raph Levien <raph@acm.org>
 * Copyright (C) 1999 Sammy Mannaert <nstalkie@tvd.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <config.h>

#include <stdio.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>

#include "list.h"
#include "misc.h"
#include "dillo.h"
#include "history.h"
#include "nav.h"
#include "doc.h"
#include "IO/Url.h"
#include "IO/IO.h"
#include "interface.h"
#ifndef DISABLE_TABS
#include "tab.h"
#endif /* !DISABLE_TABS */
#include "commands.h"
#include "menu.h"
#include "prefs.h"
#include "url.h"
#include "capi.h"

#include "dw_widget.h"
#include "dw_gtk_scrolled_window.h"
#include "dw_gtk_viewport.h"
#include "dw_gtk_statuslabel.h"
#include "dw_container.h"
#include "progressbar.h"

#include "pixmaps.h"
#include <gdk/gdkkeysyms.h>

#define DEBUG_LEVEL 0
#include "debug.h"


/*
 * Local Data
 */
/* BrowserWindow holds all the widgets (and perhaps more)
 * for each new_browser.*/
static BrowserWindow **browser_window;
static gint num_bw, num_bw_max;

/* We need only one of them. */
static GtkTooltips *tooltips = NULL;


/*
 * Initialize global data
 */
void a_Interface_init(void)
{
   num_bw = 0;
   num_bw_max = 16;
   browser_window = NULL;
   tooltips = gtk_tooltips_new ();
}

/*=== Browser Window Interface Updating =====================================*/
/*
 * Set the sensitivity on back/forw buttons and menu entries.
 */
static gint Interface_sens_idle_func(BrowserWindow *bw)
{
   gboolean back_sensitive, forw_sensitive, stop_sensitive;

   /* Stop button */
   stop_sensitive = (bw->dd->NumRootClients > 0);
   gtk_widget_set_sensitive(bw->stop_button, stop_sensitive);


   /* Back and Forward buttons */
   back_sensitive = a_Nav_stack_ptr(bw->dd) > 0;
   gtk_widget_set_sensitive(bw->back_button, back_sensitive);
   forw_sensitive = (a_Nav_stack_ptr(bw->dd) < a_Nav_stack_size(bw->dd) - 1 &&
                     !bw->dd->nav_expecting);
   gtk_widget_set_sensitive(bw->forw_button, forw_sensitive);

   if(prefs.show_popup_navigation) {
     /* adjust popup menu buttons */
     gtk_widget_set_sensitive(bw->stop_op_menu_button, stop_sensitive);
     gtk_widget_set_sensitive(bw->back_op_menu_button, back_sensitive);
     gtk_widget_set_sensitive(bw->forw_op_menu_button, forw_sensitive);
   }

   bw->sens_idle_tag = 0;
   return FALSE;
}

/*
 * Set the sensitivity on back/forw buttons and menu entries.
 */
void a_Interface_set_button_sens(BrowserWindow *bw)
{
   if (bw->sens_idle_tag != 0)
      return;
   bw->sens_idle_tag = gtk_idle_add(
                          (GtkFunction)Interface_sens_idle_func, bw);
}

/*
 * Remove a single browser window. This includes all its open documents,
 * freeing all resources associated with them, and exiting gtk
 * if no browser windows are left.
 */
static gboolean Interface_quit(GtkWidget *widget, BrowserWindow *bw)
{
   gint i;

   /* destroy the container. It should take all its contents with
    * it, relying on callbacks to destroy non-widget data
    */
   gtk_widget_destroy(GTK_WIDGET(bw->container));

   g_slist_free(bw->PanelHandles);

   if (bw->open_dialog_window != NULL)
      gtk_widget_destroy(bw->open_dialog_window);
   if (bw->openfile_dialog_window != NULL)
      gtk_widget_destroy(bw->openfile_dialog_window);
   if (bw->quit_dialog_window != NULL)
      gtk_widget_destroy(bw->quit_dialog_window);
   if (bw->findtext_dialog_window != NULL)
      gtk_widget_destroy(bw->findtext_dialog_window);
   if (bw->question_dialog_window != NULL)
      gtk_widget_destroy(bw->question_dialog_window);

   if (bw->menu_popup.over_back)
      gtk_widget_destroy(bw->menu_popup.over_back);
   if (bw->menu_popup.over_forw)
      gtk_widget_destroy(bw->menu_popup.over_forw);
#ifndef DISABLE_TABS
   if (bw->menu_popup.over_tab)
      gtk_widget_destroy(bw->menu_popup.over_tab);
#endif /* !DISABLE_TABS */
   if (bw->menu_popup.url)
      a_Url_free(bw->menu_popup.url);
  
   if (bw->sens_idle_tag)
      gtk_idle_remove(bw->sens_idle_tag);


   for (i = 0; i < num_bw; i++)
      if (browser_window[i] == bw) {
         browser_window[i] = browser_window[--num_bw];
         break;
      }

   g_free(bw);

   if (num_bw == 0)
      gtk_main_quit();

   return FALSE;
}


/*=== Browser Window Interface Construction =================================*/
/*
 * Clear a text entry
 */
static void Interface_entry_clear(GtkEntry *entry)
{
   gtk_entry_set_text(GTK_ENTRY (entry), "");
   gtk_widget_grab_focus(GTK_WIDGET(entry));
}

/*
 * Create a pixmap and return it.
 */
static GtkWidget *Interface_pixmap_new(GtkWidget *parent, gchar **data)
{
   GtkWidget *pixmapwid;
   GdkPixmap *pixmap;
   GdkBitmap *mask;
   GtkStyle *style;

   style = gtk_widget_get_style(parent);

   pixmap = gdk_pixmap_create_from_xpm_d(parent->window, &mask,
                                         &style->bg[GTK_STATE_NORMAL], data);

   pixmapwid = gtk_pixmap_new(pixmap, mask);

   return (pixmapwid);
}

/*
 * Connect button's "clicked" event with (key, key_mod) pair.
 */
static void Interface_set_button_accel(GtkButton *button,
                                       gint key,
                                       gint key_mod,
                                       GtkAccelGroup *accel_group)
{
   gtk_widget_add_accelerator(GTK_WIDGET(button), "clicked", accel_group,
                              key, key_mod, GTK_ACCEL_LOCKED);
}

/*
 * Create the "NEW" button and its location-entry.
 */
static GtkWidget *Interface_locbar_new(BrowserWindow *bw)
{
   GtkWidget *hbox, *toolbar;

   hbox = gtk_hbox_new(FALSE, 0);

   /* location entry */
   bw->location = gtk_entry_new();
   gtk_signal_connect(GTK_OBJECT(bw->location), "activate",
                      (GtkSignalFunc) a_Interface_entry_open_url, bw);
   toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
   gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar), GTK_RELIEF_NONE);
   GTK_WIDGET_UNSET_FLAGS (toolbar, GTK_CAN_FOCUS);

   bw->clear_url_button = gtk_toolbar_append_item(
                             GTK_TOOLBAR(toolbar),
                             NULL, "Clear the url-box!", "Toolbar/New",
                             Interface_pixmap_new(bw->main_window, s_new_xpm),
                             NULL, NULL);
   gtk_signal_connect_object(GTK_OBJECT(bw->clear_url_button), "clicked",
                             GTK_SIGNAL_FUNC (Interface_entry_clear),
                             GTK_OBJECT(bw->location));

   gtk_box_pack_start(GTK_BOX(hbox), toolbar, FALSE, FALSE, 0);
   gtk_widget_show(toolbar);
   gtk_box_pack_start(GTK_BOX(hbox), bw->location, TRUE, TRUE, 0);
   gtk_widget_show(bw->location);
   gtk_widget_show(hbox);
   return (hbox);
}

/*
 * Create a new toolbar (Back, Forward, Home, Reload, Save and Stop buttons)
 */
static GtkWidget *Interface_toolbar_new(BrowserWindow *bw, gint label)
{
   GtkWidget *toolbar;
   gboolean s = prefs.small_icons;

   toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
   gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar), GTK_RELIEF_NONE);

   /* back button */
   bw->back_button = gtk_toolbar_append_item(
                        GTK_TOOLBAR(toolbar),
                        label ? "Back" : NULL,
                        "Go to previous page", "Toolbar/Back",
                        Interface_pixmap_new(bw->main_window,
                                             s ? s_left_xpm : left_xpm),
                        (GtkSignalFunc) a_Commands_back_callback, bw);
   gtk_widget_set_sensitive(bw->back_button, FALSE);
   Interface_set_button_accel(GTK_BUTTON(bw->back_button), GDK_comma,
                              0, bw->accel_group);
   gtk_signal_connect(GTK_OBJECT(bw->back_button), "button-press-event",
                      GTK_SIGNAL_FUNC(a_Commands_navpress_callback), bw);

   /* forward button */
   bw->forw_button = gtk_toolbar_append_item(
                        GTK_TOOLBAR(toolbar),
                        label ? "Forward" : NULL,
                        "Go to next page", "Toolbar/Forward",
                        Interface_pixmap_new(bw->main_window,
                                             s ? s_right_xpm : right_xpm),
                        (GtkSignalFunc) a_Commands_forw_callback, bw);
   gtk_widget_set_sensitive(bw->forw_button, FALSE);
   Interface_set_button_accel(GTK_BUTTON(bw->forw_button), GDK_period,
                              0, bw->accel_group);
   gtk_signal_connect(GTK_OBJECT(bw->forw_button), "button-press-event",
                      GTK_SIGNAL_FUNC(a_Commands_navpress_callback), bw);

   /* home button */
   bw->home_button = gtk_toolbar_append_item(
                        GTK_TOOLBAR(toolbar),
                        label ? "Home" : NULL,
                        "Go to the Home page", "Toolbar/Home",
                        Interface_pixmap_new(bw->main_window,
                                             s ? s_home_xpm : home_xpm),
                        (GtkSignalFunc) a_Commands_home_callback, bw);
   gtk_signal_connect(GTK_OBJECT(bw->home_button), "button-press-event",
                      GTK_SIGNAL_FUNC(a_Commands_navpress_callback), bw);

   /* reload button */
   bw->reload_button = gtk_toolbar_append_item(
                          GTK_TOOLBAR(toolbar),
                          label ? "Reload" : NULL,
                          "Reload this page", "Toolbar/Reload",
                          Interface_pixmap_new(bw->main_window,
                                               s ? s_reload_xpm : reload_xpm),
                          (GtkSignalFunc) a_Commands_reload_callback, bw);

   /* save button */
   bw->save_button = gtk_toolbar_append_item(
                        GTK_TOOLBAR(toolbar),
                        label ? "Save" : NULL,
                        "Save this page", "Toolbar/Save",
                        Interface_pixmap_new(bw->main_window,
                                             s ? s_save_xpm : save_xpm),
                        (GtkSignalFunc) a_Commands_save_callback, bw);
   /* stop button */
   bw->stop_button = gtk_toolbar_append_item(
                        GTK_TOOLBAR(toolbar),
                        label ? "Stop" : NULL,
                        "Stop the current transfer", "Toolbar/Stop",
                        Interface_pixmap_new(bw->main_window,
                                             s ? s_stop_xpm : stop_xpm),
                        (GtkSignalFunc) a_Commands_stop_callback, bw);
   gtk_widget_set_sensitive(bw->stop_button, FALSE);

   /* bookmarks button */
   bw->bookmarks_button = gtk_toolbar_append_item(
                             GTK_TOOLBAR(toolbar),
                             label ? "Book" : NULL,
                             "View bookmarks", "Toolbar/Bookmarks",
                             Interface_pixmap_new(bw->main_window,
                                                  s ? s_bm_xpm : bm_xpm),
                             (GtkSignalFunc) a_Commands_viewbm_callback, bw);
   gtk_widget_set_sensitive(bw->bookmarks_button, TRUE);
   Interface_set_button_accel(GTK_BUTTON(bw->bookmarks_button), GDK_semicolon,
                              0, bw->accel_group);
   Interface_set_button_accel(GTK_BUTTON(bw->bookmarks_button), GDK_B,
                              GDK_MOD1_MASK, bw->accel_group);
   gtk_signal_connect(GTK_OBJECT(bw->bookmarks_button), "button-press-event",
                      GTK_SIGNAL_FUNC(a_Commands_navpress_callback), bw);

   gtk_widget_show(toolbar);
   return toolbar;
}

/*
 * Create the progressbar's box
 */
static GtkWidget *Interface_progressbox_new(BrowserWindow *bw, gint vertical)
{
   GtkWidget *progbox;

   progbox = vertical ? gtk_vbox_new(FALSE, 0) : gtk_hbox_new(FALSE, 0);
   bw->progress_box = progbox;
   bw->imgprogress = a_Progressbar_new();
   bw->progress = a_Progressbar_new();
   gtk_box_pack_start(GTK_BOX(progbox), bw->imgprogress, TRUE, TRUE, 0);
   gtk_widget_show(bw->imgprogress);
   gtk_box_pack_start(GTK_BOX(progbox), bw->progress, TRUE, TRUE, 0);
   gtk_widget_show(bw->progress);
   return (progbox);
}

/*
 * Hide/Unhide this bw's control panels.
 * toggle: Flag [toggle or set].
 */
void a_Interface_toggle_panel(BrowserWindow *bw, gboolean toggle)
{
   if (toggle)
      bw->fullwindow = !bw->fullwindow;

   if (bw->fullwindow) {
      g_slist_foreach(bw->PanelHandles, (GFunc)gtk_widget_hide, NULL);
      gtk_widget_hide(bw->status_box);
      gtk_widget_show (bw->full_screen_off_button);
   } else {
      g_slist_foreach(bw->PanelHandles, (GFunc)gtk_widget_show, NULL);
      gtk_widget_show(bw->status_box);
      gtk_widget_hide (bw->full_screen_off_button);
   }
#ifndef DISABLE_TABS
   a_Tab_visibility_update(bw);
#endif /* !DISABLE_TABS */

}

/*
 * Customize the appearance of the bw.
 */
static void Interface_browser_window_customize(BrowserWindow *bw)
{
   if ( !prefs.show_back )
      gtk_widget_hide(bw->back_button);
   if ( !prefs.show_forw )
      gtk_widget_hide(bw->forw_button);
   if ( !prefs.show_home )
      gtk_widget_hide(bw->home_button);
   if ( !prefs.show_reload )
      gtk_widget_hide(bw->reload_button);
   if ( !prefs.show_save )
      gtk_widget_hide(bw->save_button);
   if ( !prefs.show_stop )
      gtk_widget_hide(bw->stop_button);
   if ( !prefs.show_bookmarks )
      gtk_widget_hide(bw->bookmarks_button);
   if ( !prefs.show_menubar )
      gtk_widget_hide(bw->menubar);
   if ( !prefs.show_clear_url)
      gtk_widget_hide(bw->clear_url_button);
   if ( !prefs.show_url )
      gtk_widget_hide(bw->location);
   if ( !prefs.show_progress_box )
      gtk_widget_hide(bw->progress_box);

   bw->fullwindow = prefs.fullwindow_start;
   a_Interface_toggle_panel(bw, FALSE);
}

/*
 * Add the button for switching into fullscreen mode
 */
static void Interface_add_full_screen_button (BrowserWindow *bw, GtkBox *box)
{
   /* The button is put into a vbox, so that it is not enlarged vertically. */
   GtkWidget *vbox, *dummy, *button, *pixmap;

   vbox = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (box), vbox, FALSE, FALSE, 0);
   gtk_widget_show (vbox);

   /* The dummy will make the button align at the bottom.
    * (Important only when using large text fonts.) */
   dummy = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (vbox), dummy, TRUE, TRUE, 0);
   gtk_widget_show (dummy);

   button = gtk_button_new ();
   gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
   gtk_tooltips_set_tip (tooltips, button, "Hide Controls", "Show Controls");
   GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
   gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
   gtk_widget_show (button);

   pixmap = Interface_pixmap_new(bw->main_window, full_screen_on_xpm);
   gtk_container_add (GTK_CONTAINER (button), pixmap);
   gtk_widget_show (pixmap);

   gtk_signal_connect_object
      (GTK_OBJECT (button), "clicked",
       GTK_SIGNAL_FUNC (a_Commands_full_screen_callback), (gpointer)bw);
}

/*******************
 * browser windows *
 *******************/

/*
 * Create a new browser window and return it.
 * (the new window is stored in browser_window[])
 */
BrowserWindow *
a_Interface_browser_window_new(gint width, gint height, guint32 xid)
{
   GtkWidget *box1, *hbox,
             *progbox, *toolbar, *handlebox, *menubar, *locbox, *pixmap;
#ifdef XHTML_DTD_FRAMESET
#ifdef DISABLE_TABS
   GtkWidget *box2;
#endif /* DISABLE_TABS */
#endif /* XHTML_DTD_FRAMESET */
   BrowserWindow *bw;
   char buf[64];

   /* We use g_new0() to zero the memory */
   bw = g_new0(BrowserWindow, 1);
   a_List_add(browser_window, num_bw, num_bw_max);
   browser_window[num_bw++] = bw;

   if (!xid)
       bw->main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   else
       bw->main_window = gtk_plug_new(xid);

   /* set tooltips.
    * TODO: this is a global, change to browser-window local?
    */
   bw->tooltips = tooltips;
   gtk_tooltips_enable(bw->tooltips);

   gtk_window_set_policy(GTK_WINDOW(bw->main_window), TRUE, TRUE, FALSE);
   gtk_signal_connect(GTK_OBJECT(bw->main_window), "delete_event",
                      GTK_SIGNAL_FUNC(gtk_object_destroy), bw);
   gtk_signal_connect(GTK_OBJECT(bw->main_window), "destroy",
                      GTK_SIGNAL_FUNC(Interface_quit), bw);
   gtk_container_border_width(GTK_CONTAINER(bw->main_window), 0);

   gtk_window_set_wmclass(GTK_WINDOW(bw->main_window), "dillo", "Dillo");

   gtk_widget_set_usize(bw->main_window, width, height);

   /* -RL :: I must realize the window to see it correctly */
   gtk_widget_realize(bw->main_window);

   /* Create and attach an accel group to the main window */
   bw->accel_group = gtk_accel_group_new();
   gtk_window_add_accel_group(GTK_WINDOW(bw->main_window), bw->accel_group);

   box1 = gtk_vbox_new(FALSE, 0);

   /* setup the control panel */
   if (prefs.panel_size == 1) {
      handlebox = gtk_handle_box_new();
      bw->PanelHandles = g_slist_append(bw->PanelHandles, handlebox);
      hbox = gtk_hbox_new(FALSE, 0);
      /* Control Buttons */
      toolbar = Interface_toolbar_new(bw, 0);
      /* Menus */
      menubar = a_Menu_mainbar_new(bw, 1);
      /* Location entry */
      locbox = Interface_locbar_new(bw);
      /* progress bars */
      progbox = Interface_progressbox_new(bw, 0);

      gtk_box_pack_start(GTK_BOX(hbox), toolbar, FALSE, FALSE, 0);
      gtk_widget_show(toolbar);
      gtk_box_pack_start(GTK_BOX(hbox), menubar, FALSE, FALSE, 0);
      gtk_widget_show(menubar);
      gtk_box_pack_start(GTK_BOX(hbox), locbox, TRUE, TRUE, 0);
      gtk_widget_show(locbox);
      gtk_box_pack_start(GTK_BOX(hbox), progbox, FALSE, FALSE, 0);
      gtk_widget_show(progbox);
      gtk_container_add(GTK_CONTAINER(handlebox), hbox);
      gtk_widget_show(hbox);
      gtk_box_pack_start(GTK_BOX(box1), handlebox, FALSE, FALSE, 0);
      gtk_widget_show(handlebox);

   } else if (prefs.panel_size == 2) {
      handlebox = gtk_handle_box_new();
      bw->PanelHandles = g_slist_append(bw->PanelHandles, handlebox);
      hbox = gtk_hbox_new(FALSE, 0);
      menubar = a_Menu_mainbar_new(bw, 0);
      locbox = Interface_locbar_new(bw);
      gtk_box_pack_start(GTK_BOX(hbox), menubar, FALSE, FALSE, 0);
      gtk_widget_show(menubar);
      gtk_box_pack_start(GTK_BOX(hbox), locbox, TRUE, TRUE, 0);
      gtk_widget_show(locbox);
      gtk_container_add(GTK_CONTAINER(handlebox), hbox);
      gtk_widget_show(hbox);
      gtk_box_pack_start(GTK_BOX(box1), handlebox, FALSE, FALSE, 0);
      gtk_widget_show(handlebox);

      handlebox = gtk_handle_box_new();
      bw->PanelHandles = g_slist_append(bw->PanelHandles, handlebox);
      gtk_container_border_width(GTK_CONTAINER(handlebox), 4);
      hbox = gtk_hbox_new(FALSE, 0);
      toolbar = Interface_toolbar_new(bw, 1);
      progbox = Interface_progressbox_new(bw, 1);
      gtk_box_pack_start(GTK_BOX(hbox), toolbar, TRUE, TRUE, 0);
      gtk_widget_show(toolbar);
      gtk_box_pack_start(GTK_BOX(hbox), progbox, FALSE, FALSE, 0);
      gtk_widget_show(progbox);
      gtk_container_add(GTK_CONTAINER(handlebox), hbox);
      gtk_widget_show(hbox);
      gtk_box_pack_start(GTK_BOX(box1), handlebox, FALSE, FALSE, 0);
      gtk_widget_show(handlebox);

   } else {
      handlebox = gtk_handle_box_new();
      bw->PanelHandles = g_slist_append(bw->PanelHandles, handlebox);
      menubar = a_Menu_mainbar_new(bw, 0);
      gtk_container_add(GTK_CONTAINER(handlebox), menubar);
      gtk_widget_show(menubar);
      gtk_box_pack_start(GTK_BOX(box1), handlebox, FALSE, FALSE, 0);
      gtk_widget_show(handlebox);

      handlebox = gtk_handle_box_new();
      bw->PanelHandles = g_slist_append(bw->PanelHandles, handlebox);
      gtk_container_border_width(GTK_CONTAINER(handlebox), 4);
      hbox = gtk_hbox_new(FALSE, 0);
      toolbar = Interface_toolbar_new(bw, 1);
      progbox = Interface_progressbox_new(bw, 1);
      gtk_box_pack_start(GTK_BOX(hbox), toolbar, TRUE, TRUE, 0);
      gtk_widget_show(toolbar);
      gtk_box_pack_start(GTK_BOX(hbox), progbox, FALSE, FALSE, 0);
      gtk_widget_show(progbox);
      gtk_container_add(GTK_CONTAINER(handlebox), hbox);
      gtk_widget_show(hbox);
      gtk_box_pack_start(GTK_BOX(box1), handlebox, FALSE, FALSE, 0);
      gtk_widget_show(handlebox);

      handlebox = gtk_handle_box_new();
      bw->PanelHandles = g_slist_append(bw->PanelHandles, handlebox);
      locbox = Interface_locbar_new(bw);
      gtk_container_add(GTK_CONTAINER(handlebox), locbox);
      gtk_widget_show(locbox);
      gtk_box_pack_start(GTK_BOX(box1), handlebox, FALSE, FALSE, 0);
      gtk_widget_show(handlebox);
   }


   /* Add box1 */
   gtk_container_add(GTK_CONTAINER(bw->main_window), box1);

   /* Create the full_screen_off_button. This button is shared between all
    * documents in a tabbed browser. On switching tabs it is reparented
    * to the new current tab */
   bw->full_screen_off_button = gtk_button_new ();
   gtk_button_set_relief (GTK_BUTTON (bw->full_screen_off_button),
                          GTK_RELIEF_NONE);
   gtk_tooltips_set_tip (tooltips, bw->full_screen_off_button,
                         "Show Controls", "Hide Controls");
   GTK_WIDGET_UNSET_FLAGS (bw->full_screen_off_button, GTK_CAN_FOCUS);

   pixmap = Interface_pixmap_new(bw->main_window, full_screen_off_xpm);
   gtk_container_add (GTK_CONTAINER (bw->full_screen_off_button), pixmap);
   gtk_widget_show (pixmap);

   gtk_signal_connect_object
      (GTK_OBJECT (bw->full_screen_off_button), "clicked",
       GTK_SIGNAL_FUNC (a_Commands_full_screen_callback), (gpointer)bw);

#ifndef DISABLE_TABS
   /* add tab browser */
   bw->container = a_Tab_browser_new(bw);
   /* Add initial tab*/
   a_Tab_new(bw);
#else
   /* add document container */
   bw->dd = a_Doc_new();
   a_Doc_set_browserwindow(bw->dd, bw);
#ifdef XHTML_DTD_FRAMESET
   /* add box2, which will contain the docwin and the frameset.
    * Usually, only one of these is visible at any time */
   box2 = gtk_hbox_new(TRUE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(box2), 0);
   gtk_box_pack_start(GTK_BOX(box2), GTK_WIDGET(bw->dd->docwin), TRUE, TRUE, 0);
   /* as this is a new document, the frameset will be empty so there
    * is no need to pack it in the box */
   gtk_widget_show(box2);

   /* point bw->container at the box containing dd's docwin. 
    * This ensures that the docwin (and, through its callback,
    * the associated DilloDoc) is destroyed when the window is closed */
   bw->container = GTK_CONTAINER(box2);
#else
   /* point bw->container at the dd's docwin. This ensures that the
    * docwin (and, through its callback, the associated DilloDoc)
    * is destroyed when the window is closed */
   bw->container = GTK_CONTAINER(bw->dd->docwin);
#endif /* XHTML_DTD_FRAMESET */
   /* set focus to location bar */
   if(prefs.focus_location_on_new)
     gtk_widget_grab_focus(bw->location);
   else
     gtk_widget_grab_focus(GTK_BIN(bw->dd->docwin)->child);
   /* Full screen mode via double click is done in two ways: First,
    * a feature of the selection is used, since in complex pages,
    * getting events back to the viewport is quite difficult. Second,
    * a simple callback, called e.g. when viewing image resources. */
   a_Selection_set_dclick_callback(
      GTK_DW_VIEWPORT(GTK_BIN(GTK_BIN(bw->dd->docwin)->child)->child)->selection,
      (void(*)(gpointer))a_Commands_full_screen_callback, bw);

   gtk_signal_connect_object_after(GTK_OBJECT(GTK_BIN(bw->dd->docwin)->child),
                                   "button_press_event",
                                   GTK_SIGNAL_FUNC(a_Commands_click_callback),
                                   (gpointer)bw);

   a_Dw_gtk_scrolled_window_add_gadget(GTK_DW_SCROLLED_WINDOW (bw->dd->docwin),
                                       bw->full_screen_off_button);
   
   /* Catch key_press event */
   gtk_signal_connect(GTK_OBJECT(GTK_BIN(bw->dd->docwin)->child),
                      "key_press_event",
                      GTK_SIGNAL_FUNC(a_Commands_key_press_handler), bw);
#endif /* !DISABLE_TABS */
   gtk_box_pack_start(GTK_BOX(box1), GTK_WIDGET(bw->container), TRUE, TRUE, 0);
   gtk_widget_show(GTK_WIDGET(bw->container));

   /* status widget */
   bw->status = a_Dw_gtk_statuslabel_new("");
   gtk_misc_set_alignment(GTK_MISC(bw->status), 0.0, 0.5);
   bw->status_box = gtk_hbox_new(FALSE, 0);

   gtk_box_pack_start(GTK_BOX(bw->status_box), bw->status, TRUE, TRUE, 2);
   gtk_widget_show(bw->status);

   Interface_add_full_screen_button (bw, GTK_BOX (bw->status_box));

   gtk_box_pack_start(GTK_BOX(box1), bw->status_box, FALSE, FALSE, 0);
   gtk_widget_show(bw->status_box);

   gtk_widget_show(bw->main_window);
   gtk_widget_show(box1);

   /* initialize the rest of the bw's data. */
   bw->open_dialog_window = NULL;
   bw->open_dialog_entry = NULL;
   bw->openfile_dialog_window = NULL;
   bw->quit_dialog_window = NULL;
   bw->save_dialog_window = NULL;
   bw->save_link_dialog_window = NULL;
   bw->findtext_dialog_window = NULL;
   bw->findtext_dialog_check = NULL;
   bw->findtext_dialog_entry = NULL;
   bw->question_dialog_window = NULL;
   bw->question_dialog_data = NULL;
   bw->viewsource_window = NULL;

   bw->pagemarks_menuitem = NULL;
#ifdef XHTML_DTD_FRAMESET
   bw->frame_menuitem = NULL;
   bw->frameset_menuitem = NULL;
   bw->menu_popup.over_frame = NULL;
#endif /* XHTML_DTD_FRAMESET */
   bw->back_op_menu_button = NULL;
   bw->forw_op_menu_button = NULL;
   bw->stop_op_menu_button = NULL;

   bw->menu_popup.over_page = a_Menu_popup_op_new(bw);
   bw->menu_popup.over_link = a_Menu_popup_ol_new(bw);
   bw->menu_popup.over_image = a_Menu_popup_oi_new(bw);
   bw->menu_popup.over_back = NULL;
   bw->menu_popup.over_forw = NULL;

#ifndef DISABLE_TABS
   bw->menu_popup.over_tab = NULL;
#endif /* !DISABLE_TABS */
   bw->menu_popup.url = NULL;

   bw->found = FALSE;
   bw->dd_list = NULL;
   bw->dd_list_index = 0;

   bw->sens_idle_tag = 0;

   /* set document (and window) title */
   g_snprintf(buf, 64, "Version %s", VERSION);
   a_Doc_title_set(bw->dd, buf);

   /* now that the bw is made, let's customize it.. */
   Interface_browser_window_customize(bw);

   return bw;
}



/*
 * Set the title of the browser window to start with "Dillo: "
 * prepended to it.
 */
void a_Interface_set_page_title(BrowserWindow *bw, char *title)
{
   GString *buf;

   g_return_if_fail (bw != NULL && title != NULL);

   buf = g_string_new("");
   g_string_sprintfa(buf, "Dillo: %s", title);
   gtk_window_set_title(GTK_WINDOW(bw->main_window), buf->str);
   g_string_free(buf, TRUE);
}

/*
 * Set location entry's text
 */
void a_Interface_set_location_text(BrowserWindow *bw, char *text)
{
   gtk_entry_set_text(GTK_ENTRY(bw->location), text);
}

/*
 * Get location entry's text
 */
gchar *a_Interface_get_location_text(BrowserWindow *bw)
{
   return gtk_entry_get_text(GTK_ENTRY(bw->location));
}

/*
 * Reset images and text progress bars
 */
void a_Interface_reset_progress_bars(BrowserWindow *bw)
{
   a_Progressbar_update(bw->progress, "", 0);
   a_Progressbar_update(bw->imgprogress, "", 0);
}

/*
 * Set the status string on the bottom of the dillo window.
 */
void a_Interface_msg(BrowserWindow *bw, const char *format, ... )
{
   static char msg[1024];
   va_list argp;

   if ( bw ) {
      va_start(argp, format);
      vsnprintf(msg, 1024, format, argp);
      va_end(argp);
      gtk_label_set(GTK_LABEL(bw->status), msg);
      bw->status_is_link = 0;
   }
}

/*
 * Called from `destroy' callback in Interface_make_*_dialog
 */
static void Interface_destroy_window(GtkWidget *widget, GtkWidget **window)
{
   gtk_widget_destroy(*window);
   *window = NULL;
}


/*
 * Close and free every single browser_window (called at exit time)
 */
void a_Interface_quit_all(void)
{
   BrowserWindow **bws;
   gint i, n_bw;

   n_bw = num_bw;
   bws = g_malloc(sizeof(BrowserWindow *) * n_bw);

   /* we copy into a new list because destroying the main window can
    * modify the browser_window array. */
   for (i = 0; i < n_bw; i++)
      bws[i] = browser_window[i];

   for (i = 0; i < n_bw; i++)
      gtk_widget_destroy(bws[i]->main_window);

   g_free(bws);
}

/*
 * Make a dialog for choosing files (by calling
 * gtk_file_selection_*() calls)
 * This can be used for saving, opening, or whatever,
 * just set the correct callbacks
 */
static void
 Interface_make_choose_file_dialog(GtkWidget **DialogWindow,
                                   char *WmName, char *WmClass, char *WTitle,
                                   GtkSignalFunc B1CallBack, void *B1CbData)
{
   GtkAccelGroup *accel_group;

   *DialogWindow = gtk_file_selection_new(WTitle);
   gtk_window_set_modal(GTK_WINDOW(*DialogWindow), FALSE);
   gtk_window_set_wmclass(GTK_WINDOW(*DialogWindow), WmName, WmClass);

   gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(*DialogWindow));
   gtk_signal_connect(
      GTK_OBJECT(*DialogWindow),
      "destroy", (GtkSignalFunc) Interface_destroy_window, DialogWindow);
   gtk_signal_connect(
      GTK_OBJECT(GTK_FILE_SELECTION(*DialogWindow)->ok_button),
      "clicked", (GtkSignalFunc) B1CallBack, B1CbData);
   gtk_signal_connect(
      GTK_OBJECT(GTK_FILE_SELECTION (*DialogWindow)->cancel_button),
      "clicked", (GtkSignalFunc) Interface_destroy_window, DialogWindow);

   /* Make GDK_Escape close the dialog */
   accel_group = gtk_accel_group_new();
   gtk_window_add_accel_group(GTK_WINDOW(*DialogWindow), accel_group);
   gtk_widget_add_accelerator(
      GTK_FILE_SELECTION(*DialogWindow)->cancel_button, "clicked",
      accel_group, GDK_Escape, 0, GTK_ACCEL_LOCKED);
}

/*
 * Get the file URL from the widget and push it to the browser window.
 */
static void
 Interface_openfile_ok_callback(GtkWidget *widget, BrowserWindow *bw)
{
   char *fn, *Cfn, *s;
   DilloUrl *url;
   GString *UrlStr = g_string_sized_new(1024);

   fn = gtk_file_selection_get_filename(
           GTK_FILE_SELECTION(bw->openfile_dialog_window));

   Cfn = (s = a_Misc_escape_chars(fn, "% ")) ? s : fn;
   g_string_sprintf(UrlStr, "file:%s", Cfn);
   url = a_Url_new(UrlStr->str, NULL, 0, 0);
   a_Nav_push(bw->dd, url);
   a_Url_free(url);
   g_string_free(UrlStr, TRUE);
   g_free(s);

   gtk_widget_destroy(bw->openfile_dialog_window);
}

/*
 * Open an URL specified in the location entry, or in the open URL dialog.
 * The URL is not sent "as is", illegal chars are ripped out,
 * then it's fully parsed by a_Url_new().
 */
void a_Interface_entry_open_url(GtkWidget *widget, BrowserWindow *bw)
{
   gchar *text, *new_text;
   DilloUrl *url;
   GtkEntry *entry;

   /* entry = { bw->location | bw->open_dialog_entry } */
   entry = GTK_ENTRY(widget == bw->location ? widget : bw->open_dialog_entry);
   text = gtk_entry_get_text(entry);

   DEBUG_MSG(1, "entry_open_url %s\n", text);

   if ( text && *text ) {
      /* Filter URL string */
      new_text = a_Url_string_strip_delimiters(text);

      url = a_Url_new(new_text, NULL, 0, 0);
      if (url) {
         a_Nav_push(bw->dd, url);
         a_Url_free(url);
      }
      g_free(new_text);
   }

   if (bw->open_dialog_window != NULL)
      gtk_widget_hide(bw->open_dialog_window);
}

/*
 * Create and show the "Open File" dialog
 */
void a_Interface_openfile_dialog(BrowserWindow *bw)
{
   if (!bw->openfile_dialog_window) {
      Interface_make_choose_file_dialog(
         &(bw->openfile_dialog_window),
         "openfile_dialog", "Dillo", "Dillo: Open File",
         (GtkSignalFunc) Interface_openfile_ok_callback, (void *)bw);
   }

   if (!GTK_WIDGET_VISIBLE(bw->openfile_dialog_window))
      gtk_widget_show(bw->openfile_dialog_window);
   else
      gdk_window_raise(bw->openfile_dialog_window->window);
}

/*
 * Make a dialog interface with three buttons, a text entry, and an optional
 * check button.
 */
static void
 Interface_make_dialog(GtkWidget **DialogWindow, char *WmName, char *WmClass,
   char *WTitle, GtkWidget **DialogEntry, char *EntryStr,
   GtkWidget **CheckButton, char *CheckButtonText,
   char *B1Label, GtkSignalFunc B1CallBack, void *B1CbData)
{
   GtkWidget *button, *box1, *box2, *entry;
   GtkAccelGroup *accel_group;

   *DialogWindow = gtk_window_new(GTK_WINDOW_DIALOG);
   gtk_window_set_wmclass(GTK_WINDOW(*DialogWindow), WmName, WmClass);
   gtk_window_set_position(GTK_WINDOW(*DialogWindow), GTK_WIN_POS_CENTER);
   gtk_window_set_title(GTK_WINDOW(*DialogWindow), WTitle);
   gtk_signal_connect(GTK_OBJECT(*DialogWindow), "destroy",
                      (GtkSignalFunc) Interface_destroy_window, DialogWindow);

   /* Create and attach an accel group to the dialog window */
   accel_group = gtk_accel_group_new();
   gtk_window_add_accel_group(GTK_WINDOW(*DialogWindow), accel_group);

   gtk_container_border_width(GTK_CONTAINER(*DialogWindow), 5);

   box1 = gtk_vbox_new(FALSE, 5);
   gtk_container_add(GTK_CONTAINER(*DialogWindow), box1);
   gtk_widget_show(box1);

   entry = gtk_entry_new();
   GTK_WIDGET_SET_FLAGS(entry, GTK_HAS_FOCUS);
   gtk_widget_set_usize(entry, 250, 0);
   gtk_entry_set_text(GTK_ENTRY(entry), EntryStr);
   gtk_box_pack_start(GTK_BOX(box1), entry, FALSE, FALSE, 0);
   *DialogEntry = GTK_WIDGET(entry);
   gtk_widget_show(entry);

   if (CheckButton && CheckButtonText) {
      *CheckButton = gtk_check_button_new_with_label(CheckButtonText);
      gtk_box_pack_start(GTK_BOX(box1), *CheckButton, FALSE, FALSE, 0);
      gtk_widget_show(*CheckButton);
   }

   gtk_signal_connect(GTK_OBJECT(entry), "activate", B1CallBack, B1CbData);

   box2 = gtk_hbox_new(TRUE, 5);
   gtk_box_pack_start(GTK_BOX(box1), box2, FALSE, FALSE, 0);
   gtk_widget_show(box2);

   button = gtk_button_new_with_label(B1Label);
   gtk_signal_connect(GTK_OBJECT(button), "clicked", B1CallBack, B1CbData);
   GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
   gtk_box_pack_start(GTK_BOX(box2), button, FALSE, TRUE, 0);
   gtk_widget_grab_default(button);
   gtk_widget_show(button);
   gtk_signal_connect_object(GTK_OBJECT(entry), "focus_in_event",
                             (GtkSignalFunc) gtk_widget_grab_default,
                             GTK_OBJECT(button));

   button = gtk_button_new_with_label("Clear");
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                             (GtkSignalFunc) Interface_entry_clear,
                             GTK_OBJECT(entry));
   GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
   gtk_box_pack_start(GTK_BOX(box2), button, FALSE, TRUE, 0);
   gtk_widget_show(button);

   button = gtk_button_new_with_label("Cancel");
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                             (GtkSignalFunc) gtk_widget_destroy,
                             GTK_OBJECT(*DialogWindow));
   gtk_widget_add_accelerator(button, "clicked", accel_group,
                              GDK_Escape, 0, GTK_ACCEL_LOCKED);
   GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
   gtk_box_pack_start(GTK_BOX(box2), button, FALSE, TRUE, 0);
   gtk_widget_show(button);

   gtk_widget_grab_focus(entry);
}

/*
 * Make a question-dialog with a question, OK and Cancel.
 */
static void Interface_make_question_dialog(
        GtkWidget **DialogWindow, char *WmName, char *WmClass,
        char *WTitle, char *Question,
        GtkSignalFunc OkCallback, void *OkCbData,
        GtkSignalFunc CancelCallback, void *CancelCbData)
{
   GtkWidget *frame, *label, *button, *box1, *box2;

   *DialogWindow = gtk_window_new(GTK_WINDOW_DIALOG);
   gtk_window_set_wmclass(GTK_WINDOW(*DialogWindow), WmName, WmClass);
   gtk_window_set_title(GTK_WINDOW(*DialogWindow), WTitle);
   gtk_window_set_position(GTK_WINDOW(*DialogWindow), GTK_WIN_POS_CENTER);
   gtk_container_border_width(GTK_CONTAINER(*DialogWindow), 10);
   gtk_signal_connect(GTK_OBJECT(*DialogWindow), "destroy",
                      (GtkSignalFunc) Interface_destroy_window, DialogWindow);

   box1 = gtk_vbox_new(FALSE, 5);
   frame = gtk_frame_new(NULL);
   gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
   label = gtk_label_new(Question);
   gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_FILL);
   gtk_misc_set_padding(GTK_MISC(label), 20, 20);
   gtk_container_add(GTK_CONTAINER(frame), label);
   gtk_widget_show(label);
   gtk_widget_show(frame);
   gtk_box_pack_start(GTK_BOX(box1), frame, TRUE, TRUE, 0);

   box2 = gtk_hbox_new(TRUE, 5);
   button = gtk_button_new_with_label("OK");
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                             OkCallback, OkCbData);
   gtk_signal_connect(GTK_OBJECT(button), "clicked",
                      (GtkSignalFunc) Interface_destroy_window, DialogWindow);
   gtk_widget_show(button);
   gtk_box_pack_start(GTK_BOX(box2), button, FALSE, TRUE, 0);
   button = gtk_button_new_with_label("Cancel");
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                             CancelCallback, CancelCbData);
   gtk_signal_connect(GTK_OBJECT(button), "clicked",
                      (GtkSignalFunc) Interface_destroy_window, DialogWindow);
   gtk_widget_show(button);
   gtk_box_pack_start(GTK_BOX(box2), button, FALSE, TRUE, 0);
   gtk_box_pack_start(GTK_BOX(box1), box2, FALSE, FALSE, 0);
   gtk_container_add(GTK_CONTAINER(*DialogWindow), box1);

   gtk_widget_show(box2);
   gtk_widget_show(box1);
   gtk_widget_grab_focus(button);
   gtk_widget_show(*DialogWindow);
}

/*
 * Create and show an [OK|Cancel] question dialog
 */
void a_Interface_question_dialog(
        BrowserWindow *bw, gchar *QuestionTxt,
        GtkSignalFunc OkCallback, void *OkCbData,
        GtkSignalFunc CancelCallback, void *CancelCbData)
{
   if (!bw->question_dialog_window) {
      Interface_make_question_dialog(&(bw->question_dialog_window),
         "question_dialog", "Dillo", "Dillo: Question", QuestionTxt,
         OkCallback, OkCbData, CancelCallback, CancelCbData);
   } else {
      gtk_widget_destroy(bw->question_dialog_window);
   }
}

/*
 * Create and show the open URL dialog
 */
void a_Interface_open_dialog(GtkWidget *widget, BrowserWindow *bw)
{
   if (!bw->open_dialog_window) {
      Interface_make_dialog(&(bw->open_dialog_window),
         "open_dialog", "Dillo", "Dillo: Open URL",
         &(bw->open_dialog_entry), "", NULL, NULL,
         "OK", (GtkSignalFunc) a_Interface_entry_open_url, (void *)bw);
      if (prefs.transient_dialogs)
         gtk_window_set_transient_for(GTK_WINDOW(bw->open_dialog_window),
                                      GTK_WINDOW(bw->main_window));
   }

   if (!GTK_WIDGET_VISIBLE(bw->open_dialog_window))
      gtk_widget_show(bw->open_dialog_window);
   else
      gdk_window_raise(bw->open_dialog_window->window);
}

/*
 * Receive data from the cache and save it to a local file
 */
static void Interface_save_callback(int Op, CacheClient_t *Client)
{
   DilloWeb *Web = Client->Web;
   gint Bytes;

   if ( Op ){
      struct stat st;

      fflush(Web->stream);
      fstat(fileno(Web->stream), &st);
      fclose(Web->stream);
      a_Interface_msg(Web->dd->bw, "File saved (%d Bytes)", st.st_size);
   } else {
      if ( (Bytes = Client->BufSize - Web->SavedBytes) > 0 ) {
         Bytes = fwrite(Client->Buf + Web->SavedBytes, 1, Bytes, Web->stream);
         Web->SavedBytes += Bytes;
      }
   }
}

/*
 * Save current page to a local file
 */
static void Interface_file_save_url(GtkWidget *widget, BrowserWindow *bw)
{
   const char *name;
   GtkFileSelection *choosefile;
   GtkEntry *entry_url;
   DilloUrl *url;
   FILE *out;

   choosefile = GTK_FILE_SELECTION(bw->save_dialog_window);
   entry_url = GTK_ENTRY(bw->location);
   name = gtk_file_selection_get_filename(choosefile);
   url = a_Url_dup(a_History_get_url(NAV_TOP(bw->dd)));

   if ( strlen(name) && (out = fopen(name, "w")) != NULL ) {
      DilloWeb *Web = a_Web_new(url);
      Web->dd = bw->dd;
      Web->stream = out;
      Web->flags |= WEB_Download;
      /* todo: keep track of this client */
      a_Capi_open_url(Web, Interface_save_callback, Web);
   }
   a_Url_free(url);

   gtk_widget_destroy(bw->save_dialog_window);
}

/*
 * Save the link-URL to a local file
 */
static void Interface_file_save_link(GtkWidget *widget, BrowserWindow *bw)
{
   const gchar *name;
   const DilloUrl *url;
   FILE *out;

   name = gtk_file_selection_get_filename(
             GTK_FILE_SELECTION(bw->save_link_dialog_window));
   url  = a_Menu_popup_get_url(bw);

   if ( strlen(name) && (out = fopen(name, "w")) != NULL ) {
      DilloWeb *Web = a_Web_new(url);
      Web->dd = bw->dd;
      Web->stream = out;
      Web->flags |= WEB_Download;
      /* todo: keep track of this client */
      a_Capi_open_url(Web, Interface_save_callback, Web);
   }

   gtk_widget_destroy(bw->save_link_dialog_window);
}

/*
 * Scan Url and return a local-filename suggestion for saving
 */
static char *Interface_make_save_name(const DilloUrl *url)
{
   gchar *FileName;

   if ((FileName = strrchr(URL_PATH(url), '/')))
      return g_strndup(++FileName, MIN(strlen(FileName), 64));
   return g_strdup("");
}

/*
 * Show the dialog interface for saving an URL
 */
void a_Interface_save_dialog(GtkWidget *widget, BrowserWindow *bw)
{
   gchar *SuggestedName;   /* Suggested save name */
   DilloUrl* url;

   if (!bw->save_dialog_window) {
      Interface_make_choose_file_dialog(
         &bw->save_dialog_window,
         "save_dialog", "Dillo", "Dillo: Save URL as File...",
         (GtkSignalFunc) Interface_file_save_url, (void *)bw );
   }
   url = a_Url_new(a_Interface_get_location_text(bw), NULL, 0, 0);
   SuggestedName = Interface_make_save_name(url);
   gtk_file_selection_set_filename(
      GTK_FILE_SELECTION(bw->save_dialog_window), SuggestedName);
   g_free(SuggestedName);
   a_Url_free(url);

   if (!GTK_WIDGET_VISIBLE(bw->save_dialog_window))
      gtk_widget_show(bw->save_dialog_window);
   else
      gdk_window_raise(bw->save_dialog_window->window);
}

/*
 * Show the dialog interface for saving a link
 */
void a_Interface_save_link_dialog(GtkWidget *widget, BrowserWindow *bw)
{
   char *SuggestedName;   /* Suggested save name */

   if (!bw->save_link_dialog_window) {
      Interface_make_choose_file_dialog(
         &bw->save_link_dialog_window,
         "save_link_dialog", "Dillo",
         "Dillo: Save link as File...",
         (GtkSignalFunc) Interface_file_save_link,
         (void *)bw);
   }
   SuggestedName = Interface_make_save_name(a_Menu_popup_get_url(bw));
   gtk_file_selection_set_filename(
      GTK_FILE_SELECTION(bw->save_link_dialog_window), SuggestedName);
   g_free(SuggestedName);

   if (!GTK_WIDGET_VISIBLE(bw->save_link_dialog_window))
      gtk_widget_show(bw->save_link_dialog_window);
   else
      gdk_window_raise(bw->save_link_dialog_window->window);
}

/*
 * Scroll to an occurence of a string in the open page
 */
static void Interface_entry_search(GtkWidget *widget, BrowserWindow* bw)
{
   char *string;
   gboolean case_sens;
   DilloDoc *dd;

   string = gtk_editable_get_chars(GTK_EDITABLE(bw->findtext_dialog_entry),
				   0, -1);
   case_sens = gtk_toggle_button_get_active
     (GTK_TOGGLE_BUTTON(bw->findtext_dialog_check));

   /* get the document to search from dd_list */
   dd = g_list_nth_data(bw->dd_list, bw->dd_list_index);
   
   switch(a_Dw_gtk_scrolled_window_search(GTK_DW_SCROLLED_WINDOW(dd->docwin),
					  string, case_sens)) {
   case FINDTEXT_RESTART:
   case FINDTEXT_NOT_FOUND:
     a_Dw_gtk_scrolled_window_reset_search((void *) dd->docwin);
     if(++(bw->dd_list_index) < g_list_length(bw->dd_list)) {
       /* recursively repeat search in next document */
       Interface_entry_search(widget, bw);
     } else {
       if(!bw->found)
	 a_Interface_message_window("Dillo: Find text",
				    "Cannot find \"%s\".", string);
       else
	 a_Interface_message_window("Dillo: Find text",
				    "No further occurence of \"%s\". "
				    "Restarting from the beginning.", string);
       bw->found = FALSE;
       bw->dd_list_index = 0;
     }
     break;
   case FINDTEXT_SUCCESS:
     bw->found = TRUE;
     break;
   default:
     g_print("Unexpected return from a_Dw_gtk_scrolled_window_search(), in interface.c:1280...\n");
     break;
   }

   g_free(string);
}

/*
 * Callback for findtext destroy, resets search and destroys dd_list
 */
void Interface_findtext_dialog_destroy(BrowserWindow *bw)
{
   DilloDoc *dd;

   dd = g_list_nth_data(bw->dd_list, bw->dd_list_index);
   /* reset the search */
   a_Dw_gtk_scrolled_window_reset_search((void *) dd->docwin);
   /* destroy the list */
   g_list_free(bw->dd_list);
   bw->dd_list = NULL;
   bw->dd_list_index = 0;
   bw->found = FALSE;
}

/*
 * Show the dialog interface for finding text in a page
 */
void a_Interface_findtext_dialog(BrowserWindow *bw)
{
   /* make search list if it has not been defined yet */
   if(bw->dd_list == NULL)
     bw->dd_list = a_Doc_get_visible_children(bw->dd);
   bw->dd_list_index = 0;
   /* reset found state */
   bw->found = FALSE;

   if (!bw->findtext_dialog_window) {
      Interface_make_dialog(&(bw->findtext_dialog_window),
         "findtext_dialog", "Dillo", "Dillo: Find text in page",
         &(bw->findtext_dialog_entry), "",
         &(bw->findtext_dialog_check), "Case sensitive",
         "Find", (GtkSignalFunc) Interface_entry_search, (void *)bw);
      if (prefs.transient_dialogs)
         gtk_window_set_transient_for(GTK_WINDOW(bw->findtext_dialog_window),
                                      GTK_WINDOW(bw->main_window));
      gtk_signal_connect_object
         (GTK_OBJECT(bw->findtext_dialog_window), "destroy",
          (GtkSignalFunc) Interface_findtext_dialog_destroy,
          (void*) bw);
      gtk_window_set_position(GTK_WINDOW(bw->findtext_dialog_window),
                              GTK_WIN_POS_NONE);
   }

   a_Interface_set_nice_window_pos(bw->findtext_dialog_window,
                                   bw->main_window);

   if (!GTK_WIDGET_VISIBLE(bw->findtext_dialog_window))
      gtk_widget_show(bw->findtext_dialog_window);
   else
      gdk_window_raise(bw->findtext_dialog_window->window);
}

/*
 * This signal callback adjusts the position of a menu.
 * It's useful for very long menus.
 */
void a_Interface_scroll_popup(GtkWidget *widget)
{
   /*
    * todo:
    *   1) Scrolling menues should rather be the task of Gtk+. This is
    *      a hack, and I don't know if it does not break anything.
    *   2) It could be improved, e.g. a timeout could be added for
    *      better mouse navigation.
    */
   int y, h, mx, my, sh;

   y = widget->allocation.y;
   h = widget->allocation.height;
   gdk_window_get_geometry (widget->parent->parent->window,
                            &mx, &my, NULL, NULL, NULL);
   sh = gdk_screen_height ();

   if (y + my < 0)
      gdk_window_move (widget->parent->parent->window, mx, - y + 1);
   else if (y + my > sh - h)
      gdk_window_move (widget->parent->parent->window, mx, sh - h - y - 1);
}

/*
 * A general purpose message window.
 */
void a_Interface_message_window(const char *title, const char *format, ... )
{
   GtkAccelGroup *accel_group;
   GtkWidget *window, *frame, *label, *button, *box;
   static char msg[1024];
   va_list argp;

   va_start(argp, format);
   vsnprintf(msg, 1024, format, argp);
   va_end(argp);

   window = gtk_window_new(GTK_WINDOW_DIALOG);
   gtk_window_set_wmclass(GTK_WINDOW(window), "question_dialog", "dillo");
   gtk_window_set_title(GTK_WINDOW(window), title);
   gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
   gtk_container_border_width(GTK_CONTAINER(window), 10);
   gtk_signal_connect_object(GTK_OBJECT(window), "delete_event",
                             (GtkSignalFunc)gtk_widget_destroy, (void*)window);

   /* accel_group for the message window */
   accel_group = gtk_accel_group_new();
   gtk_window_add_accel_group(GTK_WINDOW(window), accel_group);

   box = gtk_vbox_new(FALSE, 5);
   frame = gtk_frame_new(NULL);
   gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
   label = gtk_label_new(msg);
   gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_FILL);
   gtk_misc_set_padding(GTK_MISC(label), 20, 20);
   gtk_container_add(GTK_CONTAINER(frame), label);
   gtk_widget_show(label);
   gtk_widget_show(frame);
   gtk_box_pack_start(GTK_BOX(box), frame, TRUE, TRUE, 0);

   button = gtk_button_new_with_label("Close");
   gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                             (GtkSignalFunc)gtk_widget_destroy, (void*)window);
   gtk_widget_add_accelerator(button, "clicked",
                              accel_group, GDK_Escape, 0, GTK_ACCEL_LOCKED);
   gtk_widget_show(button);
   gtk_box_pack_start(GTK_BOX(box), button, FALSE, TRUE, 0);
   gtk_widget_show(button);
   gtk_container_add(GTK_CONTAINER(window), box);

   gtk_widget_show(box);
   gtk_widget_grab_focus(button);
   gtk_widget_show(window);
}

/*
 * Places win1 in a way that it does not, or as less as possible, cover win2.
 */
void a_Interface_set_nice_window_pos(GtkWidget *win1, GtkWidget *win2)
{
   gint w1, h1, x2, y2, w2, h2, sw, sh, sr, sl, sb, st, max;

   gtk_widget_realize(win1);
   gdk_window_get_geometry(win1->window, NULL, NULL, &w1, &h1, NULL);
   gdk_window_get_origin(win2->window, &x2, &y2);
   gdk_window_get_geometry(win2->window, NULL, NULL, &w2, &h2, NULL);
   sw = gdk_screen_width();
   sh = gdk_screen_height();

   /* space (excluding win1 space) at right, left, bottom and top */
   sr = sw - (x2 + w2 + w1);
   sl = x2 - w1;
   sb = sh - (y2 + h2 + h1);
   st = y2 - h1;

   /* First, we test, whether win1 can be placed so that it does not
    * covor win2. */
   if (sr >= 0)
      gtk_widget_set_uposition(win1, x2 + w2, (sh - h1) / 2);
   else if (sl >= 0)
      gtk_widget_set_uposition(win1, x2 - w1, (sh - h1) / 2);
   else if (sb >= 0)
      gtk_widget_set_uposition(win1, (sh - h1) / 2, y2 + h2);
   else if (st >= 0)
      gtk_widget_set_uposition(win1, (sh - h1) / 2, y2 - h1);
   else {
      /* Second, we search for the position where the covered space
       * is (more or less) minimized. */
      max = MAX(MAX(sr, sl), MAX(sb, st));
      if (sr == max)
         gtk_widget_set_uposition(win1, sw - w1, (sh - h1) / 2);
      else if (sl == max)
         gtk_widget_set_uposition(win1, 0, (sh - h1) / 2);
      else if (sb == max)
         gtk_widget_set_uposition(win1, (sh - h1) / 2, sh - h1);
      else
         gtk_widget_set_uposition(win1, (sh - h1) / 2, 0);
   }
}
