#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fprompt.h"
#include "fb.h"
#include "iconsel.h"
#include "propdlg.h"

#include "cfg.h"
#include "cfglist.h"
#include "obj.h"
#include "win.h"    
#include "wincb.h"
#include "winlist.h"
#include "winfio.h"
#include "winreload.h"
#include "winopcb.h"
#include "core.h"
#include "help.h"
#include "config.h"

#include "images/icon_folder_menus_opened_32x32.xpm"


typedef struct {
	win_struct	*win;
	obj_struct	*obj;
} FPromptData;
#define FPROMPT_DATA(p)		((FPromptData *)(p))


static gboolean WinQueryOverwrite(
	const gchar *path, GtkWidget *toplevel
);
static void WinFPromptRenameCancelCB(gpointer data);
static void WinFPromptRenameApplyCB(gpointer data, const gchar *value);

static void WinCListSelectAll(GtkCList *clist);
static void WinCListUnselectAll(GtkCList *clist);
static void WinCListInvertSelection(GtkCList *clist);

static void WinNewNexus(win_struct *win, obj_type type);
void WinNewCB(GtkWidget *widget, gpointer data);
void WinNewItemCB(GtkWidget *widget, gpointer data);
void WinNewSeparatorCB(GtkWidget *widget, gpointer data);
void WinNewGroupCB(GtkWidget *widget, gpointer data);
void WinOpenItemCB(GtkWidget *widget, gpointer data);
void WinOpenMenusCB(GtkWidget *widget, gpointer data);
void WinSaveCB(GtkWidget *widget, gpointer data);
void WinSaveAsCB(GtkWidget *widget, gpointer data);
void WinApplyCB(GtkWidget *widget, gpointer data);
void WinAlwaysApplyOnCloseCB(GtkWidget *widget, gpointer data);
void WinRevertCB(GtkWidget *widget, gpointer data);
void WinPrintCB(GtkWidget *widget, gpointer data);
void WinCloseCB(GtkWidget *widget, gpointer data);
void WinExitCB(GtkWidget *widget, gpointer data);

static GList *WinGetSelectedObjects(win_struct *win);
void WinCutCB(GtkWidget *widget, gpointer data);
void WinCopyCB(GtkWidget *widget, gpointer data);
void WinPasteCB(GtkWidget *widget, gpointer data);
void WinRenameCB(GtkWidget *widget, gpointer data);
void WinDeleteCB(GtkWidget *widget, gpointer data);
void WinSelectAllCB(GtkWidget *widget, gpointer data);
void WinUnselectAllCB(GtkWidget *widget, gpointer data);
void WinInvertSelectionCB(GtkWidget *widget, gpointer data);
void WinPropertiesCB(GtkWidget *widget, gpointer data);

void WinRefreshCB(GtkWidget *widget, gpointer data);
void WinViewToolBarCB(GtkWidget *widget, gpointer data);
void WinViewStatusBarCB(GtkWidget *widget, gpointer data);

void WinNewMenuEditorCB(GtkWidget *widget, gpointer data);

void WinHelpContentsCB(GtkWidget *widget, gpointer data);
void WinHelpAboutCB(GtkWidget *widget, gpointer data);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Checks if path exists and if it does then it queries
 *	the user for overwrite.
 *
 *	A return of FALSE means do not proceed.
 */
static gboolean WinQueryOverwrite(
	const gchar *path, GtkWidget *toplevel  
)
{
	gint response;
	gchar *buf;
	struct stat stat_buf;

	if(!STRISEMPTY(path) ? stat(path, &stat_buf) : TRUE)
	    return(TRUE);

	buf = g_strdup_printf(
"Overwrite existing file:\n\
\n\
    %s",
	    path
	);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
	    "Confirm Overwrite",
	    buf,
	    NULL,
	    CDIALOG_ICON_WARNING,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_NO
	);
	CDialogSetTransientFor(NULL);
	g_free(buf);
	switch(response)
	{
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_YES_TO_ALL:
	  case CDIALOG_RESPONSE_OK:
	    return(TRUE);
	    break;
	  default:
	    return(FALSE);
	    break;
	}
}


/*
 *	Floating Prompt Rename Cancel callback.
 */
static void WinFPromptRenameCancelCB(gpointer data)
{
	FPromptData *d = FPROMPT_DATA(data);
	if(d == NULL)
	    return;

	g_free(d);
}

/*
 *	Floating Prompt Rename Apply callback.
 */
static void WinFPromptRenameApplyCB(gpointer data, const gchar *value)
{
	gchar *buf;
	GtkWidget *toplevel;
	obj_struct *obj, *parent_obj;
	win_struct *win;
	FPromptData *d = FPROMPT_DATA(data);
	if(d == NULL)
	    return;

	win = d->win;
	obj = d->obj;

	if((win == NULL) || (obj == NULL) || STRISEMPTY(value))
	{
	    g_free(d);
	    return;
	}

	toplevel = win->toplevel;

	/* Read only or object may not be renamed? */
	if(OBJ_READ_ONLY(obj) || !OBJ_CAN_RENAME(obj))
	{
	    g_free(d);
	    return;
	}

	/* Check if there are any invalid characters in the
	 * specified name
	 */
	if(strpbrk(value, "\\\"/\n\r\t=") != NULL)
	{
	    buf = g_strdup_printf(
"There are invalid characters in the new name:\n\
\n\
    %s\n\
\n\
Characters that may not be used in names are as\n\
follows:\n\
\n\
    \\ Backslash\n\
    \" Double Quote\n\
    / Slash\n\
    = Equal",
		value
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Rename Failed", buf, NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    g_free(d);
	    return;
	}

	/* Check if another Object in the same group has the same
	 * name as the specified name
	 */
	parent_obj = OBJ_PARENT(obj);
	if(parent_obj != NULL)
	{
	    GList *glist;
	    obj_struct *sibling_obj;

	    for(glist = OBJ_CHILDREN(parent_obj);
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		sibling_obj = OBJ(glist->data);
		if((sibling_obj != NULL) ? STRISEMPTY(sibling_obj->name) : TRUE)
		    continue;

		if(!g_strcasecmp(sibling_obj->name, value))
		{
		    buf = g_strdup_printf(
"Another item with the name \"%s\" already\n\
exists in this group.",
			sibling_obj->name
		    );
		    CDialogSetTransientFor(toplevel);
		    CDialogGetResponse(
			"Rename Failed", buf, NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		    g_free(buf);

		    g_free(d);
		    return;
		}
	    }
	}


	/* Begin renaming */

	buf = g_strdup_printf(
	    "Item \"%s\" renamed to \"%s\"",
	    obj->name, value
	);

	/* Set new name */
	g_free(obj->name);
	obj->name = STRDUP(value);

	/* Update last modified time */
	obj->last_modified = time(NULL);

	/* Report status */
	WinStatusMessage(win, buf, FALSE);
	g_free(buf);

	/* Notify the Win about this Object being modified */
	WinObjModifiedCB(win, obj);

	g_free(d);
}


/*
 *      Selects all rows on the specified GtkCList.
 */
static void WinCListSelectAll(GtkCList *clist)
{
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);
}
 
/*
 *      Unselects all rows on the specified GtkCList.
 */
static void WinCListUnselectAll(GtkCList *clist)
{ 
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);                 
}

/*
 *      Inverts selection on the specified GtkCList.
 */
static void WinCListInvertSelection(GtkCList *clist)
{
	gint i;      
	GList *glist;

	if(clist == NULL)
	    return;

	glist = (clist->selection != NULL) ?
	    g_list_copy(clist->selection) : NULL;

	gtk_clist_freeze(clist);
	for(i = 0; i < clist->rows; i++)
	{
	    if(g_list_find(glist, (gpointer)i) != NULL)
		gtk_clist_unselect_row(clist, i, 0);
	    else
		gtk_clist_select_row(clist, i, 0);  
	}                                        
	gtk_clist_thaw(clist);                   
			      
	g_list_free(glist);   
}


/*
 *	Create new Object nexus.
 */
static void WinNewNexus(win_struct *win, obj_type type)
{
	gint sibling_idx = -1;
	const gchar *name, *icon_name;
	gchar *buf;
	GtkWidget *toplevel = win->toplevel;
	GtkCList *clist = GTK_CLIST(win->clist);
	GtkCTree *ctree = GTK_CTREE(win->ctree);
	menu_format format = win->format;
	obj_struct *obj, *parent_obj = NULL, *sibling_obj = NULL;

	WinSetBusy(win, TRUE);

	/* Get selected Object to obtain the parent and sibling
	 * objects in which to create the new Object after
	 */
	if(clist->selection_end != NULL)
	{
	    GList *glist = clist->selection_end;
	    sibling_obj = OBJ(
		gtk_clist_get_row_data(clist, (gint)glist->data)
	    );
	    parent_obj = (sibling_obj != NULL) ?
		sibling_obj->parent : NULL;
	}
	else if(GTK_CLIST(ctree)->selection_end != NULL)
	{
	    GList *glist = GTK_CLIST(ctree)->selection_end;
	    parent_obj = OBJ(gtk_ctree_node_get_row_data(
		ctree, (GtkCTreeNode *)glist->data
	    ));
	}
	/* Unable to obtain parent Object? */
	if(parent_obj == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Create Failed",
"No group has been selected to create the new object in.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Parent does not allow Group Objects as childs? */
	if((type == OBJ_TYPE_GROUP) && OBJ_NO_SUBGROUPS(parent_obj))
	{
	    buf = g_strdup_printf(
"The selected group \"%s\" does not permit\n\
subgroups to be created.",
		parent_obj->name
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Create Failed", buf, NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Create new Object */
	obj = ObjNew();
	if(obj == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Create Failed",
		"Memory allocation error",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Set the new Object's type */
	obj->type = type;

	/* Set the Object's parent */
	obj->parent = parent_obj;

	/* Add this Object to the parent Object's list of children */
	if(sibling_obj != NULL)
	{
	    GList *glist = parent_obj->children;
	    if(glist != NULL)
		sibling_idx = g_list_index(glist, sibling_obj);
	}
	if(sibling_idx >= 0)
	    parent_obj->children = g_list_insert(
		parent_obj->children, obj, sibling_idx
	    );
	else  
	    parent_obj->children = g_list_append(
		parent_obj->children, obj  
	    );

	/* Set default values for the new Object by type and format */
	name = NULL;
	icon_name = NULL;
	switch(format)
	{
	  case MENU_FORMAT_ENDEAVOUR2:
	    switch(type)
	    {
	      case OBJ_TYPE_ITEM:
		name = "New Item";
		icon_name = NULL;
		break;
	      case OBJ_TYPE_ITEM_LINK:
		name = "New Item";
		icon_name = NULL;
		break;
	      case OBJ_TYPE_ITEM_SEPARATOR:
		name = "Separator";
		icon_name = NULL;
		break;
	      case OBJ_TYPE_ITEM_SPECIAL:
		name = "New Special Item";
		icon_name = NULL;
		break;
	      case OBJ_TYPE_GROUP:
		name = "New Group";
		icon_name = NULL;
		break;
	    }
	    break;

	  case MENU_FORMAT_ICEWM:
	    switch(type)
	    {
	      case OBJ_TYPE_ITEM:
		name = "New Item";
		icon_name = ICEWM_DEF_ICON_NAME_ITEM;
		break;
	      case OBJ_TYPE_ITEM_LINK:
		name = "New Item";
		icon_name = ICEWM_DEF_ICON_NAME_LINK;
		break;
	      case OBJ_TYPE_ITEM_SEPARATOR:
		name = "Separator";
		icon_name = ICEWM_DEF_ICON_NAME_SEPARATOR;
		break;
	      case OBJ_TYPE_ITEM_SPECIAL:
		name = "New Special Item";
		icon_name = ICEWM_DEF_ICON_NAME_SPECIAL;
		break;
	      case OBJ_TYPE_GROUP:
		name = "New Group";
		icon_name = ICEWM_DEF_ICON_NAME_GROUP;
		break;
	    }
	    break;
	}
	obj->name = STRDUP(name);
	obj->icon_name = STRDUP(icon_name);

	/* Set default name orientation & justification */
	obj->name_orientation = GTK_ORIENTATION_HORIZONTAL;
	obj->name_justify = GTK_JUSTIFY_LEFT;

	/* Set default icons */
	WinFIOObjLoadIcons(win, obj, NULL);

	/* Reset accelerator key */
	obj->accel_key = '\0';
	obj->accel_mods = 0;

	/* Set the last modified time to be the current time */
	obj->last_modified = time(NULL);

	/* Notify the Win about this new Object */
	WinObjAddedCB(win, obj);

	/* Find the List's new row that refers to this Object and
	 * select it
	 */
	WinListFindSelectObj(win, obj);

	/* Prompt for initial name if the Object can have a name */
	if(OBJ_CAN_RENAME(obj))
	    WinRenameCB(win->rename_mi, win);

	/* Report status */
	buf = g_strdup_printf(
	    "%s created",
	    name
	);
	WinStatusMessage(win, buf, FALSE);
	g_free(buf);

	WinSetBusy(win, FALSE);
}

/*
 *	New.
 */
void WinNewCB(GtkWidget *widget, gpointer data)
{
	WinNewItemCB(widget, data);
}

/*
 *	New Item callback.
 */
void WinNewItemCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	WinNewNexus(win, OBJ_TYPE_ITEM);
}

/*
 *	New Separator callback.
 */
void WinNewSeparatorCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	WinNewNexus(win, OBJ_TYPE_ITEM_SEPARATOR);
}

/*
 *	New Group callback.
 */
void WinNewGroupCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	WinNewNexus(win, OBJ_TYPE_GROUP);
}


/*
 *      Open Item callback.
 */
void WinOpenItemCB(GtkWidget *widget, gpointer data) 
{
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	/* Get the last selected Object on the List */
	clist = GTK_CLIST(win->clist);
	glist = clist->selection_end;
	obj = (glist != NULL) ?
	    OBJ(gtk_clist_get_row_data(clist, (gint)glist->data)) : NULL;
	if(obj == NULL)
	{
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Open the selected Object by its type */
	if(OBJ_IS_ITEM(obj))
	{
	    /* Open Items by executing its command */
	    const gchar *cmd = obj->value;
	    if(!STRISEMPTY(cmd))
	    {
		gint p = Exec(cmd);
		if(p <= 0)
		{
		    gchar *buf = g_strdup_printf(
"Unable to execute:\n\
\n\
    %s\n\
\n\
Please verify that the specified program exists, is set\n\
executable, and the specified arguments (if any) are valid.",
			cmd
		    );
		    CDialogSetTransientFor(toplevel);
		    CDialogGetResponse(
			"Open Failed", buf, NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		    g_free(buf);
		}
	    }
	}
	else if(OBJ_IS_GROUP(obj))
	{
	    /* Open Groups by expanding the Tree's current selected
	     * node and selecting the child node for this object
	     */
	    GtkCTree *ctree = GTK_CTREE(win->ctree);

	    /* Find parent node and expand it */
	    GtkCTreeNode *node = gtk_ctree_find_by_row_data(
		ctree, NULL, OBJ_PARENT(obj)
	    );
	    if(node != NULL)
		gtk_ctree_expand(ctree, node);

	    /* Find node for this Object and select it */
	    node = gtk_ctree_find_by_row_data(      
		ctree, node, obj
	    );
	    if(node != NULL)
		gtk_ctree_select(ctree, node);
 	}
	else if(OBJ_IS_LINK(obj))
	{
	    /* Open Links by selecting its destination */
	    const gchar *path = obj->value;
	    obj_struct *dest_obj = ObjGetObjFromPath(
		win->obj_menu_toplevel, path
	    );
	    if(dest_obj == NULL)
		dest_obj = ObjGetObjFromPath(
		    win->obj_tool_bar_toplevel, path
		);
	    /* Found Link's destination Object? */
	    if(dest_obj != NULL)
	    {
		/* If the destination Object is a Group then select
		 * it, otherwise select its parent and then select it
		 */
		if(OBJ_IS_GROUP(dest_obj))
		{
		    WinTreeFindSelectObj(win, dest_obj);
		}
		else
		{
		    if(WinTreeFindSelectObj(win, OBJ_PARENT(dest_obj)))
		    {
			WinListFindSelectObj(win, dest_obj);
		    }
		}
	    }
	}

	WinSetBusy(win, FALSE);
}

/*
 *	Open Menu Configuration callback.
 */
void WinOpenMenusCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;                           
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);

#warning WinOpenMenusCB() needs a menu configuration file browser

	/* Query user */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Open",
	    "Open", "Cancel",
	    win->filename,		/* Initial path */
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{         
	    const gchar *new_path = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(new_path))
	    {
		/* Open */
/* TODO */


	    }
	}

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	WinUpdate(win);
	WinSetBusy(win, FALSE);
}

/*
 *	Save.
 */
void WinSaveCB(GtkWidget *widget, gpointer data)
{
	gint status;
	gchar *filename;
	GtkWidget *toplevel;
	menu_format format;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	toplevel = win->toplevel;
	format = win->format;
	filename = STRDUP(win->filename);

	WinSetBusy(win, TRUE);

	/* Do not save menu configuration file if marked read only */
	if(win->read_only)
	{
	    gchar *buf;
	    if(filename != NULL)
		buf = g_strdup_printf(
"Unable to save the menu configuration file:\n\
\n\
    %s\n\
\n\
The file was opened as read-only.",
		    filename
		);
	    else  
		buf = STRDUP(
"Unable to save the menu configuration file because it\n\
was opened as read-only"
		);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Save Failed", buf, NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK  
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    WinSetBusy(win, FALSE);
	    g_free(filename);
	    return;
	}

	GUIBlockInput(toplevel, TRUE);

	/* Save the menu configuration */
	status = WinFIOSave(
	    win, filename, format,
	    TRUE, FALSE
	);

	GUIBlockInput(toplevel, FALSE);

	/* Error saving the menu configuration? */
	if(status)
	{
	    gchar *buf;
	    if(filename != NULL)
		buf = g_strdup_printf(
"An error occured while saving the menu configuration file:\n\
\n\
    %s",
		    filename
		);
	    else
		buf = STRDUP(
"An error occured while saving the menu configuration file"
		);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Save Error", buf, NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}

	WinUpdate(win);

	WinSetBusy(win, FALSE);

	g_free(filename);
}

/*
 *	Save As.
 */
void WinSaveAsCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);
	  
	/* Query user */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Save As",
	    "Save", "Cancel",
	    NULL,			/* Initial path */
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_path = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(new_path))
	    {
		/* Do overwrite check */
		if(WinQueryOverwrite(new_path, toplevel))
		{
		    /* Save As */


 
		    g_free(win->filename);
		    win->filename = STRDUP(new_path);
		}
	    }   
	}       

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	/* Reset due to possible file related change */
	FileBrowserReset();

	WinUpdate(win);
	WinSetBusy(win, FALSE);
}


/*
 *	Apply Changes callback.
 *
 *	Saves the menu configuration to file and instructs the
 *	Window Manager to reload.
 */
void WinApplyCB(GtkWidget *widget, gpointer data)
{
	gint status;
	gchar *filename;
	GtkWidget *toplevel;
	menu_format format;
	win_struct *win = WIN(data);
	if((win == NULL) || CDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	toplevel = win->toplevel;
	format = win->format;  
	filename = STRDUP(win->filename);

	WinSetBusy(win, TRUE);

	/* Do not save menu configuration file if marked read only */
	if(win->read_only)
	{
	    gchar *buf;
	    if(filename != NULL)
		buf = g_strdup_printf(
"Unable to save the menu configuration file:\n\
\n\
    %s\n\
\n\
The file was opened as read-only.",
		    filename                      
		);
	    else
		buf = STRDUP(   
"Unable to save the menu configuration file because it\n\
was opened as read-only"
		);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Save Failed", buf, NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    WinSetBusy(win, FALSE);
	    g_free(filename);
	    return;
	}


	GUIBlockInput(toplevel, TRUE);

	/* Save the menu configuration */
	status = WinFIOSave(
	    win, filename, format,
	    TRUE, FALSE
	);

	GUIBlockInput(toplevel, FALSE);

	/* Error saving the menu configuration? */
	if(status)
	{
	    gchar *buf;
	    if(filename != NULL)
		buf = g_strdup_printf(
"An error occured while saving the menu configuration file:\n\
\n\
    %s",
		    filename
		);
	    else
		buf = STRDUP(
"An error occured while saving the menu configuration file"
		);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Save Error", buf, NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	}
	else
	{
	    /* Instruct the Window Manager to reload the menu
	     * configuration
	     */
	    WinWMReload(win);
	}

	WinUpdate(win);

	WinSetBusy(win, FALSE);

	g_free(filename);
}

/*
 *	Always Apply On Close.
 */
void WinAlwaysApplyOnCloseCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	win->always_apply_on_close = !win->always_apply_on_close;

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}


/*
 *	Revert.
 */
void WinRevertCB(GtkWidget *widget, gpointer data)
{
	gint response;
	const gchar *last_file, *last_obj_path;
	gchar *buf, *filename;
	GtkWidget *toplevel;
	menu_format format;
	const cfg_item_struct *cfg_list;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	format = win->format;
	filename = STRDUP(win->filename);
	core = CORE(win->core);
	cfg_list = core->cfg_list;

	/* Confirm revert */
	if(filename != NULL)
	    buf = g_strdup_printf(
"Revert to the last saved menu configuration file:\n\
\n\
    %s\n\
\n\
Any changes that you have made will be discarded.",
		filename
	    );
	else
	    buf = STRDUP(
"Revert to the last saved menu configuration file?\n\
\n\
Any changes that you have made will be discarded."
	    );
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
	    "Confirm Revert", buf, NULL,
	    CDIALOG_ICON_QUESTION,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_NO
	);
	CDialogSetTransientFor(NULL);
	g_free(buf);
	if(response != CDIALOG_RESPONSE_YES)
	{
	    WinSetBusy(win, FALSE);
	    g_free(filename);
	    return;
	}

	GUIBlockInput(toplevel, TRUE);

	/* Reopen the menu configuration file */
	WinFIOOpen(
	    win, filename, format,
	    TRUE, FALSE
	);

	/* Get last selected Object path and attempt to find and select
	 * it on the Win's Tree list
	 */
	last_file = CFGItemListGetValueS(cfg_list, CFG_PARM_WIN_LAST_FILE);
	last_obj_path = CFGItemListGetValueS(cfg_list, CFG_PARM_WIN_LAST_OBJ_PATH);
	if(!STRISEMPTY(last_file) && !STRISEMPTY(win->filename) &&
	   !STRISEMPTY(last_obj_path)
	)
	{
	    if(!strcmp(win->filename, last_file))
	    {
		obj_struct *obj = ObjGetObjFromPath(
		    win->obj_menu_toplevel, last_obj_path
		);
		if(obj == NULL)      
		    obj = ObjGetObjFromPath(
			win->obj_tool_bar_toplevel, last_obj_path
		    );
		WinTreeFindSelectObj(win, obj);
	    }
	}

	WinSetTitle(win);
	WinUpdate(win);

	WinMap(win);		/* Map again to raise after opening menus */

	GUIBlockInput(toplevel, FALSE);
	WinSetBusy(win, FALSE);

	g_free(filename);
}

/*
 *	Print.
 */
void WinPrintCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	WinSetBusy(win, TRUE);



	WinUpdate(win);
	WinSetBusy(win, FALSE);
}

/*
 *	Close.
 */
void WinCloseCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	/* Query user to save and apply changes if the menu
	 * configuration file is not marked read only
	 */
	if(!win->read_only)
	{
	    gint response;

	    if(win->always_apply_on_close)
	    {
		response = CDIALOG_RESPONSE_YES;
	    }
	    else
	    {
		/* Query user to apply changes before closing */
		gchar *buf = STRDUP(
"Save menu configuration and apply changes?"
		);
		win->freeze_count++;
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
		    "Confirm Save & Apply", buf, NULL,
		    CDIALOG_ICON_QUESTION,
		    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO |
			CDIALOG_BTNFLAG_CANCEL,
		    CDIALOG_BTNFLAG_YES
		);
		CDialogSetTransientFor(NULL);
		win->freeze_count--;
		g_free(buf);
	    }
	    /* Handle response */
	    switch(response)
	    {
	      case CDIALOG_RESPONSE_YES:
		WinApplyCB(win->apply_mi, win);
		break;

	      case CDIALOG_RESPONSE_NO:
		break;

	      default:
		WinSetBusy(win, FALSE);
		return;
	    }
	}

	WinSavePositions(win);
	WinUnmap(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Exit.
 */
void WinExitCB(GtkWidget *widget, gpointer data)
{
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	core = win->core;
	if(core == NULL)
	    return;

	WinSetBusy(win, TRUE);

	/* Mark the core's close_all_windows to close all windows during
	 * the core's main management check
	 */
	core->close_all_windows = TRUE;

	WinSavePositions(win);
	WinUpdate(win);

	WinSetBusy(win, FALSE);
}


/*
 *	Returns a list of selected objects.
 */
static GList *WinGetSelectedObjects(win_struct *win)
{
	GList *glist, *obj_list = NULL;
	GtkCList *clist;
	GtkCTree *ctree;
	obj_struct *obj;

	clist = GTK_CLIST(win->clist);
	ctree = GTK_CTREE(win->ctree);

	/* Check if the list has any selected objects first */
	if(clist->selection != NULL)  
	{
	    for(glist = clist->selection; 
		glist != NULL;  
		glist = g_list_next(glist)
	    )
	    {
		obj = OBJ(gtk_clist_get_row_data(clist, (gint)glist->data));
		if(obj == NULL)
		    continue;

		obj_list = g_list_append(obj_list, obj);
	    }
	}
	/* Check if the tree has any selected objects */
	else if(GTK_CLIST(ctree)->selection != NULL)
	{
	    for(glist = GTK_CLIST(ctree)->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = OBJ(gtk_ctree_node_get_row_data(
		    ctree, (GtkCTreeNode *)glist->data
		));
		if(obj == NULL)
		    continue;

		obj_list = g_list_append(obj_list, obj);
	    }
	}

	return(obj_list);
}

/*
 *	Cut callback.
 */
void WinCutCB(GtkWidget *widget, gpointer data)
{
	gchar *buf;
	GList *glist, *obj_list;
	GtkWidget *toplevel;
	obj_struct *obj, *obj_parent;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	toplevel = win->toplevel;
	core = CORE(win->core);

	WinSetBusy(win, TRUE);

	/* Delete any existing Objects in the Clipboard */
	MEditClipboardClear(core);

	/* Get list of selected Objects */
	obj_list = WinGetSelectedObjects(win);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    obj = OBJ(glist->data);
	    if(!ObjIsDescendent(win->obj_menu_toplevel, obj) &&
	       !ObjIsDescendent(win->obj_tool_bar_toplevel, obj)
	    )
		continue;

	    /* Copy and append the Object to the Clipboard's cut list */
	    MEditClipboardCutAppend(core, obj);

	    /* Do not delete if the Object is marked read only */
	    if(OBJ_READ_ONLY(obj))
	    {
		buf = g_strdup_printf(
"The object \"%s\" is marked read-only and may not be deleted.",
		    obj->name
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Cut Failed",
		    buf,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
		continue;
	    }

	    /* Notify the Win about the removal of this Object */
	    WinObjRemovedCB(win, obj);

	    /* Remove this Object from its parent's list of children */
	    obj_parent = obj->parent;
	    if(obj_parent != NULL)
		obj_parent->children = g_list_remove(
		    obj_parent->children, obj
		);

	    /* Delete this Object and all its children */
	    ObjDelete(obj);
	}
	g_list_free(obj_list);

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Copy callback.
 */
void WinCopyCB(GtkWidget *widget, gpointer data)
{
	GList *glist, *obj_list;
	obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data); 
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	core = CORE(win->core);
		   
	WinSetBusy(win, TRUE);

	/* Delete any existing Objects in the Clipboard */
	MEditClipboardClear(core);

	/* Get list of selected Objects */
	obj_list = WinGetSelectedObjects(win);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    obj = OBJ(glist->data);
	    if(!ObjIsDescendent(win->obj_menu_toplevel, obj) && 
	       !ObjIsDescendent(win->obj_tool_bar_toplevel, obj)    
	    )
		continue;

	    /* Copy and append the Object to the Clipboard's copy list */
	    MEditClipboardCopyAppend(core, obj);
	}
	g_list_free(obj_list);

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Paste callback.
 */
void WinPasteCB(GtkWidget *widget, gpointer data)
{
	gboolean is_copy = FALSE;
	gint sibling_index;
	GList *glist;
	GtkCList *clist;
	obj_struct *obj, *parent_obj = NULL, *sibling_obj = NULL;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	core = CORE(win->core);

	WinSetBusy(win, TRUE);

	/* Get parent and sibling Objects */
	clist = GTK_CLIST(win->clist);
	glist = clist->selection_end;
	sibling_obj = OBJ((glist != NULL) ?
	    gtk_clist_get_row_data(clist, (gint)glist->data) : NULL
	);
	parent_obj = (sibling_obj != NULL) ? sibling_obj->parent : NULL;
	if(parent_obj == NULL)
	    parent_obj = WinTreeGetLocation(win);
	if(parent_obj == NULL)
	{
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Get index of the sibling Object in the parent Object's list
	 * of child Objects
	 */
	if(sibling_obj != NULL)
	{
	    glist = parent_obj->children;
	    sibling_index = (glist != NULL) ?
		g_list_index(glist, sibling_obj) : -1;
	}                                                 
	else
	{
	    sibling_index = -1;
	}

	/* Get Clipboard cut or copy list */
	glist = MEditClipboardGetCutList(core);
	if(glist == NULL)
	{
	    glist = MEditClipboardGetCopyList(core);
	    if(glist != NULL)
		is_copy = TRUE;
	}
	/* Paste Objects from the Clipboard's cut or copy list */
	for(; glist != NULL; glist = g_list_next(glist))
	{
	    obj = OBJ(glist->data);
	    if(obj == NULL)
		continue;

	    /* Parent does not allow Group Objects as childs? */
	    if(OBJ_IS_GROUP(obj) && OBJ_NO_SUBGROUPS(parent_obj))
		continue;

	    /* Make a copy of the source Object and all its child
	     * Objects
	     */
	    obj = ObjCopy(obj);
	    if(obj == NULL)
		continue;

	    /* Set the new Object's parent */
	    obj->parent = parent_obj;

	    /* Add this new Object to the parent Object's list of
	     * child Objects
	     */
	    if(sibling_index > -1)
	    {
		parent_obj->children = g_list_insert(
		    parent_obj->children, obj,
		    sibling_index
		);
		sibling_index++;
	    }
	    else
		parent_obj->children = g_list_append(
		    parent_obj->children, obj
		);

	    /* Change the name if this paste operation originated
	     * from a copy
	     */
	    if(is_copy && !OBJ_IS_SEPARATOR(obj))
	    {
		gchar *s = g_strdup_printf(
		    "Copy of %s",
		    obj->name
		);
		g_free(obj->name);
		obj->name = s;
	    }

	    /* Notify the Win about this new Object */
	    WinObjAddedCB(win, obj);
	}

	WinUpdate(win);
 
	WinSetBusy(win, FALSE);
}

/*
 *	Rename callback.
 */
void WinRenameCB(GtkWidget *widget, gpointer data)
{
	gint x = 0, y = 0, width = -1, height = -1;
	GtkWidget *toplevel;
	GtkCList *clist;
	GtkCTree *ctree;
	obj_struct *obj = NULL;
	win_struct *win = WIN(data);
	if((win == NULL) || FPromptIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;
		   
	toplevel = win->toplevel;
	clist = GTK_CLIST(win->clist);
	ctree = GTK_CTREE(win->ctree);

	/* List has a selected object? */
	if(clist->selection_end != NULL)
	{
	    GdkWindow *window = clist->clist_window;
	    GList *glist = clist->selection_end;
	    gint row = (gint)glist->data;
	    gint cell_x, cell_y, cell_width, cell_height;

	    /* Get cell geometry */
	    if(gtk_clist_get_cell_geometry(
		clist, 0, row,
		&cell_x, &cell_y, &cell_width, &cell_height
	    ))
	    {
		gdk_window_get_deskrelative_origin(window, &x, &y);
		x += cell_x + 0;
		y += cell_y - 2;
		width = cell_width;
		height = cell_height;
		obj = OBJ(gtk_clist_get_row_data(clist, row));
	    }
	}
	/* Tree has a selected object? */
	else if(GTK_CLIST(ctree)->selection_end != NULL)
	{
	    GdkWindow *window = GTK_CLIST(ctree)->clist_window;
	    GList *glist = GTK_CLIST(ctree)->selection_end;
	    GtkCTreeNode *node = (GtkCTreeNode *)glist->data;
	    gint row = gtk_ctree_get_node_row_index(ctree, node);
	    gint cell_x, cell_y, cell_width, cell_height;

	    /* Get cell geometry */
	    if(gtk_clist_get_cell_geometry(
		GTK_CLIST(ctree), 0, row,
		&cell_x, &cell_y, &cell_width, &cell_height
	    ))
	    { 
		gdk_window_get_deskrelative_origin(window, &x, &y);
		x += cell_x + 0;
		y += cell_y - 2;
		gdk_window_get_size(
		    GTK_CLIST(ctree)->clist_window, &width, &height
		);
		height = cell_height;
		obj = OBJ(gtk_ctree_node_get_row_data(ctree, node));
	    }
	}

	/* Object does not exist? */
	if(!ObjIsDescendent(win->obj_menu_toplevel, obj) &&
	   !ObjIsDescendent(win->obj_tool_bar_toplevel, obj)
	)
	    return;

	/* Object is read only? */
	if(OBJ_READ_ONLY(obj))
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
"Rename Failed",
"The object is marked read-only and may not be renamed.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return;
	}

	/* Object type may not be renamed? */
	if(!OBJ_CAN_RENAME(obj))
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
"Rename Failed",
"This type of object may not be renamed.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return;
	}

	if(TRUE)
	{
	    FPromptData *d = FPROMPT_DATA(g_malloc0(
		sizeof(FPromptData)
	    ));

	    /* Set up callback data */
	    d->win = win;
	    d->obj = obj;
	     
	    /* Map the Floating Prompt to rename */
	    FPromptSetTransientFor(toplevel);
	    FPromptSetPosition(x, y);
	    FPromptMapQuery(
		NULL,			/* No label */
		obj->name,
		NULL,			/* No tooltip message */
		FPROMPT_MAP_NO_MOVE,	/* Map code */
		width, -1,		/* Width and height */
		0,			/* No buttons */
		d,			/* Callback data */
		NULL,			/* No browse callback */
		WinFPromptRenameApplyCB,
		WinFPromptRenameCancelCB
	    );
	}
}

/*
 *	Delete callback.
 */
void WinDeleteCB(GtkWidget *widget, gpointer data)
{
	gchar *buf;
	gint response, objs_deleted = 0, nobjs;
	GList *glist, *obj_list = NULL;
	GtkWidget *toplevel;
	obj_struct *obj, *obj_parent;
	win_struct *win = WIN(data);
	if((win == NULL) || CDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	/* Get list of selected Objects */
	obj_list = WinGetSelectedObjects(win);
	if(obj_list == NULL)
	{
	    WinSetBusy(win, FALSE);
	    return;
	}

	nobjs = g_list_length(obj_list);

	/* Confirm delete */
	if(nobjs == 1)
	{
	    obj = OBJ(obj_list->data);
	    buf = g_strdup_printf(
		"Delete object \"%s\"?",
		(obj != NULL) ? obj->name : "(null)"
	    );
	}
	else
	{
	    buf = g_strdup_printf(
		"Delete %i selected object%s?",
		nobjs, (nobjs == 1) ? "" : "s"
	    );
	}
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
	    "Confirm Delete",
	    buf,
	    NULL,
	    CDIALOG_ICON_QUESTION,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_NO
	);
	CDialogSetTransientFor(NULL);
	g_free(buf);
	if(response != CDIALOG_RESPONSE_YES)
	{
	    g_list_free(obj_list);
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Delete selected Objects */
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    /* Get selected Object and make sure it exists */
	    obj = OBJ(glist->data);
	    if(!ObjIsDescendent(win->obj_menu_toplevel, obj) &&
	       !ObjIsDescendent(win->obj_tool_bar_toplevel, obj)
	    )
		continue;

	    /* Do not delete if the Object is marked read only */
	    if(OBJ_READ_ONLY(obj))
	    {
		buf = g_strdup_printf(
"The object \"%s\" is marked read-only and may not be deleted.",
		    obj->name
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Delete Failed",
		    buf,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
		continue;
	    }

	    /* Notify the Win about the removal of this Object */
	    WinObjRemovedCB(win, obj);

	    /* Remove this Object from its parent's list of children */
	    obj_parent = obj->parent;
	    if(obj_parent != NULL)
		obj_parent->children = g_list_remove(
		    obj_parent->children, obj
		);

	    /* Delete this Object and all its children */
	    ObjDelete(obj);

	    objs_deleted++;
	}

	WinUpdate(win);

	buf = g_strdup_printf(
	    "%i object%s deleted",
	    objs_deleted, (objs_deleted == 1) ? "" : "s"
	);
	WinStatusMessage(win, buf, FALSE);
	g_free(buf);

	g_list_free(obj_list);

	WinSetBusy(win, FALSE);
}

/*
 *	Select All callback.
 */
void WinSelectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	WinCListSelectAll(GTK_CLIST(win->clist));
}

/*
 *	Unselect All callback.
 */
void WinUnselectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	WinCListUnselectAll(GTK_CLIST(win->clist));
}

/*
 *	Invert Selection callback.
 */
void WinInvertSelectionCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	WinCListInvertSelection(GTK_CLIST(win->clist));
}

/*
 *	Properties callback.
 */
void WinPropertiesCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	win_struct *win = WIN(data);
	if((win == NULL) || PropDlgIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	/* Get selected Object */
	clist = GTK_CLIST(win->clist);
	glist = clist->selection_end;
	obj = OBJ((glist != NULL) ?
	    gtk_clist_get_row_data(clist, (gint)glist->data) : NULL
	);
	if(obj == NULL)
	{
	    GtkCTree *ctree = GTK_CTREE(win->ctree);
	    glist = GTK_CLIST(ctree)->selection_end;
	    obj = OBJ((glist != NULL) ?
		gtk_ctree_node_get_row_data(ctree, (GtkCTreeNode *)glist->data) : NULL
	    );
	}
	if(!ObjIsDescendent(win->obj_menu_toplevel, obj) &&
	   !ObjIsDescendent(win->obj_tool_bar_toplevel, obj)
	)
	{
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Map the Properties Dialog with the values from the selected
	 * object and wait for user response
	 */
	PropDlgSetTransientFor(toplevel);
	status = PropDlgGetResponse(win, obj);
	PropDlgSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    /* Notify the Win about this Object being modified */
	    WinObjModifiedCB(win, obj);
	}

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}


/*
 *	Refresh.
 */
void WinRefreshCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	WinSetBusy(win, TRUE);




	WinUpdate(win);
	WinSetBusy(win, FALSE);
}

/*
 *	Show/Hide Tool Bar.
 */
void WinViewToolBarCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	WinSetBusy(win, TRUE);

	if(WinToolBarIsShown(win))
	    WinToolBarSetShow(win, FALSE);
	else
	    WinToolBarSetShow(win, TRUE);

	WinSetBusy(win, FALSE);
}

/*
 *	Show/Hide Status Bar.
 */
void WinViewStatusBarCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	WinSetBusy(win, TRUE);

	if(WinStatusBarIsShown(win))
	    WinStatusBarSetShow(win, FALSE);
	else
	    WinStatusBarSetShow(win, TRUE);

	WinSetBusy(win, FALSE);
}


/*
 *	New Menu Editor.
 */
void WinNewMenuEditorCB(GtkWidget *widget, gpointer data)
{
	gint x = 0, y = 0;
	GdkWindow *window;
	GtkWidget *w;
	cfg_item_struct *cfg_list;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	w = win->toplevel;
	window = (w != NULL) ? w->window : NULL;
	core = CORE(win->core);
	cfg_list = core->cfg_list;

	/* Get position of toplevel and apply cascade */
	if(window != NULL)
	{
	    gdk_window_get_root_origin(window, &x, &y);
	    x += 30;
	    y += 30;
	}

	/* Set new position in the configuration */
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_X,
	    x, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_Y,
	    y, FALSE
	);

	/* Create new Menu Editor */
	MEditWinNew(CORE(win->core));

	WinSetBusy(win, FALSE);
}


/*
 *	Help Contents.
 */
void WinHelpContentsCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);   

	Help(CORE(win->core), "Contents", win->toplevel);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	About.
 */
void WinHelpAboutCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	CDialogSetTransientFor(toplevel);
#warning Need to include information about the opened menu configuration
	CDialogGetResponseIconData(
"About " PROG_NAME_FULL,
PROG_NAME_FULL "\nVersion " PROG_VERSION "\n\n" PROG_COPYRIGHT,
	    NULL,
	    (guint8 **)icon_folder_menus_opened_32x32_xpm,
	    CDIALOG_BTNFLAG_OK,
	    CDIALOG_BTNFLAG_OK
	);
	CDialogSetTransientFor(NULL);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}
