#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#ifdef HAVE_IMLIB
#include <Imlib.h>
#endif
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkrgb.h>

#include "../include/disk.h"

#include "guiutils.h"
#include "conmsg.h"
#include "cdialog.h"
#include "fprompt.h"
#include "fb.h"
#include "iconsel.h"
#include "propdlg.h"

#include "cfg.h"
#include "cfgfio.h"
#include "obj.h"
#include "win.h"
#include "winlist.h"
#include "winfio.h"
#include "core.h"
#include "cfglist.h"
#include "config.h"


#ifdef HAVE_IMLIB
gpointer	imlib_handle;
#endif


static void MainSignalCB(int s);
static gint MainTOCB(gpointer data);

static cfg_item_struct *MainCfgListNew(void);

void MEditClipboardCutAppend(core_struct *core, obj_struct *obj);
void MEditClipboardCopyAppend(core_struct *core, obj_struct *obj);
GList *MEditClipboardGetCutList(core_struct *core);
GList *MEditClipboardGetCopyList(core_struct *core);
void MEditClipboardClear(core_struct *core);

win_struct *MEditWinNew(core_struct *core);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	UNIX signal callback.
 */
static void MainSignalCB(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
	    exit(1);
	    break;
	}
}

/*
 *	Main timeout callback.
 */
static gint MainTOCB(gpointer data)
{
	gint total_windows = 0;
	GList *glist;
	win_struct *win;
	core_struct *core = CORE(data);
	if(core == NULL)
	    return(FALSE);

	/* Manage all windows, count the number of windows that are
	 * mapped and delete any unmapped windows
	 */
	for(glist = core->win;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    win = WIN(glist->data);
	    if(win == NULL)
		continue;

	    if(WinIsMapped(win))
	    {
		total_windows++;
	    }
	    else
	    {
		WinDelete(win);
		glist->data = NULL;
		continue;
	    }
	}

	/* If no windows are mapped then break out of the main loop */
	if((total_windows == 0) || core->close_all_windows)
	{
	    gtk_main_quit();
	    return(FALSE);
	}

	return(TRUE);
}


/*
 *	Creates a new default Configuration List.
 */
static cfg_item_struct *MainCfgListNew(void)
{
	cfg_intlist_struct *ilist;
	const cfg_item_struct src_cfg_list[] = CFG_LIST;
	cfg_item_struct *cfg_list = CFGItemListCopyList(src_cfg_list);
	const gchar *home;
	gchar cwd[PATH_MAX];

	if(getcwd(cwd, sizeof(cwd)) != NULL)
	    cwd[sizeof(cwd) - 1] = '\0';
	else
	    *cwd = '\0';

	home = getenv("HOME");

	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_MAJOR,
	    PROG_VERSION_MAJOR, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_MINOR,
	    PROG_VERSION_MINOR, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_RELEASE,
	    PROG_VERSION_RELEASE, FALSE
	);

	CFGItemListSetValueS(
	    cfg_list, CFG_PARM_DATA_DIR,
	    PrefixPaths(home, PROG_DATA_DIR), FALSE 
	);
	CFGItemListSetValueS(
	    cfg_list, CFG_PARM_DATA_DIR_GLOBAL,
	    PROG_DATA_DIR_GLOBAL, FALSE
	);

	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_X,
	    0, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_Y,
	    0, FALSE
	);   
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_WIDTH,
	    640, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_HEIGHT,
	    480, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_PANE_POS,
	    180, FALSE
	);
	ilist = CFGIntListNew(NULL, WIN_LIST_COLUMNS);
	if((ilist != NULL) ? (ilist->i != NULL) : FALSE)
	{
	    gint *i = ilist->i;
	    i[0] = 130;		/* Name */
	    i[1] = 60;		/* Type */
	    i[2] = 170;		/* Value */
	    i[3] = 80;		/* Date Modified */
	    CFGItemListSetValueIntList(
		cfg_list, CFG_PARM_WIN_LIST_COLUMN_WIDTHS,
		ilist, FALSE
	    );
	}
	CFGIntListDelete(ilist);
        CFGItemListSetValueI(
            cfg_list, CFG_PARM_WIN_ALWAYS_APPLY_ON_CLOSE,
            FALSE, FALSE
        );

	return(cfg_list);
}


/*
 *	Coppies and appends the specified Object to the Clipboard's
 *	cut list.
 */
void MEditClipboardCutAppend(core_struct *core, obj_struct *obj)
{
	if((core == NULL) || (obj == NULL))
	    return;

	/* Copy the specified Object */
	obj = ObjCopy(obj);
	if(obj == NULL)
	    return;

	/* Append the coppied Object to the Clipboard's cut list */
	core->obj_cut_list = g_list_append(
	    core->obj_cut_list, obj
	);
}

/*
 *	Coppies and appends the specified Object to the Clipboard's
 *	copy list.
 */
void MEditClipboardCopyAppend(core_struct *core, obj_struct *obj)
{
	if((core == NULL) || (obj == NULL))
	    return;

	/* Copy the specified Object */
	obj = ObjCopy(obj);
	if(obj == NULL)
	    return;

	/* Append the coppied Object to the Clipboard's copy list */
	core->obj_copy_list = g_list_append(
	    core->obj_copy_list, obj
	);
}

/*
 *	Returns the Clipboard's cut list.
 */
GList *MEditClipboardGetCutList(core_struct *core)
{
	return((core != NULL) ? core->obj_cut_list : NULL);
}

/*
 *	Returns the Clipboard's copy list.
 */
GList *MEditClipboardGetCopyList(core_struct *core)
{
	return((core != NULL) ? core->obj_copy_list : NULL);
}

/*
 *	Clears the Clipboard's cut and copy lists.
 */
void MEditClipboardClear(core_struct *core)
{
	if(core == NULL)
	    return;

	/* Delete all Objects in the cut list and delete the cut
	 * list itself
	 */
	g_list_foreach(core->obj_cut_list, (GFunc)ObjDelete, NULL);
	g_list_free(core->obj_cut_list);
	core->obj_cut_list = NULL;

	/* Delete all Objects in the copy list and delete the copy
	 * list itself
	 */
	g_list_foreach(core->obj_copy_list, (GFunc)ObjDelete, NULL);
	g_list_free(core->obj_copy_list);
	core->obj_copy_list = NULL;
}


/*
 *	Creates a new Win.
 */
win_struct *MEditWinNew(core_struct *core)
{
	const gchar *last_file, *last_obj_path;
	GList *glist;
	const cfg_item_struct *cfg_list;
	win_struct *win = WinNew(core);
	if(win == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;

	WinMap(win);

	/* Add pointer to the list */
	for(glist = core->win;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    if(glist->data == NULL)
	    {
		glist->data = win;
		return(win);
	    }
	}
	core->win = g_list_append(core->win, win);

	WinSetBusy(win, TRUE);
	GUIBlockInput(win->toplevel, TRUE);

	/* Open menu configuration */
	WinFIOOpen(
	    win, core->filename,
	    core->format,
	    TRUE, FALSE
	);

	/* Get last selected Object path and attempt to find and select
	 * it on the Win's Tree list
	 */
	last_file = CFGItemListGetValueS(cfg_list, CFG_PARM_WIN_LAST_FILE);
	last_obj_path = CFGItemListGetValueS(cfg_list, CFG_PARM_WIN_LAST_OBJ_PATH);
	if(!STRISEMPTY(last_file) && !STRISEMPTY(win->filename) &&
	   !STRISEMPTY(last_obj_path)
	)
	{
	    if(!strcmp(win->filename, last_file))
	    {
		obj_struct *obj = ObjGetObjFromPath(
		    win->obj_menu_toplevel, last_obj_path
		);
		if(obj == NULL)
		    obj = ObjGetObjFromPath(
			win->obj_tool_bar_toplevel, last_obj_path
		    );
		WinTreeFindSelectObj(win, obj);
	    }
	}

	WinSetTitle(win);
	WinUpdate(win);

	WinMap(win);	/* Map again to raise after opening menus */

	GUIBlockInput(win->toplevel, FALSE);
	WinSetBusy(win, FALSE);

	return(win);
}


int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	menu_format	format = MENU_FORMAT_ICEWM;
	const gchar	*filename = NULL,
			*wm_name = NULL,
			*wm_class = NULL,
			*title = NULL;
	gint i;
	const gchar *arg;
	gchar *cfg_file;
	cfg_item_struct *cfg_list;
	win_struct *win;
	core_struct *core;


	/* Set up time zone */
	tzset();

	/* Set up signal callbacks */
	signal(SIGINT, MainSignalCB);
	signal(SIGTERM, MainSignalCB);
	signal(SIGSEGV, MainSignalCB);
	signal(SIGSTOP, MainSignalCB);
	signal(SIGCONT, MainSignalCB);
	signal(SIGPIPE, MainSignalCB);

	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    /* Help */
	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help") ||
	       !g_strcasecmp(arg, "--h") ||
	       !g_strcasecmp(arg, "-h") ||
	       !g_strcasecmp(arg, "-?")
	    )
	    {
		g_print("%s", PROG_USAGE_MESG);
		return(0);
	    }
	    /* Version */
	    else if(!g_strcasecmp(arg, "--version") ||
		    !g_strcasecmp(arg, "-version") ||
		    !g_strcasecmp(arg, "--v") ||
		    !g_strcasecmp(arg, "-v")
	    )
	    {
		g_print(
		    "%s Version %s\n%s",
		    PROG_NAME_FULL, PROG_VERSION, PROG_COPYRIGHT
		);
		return(0);
	    }
	    /* Format */
	    else if(!g_strcasecmp(arg, "--format") ||
		    !g_strcasecmp(arg, "-format") ||
		    !g_strcasecmp(arg, "--f") ||
		    !g_strcasecmp(arg, "-f")
	    )
	    {
		i++;
		arg = (argc > i) ? argv[i] : NULL;
		if(arg != NULL)
		{
		    if(!g_strcasecmp(arg, "IceWM"))
			format = MENU_FORMAT_ICEWM;
		    else
			g_printerr(
"%s: Unsupported value \"%s\".\n",
			    argv[i - 1], arg
			);
		}
		else
		    g_printerr(
"%s: Requires argument.\n",
			argv[i - 1]
		    );
	    }
	    /* WM Name */
	    else if(!g_strcasecmp(arg, "--wm_name") ||
		    !g_strcasecmp(arg, "-wm_name") ||
		    !g_strcasecmp(arg, "--wmname") ||
		    !g_strcasecmp(arg, "-wmname")
	    )
	    {
		i++;
		arg = (argc > i) ? argv[i] : NULL;
		if(arg != NULL)
		    wm_name = arg;
		else
		    g_printerr(
"%s: Requires argument.\n",
			argv[i - 1]
		    );
	    }
	    /* WM Class */
	    else if(!g_strcasecmp(arg, "--wm_class") ||
		    !g_strcasecmp(arg, "-wm_class") ||
		    !g_strcasecmp(arg, "--wmclass") ||
		    !g_strcasecmp(arg, "-wmclass")
	    )
	    {
		i++;
		arg = (argc > i) ? argv[i] : NULL;
		if(arg != NULL)
		    wm_class = arg;
		else
		    g_printerr(
"%s: Requires argument.\n",    
			argv[i - 1]
		    );   
	    }
	    /* Title */
	    else if(!g_strcasecmp(arg, "--title") ||
		    !g_strcasecmp(arg, "-title")
	    )
	    {
		i++;
		arg = (argc > i) ? argv[i] : NULL;
		if(arg != NULL)
		    title = arg;
		else
		    g_printerr(
"%s: Requires argument.\n",
			argv[i - 1]
		    );
	    }
	    /* All else assume file name */
	    else if((*arg != '-') && (*arg != '+'))
	    {
		filename = arg;
	    }
	}

	/* Set GTK locale */
	gtk_set_locale();

	/* Initialize GTK as needed */
	if(!initialized_gtk)
	{
	    if(!gtk_init_check(&argc, &argv))
	    {
		g_printerr(
"Unable to initialize GTK.\n"
		);
		return(1);
	    }

	    /* Initialize the GDK RGB buffers system */
	    gdk_rgb_init();

	    initialized_gtk = TRUE;
	}

#ifdef HAVE_IMLIB
	imlib_handle = Imlib_init(GDK_DISPLAY());
#endif

	/* Initialize the Console Message Display System */
	ConMsgInit(PROG_NAME_FULL, 0, 0, TRUE, TRUE);

	/* Initialize dialogs */
	CDialogInit();
	FPromptInit();
	FileBrowserInit();
	IconSelInit(
#ifdef HAVE_IMLIB
	    imlib_handle
#else
	    NULL
#endif
	);
	PropDlgInit();

	/* Create core */
	core = CORE(
	    g_malloc0(sizeof(core_struct))
	);
	core->cfg_list = cfg_list = MainCfgListNew();
	core->pid = (gint)getpid();
	core->uid = (gint)getuid();
	core->euid = (gint)geteuid();
	core->home_dir = STRDUP(getenv("HOME"));
	core->prog_path = STRDUP((argc > 0) ? argv[0] : "");
	core->prog_name = STRDUP((argc > 0) ? argv[0] : "");
	core->user_name = STRDUP(getenv("USERNAME"));
	if(core->user_name == NULL)
	    core->user_name = STRDUP(getenv("USER"));
	core->menu_item_ptr_atom = gdk_atom_intern(
	    "menueditor/obj-ptr", FALSE
	);
	core->busy_cur = gdk_cursor_new(GDK_WATCH);
	core->translate_cur = gdk_cursor_new(GDK_FLEUR); 
	core->zoom_cur = gdk_cursor_new(GDK_SIZING); 
	core->resize_cur = gdk_cursor_new(GDK_SIZING);
	core->close_all_windows = FALSE;
	core->win = NULL;
	core->obj_cut_list = NULL;
	core->obj_copy_list = NULL;
	core->format = format;
	core->filename = STRDUP(filename);
	core->argc = (gint)argc;
	core->argv = (gchar **)argv;
	core->wm_name = STRDUP(wm_name);
	core->wm_class = STRDUP(wm_class);
	core->title = STRDUP(title);

	/* Load configuration from file */
	cfg_file = STRDUP(PrefixPaths(
	    CFGItemListGetValueS(cfg_list, CFG_PARM_DATA_DIR),
	    PROG_CFG_FILE
	));
	CFGLoadFromFile(cfg_file, cfg_list);
	g_free(cfg_file);

	/* Check version */
/* TODO */

	/* Update version */
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_MAJOR,
	    PROG_VERSION_MAJOR, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_MINOR,
	    PROG_VERSION_MINOR, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_RELEASE,
	    PROG_VERSION_RELEASE, FALSE
	);


	/* Create first Win */
	win = MEditWinNew(core);

	/* Set main timeout callback */
	gtk_timeout_add(
	    1000,		/* In milliseconds */
	    MainTOCB, core
	);
	gtk_main();


	/* Save configuration to file */
	cfg_file = STRDUP(PrefixPaths(
	    CFGItemListGetValueS(cfg_list, CFG_PARM_DATA_DIR),
	    PROG_CFG_FILE
	));
	CFGSaveToFile(cfg_file, cfg_list);
	g_free(cfg_file);


	/* Delete core */
	g_list_foreach(core->win, (GFunc)WinDelete, NULL);
	g_list_free(core->win);
	core->win = NULL;

	MEditClipboardClear(core);

	GDK_CURSOR_DESTROY(core->busy_cur);
	GDK_CURSOR_DESTROY(core->translate_cur);
	GDK_CURSOR_DESTROY(core->zoom_cur);
	GDK_CURSOR_DESTROY(core->resize_cur);
	CFGItemListDeleteList(core->cfg_list);
	g_free(core->home_dir);
	g_free(core->prog_path);
	g_free(core->prog_name);
	g_free(core->user_name);
	g_free(core->filename);
	g_free(core->title);
	g_free(core);


	/* Shutdown dialogs */
	PropDlgShutdown();
	IconSelShutdown();
	FileBrowserShutdown();
	FPromptShutdown();
	CDialogShutdown();

	/* Shutdown the Console Message Display System */
	ConMsgShutdown();

	return(0);
}
