/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "filelist.h"

#include "cache.h"
#include "rcfile.h"
#include "ui_fileops.h"


/*
 *-----------------------------------------------------------------------------
 * file filtering
 *-----------------------------------------------------------------------------
 */

static GList *filter_list = NULL;
static GList *extension_list = NULL;

gint ishidden(const gchar *name)
{
	if (name[0] != '.') return FALSE;
	if (name[1] == '\0' || (name[1] == '.' && name[2] == '\0')) return FALSE;
	return TRUE;
}

static FilterEntry *filter_entry_new(const gchar *description, const gchar *extensions, gint enabled)
{
	FilterEntry *fe;

	fe = g_new0(FilterEntry, 1);
	fe->description = g_strdup(description);
	fe->extensions = g_strdup(extensions);
	fe->enabled = enabled;
	
	return fe;
}

static void filter_entry_free(FilterEntry *fe)
{
	if (!fe) return;

	g_free(fe->description);
	g_free(fe->extensions);
	g_free(fe);
}

GList *filter_get_list(void)
{
	return filter_list;
}

void filter_remove_entry(FilterEntry *fe)
{
	if (!g_list_find(filter_list, fe)) return;

	filter_list = g_list_remove(filter_list, fe);
	filter_entry_free(fe);
}

void filter_add(const gchar *description, const gchar *extensions, gint enabled)
{
	filter_list = g_list_append(filter_list, filter_entry_new(description, extensions, enabled));
}

static void filter_add_if_missing(const gchar *description, const gchar *extensions, gint enabled)
{
	GList *work;

	work = filter_list;
	while (work)
		{
		FilterEntry *fe = work->data;
		work = work->next;
		if (fe->description && strcmp(fe->description, description) == 0) return;
		}

	filter_add(description, extensions, enabled);
}

void filter_reset(void)
{
	GList *work;

	work = filter_list;
	while (work)
		{
		FilterEntry *fe = work->data;
		work = work->next;
		filter_entry_free(fe);
		}

	g_list_free(filter_list);
	filter_list = NULL;
}

void filter_add_defaults(void)
{
	filter_add_if_missing("JPEG group", ".jpg;.jpeg;.jpe", TRUE);
	filter_add_if_missing("Portable Network Graphic", ".png", TRUE);
	filter_add_if_missing("Tiff", ".tif;.tiff", TRUE);
	filter_add_if_missing("Packed Pixel formats", ".pbm;.pgm;.pnm;.ppm", TRUE);
	filter_add_if_missing("Graphics Interchange Format", ".gif", TRUE);
	filter_add_if_missing("X bitmap", ".xpm;.xbm", TRUE);
	filter_add_if_missing("Bitmap", ".bmp", TRUE);
	filter_add_if_missing("Icon file", ".ico", TRUE);
	filter_add_if_missing("Raster", ".ras", TRUE);
}

static GList *filter_to_list(const gchar *extensions)
{
	GList *list = NULL;
	const gchar *p;

	if (!extensions) return NULL;

	p = extensions;
	while (*p != '\0')
		{
		const gchar *b;
		gint l = 0;

		b = p;
		while (*p != '\0' && *p != ';')
			{
			p++;
			l++;
			}
		list = g_list_append(list, g_strndup(b, l));
		if (*p == ';') p++;
		}

	return list;
}

void filter_rebuild(void)
{
	GList *work;

	path_list_free(extension_list);
	extension_list = NULL;

	work = filter_list;
	while (work)
		{
		FilterEntry *fe;

		fe = work->data;
		work = work->next;

		if (fe->enabled)
			{
			GList *ext;

			ext = filter_to_list(fe->extensions);
			if (ext) extension_list = g_list_concat(extension_list, ext);
			}
		}
}

gint filter_name_exists(const gchar *name)
{
	GList *work;
	if (!extension_list || file_filter_disable) return TRUE;

	work = extension_list;
	while (work)
		{
		gchar *filter = work->data;
		gint lf = strlen(filter);
		gint ln = strlen(name);
		if (ln >= lf)
			{
			if (strncasecmp(name + ln - lf, filter, lf) == 0) return TRUE;
			}
		work = work->next;
		}

	return FALSE;
}

void filter_write_list(FILE *f)
{
	GList *work;

	work = filter_list;
	while (work)
		{
		FilterEntry *fe = work->data;
		work = work->next;

		fprintf(f, "filter: \"%s\" \"%s%s\"\n", fe->extensions, (fe->enabled) ? "" : "#",
			(fe->description) ? fe->description : "");
		}
}

void filter_parse(const gchar *text)
{
	const gchar *p;
	gchar *ext;
	gchar *desc;
	gint enabled = TRUE;

	if (!text || text[0] != '"') return;

	ext = quoted_value(text);

	p = text;
	p++;
	while (*p != '"' && *p != '\0') p++;
	if (*p == '"') p++;
	while (*p != '"' && *p != '\0') p++;

	if (*p == '"')
		{
		desc = quoted_value(p);

		if (desc && desc[0] == '#')
			{
			gchar *tmp;
			tmp = g_strdup(desc + 1);
			g_free(desc);
			desc = tmp;

			enabled = FALSE;
			}
		}
	else
		{
		desc = NULL;
		}

	filter_add(desc, ext, enabled);

	g_free(ext);
	g_free(desc);
}

#if 0
gint filter_name_exists(const gchar *name)
{
	GList *work;
	if (!filter_list || file_filter_disable) return TRUE;

	work = filter_list;
	while (work)
		{
		gchar *filter = work->data;
		gint lf = strlen(filter);
		gint ln = strlen(name);
		if (ln >= lf)
			{
			if (strncasecmp(name + ln - lf, filter, lf) == 0) return TRUE;
			}
		work = work->next;
		}

	return FALSE;
}

static void filter_add(gchar *text, gint add)
{
	if (add) filter_list = g_list_append(filter_list, g_strdup(text));
}

void filter_rebuild(void)
{
	if (filter_list)
		{
		path_list_free(filter_list);
		filter_list = NULL;
		}

	filter_add(".jpg", filter_include_jpg);
	filter_add(".jpeg", filter_include_jpg);
	filter_add(".xpm", filter_include_xpm);
	filter_add(".tif", filter_include_tif);
	filter_add(".tiff", filter_include_tif);
	filter_add(".gif", filter_include_gif);
	filter_add(".png", filter_include_png);
	filter_add(".ppm", filter_include_ppm);
	filter_add(".pgm", filter_include_pgm);
	filter_add(".pcx", filter_include_pcx);
	filter_add(".bmp", filter_include_bmp);

	if (custom_filter)
		{
		gchar *buf = g_strdup(custom_filter);
		gchar *pos_ptr_b;
		gchar *pos_ptr_e = buf;
		while(pos_ptr_e[0] != '\0')
			{
			pos_ptr_b = pos_ptr_e;
			while (pos_ptr_e[0] != ';' && pos_ptr_e[0] != '\0') pos_ptr_e++;
			if (pos_ptr_e[0] == ';')
				{
				pos_ptr_e[0] = '\0';
				pos_ptr_e++;
				}
			filter_add(pos_ptr_b, TRUE);
			}
		g_free(buf);
		}
}
#endif

GList *path_list_filter(GList *list, gint is_dir_list)
{
	GList *work;

	if (file_filter_disable && show_dot_files) return list;

	work = list;
	while (work)
		{
		gchar *name = work->data;
		const gchar *base;

		base = filename_from_path(name);

		if ((!show_dot_files && ishidden(base)) ||
		    (!is_dir_list && !filter_name_exists(base)) )
			{
			GList *link = work;
			work = work->next;
			list = g_list_remove_link(list, link);
			g_free(name);
			g_list_free(link);
			}
		else
			{
			work = work->next;
			}
		}

	return list;
}

/*
 *-----------------------------------------------------------------------------
 * path list recursive
 *-----------------------------------------------------------------------------
 */

static gint path_list_sort_cb(gconstpointer a, gconstpointer b)
{
	return strcmp((gchar *)a, (gchar *)b);
}

GList *path_list_sort(GList *list)
{
	return g_list_sort(list, path_list_sort_cb);
}

static void path_list_recursive_append(GList **list, GList *dirs)
{
	GList *work;

	work = dirs;
	while (work)
		{
		const gchar *path = work->data;
		GList *f = NULL;
		GList *d = NULL;

		if (path_list(path, &f, &d))
			{
			f = path_list_filter(f, FALSE);
			f = path_list_sort(f);
			*list = g_list_concat(*list, f);

			d = path_list_filter(d, TRUE);
			d = path_list_sort(d);
			path_list_recursive_append(list, d);
			g_list_free(d);
			}

		work = work->next;
		}
}

GList *path_list_recursive(const gchar *path)
{
	GList *list = NULL;
	GList *d = NULL;

	if (!path_list(path, &list, &d)) return NULL;
	list = path_list_filter(list, FALSE);
	list = path_list_sort(list);

	d = path_list_filter(d, TRUE);
	d = path_list_sort(d);
	path_list_recursive_append(&list, d);
	path_list_free(d);

	return list;
}

/*
 *-----------------------------------------------------------------------------
 * text conversion utils
 *-----------------------------------------------------------------------------
 */

gchar *text_from_size(gint size)
{
	gchar *a, *b;
	gchar *s, *d;
	gint l, n, i;

	/* what I would like to use is printf("%'d", size)
	 * BUT: not supported on every libc :(
	 */

	a = g_strdup_printf("%d", size);
	l = strlen(a);
	n = (l - 1)/ 3;
	if (n < 1) return a;

	b = g_new(gchar, l + n + 1);

	s = a;
	d = b;
	i = l - n * 3;
	while (*s != '\0')
		{
		if (i < 1)
			{
			i = 3;
			*d = ',';
			d++;
			}

		*d = *s;
		s++;
		d++;
		i--;
		}
	*d = '\0';

	g_free(a);
	return b;
}

gchar *text_from_size_abrev(gint size)
{
	if (size < 1024)
		{
		return g_strdup_printf(_("%d bytes"), size);
		}
	if (size < 1048576)
		{
		return g_strdup_printf(_("%.1f K"), (gfloat)size / 1024.0);
		}

	return g_strdup_printf(_("%.1f MB"), (gfloat)size / 1048576.0);
}

const gchar *text_from_time(time_t t)
{
	static gchar ret[64];
	struct tm *btime;

	btime = localtime(&t);

	/* the %x warning about 2 digit years is not an error */
	if (strftime(ret, sizeof(ret), "%x %H:%M", btime) < 1) return "";

	return ret;
}

/*
 *-----------------------------------------------------------------------------
 * file info struct
 *-----------------------------------------------------------------------------
 */

FileData *file_data_new(gchar *path, struct stat *st)
{
	FileData *fd;

	fd = g_new0(FileData, 1);
	fd->path = path;
	fd->name = filename_from_path(fd->path);
	fd->size = st->st_size;
	fd->date = st->st_mtime;

	return fd;
}

void file_data_free(FileData *fd)
{
	g_free(fd->path);
	g_free(fd);
}

/*
 *-----------------------------------------------------------------------------
 * load file list
 *-----------------------------------------------------------------------------
 */

static SortType filelist_sort_method = SORT_NONE;
static gint filelist_sort_ascend = TRUE;

static gint sort_file_cb(void *a, void *b)
{
	FileData *fa = a;
	FileData *fb = b;

	if (!filelist_sort_ascend)
		{
		fa = b;
		fb = a;
		}

	switch (filelist_sort_method)
		{
		case SORT_SIZE:
			if (fa->size < fb->size) return -1;
			if (fa->size > fb->size) return 1;
			return 0;
			break;
		case SORT_TIME:
			if (fa->date < fb->date) return -1;
			if (fa->date > fb->date) return 1;
			return 0;
			break;
#ifdef HAVE_STRVERSCMP
		case SORT_NUMBER:
			return strverscmp(fa->name, fb->name);
			break;
#endif
		case SORT_NAME:
		default:
			return strcmp(fa->name, fb->name);
			break;
		}
}

GList *filelist_sort(GList *list, SortType method, gint ascend)
{
	filelist_sort_method = method;
	filelist_sort_ascend = ascend;
	return g_list_sort(list, (GCompareFunc) sort_file_cb);
}

GList *filelist_insert_sort(GList *list, FileData *fd, SortType method, gint ascend)
{
	filelist_sort_method = method;
	filelist_sort_ascend = ascend;
	return g_list_insert_sorted(list, fd, (GCompareFunc) sort_file_cb);
}

gint filelist_read(const gchar *path, GList **files, GList **dirs)
{
	DIR *dp;
	struct dirent *dir;
	struct stat ent_sbuf;
	GList *dlist;
	GList *flist;

	dlist = NULL;
	flist = NULL;

	if (!path || (dp = opendir(path)) == NULL)
		{
		if (files) *files = NULL;
		if (dirs) *dirs = NULL;
		return FALSE;
		}

	/* root dir fix */
	if (path[0] == '/' && path[1] == '\0') path = "";

	while ((dir = readdir(dp)) != NULL)
		{
		gchar *name = dir->d_name;
		if (show_dot_files || !ishidden(name))
			{
			gchar *filepath = g_strconcat(path, "/", name, NULL);
			if (stat(filepath, &ent_sbuf) >= 0)
				{
				if (S_ISDIR(ent_sbuf.st_mode))
					{
					/* we ignore the .thumbnails dir for cleanliness */
					if ((dirs) &&
					    !(name[0] == '.' && (name[1] == '\0' || (name[1] == '.' && name[2] == '\0'))) &&
					    strcmp(name, GQVIEW_CACHE_DIR) != 0)
						{
						dlist = g_list_prepend(dlist, file_data_new(filepath, &ent_sbuf));
						filepath = NULL;
						}
					}
				else
					{
					if ((files) && filter_name_exists(name))
						{
						flist = g_list_prepend(flist, file_data_new(filepath, &ent_sbuf));
						filepath = NULL;
						}
					}
				}
			g_free(filepath);
			}
		}

	closedir(dp);

	if (dirs) *dirs = dlist;
	if (files) *files = flist;

	return TRUE;
}

void filelist_free(GList *list)
{
	GList *work;

	work = list;
	while (work)
		{
		file_data_free((FileData *)work->data);
		work = work->next;
		}

	g_list_free(list);
}


