/*
 * File: tab.c
 *
 * Copyright (C) 2003 Frank de Lange <frank@unternet.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. */

#ifndef DISABLE_TABS

#include <config.h>

#include <stdio.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>

#include "misc.h"
#include "dillo.h"
#include "history.h"
#include "nav.h"
#include "doc.h"
#include "IO/Url.h"
#include "interface.h"
#include "tab.h"
#include "commands.h"
#include "prefs.h"
#include "menu.h"

#include "dw_gtk_scrolled_window.h"
#include "dw_gtk_viewport.h"

#include <gdk/gdkkeysyms.h>

#define DEBUG_EVENT  0
#define DEBUG_SIZE  10
#define DEBUG_ALLOC 10

#define DEBUG_LEVEL 0
#include "debug.h"

/* BROWSER TABS */

/* callbacks */

/*
 * callback for tab label resize_request. This adjusts
 * the tab title to the width of the tab, compressing
 * or shortening (according to preference) the title if needed 
 */
void
Tab_label_size_request_callback(GtkWidget *widget,
                                GtkAllocation *alloc,
                                gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *) client_data;
   GtkLabel *tab_label = (GtkLabel *) widget;
   gchar *tab_title, *short_title;
   gint tab_title_size;

   /* the tool tip is attached to the parent of the label, an event box */
   tab_title = ((GtkTooltipsData *) gtk_tooltips_data_get(
                                       GTK_WIDGET(
                                        GTK_WIDGET(
                                         tab_label)->parent)))->tip_text;

   tab_title_size = (gint) ((gfloat) alloc->width / dd->bw->tab_label_average_character_width);

   /* compress or shorten the tab title? */
   if (prefs.tab_title_compress) {
     /* use title compression algorithm */
     short_title = a_Misc_string_compress(tab_title,tab_title_size);
   } else {
     /* use simple shortening */
     short_title = a_Misc_string_shorten(tab_title,tab_title_size);
   }
   gtk_label_set_text(GTK_LABEL(tab_label), short_title);
   g_free(short_title);
}

/*
 * callback for tab label button_press_event
 *
 * used to popup menu with tab options
 *
 * even though this is called 'tab_label_..., the callback is in
 * fact attached to an event box surrounding the tab_label, as gtk
 * label widgets can not receive events
 */
void
Tab_label_button_press_event_callback(GtkWidget *widget,
                                      GdkEventButton *event,
                                      gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *) client_data;
   switch (event->button) {
   case 1:
     a_Tab_switch(dd);
     break;
   case 2:
     /* currently unused */
     DEBUG_MSG(DEBUG_EVENT, "You pressed button (%d) over a tab...\n", (gint) event->button);
     break;
   case 3:
     /* popup tab options menu */
     if (dd->bw->menu_popup.over_tab)
       gtk_widget_destroy(dd->bw->menu_popup.over_tab);
     dd->bw->menu_popup.over_tab = a_Menu_popup_tab_new(dd);
     gtk_menu_popup(GTK_MENU(dd->bw->menu_popup.over_tab), NULL, NULL,
                    NULL, NULL, event->button, event->time);
     break;
   default:
     /* weird pointer with more than 3 buttons, ignore */
     DEBUG_MSG(DEBUG_EVENT, "You pressed button (%d) over a tab...\n", (gint) event->button);
     break;
   }
}
  
/*
 * callback for tab switch. This just calls a_Tab_switch for the docwin
 * passed as page
 */
void
Tab_switch_callback(GtkWidget *tabbrowser,
                    GtkNotebookPage *page,
                    gint page_num,
                    gpointer client_data)
{
   DilloDoc *dd;
   GtkWidget *docwin;

   /* the notebook page contains a box, which contains the docwin as its first child
    * hence this hairy function... */
   docwin = g_list_nth_data(gtk_container_children(GTK_CONTAINER(gtk_notebook_get_nth_page(GTK_NOTEBOOK(tabbrowser), page_num))), 0);
   dd = a_Doc_get_by_docwin(docwin);

   if(dd)
     a_Tab_switch(dd);
}

/*
 * callback for tab add
 */
void
Tab_add_callback(GtkContainer *container,
                 GtkWidget *widget,
                 gpointer client_data)
{
   DEBUG_MSG(DEBUG_EVENT, "I am the tab_add callback in src/tab.c!\n");
}

/*
 * callback for tab remove
 */
void
Tab_remove_callback(GtkContainer *container,
                    GtkWidget *widget,
                    gpointer client_data)
{
   DEBUG_MSG(DEBUG_EVENT, "I am the tab remove callback, in src/tab.c!\n");
}

/*
 * callback for tab browser destroy
 */
void
Tab_browser_destroy_callback(GtkContainer *container,
                    GtkWidget *widget,
                    gpointer client_data)
{
   DEBUG_MSG(DEBUG_EVENT, "I am the tab browser destroy callback, in src/tab.c!\n");
}

/* Public functions */

/*
 * return a new tab browser
 */
GtkContainer *
a_Tab_browser_new(BrowserWindow *bw)
{
   GtkRequisition *req;
   GtkWidget *label;
   GtkNotebook *tabbrowser;

   /* try to calulate the average width of a character in a label
    * in the current context. This is something of a hack...
    */
   req = g_new0(GtkRequisition, 1);
   label = gtk_label_new(D_QUICK_BROWN_FOX);
   gtk_widget_show(label);
   gtk_widget_size_request(GTK_WIDGET(label), req);
   bw->tab_label_average_character_width = (gfloat) ((gfloat) req->width / (gfloat) strlen(GTK_LABEL(label)->label));
   g_free(req);

   /* Add tabbrowser */
   tabbrowser = GTK_NOTEBOOK(gtk_notebook_new());
   gtk_notebook_set_show_border(GTK_NOTEBOOK(tabbrowser), FALSE);
   gtk_notebook_set_scrollable(GTK_NOTEBOOK(tabbrowser), prefs.tab_bar_scroller);
   gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(tabbrowser), prefs.tab_bar_homogeneous);
   gtk_signal_connect_after(GTK_OBJECT(tabbrowser),
			    "switch_page",
			    GTK_SIGNAL_FUNC(Tab_switch_callback),
			    (gpointer) bw);
   
   gtk_signal_connect_after(GTK_OBJECT(tabbrowser),
                            "add",
                            GTK_SIGNAL_FUNC(Tab_add_callback),
                            (gpointer) bw);

   gtk_signal_connect(GTK_OBJECT(tabbrowser),
                            "remove",
                            GTK_SIGNAL_FUNC(Tab_remove_callback),
                            (gpointer) bw);
   /* commented out, currently not in use...  
   gtk_signal_connect(GTK_OBJECT(tabbrowser),
                            "destroy",
                            GTK_SIGNAL_FUNC(Tab_browser_destroy_callback),
                            (gpointer) bw);
   */
   
   return GTK_CONTAINER(tabbrowser);
}

/*
 * switch tab, update interface
 */
void
a_Tab_switch(DilloDoc *dd)
{
   /* set bw's current root document */
   dd->bw->dd = a_Doc_get_root(dd);

   /* add the browser window's full_screen_off_button to the dd's
    * docwin. This button is owned by the window (this makes it easier
    * to move tabs between windows, if needed), so it needs to
    * be moved from tab to tab when the tab becomes current (as
    * GTK does not allow a widget to have more than one parent).
    *
    * XXX:HACK
    */
   if (GTK_WIDGET(dd->bw->full_screen_off_button)->parent != dd->docwin) {
     /* keep a ref to the widget to keep it alive in transit... */
     gtk_object_ref(GTK_OBJECT(dd->bw->full_screen_off_button));
     if (GTK_WIDGET(dd->bw->full_screen_off_button)->parent)
       gtk_container_remove(GTK_CONTAINER(GTK_WIDGET(dd->bw->full_screen_off_button)->parent),
			    GTK_WIDGET(dd->bw->full_screen_off_button));
     a_Dw_gtk_scrolled_window_add_gadget(GTK_DW_SCROLLED_WINDOW (dd->docwin),
					 dd->bw->full_screen_off_button);
     /* ref is no longer needed, discard it... */
     gtk_object_unref(GTK_OBJECT(dd->bw->full_screen_off_button));

     /* XXX:HACK the docwin does not show (or raise?) the widget when needed... */
     if(GTK_WIDGET_VISIBLE(dd->bw->full_screen_off_button)) {
       gtk_widget_hide(dd->bw->full_screen_off_button);
       gtk_widget_show(dd->bw->full_screen_off_button);
     }
   }

   /* update progress bar and adjust button sensitivity */
   a_Doc_progress_update(dd);

   /* adjust tab specific interface elements */
   if (dd->nav_stack_ptr != -1)
     /* this tab has a document loaded, set values accordingly */
     {
       a_Interface_set_page_title(dd->bw, (gchar *) a_History_get_title(NAV_TOP(dd)));
       a_Interface_set_location_text(dd->bw,URL_STR_(a_History_get_url(NAV_TOP(dd))));
     } else {
       /* no document loaded in tab, use default values */
       a_Interface_set_page_title(dd->bw, "(Untitled)");
       a_Interface_set_location_text(dd->bw,"");
     }

   /* show tab */
   gtk_notebook_set_page(GTK_NOTEBOOK(dd->bw->container),
			 gtk_notebook_page_num(GTK_NOTEBOOK(dd->bw->container),
			 GTK_WIDGET(dd->docwin)->parent));

   /* set focus to docwin */
   gtk_widget_grab_focus(GTK_BIN(dd->docwin)->child);
}

/*
 * set visibility of tab bar
 */
void
a_Tab_visibility_update(BrowserWindow *bw)
{
   gboolean hide_tabs;

   g_return_if_fail(bw != NULL);

   /* this gets called in the tab switching code, which in turn gets called when
    * a tab is deleted. This includes the final tab, which in turn causes the main_window
    * to be destroyed, which of course includes the tabbrowser... So, check first if
    * there is still a tabbrowser around to update before doing anything else...
    */
   if(!GTK_IS_CONTAINER(bw->container))
     return;
   
   /* HIDE tab bar IF
    * -  there is only one tab left AND the preference tab_bar_show_single_tab
    * is set to NO
    *  OR
    * - when using fullscreen AND the preference tab_bar_show_fullscreen
    * is set to NO
    * OTHERWISE
    * SHOW tab bar
    */
   hide_tabs = ((g_list_length(gtk_container_children(GTK_CONTAINER(bw->container))) == 1)
                && (!prefs.tab_bar_show_single_tab));
   hide_tabs |= (bw->fullwindow
                 && (!prefs.tab_bar_show_fullscreen));
   gtk_notebook_set_show_tabs(GTK_NOTEBOOK(bw->container), !hide_tabs);
}


/*
 * close a specific browser tab,
 * close window if no tabs left
 *
 * NOTE: this will destroy the document which
 * was displayed in the tab, through the destroy callback
 * for the docwin (in src/doc.c). If documents have to
 * be moved between tabs or frames, unparent the
 * pagemarks menu and reparent dd->docwin before calling
 * this function!
 */
void
a_Tab_close(DilloDoc *dd)
{
   gint page_num;
   BrowserWindow *bw = dd->bw;

   g_return_if_fail (dd != NULL);

   page_num = gtk_notebook_page_num(GTK_NOTEBOOK(bw->container),
				    GTK_WIDGET(dd->docwin)->parent);

   /* left-to-right tab deletion: if this is the current page, show next page */
   if(gtk_notebook_get_current_page(GTK_NOTEBOOK(bw->container)) == page_num)
     gtk_notebook_next_page(GTK_NOTEBOOK(bw->container));

   /* this will destroy dd->docwin (and the DilloDoc)
    * if there are no reference left to it */
   gtk_notebook_remove_page(GTK_NOTEBOOK(bw->container), page_num);

   if (gtk_notebook_get_current_page(GTK_NOTEBOOK(bw->container)) == -1)
     {
       /* destroying the main window causes its destroy
          callback to be fired. This takes down the
          rest of the BrowserWindow */
       gtk_widget_destroy(GTK_WIDGET(bw->main_window));
     }

   a_Tab_visibility_update(bw);
}

/*
 * Add a tab (containing a new DilloDoc) to a browser window
 */
void
a_Tab_new(BrowserWindow *bw)
{
   DilloDoc *dd;

   g_return_if_fail (bw != NULL);

   dd = a_Doc_new();
   a_Tab_add(bw, dd);

   /* always show new tabs, even when background loading */
   a_Tab_switch(dd);

   /* set focus to url bar */
   gtk_widget_grab_focus(dd->bw->location);
}


/*
 * Add a tab (containing an existing DilloDoc) to a browser window
 */
void
a_Tab_add(BrowserWindow *bw, DilloDoc *dd)
{
   GString *tab_title;
   GtkWidget *tab_label, *tab_label_event_box, *box;
   GtkPackType *pack_type = g_new0(GtkPackType, 1);

   g_return_if_fail (bw != NULL && dd != NULL);

   a_Doc_set_browserwindow(dd, bw);

   if (dd->nav_stack_ptr != -1)
     {
       tab_title = g_string_new(a_History_get_title(NAV_TOP(dd)));
     } else {
       tab_title = g_string_new("(Untitled)");
     }
   
   tab_label_event_box = gtk_event_box_new();
   tab_label = gtk_label_new(tab_title->str);
   gtk_widget_show(tab_label);
   gtk_container_add(GTK_CONTAINER(tab_label_event_box), tab_label);
   box = gtk_hbox_new(TRUE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(box), 0);
   gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(dd->docwin), TRUE, TRUE, 0);
#ifdef XHTML_DTD_FRAMESET
   if(dd->frameset)
     gtk_box_pack_start(GTK_BOX(box), GTK_WIDGET(dd->frameset), TRUE, TRUE, 0);
#endif /* XHTML_DTD_FRAMESET */
   gtk_widget_show(box);
   gtk_notebook_append_page(GTK_NOTEBOOK(bw->container), box, tab_label_event_box);

   /* add tooltip with full title. This has to be attached to the event box,
    * since label widgets are windowless and hence can not receive events */
   gtk_tooltips_set_tip(GTK_TOOLTIPS(dd->bw->tooltips),
                        GTK_WIDGET(tab_label_event_box),
                        tab_title->str,
                        tab_title->str);

   /* make sure the tab labels adjust their size according to available space */
   gtk_signal_connect_after(GTK_OBJECT(tab_label),
                            "size_allocate",
                            GTK_SIGNAL_FUNC(Tab_label_size_request_callback),
                            (gpointer) dd);

   /* catch button press events for showing menu, etc. This has to be attached
    * to the event box, since label widgets are windowless and hence can not
    * receive events */
   gtk_signal_connect_after(GTK_OBJECT(tab_label_event_box),
                      "button_press_event",
                      GTK_SIGNAL_FUNC(Tab_label_button_press_event_callback),
                      (gpointer) dd);

   g_string_free(tab_title, TRUE);
   gtk_notebook_query_tab_label_packing(GTK_NOTEBOOK(bw->container), GTK_WIDGET(dd->docwin)->parent, NULL, NULL, pack_type);
   gtk_notebook_set_tab_label_packing(GTK_NOTEBOOK(bw->container), GTK_WIDGET(dd->docwin)->parent, TRUE, TRUE, *pack_type);
   g_free(pack_type);
   a_Tab_visibility_update(bw);
}


/*
 * set tab title
 * also sets window title if this tab is the current tab
 */
void
a_Tab_title_set(DilloDoc *dd, gchar *title)
{
   GtkWidget *tab_label;

   g_return_if_fail (dd != NULL && title != NULL);

   /* set tooltip on tab label container (a GtkEventBox), and queue a resize
    * for the label itself so the title is redrawn */
   tab_label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(dd->bw->container),GTK_WIDGET(dd->docwin)->parent);
   gtk_tooltips_set_tip(GTK_TOOLTIPS(dd->bw->tooltips),
                       GTK_WIDGET(tab_label),
                       title,
                       title);

   gtk_widget_queue_resize(GTK_WIDGET(GTK_BIN(tab_label)->child));

   /* if this is the current tab, set window title */
   if (dd->bw->dd == dd)
     a_Interface_set_page_title(dd->bw, title);
}

#endif /* !DISABLE_TABS */
