#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "guiutils.h"
#include "keyscan.h"

#include "images/icon_clear_20x20.xpm"

/*
 *	Key Scan Data:
 */
typedef struct {

	GtkWidget	*toplevel,
			*key_da,
			*modifiers_da,
			*clear_btn;

	guint		key,
			modifiers;

	void		(*changed_cb)(
		GtkWidget *, gpointer
	);
	gpointer	changed_data;

} KeyScanData;
#define KEY_SCAN_DATA(p)	((KeyScanData *)(p))
#define KEY_SCAN_DATA_KEY	"key_scan_data"


static KeyScanData *KeyScanGetData(GtkWidget *w);

static void KeyScanDestroyCB(gpointer data);
static gint KeyScanEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void KeyScanClearCB(GtkWidget *widget, gpointer data);

static void KeyScanDrawKey(KeyScanData *d);
static void KeyScanDrawModifiers(KeyScanData *d); 

GtkWidget *KeyScanNew(void);

void KeyScanSetChangedCB(
	GtkWidget *w,
	void (*func)(GtkWidget *, gpointer),
	gpointer data
);

guint KeyScanGetKey(GtkWidget *w);
guint KeyScanGetModifiers(GtkWidget *w);

void KeyScanSetKey(GtkWidget *w, guint key);
void KeyScanSetModifiers(GtkWidget *w, guint modifiers);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the Key Scan Data from the specified Widget.
 */
static KeyScanData *KeyScanGetData(GtkWidget *w)
{
	return(KEY_SCAN_DATA(GTK_OBJECT_GET_DATA(w, KEY_SCAN_DATA_KEY)));
}


/*
 *	Key Scan Data destroy callback.
 */
static void KeyScanDestroyCB(gpointer data)
{
	KeyScanData *d = KEY_SCAN_DATA(data);
	if(d == NULL)
	    return;

	g_free(d);
}

/*
 *	Key Scan event signal callback.
 */
static gint KeyScanEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventConfigure *configure;
	GdkEventExpose *expose;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	KeyScanData *d = KEY_SCAN_DATA(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;
	    status = TRUE;
	    break;

	  case GDK_EXPOSE:
	    expose = (GdkEventExpose *)event;
	    if(widget == d->key_da)
		KeyScanDrawKey(d);
	    else if(widget == d->modifiers_da)
		KeyScanDrawModifiers(d);
	    status = TRUE;
	    break;

	  case GDK_FOCUS_CHANGE:
	    focus = (GdkEventFocus *)event;
	    if(!GTK_WIDGET_HAS_FOCUS(widget) && focus->in)
	    {
		GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
		gtk_widget_queue_draw(widget);
	    }
	    else if(GTK_WIDGET_HAS_FOCUS(widget) && !focus->in)
	    {
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
		gtk_widget_queue_draw(widget);
	    }
	    status = TRUE;
	    break;

	  case GDK_KEY_PRESS:
	    key = (GdkEventKey *)event;
	    if(widget == d->key_da)
	    {
		d->key = key->keyval;
		gtk_widget_queue_draw(widget);

		/* Report change */
		if(d->changed_cb != NULL)
		    d->changed_cb(
			d->toplevel,
			d->changed_data
		    );
	    }
	    else if(widget == d->modifiers_da)
	    {
		switch(key->keyval)
		{
		  case GDK_Control_L:
		  case GDK_Control_R:
		    if(d->modifiers & GDK_CONTROL_MASK)
			d->modifiers &= ~GDK_CONTROL_MASK;
		    else
			d->modifiers |= GDK_CONTROL_MASK;
		    break;
		  case GDK_Alt_L:
		  case GDK_Alt_R:
		    if(d->modifiers & GDK_MOD1_MASK)
			d->modifiers &= ~GDK_MOD1_MASK;
		    else
			d->modifiers |= GDK_MOD1_MASK;
		    break;
		  case GDK_Meta_L:
		  case GDK_Meta_R:
		    if(d->modifiers & GDK_MOD2_MASK)
			d->modifiers &= ~GDK_MOD2_MASK;
		    else
			d->modifiers |= GDK_MOD2_MASK;
		    break;
		  case GDK_Caps_Lock:
		  case GDK_Shift_Lock:
		  case GDK_ISO_Lock:
		    if(d->modifiers & GDK_LOCK_MASK)
			d->modifiers &= ~GDK_LOCK_MASK;
		    else
			d->modifiers |= GDK_LOCK_MASK;
		    break;
		  case GDK_Shift_L:
		  case GDK_Shift_R:
		    if(d->modifiers & GDK_SHIFT_MASK)
			d->modifiers &= ~GDK_SHIFT_MASK;
		    else
			d->modifiers |= GDK_SHIFT_MASK;
		    break;
		}
		gtk_widget_queue_draw(widget);

		/* Report change */
		if(d->changed_cb != NULL)
		    d->changed_cb(
			d->toplevel, 
			d->changed_data 
		    );
	    }
	    status = TRUE;
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    if(!GTK_WIDGET_HAS_FOCUS(widget) &&
	       GTK_WIDGET_CAN_FOCUS(widget)
	    )
		gtk_widget_grab_focus(widget);

	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	Key Scan Clear callback.
 */
static void KeyScanClearCB(GtkWidget *widget, gpointer data)
{
	KeyScanData *d = KEY_SCAN_DATA(data);
	if(d == NULL)
	    return;

	d->key = '\0';
	d->modifiers = 0;
	gtk_widget_queue_draw(d->key_da);
	gtk_widget_queue_draw(d->modifiers_da);

	/* Report change */
	if(d->changed_cb != NULL)
	    d->changed_cb(
		d->toplevel,
		d->changed_data
	    );
}


/*
 *	Draws the Key Scan key.
 */
static void KeyScanDrawKey(KeyScanData *d)
{
	gint width, height;
	GdkFont *font;
	GdkWindow *window;
	GdkDrawable *drawable;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w = (d != NULL) ? d->key_da : NULL;
	if(w == NULL)
	    return;

	window = w->window;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return;

	gdk_window_get_size(window, &width, &height);
	if((width <= 0) || (height <= 0))
	    return;

	font = style->font;
	drawable = (GdkDrawable *)window;

	/* Draw background */
	gdk_draw_rectangle(
	    drawable, style->base_gc[state], TRUE,
	    0, 0, width, height
	);

	/* Draw key */
	if((d->key != '\0') && (font != NULL))
	{
	    const gchar *s = gdk_keyval_name(d->key);
	    GdkTextBounds b;
	    gdk_string_bounds(font, s, &b);
	    gdk_draw_string(
		drawable, font, style->text_gc[state],
		(2 * 2) + b.lbearing,
		((height - (font->ascent + font->descent)) / 2) +
		    font->ascent,
		s
	    );   
	}

	/* Draw frame */
	gtk_draw_shadow(
	    style, drawable, state, GTK_SHADOW_IN,
	    0, 0, width, height
	);

	/* Draw focus */
	if(GTK_WIDGET_SENSITIVE(w) && GTK_WIDGET_HAS_FOCUS(w))
	    gtk_draw_focus(style, drawable, 0, 0, width - 1, height - 1);
}

/*
 *	Draws the Key Scan modifiers.
 */
static void KeyScanDrawModifiers(KeyScanData *d)
{
	gint width, height;
	GdkFont *font;
	GdkWindow *window;
	GdkDrawable *drawable;
	GtkStateType state;
	GtkStyle *style;

	GtkWidget *w = (d != NULL) ? d->modifiers_da : NULL;
	if(w == NULL)
	    return;

	window = w->window;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return;

	gdk_window_get_size(window, &width, &height);
	if((width <= 0) || (height <= 0))
	    return;

	font = style->font;
	drawable = (GdkDrawable *)window;
	 
	/* Draw background */
	gdk_draw_rectangle(  
	    drawable, style->base_gc[state], TRUE,
	    0, 0, width, height
	);

	/* Draw modifiers */
	if((d->modifiers != 0) && (font != NULL))
	{
	    const guint mods = d->modifiers;
	    gchar *s = NULL;
	    GdkTextBounds b;

#define APPEND_STRING(_s_) {	\
 gchar *s2;			\
				\
 if(s != NULL) {		\
  s2 = g_strdup_printf(		\
   "%s+%s", s, (_s_)		\
  );				\
  g_free(s);			\
 } else {			\
  s2 = STRDUP(_s_);		\
 }				\
 s = s2;			\
}

	    if(mods & GDK_CONTROL_MASK)
		APPEND_STRING("Ctrl");
	    if(mods & GDK_MOD1_MASK)
		APPEND_STRING("Alt");
	    if(mods & GDK_LOCK_MASK)
		APPEND_STRING("CapsLock");
	    if(mods & GDK_SHIFT_MASK)
		APPEND_STRING("Shift");

	    gdk_string_bounds(font, s, &b);
	    gdk_draw_string(
		drawable, font, style->text_gc[state],
		(2 * 2) + b.lbearing,
		((height - (font->ascent + font->descent)) / 2) +
		    font->ascent,
		s
	    );

	    g_free(s);
#undef APPEND_STRING
	}

	/* Draw frame */
	gtk_draw_shadow( 
	    style, drawable, state, GTK_SHADOW_IN,
	    0, 0, width, height    
	);                 

	/* Draw focus */
	if(GTK_WIDGET_SENSITIVE(w) && GTK_WIDGET_HAS_FOCUS(w))
	    gtk_draw_focus(style, drawable, 0, 0, width - 1, height - 1);
}


/*
 *	Creates a new Key Scan.
 */
GtkWidget *KeyScanNew(void)
{
	const gint border_minor = 2;
	GdkFont *font;
	GtkStyle *style;
	GtkWidget *w, *parent;
	KeyScanData *d = KEY_SCAN_DATA(g_malloc0(sizeof(KeyScanData)));
	if(d == NULL)
	    return(NULL);

	d->key = '\0';
	d->modifiers = 0;
	d->changed_cb = NULL;
	d->changed_data = NULL;

	/* Toplevel */
	d->toplevel = parent = w = gtk_hbox_new(FALSE, border_minor);
	gtk_object_set_data_full(
	    GTK_OBJECT(w), KEY_SCAN_DATA_KEY,
	    d, KeyScanDestroyCB
	);

	/* Key GtkLabel */
	w = gtk_label_new("Key:");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Key GtkDrawingArea */
	d->key_da = w = gtk_drawing_area_new();
	style = gtk_widget_get_style(w);
	font = (style != NULL) ? style->font : NULL;
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
	gtk_widget_set_usize(
	    w,
	    100,
	    ((font != NULL) ? (font->ascent + font->descent) : 0) + (4 * 2)
	);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
	    GDK_FOCUS_CHANGE_MASK |
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "focus_in_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "focus_out_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Modifiers Label */
	w = gtk_label_new("Mods:");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Modifiers GtkDrawingArea */
	d->modifiers_da = w = gtk_drawing_area_new();
	style = gtk_widget_get_style(w);
	font = (style != NULL) ? style->font : NULL;
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
	gtk_widget_set_usize(
	    w,
	    150,
	    ((font != NULL) ? (font->ascent + font->descent) : 0) + (4 * 2)
	);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
	    GDK_FOCUS_CHANGE_MASK |
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "focus_in_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "focus_out_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(KeyScanEventCB), d
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Clear Button */
	w = GUIButtonPixmap((guint8 **)icon_clear_20x20_xpm);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(KeyScanClearCB), d
	);
	GUISetWidgetTip(w, "Clear");
	gtk_widget_show(w);


	return(d->toplevel);
}


/*
 *	Sets the Key Scan's changed callback.
 */
void KeyScanSetChangedCB(
	GtkWidget *w, 
	void (*func)(GtkWidget *, gpointer), 
	gpointer data  
)
{
	KeyScanData *d = KeyScanGetData(w);
	if(d == NULL)
	    return;

	d->changed_cb = func;
	d->changed_data = data;
}


/*
 *	Gets the Key Scan's key.
 */
guint KeyScanGetKey(GtkWidget *w)
{
	KeyScanData *d = KeyScanGetData(w);
	return((d != NULL) ? d->key : '\0');
}

/*
 *	Gets the Key Scan's modifiers.
 */
guint KeyScanGetModifiers(GtkWidget *w)
{
	KeyScanData *d = KeyScanGetData(w);
	return((d != NULL) ? d->modifiers : 0);
}

/*
 *	Sets the Key Scan's key.
 */
void KeyScanSetKey(GtkWidget *w, guint key)
{
	KeyScanData *d = KeyScanGetData(w);
	if(d == NULL)
	    return;

	if(d->key != key)
	{
	    d->key = key;
	    gtk_widget_queue_draw(d->key_da);
	}
}

/*
 *	Sets the Key Scan's modifiers.
 */
void KeyScanSetModifiers(GtkWidget *w, guint modifiers)
{
	KeyScanData *d = KeyScanGetData(w);
	if(d == NULL)
	    return;

	if(d->modifiers != modifiers)
	{
	    d->modifiers = modifiers;
	    gtk_widget_queue_draw(d->modifiers_da);
	}
}
