#include <string.h>
#ifdef HAVE_IMLIB
# include <Imlib.h>
#endif
#include <gdk/gdk.h>
#include "../include/tga.h"
#include "guiutils.h"
#include "imgfio.h"


static guint8 *CombineRGBAndAlphaToRGBA(
	const guint8 *rgb, const guint8 *alpha,
	gint width, gint height,               
	gint rgb_bpl, gint alpha_bpl 
);

GdkPixmap *ImgOpenPixmap(
	const gchar *filename,
	gint *width_rtn, gint *height_rtn,
	GdkBitmap **mask_rtn,
	gpointer imlib_h
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Combines the specified RGB and Alpha image data to RGBA.
 */
static guint8 *CombineRGBAndAlphaToRGBA(
	const guint8 *rgb, const guint8 *alpha,
	gint width, gint height,
	gint rgb_bpl, gint alpha_bpl
)
{
	gint rgba_bpl;
	guint8 *rgba;

	if((rgb == NULL) || (width <= 0) || (height <= 0))
	    return(NULL);

	/* Calculate bytes per line for all buffers */
	if(rgb_bpl <= 0)
	    rgb_bpl = width * 3;
	if(alpha_bpl <= 0)
	    alpha_bpl = width * 1;
	rgba_bpl = width * 4;

	/* Allocate RGBA data */
	rgba = (guint8 *)g_malloc(rgba_bpl * height);
	if(rgba == NULL)
	    return(NULL);

	/* Alpha channel available? */
	if(alpha != NULL)
	{
	    guint8 *rgba_line, *rgba_line_end, *rgba_ptr, *rgba_end;
	    const guint8 *rgb_line, *rgb_ptr, *alpha_line, *alpha_ptr;
 
	    for(rgba_line       = rgba,
		rgba_line_end   = rgba_line + (rgba_bpl * height),
		rgb_line        = rgb,
		alpha_line      = alpha;
		rgba_line < rgba_line_end;
		rgba_line       += rgba_bpl,
		rgb_line        += rgb_bpl,
		alpha_line      += alpha_bpl
	    )
	    {
		rgba_ptr        = rgba_line;
		rgba_end        = rgba_ptr + (width * 4);
		rgb_ptr         = rgb_line;
		alpha_ptr       = alpha_line;

		while(rgba_ptr < rgba_end)
		{
		    *rgba_ptr++ = *rgb_ptr++;
		    *rgba_ptr++ = *rgb_ptr++;
		    *rgba_ptr++ = *rgb_ptr++;
		    *rgba_ptr++ = *alpha_ptr++;
		}
	    }
	}
	else
	{
	    guint8 *rgba_line, *rgba_line_end, *rgba_ptr, *rgba_end;
	    const guint8 *rgb_line, *rgb_ptr;

	    for(rgba_line       = rgba,
		rgba_line_end   = rgba_line + (rgba_bpl * height),
		rgb_line        = rgb;
		rgba_line < rgba_line_end;
		rgba_line       += rgba_bpl,
		rgb_line        += rgb_bpl
	    )
	    {
		rgba_ptr        = rgba_line;
		rgba_end        = rgba_ptr + (width * 4);
		rgb_ptr         = rgb_line;

		while(rgba_ptr < rgba_end)
		{
		    *rgba_ptr++ = rgb_ptr[0];
		    *rgba_ptr++ = rgb_ptr[1];
		    *rgba_ptr++ = rgb_ptr[2];
		    *rgba_ptr++ = (
			(rgb_ptr[0] == 0xff) &&
			(rgb_ptr[1] == 0x00) &&
			(rgb_ptr[2] == 0xff)
		    ) ?
			0x00 : 0xff;

		    rgb_ptr += 3;
		}
	    }
	}

	return(rgba);
}


/*
 *	Opens the image from file to a pixmap & mask.
 */
GdkPixmap *ImgOpenPixmap(
	const gchar *filename,
	gint *width_rtn, gint *height_rtn,
	GdkBitmap **mask_rtn,
	gpointer imlib_h
)
{
	gint width = 0, height = 0;
	const gchar *ext = (filename != NULL) ?
	    strrchr(filename, '.') : NULL;
	GdkBitmap *mask = NULL;
	GdkPixmap *pixmap = NULL;

	if(width_rtn != NULL)
	    *width_rtn = width;
	if(height_rtn != NULL)
	    *height_rtn = height;
	if(mask_rtn != NULL)
	    *mask_rtn = mask;

	if(ext == NULL)
	    return(pixmap);

	/* Targa */
	if(!g_strcasecmp(ext, ".tga") ||
	   !g_strcasecmp(ext, ".targa")
	)
	{
	    guint8 *rgba = TgaReadFromFileFastRGBA(
		(const char *)filename, (int *)&width, (int *)&height,
		0xff000000      /* Transparent pixel rgba */
	    );
	    if(rgba != NULL)
	    {
		const gint bpl = width * 4;
		GdkGC *gc = GDK_GC_NEW();

		/* Create a pixmap and mask from the rgba data */
		pixmap = GDK_PIXMAP_NEW(width, height);
	       if(pixmap != NULL)
		    gdk_draw_rgb_32_image(
			pixmap, gc,
			0, 0, width, height,
			GDK_RGB_DITHER_NONE,
			(guchar *)rgba,
			bpl
		    );
		mask = GUICreateBitmapFromDataRGBA(
		    width, height, bpl,
		    rgba,
		    0x80,	/* Alpha byte threshold */
		    NULL
		);
		GDK_GC_UNREF(gc);
		g_free(rgba);
	    }
	}
	/* XPM */
	else if(!g_strcasecmp(ext, ".xpm"))
	{
	    pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
		&mask, filename
	    );
	    if(pixmap != NULL)
	    {
		gdk_window_get_size(pixmap, &width, &height);
	    }
	}
	else
	{   
#ifdef HAVE_IMLIB
	    /* All else open as an Imlib image */
	    ImlibImage *img = (imlib_h != NULL) ?
		Imlib_load_image(imlib_h, (char *)filename) : NULL;
	    if(img != NULL)
	    {
		guint8 *rgba;

		/* Realize image changes and get image geometry */
		Imlib_changed_image(imlib_h, img);
		width = img->rgb_width;
		height = img->rgb_height;

		/* Combine Imlib image data to rgba image data */
		rgba = CombineRGBAndAlphaToRGBA(
		    img->rgb_data, img->alpha_data,
		    width, height,
		    -1, -1              /* Autocalculate bpl values */
		);

		/* Unref the Imlib image */
		Imlib_destroy_image(imlib_h, img);

		if(rgba != NULL)
		{
		    const gint bpl = width * 4;
		    GdkGC *gc = GDK_GC_NEW();

		    /* Create a pixmap and mask from the rgba data */
		    pixmap = GDK_PIXMAP_NEW(width, height);
		    if(pixmap != NULL)
			gdk_draw_rgb_32_image(
			    pixmap, gc,
			    0, 0, width, height,
			    GDK_RGB_DITHER_NONE,
			    (guchar *)rgba,
			    bpl
			);
		    mask = GUICreateBitmapFromDataRGBA(
			width, height, bpl,
			rgba,
			0x80,		/* Alpha byte threshold */
			NULL
		    );
		    GDK_GC_UNREF(gc);
		    g_free(rgba);
		}
	    }
#endif
	}

	/* Update returns */
	if(width_rtn != NULL)
	    *width_rtn = width;
	if(height_rtn != NULL)
	    *height_rtn = height;
	if(mask_rtn != NULL)
	    *mask_rtn = mask;

	return(pixmap);
}
