/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "img-view.h"

#include "collect.h"
#include "dnd.h"
#include "filelist.h"
#include "fullscreen.h"
#include "image.h"
#include "info.h"
#include "menu.h"
#include "slideshow.h"
#include "utilops.h"
#include "ui_fileops.h"
#include "ui_menu.h"

#include <gdk/gdkkeysyms.h> /* for keyboard values */

#include "icons/view.xpm"


typedef struct _ViewWindow ViewWindow;
struct _ViewWindow
{
	GtkWidget *window;
	ImageWindow *imd;
	FullScreenData *fs;
	SlideShowData *ss;

	GList *list;
};


static GList *view_window_list = NULL;


static GtkWidget *view_popup_menu(ViewWindow *vw);
static void view_fullscreen_toggle(ViewWindow *vw, gint force_off);

static void view_slideshow_next(ViewWindow *vw);
static void view_slideshow_prev(ViewWindow *vw);
static void view_slideshow_start(ViewWindow *vw);
static void view_slideshow_stop(ViewWindow *vw);

static void view_window_close(ViewWindow *vw);

static void view_window_dnd_init(ViewWindow *vw);


/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */ 

static ImageWindow *view_window_active_image(ViewWindow *vw)
{
	if (vw->fs) return vw->fs->imd;

	return vw->imd;
}

static GList *path_list_copy(GList *list)
{
	GList *new_list = NULL;
	GList *work;

	work = list;
	while (work)
		{
		gchar *path;

		path = work->data;
		work = work->next;

		new_list = g_list_prepend(new_list, g_strdup(path));
		}

	return g_list_reverse(new_list);
}

static void view_window_set_list(ViewWindow *vw, GList *list)
{

	path_list_free(vw->list);
	vw->list = NULL;

	vw->list = path_list_copy(list);
}

static gint view_window_contains_collection(ViewWindow *vw)
{
	CollectionData *cd;
	CollectInfo *info;

	cd = image_get_collection(view_window_active_image(vw), &info);

	return (cd && info);
}

static void view_collection_step(ViewWindow *vw, gint next)
{
	ImageWindow *imd = view_window_active_image(vw);
	CollectionData *cd;
	CollectInfo *info;
	CollectInfo *read_ahead_info = NULL;

	cd = image_get_collection(imd, &info);

	if (!cd || !info) return;

	if (next)
		{
		info = collection_next_by_info(cd, info);
		if (enable_read_ahead)
			{
			read_ahead_info = collection_next_by_info(cd, info);
			if (!read_ahead_info) read_ahead_info = collection_prev_by_info(cd, info);
			}
		}
	else
		{
		info = collection_prev_by_info(cd, info);
		if (enable_read_ahead)
			{
			read_ahead_info = collection_prev_by_info(cd, info);
			if (!read_ahead_info) read_ahead_info = collection_next_by_info(cd, info);
			}
		}

	if (info)
		{
		image_change_from_collection(imd, cd, info, image_zoom_get_default(imd, zoom_mode));

		if (read_ahead_info) image_prebuffer_set(imd, read_ahead_info->path);
		}
	
}

static void view_collection_step_to_end(ViewWindow *vw, gint last)
{
	ImageWindow *imd = view_window_active_image(vw);
	CollectionData *cd;
	CollectInfo *info;
	CollectInfo *read_ahead_info = NULL;

	cd = image_get_collection(imd, &info);

	if (!cd || !info) return;

	if (last)
		{
		info = collection_get_last(cd);
		if (enable_read_ahead) read_ahead_info = collection_prev_by_info(cd, info);
		}
	else
		{
		info = collection_get_first(cd);
		if (enable_read_ahead) read_ahead_info = collection_next_by_info(cd, info);
		}

	if (info)
		{
		image_change_from_collection(imd, cd, info, image_zoom_get_default(imd, zoom_mode));
		if (read_ahead_info) image_prebuffer_set(imd, read_ahead_info->path);
		}
	
}

static void view_list_step(ViewWindow *vw, gint next)
{
	ImageWindow *imd = view_window_active_image(vw);
	const gchar *path;
	GList *work;
	GList *work_ahead;
	gint found;

	if (!vw->list) return;

	path = image_get_path(imd);
	if (!path) return;

	found = FALSE;
	work = vw->list;
	while (work && !found)
		{
		gchar *temp;

		temp = work->data;

		if (strcmp(path, temp) == 0)
			{
			found = TRUE;
			}
		else
			{
			work = work->next;
			}
		}
	if (!work) return;

	work_ahead = NULL;
	if (next)
		{
		work = work->next;
		if (work) work_ahead = work->next;
		}
	else
		{
		work = work->prev;
		if (work) work_ahead = work->prev;
		}
	if (!work) return;

	path = work->data;
	image_change_path(imd, path, image_zoom_get_default(imd, zoom_mode));

	if (enable_read_ahead && work_ahead)
		{
		const gchar *next_path = work_ahead->data;
		image_prebuffer_set(imd, next_path);
		}
}

static void view_list_step_to_end(ViewWindow *vw, gint last)
{
	ImageWindow *imd = view_window_active_image(vw);
	const gchar *path;
	GList *work;
	GList *work_ahead;

	if (!vw->list) return;

	if (last)
		{
		work = g_list_last(vw->list);
		work_ahead = work->prev;
		}
	else
		{
		work = vw->list;
		work_ahead = work->next;
		}

	path = work->data;
	image_change_path(imd, path, image_zoom_get_default(imd, zoom_mode));

	if (enable_read_ahead && work_ahead)
		{
		const gchar *next_path = work_ahead->data;
		image_prebuffer_set(imd, next_path);
		}
}

static void view_step_next(ViewWindow *vw)
{
	if (vw->ss)
		{
		view_slideshow_next(vw);
		}
	else if (vw->list)
		{
		view_list_step(vw, TRUE);
		}
	else
		{
		view_collection_step(vw, TRUE);
		}
}

static void view_step_prev(ViewWindow *vw)
{
	if (vw->ss)
		{
		view_slideshow_prev(vw);
		}
	else if (vw->list)
		{
		view_list_step(vw, FALSE);
		}
	else
		{
		view_collection_step(vw, FALSE);
		}
}

static void view_step_to_end(ViewWindow *vw, gint last)
{
	if (vw->list)
		{
		view_list_step_to_end(vw, last);
		}
	else
		{
		view_collection_step_to_end(vw, last);
		}
}

/*
 *-----------------------------------------------------------------------------
 * view window keyboard
 *-----------------------------------------------------------------------------
 */

static gint view_window_key_press_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	ViewWindow *vw = data;
	ImageWindow *imd;
	gint stop_signal = FALSE;
	gint x = 0;
	gint y = 0;

	imd = view_window_active_image(vw);

	if ( !(event->state & GDK_CONTROL_MASK) )
	    switch (event->keyval)
		{
		case GDK_Left: case GDK_KP_Left:
			x -= 1;
			stop_signal = TRUE;
			break;
		case GDK_Right: case GDK_KP_Right:
			x += 1;
			stop_signal = TRUE;
			break;
		case GDK_Up: case GDK_KP_Up:
			y -= 1;
			stop_signal = TRUE;
			break;
		case GDK_Down: case GDK_KP_Down:
			y += 1;
			stop_signal = TRUE;
			break;
		case GDK_Page_Up: case GDK_KP_Page_Up:
		case GDK_BackSpace:
		case 'B': case 'b':
			view_step_prev(vw);
			stop_signal = TRUE;
			break;
		case GDK_Page_Down: case GDK_KP_Page_Down:
		case GDK_space:
			view_step_next(vw);
			stop_signal = TRUE;
			break;
		case GDK_Home: case GDK_KP_Home:
			view_step_to_end(vw, FALSE);
			stop_signal = TRUE;
			break;
		case GDK_End: case GDK_KP_End:
			view_step_to_end(vw, TRUE);
			stop_signal = TRUE;
			break;
		case '+': case '=': case GDK_KP_Add:
			image_zoom_adjust(imd, get_zoom_increment());
			break;
		case '-': case GDK_KP_Subtract:
			image_zoom_adjust(imd, -get_zoom_increment());
			break;
		case 'X': case 'x': case GDK_KP_Multiply:
			image_zoom_set(imd, 0.0);
			break;
		case 'Z': case 'z': case GDK_KP_Divide: case '1':
			image_zoom_set(imd, 1.0);
			break;
		case '2':
			image_zoom_set(imd, 2.0);
			break;
		case '3':
			image_zoom_set(imd, 3.0);
			break;
		case '4':
			image_zoom_set(imd, 4.0);
			break;
		case '7':
			image_zoom_set(imd, -4.0);
			break;
		case '8':
			image_zoom_set(imd, -3.0);
			break;
		case '9':
			image_zoom_set(imd, -2.0);
			break;
		case 'S': case 's':
			if (vw->ss)
				view_slideshow_stop(vw);
			else
				view_slideshow_start(vw);
			stop_signal = TRUE;
			break;
		case 'P': case 'p':
			slideshow_pause_toggle(vw->ss);
			break;
		case 'V': case 'v':
			view_fullscreen_toggle(vw, FALSE);
			stop_signal = TRUE;
			break;
		case ']':
			image_alter(imd, ALTER_ROTATE_90);
			stop_signal = TRUE;
			break;
		case '[':
			image_alter(imd, ALTER_ROTATE_90_CC);
			stop_signal = TRUE;
			break;
		case GDK_Delete: case GDK_KP_Delete:
			if (enable_delete_key)
				{
				view_fullscreen_toggle(vw, TRUE);
				file_util_delete(image_get_path(vw->imd), NULL);
				stop_signal = TRUE;
				}
			break;
		case GDK_Escape:
			if (vw->fs)
				{
				view_fullscreen_toggle(vw, TRUE);
				}
			else
				{
				gtk_widget_destroy(vw->window);
				}
			stop_signal = TRUE;
			break;
		}

	if (event->state & GDK_CONTROL_MASK)
		{
		gint n = -1;
		switch (event->keyval)
			{
			case '1':
				n = 0;
				break;
			case '2':
				n = 1;
				break;
			case '3':
				n = 2;
				break;
			case '4':
				n = 3;
				break;
			case '5':
				n = 4;
				break;
			case '6':
				n = 5;
				break;
			case '7':
				n = 6;
				break;
			case '8':
				n = 7;
				break;
			case 'C': case 'c':
				view_fullscreen_toggle(vw, TRUE);
				file_util_copy(image_get_path(vw->imd), NULL, NULL);
				stop_signal = TRUE;
				break;
			case 'M': case 'm':
				view_fullscreen_toggle(vw, TRUE);
				file_util_move(image_get_path(vw->imd), NULL, NULL);
				stop_signal = TRUE;
				break;
			case 'R': case 'r':
				view_fullscreen_toggle(vw, TRUE);
				file_util_rename(image_get_path(vw->imd), NULL);
				stop_signal = TRUE;
				break;
			case 'D': case 'd':
				view_fullscreen_toggle(vw, TRUE);
				file_util_delete(image_get_path(vw->imd), NULL);
				stop_signal = TRUE;
				break;
			case 'P': case 'p':
				view_fullscreen_toggle(vw, TRUE);
				info_window_new(image_get_path(vw->imd), NULL);
				stop_signal = TRUE;
				break;
			case 'W': case 'w':
				view_window_close(vw);
				break;
			}
		if (n != -1)
			{
			view_fullscreen_toggle(vw, TRUE);
			start_editor_from_file(n, image_get_path(imd));
			}
		}
	else if (event->state & GDK_SHIFT_MASK)
		{
		switch (event->keyval)
			{
			case 'R': case 'r':
				image_alter(imd, ALTER_ROTATE_180);
				stop_signal = TRUE;
				break;
			case 'M': case 'm':
				image_alter(imd, ALTER_MIRROR);
				stop_signal = TRUE;
				break;
			case 'F': case 'f':
				image_alter(imd, ALTER_FLIP);
				stop_signal = TRUE;
				break;
			default:
				break;
			}
		x *= 3;
		y *= 3;
		}

	if (x != 0 || y!= 0)
		{
		keyboard_scroll_calc(&x, &y, event);
		image_scroll(imd, x, y);
		}

	return stop_signal;
}

/*
 *-----------------------------------------------------------------------------
 * view window main routines
 *-----------------------------------------------------------------------------
 */ 

static void button1_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	ViewWindow *vw = data;

	view_step_next(vw);
}

static void button2_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	ViewWindow *vw = data;

	view_step_prev(vw);
}

static void button3_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	ViewWindow *vw = data;
	GtkWidget *menu;

	menu = view_popup_menu(vw);
	gtk_menu_popup (GTK_MENU(menu), NULL, NULL, NULL, NULL,
		bevent->button, bevent->time);
}

static void button4_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	ViewWindow *vw = data;

	if (bevent->state & GDK_CONTROL_MASK)
		{
		image_zoom_adjust_at_point(imd, get_zoom_increment(), bevent->x, bevent->y);
		}
	else if ( (bevent->state & GDK_SHIFT_MASK) != (mousewheel_scrolls))
		{
		image_scroll(imd, 0, -MOUSEWHEEL_SCROLL_SIZE);
		}
	else
		{
		view_step_prev(vw);
		}
}

static void button5_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	ViewWindow *vw = data;

	if (bevent->state & GDK_CONTROL_MASK)
		{
		image_zoom_adjust_at_point(imd, -get_zoom_increment(), bevent->x, bevent->y);
		}
	else if ( (bevent->state & GDK_SHIFT_MASK) != (mousewheel_scrolls))
		{
		image_scroll(imd, 0, MOUSEWHEEL_SCROLL_SIZE);
		}
	else
		{
		view_step_next(vw);
		}
}

static void view_image_set_buttons(ViewWindow *vw, ImageWindow *imd)
{
	image_set_button_func(imd, 1, button1_cb, vw);
	image_set_button_func(imd, 2, button2_cb, vw);
	image_set_button_func(imd, 3, button3_cb, vw);
	/* for wheel mice */
	image_set_button_func(imd, 4, button4_cb, vw);
	image_set_button_func(imd, 5, button5_cb, vw);
}

static void view_fullscreen_stop_func(FullScreenData *fs, gpointer data)
{
	ViewWindow *vw = data;

	vw->fs = NULL;

	if (vw->ss) vw->ss->imd = vw->imd;
}

static void view_fullscreen_toggle(ViewWindow *vw, gint force_off)
{
	if (force_off && !vw->fs) return;

	if (vw->fs)
		{
		fullscreen_stop(vw->fs);
		}
	else
		{
		vw->fs = fullscreen_start(vw->window, vw->imd, view_fullscreen_stop_func, vw);

		view_image_set_buttons(vw, vw->fs->imd);
		gtk_signal_connect(GTK_OBJECT(vw->fs->window), "key_press_event",
				   GTK_SIGNAL_FUNC(view_window_key_press_cb), vw);

		if (vw->ss) vw->ss->imd = vw->fs->imd;
		}
}

static void view_slideshow_next(ViewWindow *vw)
{
	if (vw->ss) slideshow_next(vw->ss);
}

static void view_slideshow_prev(ViewWindow *vw)
{
	if (vw->ss) slideshow_prev(vw->ss);
}

static void view_slideshow_stop_func(SlideShowData *fs, gpointer data)
{
	ViewWindow *vw = data;

	vw->ss = NULL;
}

static void view_slideshow_start(ViewWindow *vw)
{
	if (!vw->ss)
		{
		CollectionData *cd;
		CollectInfo *info;

		if (vw->list)
			{
			vw->ss = slideshow_start_from_path_list(view_window_active_image(vw),
								path_list_copy(vw->list),
								view_slideshow_stop_func, vw);
			return;
			}

		cd = image_get_collection(view_window_active_image(vw), &info);
		if (cd && info)
			{
			vw->ss = slideshow_start_from_collection(view_window_active_image(vw), cd,
								 view_slideshow_stop_func, vw, info);
			}
		}
}

static void view_slideshow_stop(ViewWindow *vw)
{
	if (vw->ss) slideshow_free(vw->ss);
}

static void view_window_close(ViewWindow *vw)
{
	view_window_list = g_list_remove(view_window_list, vw);

	view_slideshow_stop(vw);
	view_fullscreen_toggle(vw, TRUE);
	gtk_widget_destroy(vw->window);
	path_list_free(vw->list);
	g_free(vw);
}

static gint view_window_delete_cb(GtkWidget *w, GdkEventAny *event, gpointer data)
{
	ViewWindow *vw = data;

	view_window_close(vw);
	return TRUE;
}

static void real_view_window_new(const gchar *path, GList *list, CollectionData *cd, CollectInfo *info)
{
	ViewWindow *vw;
	GtkAllocation req_size;
	gint w, h;

	if (!path && !list && (!cd || !info)) return;

	vw = g_new0(ViewWindow, 1);
	vw->fs = NULL;
	vw->ss = NULL;
	vw->list = NULL;

	vw->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(vw->window), TRUE, TRUE, FALSE);
	gtk_window_set_title (GTK_WINDOW (vw->window), "GQview");
        gtk_window_set_wmclass(GTK_WINDOW (vw->window), "view", "GQview");
        gtk_container_border_width (GTK_CONTAINER (vw->window), 0);

	window_set_icon(vw->window, (const gchar **)view_xpm, NULL);

	vw->imd = image_new(FALSE);

	if (black_window_background) image_background_set_black(vw->imd, TRUE);

	image_attach_window(vw->imd, vw->window, NULL, "GQview", TRUE);

	image_top_window_set_sync(vw->imd, TRUE);

	gtk_container_add(GTK_CONTAINER(vw->window), vw->imd->widget);
	gtk_widget_show(vw->imd->widget);

	view_window_dnd_init(vw);

	view_image_set_buttons(vw, vw->imd);

	gtk_signal_connect(GTK_OBJECT(vw->window), "delete_event",
			   GTK_SIGNAL_FUNC(view_window_delete_cb), vw);
	gtk_signal_connect(GTK_OBJECT(vw->window), "key_press_event",
			   GTK_SIGNAL_FUNC(view_window_key_press_cb), vw);
	if (cd && info)
		{
		image_change_from_collection(vw->imd, cd, info, image_zoom_get_default(NULL, zoom_mode));
		if (enable_read_ahead)
			{
			CollectInfo * r_info = collection_next_by_info(cd, info);
			if (!r_info) r_info = collection_prev_by_info(cd, info);
			if (r_info) image_prebuffer_set(vw->imd, r_info->path);
			}
		}
	else if (list)
		{
		view_window_set_list(vw, list);
		image_change_path(vw->imd, (gchar *)vw->list->data, image_zoom_get_default(NULL, zoom_mode));
		if (enable_read_ahead)
			{
			GList *work = vw->list->next;
			if (work) image_prebuffer_set(vw->imd, (gchar *)work->data);
			}
		}
	else
		{
		image_change_path(vw->imd, path, image_zoom_get_default(NULL, zoom_mode));
		}

	if (image_zoom_get(vw->imd) == 0.0)
		{
		w = vw->imd->image_width;
		h = vw->imd->image_height;
		}
	else
		{
		w = vw->imd->width;
		h = vw->imd->height;
		}
	if (limit_window_size)
		{
		gint mw = gdk_screen_width() * max_window_size / 100;
		gint mh = gdk_screen_height() * max_window_size / 100;

		if (w > mw) w = mw;
		if (h > mh) h = mh;
		}

	gtk_window_set_default_size (GTK_WINDOW(vw->window), w, h);
	req_size.x = req_size.y = 0;
	req_size.width = w;
	req_size.height = h;
	gtk_widget_size_allocate(GTK_WIDGET(vw->window), &req_size);

	gtk_widget_set_usize(vw->imd->eventbox, w, h);

	gtk_widget_show(vw->window);

	view_window_list = g_list_append(view_window_list, vw);
}

void view_window_new(const gchar *path)
{
	real_view_window_new(path, NULL, NULL, NULL);
}

void view_window_new_from_list(GList *list)
{
	real_view_window_new(NULL, list, NULL, NULL);
}

void view_window_new_from_collection(CollectionData *cd, CollectInfo *info)
{
	real_view_window_new(NULL, NULL, cd, info);
}

/*
 *-----------------------------------------------------------------------------
 * public
 *-----------------------------------------------------------------------------
 */

void view_window_colors_update(void)
{
	GList *work;

	work = view_window_list;
	while (work)
		{
		ViewWindow *vw = work->data;
		work = work->next;

		image_background_set_black(vw->imd, black_window_background);
		}
}

/*
 *-----------------------------------------------------------------------------
 * view window menu routines and callbacks
 *-----------------------------------------------------------------------------
 */ 

static void view_new_window_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;
	CollectionData *cd;
	CollectInfo *info;

	cd = image_get_collection(vw->imd, &info);

	if (cd && info)
		{
		view_window_new_from_collection(cd, info);
		}
	else
		{
		view_window_new(image_get_path(vw->imd));
		}
}

static void view_edit_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw;
	gint n;

	vw = submenu_item_get_data(widget);
	n = GPOINTER_TO_INT(data);
	if (!vw) return;

	view_fullscreen_toggle(vw, TRUE);
	start_editor_from_file(n, image_get_path(vw->imd));
}

static void view_alter_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw;
	AlterType type;

	vw = submenu_item_get_data(widget);
	type = GPOINTER_TO_INT(data);

	if (!vw) return;
	image_alter(vw->imd, type);
}

static void view_info_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	info_window_new(image_get_path(vw->imd), NULL);
}

static void view_wallpaper_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;
	ImageWindow *imd;

	imd = view_window_active_image(vw);

	image_to_root_window(imd, (image_zoom_get(imd) == 0.0));
}

static void view_zoom_in_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	image_zoom_adjust(view_window_active_image(vw), get_zoom_increment());
}

static void view_zoom_out_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	image_zoom_adjust(view_window_active_image(vw), -get_zoom_increment());
}

static void view_zoom_1_1_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	image_zoom_set(view_window_active_image(vw), 1.0);
}

static void view_zoom_fit_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	image_zoom_set(view_window_active_image(vw), 0.0);
}

static void view_copy_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	file_util_copy(image_get_path(vw->imd), NULL, NULL);
}

static void view_move_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	file_util_move(image_get_path(vw->imd), NULL, NULL);
}

static void view_rename_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	file_util_rename(image_get_path(vw->imd), NULL);
}

static void view_delete_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, TRUE);
	file_util_delete(image_get_path(vw->imd), NULL);
}

static void view_fullscreen_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_fullscreen_toggle(vw, FALSE);
}

static void view_slideshow_start_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_slideshow_start(vw);
}

static void view_slideshow_stop_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_slideshow_stop(vw);
}

static void view_slideshow_pause_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	slideshow_pause_toggle(vw->ss);
}

static void view_close_cb(GtkWidget *widget, gpointer data)
{
	ViewWindow *vw = data;

	view_window_close(vw);
}

static GtkWidget *view_popup_menu(ViewWindow *vw)
{
	GtkWidget *menu;
	GtkWidget *item;

	menu = popup_menu_short_lived();

	menu_item_add(menu, _("Zoom in"), GTK_SIGNAL_FUNC(view_zoom_in_cb), vw);
	menu_item_add(menu, _("Zoom out"), GTK_SIGNAL_FUNC(view_zoom_out_cb), vw);
	menu_item_add(menu, _("Zoom 1:1"), GTK_SIGNAL_FUNC(view_zoom_1_1_cb), vw);
	menu_item_add(menu, _("Fit image to window"), GTK_SIGNAL_FUNC(view_zoom_fit_cb), vw);
	menu_item_add_divider(menu);

	item = submenu_add_edit(menu, NULL, GTK_SIGNAL_FUNC(view_edit_cb), vw);
	menu_item_add_divider(item);
	menu_item_add(item, _("Set as wallpaper"), GTK_SIGNAL_FUNC(view_wallpaper_cb), vw);

	submenu_add_alter(menu, GTK_SIGNAL_FUNC(view_alter_cb), vw);

	menu_item_add(menu, _("Properties"), GTK_SIGNAL_FUNC(view_info_cb), vw);

	menu_item_add(menu, _("View in new window"), GTK_SIGNAL_FUNC(view_new_window_cb), vw);

	menu_item_add_divider(menu);
	menu_item_add(menu, _("Copy..."), GTK_SIGNAL_FUNC(view_copy_cb), vw);
	menu_item_add(menu, _("Move..."), GTK_SIGNAL_FUNC(view_move_cb), vw);
	menu_item_add(menu, _("Rename..."), GTK_SIGNAL_FUNC(view_rename_cb), vw);
	menu_item_add(menu, _("Delete..."), GTK_SIGNAL_FUNC(view_delete_cb), vw);

	menu_item_add_divider(menu);

	if (vw->ss)
		{
		menu_item_add(menu, _("Stop slideshow"), GTK_SIGNAL_FUNC(view_slideshow_stop_cb), vw);
		if (slideshow_paused(vw->ss))
			{
			item = menu_item_add(menu, _("Continue slideshow"),
					     GTK_SIGNAL_FUNC(view_slideshow_pause_cb), vw);
			}
		else
			{
			item = menu_item_add(menu, _("Pause slideshow"),
					     GTK_SIGNAL_FUNC(view_slideshow_pause_cb), vw);
			}
		}
	else
		{
		item = menu_item_add(menu, _("Start slideshow"), GTK_SIGNAL_FUNC(view_slideshow_start_cb), vw);
		gtk_widget_set_sensitive(item, (vw->list != NULL) || view_window_contains_collection(vw));
		item = menu_item_add(menu, _("Pause slideshow"), GTK_SIGNAL_FUNC(view_slideshow_pause_cb), vw);
		gtk_widget_set_sensitive(item, FALSE);
		}

	if (vw->fs)
		{
		menu_item_add(menu, _("Exit full screen"), GTK_SIGNAL_FUNC(view_fullscreen_cb), vw);
		}
	else
		{
		menu_item_add(menu, _("Full screen"), GTK_SIGNAL_FUNC(view_fullscreen_cb), vw);
		}

	menu_item_add_divider(menu);
	menu_item_add(menu, _("Close window"), GTK_SIGNAL_FUNC(view_close_cb), vw);

	return menu;
}

/*
 *-------------------------------------------------------------------
 * dnd confirm dir
 *-------------------------------------------------------------------
 */

typedef struct {
	ViewWindow *vw;
	GList *list;
} CViewConfirmD;

static void view_dir_list_cancel(GtkWidget *widget, gpointer data)
{
	/* do nothing */
}

static void view_dir_list_do(ViewWindow *vw, GList *list, gint skip, gint recurse)
{
	GList *work;

	view_window_set_list(vw, NULL);

	work = list;
	while (work)
		{
		gchar *path = work->data;
		work = work->next;

		if (isdir(path))
			{
			if (!skip)
				{
				GList *list = NULL;

				if (recurse)
					{
					list = path_list_recursive(path);
					}
				else
					{
					path_list(path, &list, NULL);
					list = path_list_sort(list);
					list = path_list_filter(list, FALSE);
					}
				if (list) vw->list = g_list_concat(vw->list, list);
				}
			}
		else
			{
			/* FIXME: no filtering here */
			vw->list = g_list_append(vw->list, g_strdup(path));
			}
		}

	if (vw->list)
		{
		gchar *path;

		path = vw->list->data;
		image_change_path(vw->imd, path, image_zoom_get_default(vw->imd, zoom_mode));

		work = vw->list->next;
		if (enable_read_ahead && work)
			{
			path = work->data;
			image_prebuffer_set(vw->imd, path);
			}
		}
	else
		{
		image_change_path(vw->imd, NULL, image_zoom_get_default(vw->imd, zoom_mode));
		}
}

static void view_dir_list_add(GtkWidget *widget, gpointer data)
{
	CViewConfirmD *d = data;
	view_dir_list_do(d->vw, d->list, FALSE, FALSE);
}

static void view_dir_list_recurse(GtkWidget *widget, gpointer data)
{
	CViewConfirmD *d = data;
	view_dir_list_do(d->vw, d->list, FALSE, TRUE);
}

static void view_dir_list_skip(GtkWidget *widget, gpointer data)
{
	CViewConfirmD *d = data;
	view_dir_list_do(d->vw, d->list, TRUE, FALSE);
}

static void view_dir_list_destroy(GtkWidget *widget, gpointer data)
{
	CViewConfirmD *d = data;
        path_list_free(d->list);
        g_free(d);
}

static GtkWidget *view_confirm_dir_list(ViewWindow *vw, GList *list)
{
	GtkWidget *menu;
	CViewConfirmD *d;

	d = g_new(CViewConfirmD, 1);
	d->vw = vw;
	d->list = list;

	menu = popup_menu_short_lived();
	gtk_signal_connect(GTK_OBJECT(menu), "destroy",
			   GTK_SIGNAL_FUNC(view_dir_list_destroy), d);

	menu_item_add(menu, _("Dropped list includes directories."), NULL, NULL);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Add contents"), GTK_SIGNAL_FUNC(view_dir_list_add), d);
	menu_item_add(menu, _("Add contents recursive"), GTK_SIGNAL_FUNC(view_dir_list_recurse), d);
	menu_item_add(menu, _("Skip directories"), GTK_SIGNAL_FUNC(view_dir_list_skip), d);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Cancel"), GTK_SIGNAL_FUNC(view_dir_list_cancel), d);

	return menu;
}

/*
 *-----------------------------------------------------------------------------
 * image drag and drop routines
 *-----------------------------------------------------------------------------
 */

static void view_window_get_dnd_data(GtkWidget *widget, GdkDragContext *context,
				     gint x, gint y,
				     GtkSelectionData *selection_data, guint info,
				     guint time, gpointer data)
{
	ViewWindow *vw = data;
	ImageWindow *imd;

	if (gtk_drag_get_source_widget(context) == vw->imd->image) return;

	imd = vw->imd;

	if (info == TARGET_URI_LIST || info == TARGET_APP_COLLECTION_MEMBER)
		{
		CollectionData *source;
		GList *list;
		GList *info_list;

		if (info == TARGET_URI_LIST)
			{
			GList *work;

			list = get_uri_file_list(selection_data->data);

			work = list;
			while (work)
				{
				if (isdir((gchar *)work->data))
					{
					GtkWidget *menu;
					menu = view_confirm_dir_list(vw, list);
					gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, time);
					return;
					}
				work = work->next;
				}

			list = path_list_filter(list, FALSE);

			source = NULL;
			info_list = NULL;
			}
		else
			{
			source = collection_from_dnd_data((gchar *)selection_data->data, &list, &info_list);
			}

		if (list)
			{
			gchar *path;

			path = list->data;
			if (isfile(path))
				{
				view_slideshow_stop(vw);
				view_window_set_list(vw, NULL);
				if (source && info_list)
					{
					image_change_from_collection(imd, source, info_list->data, image_zoom_get_default(imd, zoom_mode));
					}
				else
					{
					if (list->next)
						{
						vw->list = list;
						list = NULL;
						}
					image_change_path(imd, path, image_zoom_get_default(imd, zoom_mode));
					}
				}
			}
		path_list_free(list);
		g_list_free(info_list);
		}
}

static void view_window_set_dnd_data(GtkWidget *widget, GdkDragContext *context,
				     GtkSelectionData *selection_data, guint info,
				     guint time, gpointer data)
{
	ViewWindow *vw = data;
	const gchar *path;

	path = image_get_path(vw->imd);

	if (path)
		{
		gchar *text = NULL;
		switch (info)
			{
			case TARGET_URI_LIST:
				text = g_strconcat("file:", path, "\r\n", NULL);
				break;
			case TARGET_TEXT_PLAIN:
				text = g_strdup(path);
				break;
			}
		if (text)
			{
			gtk_selection_data_set (selection_data, selection_data->target,
						8, text, strlen(text));
			g_free(text);
			}
		}
	else
		{
		gtk_selection_data_set (selection_data, selection_data->target,
					8, NULL, 0);
		}
}

static void view_window_dnd_init(ViewWindow *vw)
{
	ImageWindow *imd;

	imd = vw->imd;

	gtk_drag_source_set(imd->image, GDK_BUTTON2_MASK,
			    dnd_file_drag_types, dnd_file_drag_types_count,
			    GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_signal_connect(GTK_OBJECT(imd->image), "drag_data_get",
			   GTK_SIGNAL_FUNC(view_window_set_dnd_data), vw);

	gtk_drag_dest_set(imd->image,
			  GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
			  dnd_file_drop_types, dnd_file_drop_types_count,
                          GDK_ACTION_COPY);
	gtk_signal_connect(GTK_OBJECT(imd->image), "drag_data_received",
			   GTK_SIGNAL_FUNC(view_window_get_dnd_data), vw);
}

/*
 *-----------------------------------------------------------------------------
 * maintenance (for rename, move, remove)
 *-----------------------------------------------------------------------------
 */

static void view_real_removed(ViewWindow *vw, const gchar *path, GList *ignore_list)
{
	ImageWindow *imd;
	const gchar *image_path;

	imd = view_window_active_image(vw);
	image_path = image_get_path(imd);

	if (image_path && strcmp(image_path, path) == 0)
		{
		if (vw->list)
			{
			view_list_step(vw, TRUE);
			if (image_get_path(imd) == image_path)
				{
				view_list_step(vw, FALSE);
				}
			}
		else if (view_window_contains_collection(vw))
			{
			view_collection_step(vw, TRUE);
			if (image_get_path(imd) == image_path)
				{
				view_collection_step(vw, FALSE);
				}
			}
		if (image_get_path(imd) == image_path)
			{
			image_change_path(imd, NULL, image_zoom_get_default(imd, zoom_mode));
			}
		}

	if (vw->list)
		{
		GList *work;
		work = vw->list;
		while (work)
			{
			gchar *chk_path;

			chk_path = work->data;
			work = work->next;

			if (strcmp(chk_path, path) == 0)
				{
				vw->list = g_list_remove(vw->list, chk_path);
				g_free(chk_path);
				}
			}
		}
}

static void view_real_moved(ViewWindow *vw, const gchar *source, const gchar *dest)
{
	ImageWindow *imd;
	const gchar *image_path;

	imd = view_window_active_image(vw);
	image_path = image_get_path(imd);

	if (image_path && strcmp(image_path, source) == 0)
		{
		image_set_path(imd, dest);
		}

	if (vw->list)
		{
		GList *work;
		work = vw->list;
		while (work)
			{
			gchar *chk_path;

			chk_path = work->data;

			if (strcmp(chk_path, source) == 0)
				{
				work->data = g_strdup(dest);
				g_free(chk_path);
				}

			work = work->next;
			}
		}
}

void view_window_maint_removed(const gchar *path, GList *ignore_list)
{
	GList *work = view_window_list;
	while (work)
		{
		ViewWindow *vw = work->data;
		work = work->next;

		view_real_removed(vw, path, ignore_list);
		}
}

void view_window_maint_moved(const gchar *source, const gchar *dest)
{
	GList *work = view_window_list;
	while (work)
		{
		ViewWindow *vw = work->data;
		work = work->next;

		view_real_moved(vw, source, dest);
		}
}

