; "fhilbert.scm": Hilbert's space filling mapping
; Copyright (c) 2003 Aubrey Jaffer
;
;Permission to copy this software, to modify it, to redistribute it,
;to distribute modified versions, and to use it for any purpose is
;granted, subject to the following restrictions and understandings.
;
;1.  Any copy made of this software must include this copyright notice
;in full.
;
;2.  I have made no warrantee or representation that the operation of
;this software will be error-free, and I am under no obligation to
;provide any services, by way of maintenance, update, or otherwise.
;
;3.  In conjunction with products arising from the use of this
;material, there shall be no use of my name in any advertising,
;promotional, or sales literature without prior written consent in
;each case.

(require 'logical)

;;@code{(require 'hilbert-fill)}
;;@ftindex hilbert-fill
;;
;;@noindent
;;@cindex Hilbert
;;@cindex Space-Filling
;;The @dfn{Hilbert Space-Filling Curve} is a one-to-one mapping
;;between a unit line segment and an @var{n}-dimensional unit cube.
;;
;;@noindent
;;The integer procedures map the non-negative integers to an
;;arbitrarily large @var{n}-dimensional cube with its corner at the
;;origin and all coordinates are non-negative.
;;
;;@noindent
;;For any exact nonnegative integers @var{scalar} and @var{rank},
;;
;;@example
;;(= @var{scalar} (hilbert-coordinates->integer
;;           (integer->hilbert-coordinates @var{scalar} @var{rank})))
;;                                       @result{} #t
;;@end example

;;@body 
;;Returns a list of @2 integer coordinates corresponding to exact
;;non-negative integer @1.  The lists returned by @0 for @1 arguments
;;0 and 1 will differ in the first element.
(define (integer->hilbert-coordinates scalar rank)
  (define bignum 0)
  (define flipBit 0)
  (define ndOnes (logical:ones rank))
  (define rank*nBits
    (let ((rank^2 (* rank rank)))
      (* (quotient (+ -1 rank^2 (integer-length scalar)) rank^2)
	 rank^2)))
  (let ((nthbits (quotient (logical:ones rank*nBits) ndOnes)))
    (define igry (logxor (integer->gray-code scalar) (ash nthbits -1)))
    (do ((bdxn (- rank rank*nBits) (+ rank bdxn))
	 (chnk (logand (ash igry (- rank rank*nBits)) ndOnes)
	       (logand (ash igry (+ rank bdxn)) ndOnes))
	 (rotation 0 (modulo (+ (integer-length (logand (- chnk) chnk))
				1 rotation)
			     rank)))
	((positive? bdxn))
      (set! bignum (+ (logxor flipBit (logical:rotate chnk rotation rank))
		      (ash bignum rank)))
      (set! flipBit (ash 1 rotation))))
  (map gray-code->integer (bitwise:delaminate rank bignum)))

;;@body
;;Returns an exact non-negative integer corresponding to @1, a list
;;of non-negative integer coordinates.
(define (hilbert-coordinates->integer coords)
  (define rank (length coords))
  (define bignum (apply bitwise:laminate (map integer->gray-code coords)))
  (let ((rank*nBits
	 (* (quotient (+ -1 rank (integer-length (apply max coords))) rank)
	    rank rank))
	(ndOnes (logical:ones rank))
	(rotation 0)
	(flipBit 0)
	(scalar 0))
    (define nthbits (quotient (logical:ones rank*nBits) ndOnes))
    (do ((bdxn (- rank rank*nBits) (+ rank bdxn)))
	((positive? bdxn))
      (let ((chnk (logical:rotate
		   (logxor flipBit (logand ndOnes (ash bignum bdxn)))
		   (- rotation)
		   rank)))
	(set! scalar (+ chnk (ash scalar rank)))
	(set! flipBit (ash 1 rotation))
	(set! rotation (modulo (+ (integer-length (logand (- chnk) chnk))
				  1 rotation)
			       rank))))
    (gray-code->integer (logxor scalar (ash nthbits -1)))))
