/*osTime.c

This file includes os specific code for timing related functionalities.

*/

/*
 * Copyright (c) 2001 Conexant Systems, Inc.
 * 
 * 1.   Permitted use. Redistribution and use in source and binary forms,
 * with or without modification, are permitted under the terms set forth
 * herein.
 * 
 * 2.   Disclaimer of Warranties. CONEXANT AND OTHER CONTRIBUTORS MAKE NO
 * REPRESENTATION ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.
 * IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTIES OF ANY KIND.
 * CONEXANT AND OTHER CONTRIBUTORS DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, GOOD TITLE AND AGAINST INFRINGEMENT.
 * 
 * This software has not been formally tested, and there is no guarantee that
 * it is free of errors including, but not limited to, bugs, defects,
 * interrupted operation, or unexpected results. Any use of this software is
 * at user's own risk.
 * 
 * 3.   No Liability.
 * 
 * (a) Conexant or contributors shall not be responsible for any loss or
 * damage to Company, its customers, or any third parties for any reason
 * whatsoever, and CONEXANT OR CONTRIBUTORS SHALL NOT BE LIABLE FOR ANY
 * ACTUAL, DIRECT, INDIRECT, SPECIAL, PUNITIVE, INCIDENTAL, OR CONSEQUENTIAL
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED, WHETHER IN CONTRACT, STRICT OR OTHER LEGAL THEORY OF
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY
 * WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 * 
 * (b) User agrees to hold Conexant and contributors harmless from any
 * liability, loss, cost, damage or expense, including attorney's fees,
 * as a result of any claims which may be made by any person, including
 * but not limited to User, its agents and employees, its customers, or
 * any third parties that arise out of or result from the manufacture,
 * delivery, actual or alleged ownership, performance, use, operation
 * or possession of the software furnished hereunder, whether such claims
 * are based on negligence, breach of contract, absolute liability or any
 * other legal theory.
 * 
 * 4.   Notices. User hereby agrees not to remove, alter or destroy any
 * copyright, trademark, credits, other proprietary notices or confidential
 * legends placed upon, contained within or associated with the Software,
 * and shall include all such unaltered copyright, trademark, credits,
 * other proprietary notices or confidential legends on or in every copy of
 * the Software.
 * 
 */
#include "oscompat.h"
#include <linux/timer.h>
#include <linux/time.h>
#include <linux/ptrace.h>
#include <linux/interrupt.h>
#include <asm/semaphore.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <asm/system.h>

#include "oscompat.h"
#include "ossysenv.h"
#include "ostime_ex.h"
#include "osmemory_ex.h"
#include "osstring_ex.h"

#define MSECS_TO_TICKS(msecs) (((msecs) && ((msecs) < 1000/HZ)) ? 1 : \
									(msecs)/(1000/HZ))
#define TICKS_TO_MSECS(ticks) (ticks*(1000/HZ))

typedef struct TIME_OUT_INSTANCE_TYPE
{
	struct tq_struct	TaskQueue;
	struct wait_queue	*WaitQueue;
	struct timer_list	Timer;
	BOOL                active;
	int					Count;

	UINT32			mSec;
	BOOL			bLocked;

	PFREE_FUNC		pFuncFree;
	PVOID			pRefData;

	PCBFUNC			pTimeOutCallBack;

}TIME_OUT_INSTANCE_T, *PTIME_OUT_INSTANCE_T;
 
typedef struct TIMER_TAG
{
	UINT32 msec;
	struct timer_list timer;
} TIMER_T, *PTIMER_T;

/********************************************************************/

STATIC	VOID TimerThreadFunction(PVOID pData)
{
	PTIME_OUT_INSTANCE_T	pTimeOutInstance = (PTIME_OUT_INSTANCE_T)pData;

	if (pTimeOutInstance->active == TRUE )
	{
		pTimeOutInstance->pTimeOutCallBack(pTimeOutInstance->pRefData);
		// callback might have set active to FALSE
		if (pTimeOutInstance->active == TRUE ) {
			mod_timer(&pTimeOutInstance -> Timer, jiffies + MSECS_TO_TICKS(pTimeOutInstance -> mSec));
		}
	}

	pTimeOutInstance->Count--;
	SCHEDULE_TASK_MOD_DEC_USE_COUNT;
}

STATIC VOID TimeOutHandler(	unsigned long Data )
{
	PTIME_OUT_INSTANCE_T	pTimeOutInstance = (PTIME_OUT_INSTANCE_T)Data;

	if (pTimeOutInstance->active == FALSE )
	{
		return;
	}

	SCHEDULE_TASK_MOD_INC_USE_COUNT;
	pTimeOutInstance->Count++;
	if ( schedule_task(&(pTimeOutInstance -> TaskQueue)) == 0)
	{
		//printk(KERN_ERR __FUNCTION__": TimeOutHandler(%d) - schedule_task FAILED (jiffies=%ld)\n", pTimeOutInstance->Count, jiffies);
		SCHEDULE_TASK_MOD_DEC_USE_COUNT;
		pTimeOutInstance->Count--;
	}
}

/********************************************************************/
/* Construct a periodic time out instance.RunTime manager does not	*/
/* simply use global timers , since these are called with the		*/
/* same thread handle as the ring3 application. This prevent		*/
/* synchronization between passive and async calls with semaphores	*/
/* or mutexes.                                                      */
/* Parameters :                                                     */
/* Irql - run time IQRL in which the timeout call back              */  
/*        should be called. In Wi95 this parameter in ignored,      */
/*        and all timeout are called at dispatch level.             */
/* InitialTimeout - Initial timeout interval in ms pRefData         */
/*                  should be passed as a parameter to this function*/
/* PTimeOutCallBack - CallBack to be call every timeout duration.   */
/* pFuncAlloc - alloc function,pRefData should be passed as         */ 
/*              parameter for thids function. function should       */
/*              allocate memory in non-paged pool                   */
/* pFuncFree - free function,pRefData should be passed as           */
/*             parameter for thids function                         */
/* pRefData -  reference data to be passed to pTimeOutCallBack      */
/*             pFuncAllc and pFuncFree                              */
/********************************************************************/
// @@@@ Irql unreferenced

GLOBAL	HANDLE	OSCreatePeriodicTimeOut	(	IN	TIMER_IRQL_TYPE	Irql,
											IN	UINT32			InitialTimeOut,
											IN	PCBFUNC			pTimeOutCallBack,
											IN	PALLOC_FUNC		pFuncAlloc,
											IN	PFREE_FUNC		pFuncFree,
											IN	PVOID			pRefData)
{
	PTIME_OUT_INSTANCE_T	pTimeOutInstance;
	if(pFuncAlloc) {
		pTimeOutInstance= pFuncAlloc(sizeof(TIME_OUT_INSTANCE_T), pRefData);
	} else {
		ASSERT(!in_interrupt());
		pTimeOutInstance= kmalloc(sizeof(TIME_OUT_INSTANCE_T), GFP_KERNEL);
	}
	if (NULL == pTimeOutInstance )
		return(NULL);
	pTimeOutInstance->pFuncFree			= pFuncFree;
	pTimeOutInstance->pRefData			= pRefData;
	pTimeOutInstance->pTimeOutCallBack	= pTimeOutCallBack;
	pTimeOutInstance->bLocked			= FALSE;
	pTimeOutInstance->mSec				= InitialTimeOut;
 
	INIT_TQUEUE(&pTimeOutInstance -> TaskQueue, TimerThreadFunction, pTimeOutInstance);

	init_timer(&pTimeOutInstance -> Timer);
	pTimeOutInstance -> Timer.function = TimeOutHandler;
	pTimeOutInstance -> Timer.data = (unsigned long)pTimeOutInstance;
	pTimeOutInstance -> Timer.expires = jiffies + MSECS_TO_TICKS(100);
	pTimeOutInstance -> Count = 0;
	if(InitialTimeOut != 0) {
		pTimeOutInstance -> active = TRUE;
		add_timer(&(pTimeOutInstance -> Timer));
	} else
		pTimeOutInstance -> active = FALSE;

	return( (HANDLE)pTimeOutInstance);
}

/********************************************************************/
GLOBAL	VOID	OSDestroyPeriodicTimeOut	(IN	HANDLE	hTimeOut)
{
	PTIME_OUT_INSTANCE_T	pTimeOutInstance = (PTIME_OUT_INSTANCE_T)hTimeOut;
	int						flushattempts;

	ASSERT(!in_interrupt());

	pTimeOutInstance -> bLocked	= TRUE;
	pTimeOutInstance -> active = FALSE;

	del_timer(&pTimeOutInstance -> Timer);

	for(flushattempts = 0; pTimeOutInstance->Count > 0 && flushattempts < 10;
			flushattempts++)
	{
		flush_scheduled_tasks();
	}

	if(pTimeOutInstance->pFuncFree) {
		pTimeOutInstance->pFuncFree(pTimeOutInstance,pTimeOutInstance->pRefData);
	} else {
		kfree(pTimeOutInstance);
	}

}
/********************************************************************/
GLOBAL	BOOL	OSSetPeriodicTimeOut		(	IN	HANDLE			hTimeOut,
												IN	UINT32			NewTimeOut)
{
	PTIME_OUT_INSTANCE_T	pTimeOutInstance = (PTIME_OUT_INSTANCE_T)hTimeOut;

	pTimeOutInstance->mSec = NewTimeOut;
	if(NewTimeOut != 0) {
		pTimeOutInstance -> active = TRUE;
		mod_timer(&pTimeOutInstance -> Timer, jiffies + MSECS_TO_TICKS(pTimeOutInstance -> mSec));
	} else {
		pTimeOutInstance -> active = FALSE;
		del_timer(&pTimeOutInstance -> Timer);
	}
	return TRUE;
}

#if 0
// This has to be a separate function due to Mandrake compiler bug
static
VOID	OSCallOnMyStack_asm(			IN	PCBFUNC			pFunc, 
											IN	UINT32			lParam, 
											IN	PUINT32			pTopOfStack) 
{
	static volatile void *pstack;

	asm volatile ( "mov %%esp, %0\n"
				   "mov %1, %%esp\n"
				   "push %3\n"
				   "call *%2\n"
				   "mov %0, %%esp\n"
				   : "=r" (pstack) : "r" (pTopOfStack), "r" (pFunc), "r" (lParam) );
}

GLOBAL	VOID	OSCallOnMyStack(			IN	PCBFUNC			pFunc, 
											IN	UINT32			lParam, 
											IN	PUINT32			pTopOfStack, 
											IN	UINT32			StackSizeBytes)
{
	static spinlock_t pstack_lock __attribute__((unused)) = SPIN_LOCK_UNLOCKED;
	unsigned long flags;

	spin_lock_irqsave(&pstack_lock, flags);
	OSCallOnMyStack_asm(pFunc, lParam, pTopOfStack);
	spin_unlock_irqrestore(&pstack_lock, flags);
}
#else
GLOBAL	VOID	OSCallOnMyStack(			IN	PCBFUNC			pFunc, 
											IN	UINT32			lParam, 
											IN	PUINT32			pTopOfStack, 
											IN	UINT32			StackSizeBytes)
{
	static volatile void *pstack;
//	static spinlock_t pstack_lock __attribute__((unused)) = SPIN_LOCK_UNLOCKED;
//	unsigned long flags;

//	spin_lock_irqsave(&pstack_lock, flags);
	asm volatile ( "mov %%esp, %0\n"
				   "mov %1, %%esp\n"
				   : "=r" (pstack) : "r" (pTopOfStack) );
	pFunc((PVOID)lParam); 
	asm volatile ( "mov %0, %%esp\n"
				   : "=r" (pstack) );
//	spin_unlock_irqrestore(&pstack_lock, flags);
}
#endif

static time_t epoch = 0;

/********************************************************************/
GLOBAL	UINT32	OSGetSystemTime(			VOID	)
{
	struct timeval timestamp;
	do_gettimeofday(&timestamp);
	// result returned in milliseconds
	return ((UINT32)(((timestamp.tv_sec-epoch)*1000) + (timestamp.tv_usec/1000)));
}

GLOBAL	VOID	OSInitTime(			VOID	)
{
	struct timeval timestamp;
	do_gettimeofday(&timestamp);

	epoch = timestamp.tv_sec;
}

/********************************************************************/

HANDLE OSCreateTimer(UINT32 msec, PVOID pCBFunc, PVOID pRefData)
{
	PTIMER_T pTimer;

	if ( pCBFunc == NULL )
	{
		ASSERT(pCBFunc);
		return (NULL);
	}

	ASSERT(!in_interrupt());

	pTimer = kmalloc(sizeof(TIMER_T),GFP_KERNEL);
	if ( NULL == pTimer )
	{
		ASSERT(pTimer);
		return (NULL);
	}

	memset(pTimer,0,sizeof(TIMER_T));

	init_timer(&(pTimer->timer));
	pTimer->timer.function = pCBFunc ;
	pTimer->timer.data = (unsigned long)pRefData;
	pTimer->timer.expires = jiffies + MSECS_TO_TICKS(msec);
	pTimer->msec = msec;
	return ((HANDLE)pTimer);
}

void OSSetTimer(PVOID pTimer)
{
	struct timer_list*	pTimerH;
	PTIMER_T pTimerS = (PTIMER_T)pTimer;
	if ( pTimer == NULL )
	{
		ASSERT(pTimer);
		return;
	}
	pTimerH = &(pTimerS->timer);
	mod_timer(pTimerH, jiffies + MSECS_TO_TICKS(pTimerS->msec));
}

void OSCancelTimer(PVOID pTimer)
{
	struct timer_list*	pTimerH;
	if ( pTimer == NULL )
	{
		ASSERT(pTimer);
		return;
	}
	pTimerH = &((PTIMER_T)pTimer)->timer;
	
	del_timer(pTimerH);
}

void OSChangeTimerTimeOut(PVOID pTimer, UINT32 msec)
{
	PTIMER_T pTimerS = (PTIMER_T)pTimer;
	if ( pTimer == NULL )
	{
		ASSERT(pTimer);
		return;
	}
	pTimerS->msec = msec;
}

void OSDestroyTimer(PVOID pTimer)
{
	if ( pTimer == NULL )
	{
		ASSERT(pTimer);
		return;
	}

	del_timer(&((PTIMER_T)pTimer)->timer);
	kfree(pTimer);
}

/********************************************************************/
/* Utilities                                                        */
/********************************************************************/

static spinlock_t atomic_lock __attribute__((unused)) = SPIN_LOCK_UNLOCKED;

/****************************************************************************************
    The OsAtomicCompareAndSwap function compares the value at the specified address with 
	oldVal. The value of newValue is written to the address only if oldValue and the 
	value at the address are equal. OSCompareAndSwap returns true if newValue is written 
	to the address; otherwise, it returns false. 
    
    Params:
        oldValue The value to compare at address.
        newValue The value to write to address if oldValue compares true.
        address The 4-byte aligned address of the data to update atomically.
    Result: true if newValue was written to the address. 
****************************************************************************************/
BOOL OsAtomicCompareAndSwap (PVOID oldValue, PVOID newValue, PVOID* address)
{
	unsigned long flags;

	spin_lock_irqsave(&atomic_lock, flags);
  	if (*(PUINT32)address ==  (UINT32)oldValue)
  	{
		*(PUINT32)address = (UINT32)newValue;
		spin_unlock_irqrestore(&atomic_lock, flags);
  		return TRUE;
  	}
	spin_unlock_irqrestore(&atomic_lock, flags);

	return FALSE;
}
/****************************************************************************************
    OsAtomicAdd
   
    OsAtomicAdd function adds the specified amount to the value at the specified 
    address and returns the result.
    Params:
        amount  The amount to add.
        address The 4-byte aligned address of the value to update atomically.
    Result: The result of the addition.
****************************************************************************************/
INT32	OsAtomicAdd   (INT32 amount, PINT32 address)
{
	unsigned long flags;
	atomic_t *v = (atomic_t *)address;

	spin_lock_irqsave(&atomic_lock, flags);
	atomic_add(amount, v);
	amount = atomic_read(v);
	spin_unlock_irqrestore(&atomic_lock, flags);

	return amount;
}

void OsSleep(UINT32 ms)
{
	if(!in_interrupt() && (ms > TICKS_TO_MSECS(1))) {
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(MSECS_TO_TICKS(ms));
	} else {
		// short delay or in interrupt
		mdelay(ms);
	}
}

/* Semaphores */
/********************************************************************/
GLOBAL HANDLE	OSSemaphoreCreate(	IN	int InitCount)
{
	struct semaphore *pSemaphore;
	
	ASSERT(!in_interrupt());

	pSemaphore = kmalloc(sizeof(struct semaphore), GFP_KERNEL);
	if (NULL == pSemaphore)
	{
		ASSERT(pSemaphore);
		return (NULL);
	}
	sema_init(pSemaphore, InitCount);
	return ((HANDLE)pSemaphore);
}

/********************************************************************/
GLOBAL VOID	OSSemaphoreDestroy(		IN	HANDLE hSemaphore)
{
	if (NULL != hSemaphore)
		kfree((void *)hSemaphore);
}

/********************************************************************/
GLOBAL VOID	OSSemaphoreWait(		IN	HANDLE hSemaphore)
{
	struct semaphore *pSemaphore = (struct semaphore *)hSemaphore;
	if (NULL == hSemaphore) {
		ASSERT(hSemaphore);
		return;
	}
	down(pSemaphore);
}

/********************************************************************/
GLOBAL VOID	OSSemaphoreSignal(		IN	HANDLE hSemaphore)
{
	struct semaphore *pSemaphore = (struct semaphore *)hSemaphore;
	if (NULL == hSemaphore) {
		ASSERT(hSemaphore);
		return;
	}
	up(pSemaphore);
}

/* Critical Section (Mutex) */
/********************************************************************/
typedef struct CRIT {
	unsigned long int saved_flags;
	BOOL valid;
	struct CRIT *prev;
} CRIT_T;

GLOBAL HANDLE __OSCriticalSectionCreate( char *file, int line )
{
    CRIT_T *mutex;

	ASSERT(!in_interrupt());

    mutex  = kmalloc(sizeof(CRIT_T), GFP_KERNEL);
	if(!mutex) {
		ASSERT(mutex);
		return NULL;
	}
	mutex->saved_flags = 0;
	mutex->valid = FALSE;
	mutex->prev = NULL;
//	printk( "From file %s, line %d: got mutex 0x%lx\n", file, line, mutex );
    return ((HANDLE)mutex);
}

/********************************************************************/
GLOBAL VOID	OSCriticalSectionDestroy(	IN	HANDLE hMutex)
{
	if ( NULL == hMutex )
	{
		ASSERT(hMutex);
		return;
	}
	kfree((CRIT_T *)hMutex);
}

#ifdef DEBUG_CHECK_MUTEX_TIMES
static DWORD cs_start;
#endif
/*  static int cs_count=0; */
static CRIT_T *curr_mutex=NULL;

/********************************************************************/
GLOBAL VOID	OSCriticalSectionAcquire(	IN	HANDLE hMutex)
{
	CRIT_T *mutex = (CRIT_T *) hMutex;
	unsigned long int old_fl;

	if(!hMutex) {
		ASSERT(hMutex);
		return;
	}

	old_fl = mutex->saved_flags;
	//printk("Entering CriticalSection %x\n",hMutex);
	save_flags( mutex->saved_flags );
	cli();
	if ( mutex->valid )
	{
		printk(KERN_ERR __FUNCTION__": mutex 0x%08lx already in use (0x%08lx)\n",
				(UINT32) hMutex, (UINT32) old_fl );
	}
	mutex->valid = TRUE;
	mutex->prev = curr_mutex;
	curr_mutex = mutex;
#ifdef DEBUG_CHECK_MUTEX_TIMES
	rdtscl(cs_start);
#endif
/*  	if ( ++cs_count != 1 ) */
/*  	{ */
/*  		printk( "OSCriticalSectionAcquire: count is %d!!!\n", cs_count ); */
/*  	} */
}

/********************************************************************/
GLOBAL VOID	OSCriticalSectionRelease(	IN	HANDLE hMutex)
{
	CRIT_T *mutex = (CRIT_T *) hMutex;
#ifdef DEBUG_CHECK_MUTEX_TIMES
	DWORD cs_end;
	DWORD tmp;
#endif

	if(!hMutex) {
		ASSERT(hMutex);
		return;
	}

/*  	if ( --cs_count != 0 ) */
/*  	{ */
/*  		printk( "OSCriticalSectionRelease: count is %d!!!\n", cs_count ); */
/*  	} */
#ifdef DEBUG_CHECK_MUTEX_TIMES
	rdtscl(cs_end);
	tmp = cs_end;
#endif
	if ( curr_mutex != mutex )
	{
		printk(KERN_ERR __FUNCTION__": acquired 0x%08lx, release 0x%08lx\n",
			   (UINT32) curr_mutex, (UINT32) mutex );
	}
	if ( !mutex->valid )
	{
		printk(KERN_ERR __FUNCTION__": mutex 0x%08lx invalid (0x%08lx)\n",
				(UINT32) hMutex, (UINT32) mutex->saved_flags );
	}
	curr_mutex = mutex->prev;
	mutex->valid = FALSE;
#ifdef DEBUG_CHECK_MUTEX_TIMES
	cs_end -= cs_start;
#endif
	restore_flags( mutex->saved_flags );
#ifdef DEBUG_CHECK_MUTEX_TIMES
	if ( cs_end > 2660000 ) /* More than 10mSec */
	{
		printk( "CriticalSection too long (%ld-%ld=%ld)\n",
				tmp, cs_start, cs_end );
	}
#endif
	//printk("Left CriticalSection %x\n",hMutex);
}

/********************************************************************/
GLOBAL	VOID	OSSetTimeSensitivity(	IN	UINT32				Interval)			
{
}
/********************************************************************/
GLOBAL	VOID	OSRestoreTimeSensitivity(		IN	UINT32				Interval)
{
}

/********************************************************************/
#if 0
GLOBAL	HANDLE	OSDisableInterrupt(void)
{
	UINT16 wPreviousState;
#if ( COMPILER == CMP_MSDEV)
	__asm
    {
	    push  eax;
	    pushf;
	    pop   ax;
	    mov   wPreviousState, ax;
	    pop   eax;
	    cli;
    }
#else
    asm ( "\n\t"
	  "pushl %%eax \n\t"
	  "pushf \n\t"
	  "popw  %%ax \n\t"
	  "movw  %%ax, %0 \n\t"
	  "popl  %%eax \n\t"
	  "cli"

	  : "=m" (wPreviousState) );
#endif
	return( (HANDLE) (UINT) wPreviousState);
}

/********************************************************************/
GLOBAL	VOID	OSRestoreInterrupts(	IN	HANDLE hStatus)
{
 	UINT16 wPreviousState = (UINT16) (UINT) hStatus;
#if ( COMPILER == CMP_MSDEV)
	__asm
    {
	    push  eax;
	    mov   ax, wPreviousState;
	    test  ah, 0x02;
	    pop	eax;
	    jz    SHORT No_Enable;
	    sti;
	No_Enable:
	}
#else
   asm( "\n\t"
	"pushl  %%eax \n\t"
	"movw   %0, %%ax \n\t"
	"testb  $2, %%ah \n\t"
	"popl   %%eax \n\t"
	"jz     No_Enable \n\t"
	"sti \n"
        "No_Enable:\n"

	: : "m" (wPreviousState) );
#endif
}
#endif
