/*************************************************************************

  File Name: OsKernStdio.c

  File Description: Functions to emulate stdio within the kernel.

  ************************************************************************/

/*
 * Copyright (c) 2001 Conexant Systems, Inc.
 * 
 * 1.   Permitted use. Redistribution and use in source and binary forms,
 * with or without modification, are permitted under the terms set forth
 * herein.
 * 
 * 2.   Disclaimer of Warranties. CONEXANT AND OTHER CONTRIBUTORS MAKE NO
 * REPRESENTATION ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.
 * IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTIES OF ANY KIND.
 * CONEXANT AND OTHER CONTRIBUTORS DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, GOOD TITLE AND AGAINST INFRINGEMENT.
 * 
 * This software has not been formally tested, and there is no guarantee that
 * it is free of errors including, but not limited to, bugs, defects,
 * interrupted operation, or unexpected results. Any use of this software is
 * at user's own risk.
 * 
 * 3.   No Liability.
 * 
 * (a) Conexant or contributors shall not be responsible for any loss or
 * damage to Company, its customers, or any third parties for any reason
 * whatsoever, and CONEXANT OR CONTRIBUTORS SHALL NOT BE LIABLE FOR ANY
 * ACTUAL, DIRECT, INDIRECT, SPECIAL, PUNITIVE, INCIDENTAL, OR CONSEQUENTIAL
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED, WHETHER IN CONTRACT, STRICT OR OTHER LEGAL THEORY OF
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY
 * WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 * 
 * (b) User agrees to hold Conexant and contributors harmless from any
 * liability, loss, cost, damage or expense, including attorney's fees,
 * as a result of any claims which may be made by any person, including
 * but not limited to User, its agents and employees, its customers, or
 * any third parties that arise out of or result from the manufacture,
 * delivery, actual or alleged ownership, performance, use, operation
 * or possession of the software furnished hereunder, whether such claims
 * are based on negligence, breach of contract, absolute liability or any
 * other legal theory.
 * 
 * 4.   Notices. User hereby agrees not to remove, alter or destroy any
 * copyright, trademark, credits, other proprietary notices or confidential
 * legends placed upon, contained within or associated with the Software,
 * and shall include all such unaltered copyright, trademark, credits,
 * other proprietary notices or confidential legends on or in every copy of
 * the Software.
 * 
 */
#if ( UK_MODE == UK_KERNEL )

#include "oscompat.h"
#include <linux/fs.h>
#include <asm/uaccess.h>

#include "ossysenv.h"
#include "osstdio.h"

FILE *
OsFOpen (const char *path, const char *mode)
{
	FILE *filp;
	int error = 0;
	int flags, creatmode = 0;

	if(!path || !mode) {
		error = -EIO;
		filp = NULL;
		goto out;
	}

	//printk(KERN_INFO __FUNCTION__"(\"%s\", \"%s\")\n", path, mode);

	if(!strcmp(mode, "r") || !strcmp(mode, "rb")) {
		flags = O_RDONLY;
	} else if(!strcmp(mode, "r+") || !strcmp(mode, "r+b") || !strcmp(mode, "rb+")) {
		flags = O_RDWR;
	} else if(!strcmp(mode, "w") || !strcmp(mode, "wb")) {
		flags = O_WRONLY | O_TRUNC | O_CREAT;
		creatmode = 0600;
	} else if(!strcmp(mode, "w+") || !strcmp(mode, "w+b") || !strcmp(mode, "wb+")) {
		flags = O_RDWR | O_TRUNC | O_CREAT;
		creatmode = 0600;
	} else {
		error = -ENOSYS; // not implemented
		filp = NULL;
		goto out;
	}

	filp = filp_open(path, flags, creatmode);

	if (IS_ERR(filp)) {
		error = PTR_ERR(filp);
		filp = NULL;
		goto out;
	}

	if (!S_ISREG(filp->f_dentry->d_inode->i_mode)) {
		error = -EACCES;
		filp_close(filp, NULL);
		filp = NULL;
		goto out;
	}

	if (!filp->f_op->read) {
		error = -EIO;
		filp_close(filp, NULL);
		filp = NULL;
		goto out;
	}

out:
	if(error)
		printk(KERN_ERR __FUNCTION__"(\"%s\", \"%s\"): error %d\n", path, mode, error);

	if (filp) {
		filp->f_pos = 0;            /* start offset */
	}

	return filp;
}

int
OsFClose(FILE *filp)
{
	if(!IS_ERR(filp)) {
		filp_close(filp, NULL);
	}
	return 0;
}

size_t
OsFRead(void *ptr, size_t size, size_t nmemb, FILE *filp)
{
	int bytes;

	if(IS_ERR(filp)) {
		bytes = -EINVAL;
	} else {
		mm_segment_t oldfs;

		oldfs = get_fs();
		set_fs(get_ds());
		bytes = filp->f_op->read(filp, ptr, size*nmemb, &filp->f_pos);
		set_fs(oldfs);
	}

	if(bytes < 0) {
		printk(KERN_ERR __FUNCTION__"(filp %p): error %d\n", filp, bytes);
		return 0;
	} else
		return bytes / size;
}

size_t
OsFWrite(const void *ptr, size_t size, size_t nmemb, FILE *filp)
{
	int bytes;

	if(IS_ERR(filp)) {
		bytes = -EINVAL;
	} else {
		mm_segment_t oldfs;

		oldfs = get_fs();
		set_fs(get_ds());
		bytes = filp->f_op->write(filp, ptr, size*nmemb, &filp->f_pos);
		set_fs(oldfs);
	}

	if(bytes < 0) {
		printk(KERN_ERR __FUNCTION__"(filp %p): error %d\n", filp, bytes);
		return 0;
	} else
		return bytes / size;
}


int
OsFSeek(FILE *filp, long offset, int origin)
{
	loff_t retval;
	mm_segment_t oldfs;

	if(IS_ERR(filp)) {
		return -EINVAL;
	}

	if(!filp->f_op || !filp->f_op->llseek) {
		return -ESPIPE;
	}

	oldfs = get_fs();
	set_fs(get_ds());
	retval = filp->f_op->llseek(filp, offset, origin);
	set_fs(oldfs);
	if(retval < 0) {
		printk(KERN_ERR __FUNCTION__"(filp %p, offset=%ld origin=%d): error %d\n", filp, offset, origin, (int)retval);
		return retval;
	} else
		return 0;
}

#endif // UK_MODE == UK_KERNEL

