#
# This file contains common functions used by pkgtools scripts.
#
# Copyright (C) 2005 Lasse Collin <lasse.collin@slackware.fi>
#
# Redistribution and use of this script, with or without modification, is
# permitted provided that the following conditions are met:
#
# 1. Redistributions of this script must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
#
#  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
#  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
#  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO
#  EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
#  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
#  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
#  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
#  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
#  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# Last modified: Wed May 18 20:20:29 EEST 2005

# All scripts in pkgtools expect umask 0022:
umask 0022

# This is to keep 'sort' fast:
LC_ALL=C
export LC_ALL

# Location of tar-1.13. I think it is more comfortable to have tar-1.13 included
# in the package of pkgtools instead of tar package. Your opinion may vary.
TAR=tar-1.13

# Location of Wget:
WGET=wget

# ash does not have $UID set by default:
[ "$UID" = "" ] && UID=$(id -u)

# We do not want these to be set with environment variables:
unset MODE

# Check that directory specified with --root exists. If not, show error and exit.
check_root_dir_exists() {
  [ -d "$1" ] && return 0
  echo "The specified root directory does not exist: $1"
  exit 99
}

# Check that we are running with root priviledges. If not, quit.
check_is_run_by_root() {
  [ "$UID" = "0" ] && return 0
  echo "You must be root to use this command."
  exit 10
}

# Error message used by installpkg, removepkg and upgradepkg:
exit_no_packages_specified() {
  echo "Error: No packages specified on the command line."
  exit 99
}

# Error message used by many scripts, hopefully never needed:
exit_getopt_error() {
  echo "Fatal error parsing command line options."; exit 97
}

# '/path/to/foo-bar-0.12-i486-1.tgz' => 'foo-bar-0.12-i486-1'
package_fullname() {
  echo "$1" | sed 's,^.*/,,;s,\.\(tgz\|tlz\|tbz\|tar\)$,,'
}

# Returns the basename of the package. E.g. '/path/to/foo-bar-0.12-i486-1.tgz' => 'foo-bar'
package_basename() {
  echo "$1" | sed 's,^.*/,,;s,\.\(tgz\|tlz\|tbz\|tar\)$,,;s,-[^-]*-[^-]*-[^-]*$,,'
}

# '/path/to/foo-bar-0.12-i486-1.tgz' => '0.12'
package_version() {
  echo "$1" | sed -n 's,^.*/,,;s,^.*-\([^-]*\)-[^-]*-[^-]*\.\(tgz\|tlz\|tbz\|tar\)$,\1,p'
}

# '/path/to/foo-bar-0.12-i486-1.tgz' => 'i486'
package_arch() {
  echo "$1" | sed -n 's,^.*/,,;s,^.*-[^-]*-\([^-]*\)-[^-]*\.\(tgz\|tlz\|tbz\|tar\)$,\1,p'
}

# '/path/to/foo-bar-0.12-i486-1.tgz' => '1'
package_buildversion() {
  echo "$1" | sed -n 's,^.*/,,;s,^.*-[^-]*-[^-]*-\([^-]*\)\.\(tgz\|tlz\|tbz\|tar\)$,\1,p'
}

# Returns package type which can be "tgz", "tlz", "tbz", "tar" or "".
# Empty means invalid package name/type.
package_type() {
  echo "$1" | sed -n 's/^.*\.\(tgz\|tlz\|tbz\|tar\)$/\1/p'
}

# '/path/to/foo-bar-0.12-i486-1.tgz' => '/path/to/foo-bar-0.12-i486-1'
package_strip_extension() {
  echo "$1" | sed 's,\.\(tgz\|tlz\|tbz\|tar\)$,,'
}

# Returns true (zero) if the argument is an URL to a package file:
is_url_package() {
  [ "$(echo "$1" | sed -n 's/^\(http\|ftp\):\/\/.*\/.*\.\(tgz\|tlz\|tbz\|tar\)$/\1/p')" = "" ] && return 1
  return 0
}

# Returns true (zero) if the given package name conforms to Slackware
# specifications. This function accepts empty fields, e.g. 'foo-0.12--.tgz',
# you can take this as an undocumented feature (or bug). ;-)
is_valid_package_name() {
  [ "$(echo "$1" | sed -n \
      's,^.*/,,;s,^[^-].*-[^-]*-[^-]*-[^-]*\.\(tgz\|tlz\|tbz\|tar\)$,&,p')" = "" -a \
      "$(echo "$1" | sed -n 's,^.*/,,;/^[a-zA-Z0-9.,!@_+-]*$/p')" != "" ] && return 1
  return 0 # $1 is a valid package name
}

# Reads stdin, return true if a kernel package except kernel-headers or kernel-source is found:
is_kernel_packages() {
  [ "$(sed -n 's,^.*/,,;/^kernel-headers-[^-]*-[^-]*-[^-]*$/d
      /kernel-source-[^-]*-[^-]*-[^-]*/d;/^kernel-/p')" = "" ] && return 1
  return 0
}

show_new_kernel_message() {
  cat << EOF
If you installed a new kernel, remember to check your boot loader
configuration and in case of LILO run /sbin/lilo. If you are using
another boot loader, such as GRUB, refer to its manual to learn how
to select the new kernel at boot.
EOF
}

# Extracts symlinks from doinst.sh style script. Reads stdin and outputs to stdout.
extract_links() {
  # The first line detects the original Slackware style symlinks and
  # the second new style quoted symlinks. After that the second sed
  # unquotes the symlink names.
  sed -n "s,^( *cd \([^ ;][^ ;]*\) *; *rm -rf \([^ )][^ )]*\) *) *$,\1/\2,p
      s,^rm -rf -- '\(.*\)' #Symlink#$,\1,p" | sed "s,^'\(.*\)'$,\1,;s,'\\\\'',',g"
}

# Uncompress a file to stdout:
uncompress_pkg() {
  case "$(package_type "$1")" in
    tlz)        lzma d -si -so < "$1" 2> /dev/null ;;
    # Use bunzip2 instead of bzip2 for BusyBox compatibility:
    tbz)        bunzip2 -c "$1" 2> /dev/null ;;
    tar)        cat "$1" 2> /dev/null ;;
    # For explodepkg compatibility we default to tgz. Do no add -f as it does not work with busybox.
    *)          gunzip -c "$1" 2> /dev/null ;;
  esac
}

# Compress a file from stdin to stdout:
compress_pkg() {
  case "$1" in
    tgz)   gzip -9 ;;
    tlz)   lzma e -a1 -si -so 2> /dev/null ;;
    tbz)   bzip2 -9 ;;
    tar)   cat ;;
  esac
}

download() {
# $1 = URL ; $2 = destination file
  (
    # WGET_FLAGS will not be splitted correctly without this:
    unset IFS
    # trap is ignored by busybox but I hope we can live with it. Without trap
    # Ctrl-c would break execution of the whole script instead of only wget.
    trap "rm -f \"$2\"; exit 1" SIGINT
    $WGET $WGET_FLAGS -O "$2" "$1"
  )
}

# Reads config file, initalizes some commonly used variables and creates
# package database directories and files if they do not already exist.
# Note: This function must called *after* the ROOT is set!
initialize_variables_and_package_database() {
  PACKAGE_CACHE_DIR=/var/cache/packages
  KEEP_DOWNLOADED=1
  WGET_FLAGS="--passive-ftp"
  BLACKLIST_FILE=/etc/pkgtools/blacklist
  [ -f /etc/pkgtools/config ] && . /etc/pkgtools/config
  [ ! -f "$BLACKLIST_FILE" ] && BLACKLIST_FILE=/dev/null
  ADM_DIR=$ROOT/var/log
  TMP=$ADM_DIR/setup/tmp
  REPO_DIR=$ADM_DIR/setup/repositories
  [ "$ROOT" != "" ] && export ROOT
  for I in  "$ADM_DIR/packages" \
            "$ADM_DIR/scripts" \
            "$ADM_DIR/removed_packages" \
            "$ADM_DIR/removed_scripts" \
            "$REPO_DIR" \
            "$PACKAGE_CACHE_DIR"
    do
    if [ ! -d "$I" ]; then
      rm -rf "$I"
      mkdir -m 0755 -p "$I"
    fi
  done
  # $TMP has different default permissions:
  if [ ! -d "$TMP" ]; then
    rm -rf "$TMP"
    mkdir -m 0700 -p "$TMP"
  fi
}
