/*
# The contents of this file are subject to the Netscape Public License
# Version 1.1 (the "License"); you may not use this file except in
# compliance with the License. You may obtain a copy of the License at
# http://www.mozilla.org/NPL/
#
# Software distributed under the License is distributed on an "AS IS"
# basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
# the License for the specific language governing rights and limitations
# under the License.
#
# The Original Code was released August, 1999.  The Initial Developer
# of the Original Code is Netscape Communications Corporation.  Portions
# created by Netscape are Copyright (C) 1999, 2000 Netscape Communications
# Corporation.  All Rights Reserved.
#
# Contributor(s): John M. Kristian <kristian@netscape.com>
#                 Richard Megginson <richm@netscape.com>
*/
#include "interpret.h"

#include <stdio.h> /* fprintf perror */
#include <stdlib.h> /* getenv malloc */
#include <string.h>
#ifdef ARCH_AIX
#include <strings.h> /* for strcasecmp */
#endif

#ifdef ARCH_MSWin32
#define strcasecmp(L,R) stricmp(L,R)
#endif

#if defined(ARCH_MSWin32) || defined(ARCH_HPUX) || defined(ARCH_OSF1)
static char*
strtok_r(char* sp, const char* brk, char** next)
{
    const char *bp;
    unsigned long sc, bc;
    char *tok;

    if (sp == NULL && (sp = *next) == NULL)
      return NULL;

    /* Skip leading delimiters; roughly, sp += strspn(sp, brk) */
  cont:
    sc = *sp++;
    for (bp = brk; (bc = *bp++) != 0;) {
	if (sc == bc)
	  goto cont;
    }

    if (sc == 0) { /* no non-delimiter characters */
	*next = NULL;
	return NULL;
    }
    tok = sp-1;

    /* Scan token; roughly, sp += strcspn(sp, brk)
     * Note that brk must be 0-terminated; we stop if we see that, too.
     */
    while (1) {
	sc = *sp++;
	bp = brk;
	do {
	    if ((bc = *bp++) == sc) {
		if (sc == 0) {
		    *next = NULL;
		} else {
		    *next = sp;
		    *(sp-1) = 0;
		}
		return tok;
	    }
	} while (bc != 0);
    }
    /* NOTREACHED */
}
#endif /* ARCH_MSWin32 */

static const char space[] = " \t\n\r";

static const char*
get_value (const char* t)
{
    if (t && *t == '$') {
	t = getenv (t+1);
    }
    return t;
}

extern "C" int
interpret (char* script, int (*execute) (const char* source, const char* op))
{
    const char* source = "?";
    auto int decided = 1;
    auto char* next_line = NULL;

    auto char* line;
    for (line = strtok_r (script, "\n\r", &next_line); line;
	 line = strtok_r (NULL,   "\n\r", &next_line)) {

	auto const char* op;
	for (op = strtok (line, space); op;
	     op = strtok (NULL, space)) {
	    if (*op == '#') { /* comment */
		break; /* skip to next line */
	    } else if (!strcasecmp (op, "source:")) {
		source = strtok (NULL, "\n\r");

	    /* The syntax for ifdef { }else{ } looks like C, but it's not really.
	       There's no nesting and {} aren't special (so whitespace matters).
	       You're welcome to implement a fancier interpreter.  It would be
	       compatible with current scripts.
	    */
	    } else if (!strcasecmp (op, "ifdef")) {
		auto const char* v = strtok (NULL, space);
		if (strcmp ("{", strtok (NULL, space))) {
		    fprintf (stderr, "%s: missing { after %s %s\n", source, op, v);
		    return -1;
		}
		decided = v && (*v == '$') && getenv (v+1);
		if (!decided) break; /* skip to next line */
	    } else if (!strcasecmp (op, "}else{")) {
		if (decided) break; /* skip to next line */
		decided = 1;
	    } else if (!strcmp (op, "}")) {
		/* do nothing */

	    } else if (!strcasecmp (op, "addPath")) {
		auto const char* L = strtok (NULL, space);
		auto const char* sep = strtok (NULL, space);
		auto const char* R = strtok (NULL, space);
		if (!(L && sep && R)) {
		    fprintf (stderr, "%s: %s requires 3 parameters\n", source, op);
		    return -1;
		}
		auto const char* set = (*L == '$') ? L+1 : ((*R == '$') ? R+1 : NULL);
		if (set) {
		    L = get_value (L);
		    R = get_value (R);
		    auto char* result = (char*)malloc (strlen(set)
						       + (L ? strlen(L) : 0)
						       + strlen (sep)
						       + (R ? strlen(R) : 0)
						       + 2);
		    strcpy (result, set);
		    strcat (result, "=");
		    if (L) strcat (result, L);
		    if (L && R) strcat (result, sep);
		    if (R) strcat (result, R);
		    auto const int err = putenv (result);
		    if (err) {
			fprintf (stderr, "%s: putenv(%s): %i\n", source, result, err);
			free (result);
			return err;
		    }
		}

	    } else if (!strcasecmp (op, "echoError")) {
		fprintf (stderr, "%s:", source);
		auto const char* p;
		while (p = get_value (strtok (NULL, space))) {
		    fprintf (stderr, " %s", p);
		}
		fprintf (stderr, "\n");

	    } else if (execute) {
		auto const int err = execute (source, op);
		if (err) return err;
	    } else {
		fprintf (stderr, "%s: %s is not supported\n", source, op);
		return -1;
	    }
	}
    }
    return 0;
}
