/*
 *  linux/fs/supermount/subfs.c
 *
 *  Original version:
 *      Copyright (C) 1995, 1997
 *      Stephen Tweedie (sct@dcs.ed.ac.uk)
 *
 *      from
 *
 *      linux/fs/minix/inode.c
 *      Copyright (C) 1991, 1992  Linus Torvalds
 *
 *      and
 *
 *      linux/fs/ext2/super.c
 *      Copyright (C) 1992, 1993, 1994, 1995  Remy Card
 *
 *  Rewriten for kernel 2.2, 2.4. (C) 1999, 2000 Alexis Mikhailov
 *                                    (alexis@abc.cap.ru)
 *  Rewritten for kernel 2.4.21 (C) 2003 Andrey Borzenkov
 *                                       (arvidjaar@mail.ru)
 *  Rewritten for kernel 2.5.70 (C) 2003 Andrey Borzenkov
 *                                       (arvidjaar@mail.ru)
 *
 *  $Id: subfs.c,v 1.29.2.11 2004/01/14 19:24:10 bor Exp $
 */

#define S_DBG_TRACE_CURRENT S_DBG_TRACE_SUBFS
#include "supermount.h"

/*
 * close all open files on subfs
 */
static void
supermount_clean_files(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	struct list_head *ptr, *n;

	ENTER(sb);

	list_for_each_safe(ptr, n, &sbi->s_files) {
		struct supermount_file_info *sfi;
		struct file *subfile;

		sfi = list_entry(ptr, struct supermount_file_info, list);

		subfile = sfi->file;
		sfi->file = 0;
		list_del_init(&sfi->list);

		SUPERMOUNT_BUG_LOCKED_ON(sb, !subfile);

		fput(subfile);
	}

	LEAVE(sb);

}

/*
 * close all open dentries on subfs
 */
static void
supermount_clean_dentries(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	struct list_head *ptr, *n;

	ENTER(sb);

	list_for_each_safe(ptr, n, &sbi->s_dentries) {
		struct supermount_dentry_info *sdi;
		struct dentry *subdent;

		sdi = list_entry(ptr, struct supermount_dentry_info, list);
		/*
		 * HACK - FIXME
		 * see dentry.c:supermount_d_compare
		 */
		write_lock(&d_compare_lock);
		subdent = sdi->dentry;
		sdi->dentry = 0;
		write_unlock(&d_compare_lock);

		list_del_init(&sdi->list);
		d_drop(sdi->host);
		dnotify_parent(sdi->host, DN_DELETE);

		SUPERMOUNT_BUG_LOCKED_ON(sb, !subdent);

		dput(subdent);
	}

	LEAVE(sb);

}

/*
 * close all open inodes on subfs
 */
static void
supermount_clean_inodes(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	struct list_head *ptr, *n;

	ENTER(sb);

	list_for_each_safe(ptr, n, &sbi->s_inodes) {
		struct supermount_inode_info *sii;
		struct inode *host;
		struct inode *subi;

		sii = list_entry(ptr, struct supermount_inode_info, list);
		host = &sii->vfs_inode;

		subi = sii->inode;
		sii->inode = NULL;
		list_del_init(&sii->list);
		remove_inode_hash(host);
		host->i_mapping = &host->i_data;

		/*
		 * it is possible to have no subi here. clear_inode does
		 * release lock after removing subi but before unlinking
		 * it
		 */
		if (subi)
			iput(subi);

		SUPERMOUNT_BUG_LOCKED_ON(sb, sii->writecount < 0);
		SUPERMOUNT_BUG_LOCKED_ON(sb, sii->readcount < 0);
		while (sii->writecount) {
			sii->writecount--;
			sbi->writecount--;
		}
		while (sii->readcount) {
			sii->readcount--;
			sbi->readcount--;
		}
	}

	LEAVE(sb);
}

/*
 * reason can be
 *   SUBFS_UMNT_NORMAL - normal umount, do not try to release subfs
 *   SUBFS_UMNT_MEDIA  - media change detected, release subfs,
 *   			 do not remount ro (as media is already gone)
 *   SUBFS_UMNT_USER   - user request, release subfs, remount ro before
 *   			 releasing tray lock
 *
 *   unlock_door is always needed to keep device usage count correct
 */
static inline int subfs_remount_ro(struct super_block *sb);
void
subfs_umount(struct super_block *sb, int reason)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	int writecount = sbi->writecount;

	ENTER(sb);

	if (reason != SUBFS_UMNT_NORMAL) {
		/*
		 * we used to did shrink_dcache here. It compicates locking
		 * (clear_inode is called under sbi->sem thus requiring either
		 * recursive lock or separate lock just for inode list).
		 * This is not needed any more to ensure subfs can be umounted
		 * so we let dentries die and rely on dentry_revalidate to
		 * reject stale dentries
		 */

		if (subfs_is_busy(sb))
			supermount_warning(sb, "opened files during media change");

		supermount_clean_files(sb);
		supermount_clean_dentries(sb);
		supermount_clean_inodes(sb);

		if (reason == SUBFS_UMNT_USER && writecount > 0)
			subfs_remount_ro(sb);
		if (sbi->lockcount > 0)
			supermount_unlock_door(sb);
		/*
		 * this is quite ugly but so far I have no idea how to
		 * do it cleanly
		 */
		sbi->lockcount = 0;
	}

	/*
	 * all files are expected to be closed
	 */
	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->writecount);
	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->readcount);
	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->lockcount);
	SUPERMOUNT_BUG_LOCKED_ON(sb, !list_empty(&sbi->s_files));
	SUPERMOUNT_BUG_LOCKED_ON(sb, !list_empty(&sbi->s_dentries));
	SUPERMOUNT_BUG_LOCKED_ON(sb, !list_empty(&sbi->s_inodes));

	unmark_media_supermounted(sb);
	mntput(sbi->s_undermount);
	sbi->s_undermount = NULL;
	if (sbi->rw)
		sb->s_flags &= ~MS_RDONLY;
	sb->s_blocksize = 1024;
	sb->s_blocksize_bits = 10;

	/*
	 * FIXME
	 * again the problem of unmounting subfs from inside of put_super
	 */
	if (sb->s_root)
		supermount_init_root_inode(sb->s_root->d_inode);

	LEAVE(sb);
}

static inline int
subfs_remount_ro(struct super_block *sb)
{
	struct super_block *subsb = subfs_sb(sb);
	int rc;

	ENTER(sb);

	rc = do_remount_sb(subsb, subsb->s_flags | MS_RDONLY, NULL, 0);

	LEAVE(sb);

	return rc;
}

static inline int
subfs_remount_rw(struct super_block *sb)
{
	struct super_block *subsb = subfs_sb(sb);
	int rc;

	ENTER(sb);

	rc = do_remount_sb(subsb, subsb->s_flags & ~MS_RDONLY, NULL, 0);

	LEAVE(sb);

	return rc;
}

static struct vfsmount *
subfs_real_mount(struct super_block *sb, char *type)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	struct vfsmount *mnt;
	char *opts = NULL;

	ENTER(sb, "type=%s", type);

	if (sbi->s_data) {
		opts = strdup(sbi->s_data);
		if (!opts) {
			mnt = ERR_PTR(-ENOMEM);
			goto fail;
		}
	}

	mnt = do_kern_mount(type,
			    (sb->s_flags & MS_RMT_MASK) | MS_SUPERMOUNTED,
			    sbi->devname, opts);
	if (opts)
		kfree(opts);

fail:
	LEAVE(sb, "submnt=%p", mnt);

	return mnt;
}

static int
subfs_real_mount2(struct super_block *sb, char *type)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	struct vfsmount *mnt;
	struct super_block *subsb;
	int rc = 0;

	ENTER(sb, "type=%s", type);

	mnt = subfs_real_mount(sb, type);

	if (IS_ERR(mnt) && PTR_ERR(mnt) == -EROFS) {
		sb->s_flags |= MS_RDONLY;
		mnt = subfs_real_mount(sb, type);
	}
	rc = PTR_ERR(mnt);
	if (IS_ERR(mnt))
		goto out;

	/* paranoid check for double-mounting */
	SUPERMOUNT_BUG_LOCKED_ON(sb, atomic_read(&mnt->mnt_sb->s_active) != 1);

	sbi->s_undermount = mnt;

	if (!(sb->s_flags & MS_RDONLY)) {
		rc = subfs_remount_ro(sb);
		if (rc)
			goto mntput;
	}

	subsb = mnt->mnt_sb;
	sb->s_blocksize = subsb->s_blocksize;
	sb->s_blocksize_bits = subsb->s_blocksize_bits;
	attach_subfs_dentry(sb->s_root, mnt->mnt_root);
	attach_subfs_inode(sb->s_root->d_inode, mnt->mnt_root->d_inode);
	insert_inode_hash(sb->s_root->d_inode);
	dnotify_parent(sb->s_root, DN_CREATE);
	rc = 0;
	goto out;

	/*
	 * error clean up
	 */
mntput:
	sbi->s_undermount = 0;
	mntput(mnt);
out:
	LEAVE(sb, "type=%s rc=%d", type, rc);

	return rc;
}

/*
 * Error values from mount
 *   ENOENT     - no device file (quite possible with devfs)
 *   ENXIO      - device does not exist
 *   ENOMEDIUM  - no medium inserted (surprise, surprise :)
 *   EBUSY      - attempt to mount on already mounted device
 *                we specifically disallow it even when both
 *                file system and mode (ro/rw) are the same
 */
static int
subfs_mount(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	int retval = -ENODEV;
	char *types = strdup(sbi->s_type);

	ENTER(sb);

	if (sbi->disabled)
		retval = -EPERM;
	else if (!types)
	       retval = -ENOMEM;
	else {
		char *p = types, *fs;

		while ((fs = strsep(&p, ":")) != NULL && retval
		       && retval != -ENXIO
		       && retval != -ENOMEDIUM
		       && retval != -ENOENT
		       && retval != -EBUSY)
			retval = subfs_real_mount2(sb, fs);
	}

	if (types)
		kfree(types);

	if (!retval)
		mark_media_supermounted(sb);

	LEAVE(sb, "rc=%d", retval);

	return retval;
}

static int
__subfs_check_disk_change(struct super_block *sb)
{
	struct super_block *subsb = subfs_sb(sb);
	int rc;
	struct block_device *dev = subsb->s_bdev;

	ENTER(sb);

	rc = atomic_read(&subsb->s_media_changed);
	if (!rc) {
		lock_kernel();
		rc = check_disk_change(dev);
		unlock_kernel();
	}

	if (rc)
		subfs_umount(sb, SUBFS_UMNT_MEDIA);

	LEAVE(sb, "rc=%d", rc);

	return rc;
}

/*
 * this must really be called subfs_active ...
 */
int
subfs_check_disk_change(struct super_block *sb)
{
	int rc;

	ENTER(sb);

	subfs_lock(sb);
	if (subfs_is_mounted(sb))
		rc = __subfs_check_disk_change(sb);
	else
		rc = 1;
	subfs_unlock(sb);


	LEAVE(sb, "rc=%d", rc);

	return rc;
}

static int
check_and_remount_subfs(struct super_block *sb)
{
	int rc;

	ENTER(sb);

	if (subfs_is_mounted(sb) && !__subfs_check_disk_change(sb)) {
		rc = 0;
		goto out;
	}

	rc = subfs_mount(sb);

out:
	LEAVE(sb, "rc=%d", rc);

	return rc;
}

static inline void
subfs_tray_lock(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	ENTER(sb);

	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->lockcount < 0);

	if (!sbi->lockcount++)
		supermount_lock_door(sb);

	LEAVE(sb);
}

static inline void
subfs_tray_unlock(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	ENTER(sb);

	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->lockcount <= 0);

	if (!--sbi->lockcount)
		supermount_unlock_door(sb);

	LEAVE(sb);
}

static void
subfs_get_read_access(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	ENTER(sb);

	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->readcount < 0);

	if (!sbi->readcount++ && sbi->tray_lock == TRAY_LOCK_ALWAYS)
		subfs_tray_lock(sb);

	LEAVE(sb);
}

static int
subfs_get_write_access(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	int rc;

	ENTER(sb);

	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->writecount < 0);

	rc = 0;
	if (!sbi->writecount) {
		if (sb->s_flags & MS_RDONLY)
			rc = -EROFS;
		else
			rc = subfs_remount_rw(sb);
	}

	if (!rc && !sbi->writecount++ && sbi->tray_lock != TRAY_LOCK_NEVER)
		subfs_tray_lock(sb);

	LEAVE(sb, "rc=%d", rc);

	return rc;
}

static void
subfs_put_read_access(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	ENTER(sb);

	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->readcount <= 0);

	if (!--sbi->readcount && sbi->tray_lock == TRAY_LOCK_ALWAYS)
		subfs_tray_unlock(sb);

	LEAVE(sb);
}

static void
subfs_put_write_access(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	ENTER(sb);

	SUPERMOUNT_BUG_LOCKED_ON(sb, sbi->writecount <= 0);

	if (!--sbi->writecount) {
		/*
		 * no need to fsync, it is done automatically on remount
		 */
		int rc = subfs_remount_ro(sb);
		if (rc)
			supermount_error(sb, "failed to remount ro, error = %d", rc);
		if (sbi->tray_lock != TRAY_LOCK_NEVER)
			subfs_tray_unlock(sb);
	}

	LEAVE(sb);
}

int
subfs_get_access(struct inode *inode, int rw)
{
	struct super_block *sb = inode->i_sb;
	int rc = 0;
	
	ENTER(sb);

	subfs_lock(sb);
	if (is_inode_obsolete(inode))
		rc = -ESTALE;
	else {
		struct supermount_inode_info *sii = supermount_i(inode);

		if (rw) {
			SUPERMOUNT_BUG_LOCKED_ON(sb, sii->writecount < 0);
			rc = subfs_get_write_access(sb);
			if (!rc)
				sii->writecount++;
		} else {
			SUPERMOUNT_BUG_LOCKED_ON(sb, sii->readcount < 0);
			subfs_get_read_access(sb);
			sii->readcount++;
		}
	}
	subfs_unlock(sb);

	LEAVE(sb, "rc=%d", rc);

	return rc;
}

void
subfs_put_access(struct inode *inode, int rw)
{
	struct super_block *sb = inode->i_sb;

	ENTER(sb);

	subfs_lock(sb);
	if (!is_inode_obsolete(inode)) {
		struct supermount_inode_info *sii = supermount_i(inode);

		if (rw) {
			SUPERMOUNT_BUG_LOCKED_ON(sb, sii->writecount <= 0);
			sii->writecount--;
			subfs_put_write_access(sb);
		} else {
			SUPERMOUNT_BUG_LOCKED_ON(sb, sii->readcount <= 0);
			sii->readcount--;
			subfs_put_read_access(sb);
		}

	}
	subfs_unlock(sb);

	LEAVE(sb);
}

struct vfsmount *
subfs_go_online(struct super_block *sb)
{
	int rc;
	struct vfsmount *mnt;

	ENTER(sb);

	subfs_lock(sb);

	rc = check_and_remount_subfs(sb);
	mnt = ERR_PTR(rc);
	if (!rc)
		mnt = mntget(subfs_mnt(sb));
	subfs_unlock(sb);

	LEAVE(sb, "mnt=%p", mnt);

	return mnt;
}

void
subfs_go_offline(struct super_block *sb, struct vfsmount *mnt)
{
	ENTER(sb);

	SUPERMOUNT_BUG_ON(!mnt);
	mntput(mnt);

	LEAVE(sb);

}

struct vfsmount *
subfs_prevent_umount(struct super_block *sb)
{
	struct vfsmount *mnt;

	ENTER(sb);

	subfs_lock(sb);
	mnt = subfs_mnt(sb);
	if (mnt)
		mnt = mntget(mnt);
	subfs_unlock(sb);

	LEAVE(sb, "mnt=%p", mnt);

	return mnt;
}

void
subfs_allow_umount(struct super_block *sb, struct vfsmount *mnt)
{
	ENTER(sb);

	SUPERMOUNT_BUG_ON(!mnt);
	mntput(mnt);

	LEAVE(sb);

}

struct vfsmount *
subfs_get_mnt(struct super_block *sb)
{
	struct vfsmount *mnt = NULL;

	ENTER(sb);

	subfs_lock(sb);
	if (subfs_is_mounted(sb))
		mnt = mntget(subfs_mnt(sb));
	subfs_unlock(sb);

	LEAVE(sb, "mnt=%p", mnt);

	return mnt;
}

struct super_block *
subfs_get_sb(struct super_block *sb)
{
	struct super_block *subsb = NULL;

	ENTER(sb);

	subfs_lock(sb);
	if (subfs_is_mounted(sb))
		subsb = subfs_sb(sb);
	subfs_unlock(sb);

	LEAVE(sb, "subsb=%p", subsb);

	return subsb;
}

/*
 * contrary to its name this function deals with _supermount_ inode
 */
void
subfs_clear_inode(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;
	struct supermount_inode_info *sii = supermount_i(inode);

	ENTER(sb, "inode=%p", inode);

	subfs_lock(sb);

	/*
	 * this is safe. If subfs has been unmounted, counters has been
	 * set to 0 already
	 */
	while(sii->writecount > 0) {
		sii->writecount--;
		subfs_put_write_access(sb);
	}

	while(sii->readcount > 0) {
		sii->readcount--;
		subfs_put_read_access(sb);
	}

	list_del(&sii->list);

	subfs_unlock(sb);

	LEAVE(sb, "inode=%p", inode);
}
