/*
 *  linux/fs/supermount/file.c
 *
 *  Original version:
 *      Copyright (C) 1995, 1997
 *      Stephen Tweedie (sct@dcs.ed.ac.uk)
 *
 *      from
 *
 *      linux/fs/minix/dir.c
 *      Copyright (C) 1991, 1992  Linus Torvalds
 *
 *      and
 *
 *      linux/fs/ext2/dir.c
 *      Copyright (C) 1992, 1993, 1994, 1995  Remy Card
 *
 *  Rewriten for kernel 2.2 & 2.4. (C) 1999, 2000 Alexis Mikhailov
 *                                    (alexis@abc.cap.ru)
 *  Rewriten for kernel 2.4. (C) 2001 MandrakeSoft Inc.
 *                                    Juan Quintela (quintela@mandrakesoft.com)
 *  Rewritten for kernel 2.4.21 (C) 2003 Andrey Borzenkov
 *                                       (arvidjaar@mail.ru)
 *  Rewritten for kernel 2.5.70 (C) 2003 Andrey Borzenkov
 *                                       (arvidjaar@mail.ru)
 *
 *  $Id: file.c,v 1.13.4.4 2003/07/13 14:52:43 bor Exp $
 */

#define S_DBG_TRACE_CURRENT S_DBG_TRACE_FILE
#include "supermount.h"

#define DEFAULT_POLLMASK (POLLIN | POLLOUT | POLLRDNORM | POLLWRNORM)

static inline int
init_file_info(struct file *file, unsigned int fake)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct supermount_file_info *sfi;
	int rc;

	ENTER(sb, "file=%p fake=%d", file, fake);

	rc = 1;
	sfi = kmalloc(sizeof(*sfi), GFP_KERNEL);
	if (!sfi)
		goto out;

	memset(sfi, 0, sizeof(*sfi));

	INIT_LIST_HEAD(&sfi->list);
	sfi->host = file;
	sfi->owner = current->pid;
	sfi->vm_ops = 0;
	sfi->file = 0;
	sfi->fake = fake;
	
	file->f_supermount = sfi;

	rc = 0;
out:
	LEAVE(sb, "file=%p fake=%d rc=%d", file, fake, rc);

	return rc;
}

static inline void
attach_subfs_file(struct file *file, struct file *subfile)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	struct supermount_file_info *sfi;

	ENTER(sb, "file=%p subfile=%p", file, subfile);

	sfi = supermount_f(file);
	sfi->file = subfile;
	list_add(&sfi->list, &sbi->s_files);

	LEAVE(sb, "file=%p subfile=%p", file, subfile);
}

static inline int
prepare_file(struct file *file, struct file *subfile)
{
	struct super_block *sb = file->f_dentry->d_sb;
	int rc;

	ENTER(sb, "file=%p subfile=%p", file, subfile);

	subfs_lock(sb);

	rc = -ENOMEDIUM;
	if (!subfs_is_mounted(sb))
		goto out;

	rc = -ESTALE;
	if (is_dentry_obsolete(file->f_dentry))
		goto out;

	rc = -ENOMEM;
	if (init_file_info(file, 0))
		goto out;

	attach_subfs_file(file, subfile);
	rc = 0;

out:
	subfs_unlock(sb);

	LEAVE(sb, "file=%p subfile=%p", file, subfile);

	return rc;
}

struct file *
get_subfs_file(struct file *file)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct supermount_file_info *sfi = supermount_f(file);
	struct file *err;

	ENTER(sb, "file=%p", file);

	subfs_lock(sb);
	
	err = ERR_PTR(-ENOMEDIUM);
	if (!subfs_is_mounted(sb))
		goto out;

	err = ERR_PTR(-ESTALE);
	if (is_file_fake(file) || is_file_obsolete(file))
		goto out;

	err = sfi->file;
	SUPERMOUNT_BUG_LOCKED_ON(sb, !err->f_dentry);
	SUPERMOUNT_BUG_LOCKED_ON(sb, !err->f_dentry->d_inode);
	get_file(err);

out:
	subfs_unlock(sb);

	LEAVE(sb, "file=%p subfile=%p", file, err);

	return err;
}

static loff_t
supermount_llseek(struct file *file, loff_t offset, int origin)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct file *subfile;
	loff_t rc;

	ENTER(sb, "file=%p offset=%lld origin=%d", file, offset, origin);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	if (subfile->f_op && subfile->f_op->llseek)
		rc = subfile->f_op->llseek(subfile, offset, origin);
	else
		rc = default_llseek(subfile, offset, origin);
	file->f_pos = subfile->f_pos;
	file->f_version = subfile->f_version;

	fput(subfile);
out:
	LEAVE(sb, "file=%p rc=%lld", file, rc);

	return rc;
}

static ssize_t
supermount_read(struct file *file, char *buf, size_t count, loff_t * ppos)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct inode *inode = file->f_dentry->d_inode;
	struct file *subfile;
	int write_on = NEED_WRITE_ATIME(inode);
	int rc;

	ENTER(sb, "file=%p", file);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = -EINVAL;
	if (!subfile->f_op || !subfile->f_op->read)
		goto put_subfile;

	rc = subfs_get_access(inode, write_on);
	if (rc)
		goto put_subfile;

	rc = subfile->f_op->read(subfile, buf, count, ppos);
	subfs_put_access(inode, write_on);
	if (rc < 0)
		goto put_subfile;

	inode->i_atime = subfile->f_dentry->d_inode->i_atime;

	if (rc > 0)
		file->f_pos = subfile->f_pos = *ppos;

put_subfile:
	fput(subfile);
out:
	LEAVE(sb, "file=%p rc=%d", file, rc);

	return rc;
}

static ssize_t
supermount_write(struct file *file, const char *buf,
		 size_t count, loff_t * ppos)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct file *subfile;
	int rc;

	ENTER(sb, "file=%p", file);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = 0;
	if (subfile->f_op && subfile->f_op->write)
		rc = subfile->f_op->write(subfile, buf, count, ppos);
	if (rc > 0) {
		struct inode *subinode = subfile->f_dentry->d_inode;

		file->f_pos = subfile->f_pos = *ppos;
		file->f_mode = subfile->f_mode;
		file->f_dentry->d_inode->i_size = subinode->i_size;
		file->f_dentry->d_inode->i_blocks = subinode->i_blocks;
		file->f_dentry->d_inode->i_mode = subinode->i_mode;
		file->f_dentry->d_inode->i_ctime = subinode->i_ctime;
		file->f_dentry->d_inode->i_mtime = subinode->i_mtime;
	}

	fput(subfile);
out:
	LEAVE(sb, "file=%p rc=%d", file, rc);

	return rc;
}

int
supermount_readdir(struct file *file, void *buf, filldir_t fill_fn)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct inode *inode = file->f_dentry->d_inode;
	struct file *subfile;
	int write_on = NEED_WRITE_ATIME(inode);
	int fake_readdir = 1;
	int rc;

	ENTER(sb, "file=%p", file);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = -ENOTDIR;
	if (!subfile->f_op || !subfile->f_op->readdir)
		goto put_subfile;

	rc = subfs_get_access(inode, write_on);
	if (rc)
		goto put_subfile;

	/* FIXME should it go before get_access? */
	fake_readdir = 0;
	rc = subfile->f_op->readdir(subfile, buf, fill_fn);
	subfs_put_access(inode, write_on);
	if (rc)
		goto put_subfile;

	inode->i_atime = subfile->f_dentry->d_inode->i_atime;
	file->f_pos = subfile->f_pos;

put_subfile:
	fput(subfile);
out:
	if (fake_readdir && is_file_fake(file)) {
		/* cf. supermount_open */
		rc = 0;
	}
	LEAVE(sb, "file=%p rc=%d fpos=%lld", file, rc, file->f_pos);

	return rc;
}

static unsigned int
supermount_poll(struct file *file, struct poll_table_struct *table)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct file *subfile;
	int rc;

	ENTER(sb, "file=%p", file);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = DEFAULT_POLLMASK;
	if (subfile->f_op && subfile->f_op->poll)
		rc = subfile->f_op->poll(subfile, table);

	fput(subfile);
out:
	LEAVE(sb, "file=%p rc=%d", file, rc);

	return rc;
}

static int
supermount_ioctl(struct inode *inode, struct file *file,
		 unsigned int cmd, unsigned long arg)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct file *subfile;
	struct inode *subinode;
	int rc;

	ENTER(sb, "file=%p cmd=%u arg=%lu", file, cmd, arg);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = -ENOTTY;
	subinode = subfile->f_dentry->d_inode;
	if (subfile->f_op && subfile->f_op->ioctl)
		rc = subfile->f_op->ioctl(subinode, subfile, cmd, arg);

	/* flags may have been changed by ioctl */
	if (!rc)
		set_inode_flags(file->f_dentry->d_inode, subinode);

	fput(subfile);
out:
	LEAVE(sb, "file=%p rc=%d", file, rc);

	return rc;
}

int
supermount_open(struct inode *inode, struct file *file)
{
	struct super_block *sb = inode->i_sb;
	struct dentry *subdent;
	struct file *subfile = 0;
	struct vfsmount *submnt;
	int write_on = file->f_mode & FMODE_WRITE;
	int fake_open = 1;
	int rc;

	ENTER(sb, "inode=%p file=%p", inode, file);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	submnt = subfs_get_mnt(sb);
	if (!submnt)
		goto out;

	subdent = get_subfs_dentry(file->f_dentry);
	rc = PTR_ERR(subdent);
	if (IS_ERR(subdent))
		goto put_submnt;
	
	rc = subfs_get_access(inode, write_on);
	if (rc)
		goto put_subdent;

	/*
	 * the following is used to simplify error processing. dentry_open
	 * automatically does mntput and dput in error case, this may result
	 * in subfs being destroyed
	 * We just make sure we need to do mntput exactly once here;
	 * additionally it guards against accidental remounting of subfs
	 * until we has cleaned up
	 */
	submnt = mntget(submnt);
	subdent = dget(subdent);

	subfile = dentry_open(subdent, submnt, file->f_flags);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto put_access;
	/*
	 * no need to do extra mntput and dput, it is done automatically in
	 * dentry_open on error
	 */

	rc = prepare_file(file, subfile);
	if (rc)
		goto put_subfile;

	subfile->f_mode = file->f_mode;
	/*
	 * this is needed for mmap to work. In current model vm_area
	 * is associated with superfile; we never explicitly call
	 * any vm method with subfile as pointer. But many drivers
	 * attach private structures to this field and mmap of special
	 * files on supermount fs won't work without it
	 */
	file->private_data = subfile->private_data;
	/*
	 * we have real subfile now, do not fake anything
	 */
	fake_open = 0;

	/*
	 * Now get rid of extra mntget and dget
	 */
	goto put_subdent;

	/*
	 * error cleanup
	 */

put_subfile:
	fput(subfile);
	subfile = 0;
put_access:
	subfs_put_access(inode, write_on);
put_subdent:
	dput(subdent);
put_submnt:
	mntput(submnt);
out:
	if (fake_open && inode == sb->s_root->d_inode) {
		/*
		 * always appear to succeed for root open. It allows active
		 * monitoring of mountpoint using FAM/dnotify and also is less
		 * surprising for other programs
		 */
		rc = init_file_info(file, 1);
		if (rc)
			rc = -ENOMEM;
	}
	LEAVE(sb, "inode=%p file=%p rc=%d subfile=0x%p", inode, file, rc, subfile);

	return rc;
}

static int
supermount_flush(struct file *file)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct file *subfile;
	int fake_flush = 1;
	int rc;

	ENTER(sb, "file=%p", file);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = 0;
	fake_flush = 0;
	if (subfile->f_op && subfile->f_op->flush)
		rc = subfile->f_op->flush(subfile);

	fput(subfile);
out:
	if (fake_flush && is_file_fake(file)) {
		/* cf. supermount_open */
		rc = 0;
	}
	LEAVE(sb, "file=%p rc=%d fake=%d", file, rc, fake_flush);

	return rc;
}

/*
 * if subfile is NULL it has already been released in supermount_clean_files
 * together with adjusting open/write counters. Else we do it here.
 *
 * The reason is, it may be called long after media has been changed
 * and we definitely do not want this function to mess up the
 * new subfs state.
 */
static int
supermount_release(struct inode *inode, struct file *file)
{
	struct file *subfile = 0;
	struct super_block *sb = inode->i_sb;
	struct supermount_file_info *sfi = file->f_supermount;

	ENTER(sb, "inode=%p file=%p", inode, file);

	subfs_lock(sb);
	/*
	 * FIXME
	 * this sucks. But there does not seem to be any way
	 * to distinguish between ENOMEM on _open (legitimate
	 * case) and anything else (plain bug)
	 */
	if (sfi) {
		list_del(&sfi->list);
		subfile = sfi->file;
		sfi->file = 0;
	} else
		supermount_warning(sb, "no supermount file info attached");
	subfs_unlock(sb);

	if (subfile) {
		int bug = atomic_read(&subfile->f_count) != 1;
		fput(subfile);
		subfs_put_access(inode, file->f_mode & FMODE_WRITE);
		SUPERMOUNT_BUG_ON(bug);
	}

	if (sfi)
		kfree(sfi);

	LEAVE(sb, "inode=%p file=%p", inode, file);

	return 0;

}

static int
supermount_fsync(struct file *file, struct dentry *dentry, int datasync)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct file *subfile;
	int rc;

	ENTER(sb, "file=%p dentry=%s sync=%d", file, dentry->d_name.name, datasync);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = -EINVAL;
	if (subfile->f_op && subfile->f_op->fsync)
		rc = subfile->f_op->fsync(subfile, subfile->f_dentry, datasync);

	fput(subfile);
out:
	ENTER(sb, "file=%p dentry=%s rc=%d", file, dentry->d_name.name, rc);

	return rc;
}

static int
supermount_fasync(int fd, struct file *file, int on)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct file *subfile;
	int rc;

	ENTER(sb, "fd=%d file=%p on=%d", fd, file, on);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = -EINVAL;
	if (subfile->f_op && subfile->f_op->fasync)
		rc = subfile->f_op->fasync(fd, subfile, on);

	fput(subfile);
out:
	LEAVE(sb, "fd=%d file=%p rc=%d", fd, file, rc);

	return rc;
}

static int
supermount_lock(struct file *file, int cmd, struct file_lock *fl)
{
	struct super_block *sb = file->f_dentry->d_sb;
	struct file *subfile;
	int rc;

	ENTER(sb, "file=%p cmd=%d", file, cmd);

	rc = -ESTALE;
	if (subfs_check_disk_change(sb))
		goto out;

	subfile = get_subfs_file(file);
	rc = PTR_ERR(subfile);
	if (IS_ERR(subfile))
		goto out;

	rc = 0;
	if (subfile->f_op && subfile->f_op->lock)
		rc = subfile->f_op->lock(subfile, cmd, fl);
	else if (cmd == F_GETLK)
		posix_test_lock(file, fl);

	fput(subfile);
out:
	LEAVE(sb, "file=%p rc=%d", file, rc);

	return rc;
}

/* Fixme:
 * readv: easy, export churnk from vfs
 * writev: easy, export churnk from vfs
 * sendpage: only used for networking, not needed
 * get_unmmapped_area: only used for devices, not needed
 */

struct file_operations supermount_dir_operations = {
	.llseek		= supermount_llseek,
	.read		= supermount_read,
	.readdir	= supermount_readdir,
	.ioctl		= supermount_ioctl,
	.open		= supermount_open,
	.flush		= supermount_flush,
	.release	= supermount_release,
	.fsync		= supermount_fsync,
	.fasync		= supermount_fasync,
};

struct file_operations supermount_file_operations = {
	.llseek		= supermount_llseek,
	.read		= supermount_read,
	.write		= supermount_write,
	.poll		= supermount_poll,
	.ioctl		= supermount_ioctl,
	.mmap		= supermount_file_mmap, /* from filemap.c */
	.open		= supermount_open,
	.flush		= supermount_flush,
	.release	= supermount_release,
	.fsync		= supermount_fsync,
	.fasync		= supermount_fasync,
	.lock		= supermount_lock,
};
