/* RWDOCS */
/* filename	: motor.c
 * purpose	: provide event drive and support routines
 *		: for cohulip package.
 * author	: Randy Wright
 * copyright	: ((C)) Copyright 1993 Randy Wright
 *
 *
 * this file is copyright and not public domain. anyone
 * may copy this file, redistribute it and store it in any
 * medium they see fit provided they do not charge for the
 * "content" of this file and they do not remove this notice
 * or the copyright notice.
 * 
 *
 *

method: motor provides a poll driven metabolism 
for cohulip. Cohulip is derived from ka9q code 
that used a commutator loop as the main drive. 
Commutator archetecture is suitable for a single
user user environment but not for mulitasking, 
mulituser systems. On the other hand, the poll 
driven archetecture provided by this file is not
portable to any single user environments that I 
know of. This simple poll based code will only work
on operating systems that provide a poll service.
This code is specifically designed to run on
Coherent 4.2.

How it works:
A call to makepiston() will put a device onto the list
of polled devices, while rmpiston() will stop polling
the device.

A single call to revolve() will start the polling. revolve()
does not return. Instead exit is called when needed. Revolve
calls poll on the various devices and when a revents for a
given device is non zero, revolve calls a function that was
registered with makepiston. the call to poll in revolve times
out after throttle miliseconds. This allows timers to be updated,
the loopback non-device to be updated and any remaining telunix
stuff to be cleaned up. 

The list of fd's that are to be polled are kept in a
struct pollfd[] called piston. The data to link to the
device calls is kept in a CYLINDER. The individual fd
is the index into these arrays. The value MAXPOLLS in motor.h
limits the number of CYLINDERS in crankshaft[].

The int throttle can be varied dynamically from elsewhere in
the program. A lower number makes timeouts come more quickly.

If you decide to add a device, you'll need to include motor.h
in your code and do a makepiston to get your device onto the
polling list. If your device is a network device, makepiston()
will fill in the interface struct member cylinder with the fd.
This will be needed later if you want to stop polling on your
device. Always stop polling with rmpiston BEFORE you close()
the device.

The function initpistons() should not be used.
 */

#include <stdio.h>
#include "unixopt.h"
#include "global.h"
#include "motor.h"
#include "tcp.h"
#include "ip.h"
#include "mbuf.h"
#include "telnet.h"

#ifdef TRACE
#include "trace.h"
extern struct interface loopback; /* in main.c */
#endif /* TRACE */

extern struct mbuf *loopq;

struct pollfd piston[MAXPOLLS];
CYLINDER crankshaft[MAXPOLLS];

#ifdef TELUNIX
extern struct tcp *tnix_tcb;
#endif /* TELUNIX */

int numcyls = MAXPOLLS;	/* the current number of pistons */
int throttle = 1000;	/* the timeout value for poll */

int
revolve()
{

/* RWDOCE */
	int n, i;
	struct mbuf *bp;
	extern void check_time();
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif

	for( ; ; ) {
			/* note that an interface will not
			 * receive poll service if its fd is not on the
			 * piston list, or if its calls are NULLFP
			 * or if its piston[].events is set to zero.
			 * the only other way a routine can get serviced
			 * is if it is on the timer chain. Almost any
			 * active interface should be set up for POLLIN.
			 * however, output may be the result of processing
			 * input or retransmission from a timeout. If an
			 * entire datagram is handled in a single output stroke
			 * by a given interface (such as ethernet) there is
			 * probably no need for output polling. 
			 */
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
		n = poll( piston, numcyls, throttle);

			/* this loop continues until all polls
			 * have been resolved or MAXPOLLS is
			 * reached. 
			 */
		for( i = 0; i < MAXPOLLS && n; i++ ) {
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
			if( piston[i].revents & POLLIN ) {
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
				n--;
				if( crankshaft[i].r_stroke != NULL )
				(* crankshaft[i].r_stroke)(crankshaft[i].ifp);
			}
			if( piston[i].revents & POLLOUT ) {
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
				n--;
				if( crankshaft[i].t_stroke != NULL )
				(* crankshaft[i].t_stroke)(crankshaft[i].ifp);
			}
	
			if( piston[i].revents & (POLLPRI | POLLERR | POLLHUP)) {
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
				n--;
				if( crankshaft[i].r_stroke != NULL )
				(* crankshaft[i].p_stroke	)(crankshaft[i].ifp);
			}
		}
				/* this call provides timed services
				 * for various timers. The timer chain
				 * can cause transmissions and can call
				 * periodic functions.
				 */
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
			check_time();

		/* Service the loopback queue */
		if((bp = dequeue(&loopq)) != NULLBUF) {
			struct ip ip;
#ifdef  TRACE
			dump(&loopback,IF_TRACE_IN,TRACE_IP,bp);
#endif
			/* Extract IP header */
			ntohip(&ip,&bp);
			ip_recv(&ip,bp,0);
		}
#ifdef TELUNIX
				/* process leftover tcp input */
			if( (struct tcp * )tnix_tcb != (struct tcp *)NULLTCB )
				 tnix_scan();
#endif /* TELUNIX */
	}
}

/* RWDOCS */
/* add a piston to the crankshaft */
int
makepiston(fd, r_call, t_call, p_call, ifp, events)
int fd;			/* file descriptor for polling */

				/* unused function pointers
				 * should be set to NULLFP */
void (*r_call)();	/* call when data arrives */
void (*t_call)();	/* call when space is avaliable for output */
void (*p_call)();	/* call when priority or hangup occurs */

struct interface *ifp;	/* the interface pointer for fd (NULL if none) */

int events;		/* initial events setting */
{

/* RWDOCE */
	int n;

#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
	n = -1; /*  numcyls; */
	if( fd < 0 ) {
		fprintf( stderr, "motor: out of cylinders\n" );
		return -1;
	}
	
	if( fd < MAXPOLLS ) {
	
			/* set up the calls */
		crankshaft[fd].r_stroke = r_call;
		crankshaft[fd].t_stroke = t_call;
		crankshaft[fd].p_stroke = p_call;
		crankshaft[fd].ifp = ifp;
		crankshaft[fd].status = 0;
	
			/* set up the pollfd struct */
		piston[fd].fd = fd;
		piston[fd].events = events;
		piston[fd].revents = 0;
	
		n = fd;	
		/* give access to the struct so the
			 * intercase routine can flow control
			 */
		if( ifp != NULLIF )
		ifp->cylinder = fd;
	
			/* update the number of pfds * /
		numcyls++; */
	
	
	
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
		/* return the index to the caller */
		return(n);
	} else	{
		printf( "can't add %d to %d cyls\n", fd, numcyls );
#ifdef COHPROF
	profile(__LINE__,__FILE__);
#endif
			/* no cylinders left, return error */
		return( -1 );
	}
}

/* RWDOCS */
/* stop polling on fd, call before closing fd */
int
rmpiston(fd)
int fd;
{

/* RWDOCE */
	if( fd < 0 ) {
		fprintf( stderr, "rmpiston : close called more than once\n" );
		return -1;
	}
	if( fd < MAXPOLLS ) {
	
			/* set up the calls */
		crankshaft[fd].r_stroke = NULL;
		crankshaft[fd].t_stroke = NULL;
		crankshaft[fd].p_stroke = NULL;
		crankshaft[fd].ifp = NULLIF;
		crankshaft[fd].status = 0;
	
			/* set up the pollfd struct */
		piston[fd].fd = -1;
		piston[fd].events = 0;
		piston[fd].revents = 0;
	}
	return 0;
}

/** end of motor.c **/
