/*
  Copyright (C) 2000-2008

  Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_vector.cc 1885 2008-03-18 15:24:56Z paultcochrane $
*/

/*! @file xmds_vector.cc
  @brief Vector object classes and methods

  More detailed explanation...
*/

#include <xmds_common.h>
#include <xmds_vector.h>
#include <xmds_simulation.h>

// **************************************************************************
// **************************************************************************
//                              xmdsVector public
// **************************************************************************
// **************************************************************************

extern bool debugFlag;

long nxmdsVectors=0;  //!< The number of xmds vectors

// **************************************************************************
xmdsVector::xmdsVector(
                       const xmdsField *const yourField) :
  myField(yourField) {
  if (debugFlag) {
    nxmdsVectors++;
    printf("xmdsVector::xmdsVector\n");
    printf("nxmdsVectors=%li\n", nxmdsVectors);
  }
  myNeedsFFTWRoutines=0;
}

// **************************************************************************
xmdsVector::~xmdsVector() {
  if (debugFlag) {
    nxmdsVectors--;
    printf("xmdsVector::~xmdsVector\n");
    printf("nxmdsVectors=%li\n", nxmdsVectors);
  }
}

// **************************************************************************
void xmdsVector::writeDefines(
                              FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsVector::writeDefines\n");
  }
  
  fprintf(outfile, "// vector %s defines\n", myName.c_str());
  if (myComponentsLengthsList.size()>0) {
    long totalComponents = 0;
    for (list<long>::const_iterator pLong = myComponentsLengthsList.begin(); pLong != myComponentsLengthsList.end(); pLong++) {
      totalComponents += *pLong;
    }
    fprintf(outfile, "#define _%s_%s_ncomponents %li\n", field()->name()->c_str(), myName.c_str(), totalComponents);
  }
  else {
    fprintf(outfile, "#define _%s_%s_ncomponents %li\n", field()->name()->c_str(), myName.c_str(), (long)myComponentsNamesList.size());
  }
  
  if (myComponentsLengthsList.size()>0) {
    if (myComponentsLengthsList.size()!=myComponentsNamesList.size()) {
      throw xmdsException("List of component lengths has a different size than the list of components");
    }
    else {
      unsigned long totalComponents = 0;
      list<long>::const_iterator pLong = myComponentsLengthsList.begin();
      list<XMLString>::const_iterator pXMLString = myComponentsNamesList.begin();
      
      if (*pLong==1) {
        fprintf(outfile, "#define %s _active_%s_%s[_%s_%s_index_pointer]\n",
            pXMLString->c_str(), field()->name()->c_str(), myName.c_str(), field()->name()->c_str(), myName.c_str());
      }
      else if (*pLong>1) {
        fprintf(outfile, "#define %s(j) _active_%s_%s[_%s_%s_index_pointer + j - 1]\n",
            pXMLString->c_str(), field()->name()->c_str(), myName.c_str(), field()->name()->c_str(), myName.c_str());
//  Moved this next line into the integrate blocks themselves
//        fprintf(outfile, "#define d%s_d%s(j) d%s_d%s[j-1]\n",
//            pXMLString->c_str(),field()->simulation()->parameters()->propDimName.c_str(),pXMLString->c_str(),field()->simulation()->parameters()->propDimName.c_str());
      }
      else
        throw xmdsException("Component lengths has a nonpositive integer.");
      
      totalComponents+=*pLong;
      pLong++;
      pXMLString++;
      
      while (pXMLString != myComponentsNamesList.end()) {
        if (*pLong==1) {
          fprintf(outfile, "#define %s _active_%s_%s[_%s_%s_index_pointer + %li]\n",
              pXMLString->c_str(), field()->name()->c_str(), myName.c_str(), field()->name()->c_str(), myName.c_str(),totalComponents);
        }
        else if (*pLong>1) {
          fprintf(outfile, "#define %s(j) _active_%s_%s[_%s_%s_index_pointer + j + %li]\n",
              pXMLString->c_str(), field()->name()->c_str(), myName.c_str(), field()->name()->c_str(), myName.c_str(),totalComponents-1);
//  Moved this next line into the integrate blocks themselves
//          fprintf(outfile, "#define d%s_d%s(j) d%s_d%s[j-1]\n",
//              pXMLString->c_str(),field()->simulation()->parameters()->propDimName.c_str(),pXMLString->c_str(),field()->simulation()->parameters()->propDimName.c_str());
        }
        else
          throw xmdsException("Component lengths has a nonpositive integer.");
          totalComponents+=*pLong;
        pLong++;
        pXMLString++;
      }
    }
  }
  else {
    unsigned long i=0;
    for (list<XMLString>::const_iterator pXMLString = myComponentsNamesList.begin(); pXMLString != myComponentsNamesList.end(); pXMLString++) {
      fprintf(outfile, "#define %s _active_%s_%s[_%s_%s_index_pointer + %li]\n",
          pXMLString->c_str(), field()->name()->c_str(), myName.c_str(), field()->name()->c_str(), myName.c_str(), i);
      i++;
    }
  }
  fprintf(outfile, "\n");
                }

// **************************************************************************
void xmdsVector::writeGlobals(
                              FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsVector::writeGlobals\n");
  }

  fprintf(outfile, "// vector %s globals \n", myName.c_str());
  fprintf(outfile, "\n");

  const char* typeName="";
  if (myType==COMPLEX) {
    typeName="complex";
  }
  else if (myType==DOUBLE) {
    typeName="double";
  }

  if ((field()->simulation()->parameters()->usempi)&!(field()->simulation()->parameters()->stochastic)){
    fprintf(outfile, "%s *_%s_%s;\n", typeName, field()->name()->c_str(), myName.c_str());
    fprintf(outfile, "%s* _active_%s_%s;\n", typeName, field()->name()->c_str(), myName.c_str());
    fprintf(outfile, "%s *_%s_%s_work;\n", typeName, field()->name()->c_str(), myName.c_str());
    fprintf(outfile, "\n");
  }
  else if (field()->simulation()->parameters()->fftwVersion == 2) {
    fprintf(outfile, "%s *_%s_%s = new %s[_%s_size*_%s_%s_ncomponents];\n", typeName,
            field()->name()->c_str(), myName.c_str(), typeName,
            field()->name()->c_str(), field()->name()->c_str(), myName.c_str());
    fprintf(outfile, "%s* _active_%s_%s;\n", typeName, field()->name()->c_str(), myName.c_str());
    fprintf(outfile, "\n");
  }
  else if (field()->simulation()->parameters()->fftwVersion == 3) {
    fprintf(outfile, "%s *_%s_%s = (%s*) fftw_malloc(_%s_size*_%s_%s_ncomponents*sizeof(%s));\n", typeName,
            field()->name()->c_str(), myName.c_str(), typeName,
            field()->name()->c_str(), field()->name()->c_str(), myName.c_str(), typeName);
    fprintf(outfile, "%s* _active_%s_%s;\n", typeName, field()->name()->c_str(), myName.c_str());
    fprintf(outfile, "\n");
  }

  if (!myNeedsFFTWRoutines) {
    return;
  }

  fprintf(outfile, "unsigned long _%s_%s_space;\n", field()->name()->c_str(), myName.c_str());
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsVector::writePrototypes(
                                 FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsVector::writePrototypes\n");
  }

  fprintf(outfile, "// vector %s prototypes \n", myName.c_str());
  fprintf(outfile, "\n");

  fprintf(outfile, "void _%s_%s_initialise();\n", field()->name()->c_str(), name()->c_str());
  fprintf(outfile, "\n");

  if (!myNeedsFFTWRoutines) {
    return;
  }

  fprintf(outfile, "void _%s_%s_go_space(\n", field()->name()->c_str(), myName.c_str());
  fprintf(outfile, "     const unsigned long& _space);\n");
  fprintf(outfile, "\n");

}

// **************************************************************************
void xmdsVector::writeRoutines(
                               FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsVector::writeRoutines\n");
  }

  const char *const fieldName = field()->name()->c_str();

  fprintf(outfile, "// ********************************************************\n");
  fprintf(outfile, "//           field %s vector %s routines\n", fieldName, myName.c_str());
  fprintf(outfile, "// ********************************************************\n");
  fprintf(outfile, "\n");

  this->writeInitialiseRoutine(outfile);

  if (!myNeedsFFTWRoutines) {
    return;
  }

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _%s_%s_go_space(\n", fieldName, myName.c_str());
  fprintf(outfile, "     const unsigned long& _newspace) {\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "if (_%s_%s_space==_newspace)\n", fieldName, myName.c_str());
  fprintf(outfile, "     return;\n");
  fprintf(outfile, "\n");

  unsigned long fullSpace = myField->geometry()->fullSpace();

  fprintf(outfile, "double _c=1.0;\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "if ((_%s_%s_space==0)&(_newspace==%li)) {\n", fieldName, myName.c_str(), fullSpace);
  fprintf(outfile, "\n");

  if (field()->simulation()->parameters()->nThreads>1 && field()->simulation()->parameters()->fftwVersion == 2) {
    fprintf(outfile, "   fftwnd_threads(_num_threads, _%s_forward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_ncomponents, 1, 0, 0, 0);\n",
            fieldName, fieldName, myName.c_str(),
            fieldName, myName.c_str(), fieldName, myName.c_str());
  }
  else if ((field()->simulation()->parameters()->usempi) && !(field()->simulation()->parameters()->stochastic)) {
    fprintf(outfile, "   fftwnd_mpi(_%s_forward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_TRANSPOSED_ORDER);\n",
            fieldName, fieldName, myName.c_str(), fieldName, myName.c_str(),
            fieldName, myName.c_str());
  }
  else if (field()->simulation()->parameters()->fftwVersion == 2) {
    fprintf(outfile, "   fftwnd(_%s_forward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_ncomponents, 1, 0, 0, 0);\n",
            fieldName, fieldName, myName.c_str(),
            fieldName, myName.c_str(), fieldName, myName.c_str());
  }
  else if (field()->simulation()->parameters()->fftwVersion == 3) {
    fprintf(outfile, "   fftw_execute_dft(_%s_%s_forward_plan, reinterpret_cast<fftw_complex*>(_active_%s_%s), reinterpret_cast<fftw_complex*>(_active_%s_%s));\n",
            fieldName, myName.c_str(), fieldName, myName.c_str(), fieldName, myName.c_str());
  }
  fprintf(outfile, "\n");
  for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
    fprintf(outfile, "   _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "     _%s_%s_space=_newspace;\n", fieldName, myName.c_str());
  fprintf(outfile, "     }\n");

  fprintf(outfile, "else if ((_%s_%s_space==%li)&(_newspace==0)) {\n", fieldName, myName.c_str(), fullSpace);
  fprintf(outfile, "\n");
  if (field()->simulation()->parameters()->nThreads>1 && field()->simulation()->parameters()->fftwVersion == 2) {
    fprintf(outfile, "   fftwnd_threads(_num_threads, _%s_backward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_ncomponents, 1, 0, 0, 0);\n",
            fieldName, fieldName, myName.c_str(),
            fieldName, myName.c_str(), fieldName, myName.c_str());
  }
  else if ((field()->simulation()->parameters()->usempi)&!(field()->simulation()->parameters()->stochastic)) {
    fprintf(outfile, "   fftwnd_mpi(_%s_backward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_TRANSPOSED_ORDER);\n",
            fieldName, fieldName, myName.c_str(), fieldName, myName.c_str(),
            fieldName, myName.c_str());
  }
  else if (field()->simulation()->parameters()->fftwVersion == 2) {
    fprintf(outfile, "   fftwnd(_%s_backward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_ncomponents, 1, 0, 0, 0);\n",
            fieldName, fieldName, myName.c_str(),
            fieldName, myName.c_str(), fieldName, myName.c_str());
  }
  else if (field()->simulation()->parameters()->fftwVersion == 3) {
    fprintf(outfile, "   fftw_execute_dft(_%s_%s_backward_plan, reinterpret_cast<fftw_complex*>(_active_%s_%s), reinterpret_cast<fftw_complex*>(_active_%s_%s));\n",
            fieldName, myName.c_str(), fieldName, myName.c_str(), fieldName, myName.c_str());
  }
  fprintf(outfile, "\n");
  for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
    fprintf(outfile, "   _c *= _%s_dk%li/sqrt(2*M_PI);\n", fieldName, i);
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "     _%s_%s_space=_newspace;\n", fieldName, myName.c_str());
  fprintf(outfile, "     }\n");

  if (myField->geometry()->nDims()>1) {
    if ((field()->simulation()->parameters()->usempi) && !(field()->simulation()->parameters()->stochastic)) {

      // mixed transforms stuff
      // note this does not need to be ultra efficient since it is probably only
      // ever used in filters, moment stuff, and after initialisation

      // For mixed space situations, we need to arbitrarily define when we will be transposed, and when we shall not.
      // This has efficiency effects, as transposing can only be done safely with a full transform.
      // We shall declare that the first two dimensions will only be swapped if both of them are in Fourier space

      // There are three cases.  In order of complexity:
      //   1.  Swapped -> swapped
      //   2.  Unswapped -> unswapped
      //   3.  Change of swappedness

      fprintf(outfile, "else {\n");
      fprintf(outfile, "    long _howmany;\n");
      fprintf(outfile, "    int _lattice;\n");
      fprintf(outfile, "\n");

      //  1. swapped -> swapped
      //     Only need to transform the lower levels normally
      fprintf(outfile, "    if ((_newspace&1)&&(_newspace&2)&&(_%s_%s_space&1)&&(_%s_%s_space&2)) { // swapped -> swapped \n", fieldName, myName.c_str(), fieldName, myName.c_str());
      fprintf(outfile, "           // no preparation before doing the lower levels needs to be done in this case  \n\n");
      fprintf(outfile, "        }  // end of swapped -> swapped case  \n\n");

      //  2. unswapped -> unswapped
      //     If first transverse dimension is unchanged, then we only need to change the lower levels.
      //     Otherwise, we homogenise the space (all to real or all to Fourier), transform keeping order, and then change the lower levels.
      fprintf(outfile, "    else if (!((_newspace&1)&&(_newspace&2))&&!((_%s_%s_space&1)&&(_%s_%s_space&2))) { // unswapped -> unswapped \n", fieldName, myName.c_str(), fieldName, myName.c_str());
      fprintf(outfile, "        if ((_%s_%s_space&1)&&!(_newspace&1)) { // first dimension goes from Fourier space to normal space  \n", fieldName, myName.c_str());
      { // change all the lower levels to Fourier space
        unsigned long two2n=2;
        for (unsigned long i=1; i<myField->geometry()->nDims(); i++) {
          fprintf(outfile, "            _howmany = _%s_%s_ncomponents;\n", fieldName, myName.c_str());
          for (unsigned long j=myField->geometry()->nDims(); j>i+1; j--) {
            fprintf(outfile, "            _howmany *= _%s_lattice%li;\n", fieldName, j-1);
          }
          fprintf(outfile, "            _lattice = _%s_lattice%li;\n",    fieldName, i);
          fprintf(outfile, "            if (!(_%s_%s_space&%li)) {\n", fieldName, myName.c_str(), two2n);
          fprintf(outfile, "               fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_FORWARD, FFTW_IN_PLACE);\n");
          fprintf(outfile, "               for (unsigned long _i0=0; _i0<local_nx");
          for (unsigned long j=1; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "                   fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());
          fprintf(outfile, "               _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
          fprintf(outfile, "               fftwnd_destroy_plan(_plan);\n");
          fprintf(outfile, "            }\n\n");
          two2n *= 2;
        }
      }
      // Change whole field to normal space without changing order
      fprintf(outfile, "         int _fftw_latt[%li];\n", myField->geometry()->nDims());
      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "         _fftw_latt[%li] = _%s_lattice%li;\n", i, fieldName, i);
      }
      fprintf(outfile, "         fftwnd_mpi_plan _plan = fftwnd_mpi_create_plan(MPI_COMM_WORLD, %li, _fftw_latt, FFTW_BACKWARD, FFTW_IN_PLACE|FFTW_ESTIMATE|FFTW_USE_WISDOM);\n",
              myField->geometry()->nDims());

      fprintf(outfile, "         fftwnd_mpi(_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_NORMAL_ORDER);\n",
              fieldName, myName.c_str(), fieldName, myName.c_str(),
              fieldName, myName.c_str());
      fprintf(outfile, "         fftwnd_mpi_destroy_plan(_plan);\n");

      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "            _c *= _%s_dk%li/sqrt(2*M_PI);\n", fieldName, i);
      }
      fprintf(outfile, "     _%s_%s_space=0;\n", fieldName, myName.c_str());
      fprintf(outfile, "          } \n");

      fprintf(outfile, "        else if (!(_%s_%s_space&1)&&(_newspace&1)) { // first dimension goes from normal space to Fourier space  \n", fieldName, myName.c_str());
      { // change all the lower levels to normal space
        unsigned long two2n=2;
        for (unsigned long i=1; i<myField->geometry()->nDims(); i++) {
          fprintf(outfile, "          _howmany = _%s_%s_ncomponents;\n", fieldName, myName.c_str());
          for (unsigned long j=myField->geometry()->nDims(); j>i+1; j--) {
            fprintf(outfile, "          _howmany *= _%s_lattice%li;\n", fieldName, j-1);
          }
          fprintf(outfile, "          _lattice = _%s_lattice%li;\n",    fieldName, i);
          fprintf(outfile, "          if (_%s_%s_space&%li) {\n", fieldName, myName.c_str(), two2n);
          fprintf(outfile, "            fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_BACKWARD, FFTW_IN_PLACE);\n");
          fprintf(outfile, "            for (unsigned long _i0=0; _i0<local_nx");
          for (unsigned long j=1; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "                 fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());
          fprintf(outfile, "             _c *= _%s_dk%li/sqrt(2*M_PI);\n", fieldName, i);
          fprintf(outfile, "             fftwnd_destroy_plan(_plan);\n");
          fprintf(outfile, "           }\n\n");
          two2n *= 2;
        }
      }
      // Change whole field to Fourier space without changing order
      fprintf(outfile, "           fftwnd_mpi(_%s_forward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_NORMAL_ORDER);\n",
              fieldName, fieldName, myName.c_str(), fieldName, myName.c_str(),
              fieldName, myName.c_str());
      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "           _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
      }
      fprintf(outfile, "      _%s_%s_space=%li;\n", fieldName, myName.c_str(), fullSpace);
      fprintf(outfile, "        } \n");

      fprintf(outfile, "      }  // end of unswapped -> unswapped case  \n\n");

      //  3. changing swappedness
      //    First we change all lower dimensions to be the same as the first, then we do a full transposed transform
      //      If the first transverse dimension needs changing, then we do a full in order transform
      //      Then we change the lower levels.

      fprintf(outfile, "   else { // changing swappedness  \n\n");
      fprintf(outfile, "        if ((_%s_%s_space&1)&&(_%s_%s_space&2)) { // initially swapped \n", fieldName, myName.c_str(), fieldName, myName.c_str());
      { // Put all lower dimensions to Fourier space and Fourier transform (transposed)
        unsigned long two2n=4;
        for (unsigned long i=2; i<myField->geometry()->nDims(); i++) {

          fprintf(outfile, "    _howmany = _%s_%s_ncomponents;\n", fieldName, myName.c_str());
          for (unsigned long j=myField->geometry()->nDims(); j>i+1; j--) {
            fprintf(outfile, "    _howmany *= _%s_lattice%li;\n", fieldName, j-1);
          }
          fprintf(outfile, "    _lattice = _%s_lattice%li;\n",    fieldName, i);
          fprintf(outfile, "\n");
          fprintf(outfile, "    if (!(_%s_%s_space&%li)) {\n", fieldName, myName.c_str(), two2n);
          fprintf(outfile, "       fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_FORWARD, FFTW_IN_PLACE);\n");

          fprintf(outfile, "       for (unsigned long _i0=0; _i0<local_ny_after_transpose*_%s_lattice0", fieldName);
          for (unsigned long j=2; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "            fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());

          fprintf(outfile, "      _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
          fprintf(outfile, "      fftwnd_destroy_plan(_plan);\n");
          fprintf(outfile, "     }\n");
          fprintf(outfile, "\n");

          two2n *= 2;
        }
      }
      fprintf(outfile, "            fftwnd_mpi(_%s_backward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_TRANSPOSED_ORDER);\n",
              fieldName, fieldName, myName.c_str(), fieldName, myName.c_str(),
              fieldName, myName.c_str());
      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "            _c *= _%s_dk%li/sqrt(2*M_PI);\n", fieldName, i);
      }
      fprintf(outfile, "       _%s_%s_space=0;\n", fieldName, myName.c_str());
      //  to get from 0 to unswapped, could do with a recursive call
      // but really it's a very small step to do it straight...
      fprintf(outfile, "        if (_newspace&1) { // first dimension needs switching to Fourier again \n");
      fprintf(outfile, "        fftwnd_mpi(_%s_forward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_NORMAL_ORDER);\n",
              fieldName, fieldName, myName.c_str(), fieldName, myName.c_str(),
              fieldName, myName.c_str());
      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "        _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
      }
      fprintf(outfile, "    _%s_%s_space=%li;\n", fieldName, myName.c_str(), fullSpace);
      fprintf(outfile, "                  } \n");
      fprintf(outfile, "           } // end initially swapped case  \n");
      fprintf(outfile, "        else { // initially unswapped \n");
      fprintf(outfile, "           if (_%s_%s_space&1) { // put all lower dimensions to Fourier space and Fourier transform (transposed) \n", fieldName, myName.c_str());
      { // change all the lower levels (brackets largely for cut and paste convenience)
        unsigned long two2n=2;
        for (unsigned long i=1; i<myField->geometry()->nDims(); i++) {

          fprintf(outfile, "    _howmany = _%s_%s_ncomponents;\n", fieldName, myName.c_str());
          for (unsigned long j=myField->geometry()->nDims(); j>i+1; j--) {
            fprintf(outfile, "    _howmany *= _%s_lattice%li;\n", fieldName, j-1);
          }
          fprintf(outfile, "    _lattice = _%s_lattice%li;\n",    fieldName, i);
          fprintf(outfile, "\n");
          fprintf(outfile, "    if (!(_%s_%s_space&%li)) {\n", fieldName, myName.c_str(), two2n);
          fprintf(outfile, "       fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_FORWARD, FFTW_IN_PLACE);\n");
          fprintf(outfile, "       for (unsigned long _i0=0; _i0<local_nx");
          for (unsigned long j=1; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "             fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());
          fprintf(outfile, "      _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
          fprintf(outfile, "      fftwnd_destroy_plan(_plan);\n");
          fprintf(outfile, "     }\n");
          fprintf(outfile, "\n");

          two2n *= 2;
        }
      }
      fprintf(outfile, "         int _fftw_latt[%li];\n", myField->geometry()->nDims());
      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "         _fftw_latt[%li] = _%s_lattice%li;\n", i, fieldName, i);
      }
      fprintf(outfile, "         fftwnd_mpi_plan _plan = fftwnd_mpi_create_plan(MPI_COMM_WORLD, %li, _fftw_latt, FFTW_BACKWARD, FFTW_IN_PLACE|FFTW_ESTIMATE|FFTW_USE_WISDOM);\n",
              myField->geometry()->nDims());

      fprintf(outfile, "         fftwnd_mpi(_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_NORMAL_ORDER);\n",
              fieldName, myName.c_str(), fieldName, myName.c_str(),
              fieldName, myName.c_str());
      fprintf(outfile, "         fftwnd_mpi_destroy_plan(_plan);\n");

      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "        _c *= _%s_dk%li/sqrt(2*M_PI);\n", fieldName, i);
      }
      fprintf(outfile, "        fftwnd_mpi(_%s_forward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_TRANSPOSED_ORDER);\n",
              fieldName, fieldName, myName.c_str(), fieldName, myName.c_str(),
              fieldName, myName.c_str());
      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "        _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
      }
      fprintf(outfile, "    _%s_%s_space=%li;\n", fieldName, myName.c_str(), fullSpace);
      fprintf(outfile, "          }\n");
      fprintf(outfile, "        else { // put all lower dimensions to normal space and Fourier transform (transposed)  \n");
      { // change all the lower levels (brackets largely for cut and paste convenience)
        unsigned long two2n=2;
        for (unsigned long i=1; i<myField->geometry()->nDims(); i++) {

          fprintf(outfile, "    _howmany = _%s_%s_ncomponents;\n", fieldName, myName.c_str());
          for (unsigned long j=myField->geometry()->nDims(); j>i+1; j--) {
            fprintf(outfile, "    _howmany *= _%s_lattice%li;\n", fieldName, j-1);
          }
          fprintf(outfile, "    _lattice = _%s_lattice%li;\n",    fieldName, i);
          fprintf(outfile, "\n");
          fprintf(outfile, "    if (_%s_%s_space&%li) {\n", fieldName, myName.c_str(), two2n);
          fprintf(outfile, "       fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_BACKWARD, FFTW_IN_PLACE);\n");
          fprintf(outfile, "       for (unsigned long _i0=0; _i0<local_nx");
          for (unsigned long j=1; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "             fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());
          fprintf(outfile, "      _c *= _%s_dk%li/sqrt(2*M_PI);\n", fieldName, i);
          fprintf(outfile, "      fftwnd_destroy_plan(_plan);\n");
          fprintf(outfile, "    }\n");
          fprintf(outfile, "\n");

          two2n *= 2;
        }
      }
      fprintf(outfile, "        fftwnd_mpi(_%s_forward_plan, _%s_%s_ncomponents, _active_%s_%s, _%s_%s_work, FFTW_TRANSPOSED_ORDER);\n",
              fieldName, fieldName, myName.c_str(), fieldName, myName.c_str(),
              fieldName, myName.c_str());
      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {
        fprintf(outfile, "        _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
      }
      fprintf(outfile, "        _%s_%s_space=%li;\n", fieldName, myName.c_str(), fullSpace);
      fprintf(outfile, "           }\n");
      fprintf(outfile, "           } // end initially unswapped case  \n");
      fprintf(outfile, "        }  // end of changing swappedness case  \n\n");

      //  By this point, the vector's space must be set to its actual space if it has changed
      fprintf(outfile, "   // Now the lower transverse dimensions need changing to the correct state  \n\n");

      { // change all the lower levels (brackets largely for cut and paste convenience)
        unsigned long two2n=2;

        for (unsigned long i=1; i<myField->geometry()->nDims(); i++) {

          fprintf(outfile, "    _howmany = _%s_%s_ncomponents;\n", fieldName, myName.c_str());
          for (unsigned long j=myField->geometry()->nDims(); j>i+1; j--) {
            fprintf(outfile, "    _howmany *= _%s_lattice%li;\n", fieldName, j-1);
          }
          if (i==1) {
            fprintf(outfile, "    if ((_newspace&1)&&(_newspace&2)) \n");
            fprintf(outfile, "       _lattice = _%s_lattice0;\n",    fieldName);
            fprintf(outfile, "    else\n");
            fprintf(outfile, "       _lattice = _%s_lattice1;\n",    fieldName);
          }
          else
            fprintf(outfile, "    _lattice = _%s_lattice%li;\n",    fieldName, i);
          fprintf(outfile, "\n");
          fprintf(outfile, "    if ((!(_%s_%s_space&%li))&&(_newspace&%li)) {\n", fieldName, myName.c_str(), two2n, two2n);
          fprintf(outfile, "       fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_FORWARD, FFTW_IN_PLACE);\n");

          fprintf(outfile, "       if ((_newspace&1)&&(_newspace&2)) {\n");
          fprintf(outfile, "          for (unsigned long _i0=0; _i0<local_ny_after_transpose");
          if (i>1)
            fprintf(outfile, "*_%s_lattice0", fieldName);
          for (unsigned long j=2; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "               fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());
          fprintf(outfile, "         }\n");
          fprintf(outfile, "       else {\n");
          fprintf(outfile, "          for (unsigned long _i0=0; _i0<local_nx");
          for (unsigned long j=1; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "               fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());
          fprintf(outfile, "         }\n");

          fprintf(outfile, "      _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
          fprintf(outfile, "      fftwnd_destroy_plan(_plan);\n");
          fprintf(outfile, "     }\n");
          fprintf(outfile, "    else if ((_%s_%s_space&%li)&&!(_newspace&%li)) {\n", fieldName, myName.c_str(), two2n, two2n);
          fprintf(outfile, "       fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_BACKWARD, FFTW_IN_PLACE);\n");
          fprintf(outfile, "       if ((_newspace&1)&&(_newspace&2)) {\n");
          fprintf(outfile, "          for (unsigned long _i0=0; _i0<local_ny_after_transpose");
          if (i>1)
            fprintf(outfile, "*_%s_lattice0", fieldName);
          for (unsigned long j=2; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "               fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());
          fprintf(outfile, "         }\n");
          fprintf(outfile, "       else {\n");
          fprintf(outfile, "          for (unsigned long _i0=0; _i0<local_nx");
          for (unsigned long j=1; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          fprintf(outfile, "               fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                  fieldName, myName.c_str());
          fprintf(outfile, "         }\n");
          fprintf(outfile, "      _c *= _%s_dk%li/sqrt(2*M_PI);\n", fieldName, i);
          fprintf(outfile, "      fftwnd_destroy_plan(_plan);\n");
          fprintf(outfile, "    }\n");
          fprintf(outfile, "\n");

          two2n *= 2;
        }
      } // lower levels now changed
      fprintf(outfile, "   _%s_%s_space=_newspace;\n", fieldName, myName.c_str());
      fprintf(outfile, "   }\n\n");  // end of the mixed space case
    }
    else {
      // We are not doing MPI fourier transforms.

      // mixed transforms stuff
      // note this does not need to be ultra efficient since it is probably only
      // ever used in filters, moment stuff, and after initialisation

      fprintf(outfile, "else {\n");
      fprintf(outfile, "   unsigned long _howmany;\n");
      fprintf(outfile, "   int _lattice;\n");
      fprintf(outfile, "\n");

      unsigned long two2n=1;

      for (unsigned long i=0; i<myField->geometry()->nDims(); i++) {

        fprintf(outfile, "   _howmany = _%s_%s_ncomponents;\n", fieldName, myName.c_str());
        for (unsigned long j=myField->geometry()->nDims(); j>i+1; j--) {
          fprintf(outfile, "   _howmany *= _%s_lattice%li;\n", fieldName, j-1);
        }
        fprintf(outfile, "\n");
        fprintf(outfile, "   _lattice = _%s_lattice%li;\n",    fieldName, i);
        fprintf(outfile, "\n");
        fprintf(outfile, "   if ((!(_%s_%s_space&%li))&&(_newspace&%li)) {\n", fieldName, myName.c_str(), two2n, two2n);
        fprintf(outfile, "\n");
        if (field()->simulation()->parameters()->fftwVersion == 2)
          fprintf(outfile, "     fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_FORWARD, FFTW_IN_PLACE);\n");
        else
          fprintf(outfile, "     fftw_plan _plan = fftw_plan_many_dft(1, &_lattice, _howmany, \n"
                  "                                          reinterpret_cast<fftw_complex*>(_active_%s_%s), NULL, _howmany, 1, \n"
                  "                                          reinterpret_cast<fftw_complex*>(_active_%s_%s), NULL, _howmany, 1, \n"
                  "                                          FFTW_FORWARD, FFTW_ESTIMATE);\n",
                  fieldName, myName.c_str(), fieldName, myName.c_str());
        // We need estimate planning as measure planning would overwrite the field, and that would be bad, right?
        fprintf(outfile, "\n");
        if (i>0) {
          fprintf(outfile, "           for (unsigned long _i0=0; _i0<_%s_lattice0", fieldName);
          for (unsigned long j=1; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          if (field()->simulation()->parameters()->nThreads>1 && field()->simulation()->parameters()->fftwVersion == 2) {
            fprintf(outfile, "         fftwnd_threads(_num_threads, _plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                    fieldName, myName.c_str());
          }
          else if (field()->simulation()->parameters()->fftwVersion == 2) {
            fprintf(outfile, "         fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                    fieldName, myName.c_str());
          }
          else if (field()->simulation()->parameters()->fftwVersion == 3) {
            fprintf(outfile, "         fftw_execute_dft(_plan, reinterpret_cast<fftw_complex*>(_active_%s_%s) + _i0*_lattice*_howmany, \n"
                    "                          reinterpret_cast<fftw_complex*>(_active_%s_%s) + _i0*_lattice*_howmany);\n",
                    fieldName, myName.c_str(), fieldName, myName.c_str());
          }
        }
        else {
          if (field()->simulation()->parameters()->nThreads>1 && field()->simulation()->parameters()->fftwVersion == 2) {
            fprintf(outfile, "    fftwnd_threads(_num_threads, _plan, _howmany, _active_%s_%s, _howmany, 1, 0, 0, 0);\n",
                    fieldName, myName.c_str());
          }
          else if (field()->simulation()->parameters()->fftwVersion == 2) {
            fprintf(outfile, "    fftwnd(_plan, _howmany, _active_%s_%s, _howmany, 1, 0, 0, 0);\n",
                    fieldName, myName.c_str());
          }
          else if (field()->simulation()->parameters()->fftwVersion == 3) {
            fprintf(outfile, "    fftw_execute(_plan);\n");
          }
        }
        fprintf(outfile, "\n");
        fprintf(outfile, "        _c *= _%s_dx%li/sqrt(2*M_PI);\n", fieldName, i);
        fprintf(outfile, "\n");
        if (field()->simulation()->parameters()->fftwVersion == 2) {
          fprintf(outfile, "         fftwnd_destroy_plan(_plan);\n");
        }
        else if (field()->simulation()->parameters()->fftwVersion == 3) {
          fprintf(outfile, "        fftw_destroy_plan(_plan);\n");
        }
        fprintf(outfile, "        }\n");
        fprintf(outfile, "   else if ((_%s_%s_space&%li)&&!(_newspace&%li)) {\n", fieldName, myName.c_str(), two2n, two2n);
        fprintf(outfile, "\n");
        if (field()->simulation()->parameters()->fftwVersion == 2)
          fprintf(outfile, "     fftwnd_plan _plan = fftwnd_create_plan(1, &_lattice, FFTW_BACKWARD, FFTW_IN_PLACE);\n");
        else
          // We need estimate planning as measure planning in fftw3 overwrites the field, and that would be bad, right?
          fprintf(outfile, "     fftw_plan _plan = fftw_plan_many_dft(1, &_lattice, _howmany, \n"
                  "                                          reinterpret_cast<fftw_complex*>(_active_%s_%s), NULL, _howmany, 1, \n"
                  "                                          reinterpret_cast<fftw_complex*>(_active_%s_%s), NULL, _howmany, 1, \n"
                  "                                          FFTW_BACKWARD, FFTW_ESTIMATE);\n",
                  fieldName, myName.c_str(), fieldName, myName.c_str());
        fprintf(outfile, "\n");
        if (i>0) {
          fprintf(outfile, "           for (unsigned long _i0=0; _i0<_%s_lattice0", fieldName);
          for (unsigned long j=1; j<i; j++) {
            fprintf(outfile, "*_%s_lattice%li", fieldName, j);
          }
          fprintf(outfile, "; _i0++)\n");
          if (field()->simulation()->parameters()->nThreads>1 && field()->simulation()->parameters()->fftwVersion == 2) {
            fprintf(outfile, "         fftwnd_threads(_num_threads, _plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                    fieldName, myName.c_str());
          }
          else if (field()->simulation()->parameters()->fftwVersion == 2) {
            fprintf(outfile, "         fftwnd(_plan, _howmany, _active_%s_%s + _i0*_lattice*_howmany, _howmany, 1, 0, 0, 0);\n",
                    fieldName, myName.c_str());
          }
          else if (field()->simulation()->parameters()->fftwVersion == 3) {
            fprintf(outfile, "         fftw_execute_dft(_plan, reinterpret_cast<fftw_complex*>(_active_%s_%s) + _i0*_lattice*_howmany, \n"
                    "                          reinterpret_cast<fftw_complex*>(_active_%s_%s) + _i0*_lattice*_howmany);\n",
                    fieldName, myName.c_str(), fieldName, myName.c_str());
          }

        }
        else {
          if (field()->simulation()->parameters()->nThreads>1 && field()->simulation()->parameters()->fftwVersion == 2) {
            fprintf(outfile, "    fftwnd_threads(_num_threads, _plan, _howmany, _active_%s_%s, _howmany, 1, 0, 0, 0);\n",
                    fieldName, myName.c_str());
          }
          else if (field()->simulation()->parameters()->fftwVersion == 2) {
            fprintf(outfile, "    fftwnd(_plan, _howmany, _active_%s_%s, _howmany, 1, 0, 0, 0);\n",
                    fieldName, myName.c_str());
          }
          else if (field()->simulation()->parameters()->fftwVersion == 3) {
            fprintf(outfile, "    fftw_execute(_plan);\n");
          }
        }
        fprintf(outfile, "\n");
        fprintf(outfile, "        _c *= _%s_dk%li/sqrt(2*M_PI);\n", fieldName, i);
        fprintf(outfile, "\n");
        if (field()->simulation()->parameters()->fftwVersion == 2) {
          fprintf(outfile, "         fftwnd_destroy_plan(_plan);\n");
        }
        else if (field()->simulation()->parameters()->fftwVersion == 3) {
          fprintf(outfile, "        fftw_destroy_plan(_plan);\n");
        }
        fprintf(outfile, "        }\n");
        fprintf(outfile, "\n");

        two2n *= 2;
      }

      fprintf(outfile, "   _%s_%s_space=_newspace;\n", fieldName, myName.c_str());
      fprintf(outfile, "   }\n\n");
    }
  }

  if ((field()->simulation()->parameters()->usempi) && !(field()->simulation()->parameters()->stochastic)) {
    fprintf(outfile, "  for (long _i0=0; _i0<total_local_size*_%s_%s_ncomponents; _i0++) {\n", fieldName, myName.c_str());
  }
  else {
    if (field()->simulation()->parameters()->useOpenMP) {
      // loop iterations are independent, so parallelise!
      fprintf(outfile, "#ifdef _OPENMP\n"
              "#pragma omp parallel for\n"
              "#endif\n");
    }
    fprintf(outfile, "  for (long _i0=0; _i0<_%s_size*_%s_%s_ncomponents; _i0++) {\n", fieldName, fieldName, myName.c_str());
  }
  fprintf(outfile, "     _active_%s_%s[_i0].re *= _c;\n", fieldName, myName.c_str());
  fprintf(outfile, "     _active_%s_%s[_i0].im *= _c;\n", fieldName, myName.c_str());
  fprintf(outfile, "     }\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsVector::writeInitialisationCall(
                                         FILE *const outfile,
                                         const char *const indent) const {
  if (debugFlag) {
    printf("xmdsVector::writeInitialisationCall\n");
  }

  fprintf(outfile, "%s_%s_%s_initialise();\n", indent, field()->name()->c_str(), name()->c_str());
}

// **************************************************************************
const XMLString* xmdsVector::name() const {
  if (debugFlag) {
    printf("xmdsVector::name\n");
  }

  return &myName;
}

// **************************************************************************
void xmdsVector::setName(
                         const XMLString& yourName) {
  if (debugFlag) {
    printf("xmdsVector::setName\n");
  }

  myName = yourName;
}

// **************************************************************************
xmdsVectorType xmdsVector::vectorType() const {
  if (debugFlag) {
    printf("xmdsVector::vectorType\n");
  }

  return myType;
}

// **************************************************************************
void xmdsVector::setVectorType(
                               const xmdsVectorType& yourType) {
  if (debugFlag) {
    printf("xmdsVector::setVectorType\n");
  }

  myType = yourType;
}

// **************************************************************************
unsigned long xmdsVector::nComponents() const {
  if (debugFlag) {
    printf("xmdsVector::nComponents\n");
  }

  return myComponentsNamesList.size();
}

// **************************************************************************
const XMLString* xmdsVector::componentName(
                                           const unsigned long& index) const {
  if (debugFlag) {
    printf("xmdsVector::componentName\n");
  }

  if (index>=myComponentsNamesList.size()) {
    return 0;
  }

  list<XMLString>::const_iterator pXMLString = myComponentsNamesList.begin();
  for (unsigned long i=0; i<index; i++) {
    pXMLString++;
  }

  return &*pXMLString;
}

// **************************************************************************
const long xmdsVector::componentLength(
                                           const unsigned long& index) const {
  if (debugFlag) {
    printf("xmdsVector::componentLength\n");
  }

  if (index>=myComponentsLengthsList.size()) {
    return 0;
  }

  list<long>::const_iterator pLong = myComponentsLengthsList.begin();
  for (unsigned long i=0; i<index; i++) {
    pLong++;
  }

  return *pLong;
}

// **************************************************************************
bool xmdsVector::getComponent(
                              const XMLString& ofName,
                              unsigned long& index) const {
  if (debugFlag) {
    printf("xmdsVector::getComponent\n");
  }

  index=0;
  for (list<XMLString>::const_iterator pXMLString = myComponentsNamesList.begin(); pXMLString != myComponentsNamesList.end(); pXMLString++) {
    if (*pXMLString==ofName) {
      return 1;
    }
    index++;
  }

  return 0;
}

// **************************************************************************
void xmdsVector::setComponents(
                               const list<XMLString>& yourComponentsNamesList) {
  if (debugFlag) {
    printf("xmdsVector::setComponents\n");
  }

  myComponentsNamesList = yourComponentsNamesList;
}

// **************************************************************************
void xmdsVector::setLengths(
                               const list<long>& yourComponentsLengthsList) {
  if (debugFlag) {
    printf("xmdsVector::setComponents\n");
  }

  myComponentsLengthsList = yourComponentsLengthsList;
}

// **************************************************************************
bool xmdsVector::needsFFTWRoutines() const {
  if (debugFlag) {
    printf("xmdsVector::needsFFTWRoutines\n");
  }

  return myNeedsFFTWRoutines;
}

// **************************************************************************
void xmdsVector::setNeedsFFTWRoutines() const {
  if (debugFlag) {
    printf("xmdsVector::setNeedsFFTWRoutines\n");
  }

  myNeedsFFTWRoutines=1;
}

// **************************************************************************
unsigned long xmdsVector::initialSpace() const {
  if (debugFlag) {
    printf("xmdsVector::initialSpace\n");
  }

  return myInitialSpace;
}

// **************************************************************************
void xmdsVector::setInitialSpace(
                                 const unsigned long& yourInitialSpace) {
  if (debugFlag) {
    printf("xmdsVector::setInitialSpace\n");
  }

  myInitialSpace=yourInitialSpace;
}

// **************************************************************************
// **************************************************************************
//                              xmdsVector protected
// **************************************************************************
// **************************************************************************

// **************************************************************************
void xmdsVector::writeInitialiseRoutine(FILE *const outfile) const
{
  if (debugFlag) {
    printf("xmdsVector::writeInitialise\n");
  }

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _%s_%s_initialise() {\n", field()->name()->c_str(), myName.c_str());
  fprintf(outfile, "\n");
  // This is so silly that we need to ask the field for the simulation object.
  if (field()->simulation()->parameters()->useOpenMP) {
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel for\n"
            "#endif\n");
  }

  if (field()->simulation()->parameters()->usempi && !field()->simulation()->parameters()->stochastic && *field()->name() == "main") {
    fprintf(outfile, "for (long _i0=0; _i0<total_local_size*_%s_%s_ncomponents; _i0++)\n",
            field()->name()->c_str(), myName.c_str());
  }
  else {
    fprintf(outfile, "for (long _i0=0; _i0<_%s_size*_%s_%s_ncomponents; _i0++)\n",
            field()->name()->c_str(), field()->name()->c_str(), myName.c_str());
  }
  fprintf(outfile, "     _%s_%s[_i0] = 0;\n", field()->name()->c_str(), myName.c_str());

  if (myNeedsFFTWRoutines) {
    fprintf(outfile, "\n");
    fprintf(outfile, "_%s_%s_space=%li;\n", field()->name()->c_str(), myName.c_str(), myInitialSpace);
  }

  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
const xmdsField* xmdsVector::field() const {
  if (debugFlag) {
    printf("xmdsVector::field\n");
  }

  return myField;
}

// **************************************************************************
bool xmdsVector::space(
                       const long unsigned int& index) const {
  if (debugFlag) {
    printf("xmdsVector::space\n");
  }

  if (index>=field()->geometry()->nDims()) {
    throw xmdsException("Internal range error in xmdsVector::space()");
  }

  return (myInitialSpace >> index)&1;
}

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
