/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmBitmaps.c

  (c) Simon Marlow 1990-92
  (c) Albert Graef 1994
  
  modified 7-1997 by strauman@sun6hft.ee.tu-berlin.de to add
  different enhancements (see README-1.4).

  modified 2004,2005,2006 by Bernhard R. Link (see Changelog)

  Functions & data for handling the bitmaps and cursors.
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/Drawing.h>

#include <X11/xpm.h>

/* Large cursors */
#include "../images/xfm_file.xbm"
#include "../images/xfm_filemsk.xbm"
#include "../images/xfm_files.xbm"
#include "../images/xfm_filesmsk.xbm"
#include "../images/xfm_noentry32.xbm"
#include "../images/xfm_noentrymsk32.xbm"
#include "../images/xfm_dir.xbm"
#include "../images/xfm_dirmsk.xbm"
#include "../images/xfm_exec.xbm"
#include "../images/xfm_execmsk.xbm"
#include "../images/xfm_watch.xbm"
#include "../images/xfm_watchmsk.xbm"
/* small cursors */
#include "../images/xfm_file_s.xbm"
#include "../images/xfm_filemsk_s.xbm"
#include "../images/xfm_files_s.xbm"
#include "../images/xfm_filesmsk_s.xbm"
#include "../images/xfm_noentry_s.xbm"
#include "../images/xfm_noentrymsk_s.xbm"
#include "../images/xfm_dir_s.xbm"
#include "../images/xfm_dirmsk_s.xbm"
#include "../images/xfm_exec_s.xbm"
#include "../images/xfm_execmsk_s.xbm"
/* different stuff */
#include "../images/xfm_lline.xbm"
#include "../images/xfm_tline.xbm"
#include "../images/xfm_fline.xbm"
#include "../images/xfm_cline.xbm"
#include "../images/xfm_larrow.xbm"
#include "../images/xfm_rarrow.xbm"
#include "../images/xfm_wavy_arrow.xbm"
#include "../images/xfm_tick.xbm"
#include "../images/xfm_notick.xbm"
#include "../images/xfm_excl.xbm"
#ifdef ENHANCE_PERMS
#include "../images/xfm_suid.xbm"
#include "../images/xfm_Suid.xbm"
#include "../images/xfm_sticky.xbm"
#include "../images/xfm_Sticky.xbm"
#endif

/* pixmaps for some popup windows */
#include "../images/xfm_file.xpm"
#include "../images/xfm_dir.xpm"
#include "../images/xfm_files.xpm"
#include "../images/xfm_symlnk.xpm"

/* default icons for built in file types */
#include "../icons/inode/directory.xpm"
#include "../icons/inode/x-parent-directory.xpm"
#include "../icons/default.xpm"
#include "../icons/inode/default.xpm"
#include "../icons/inode/x-bad-link.xpm"

/* The icons for the application and file window */
#include "../images/xfm_icon.xpm"
#include "../images/xfm_appmgr.xpm"

/* symbols to express special states of a file */
#include "../images/xfm_linksymbol.xpm"
#include "../images/xfm_gzipsymbol.xpm"
#include "../images/xfm_bzipsymbol.xpm"

#include "global.h"
#include "Am.h"
#include "Fm.h"
#include "mime.h"

/*-----------------------------------------------------------------------------
  STATIC DATA
-----------------------------------------------------------------------------*/

typedef struct {
  char *bits;
  int width, height;
} BitmapRec;


#ifdef __STDC__
#define ICON(x) { (char*)x##_bits, x##_width, x##_height }
#else
#define ICON(x) { x/**/_bits, x/**/_width, x/**/_height }
#endif

static BitmapRec bitmaps[] = {
  ICON(xfm_noentry),
  /* helper images: */
  ICON(xfm_lline), ICON(xfm_tline), ICON(xfm_fline), ICON(xfm_cline),
  ICON(xfm_larrow), ICON(xfm_rarrow), ICON(xfm_wavy_arrow), ICON(xfm_tick),
  ICON(xfm_notick), ICON(xfm_excl),
#ifdef ENHANCE_PERMS
  ICON(xfm_suid),ICON(xfm_Suid),ICON(xfm_sticky),ICON(xfm_Sticky),
#endif
};

static char **pixmaps[] = {
  xfm_files_xpm, xfm_dir_xpm, xfm_file_xpm, xfm_symlnk_xpm,
};

static char **icons[TYPE_COUNT] = {
  directory_xpm, x_parent_directory_xpm, default_xpm,
  inode_default_xpm, x_bad_link_xpm,
};

static char **modifier_pixmaps[FILE_MODIFIER_COUNT] = {
  xfm_linksymbol_xpm, xfm_gzipsymbol_xpm, xfm_bzipsymbol_xpm
};

typedef struct {
  int source, mask;
} CursorRec;

/* large cursor images */
static BitmapRec cursor_bitmaps[ 2*CURSOR_COUNT ] = {
  ICON(xfm_file), ICON(xfm_filemsk), ICON(xfm_files), ICON(xfm_filesmsk),
  ICON(xfm_noentry), ICON(xfm_noentrymsk), ICON(xfm_dir), ICON(xfm_dirmsk),
  ICON(xfm_exec), ICON(xfm_execmsk), 
  ICON(xfm_watch), ICON(xfm_watchmsk),
};
/* small fallback cursor images */
static BitmapRec safe_cursor_bitmaps[ 2*CURSOR_COUNT ] = {
  ICON(xfm_file_s), ICON(xfm_filemsk_s),
  ICON(xfm_files_s), ICON(xfm_filesmsk_s),
  ICON(xfm_noentry_s), ICON(xfm_noentrymsk_s),
  ICON(xfm_dir_s), ICON(xfm_dirmsk_s),
  ICON(xfm_exec_s), ICON(xfm_execmsk_s),
  ICON(xfm_watch), ICON(xfm_watchmsk),
};


/*-----------------------------------------------------------------------------
  PUBLIC DATA
-----------------------------------------------------------------------------*/

Pixmap bm[BM_COUNT];
struct dimensionedpixmap file_modifiers[FILE_MODIFIER_COUNT];
Pixmap bm_icon,bm_iconmsk,bm_appmgr,bm_appmgrmsk;
Cursor curs[CURSOR_COUNT];

static const char * const mimetypes[TYPE_COUNT]={
	"inode/directory",
	"inode/x-parent-directory",
	"*",
	"inode/*",
	"inode/x-bad-link.xpm",
};

/* PRIVATE FUNCTIONS */
static int readPixmapFromFile(Display *dpy, Window win, Widget top, String searchpath, Pixmap *pm, Pixmap *msk, XpmAttributes *atts);
static int readPixmapFromData(Display *dpy, Window win, Widget top, char **data, Pixmap *pm, Pixmap *msk, XpmAttributes *atts);
#ifdef ENHANCE_CMAP
static Boolean switchColormap(Widget wid);
#endif

/*-----------------------------------------------------------------------------
  PUBLIC FUNCTIONS
-----------------------------------------------------------------------------*/


void readBitmaps(void)
{
  Cardinal i;
  int j;
  Display *dpy;
  int scrn;
  Colormap cmp;
  Window win;
  XColor black, white;
  BitmapRec *cb;
  unsigned int w,h,rw,rh;
  XpmAttributes xpm_attr;
  static XpmColorSymbol none_color = { NULL, "None", (Pixel)0 };

  dpy = XtDisplay(aw.shell);
  win = DefaultRootWindow(dpy);
  scrn = DefaultScreen(dpy);
  cmp = DefaultColormap(dpy, scrn);

  black.pixel = BlackPixel(dpy, scrn);
  XQueryColor(dpy, cmp, &black);
  white.pixel = WhitePixel(dpy, scrn);
  XQueryColor(dpy, cmp, &white);

  /* create the hardcoded bitmaps */

  for (i=0; i<XtNumber(bitmaps); i++)
    bm[i] = XCreateBitmapFromData(dpy, win, bitmaps[i].bits,
				  bitmaps[i].width, bitmaps[i].height);
#ifdef ENHANCE_CMAP
  xpm_attr.valuemask = XpmColormap|XpmColorSymbols|XpmSize|XpmCloseness;
#else
  xpm_attr.valuemask = XpmReturnPixels|XpmColorSymbols|XpmSize;
#endif
  xpm_attr.colorsymbols = &none_color;
  xpm_attr.numsymbols = 1;
#ifdef ENHANCE_CMAP
  xpm_attr.closeness = (unsigned int) resources.color_closeness;
  XtVaGetValues(aw.shell, XtNbackground, &none_color.pixel,
			  XtNcolormap,&xpm_attr.colormap, NULL);
#else
  XtVaGetValues(aw.shell, XtNbackground, &none_color.pixel, NULL);
#endif
  for (i=XtNumber(bitmaps); i<BM_COUNT; i++)
    readPixmapFromData(dpy, win, aw.shell,pixmaps[i-XtNumber(bitmaps)],&bm[i],NULL,&xpm_attr);

  for (i=0; i <FILE_MODIFIER_COUNT; i++ ) {
	readPixmapFromData(dpy, win, aw.shell,
			modifier_pixmaps[i],&file_modifiers[i].pm,
			NULL,&xpm_attr);
	file_modifiers[i].width = xpm_attr.width;
	file_modifiers[i].height = xpm_attr.height;

  }

  /* create the cursors */
  /* Check if the server supports our maximal cursor
   * size; 
   * Note: we assume the bitmap size can be found in
   * 'bitmaps'.
   */
  w=h=0;
  for (i=0; i<CURSOR_COUNT; i++) {
    if ((rw=cursor_bitmaps[2*i].width)>w) w=rw;
    if ((rh=cursor_bitmaps[2*i].height)>h) h=rh;
  }
  XQueryBestCursor(dpy,win,w,h,&rw,&rh);
  if ( rw<w || rh<h )
   /* switch to 16 bit cursors which are safe */
	  cb = cursor_bitmaps;
  else
	  cb = safe_cursor_bitmaps;
  
  for (i=0; i<CURSOR_COUNT; i++) {
	Pixmap source,mask;

   	source = XCreateBitmapFromData(dpy, win, cb->bits,
		cb->width, cb->height);
	cb++;
   	mask = XCreateBitmapFromData(dpy, win, cb->bits,
		cb->width, cb->height);
	cb--;
	curs[i] = XCreatePixmapCursor(dpy, source,mask,
	        &black, &white, cb->width/2, cb->height/2);
	cb += 2;
  }

  /* create the application icons */

  readPixmapFromData(dpy,win,aw.shell,xfm_icon_xpm,&bm_icon,&bm_iconmsk,&xpm_attr);
  readPixmapFromData(dpy,win,aw.shell,xfm_appmgr_xpm,&bm_appmgr,&bm_appmgrmsk,&xpm_attr);

  for (j=0; j<TYPE_COUNT; j++) {
	  struct mime_filetype *type = mime_get_filetype(mimetypes[j]);
	  if( type == NULL )
		  abortXfm("Ouf of Memory");
	  type->data = (struct filetype_data*)XtCalloc(1,sizeof(struct filetype_data));
	  readPixmapFromData(dpy,win,aw.shell,icons[j],&type->data->icon_bm,NULL,&xpm_attr);
	  type->data->bm_width = xpm_attr.width;
	  type->data->bm_height = xpm_attr.height;
	  type->data->builtin = true;
  }
}

Pixmap loadFileIcon(const char *name, Cardinal *pm_width, Cardinal *pm_height) {
  Display *dpy = XtDisplay(aw.shell);
  Window win = DefaultRootWindow(dpy);
  Pixmap icon_bm;
  char fullname[MAXPATHLEN];

  XpmAttributes xpm_attr;
  static XpmColorSymbol none_color = { NULL, "None", (Pixel)0 };

  /* first search for xpm icons: */

#ifdef ENHANCE_CMAP
  xpm_attr.valuemask = XpmColormap|XpmColorSymbols|XpmSize|XpmCloseness;
#else
  xpm_attr.valuemask = XpmReturnPixels|XpmColorSymbols|XpmSize;
#endif
  xpm_attr.colorsymbols = &none_color;
  xpm_attr.numsymbols = 1;
#ifdef ENHANCE_CMAP
  xpm_attr.closeness = (unsigned int) resources.color_closeness;
  XtVaGetValues(aw.shell, XtNbackground, &none_color.pixel,
			  XtNcolormap, &xpm_attr.colormap, NULL);
#else
  XtVaGetValues(aw.shell, XtNbackground, &none_color.pixel, NULL);
#endif
  if (NULL==searchPathWithSuffix(fullname, resources.icon_path, name, ".xpm"))
	  return None;
  if (XpmSuccess==readPixmapFromFile(dpy,win,aw.shell,fullname,&icon_bm,NULL,&xpm_attr)) {
    if (pm_width) *pm_width=(unsigned int)xpm_attr.width;
    if (pm_height) *pm_height=(unsigned int)xpm_attr.height;
    return icon_bm;
  }
  return None;
}

static Pixmap actualReadIcon(const char *name, Cardinal *pm_width, Cardinal *pm_height)
{
  Display *dpy = XtDisplay(aw.shell);
  Window win = DefaultRootWindow(dpy);
  Screen *scrn = XtScreen(aw.shell);
  Pixmap icon_bm;
  char fullname[MAXPATHLEN];
  unsigned int w, h;
  int x, y;

  XpmAttributes xpm_attr;
  static XpmColorSymbol none_color = { NULL, "None", (Pixel)0 };

  /* first search for xpm icons: */

#ifdef ENHANCE_CMAP
  xpm_attr.valuemask = XpmColormap|XpmColorSymbols|XpmSize|XpmCloseness;
#else
  xpm_attr.valuemask = XpmReturnPixels|XpmColorSymbols|XpmSize;
#endif
  xpm_attr.colorsymbols = &none_color;
  xpm_attr.numsymbols = 1;
#ifdef ENHANCE_CMAP
  xpm_attr.closeness = (unsigned int) resources.color_closeness;
  XtVaGetValues(aw.shell, XtNbackground, &none_color.pixel,
			  XtNcolormap, &xpm_attr.colormap, NULL);
#else
  XtVaGetValues(aw.shell, XtNbackground, &none_color.pixel, NULL);
#endif
  (void) searchPath(fullname, resources.pixmap_path, name);
  if (XpmSuccess==readPixmapFromFile(dpy,win,aw.shell,fullname,&icon_bm,NULL,&xpm_attr)) {
    if (pm_width) *pm_width=(unsigned int)xpm_attr.width;
    if (pm_height) *pm_height=(unsigned int)xpm_attr.height;
    return icon_bm;
  }

  /* now search bitmap in standard locations (*bitmapFilePath): */

  icon_bm = XmuLocateBitmapFile(scrn, name, NULL, 0, (int *)&w, (int *)&h,
				 &x, &y);
  if (icon_bm != None) {
    if (pm_width) *pm_width=w;
    if (pm_height) *pm_height=h;
    return icon_bm;
  }

  /* finally search along *bitmapPath: */

  if (XReadBitmapFile(dpy, win,
		      searchPath(fullname, resources.bitmap_path, name),
		      &w, &h, &icon_bm, &x, &y) == BitmapSuccess) {
    if (pm_width) *pm_width=w;
    if (pm_height) *pm_height=h;
    return icon_bm;
   }
  if (pm_width) *pm_width=(unsigned int)0;
  if (pm_height) *pm_height=(unsigned int)0;
  return None;
}

/* This introduces some caching mechanism, as any icon needed will
 * most likely needed soon again, do not free them but keep them in
 * a list and look for each name if we have loaded it already before. - brl */

static struct iconMemory {
	struct {
		String name;
		Pixmap icon;
		unsigned int width,height;
	} icons[100];
	struct iconMemory *next;
} *iconRoot = NULL;

Pixmap loadIcon(const char *name, Cardinal *pm_width, Cardinal *pm_height) {
	struct iconMemory *m;
	int i;

	for( m = iconRoot; m != NULL ; m = m->next ) {
		for( i = 0 ; i < 100 ; i++ ) {
			if( m->icons[i].name == NULL )
				break;
			if( strcmp(m->icons[i].name,name) == 0 )
				break;
		}
		if( i < 100 )
			break;
	}
	if( m == NULL ) {
		m = (struct iconMemory*)XtCalloc(1,sizeof(struct iconMemory));
		m->next = iconRoot;
		iconRoot = m;
		i = 0;
	}
	if( m->icons[i].name == NULL ) {
		m->icons[i].icon = actualReadIcon(name,&m->icons[i].width,
						      &m->icons[i].height);
		if( m->icons[i].icon == None ) {
			if (pm_width) 
				*pm_width=(unsigned int)0;
			if (pm_height) 
				*pm_height=(unsigned int)0;
			return None;
		}
		m->icons[i].name = XtNewString(name);
	}
	if (pm_width) 
		*pm_width=m->icons[i].width;
	if (pm_height) 
		*pm_height=m->icons[i].height;
	return m->icons[i].icon;
}

#ifdef ENHANCE_CMAP

static Boolean switchColormap(Widget wid)
{
static Boolean has_private_cm=False;

XtAppContext app=XtWidgetToApplicationContext(wid);
Colormap     new,old;
Display	     *di=XtDisplay(wid);
Widget	     top,wtmp;

 if (has_private_cm) {
   XtAppWarning(app,"switchColormap failed: have private colormap already");
   return False;
 }
 top=wid;
 while((wtmp=XtParent(top))) top=wtmp;

 XtVaGetValues(top,XtNcolormap,&old,(char*)0);
 
 if (0==(new=XCopyColormapAndFree(di,old))) {
   XtAppWarning(app,"switchColormap failed: cannot allocate new colormap");
   return False;
 }

 has_private_cm=True;
 XtVaSetValues(top,XtNcolormap,new,(char*)0);
 XtSetWMColormapWindows(top,&top,1);

 XtAppWarning(app,"switching to private colormap...");
 return True;
}
#endif /*ENHANCE_CMAP*/

static int readPixmapFromFile(Display *dpy, Window win, Widget top, String searchpath, Pixmap *pm, Pixmap *msk, XpmAttributes *atts)
{
int      rval;

rval=XpmReadFileToPixmap(dpy,win,searchpath,pm,msk,atts);
#ifdef ENHANCE_CMAP
if (rval==XpmColorFailed && switchColormap(top)) {
    /* get the new colormap */
    XtVaGetValues(top,XtNcolormap,&atts->colormap,(char*)0);
    /* and repeat */
    return XpmReadFileToPixmap(dpy,win,searchpath,pm,msk,atts);
} else
#endif
    return rval;
}

static int readPixmapFromData(Display *dpy, Window win, Widget top, char **data, Pixmap *pm, Pixmap *msk, XpmAttributes *atts)
{
int      rval;

rval=XpmCreatePixmapFromData(dpy,win,data,pm,msk,atts);
#ifdef ENHANCE_CMAP
if (rval==XpmColorFailed && switchColormap(top)) {
    /* get the new colormap */
    XtVaGetValues(top,XtNcolormap,&atts->colormap,(char*)0);
    /* and repeat */
    return XpmCreatePixmapFromData(dpy,win,data,pm,msk,atts);
} else
#endif
    return rval;
}
