/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file RTDrivers.h
    \author Romain BONDUE
    \date 30/08/2005 */
#ifndef __RTDRIVERS_H__
#define __RTDRIVERS_H__

#include <ctime>

#include "WE17Driver.h"



namespace nsWireless
{
    typedef unsigned short u16;


    class CRTDriver : public CWE17Driver
    {
      public :
        CRTDriver (const std::string&) throw (nsErrors::CSystemExc);


      protected :
        void GetScanResult (int, u16, std::vector<CCell>&) const
                                throw (nsErrors::CException, std::bad_alloc);

    }; // CRTDriver


    class CRT2400Driver : public CRTDriver
    {
      public :
        CRT2400Driver (const std::string&) throw (nsErrors::CException);
        void Scan () throw (nsErrors::CSystemExc);
        void GetScanResult (std::vector<CCell>&) const
                                throw (nsErrors::CException, std::bad_alloc);


      private :
        const int m_PrivateIoctl;

    }; // CRT2400Driver


    enum WirelessMode_e {Mixed, B_Only};
    enum BGProtection_e {AutoProtection, On, Off};
    enum TxPreamble_e {Long, Short, AutoPreamble};


    class CRTAdvancedDriver : public CRTDriver
    {
      public :
        void GetScanResult (u16, std::vector<CCell>&) const
                                throw (nsErrors::CException, std::bad_alloc);
        int GetTxRate () const throw (nsErrors::CSystemExc);
        void SetTxRate (int) throw (nsErrors::CSystemExc);

            // TODO CountryRegion (has an influence on GetSupportedFreq()).

        virtual WirelessMode_e GetWirelessMode () const throw();
        virtual void SetWirelessMode (WirelessMode_e)
                                                throw (nsErrors::CSystemExc);

        virtual bool GetTxBurst () const throw();
        virtual void SetTxBurst (bool) throw (nsErrors::CSystemExc);

        virtual BGProtection_e GetBGProtection () const throw();
        virtual void SetBGProtection (BGProtection_e)
                                                throw (nsErrors::CSystemExc);

        virtual TxPreamble_e GetTxPreamble () const throw();
        virtual void SetTxPreamble (TxPreamble_e) throw (nsErrors::CSystemExc);

        virtual bool GetTurboRate () const throw();
        virtual void SetTurboRate (bool) throw (nsErrors::CSystemExc);

        virtual bool GetAdHocOFDM () const throw();
        virtual void SetAdHocOFDM (bool) throw (nsErrors::CSystemExc);

        virtual bool GetRfmontx () const throw();
        virtual void SetRfmontx (bool) throw (nsErrors::CSystemExc);


      protected :
        class IWPrivHandler
        {
          public :
            virtual void operator () (const ::iw_priv_args&)
                                      throw (nsErrors::CException);
            virtual ~IWPrivHandler () throw();

        }; // IWPrivHandler


        CRTAdvancedDriver (const std::string&, int, u16, u16, u16,
                   IWPrivHandler*) throw (nsErrors::CException, std::bad_alloc);
        CEncryptionD GetEncryption (u16, u16) throw (nsErrors::CSystemExc,
                                                     std::bad_alloc);
        void SetIoctl (const char*, const std::string&, const std::string&)
                                            const throw (nsErrors::CSystemExc);
        void SetIoctl (const char*, int, const std::string&) const
                                                throw (nsErrors::CSystemExc);
        enum {IsB_Only = 1, TxBurst = 2, TurboRate = 4, AdHocOFDM = 8,
              Rfmontx = 16};
        void SetFlag (int, bool) throw();
        bool GetFlag (int) const throw();
        BGProtection_e GetBGProtectionFlag () const throw();
        void SetBGProtectionFlag (BGProtection_e) throw();
        TxPreamble_e GetTxPreambleFlag () const throw();
        void SetTxPreambleFlag (TxPreamble_e) throw();
        int GetRfmontxIoctl () const throw();
        static const int InvalidIoctl = -1;


      private :
        const int m_PrivateIoctl;
        int m_SetIoctl;
        int m_RfmontxIoctl;

        char m_Flags;

        BGProtection_e m_BGProtection;
        TxPreamble_e m_TxPreamble;

    }; // CRTAdvancedDriver


    class CRT2500Driver : public CRTAdvancedDriver
    {
      public :
        CRT2500Driver (const std::string&) throw (nsErrors::CException,
                                                  std::bad_alloc);
        CEncryptionD GetEncryption () throw (nsErrors::CSystemExc,
                                             std::bad_alloc);
        void SetEncryption (const CEncryptionD&)
                                throw (nsErrors::CSystemExc, std::bad_alloc);
        void GetScanResult (std::vector<CCell>&) const
                                throw (nsErrors::CException, std::bad_alloc);

    }; // CRT2500Driver


    class CRT2570Driver : public CRTDriver
    {
      public :
        CRT2570Driver (const std::string&) throw (nsErrors::CException);
        void SetEncryption (const CEncryptionD&)
                                throw (nsErrors::CSystemExc, std::bad_alloc);
        void GetScanResult (std::vector<CCell>&) const
                                throw (nsErrors::CException, std::bad_alloc);
        void GetSupportedRates (std::vector<int>&) const throw();
        virtual bool GetRfmontx () const throw();
        virtual void SetRfmontx (bool) throw (nsErrors::CSystemExc);

        enum PrismMode_e {Default = 0, Force = 1, ForceNo = 2};
        virtual PrismMode_e GetPrismHeader () const throw();
        virtual void SetPrismHeader (PrismMode_e) throw (nsErrors::CSystemExc);

        enum AdHocMode_e {Std11 = 0, Auto = 1, Only54 = 2};
        virtual AdHocMode_e GetAdHocMode () const throw();
        virtual void SetAdHocMode (AdHocMode_e) throw (nsErrors::CSystemExc);


      protected :
        void PrivateIoctl (int, int, const std::string&)
                                                throw (nsErrors::CSystemExc);
        void PrivateIoctl (int, const std::string&, const std::string&)
                                                throw (nsErrors::CSystemExc);
        int GetIntIoctl (int, const std::string&)
                                                throw (nsErrors::CSystemExc);
        static const int InvalidIoctl = -1;


      private :
        int m_AuthIoctl;
        int m_EncIoctl;
        int m_KeyIoctl;
        int m_AdHocModeIoctl;
        int m_RfmontxIoctl;
        int m_PrismHeaderIoctl;
        int m_RTPrivIoctl;
        enum {RfMontx = 1};
        char m_Flags;
        PrismMode_e m_PrismMode;
        AdHocMode_e m_AdHocMode;

    }; // CRT2570Driver


    class CRT61Driver : public CRTAdvancedDriver
    {
      public :
        CRT61Driver (const std::string&) throw (nsErrors::CException,
                                                std::bad_alloc);
        void Scan () throw (nsErrors::CSystemExc);
        void GetScanResult (std::vector<CCell>&) const
                                throw (nsErrors::CException, std::bad_alloc);
        void SetEncryption (const CEncryptionD&)
                                throw (nsErrors::CSystemExc, std::bad_alloc);
        CEncryptionD GetEncryption () throw (nsErrors::CSystemExc,
                                             std::bad_alloc);


      protected :
        class IWPrivHandlerRT61 : public IWPrivHandler
        {
          public :
            IWPrivHandlerRT61 (int&) throw();
            void operator () (const ::iw_priv_args&)
                                                throw (nsErrors::CException);


          private :
            int& m_GetSiteSurveyIoctl;

        }; // IWPrivHandler

        void UpdateScanTime () throw();
        time_t GetLastScanTime () const throw();
        int GetSiteSurveyIoctl () const throw();


      private :
        int m_GetSiteSurveyIoctl;
        time_t m_LastScanTime;

    }; // CRT61Driver


        // Maximize shared code (could more elegant though).
    class CRT73Driver : public CRT61Driver
    {
      public :
        CRT73Driver (const std::string&) throw (nsErrors::CException,
                                                std::bad_alloc);
        void GetScanResult (std::vector<CCell>&) const
                                throw (nsErrors::CException, std::bad_alloc);
        void SetRfmontx (bool) throw (nsErrors::CSystemExc);

    }; // CRT73Driver

} // namespace nsWireless

#include "private/RTDrivers.hxx"

#endif // __RTDRIVERS_H__
