//                                               -*- C++ -*-
/**
 *  @file  MeixnerDistribution.hxx
 *  @brief The MeixnerDistribution distribution
 *
 *  Copyright 2005-2015 Airbus-EDF-IMACS-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  @author schueller
 *  @date   2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012)
 */
#ifndef OPENTURNS_MEIXNERDISTRIBUTION_HXX
#define OPENTURNS_MEIXNERDISTRIBUTION_HXX

#include "OTprivate.hxx"
#include "ContinuousDistribution.hxx"
#include "PiecewiseHermiteEvaluationImplementation.hxx"

BEGIN_NAMESPACE_OPENTURNS

/**
 * @class MeixnerDistribution
 *
 * The MeixnerDistribution distribution.
 */
class OT_API MeixnerDistribution
  : public ContinuousDistribution
{
  CLASSNAME;
public:

  enum ParameterSet { KLAMBDA, MUSIGMA };

  typedef Pointer<DistributionImplementation> Implementation;

  /** Default constructor */
  MeixnerDistribution();

  /** Parameters constructor */
  MeixnerDistribution(const NumericalScalar alpha,
                      const NumericalScalar beta,
                      const NumericalScalar delta,
                      const NumericalScalar mu);


  /** Comparison operator */
  Bool operator ==(const MeixnerDistribution & other) const;

  /** String converter */
  String __repr__() const;
  String __str__(const String & offset = "") const;

  /* Interface inherited from Distribution */

  /** Virtual constructor */
  virtual MeixnerDistribution * clone() const;

  /** Get one realization of the distribution */
  NumericalPoint getRealization() const;

  /** Get the PDF of the distribution */
  using ContinuousDistribution::computePDF;
  NumericalScalar computePDF(const NumericalPoint & point) const;
  using ContinuousDistribution::computeLogPDF;
  NumericalScalar computeLogPDF(const NumericalPoint & point) const;

  /** Get the CDF of the distribution */
  using ContinuousDistribution::computeCDF;
  NumericalScalar computeCDF(const NumericalPoint & point) const;
  using ContinuousDistribution::computeComplementaryCDF;
  NumericalScalar computeComplementaryCDF(const NumericalPoint & point) const;

  /** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
  NumericalComplex computeCharacteristicFunction(const NumericalScalar x) const;
  NumericalComplex computeLogCharacteristicFunction(const NumericalScalar x) const;

  /** Get the standard deviation of the distribution */
  NumericalPoint getStandardDeviation() const;

  /** Get the skewness of the distribution */
  NumericalPoint getSkewness() const;

  /** Get the kurtosis of the distribution */
  NumericalPoint getKurtosis() const;

  /** Get the standard representative in the parametric family, associated with the standard moments */
  Implementation getStandardRepresentative() const;

  /** Parameters value and description accessor */
  NumericalPointWithDescriptionCollection getParametersCollection() const;
  using ContinuousDistribution::setParametersCollection;
  void setParametersCollection(const NumericalPointCollection & parametersCollection);

  /* Interface specific to MeixnerDistribution */

  /** Alpha accessor */
  void setAlpha(const NumericalScalar alpha);
  NumericalScalar getAlpha() const;

  /** Beta accessor */
  void setBeta(const NumericalScalar beta);
  NumericalScalar getBeta() const;

  /** Delta accessor */
  void setDelta(const NumericalScalar delta);
  NumericalScalar getDelta() const;

  /** Mu accessor */
  void setMu(const NumericalScalar mu);
  NumericalScalar getMu() const;

  /** Method save() stores the object through the StorageManager */
  void save(Advocate & adv) const;

  /** Method load() reloads the object from the StorageManager */
  void load(Advocate & adv);


protected:


private:

  /** Get the quantile of the distribution, i.e the value Xp such that P(X <= Xp) = prob */
  NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                        const Bool tail = false) const;

  /** Compute the numerical range of the distribution given the parameters values */
  void computeRange();

  /** Set simultaneously the tree scale and shape parameters */
  void setAlphaBetaDelta(const NumericalScalar alpha,
                         const NumericalScalar beta,
                         const NumericalScalar delta);

  /** Compute the mean of the distribution */
  void computeMean() const;

  /** Compute the covariance of the distribution */
  void computeCovariance() const;

  /** Update the derivative attributes */
  void update();

  /** The main parameter set of the distribution */
  NumericalScalar alpha_;
  NumericalScalar beta_;
  NumericalScalar delta_;
  NumericalScalar mu_;

  /** The logarithm of the normalization factor */
  NumericalScalar logNormalizationFactor_;

  /** Bounds for the ratio of uniform sampling algorithm */
  NumericalScalar b_;
  NumericalScalar c_;
  NumericalScalar dc_;

  /** CDF approximation */
  PiecewiseHermiteEvaluationImplementation cdfApproximation_;

  /** Complementary CDF approximation */
  PiecewiseHermiteEvaluationImplementation ccdfApproximation_;

}; /* class MeixnerDistribution */


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_MEIXNERDISTRIBUTION_HXX */
