/* pdu-vtp.c
 
   Builder for VLAN Trunking Protocol (VTP) PDUs.

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*

* VTP References:

  - "Understanding VLAN Trunk Protocol (VTP)"
  (http://www.cisco.com/en/US/tech/tk389/tk689/technologies_tech_note09186a0080094c52.shtml)

  - Yersinia (www.yersinia.net): src/vtp.[ch]

  - Scapy VTP bug (http://trac.secdev.org/scapy/attachment/ticket/179/scapy_vtp.py)

  - http://www.cisco.com/univercd/cc/td/doc/product/lan/trsrb/frames.htm#22187

  - debug sw-vlan vtp {events, packets, pruning, xmit}

* MD5 hash calculation: ???

* NetExpect VTP PDU examples:

1. Summary Advertisement:

dot3(dst = 01:00:0c:cc:cc:cc)/llc(dsap=0xaa, ssap=0xaa, ctrl = 0x3)/snap(oui='\\x00\\x00\\x0c', code = 0x2003)/vtp-summary(domain-name = 'chapulandia', revision = 4, updater = 10.10.11.19, timestamp = '930301000818')

2. Subset Advertisement:

dot3(dst = 01:00:0c:cc:cc:cc)/llc(dsap=0xaa, ssap=0xaa, ctrl = 0x3)/snap(oui='\\x00\\x00\\x0c', code = 0x2003)/vtp-subset(domain-name = 'chapulandia', revision = 4)/vlan-info(status = 0x1, type = 'Ethernet', id = 123, mtu = 1500, name = 'voicenet', options(tlv(value='1324') ) )/vlan-info(status = 0x1, type = 'Ethernet', id = 124, mtu = 1500, name = 'printer-net', options(tlv(value='4567') ) )

3. Advertisement Request:

dot3(dst = 01:00:0c:cc:cc:cc)/llc(dsap=0xaa, ssap=0xaa, ctrl = 0x3)/snap(oui='\\x00\\x00\\x0c', code = 0x2003)/vtp-request(domain-name = 'chapulandia', start = 1)

*/

#include "pbuild-priv.h"
#include "pdu-vtp.h"

static const pdu_t pdu_vtp_summary = {
    .name = "vtp-summary",
    .description = "VTP Summary Advertisement",
    .documented_in = "www.cisco.com",
    .fields = (field_t []) {
	{
	    .name = "version",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_summary, version),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_NUM, ._number = VTP_DFL_VERSION}
	    }
	},
	{
	    .name = "code",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_summary, code),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_NUM, ._number = VTP_SUMM_ADVERT}
	    }
	},
	{
	    .name = "followers",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_summary, followers),
	},
	{
	    .name = "domain-length",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_summary, dom_len),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_LEN, ._fname = "domain-name"}
	    }
	},
	{
	    .name = "domain-name",
	    .type = PDU_FTYPE_DATA,
	    .offset = offsetof(struct vtp_summary, domain),
	    .size = VTP_DOMAIN_SIZE
	},
	{
	    .name = "revision",
	    .type = PDU_FTYPE_UINT32,
	    .offset = offsetof(struct vtp_summary, revision)
	},
	{
	    .name = "updater",
	    .type = PDU_FTYPE_IP,
	    .offset = offsetof(struct vtp_summary, updater)
	},
	{
	    .name = "timestamp",
	    .type = PDU_FTYPE_DATA,
	    .offset = offsetof(struct vtp_summary, timestamp),
	    .size = VTP_TIMESTAMP_SIZE
	},
	{
	    .name = "md5",
	    .type = PDU_FTYPE_DATA,
	    .offset = offsetof(struct vtp_summary, md5),
	    .size = 16
	},
	{
	    .name = NULL
	}
    },
    .len = sizeof(struct vtp_summary)
};

static const pdu_t pdu_vtp_subset = {
    .name = "vtp-subset",
    .description = "VTP Subset Advertisement",
    .documented_in = "www.cisco.com",
    .fields = (field_t []) {
	{
	    .name = "version",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_subset, version),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_NUM, ._number = VTP_DFL_VERSION}
	    }
	},
	{
	    .name = "code",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_subset, code),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_NUM, ._number = VTP_SUBSET_ADVERT}
	    }
	},
	{
	    .name = "seq",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_subset, seq)
	},
	{
	    .name = "domain-length",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_subset, dom_len),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_LEN, ._fname = "domain-name"}
	    }
	},
	{
	    .name = "domain-name",
	    .type = PDU_FTYPE_DATA,
	    .offset = offsetof(struct vtp_subset, domain),
	    .size = VTP_DOMAIN_SIZE
	},
	{
	    .name = "revision",
	    .type = PDU_FTYPE_UINT32,
	    .offset = offsetof(struct vtp_subset, revision)
	},
	{
	    .name = NULL
	}
    },
    .len = sizeof(struct vtp_subset)
};

static const pdu_t pdu_vtp_request = {
    .name = "vtp-request",
    .description = "VTP Advertisement Request",
    .documented_in = "www.cisco.com",
    .fields = (field_t []) {
	{
	    .name = "version",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_request, version),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_NUM, ._number = VTP_DFL_VERSION}
	    }
	},
	{
	    .name = "code",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_request, code),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_NUM, ._number = VTP_REQUEST}
	    }
	},
	{
	    .name = "domain-length",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vtp_request, dom_len),
	    .defval = (defval_t []) {
		{.type = PDU_DEF_LEN, ._fname = "domain-name"}
	    }
	},
	{
	    .name = "domain-name",
	    .type = PDU_FTYPE_DATA,
	    .offset = offsetof(struct vtp_request, domain),
	    .size = VTP_DOMAIN_SIZE
	},
	{
	    .name = "start",
	    .type = PDU_FTYPE_UINT16,
	    .offset = offsetof(struct vtp_request, start_val)
	},
	{
	    .name = NULL
	}
    },
    .len = sizeof(struct vtp_request)
};

/*
 * Returns the size of a VLAN information PDU, which depends on the size
 * of the VLAN name inside the VLAN information PDU and on the TLV
 * options that may follow the VLAN information header.
 */
static size_t
vlan_info_len(const GNode *pdu)
{
    size_t len;
    struct payload *vlan_name;

    len = sizeof(struct vlan_info);

    /*
     * In a VLAN information PDU the only field that has a variable
     * size is the VLAN name.
     */
    vlan_name = _pb_pdata(pdu, "name");
    if (vlan_name)
	len += vlan_name->len
	       + (3 - (vlan_name->len - 1) % 4); /* to pad w/ zeros to
						    multiple of 4 bytes */

    len += ( (struct node_data *) pdu->data)->_data_pdu.opts_len;

    return len;
}

static struct pdu_dict vlan_info_tlv_types[] = { /* Enumeration */
    {"Source-Routing Ring Number",    (uint32_t []) {0x01} },
    {"Source-Routing Bridge Number",  (uint32_t []) {0x02} },
    {"Spanning-Tree Protocol Type",   (uint32_t []) {0x03} },
    {"Parent VLAN",                   (uint32_t []) {0x04} },
    {"Translationally Bridged VLANs", (uint32_t []) {0x05} },
    {"Pruning",                       (uint32_t []) {0x06} },
    {"Bridge Type",                   (uint32_t []) {0x07} },
    {"Max ARE Hop Count",             (uint32_t []) {0x08} },
    {"Max STE Hop Count",             (uint32_t []) {0x09} },
    {"Backup CRF Mode",               (uint32_t []) {0x0a} },
    {NULL,                            NULL}
};

static const pdu_t pdu_vtp_vlaninfo = {
    .name = "vlan-info",
    .description = "VTP VLAN Information",
    .documented_in = "www.cisco.com",
    .fields = (field_t []) {
	{
	    .name = "length",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vlan_info, len),
	    .defval = (defval_t []) { {.type = PDU_DEF_LEN_HDR} }
	},
	{
	    .name = "status",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vlan_info, status)
	},
	{
	    .name = "type",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vlan_info, type),
	    .data = (struct pdu_dict []) { /* Enumeration */
		{"Ethernet", (uint32_t []) {0x01} },
		{"FDDI",     (uint32_t []) {0x02} },
		{"TrCRF",    (uint32_t []) {0x03} },
		{"FDDI-net", (uint32_t []) {0x04} },
		{"TrBRF",    (uint32_t []) {0x05} },
		{NULL,	     NULL}
	    },
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 1} }
	},
	{
	    .name = "name-len",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct vlan_info, name_len),
	    .defval = (defval_t []) { {.type = PDU_DEF_LEN, ._fname = "name"} }
	},
	{
	    .name = "id",
	    .type = PDU_FTYPE_UINT16,
	    .offset = offsetof(struct vlan_info, id),
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 1} }
	},
	{
	    .name = "mtu",
	    .type = PDU_FTYPE_UINT16,
	    .offset = offsetof(struct vlan_info, mtu),
	    .defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 1500} }
	},
	{
	    .name = "dot10index",
	    .type = PDU_FTYPE_UINT32,
	    .offset = offsetof(struct vlan_info, dot10)
	},
	{
	    .name = "name",
	    .type = PDU_FTYPE_DATA,
	    .offset = sizeof(struct vlan_info),
	    .size = VLAN_NAME_SIZE
	},
	{
	    .name = "options", /* VTP VLAN Info TLV */
	    .type = PDU_FTYPE_BRACED_ARGS,
	    .data = (field_t []) {
		{
		    .name = "tlv",
		    .type = PDU_FTYPE_BRACED_ARGS,
		    .data = (field_t []) {
			{
			    .name = "type",
			    .type = PDU_FTYPE_UINT8,
			    .offset = 0,
			    .data = vlan_info_tlv_types /* Enumeration */
			},
			{
			    .name = "length",
			    .type = PDU_FTYPE_UINT8,
			    .offset = 1,
			    .defval = (defval_t []) {
				{.type = PDU_DEF_LEN_TLV4}
			    }
			},
			{
			    .name = "value",
			    .type = PDU_FTYPE_DATA,
			    .offset = 2
			},
			{
			    .name = NULL
			}
		    }, .defval = NULL
		},
		{
		    .name = NULL
		}
	    }, .defval = NULL
	},
	{
	    .name = NULL
	}
    },
    .get_len = vlan_info_len,
    .options_class = "vlan-info"
};

static const pdu_option_t vlan_info_tlv = {
    .name = "tlv",
    .description = "VTP VLAN Info TLV",
    .get_len = &pdu_generic_tlv_len4
};

void
_pb_register_vtp(void)
{
    _pb_register_protocol(&pdu_vtp_summary);
    _pb_register_protocol(&pdu_vtp_subset);
    _pb_register_protocol(&pdu_vtp_request);
    _pb_register_protocol(&pdu_vtp_vlaninfo);

    _pb_register_protocol_option("vlan-info", &vlan_info_tlv);
}
