/* packets.c

   Initialization of the libpackets subsystem.

   Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012 Eloy Paris <peloy@netexpect.org>

   Some code here has been borrowed from the Wireshark source code, and
   is copyright Gerald Combs and others.

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <tcl.h>
#include <epan/timestamp.h>
#include <epan/packet.h>
#include <epan/column.h>
#include <epan/prefs.h>
#include <wsutil/privileges.h>
#include <epan/filesystem.h>

#include "util-tcl.h"
#include "packets-priv.h"

/* These are used by epan_init() */
static void read_failure_message(const char *, int);
static void write_failure_message(const char *, int);
static void failure_message(const char *, va_list);
static void open_failure_message(const char *, int, gboolean);

column_info cinfo;

/*
 * Initialized everything related to libwireshark use. Lots of this comes
 * from tshark.c in the wireshark distribution.
 */
static void
init_libwireshark(const char *ws_conf_profile)
{
    int i;
    e_prefs *prefs;
    char *gpf_path, *pf_path;
    int gpf_open_errno, gpf_read_errno;
    int pf_open_errno, pf_read_errno;

    /*
     * Get credential information for later use.
     */
    init_process_policies();

    /*
     * Handle the wireshark configuration profile (nexp's -C option). Set a
     * configuration profile only if the user specified a specific profile via
     * nexp's -C option, i.e.  ws_conf_profile != NULL. If no configuration
     * profile was specified then we don't set a specific profile and use
     * libwireshark defaults.
     */
    if (ws_conf_profile) {
	if (profile_exists(ws_conf_profile, FALSE) )
	    set_profile_name(ws_conf_profile);
	else {
	    fprintf(stderr, "libwireshark configuration profile \"%s\" "
		    "does not exist.\n", ws_conf_profile);
	    exit(EXIT_FAILURE);
	}
    }

    timestamp_set_type(TS_RELATIVE);
    timestamp_set_precision(TS_PREC_AUTO_MSEC);
    timestamp_set_seconds_type(TS_SECONDS_DEFAULT);

    epan_init(register_all_protocols, register_all_protocol_handoffs,
	      NULL, NULL, failure_message, open_failure_message,
	      read_failure_message, write_failure_message);

    /*
     * Now register the preferences for any non-dissector modules.
     * We must do that before we read the preferences.
     */
    prefs_register_modules();

    prefs = read_prefs(&gpf_open_errno, &gpf_read_errno, &gpf_path,
		       &pf_open_errno, &pf_read_errno, &pf_path);
    if (gpf_path != NULL) {
	if (gpf_open_errno != 0)
	    fprintf(stderr, "Can't open global preferences file \"%s\": %s.\n",
		    pf_path, strerror(gpf_open_errno) );
	if (gpf_read_errno != 0)
	    fprintf(stderr, "I/O error reading global preferences file "
		    "\"%s\": %s.\n", pf_path, strerror(gpf_read_errno) );
    }

    if (pf_path != NULL) {
	if (pf_open_errno != 0)
	    fprintf(stderr, "Can't open your preferences file \"%s\": %s.\n",
		    pf_path, strerror(pf_open_errno));
	if (pf_read_errno != 0)
	    fprintf(stderr, "I/O error reading your preferences file "
		    "\"%s\": %s.\n", pf_path, strerror(pf_read_errno));
	g_free(pf_path);
	pf_path = NULL;
    }

    /* Build the column format array */
    col_setup(&cinfo, prefs->num_cols);

    for (i = 0; i < cinfo.num_cols; i++) {
	cinfo.col_fmt[i] = get_column_format(i);
	cinfo.col_title[i] = g_strdup(get_column_title(i));
	if (cinfo.col_fmt[i] == COL_CUSTOM) {
	    cinfo.col_custom_field[i] = g_strdup(get_column_custom_field(i));
	} else {
	    cinfo.col_custom_field[i] = NULL;
	}
	cinfo.fmt_matx[i] = (gboolean *) g_malloc0(sizeof(gboolean) * NUM_COL_FMTS);
	get_column_format_matches(cinfo.fmt_matx[i], cinfo.col_fmt[i]);
	cinfo.col_data[i] = NULL;
	if (cinfo.col_fmt[i] == COL_INFO)
	    cinfo.col_buf[i] = (gchar *) g_malloc(sizeof(gchar) * COL_MAX_INFO_LEN);
	else
	    cinfo.col_buf[i] = (gchar *) g_malloc(sizeof(gchar) * COL_MAX_LEN);
	cinfo.col_fence[i] = 0;
	cinfo.col_expr.col_expr[i] = (gchar *) g_malloc(sizeof(gchar) * COL_MAX_LEN);
	cinfo.col_expr.col_expr_val[i] = (gchar *) g_malloc(sizeof(gchar) * COL_MAX_LEN);
    }

    for (i = 0; i < cinfo.num_cols; i++) {
	int j;

	for (j = 0; j < NUM_COL_FMTS; j++) {
	    if (!cinfo.fmt_matx[i][j])
		continue;

	    if (cinfo.col_first[j] == -1)
		cinfo.col_first[j] = i;

	    cinfo.col_last[j] = i;
	}
    }

    /* Initialize all data structures used for dissection. */
    init_dissection();
}

static void
read_failure_message(const char *filename, int err)
{
    fprintf(stderr, "An error occurred while reading from the file \"%s\": %s.",
	    filename, strerror(err) );
}

static void
write_failure_message(const char *filename, int err)
{
    fprintf(stderr, "An error occurred while writing to the file \"%s\": %s.",
	    filename, strerror(err) );
}

static void
failure_message(const char *msg_format, va_list ap)
{
    vfprintf(stderr, msg_format, ap);
    fprintf(stderr, "\n");
}

static void
open_failure_message(const char *filename, int err, gboolean for_writing)
{
    fprintf(stderr, "open error. filename = %s, err = %d, for_writing = %d\n",
	    filename, err, for_writing);
}

/*
 * Function registered with atexit() to clean up libwireshark at program
 * termination.
 */
static void
packets_cleanup(void)
{
    g_hash_table_destroy(dissection_vars);
    epan_cleanup();
}

void
pkt_init(const char *ws_conf_profile)
{
    init_libwireshark(ws_conf_profile);

    dissection_vars = g_hash_table_new_full(g_str_hash, g_str_equal,
					    g_free, NULL);

    if (atexit(packets_cleanup) != 0) {
	fprintf(stderr,
		"cannot set atexit() function packets_cleanup()\n");
	exit(EXIT_FAILURE);
    }
}

/********************************************************************
 *				clear                               *
 ********************************************************************/

#if !(GLIB_MAJOR_VERSION > 2 || (GLIB_MAJOR_VERSION == 2 && GLIB_MINOR_VERSION >= 12) )
static void
remove_all(gpointer key, gpointer value, gpointer user_data)
{
    return TRUE;
}
#endif

static void
unset_var(gpointer key, gpointer value _U_, gpointer user_data)
{
    Tcl_Interp *interp;

    interp = user_data;

    Tcl_UnsetVar(interp, key, 0);
}

int
pkt_clear_dissection(Tcl_Interp *interp, int argc, Tcl_Obj * const *objv)
{
    int index;
    static const char *subcmds[] = {
	"vars", NULL
    };
    enum subcmds {
	SUBCMD_VARS
    };

    if (argc == 1) {
	/* XXX - need to print out usage information */
	nexp_error(interp, "wrong # args");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    /*
     * Remove all Tcl variables created during dissection.
     */
    g_hash_table_foreach(dissection_vars, unset_var, interp);

    /*
     * Now remove all entries from the hash table.
     */
#if GLIB_MAJOR_VERSION > 2 || (GLIB_MAJOR_VERSION == 2 && GLIB_MINOR_VERSION >= 12)
	g_hash_table_remove_all(dissection_vars);
#else
	g_hash_table_foreach_remove(dissection_vars, remove_all, NULL);
#endif

    return TCL_OK;
}

/********************************************************************
 *				 show                               *
 ********************************************************************/

struct show_callback_data {
    Tcl_Interp *interp;
    Tcl_Obj *list;
};

static void
add_key_to_list(gpointer key, gpointer value _U_, gpointer user_data)
{
    struct show_callback_data *cb_data;
    Tcl_Obj *varname;

    cb_data = user_data;

    varname = Tcl_NewStringObj(key, strlen(key) );

    Tcl_ListObjAppendElement(cb_data->interp, cb_data->list, varname);
}

int
pkt_show_dissection(Tcl_Interp *interp)
{
    Tcl_Obj *list;
    struct show_callback_data cb_data;

    list = Tcl_NewListObj(0, NULL);

    cb_data.interp = interp;
    cb_data.list = list;

    g_hash_table_foreach(dissection_vars, add_key_to_list, &cb_data);

    Tcl_SetObjResult(interp, list);

    return TCL_OK;
}
