/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



/////////////////////// Qt includes
#include <QObject>
#include <QDebug>
#include <QDialog>
#include <QWidget>
#include <QDir>
#include <QFileDialog>
#include <QMessageBox>
#include <QDate>
#include <QSettings>


/////////////////////// Local includes
#include <minexpert/gui/SqlMassDataSlicerDlg.hpp>
#include <minexpert/nongui/SqlMassDataSlicer.hpp>


namespace msXpSmineXpert
{


	//! Construct an initialized SqlMassDataSlicerDlg instance.
	/*!

		\param parent parent widget.

		\param slicer pointer to the SqlMassDataSlicer instance to be used for the
		slicing operation.

		\param applicationName name of the application to be displayed in the window
		title.

*/
	SqlMassDataSlicerDlg::SqlMassDataSlicerDlg(QWidget *parent,
			SqlMassDataSlicer *slicer,
			const QString & applicationName)
		: QDialog{parent}, m_applicationName{applicationName}, mp_slicer{slicer}
	{
		if(mp_slicer == nullptr)
			qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

		m_ui.setupUi(this);

		initialize();
	}


	//! Destructi \c this SqlMassDataSlicerDlg instance.
	SqlMassDataSlicerDlg::~SqlMassDataSlicerDlg()
	{
	}


	//! Read the settings to restore the window geometry.
	void
		SqlMassDataSlicerDlg::readSettings()
		{
			QSettings settings;
			restoreGeometry(
					settings.value("SqlMassDataSlicerDlg_geometry").toByteArray());
		}


	//! Write the settings to restore the window geometry later.
	void
		SqlMassDataSlicerDlg::writeSettings()
		{
			QSettings settings;
			settings.setValue("SqlMassDataSlicerDlg_geometry", saveGeometry());
		}


	//! Initialize the dialog window.
	void
		SqlMassDataSlicerDlg::initialize()
		{
			// This window might be used by more than only one application, set its
			// name properly.
			setWindowTitle(QString("%1 - Data slicer options").arg(m_applicationName));

			// Restore the geometry of the window and fill in the history combo box.
			readSettings();

			// The slicer passed as parameter should have the range start and end
			// values set. We will put these in their corresponding line edit widget.
			setRangeStart(mp_slicer->m_rangeStart);
			setRangeEnd(mp_slicer->m_rangeEnd);

			// The review groupbox is hidden when the dialog window is created, it is
			// only shown if necessary.

			m_ui.reviewGroupBox->setVisible(false);

			// The file names text edit is infact read only, it is only used for
			// checking

			m_ui.fileNamesPlainTextEdit->setReadOnly(true);

			// Make the connections

			// The slicer emits a message telling what file it is exporting dat to.
			connect(mp_slicer, &SqlMassDataSlicer::updateSliceFileName, this,
					&SqlMassDataSlicerDlg::updateSliceFileName);

			connect(m_ui.destDirectoryPushButton, &QPushButton::clicked, this,
					&SqlMassDataSlicerDlg::destDirectoryPushButton);

			connect(m_ui.sliceSizeRadioButton, &QRadioButton::toggled, this,
					&SqlMassDataSlicerDlg::sliceRadioButtonToggled);

			connect(m_ui.sliceCountRadioButton, &QRadioButton::toggled, this,
					&SqlMassDataSlicerDlg::sliceRadioButtonToggled);

			connect(m_ui.validatePushButton, &QPushButton::clicked, this,
					&SqlMassDataSlicerDlg::validatePushButton);

			connect(m_ui.closePushButton, &QPushButton::clicked, this,
					&SqlMassDataSlicerDlg::closePushButton);

			connect(m_ui.displayedRegionCheckBox, &QCheckBox::stateChanged, this,
					&SqlMassDataSlicerDlg::displayedRegionCheckBoxStateChanged);

			connect(m_ui.confirmPushButton, &QPushButton::clicked, this,
					&SqlMassDataSlicerDlg::confirmPushButton);
		}


	//! Set the name of the source file.
	void
		SqlMassDataSlicerDlg::setSrcFileName(const QString &fileName)
		{
			// If the source filename is absolute, then put the directory in the
			// member datum.
			QFileInfo fileInfo(fileName);

			if(!fileInfo.exists())
			{
				QString msg("The data source file name is inexistent. Error.");
				qDebug() << __FILE__ << __LINE__ << msg;
				QMessageBox msgBox;
				msgBox.setText(msg);
				msgBox.setStandardButtons(QMessageBox::Ok);
				msgBox.exec();

				return;
			}

			if(fileInfo.isAbsolute())
			{
				// Get the directory component.
				mp_slicer->m_destDirectory = fileInfo.absolutePath();

				// Then, we can just display the file name.
				mp_slicer->m_srcFileName = fileInfo.baseName();

				// But use the format string to illustrate and as a convenient default
				// in the line edit widget.
				m_ui.formatStringLineEdit->setText(
						"%f-export-slice-%n-of-%c-range_%s-%e.db");
			}
			else
			{
				m_ui.formatStringLineEdit->setText(fileName);
				mp_slicer->m_srcFileName = fileName;
			}
		}


	//! Set the range start value in the corresponding line edit widget.
	void
		SqlMassDataSlicerDlg::setRangeStart(double start)
		{
			m_ui.rangeStartLineEdit->setText(QString("%1").arg(start, 0, 'f', 12));
		}


	//! Set the range end value in the corresponding line edit widget.
	void
		SqlMassDataSlicerDlg::setRangeEnd(double end)
		{
			m_ui.rangeEndLineEdit->setText(QString("%1").arg(end, 0, 'f', 12));
		}


	//! Set the range start and end values in their corresponding line edit widgets.
	void
		SqlMassDataSlicerDlg::setRangeLimits(double start, double end)
		{
			setRangeStart(start);
			setRangeEnd(end);
		}


	//! Ensure that there are names of the destination files.
	/*!

		The names of the destination files have been crafted on the basis of the
		data set by the user in this dialog window. To allow the user to review
		these file names, they are printed in a text edit widget that is make
		visible at that precise time (it is invisible before this call).

		*/
	void
		SqlMassDataSlicerDlg::reviewFileNames()
		{
			QString fileNames = mp_slicer->allFileNamesAsString();

			if(fileNames.isEmpty())
			{
				m_ui.reviewGroupBox->setVisible(true);
				m_ui.fileNamesPlainTextEdit->setPlainText(
						"Major error, there are no file names in the list.");

				return;
			}

			m_ui.reviewGroupBox->setVisible(true);
			m_ui.fileNamesPlainTextEdit->setPlainText(fileNames);
		}


	//! Trigger the verification of all the data set by the user.
	/*!

		The verification of the data and the crafting of the destination file names
		is followed by a call to reviewFileNames().

		*/
	void
		SqlMassDataSlicerDlg::validatePushButton()
		{
			// We'll have to perform a thorough checking of all the parameters in the
			// dialog window. Then we'll have to craft all the required stuff to
			// actually perform the requested data export.

			// First, reset the slicer to all the value that have been modified by the
			// user after this dialog window was initialized.
			mp_slicer->reset();

			// First verify that the file format string is not empty.
			if(m_ui.formatStringLineEdit->text().isEmpty())
			{
				QString msg("Please set the file name format string.");
				qDebug() << __FILE__ << __LINE__ << msg;
				QMessageBox msgBox;
				msgBox.setText(msg);
				msgBox.setStandardButtons(QMessageBox::Ok);
				msgBox.exec();

				return;
			}

			mp_slicer->m_formatString = m_ui.formatStringLineEdit->text();

			// The check that all the numerical values are sane.
			bool ok = false;
			double start = 0;
			double end = 0;
			QString text;

			// When the dialog was setup, the slicer had the proper range values set
			// from the plot widget from which the export of the data was triggered.
			// Let's get these values and set them to handy local variables.

			text = m_ui.rangeStartLineEdit->text();
			if(!text.isEmpty())
			{
				start = text.toDouble(&ok);
				if(!ok)
				{
					QString msg("Failed to convert text to double: ");
					msg += text;

					qDebug() << __FILE__ << __LINE__ << msg;

					QMessageBox msgBox;
					msgBox.setText(msg);
					msgBox.setStandardButtons(QMessageBox::Ok);
					msgBox.exec();

					return;
				}
			}

			text = m_ui.rangeEndLineEdit->text();
			if(!text.isEmpty())
			{
				end = text.toDouble(&ok);
				if(!ok)
				{
					QString msg("Failed to convert text to double: ");
					msg += text;

					qDebug() << __FILE__ << __LINE__ << msg;

					QMessageBox msgBox;
					msgBox.setText(msg);
					msgBox.setStandardButtons(QMessageBox::Ok);
					msgBox.exec();

					return;
				}
			}

			// Make sure end is greater than start...
			if(start < end)
			{
				mp_slicer->m_rangeStart = start;
				mp_slicer->m_rangeEnd = end;
			}
			else
			{
				mp_slicer->m_rangeStart = end;
				mp_slicer->m_rangeEnd = start;
			}

			if(m_ui.displayedRegionCheckBox->isChecked())
			{

				// Export the data as displayed in the corresponding widget to a single
				// file. Note that the slicer was constructed with the range values, so
				// we have all the necessary to perform the export.

				mp_slicer->m_sliceType = SliceType::SLICE_ZOOMED_REGION;
			}
			else
			{

				// Perform some kind of slicing of the data to various files.

				if(m_ui.sliceSizeRadioButton->isChecked())
				{
					mp_slicer->m_sliceType = SliceType::SLICE_BY_SIZE;

					text = m_ui.sliceSizeLineEdit->text();
					double sliceSize = text.toDouble(&ok);
					if(!ok)
					{
						QString msg("Failed to convert text to double: ");
						msg += text;

						qDebug() << __FILE__ << __LINE__ << msg;

						QMessageBox msgBox;
						msgBox.setText(msg);
						msgBox.setStandardButtons(QMessageBox::Ok);
						msgBox.exec();

						return;
					}
					mp_slicer->m_sliceSize = sliceSize;
				}
				else
				{
					// Sanity check
					if(!m_ui.sliceCountRadioButton->isChecked())
						qFatal("Fatal error at %s@%d. Program aborted.", __FILE__,
								__LINE__);

					mp_slicer->m_sliceType = SliceType::SLICE_BY_COUNT;

					text = m_ui.sliceCountLineEdit->text();
					int sliceCount = text.toInt(&ok);
					if(!ok)
					{
						QString msg("Failed to convert text to int:");
						msg += text;

						qDebug() << __FILE__ << __LINE__ << msg;

						QMessageBox msgBox;
						msgBox.setText(msg);
						msgBox.setStandardButtons(QMessageBox::Ok);
						msgBox.exec();

						return;
					}
					mp_slicer->m_sliceCount = sliceCount;
				}
			}

			// At this point, we have all the required data to validate the
			// destFileName string and thus potentially construct the various
			// fileNames.

			QString errors;
			if(!mp_slicer->configureSpecifs(&errors))
			{
				QString msg =
					"Failed to validate the input data, please check the input\n";
				msg += errors;
				QMessageBox msgBox;
				msgBox.setText(msg);
				msgBox.setStandardButtons(QMessageBox::Ok);
				msgBox.exec();

				return;
			}

			// Show all the file names to the user so she can verify, fix and confirm.

			reviewFileNames();
		}


	//! Close this dialog window.
	void
		SqlMassDataSlicerDlg::closePushButton()
		{
			qDebug() << __FILE__ << __LINE__ << "Cancel";
			setResult(QDialog::Rejected);
			done(QDialog::Rejected);
		}


	//! Confirm that the destination file names are valid.
	void
		SqlMassDataSlicerDlg::confirmPushButton()
		{
			qDebug() << __FILE__ << __LINE__
				<< "Destination file names have been confirmed.";

			if(!mp_slicer->slice())
			{
				QString msg = "An error occurred while slicing the data. Please check "
					"your configuration.";
				QMessageBox msgBox;
				msgBox.setText(msg);
				msgBox.setStandardButtons(QMessageBox::Ok);
				msgBox.exec();
			}
			else
			{
				QString msg = "The data slicing is finished. This window will close.";
				QMessageBox msgBox;
				msgBox.setText(msg);
				msgBox.setStandardButtons(QMessageBox::Ok);
				msgBox.exec();
			}

			writeSettings();
			setResult(QDialog::Accepted);
			done(QDialog::Accepted);
		}


	//! React to the clicking of UI elements.
	void
		SqlMassDataSlicerDlg::displayedRegionCheckBoxStateChanged(int checkState)
		{
			if(checkState == Qt::Checked)
			{
				m_ui.autoSliceGroupBox->setEnabled(false);
			}
			else
			{
				m_ui.autoSliceGroupBox->setEnabled(true);
			}
		}


	//! React to the clicking of UI elements.
	void
		SqlMassDataSlicerDlg::sliceRadioButtonToggled(bool checked)
		{
			QRadioButton *sender = static_cast<QRadioButton *>(QObject::sender());
			if(sender == m_ui.sliceCountRadioButton)
			{
				m_ui.sliceCountLineEdit->setEnabled(checked);
				m_ui.sliceSizeLineEdit->setEnabled(!checked);
			}
			else if(sender == m_ui.sliceSizeRadioButton)
			{
				m_ui.sliceSizeLineEdit->setEnabled(checked);
				m_ui.sliceCountLineEdit->setEnabled(!checked);
			}
		}


	//! Setup the feedback system to inform the user of the progress of the operation.
	void
		SqlMassDataSlicerDlg::setupFeedback(QString msg, int startValue,
				int endValue, int currentValue)
		{
			m_ui.feedbackProgressBar->setMinimum(startValue);
			m_ui.feedbackProgressBar->setMaximum(endValue);

			if(currentValue != -1)
				m_ui.feedbackProgressBar->setValue(currentValue);

			if(!msg.isEmpty())
				m_ui.feedbackLineEdit->setText(msg);

			QApplication::processEvents();
		}


	//! Update the feedback to the user.
	void
		SqlMassDataSlicerDlg::updateFeedback(QString msg, int currentValue)
		{
			m_ui.feedbackLineEdit->setText(msg);
			m_ui.feedbackProgressBar->setValue(currentValue);

			// Process the app events, to refresh the GUI.
			QApplication::processEvents();
		}


	//! Change the name of the destination file currently being dealt with.
	void
		SqlMassDataSlicerDlg::updateSliceFileName(QString fileName)
		{
			m_ui.feedbackFileLineEdit->setText(fileName);
		}


	//! Allow the user to define the directory where the destination files should be written.
	void
		SqlMassDataSlicerDlg::destDirectoryPushButton()
		{
			// Let the user define the destination directory for all the files
			// produced by the data slicer. However, try to open a meaningful
			// directory for choosing another.

			QString openDir;

			if(mp_slicer->m_destDirectory.isEmpty())
				openDir = QDir::homePath();
			else
				openDir = mp_slicer->m_destDirectory;

			QString newDirectory = QFileDialog::getExistingDirectory(
					this, tr("Open Directory"), openDir, QFileDialog::ShowDirsOnly);

			// If the user escapes the dialog box, then we do not change anything.
			if(!newDirectory.isEmpty())
				mp_slicer->m_destDirectory = newDirectory;
		}


} // namespace msXpSmineXpert

