/*
 *  @(#)ThrowableParser.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.util.throwable.v1;


import java.io.StringWriter;
import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.StringReader;

import java.util.Vector;

import java.lang.reflect.Method;

//import org.apache.log4j.Logger;


/**
 * Parses a Throwable's stack trace into its components.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since      March 17, 2002
 * @version    $Date: 2003/09/23 19:54:07 $
 */
public class ThrowableParser
{
    //private static final Logger LOG = Logger.getLogger(
    //    ThrowableParser.class.getName() );
    
    // JDK 1.1+
    private BufferedReader traceReader;
    
    // JDK 1.4+
    private Object[] traceList;
    private int traceIndex;
    
    private static final String JDK14_GST_METHOD = "getStackTrace";
    private static final Class[] JDK14_GST_METHOD_ARGS = new Class[0];

    private static final String JDK14_GC_METHOD = "getCause";
    private static final Class[] JDK14_GC_METHOD_ARGS = new Class[0];
    
    
    
    /**
     * Only piecemeal iterates through the stack-trace.  Supports the JDK 1.4
     * StackTraceElement class.
     */
    public ThrowableParser( Throwable t )
    {
        if (t == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        parseThrowable( t );
    }
    
    
    /**
     * @return <tt>null</tt> if there are no more lines, or the next line
     *      in the list.
     */
    public synchronized StackTraceLineParser next()
    {
        if (this.traceList != null)
        {
            return next14();
        }
        
        // else
        return next11();
    }
    
    
    //-------------------------------------------------------------------------
    
    
    /**
     * 
     */
    protected StackTraceLineParser next11()
    {
        //LOG.debug( "finding next stack trace line for JDK 1.1+" );
        while (true)
        {
            String line;
            try
            {
                line = this.traceReader.readLine();
                //LOG.debug("trace line = '"+line+"'");
            }
            catch (java.io.IOException ioe)
            {
                throw new IllegalStateException("Never supposed to happen: "+
                    ioe );
            }
            if (line == null)
            {
                //LOG.debug("end-of-stack trace.");
                return null;
            }
            try
            {
                StackTraceLineParser stlp = new StackTraceLineParser( line );
                
                // found a valid line
                return stlp;
            }
            catch (Exception e)
            {
                // not a valid line.  Continue.
                //LOG.debug(" - not a valid line",e);
            }
        }
    }
    
    
    /**
     * 
     */
    protected StackTraceLineParser next14()
    {
        //LOG.debug( "finding next stack trace line for JDK 1.4+" );
        if (this.traceIndex >= this.traceList.length)
        {
            return null;
        }
        
        // this will throw an exception if the parser does not know this
        // type.  That's fine.  That will show a bug in this implementation.
        StackTraceLineParser stlp = new StackTraceLineParser(
            this.traceList[ this.traceIndex ] );
        ++this.traceIndex;
        
        return stlp;
    }
    
    
    
    protected void parseThrowable( Throwable t )
    {
        try
        {
            // Construct the trace from the complete trace chain.
            Vector trace = new Vector();
            while (t != null)
            {
                Class c = t.getClass();
                Method m = c.getDeclaredMethod( JDK14_GST_METHOD,
                    JDK14_GST_METHOD_ARGS );
                Object[] o = (Object[])m.invoke( t, null );
                if (o != null)
                {
                    for (int i = 0; i < o.length; ++i)
                    {
                        trace.addElement( o[i] );
                    }
                }
                
                // find cause of t
                m = c.getDeclaredMethod( JDK14_GC_METHOD,
                    JDK14_GC_METHOD_ARGS );
                t = (Throwable)m.invoke( t, null );
            }
            
            Object[] o = new Object[ trace.size() ];
            trace.copyInto( o );
            this.traceList = o;
            this.traceIndex = 0;
            
            //LOG.info("Parsing: using JDK 1.4 method.");
        }
        catch (ThreadDeath td)
        {
            // don't ever gulp these
            throw td;
        }
        catch (Throwable th)
        {
            // use JDK 1.1+ compatible method
            parseThrowableBuffer( t );
        }
    }
    
    
    protected void parseThrowableBuffer( Throwable t )
    {
        //LOG.info("Parsing: reverted to JDK 1.1 compatible method.");
        
        StringWriter sw = new StringWriter();
        t.printStackTrace( new PrintWriter( sw ) );
        StringReader sr = new StringReader( sw.toString() );
        this.traceReader = new BufferedReader( sr );
    }
}

