/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     numint.c                                                       */
/*                                                                          */
/* description:  quadrature formulas and routines for numerical quadrature  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

#define  zero  0.0
#define  one   1.0
#define  half  0.5
#define  third 1.0/3.0
#define  quart 1.0/4.0

/*--------------------------------------------------------------------------*/
/*  0d quadrature formulas using 1 barycentric coordinates                  */
/*--------------------------------------------------------------------------*/

#define MAX_QUAD_DEG_0d   19

static const double  x_0d[N_LAMBDA] = {1.0, 0.0, 0.0, 0.0};
static const double  *gx_0d[1] = {x_0d};
static const double  w_0d[1] = {1.0};

static QUAD quad_0d[MAX_QUAD_DEG_0d+1] = {
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_0   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_1   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_2   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_3   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_4   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_5   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_6   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_7   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_8   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_9   */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_10  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_11  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_12  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_13  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_14  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_15  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_16  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_17  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}, /* P_18  */
  {"0d", MAX_QUAD_DEG_0d, 0, 1, gx_0d, w_0d}  /* P_19  */
};

#define  StdVol 1.0

/*--------------------------------------------------------------------------*/
/*  1d quadrature formulas using 2 barycentric coordinates                  */
/*--------------------------------------------------------------------------*/

#define MAX_QUAD_DEG_1d   19

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_1                                                 */
/*--------------------------------------------------------------------------*/

static const double  x0_1d[N_LAMBDA] = {0.5, 0.5, 0.0, 0.0};
static const double  *gx0_1d[1] = {x0_1d};
static const double  w0_1d[1] = {StdVol*1.0};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_3                                                 */
/*--------------------------------------------------------------------------*/

static const double  x1_1d[2][N_LAMBDA] =
		{{0.788675134594813, 0.211324865405187, 0.0, 0.0},
		 {0.211324865405187, 0.788675134594813, 0.0, 0.0}};

static const double  *gx1_1d[2] = {x1_1d[0], x1_1d[1]};
static const double  w1_1d[2] = {StdVol*0.5, StdVol*0.5};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_5                                                 */
/*--------------------------------------------------------------------------*/

static const double  x2_1d[3][N_LAMBDA] =
		{{0.887298334620741, 0.112701665379259, 0.0, 0.0},
		 {0.500000000000000, 0.500000000000000, 0.0, 0.0},
		 {0.112701665379259, 0.887298334620741, 0.0, 0.0}};

static const double  *gx2_1d[3] = {x2_1d[0], x2_1d[1], x2_1d[2]};
static const double  w2_1d[3] = {StdVol*0.277777777777778,
				 StdVol*0.444444444444444,
				 StdVol*0.277777777777778};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_7                                                 */
/*--------------------------------------------------------------------------*/

static const double  x3_1d[4][N_LAMBDA] = 
		{{0.930568155797026, 0.069431844202973, 0.0, 0.0},
		 {0.669990521792428, 0.330009478207572, 0.0, 0.0},
		 {0.330009478207572, 0.669990521792428, 0.0, 0.0},
		 {0.069431844202973, 0.930568155797026, 0.0, 0.0}};

static const double  *gx3_1d[4] = {x3_1d[0], x3_1d[1], x3_1d[2], x3_1d[3]};

static const double  w3_1d[4] = {StdVol*0.173927422568727,
				 StdVol*0.326072577431273,
				 StdVol*0.326072577431273,
				 StdVol*0.173927422568727};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_9                                                 */
/*--------------------------------------------------------------------------*/

static const double  x4_1d[5][N_LAMBDA] = 
		  {{0.953089922969332, 0.046910077030668, 0.0, 0.0},
		   {0.769234655052841, 0.230765344947159, 0.0, 0.0},
		   {0.500000000000000, 0.500000000000000, 0.0, 0.0},
		   {0.230765344947159, 0.769234655052841, 0.0, 0.0},
		   {0.046910077030668, 0.953089922969332, 0.0, 0.0}};

static const double  *gx4_1d[5] = {x4_1d[0], x4_1d[1], x4_1d[2], x4_1d[3],
				   x4_1d[4]};

static const double  w4_1d[5] = {StdVol*0.118463442528095,
				 StdVol*0.239314335249683,
				 StdVol*0.284444444444444,
				 StdVol*0.239314335249683,
				 StdVol*0.118463442528095};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_11                                                */
/*--------------------------------------------------------------------------*/

static const double  x5_1d[6][N_LAMBDA] =
		{{0.966234757101576, 0.033765242898424, 0.0, 0.0},
		 {0.830604693233133, 0.169395306766867, 0.0, 0.0},
		 {0.619309593041598, 0.380690406958402, 0.0, 0.0},
		 {0.380690406958402, 0.619309593041598, 0.0, 0.0},
		 {0.169395306766867, 0.830604693233133, 0.0, 0.0},
		 {0.033765242898424, 0.966234757101576, 0.0, 0.0}};

static const double  *gx5_1d[6] = {x5_1d[0], x5_1d[1], x5_1d[2],
				   x5_1d[3], x5_1d[4], x5_1d[5]};

static const double  w5_1d[6] = {StdVol*0.085662246189585,
				 StdVol*0.180380786524069,
				 StdVol*0.233956967286345,
				 StdVol*0.233956967286345,
				 StdVol*0.180380786524069,
				 StdVol*0.085662246189585};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_13                                                */
/*--------------------------------------------------------------------------*/

static const double  x6_1d[7][N_LAMBDA] = 
		{{0.974553956171380, 0.025446043828620, 0.0, 0.0},
		 {0.870765592799697, 0.129234407200303, 0.0, 0.0},
		 {0.702922575688699, 0.297077424311301, 0.0, 0.0},
		 {0.500000000000000, 0.500000000000000, 0.0, 0.0},
		 {0.297077424311301, 0.702922575688699, 0.0, 0.0},
		 {0.129234407200303, 0.870765592799697, 0.0, 0.0},
		 {0.025446043828620, 0.974553956171380, 0.0, 0.0}};

static const double  *gx6_1d[7] = {x6_1d[0], x6_1d[1], x6_1d[2], x6_1d[3],
				   x6_1d[4], x6_1d[5], x6_1d[6]};

static const double  w6_1d[7] = {StdVol*0.064742483084435,
				 StdVol*0.139852695744614,
				 StdVol*0.190915025252559,
				 StdVol*0.208979591836735,
				 StdVol*0.190915025252559,
				 StdVol*0.139852695744614,
				 StdVol*0.064742483084435};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_15                                                */
/*--------------------------------------------------------------------------*/

static const double  x7_1d[8][N_LAMBDA] =
		{{0.980144928248768, 0.019855071751232, 0.0, 0.0},
		 {0.898333238706813, 0.101666761293187, 0.0, 0.0},
		 {0.762766204958164, 0.237233795041836, 0.0, 0.0},
		 {0.591717321247825, 0.408282678752175, 0.0, 0.0},
		 {0.408282678752175, 0.591717321247825, 0.0, 0.0},
		 {0.237233795041836, 0.762766204958164, 0.0, 0.0},
		 {0.101666761293187, 0.898333238706813, 0.0, 0.0},
		 {0.019855071751232, 0.980144928248768, 0.0, 0.0}};


static const double  *gx7_1d[8] = {x7_1d[0], x7_1d[1], x7_1d[2], x7_1d[3],
				   x7_1d[4], x7_1d[5], x7_1d[6], x7_1d[7]};

static const double  w7_1d[8] = {StdVol*0.050614268145188,
				 StdVol*0.111190517226687,
				 StdVol*0.156853322938943,
				 StdVol*0.181341891689181,
				 StdVol*0.181341891689181,
				 StdVol*0.156853322938943,
				 StdVol*0.111190517226687,
				 StdVol*0.050614268145188};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_17                                                */
/*--------------------------------------------------------------------------*/

static const double  x8_1d[9][N_LAMBDA] = 
  {{0.984080119753813, 0.015919880246187, 0.0, 0.0},
   {0.918015553663318, 0.081984446336682, 0.0, 0.0},
   {0.806685716350295, 0.193314283649705, 0.0, 0.0},
   {0.662126711701905, 0.337873288298095, 0.0, 0.0},
   {0.500000000000000, 0.500000000000000, 0.0, 0.0},
   {0.337873288298095, 0.662126711701905, 0.0, 0.0},
   {0.193314283649705, 0.806685716350295, 0.0, 0.0},
   {0.081984446336682, 0.918015553663318, 0.0, 0.0},
   {0.015919880246187, 0.984080119753813, 0.0, 0.0}};

static const double  *gx8_1d[9] = {x8_1d[0], x8_1d[1], x8_1d[2], x8_1d[3],
				   x8_1d[4], x8_1d[5], x8_1d[6], x8_1d[7],
				   x8_1d[8]};
static const double  w8_1d[9] = {StdVol*0.040637194180787,
				 StdVol*0.090324080347429,
				 StdVol*0.130305348201467,
				 StdVol*0.156173538520001,
				 StdVol*0.165119677500630,
				 StdVol*0.156173538520001,
				 StdVol*0.130305348201467,
				 StdVol*0.090324080347429,
				 StdVol*0.040637194180787};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_19                                                */
/*--------------------------------------------------------------------------*/

static const double  x9_1d[10][N_LAMBDA] =
  {{0.986953264258586, 0.013046735741414, 0.0, 0.0},
   {0.932531683344493, 0.067468316655508, 0.0, 0.0},
   {0.839704784149512, 0.160295215850488, 0.0, 0.0},
   {0.716697697064623, 0.283302302935377, 0.0, 0.0},
   {0.574437169490815, 0.425562830509185, 0.0, 0.0},
   {0.425562830509185, 0.574437169490815, 0.0, 0.0},
   {0.283302302935377, 0.716697697064623, 0.0, 0.0},
   {0.160295215850488, 0.839704784149512, 0.0, 0.0},
   {0.067468316655508, 0.932531683344493, 0.0, 0.0},
   {0.013046735741414, 0.986953264258586, 0.0, 0.0}};

static const double  *gx9_1d[10] = {x9_1d[0], x9_1d[1], x9_1d[2], x9_1d[3],
				    x9_1d[4], x9_1d[5], x9_1d[6], x9_1d[7],
				    x9_1d[8], x9_1d[9]};
static const double  w9_1d[10] = {StdVol*0.033335672154344,
				  StdVol*0.074725674575291,
				  StdVol*0.109543181257991,
				  StdVol*0.134633359654998,
				  StdVol*0.147762112357376,
				  StdVol*0.147762112357376,
				  StdVol*0.134633359654998,
				  StdVol*0.109543181257991,
				  StdVol*0.074725674575291,
				  StdVol*0.033335672154344};

static QUAD quad_1d[MAX_QUAD_DEG_1d+1] = {
  {"1d-Gauss: P_1", 1, 1, 1, gx0_1d, w0_1d}, /* P_0   */
  {"1d-Gauss: P_1", 1, 1, 1, gx0_1d, w0_1d}, /* P_1   */
  {"1d-Gauss: P_3", 3, 1, 2, gx1_1d, w1_1d}, /* P_2   */
  {"1d-Gauss: P_3", 3, 1, 2, gx1_1d, w1_1d}, /* P_3   */
  {"1d-Gauss: P_5", 5, 1, 3, gx2_1d, w2_1d}, /* P_4   */
  {"1d-Gauss: P_5", 5, 1, 3, gx2_1d, w2_1d}, /* P_5   */
  {"1d-Gauss: P_7", 7, 1, 4, gx3_1d, w3_1d}, /* P_6   */
  {"1d-Gauss: P_7", 7, 1, 4, gx3_1d, w3_1d}, /* P_7   */
  {"1d-Gauss: P_9", 9, 1, 5, gx4_1d, w4_1d}, /* P_8   */
  {"1d-Gauss: P_9", 9, 1, 5, gx4_1d, w4_1d}, /* P_9   */
  {"1d-Gauss: P_11", 11, 1, 6, gx5_1d, w5_1d}, /* P_10  */
  {"1d-Gauss: P_11", 11, 1, 6, gx5_1d, w5_1d}, /* P_11  */
  {"1d-Gauss: P_13", 13, 1, 7, gx6_1d, w6_1d}, /* P_12  */
  {"1d-Gauss: P_13", 13, 1, 7, gx6_1d, w6_1d}, /* P_13  */
  {"1d-Gauss: P_15", 15, 1, 8, gx7_1d, w7_1d}, /* P_14  */
  {"1d-Gauss: P_15", 15, 1, 8, gx7_1d, w7_1d}, /* P_15  */
  {"1d-Gauss: P_17", 17, 1, 9, gx8_1d, w8_1d}, /* P_16  */
  {"1d-Gauss: P_17", 17, 1, 9, gx8_1d, w8_1d}, /* P_17  */
  {"1d-Gauss: P_19", 19, 1, 10, gx9_1d, w9_1d}, /* P_18 */
  {"1d-Gauss: P_19", 19, 1, 10, gx9_1d, w9_1d}};/* P_19 */

#undef StdVol
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*  2d quadrature formulas using 3 barycentric coordinates                  */
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

#define CYCLE(c1,c2,c3)     {c1,c2,c3,0.0},{c2,c3,c1,0.0},{c3,c1,c2,0.0}
#define ALL_COMB(c1,c2,c3)  CYCLE(c1,c2,c3), CYCLE(c1,c3,c2)
#define W_CYCLE(w1)         w1, w1, w1
#define W_ALL_COMB(w1)      W_CYCLE(w1), W_CYCLE(w1)

#define MAX_QUAD_DEG_2d   17

#define StdVol 0.5

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 1                                                 */
/*--------------------------------------------------------------------------*/

#define N1  1

#define c1  1.0/3.0
#define w1  StdVol*1.0

static const double  x1_2d[N_LAMBDA] = {c1, c1, c1, 0};
static const double  *gx1_2d[N1] = {x1_2d};
static const double  w1_2d[N1] = {w1};

#undef c1
#undef w1

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 2                                                 */
/* Stroud, A.H.: Approximate calculation of multiple integrals              */
/* Prentice-Hall Series in Automatic Computation. (1971)                    */
/* optimal number of points: 3, number of points: 3                         */
/* interior points, completly symmetric in barycentric coordinates          */
/*--------------------------------------------------------------------------*/

#define N2  3

#define c1  2.0/3.0
#define c2  1.0/6.0
#define w1  StdVol/3.0

static const double  x2_2d[3][N_LAMBDA] = {CYCLE(c1, c2, c2)};

static const double  *gx2_2d[3] = {x2_2d[0], x2_2d[1], x2_2d[2]};
static const double  w2_2d[3] = {W_CYCLE(w1)};

#undef c1
#undef c2
#undef w1

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_3                                                 */
/*--------------------------------------------------------------------------*/

#define N3  6

#define c1  0.0
#define c2  1.0/2.0
#define c3  4.0/6.0
#define c4  1.0/6.0
#define w1  StdVol*1.0/30.0
#define w2  StdVol*9.0/30.0

static const double  x3_2d[N3][N_LAMBDA] = {CYCLE(c1,c2,c2), CYCLE(c3,c4,c4)};

static const double  *gx3_2d[N3] = {x3_2d[0], x3_2d[1], x3_2d[2],
				    x3_2d[3], x3_2d[4], x3_2d[5]};

static const double  w3_2d[N3] = {W_CYCLE(w1), W_CYCLE(w2)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef w1
#undef w2

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 4                                                 */
/* Dunavant, D.A.: High degree efficient symmetrical Gaussian quadrature    */
/* rules for the triangle. Int. J. Numer. Methods Eng. 21, 1129-1148 (1985) */
/* nearly optimal number of (interior) points, positive wheights  (PI)      */
/* number of points: 6, optimal number of points: 6                         */
/*--------------------------------------------------------------------------*/

#define N4  6

#define c1  0.816847572980459
#define c2  0.091576213509771
#define c3  0.108103018168070
#define c4  0.445948490915965
#define w1  StdVol*0.109951743655322
#define w2  StdVol*0.223381589678011

static const double  x4_2d[N4][N_LAMBDA] = {CYCLE(c1,c2,c2), CYCLE(c3,c4,c4)};

static const double  *gx4_2d[N4] = {x4_2d[0], x4_2d[1], x4_2d[2],
				    x4_2d[3], x4_2d[4], x4_2d[5]};
static const double  w4_2d[N4] = {W_CYCLE(w1), W_CYCLE(w2)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef w1
#undef w2


/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 5                                                 */
/*--------------------------------------------------------------------------*/

#if 0
/*--------------------------------------------------------------------------*/
/* Stroud, A.H.: Approximate calculation of multiple integrals              */
/* Prentice-Hall Series in Automatic Computation. (1971)                    */
/* number of points: 7, optimal number of points: 7                         */
/*--------------------------------------------------------------------------*/

#define N5   7

#define c1   1.0/3.0
#define c2   0.0 
#define c3   1.0/2.0
#define c4   1.0
#define c5   0.0 
#define w1   StdVol*0.45
#define w2   StdVol*4.0/30.0
#define w3   StdVol*0.05
#else

/*--------------------------------------------------------------------------*/
/* Dunavant, D.A.: High degree efficient symmetrical Gaussian quadrature    */
/* rules for the triangle. Int. J. Numer. Methods Eng. 21, 1129-1148 (1985) */
/* nealy optimal number of (interior) points, positive wheights  (PI)       */
/* number of points: 7, optimal number of points: 7                         */
/*--------------------------------------------------------------------------*/

#define N5   7

#define c1   1.0/3.0
#define c2   0.797426985353087
#define c3   0.101286507323456
#define c4   0.059715871789770
#define c5   0.470142064105115
#define w1   StdVol*0.225000000000000
#define w2   StdVol*0.125939180544827
#define w3   StdVol*0.132394152788506
#endif

static const double  x5_2d[N5][N_LAMBDA] = {{c1, c1, c1, 0.0},
					    CYCLE(c2,c3,c3), CYCLE(c4,c5,c5)};

static const double  *gx5_2d[N5] = {x5_2d[0], x5_2d[1], x5_2d[2],  x5_2d[3], 
				    x5_2d[4], x5_2d[5], x5_2d[6]};

static const double  w5_2d[N5] = {w1, W_CYCLE(w2), W_CYCLE(w3)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef c5
#undef w1
#undef w2
#undef w3

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 6: only 12 point rule available in the literature */
/*  ->  use quadrature exact on P 7 with 12 points                          */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 7                                                 */
/*--------------------------------------------------------------------------*/

#if 1
/*--------------------------------------------------------------------------*/
/* Gatermann, Karin: The construction of symmetric cubature formulas for    */
/* the square and the triangle. Computing 40, No.3, 229-240 (1988)          */
/* optimal number of points: 12, number of points: 12                       */
/* only interior points, not completly symmetric in barycentric coordinates */
/*--------------------------------------------------------------------------*/

#define N7   12

#define c1   0.06238226509439084
#define c2   0.06751786707392436
#define c3   0.87009986783168480
#define c4   0.05522545665692000
#define c5   0.32150249385201560
#define c6   0.62327204949106440
#define c7   0.03432430294509488
#define c8   0.66094919618679800
#define c9   0.30472650086810720
#define c10  0.5158423343536001
#define c11  0.2777161669764050
#define c12  0.2064414986699949
#define w1   0.02651702815743450
#define w2   0.04388140871444811
#define w3   0.02877504278497528
#define w4   0.06749318700980879

static const double  x7_2d[N7][N_LAMBDA] = {CYCLE(c1,c2,c3), CYCLE(c4,c5,c6),
					 CYCLE(c7,c8,c9), CYCLE(c10,c11,c12)};

static const double  *gx7_2d[N7] = {x7_2d[0], x7_2d[1], x7_2d[2], x7_2d[3], 
				    x7_2d[4], x7_2d[5], x7_2d[6], x7_2d[7],
				    x7_2d[8], x7_2d[9], x7_2d[10], x7_2d[11]};

static const double  w7_2d[N7] = {W_CYCLE(w1), W_CYCLE(w2),
				  W_CYCLE(w3), W_CYCLE(w4)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef c5
#undef c6
#undef c7
#undef c8
#undef c9
#undef c10
#undef c11
#undef c12
#undef w1
#undef w2
#undef w3
#undef w4

#else  

/*--------------------------------------------------------------------------*/
/* Stroud, A.H.: Approximate calculation of multiple integrals              */
/* Prentice-Hall Series in Automatic Computation. (1971)                    */
/* optimal number of points: 12, number of points: 16                       */
/* only interior points, not symmetric in barycentric coordinates           */
/*--------------------------------------------------------------------------*/

#define N7  16

static const double  x7_2d[N7][N_LAMBDA] = {
  {0.0571041961, 0.065466992667427, 0.877428811232573, 0.0},
  {0.2768430136, 0.050210121765552, 0.672946864634448, 0.0},
  {0.5835904324, 0.028912083388173, 0.387497484211827, 0.0},
  {0.8602401357, 0.009703784843971, 0.130056079456029, 0.0},
  {0.0571041961, 0.311164552242009, 0.631731251657992, 0.0},
  {0.2768430136, 0.238648659738548, 0.484508326661452, 0.0},
  {0.5835904324, 0.137419104121164, 0.278990463478836, 0.0},
  {0.8602401357, 0.046122079890946, 0.093637784409054, 0.0},
  {0.0571041961, 0.631731251657991, 0.311164552242009, 0.0},
  {0.2768430136, 0.484508326661451, 0.238648659738549, 0.0},
  {0.5835904324, 0.278990463478836, 0.137419104121164, 0.0},
  {0.8602401357, 0.093637784409054, 0.046122079890946, 0.0},
  {0.0571041961, 0.877428809346781, 0.065466994553219, 0.0},
  {0.2768430136, 0.672946863188134, 0.050210123211866, 0.0},
  {0.5835904324, 0.387497483379007, 0.028912084220993, 0.0},
  {0.8602401357, 0.130056079176509, 0.009703785123491, 0.0}};

static const double  *gx7_2d[N7] ={x7_2d[0], x7_2d[1], x7_2d[2], x7_2d[3], 
				   x7_2d[4], x7_2d[5], x7_2d[6], x7_2d[7],
				   x7_2d[8], x7_2d[9], x7_2d[10], x7_2d[11],
				   x7_2d[12], x7_2d[13], x7_2d[14], x7_2d[15]};

static const double  w7_2d[N7] = {StdVol*0.047136736384287,
				  StdVol*0.070776135805325,
				  StdVol*0.045168098569998,
				  StdVol*0.010846451805605,
				  StdVol*0.088370177015713,
				  StdVol*0.132688432194675,
				  StdVol*0.084679449030002,
				  StdVol*0.020334519094395,
				  StdVol*0.088370177015713,
				  StdVol*0.132688432194675,
				  StdVol*0.084679449030002,
				  StdVol*0.020334519094395,
				  StdVol*0.047136736384287,
				  StdVol*0.070776135805325,
				  StdVol*0.045168098569998,
				  StdVol*0.010846451805605};

#endif


/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 8                                                 */
/* Dunavant, D.A.: High degree efficient symmetrical Gaussian quadrature    */
/* rules for the triangle. Int. J. Numer. Methods Eng. 21, 1129-1148 (1985) */
/* nealy optimal number of (interior) points, positive wheights  (PI)       */
/* number of points: 16, optimal number of points: 15                       */
/* only interior points, completly symmetric in barycentric coordinates     */
/*--------------------------------------------------------------------------*/

#define N8  16

#define c1   1.0/3.0
#define c2   0.081414823414554
#define c3   0.459292588292723
#define c4   0.658861384496480
#define c5   0.170569307751760
#define c6   0.898905543365938
#define c7   0.050547228317031
#define c8   0.008394777409958
#define c9   0.263112829634638
#define c10  0.728492392955404
#define w1   StdVol*0.144315607677787
#define w2   StdVol*0.095091634267285
#define w3   StdVol*0.103217370534718
#define w4   StdVol*0.032458497623198
#define w5   StdVol*0.027230314174435

static const double  x8_2d[N8][N_LAMBDA] = {{c1, c1, c1, 0.0},
					    CYCLE(c2,c3,c3),
					    CYCLE(c4,c5,c5),
					    CYCLE(c6,c7,c7),
					    ALL_COMB(c8,c9,c10)};

static const double  *gx8_2d[N8] ={x8_2d[0], x8_2d[1], x8_2d[2], x8_2d[3], 
				   x8_2d[4], x8_2d[5], x8_2d[6], x8_2d[7],
				   x8_2d[8], x8_2d[9], x8_2d[10], x8_2d[11],
				   x8_2d[12], x8_2d[13], x8_2d[14], x8_2d[15]};

static const double  w8_2d[N8] = {w1, W_CYCLE(w2), W_CYCLE(w3),
				  W_CYCLE(w4), W_ALL_COMB(w5)};
				  

#undef c1
#undef c2
#undef c3
#undef c4
#undef c5
#undef c6
#undef c7
#undef c8
#undef c9
#undef c10
#undef w1
#undef w2
#undef w3
#undef w4
#undef w5

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 9                                                 */
/* Dunavant, D.A.: High degree efficient symmetrical Gaussian quadrature    */
/* rules for the triangle. Int. J. Numer. Methods Eng. 21, 1129-1148 (1985) */
/* nealy optimal number of (interior) points, positive wheights  (PI)       */
/* optimal number of points: ?, number of points: 19                        */
/* only interior points, completly symmetric in barycentric coordinates     */
/*--------------------------------------------------------------------------*/

#define N9  19

#define c1   1.0/3.0
#define c2   0.020634961602525
#define c3   0.489682519198738
#define c4   0.125820817014127
#define c5   0.437089591492937
#define c6   0.623592928761935
#define c7   0.188203535619033
#define c8   0.910540973211095
#define c9   0.044729513394453
#define c10  0.036838412054736
#define c11  0.221962989160766
#define c12  0.741198598784498
#define w1   StdVol*0.097135796282799
#define w2   StdVol*0.031334700227139
#define w3   StdVol*0.077827541004774
#define w4   StdVol*0.079647738927210
#define w5   StdVol*0.025577675658698
#define w6   StdVol*0.043283539377289

static const double  x9_2d[N9][N_LAMBDA] = {{c1, c1, c1, 0.0},
					    CYCLE(c2,c3,c3),
					    CYCLE(c4,c5,c5),
					    CYCLE(c6,c7,c7),
					    CYCLE(c8,c9,c9),
					    ALL_COMB(c10,c11,c12)};

static const double  *gx9_2d[N9] ={x9_2d[0], x9_2d[1], x9_2d[2], x9_2d[3], 
				   x9_2d[4], x9_2d[5], x9_2d[6], x9_2d[7],
				   x9_2d[8], x9_2d[9], x9_2d[10], x9_2d[11],
				   x9_2d[12], x9_2d[13], x9_2d[14], x9_2d[15],
				   x9_2d[16], x9_2d[17], x9_2d[18]};

static const double  w9_2d[N9] = {w1, 
				  W_CYCLE(w2),
				  W_CYCLE(w3),
				  W_CYCLE(w4),
				  W_CYCLE(w5),
				  W_ALL_COMB(w6)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef c5
#undef c6
#undef c7
#undef c8
#undef c9
#undef c10
#undef c11
#undef c12
#undef w1
#undef w2
#undef w3
#undef w4
#undef w5
#undef w6

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 10                                                */
/* Dunavant, D.A.: High degree efficient symmetrical Gaussian quadrature    */
/* rules for the triangle. Int. J. Numer. Methods Eng. 21, 1129-1148 (1985) */
/* nealy optimal number of (interior) points, positive wheights  (PI)       */
/* optimal number of points: ?, number of points: 25                        */
/* only interior points, completly symmetric in barycentric coordinates     */
/*--------------------------------------------------------------------------*/

#define N10 25

#define c1   1.0/3.0

#define c2   0.028844733232685
#define c3   0.485577633383657

#define c4   0.781036849029926
#define c5   0.109481575485037

#define c6   0.141707219414880
#define c7   0.307939838764121
#define c8   0.550352941820999

#define c9   0.025003534762686
#define c10  0.246672560639903
#define c11  0.728323904597411

#define c12  0.009540815400299
#define c13  0.066803251012200
#define c14  0.923655933587500

#define w1   StdVol*0.090817990382754
#define w2   StdVol*0.036725957756467
#define w3   StdVol*0.045321059435528
#define w4   StdVol*0.072757916845420
#define w5   StdVol*0.028327242531057
#define w6   StdVol*0.009421666963733

static const double  x10_2d[N10][N_LAMBDA] = {{c1, c1, c1, 0.0},
					      CYCLE(c2,c3,c3),
					      CYCLE(c4,c5,c5),
					      ALL_COMB(c6,c7,c8),
					      ALL_COMB(c9,c10,c11),
					      ALL_COMB(c12,c13,c14)};

static const double  *gx10_2d[N10] = 
   {x10_2d[0], x10_2d[1], x10_2d[2], x10_2d[3],
    x10_2d[4], x10_2d[5], x10_2d[6], x10_2d[7],
    x10_2d[8], x10_2d[9], x10_2d[10], x10_2d[11],
    x10_2d[12], x10_2d[13], x10_2d[14], x10_2d[15],
    x10_2d[16], x10_2d[17], x10_2d[18], x10_2d[19],
    x10_2d[20], x10_2d[21], x10_2d[22], x10_2d[23],
    x10_2d[24]};

static const double  w10_2d[N10] = {w1, 
				    W_CYCLE(w2),
				    W_CYCLE(w3),
				    W_ALL_COMB(w4),
				    W_ALL_COMB(w5),
				    W_ALL_COMB(w6)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef c5
#undef c6
#undef c7
#undef c8
#undef c9
#undef c10
#undef c11
#undef c12
#undef c13
#undef c14
#undef w1
#undef w2
#undef w3
#undef w4
#undef w5
#undef w6

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 11                                                */
/* Dunavant, D.A.: High degree efficient symmetrical Gaussian quadrature    */
/* rules for the triangle. Int. J. Numer. Methods Eng. 21, 1129-1148 (1985) */
/* nealy optimal number of (interior) points, positive wheights  (PI)       */
/* optimal number of points: ?, number of points: 27                        */
/* only interior points, completly symmetric in barycentric coordinates     */
/*--------------------------------------------------------------------------*/

#define N11 27

#define c1  -0.069222096541517
#define c2   0.534611048270758

#define c3   0.202061394068290
#define c4   0.398969302965855

#define c5   0.593380199137435
#define c6   0.203309900431282

#define c7   0.761298175434837
#define c8   0.119350912282581

#define c9   0.935270103777448
#define c10  0.032364948111276

#define c11  0.050178138310495
#define c12  0.356620648261293
#define c13  0.593201213428213

#define c14  0.021022016536166
#define c15  0.171488980304042
#define c16  0.807489003159792

#define w1   StdVol*0.000927006328961
#define w2   StdVol*0.077149534914813
#define w3   StdVol*0.059322977380774
#define w4   StdVol*0.036184540503418
#define w5   StdVol*0.013659731002678
#define w6   StdVol*0.052337111962204
#define w7   StdVol*0.020707659639141

static const double  x11_2d[N11][N_LAMBDA] = {CYCLE(c1,c2,c2),
					      CYCLE(c3,c4,c4),
					      CYCLE(c5,c6,c6),
					      CYCLE(c7,c8,c8),
					      CYCLE(c9,c10,c10),
					      ALL_COMB(c11,c12,c13),
					      ALL_COMB(c14,c15,c16)};

static const double  *gx11_2d[N11] = 
   {x11_2d[0], x11_2d[1], x11_2d[2], x11_2d[3],
    x11_2d[4], x11_2d[5], x11_2d[6], x11_2d[7],
    x11_2d[8], x11_2d[9], x11_2d[10], x11_2d[11],
    x11_2d[12], x11_2d[13], x11_2d[14], x11_2d[15],
    x11_2d[16], x11_2d[17], x11_2d[18], x11_2d[19],
    x11_2d[20], x11_2d[21], x11_2d[22], x11_2d[23],
    x11_2d[24], x11_2d[25], x11_2d[26]};

static const double  w11_2d[N11] = {W_CYCLE(w1),
				    W_CYCLE(w2),
				    W_CYCLE(w3),
				    W_CYCLE(w4),
				    W_CYCLE(w5),
				    W_ALL_COMB(w6),
				    W_ALL_COMB(w7)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef c5
#undef c6
#undef c7
#undef c8
#undef c9
#undef c10
#undef c11
#undef c12
#undef c13
#undef c14
#undef c15
#undef c16
#undef w1
#undef w2
#undef w3
#undef w4
#undef w5
#undef w6
#undef w7

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 12                                                */
/* Dunavant, D.A.: High degree efficient symmetrical Gaussian quadrature    */
/* rules for the triangle. Int. J. Numer. Methods Eng. 21, 1129-1148 (1985) */
/* nealy optimal number of (interior) points, positive wheights  (PI)       */
/* optimal number of points: 2, number of points: 25                        */
/* only interior points, completly symmetric in barycentric coordinates     */
/*--------------------------------------------------------------------------*/

#define N12 33

#define c1   0.023565220452390
#define c2   0.488217389773805

#define c3   0.120551215411079
#define c4   0.439724392294460

#define c5   0.457579229975768
#define c6   0.271210385012116

#define c7   0.744847708916828
#define c8   0.127576145541586
	       
#define c9   0.957365299093579
#define c10  0.021317350453210

#define c11  0.115343494534698
#define c12  0.275713269685514
#define c13  0.608943235779788

#define c14  0.022838332222257
#define c15  0.281325580989940
#define c16  0.695836086787803

#define c17  0.025734050548330
#define c18  0.116251915907597
#define c19  0.858014033544073

#define w1   StdVol*0.025731066440455
#define w2   StdVol*0.043692544538038
#define w3   StdVol*0.062858224217885
#define w4   StdVol*0.034796112930709
#define w5   StdVol*0.006166261051559
#define w6   StdVol*0.040371557766381
#define w7   StdVol*0.022356773202303
#define w8   StdVol*0.017316231108659

static const double  x12_2d[N12][N_LAMBDA] = {CYCLE(c1,c2,c2),
					      CYCLE(c3,c4,c4),
					      CYCLE(c5,c6,c6),
					      CYCLE(c7,c8,c8),
					      CYCLE(c9,c10,c10),
					      ALL_COMB(c11,c12,c13),
					      ALL_COMB(c14,c15,c16),
					      ALL_COMB(c17,c18,c19)};

static const double  *gx12_2d[N12] = 
   {x12_2d[0], x12_2d[1], x12_2d[2], x12_2d[3],
    x12_2d[4], x12_2d[5], x12_2d[6], x12_2d[7],
    x12_2d[8], x12_2d[9], x12_2d[10], x12_2d[11],
    x12_2d[12], x12_2d[13], x12_2d[14], x12_2d[15],
    x12_2d[16], x12_2d[17], x12_2d[18], x12_2d[19],
    x12_2d[20], x12_2d[21], x12_2d[22], x12_2d[23],
    x12_2d[24], x12_2d[25], x12_2d[26], x12_2d[27],
    x12_2d[28], x12_2d[29], x12_2d[30], x12_2d[31],
    x12_2d[32]};

static const double  w12_2d[N12] = {W_CYCLE(w1),
				    W_CYCLE(w2),
				    W_CYCLE(w3),
				    W_CYCLE(w4),
				    W_CYCLE(w5),
				    W_ALL_COMB(w6),
				    W_ALL_COMB(w7),
				    W_ALL_COMB(w8)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef c5
#undef c6
#undef c7
#undef c8
#undef c9
#undef c10
#undef c11
#undef c12
#undef c13
#undef c14
#undef c15
#undef c16
#undef c17
#undef c18
#undef c19
#undef w1
#undef w2
#undef w3
#undef w4
#undef w5
#undef w6
#undef w7
#undef w8

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P 17                                                */
/* Dunavant, D.A.: High degree efficient symmetrical Gaussian quadrature    */
/* rules for the triangle. Int. J. Numer. Methods Eng. 21, 1129-1148 (1985) */
/* nealy optimal number of (interior) points, positive wheights  (PI)       */
/* optimal number of points: ?, number of points: 61                        */
/* only interior points, completly symmetric in barycentric coordinates     */
/*--------------------------------------------------------------------------*/

#define N17 61

#define c1   1.0/3.0
	       
#define c2   0.005658918886452
#define c3   0.497170540556774
	       
#define c4   0.035647354750751
#define c5   0.482176322624625
	       
#define c6   0.099520061958437
#define c7   0.450239969020782

#define c8   0.199467521245206
#define c9   0.400266239377397

#define c10  0.495717464058095
#define c11  0.252141267970953
	       
#define c12  0.675905990683077
#define c13  0.162047004658461

#define c14  0.848248235478508
#define c15  0.075875882260746

#define c16  0.968690546064356
#define c17  0.015654726967822

#define c18  0.010186928826919
#define c19  0.334319867363658
#define c20  0.655493203809423

#define c21  0.135440871671036
#define c22  0.292221537796944
#define c23  0.572337590532020

#define c24  0.054423924290583
#define c25  0.319574885423190
#define c26  0.626001190286228

#define c27  0.012868560833637
#define c28  0.190704224192292
#define c29  0.796427214974071

#define c30  0.067165782413524
#define c31  0.180483211648746
#define c32  0.752351005937729

#define c33  0.014663182224828
#define c34  0.080711313679564
#define c35  0.904625504095608

#define w1   StdVol*0.033437199290803
#define w2   StdVol*0.005093415440507
#define w3   StdVol*0.014670864527638
#define w4   StdVol*0.024350878353672
#define w5   StdVol*0.031107550868969
#define w6   StdVol*0.031257111218620
#define w7   StdVol*0.024815654339665
#define w8   StdVol*0.014056073070557
#define w9   StdVol*0.003194676173779
#define w10  StdVol*0.008119655318993
#define w11  StdVol*0.026805742283163
#define w12  StdVol*0.018459993210822
#define w13  StdVol*0.008476868534328
#define w14  StdVol*0.018292796770025
#define w15  StdVol*0.006665632004165

static const double  x17_2d[N17][N_LAMBDA] = {{c1, c1, c1, 0.0},
					      CYCLE(c2,c3,c3),
					      CYCLE(c4,c5,c5),
					      CYCLE(c6,c7,c7),
					      CYCLE(c8,c9,c9),
					      CYCLE(c10,c11,c11),
					      CYCLE(c12,c13,c13),
					      CYCLE(c14,c15,c15),
					      CYCLE(c16,c17,c17),
					      ALL_COMB(c18,c19,c20),
					      ALL_COMB(c21,c22,c23),
					      ALL_COMB(c24,c25,c26),
					      ALL_COMB(c27,c28,c29),
					      ALL_COMB(c30,c31,c32),
					      ALL_COMB(c33,c34,c35)};

static const double  *gx17_2d[N17] = 
   {x17_2d[0], x17_2d[1], x17_2d[2], x17_2d[3],
    x17_2d[4], x17_2d[5], x17_2d[6], x17_2d[7],
    x17_2d[8], x17_2d[9], x17_2d[10], x17_2d[11],
    x17_2d[12], x17_2d[13], x17_2d[14], x17_2d[15],
    x17_2d[16], x17_2d[17], x17_2d[18], x17_2d[19],
    x17_2d[20], x17_2d[21], x17_2d[22], x17_2d[23],
    x17_2d[24], x17_2d[25], x17_2d[26], x17_2d[27],
    x17_2d[28], x17_2d[29], x17_2d[30], x17_2d[31],
    x17_2d[32], x17_2d[33], x17_2d[34], x17_2d[35],
    x17_2d[36], x17_2d[37], x17_2d[38], x17_2d[39],
    x17_2d[40], x17_2d[41], x17_2d[42], x17_2d[43],
    x17_2d[44], x17_2d[45], x17_2d[46], x17_2d[47],
    x17_2d[48], x17_2d[49], x17_2d[50], x17_2d[51],
    x17_2d[52], x17_2d[53], x17_2d[54], x17_2d[55],
    x17_2d[56], x17_2d[57], x17_2d[58], x17_2d[59],
    x17_2d[60]};

static const double  w17_2d[N17] = {w1, 
				    W_CYCLE(w2),
				    W_CYCLE(w3),
				    W_CYCLE(w4),
				    W_CYCLE(w5),
				    W_CYCLE(w6),
				    W_CYCLE(w7),
				    W_CYCLE(w8),
				    W_CYCLE(w9),
				    W_ALL_COMB(w10),
				    W_ALL_COMB(w11),
				    W_ALL_COMB(w12),
				    W_ALL_COMB(w13),
				    W_ALL_COMB(w14),
				    W_ALL_COMB(w15)};

#undef c1
#undef c2
#undef c3
#undef c4
#undef c5
#undef c6
#undef c7
#undef c8
#undef c9
#undef c10
#undef c11
#undef c12
#undef c13
#undef c14
#undef c15
#undef c16
#undef c17
#undef c18
#undef c19
#undef c20
#undef c21
#undef c22
#undef c23
#undef c24
#undef c25
#undef c26
#undef c27
#undef c28
#undef c29
#undef c30
#undef c31
#undef c32
#undef c33
#undef c34
#undef c35

#undef w1
#undef w2
#undef w3
#undef w4
#undef w5
#undef w6
#undef w7
#undef w8
#undef w9
#undef w10
#undef w11
#undef w12
#undef w13
#undef w14
#undef w15

static QUAD quad_2d[MAX_QUAD_DEG_2d+1] = {
  {"2d-P_1", 1, 2, N1, gx1_2d, w1_2d},   /* P 0  */
  {"2d-P_1", 1, 2, N1, gx1_2d, w1_2d},   /* P 1  */
  {"2d  Stroud: P_2", 2, 2, N2, gx2_2d, w2_2d},   /* P 2  */
  {"2d  Stroud: P_3", 3, 2, N3, gx3_2d, w3_2d},   /* P 3  */
  {"2d  Dunavant: P_4", 4, 2, N4, gx4_2d, w4_2d},   /* P 4  */
  {"2d  Dunavant: P_5", 5, 2, N5, gx5_2d, w5_2d},   /* P 5  */
  {"2d  Gattermann: P_7", 7, 2, N7, gx7_2d, w7_2d},   /* P 6  */
  {"2d  Gattermann: P_7", 7, 2, N7, gx7_2d, w7_2d},   /* P 7  */
  {"2d  Dunavant: P_8", 8, 2, N8, gx8_2d, w8_2d},   /* P 8  */
  {"2d  Dunavant: P_9", 9, 2, N9, gx9_2d, w9_2d},   /* P 9  */
  {"2d  Dunavant: P_10", 10, 2, N10, gx10_2d, w10_2d},/* P 10 */
  {"2d  Dunavant: P_11", 11, 2, N11, gx11_2d, w11_2d},/* P 11 */
  {"2d  Dunavant: P_12", 12, 2, N12, gx12_2d, w12_2d},/* P 12 */
  {"2d  Dunavant: P_17", 17, 2, N17, gx17_2d, w17_2d},/* P 13 */
  {"2d  Dunavant: P_17", 17, 2, N17, gx17_2d, w17_2d},/* P 14 */
  {"2d  Dunavant: P_17", 17, 2, N17, gx17_2d, w17_2d},/* P 15 */
  {"2d  Dunavant: P_17", 17, 2, N17, gx17_2d, w17_2d},/* P 16 */
  {"2d  Dunavant: P_17", 17, 2, N17, gx17_2d, w17_2d} /* P 17 */
};

#undef StdVol
#undef N1
#undef N2
#undef N3
#undef N4
#undef N5
#undef N6
#undef N7
#undef N8
#undef N9
#undef N10
#undef N11
#undef N12
#undef N17

/*--------------------------------------------------------------------------*/
/*  3d quadrature formulas using 4 barycentric coordinates                  */
/*--------------------------------------------------------------------------*/

#define MAX_QUAD_DEG_3d   7

#define StdVol (1.0/6.0)

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_1                                                 */
/*--------------------------------------------------------------------------*/

static const double  x1_3d[N_LAMBDA] = {quart, quart, quart, quart};

static const double  *gx1_3d[1] = {x1_3d};
static const double  w1_3d[1] = {StdVol*one};



/*--------------------------------------------------------------------------*/
/*  Quad quadrature exact on P_2                                           */
/*--------------------------------------------------------------------------*/

#define c14   0.585410196624969
#define c15   0.138196601125011

static const double  x2_3d[4][N_LAMBDA] = {{c14, c15, c15, c15},
					   {c15, c14, c15, c15},
					   {c15, c15, c14, c15},
					   {c15, c15, c15, c14}};

static const double  *gx2_3d[4] = {x2_3d[0], x2_3d[1], x2_3d[2], x2_3d[3]};
static double  w2_3d[4] = {StdVol*quart, StdVol*quart,
			   StdVol*quart, StdVol*quart};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_3                                                 */
/*--------------------------------------------------------------------------*/

#define w8  1.0/40.0
#define w9  9.0/40.0

static const double  x3_3d[8][N_LAMBDA] = {{one,  zero,  zero,  zero},
					   {zero,   one,  zero,  zero},
					   {zero,  zero,   one,  zero},
					   {zero,  zero,  zero,   one},
					   {zero, third, third, third},
					   {third,  zero, third, third},
					   {third, third, zero,  third},
					   {third, third, third,  zero}};

static const double  *gx3_3d[8] = {x3_3d[0], x3_3d[1], x3_3d[2], x3_3d[3],
				   x3_3d[4], x3_3d[5], x3_3d[6], x3_3d[7]};

static const double   w3_3d[8] = {StdVol*w8, StdVol*w8, StdVol*w8, StdVol*w8,
				  StdVol*w9, StdVol*w9, StdVol*w9, StdVol*w9};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_5                                                 */
/*--------------------------------------------------------------------------*/

static const double x5_3d[15][N_LAMBDA] = 
 {{0.250000000000000, 0.250000000000000, 0.250000000000000, 0.250000000000000},
  {0.091971078052723, 0.091971078052723, 0.091971078052723, 0.724086765841831},
  {0.724086765841831, 0.091971078052723, 0.091971078052723, 0.091971078052723},
  {0.091971078052723, 0.724086765841831, 0.091971078052723, 0.091971078052723},
  {0.091971078052723, 0.091971078052723, 0.724086765841831, 0.091971078052723},
  {0.319793627829630, 0.319793627829630, 0.319793627829630, 0.040619116511110},
  {0.040619116511110, 0.319793627829630, 0.319793627829630, 0.319793627829630},
  {0.319793627829630, 0.040619116511110, 0.319793627829630, 0.319793627829630},
  {0.319793627829630, 0.319793627829630, 0.040619116511110, 0.319793627829630},
  {0.443649167310371, 0.056350832689629, 0.056350832689629, 0.443649167310371},
  {0.056350832689629, 0.443649167310371, 0.056350832689629, 0.443649167310371},
  {0.056350832689629, 0.056350832689629, 0.443649167310371, 0.443649167310371},
  {0.443649167310371, 0.056350832689629, 0.443649167310371, 0.056350832689629},
  {0.443649167310371, 0.443649167310371, 0.056350832689629, 0.056350832689629},
 {0.056350832689629, 0.443649167310371, 0.443649167310371, 0.056350832689629}};

static const double *gx5_3d[15] = {x5_3d[0],  x5_3d[1],  x5_3d[2],   
				   x5_3d[3],  x5_3d[4],  x5_3d[5],
				   x5_3d[6],  x5_3d[7],  x5_3d[8],
				   x5_3d[9],  x5_3d[10], x5_3d[11],
				   x5_3d[12], x5_3d[13], x5_3d[14]};

static const double w5_3d[15] = {StdVol*0.118518518518519,
				 StdVol*0.071937083779019,
				 StdVol*0.071937083779019,
				 StdVol*0.071937083779019,
				 StdVol*0.071937083779019,
				 StdVol*0.069068207226272,
				 StdVol*0.069068207226272,
				 StdVol*0.069068207226272,
				 StdVol*0.069068207226272,
				 StdVol*0.052910052910053,
				 StdVol*0.052910052910053,
				 StdVol*0.052910052910053,
				 StdVol*0.052910052910053,
				 StdVol*0.052910052910053,
				 StdVol*0.052910052910053};

/*--------------------------------------------------------------------------*/
/*  quadrature exact on P_7                                                 */
/*--------------------------------------------------------------------------*/

static const double x7_3d[64][N_LAMBDA] = 
         {{0.0485005494, 0.0543346112, 0.0622918076, 0.8348730318},
	  {0.0485005494, 0.0543346112, 0.2960729005, 0.6010919389},
	  {0.0485005494, 0.0543346112, 0.6010919389, 0.2960729005},
	  {0.0485005494, 0.0543346112, 0.8348730300, 0.0622918093},
	  {0.0485005494, 0.2634159753, 0.0477749033, 0.6403085720},
	  {0.0485005494, 0.2634159753, 0.2270740686, 0.4610094066},
	  {0.0485005494, 0.2634159753, 0.4610094066, 0.2270740686},
	  {0.0485005494, 0.2634159753, 0.6403085706, 0.0477749047},
	  {0.0485005494, 0.5552859758, 0.0275098315, 0.3687036433},
	  {0.0485005494, 0.5552859758, 0.1307542021, 0.2654592727},
	  {0.0485005494, 0.5552859758, 0.2654592727, 0.1307542021},
	  {0.0485005494, 0.5552859758, 0.3687036425, 0.0275098323},
	  {0.0485005494, 0.8185180165, 0.0092331459, 0.1237482881},
	  {0.0485005494, 0.8185180165, 0.0438851337, 0.0890963004},
	  {0.0485005494, 0.8185180165, 0.0890963004, 0.0438851337},
	  {0.0485005494, 0.8185180165, 0.1237482879, 0.0092331462},
	  {0.2386007376, 0.0434790928, 0.0498465199, 0.6680736497},
	  {0.2386007376, 0.0434790928, 0.2369204606, 0.4809997090},
	  {0.2386007376, 0.0434790928, 0.4809997090, 0.2369204606},
	  {0.2386007376, 0.0434790928, 0.6680736482, 0.0498465214},
	  {0.2386007376, 0.2107880664, 0.0382299497, 0.5123812464},
	  {0.2386007376, 0.2107880664, 0.1817069135, 0.3689042825},
	  {0.2386007376, 0.2107880664, 0.3689042825, 0.1817069135},
	  {0.2386007376, 0.2107880664, 0.5123812453, 0.0382299508},
	  {0.2386007376, 0.4443453248, 0.0220136390, 0.2950402987},
	  {0.2386007376, 0.4443453248, 0.1046308045, 0.2124231331},
	  {0.2386007376, 0.4443453248, 0.2124231331, 0.1046308045},
	  {0.2386007376, 0.4443453248, 0.2950402980, 0.0220136396},
	  {0.2386007376, 0.6549862048, 0.0073884546, 0.0990246030},
	  {0.2386007376, 0.6549862048, 0.0351173176, 0.0712957400},
	  {0.2386007376, 0.6549862048, 0.0712957400, 0.0351173176},
	  {0.2386007376, 0.6549862048, 0.0990246028, 0.0073884548},
	  {0.5170472951, 0.0275786260, 0.0316174612, 0.4237566177},
	  {0.5170472951, 0.0275786260, 0.1502777622, 0.3050963168},
	  {0.5170472951, 0.0275786260, 0.3050963168, 0.1502777622},
	  {0.5170472951, 0.0275786260, 0.4237566168, 0.0316174621},
	  {0.5170472951, 0.1337020823, 0.0242491141, 0.3250015085},
	  {0.5170472951, 0.1337020823, 0.1152560157, 0.2339946069},
	  {0.5170472951, 0.1337020823, 0.2339946069, 0.1152560157},
	  {0.5170472951, 0.1337020823, 0.3250015078, 0.0242491148},
	  {0.5170472951, 0.2818465779, 0.0139631689, 0.1871429581},
	  {0.5170472951, 0.2818465779, 0.0663669280, 0.1347391990},
	  {0.5170472951, 0.2818465779, 0.1347391990, 0.0663669280},
	  {0.5170472951, 0.2818465779, 0.1871429577, 0.0139631693},
	  {0.5170472951, 0.4154553004, 0.0046864691, 0.0628109354},
	  {0.5170472951, 0.4154553004, 0.0222747832, 0.0452226213},
	  {0.5170472951, 0.4154553004, 0.0452226213, 0.0222747832},
	  {0.5170472951, 0.4154553004, 0.0628109352, 0.0046864693},
	  {0.7958514179, 0.0116577407, 0.0133649937, 0.1791258477},
	  {0.7958514179, 0.0116577407, 0.0635238021, 0.1289670393},
	  {0.7958514179, 0.0116577407, 0.1289670393, 0.0635238021},
	  {0.7958514179, 0.0116577407, 0.1791258473, 0.0133649941},
	  {0.7958514179, 0.0565171087, 0.0102503252, 0.1373811482},
	  {0.7958514179, 0.0565171087, 0.0487197855, 0.0989116879},
	  {0.7958514179, 0.0565171087, 0.0989116879, 0.0487197855},
	  {0.7958514179, 0.0565171087, 0.1373811479, 0.0102503255},
	  {0.7958514179, 0.1191391593, 0.0059023608, 0.0791070620},
	  {0.7958514179, 0.1191391593, 0.0280539153, 0.0569555075},
	  {0.7958514179, 0.1191391593, 0.0569555075, 0.0280539153},
	  {0.7958514179, 0.1191391593, 0.0791070618, 0.0059023610},
	  {0.7958514179, 0.1756168040, 0.0019810139, 0.0265507642},
	  {0.7958514179, 0.1756168040, 0.0094157572, 0.0191160209},
	  {0.7958514179, 0.1756168040, 0.0191160209, 0.0094157572},
	  {0.7958514179, 0.1756168040, 0.0265507642, 0.0019810140}};

static const double *gx7_3d[64] = {x7_3d[0],  x7_3d[1],  x7_3d[2],   
				   x7_3d[3],  x7_3d[4],  x7_3d[5],
				   x7_3d[6],  x7_3d[7],  x7_3d[8],
				   x7_3d[9],  x7_3d[10], x7_3d[11],
				   x7_3d[12], x7_3d[13], x7_3d[14],
				   x7_3d[15], x7_3d[16], x7_3d[17],
				   x7_3d[18], x7_3d[19], x7_3d[20],
				   x7_3d[21], x7_3d[22], x7_3d[23],
				   x7_3d[24], x7_3d[25], x7_3d[26],
				   x7_3d[27], x7_3d[28], x7_3d[29],
				   x7_3d[30], x7_3d[31], x7_3d[32],
				   x7_3d[33], x7_3d[34], x7_3d[35],
				   x7_3d[36], x7_3d[37], x7_3d[38],
				   x7_3d[39], x7_3d[40], x7_3d[41],
				   x7_3d[42], x7_3d[43], x7_3d[44],
				   x7_3d[45], x7_3d[46], x7_3d[47],
				   x7_3d[48], x7_3d[49], x7_3d[50],
				   x7_3d[51], x7_3d[52], x7_3d[53],
				   x7_3d[54], x7_3d[55], x7_3d[56],
				   x7_3d[57], x7_3d[58], x7_3d[59],
				   x7_3d[60], x7_3d[61], x7_3d[62],
				   x7_3d[63]};

static const double w7_3d[64] = {StdVol*0.0156807540, StdVol*0.0293976870,
				 StdVol*0.0293976870, StdVol*0.0156807540,
				 StdVol*0.0235447608, StdVol*0.0441408300,
				 StdVol*0.0441408300, StdVol*0.0235447608,
				 StdVol*0.0150258564, StdVol*0.0281699100,
				 StdVol*0.0281699100, StdVol*0.0150258564,
				 StdVol*0.0036082374, StdVol*0.0067645878,
				 StdVol*0.0067645878, StdVol*0.0036082374,
				 StdVol*0.0202865376, StdVol*0.0380324358,
				 StdVol*0.0380324358, StdVol*0.0202865376,
				 StdVol*0.0304603764, StdVol*0.0571059660,
				 StdVol*0.0571059660, StdVol*0.0304603764,
				 StdVol*0.0194392824, StdVol*0.0364440336,
				 StdVol*0.0364440336, StdVol*0.0194392824,
				 StdVol*0.0046680564, StdVol*0.0087514968,
				 StdVol*0.0087514968, StdVol*0.0046680564,
				 StdVol*0.0097055322, StdVol*0.0181955664,
				 StdVol*0.0181955664, StdVol*0.0097055322,
				 StdVol*0.0145729242, StdVol*0.0273207684,
				 StdVol*0.0273207684, StdVol*0.0145729242,
				 StdVol*0.0093001866, StdVol*0.0174356394,
				 StdVol*0.0174356394, StdVol*0.0093001866,
				 StdVol*0.0022333026, StdVol*0.0041869110,
				 StdVol*0.0041869110, StdVol*0.0022333026,
				 StdVol*0.0014639124, StdVol*0.0027444882,
				 StdVol*0.0027444882, StdVol*0.0014639124,
				 StdVol*0.0021980748, StdVol*0.0041208678,
				 StdVol*0.0041208678, StdVol*0.0021980748,
				 StdVol*0.0014027730, StdVol*0.0026298660,
				 StdVol*0.0026298660, StdVol*0.0014027730,
				 StdVol*0.0003368550, StdVol*0.0006315234,
				 StdVol*0.0006315234, StdVol*0.0003368550};

/*--------------------------------------------------------------------------*/
/*  build a vector of Quad' quadrature formulars. For quadrature of degree */
/*  use that of degree (only on function evaluation also)                   */
/*--------------------------------------------------------------------------*/

static QUAD quad_3d[MAX_QUAD_DEG_3d+1] = {
  {"3d Stroud: P_1", 1, 3,  1, gx1_3d, w1_3d},   /* P_0  */
  {"3d Stroud: P_1", 1, 3,  1, gx1_3d, w1_3d},   /* P_1  */
  {"3d Stroud: P_2", 2, 3,  4, gx2_3d, w2_3d},   /* P_2  */
  {"3d Stroud: P_3", 3, 3,  8, gx3_3d, w3_3d},   /* P_3  */
  {"3d ???: P_5", 5, 3, 15, gx5_3d, w5_3d},   /* P_4  */
  {"3d ???: P_5", 5, 3, 15, gx5_3d, w5_3d},   /* P_5  */
  {"3d ???: P_7", 7, 3, 64, gx7_3d, w7_3d},   /* P_6  */
  {"3d ???: P_7", 7, 3, 64, gx7_3d, w7_3d}    /* P_7  */
};

#undef StdVol

/*--------------------------------------------------------------------------*/
/*  integration in different dimensions                                     */
/*--------------------------------------------------------------------------*/

static QUAD        *quad_nd[4] = {quad_0d, quad_1d, quad_2d, quad_3d};
static const U_CHAR max_quad_deg[4] = {MAX_QUAD_DEG_0d, MAX_QUAD_DEG_1d, 
				       MAX_QUAD_DEG_2d, MAX_QUAD_DEG_3d};

const QUAD *get_quadrature(int dim, int degree)
{
  FUNCNAME("get_quadrature");

  TEST_EXIT((dim >= 0) && (dim < 4),
    "Bad dim %d - must be between 0 and 3!\n");

  degree = MAX(0, degree);

  if(degree > max_quad_deg[dim])
  {
    MSG("degree %d too large; changing to %d\n", degree, max_quad_deg[dim]);
    degree = max_quad_deg[dim];
  }

  return((const QUAD *) (quad_nd[dim] + degree));
}

/*---8<---------------------------------------------------------------------*/
/*---                                                                    ---*/
/*--------------------------------------------------------------------->8---*/

const QUAD *get_lumping_quadrature(int dim)
{
  FUNCNAME("get_lumping_quadrature");
  static const double lambda_0[N_LAMBDA] = {1.0, 0.0, 0.0, 0.0};
  static const double lambda_1[N_LAMBDA] = {0.0, 1.0, 0.0, 0.0};
  static const double lambda_2[N_LAMBDA] = {0.0, 0.0, 1.0, 0.0};
  static const double lambda_3[N_LAMBDA] = {0.0, 0.0, 0.0, 1.0};

#define StdVol 1.0
  static const double *lambda0[1] = {lambda_0};
  static const double weight0[1]  = {StdVol*1.0};

  static const double *lambda1[2] = {lambda_0, lambda_1};
  static const double weight1[2]  = {StdVol*0.5, StdVol*0.5};
#undef StdVol

#define StdVol 0.5
  static const double *lambda2[3] = {lambda_0, lambda_1, lambda_2};
  static const double weight2[3]  = {StdVol/3.0, StdVol/3.0, StdVol/3.0};
#undef StdVol

#define StdVol (1.0/6.0)
  static const double *lambda3[4] = {lambda_0, lambda_1, lambda_2, lambda_3};
  static const double weight3[4]  = {StdVol/4.0, StdVol/4.0,
				     StdVol/4.0, StdVol/4.0};
#undef StdVol
  
  static const QUAD lumping[4]={{ "lump0", 1, 0, 1, lambda0, weight0},
				{ "lump1", 1, 1, 2, lambda1, weight1},
				{ "lump2", 1, 2, 3, lambda2, weight2},
				{ "lump3", 1, 3, 4, lambda3, weight3}};

  TEST_EXIT(dim >= 0 && dim < 4,"invalid dim: %d\n", dim);
  
  return(lumping+dim);
}


/*--------------------------------------------------------------------------*/
/*  integrate f over reference simplex in R^(dim+1) 			    */
/*--------------------------------------------------------------------------*/


REAL integrate_std_simp(const QUAD *quad, REAL (*f)(const REAL *))
{
  FUNCNAME("integrate_std_simp");
  double   val;
  int      i;

  if (!quad || !f)
  {
    if (!quad) ERROR("quad is pointer to nil; return value is 0.0\n");
    if (!f) ERROR("f() is pointer to nil; return value is 0.0\n");
    return(0.0);
  }

  for (val = i = 0; i < quad->n_points; i++)
    val += quad->w[i]*(*f)(quad->lambda[i]);

  return(val);
}

/*--------------------------------------------------------------------------*/
/*  initialize values of a function f in local coordinates at the           */
/*  quadrature points                                                       */
/*--------------------------------------------------------------------------*/

const REAL *f_at_qp(const QUAD *quad, REAL (*f)(const REAL [N_LAMBDA]),
		    REAL *vec)
{
  FUNCNAME("f_at_qp");
  static REAL   *quad_vec = nil;
  static size_t size = 0;
  REAL          *val;
  int           i;
  
  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, quad->n_points);
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL);
      size = new_size;
    }
    val = quad_vec;
  }

  for (i = 0; i < quad->n_points; i++)
    val[i] = (*f)(quad->lambda[i]);
    
  return((const REAL *) val);
}

const REAL_D *f_d_at_qp(const QUAD *quad,
			const REAL *(*f)(const REAL[N_LAMBDA]),
			REAL_D *vec)
{
  FUNCNAME("f_d_at_qp");
  static REAL_D *quad_vec = nil;
  static size_t size = 0;
  REAL_D        *val;
  const REAL    *val_iq;
  int           iq, k;
  
  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, quad->n_points);
      quad_vec = MEM_REALLOC(quad_vec, size, new_size, REAL_D);
      size = new_size;
    }
    val = quad_vec;
  }

  for (iq = 0; iq < quad->n_points; iq++)
  {
    val_iq = (*f)(quad->lambda[iq]);
    for (k = 0; k < DIM_OF_WORLD; k++)
      val[iq][k] = val_iq[k];
  }    
  return((const REAL_D *) val);
}

/*--------------------------------------------------------------------------*/
/*  initialize gradient values of a function f in local coordinates at the  */
/*  quadrature points                                                       */
/*--------------------------------------------------------------------------*/

const REAL_D *grd_f_at_qp(const QUAD *quad, 
			  const REAL *(*f)(const REAL [N_LAMBDA]), REAL_D *vec)
{
  FUNCNAME("grd_f_at_qp");
  static REAL_D  *quad_vec_d = nil;
  static size_t  size = 0;
  REAL_D         *val;
  int            i, j;
  const REAL     *grd;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, quad->n_points);
      quad_vec_d = MEM_REALLOC(quad_vec_d, size, new_size, REAL_D);
      size = new_size;
    }
    val = quad_vec_d;
  }

  for (i = 0; i < quad->n_points; i++)
  {
    grd = (*f)(quad->lambda[i]);
    for (j = 0; j < DIM_OF_WORLD; j++)
      val[i][j] = grd[j];
  }
    
  return((const REAL_D *) val);
}

const REAL_DD *grd_f_d_at_qp(const QUAD *quad, 
			     const REAL_D *(*f)(const REAL [N_LAMBDA]),
			     REAL_DD *vec)
{
  FUNCNAME("grd_f_d_at_qp");
  static REAL_DD  *quad_vec_dd = nil;
  static size_t   size = 0;
  REAL_DD         *val;
  int             i, j, n;
  const REAL_D    *grd_d;

  if (vec)
  {
    val = vec;
  }
  else
  {
    if (size < (size_t) quad->n_points) 
    {
      size_t  new_size = MAX(MAX_N_QUAD_POINTS, quad->n_points);
      quad_vec_dd = MEM_REALLOC(quad_vec_dd, size, new_size, REAL_DD);
      size = new_size;
    }
    val = quad_vec_dd;
  }

  for (i = 0; i < quad->n_points; i++)
  {
    grd_d = (*f)(quad->lambda[i]);
    for (n = 0; n < DIM_OF_WORLD; n++)
      for (j = 0; j < DIM_OF_WORLD; j++)
	val[i][n][j] = grd_d[n][j];
  }
    
  return((const REAL_DD *) val);
}

struct all_fast
{
  const QUAD        *quad;
  const BAS_FCTS    *bas_fcts;
  QUAD_FAST         *quad_fast;
  struct all_fast   *next;
};

const QUAD_FAST *get_quad_fast(const BAS_FCTS *bas_fcts, const QUAD *quad,
			       U_CHAR init_flag)
{
  FUNCNAME("get_quad_fast");
  static struct all_fast  *first_fast = nil; /* anchor to the list  */
  struct all_fast *fast;
  QUAD_FAST       *quad_fast;
  int             i, j, k, l, n_points, n_bas_fcts;
  size_t          size, size_p;
  REAL            **g_phi;
  REAL            (**g_grd_phi)[N_LAMBDA];
  REAL            (**g_D2_phi)[N_LAMBDA][N_LAMBDA];
  const REAL      *lambda;
  void            *mem = nil, *mem_p = nil, *mem_last = nil;

  for (fast = first_fast; fast; fast = fast->next)
    if (fast->bas_fcts == bas_fcts && fast->quad == quad)  break;

  if (fast  &&  ((fast->quad_fast->init_flag & init_flag) == init_flag))
    return(fast->quad_fast);

  if (init_flag & INIT_PHI && !bas_fcts->phi)
  {
    ERROR("no functions for evaluating basis functions\n");
    ERROR("can not initialize quad_fast->phi\n");
    return(nil);
  }
  if (init_flag & INIT_GRD_PHI && !bas_fcts->grd_phi)
  {
    ERROR("no functions for evaluating gradients of basis functions\n");
    ERROR("can not initialize quad_fast->grd_phi\n");
    return(nil);
  }
  if (init_flag & INIT_D2_PHI && !bas_fcts->D2_phi)
  {
    ERROR("no functions for evaluating 2nd derivatives of basis functions\n");
    ERROR("can not initialize quad_fast->D2_phi\n");
    return(nil);
  }
  if (quad->dim != bas_fcts->dim)
  {
    ERROR("Basis function dim == %d does not match quadrature dim == %d!\n", bas_fcts->dim, quad->dim);
    return(nil);
  }

  if (!fast)
  {
    quad_fast = MEM_ALLOC(1, QUAD_FAST);
    quad_fast->quad = quad;
    quad_fast->bas_fcts = bas_fcts;
    quad_fast->n_points = quad->n_points;
    quad_fast->n_bas_fcts = bas_fcts->n_bas_fcts;
    quad_fast->w = quad->w;
    quad_fast->init_flag = 0;
    quad_fast->phi = nil;
    quad_fast->grd_phi = nil;
    quad_fast->D2_phi = nil;

    fast = MEM_ALLOC(1, struct all_fast);
    fast->quad      = quad;
    fast->bas_fcts  = bas_fcts;
    fast->quad_fast = quad_fast;
    fast->next      = first_fast;
    first_fast      = fast;
  }
  else
    quad_fast = fast->quad_fast;

  n_points = quad->n_points;
  n_bas_fcts = bas_fcts->n_bas_fcts;

  size = size_p = 0;
  if (!quad_fast->phi  &&  (init_flag & INIT_PHI))
  {
    size +=  n_points*n_bas_fcts*sizeof(REAL);
    size_p += n_points*sizeof(REAL *);
  }
  if (!quad_fast->grd_phi  &&  (init_flag & INIT_GRD_PHI))
  {
    size += n_points*n_bas_fcts*sizeof(REAL [N_LAMBDA]); 
    size_p += n_points*sizeof(REAL (*)[N_LAMBDA]);
  }
  if (!quad_fast->D2_phi  &&  (init_flag & INIT_D2_PHI))
  {
    size += n_points*n_bas_fcts*sizeof(REAL [N_LAMBDA][N_LAMBDA]);
    size_p += n_points*sizeof(REAL (*)[N_LAMBDA][N_LAMBDA]);
  }

  if (size)
  {
    mem = alberta_alloc(size, funcName, __FILE__,  __LINE__);
    mem_last = (void *) (((char *) mem)+size);
    mem_p = alberta_alloc(size_p, funcName, __FILE__,  __LINE__);

    if (!quad_fast->phi  &&  (init_flag & INIT_PHI))
    {
      BAS_FCT    **phi = bas_fcts->phi;
      
      g_phi = quad_fast->phi = (REAL **) mem_p;
      mem_p = (void *) (g_phi+n_points);
      for (i = 0; i < n_points; i++)
      {
	lambda = quad->lambda[i];
	g_phi[i] = (REAL *) mem;
	mem = (void *) (g_phi[i]+n_bas_fcts);
	for (j = 0; j < n_bas_fcts; j++)
	{
	  g_phi[i][j] = phi[j](lambda);
	}
      }

      quad_fast->init_flag |= INIT_PHI;
    }

    if (!quad_fast->grd_phi  && (init_flag & INIT_GRD_PHI))
    {
      GRD_BAS_FCT  **grd_phi = bas_fcts->grd_phi;
      const REAL   *grd;

      g_grd_phi = quad_fast->grd_phi  = (REAL (**)[N_LAMBDA]) mem_p;
      mem_p = (void *) (quad_fast->grd_phi+n_points);

      for (i = 0; i < n_points; i++)
      { 
	lambda = quad->lambda[i];
	g_grd_phi[i] = (REAL (*)[N_LAMBDA]) mem;
	mem = (void *) (g_grd_phi[i]+n_bas_fcts);

	for (j = 0; j < n_bas_fcts; j++)
	{
	  grd = grd_phi[j](lambda);
	  for (k = 0; k < N_LAMBDA; k++)
	  {
	    g_grd_phi[i][j][k] = grd[k];
	  }	    
	}
      }
      quad_fast->init_flag |= INIT_GRD_PHI;
    }

    if (!quad_fast->D2_phi  && (init_flag & INIT_D2_PHI))
    {
      D2_BAS_FCT   **D2_phi = bas_fcts->D2_phi;
      const REAL   (*D2)[N_LAMBDA];

      g_D2_phi = quad_fast->D2_phi  = (REAL (**)[N_LAMBDA][N_LAMBDA]) mem_p;
      mem_p = (void *) (quad_fast->D2_phi+n_points);

      for (i = 0; i < n_points; i++)
      {
	lambda = quad->lambda[i];
	g_D2_phi[i] = (REAL (*)[N_LAMBDA][N_LAMBDA]) mem;
	mem = (void *) (g_D2_phi[i]+n_bas_fcts);
	for (j = 0; j < n_bas_fcts; j++)
	{
	  D2 = D2_phi[j](lambda);
	  for (k = 0; k < N_LAMBDA; k++)
	    for (l = 0; l < N_LAMBDA; l++)
	      g_D2_phi[i][j][k][l] = D2[k][l];
	}
      }
      quad_fast->init_flag |= INIT_D2_PHI;
    }
  }

  if (mem > mem_last)
    ERROR_EXIT("allocated size too small\n");
  if (mem < mem_last)
    ERROR_EXIT("allocated size too big\n");

  if ((quad_fast->init_flag & init_flag) != init_flag)
  {
    ERROR("could not initialize quad_fast, returning pointer to nil\n");
    return(nil);
  }

  return(quad_fast);
}

void print_quadrature(const QUAD *quad)
{
  FUNCNAME("print_quadrature");
  int      i, j;

  MSG("quadrature %s for dimension %d exact on P_%d\n", 
      quad->name, quad->dim, quad->degree);
  MSG("%d points with weights and quadrature points:\n", quad->n_points);

  for (i = 0; i < quad->n_points; i++)
  {
    MSG("w[%2d] = %le, lambda[%2d] = (", i, quad->w[i], i);
    for (j = 0; j <= quad->dim; j++)
      print_msg("%le%s", quad->lambda[i][j], j < quad->dim ? ", " : ")\n");
  }
  return;
}

