/*
 * Copyright (C) 2017 Google, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.google.auto.value.processor;

import com.google.testing.compile.CompilationRule;
import javax.lang.model.util.Elements;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;

/**
 * Test rule that allows compile-testing tests to know which {@code @Generated} annotation is
 * available.
 */
final class GeneratedImportRule implements TestRule {

  private final CompilationRule compilationRule = new CompilationRule();

  /**
   * Returns the qualified name of the {@code @Generated} annotation available during a compilation
   * task.
   */
  String generatedAnnotationType() {
    Elements elements = compilationRule.getElements();
    return elements.getTypeElement("javax.annotation.processing.Generated") != null
        ? "javax.annotation.processing.Generated"
        : "javax.annotation.Generated";
  }

  /**
   * Returns an {@code import} statement that imports the {@code @Generated} annotation {@linkplain
   * #generatedAnnotationType() available during a compilation task}.
   */
  String importGeneratedAnnotationType() {
    return "import " + generatedAnnotationType() + ";";
  }

  @Override
  public Statement apply(Statement base, Description description) {
    return compilationRule.apply(base, description);
  }
}
