/*==========================================================================
  ===== general functions, type arrays and matrices ======================== */
extern void free(void*);

// typemap to convert a python list of integers to array of ints
%typemap(in) (int *pylist, size_t length) {
  int i;
  if (PySequence_Check($input)) {
    $2 = PySequence_Size($input);
    $1 = int_array_alloc($2);
    if (!$1) {
      PyErr_SetString(PyExc_TypeError,"Could not allocate a int_array with $2 entries");
      return NULL;
    }     
    for (i = 0; i < $2; i++) {
      PyObject *o = PySequence_GetItem($input,i);
      if (!PyInt_Check(o)) {
         Py_XDECREF(o);
         PyErr_SetString(PyExc_ValueError,"Expecting a sequence of integers");
         free($1);
         return NULL;
      }
      $1[i] = (int)PyInt_AsLong(o);
      Py_DECREF(o);
    }
  }
  else {
    PyErr_SetString(PyExc_TypeError,"Expecting a sequence");
    return NULL;
  }
}

// ignore the input value for int array to python list conversion
%typemap(in, numinputs=0) size_t *int_array_length (size_t temp) {
    $1 = &temp;
}
// convert array of ints to python list
%typemap(argout) (size_t *int_array_length) {
    int i;
    Py_XDECREF($result);   /* Blow away any previous result */
    if (result) {
        $result = PyList_New(*$1);
        for (i=0; i<*$1; i++) {
            PyList_SetItem($result, i, PyInt_FromLong(result[i]));
        }
    }
    else {
        PyErr_SetString(PyExc_ValueError,"got a null pointer");
        return NULL;
    }
}

// typemap to convert a python list of floats to array of doubles
%typemap(in) (double *pylist, size_t length) {
  int i;
  if (PySequence_Check($input)) {
    $2 = PySequence_Size($input);
    $1 = double_array_alloc($2);
    if (!$1) {
      PyErr_SetString(PyExc_TypeError,"Could not allocate a double_array with $2 entries");
      return NULL;
    }
    for (i = 0; i < $2; i++) {
        PyObject *o = PySequence_GetItem($input,i);
        if (PyFloat_Check(o)) {
            $1[i] = PyFloat_AsDouble(o);
            Py_DECREF(o);
        }
        // conversion from int to double is without loss
        else if (PyInt_Check(o)) {
            $1[i] = (double)PyInt_AsLong(o);
            Py_DECREF(o);
        }
        else {
            Py_XDECREF(o);
            PyErr_SetString(PyExc_ValueError,"Expecting a sequence of numbers");
            free($1);
            return NULL;
        }
    }
  }
  else {
    PyErr_SetString(PyExc_TypeError,"Expecting a sequence");
    return NULL;
  }
}

// ignore the input value for double array to python list conversion
%typemap(in, numinputs=0) size_t *double_array_length (size_t temp) {
    $1 = &temp;
}
// convert array of doubles to python list
%typemap(argout) (size_t *double_array_length) {
    int i;
    Py_XDECREF($result);   /* Blow away any previous result */
    if (result) {
        $result = PyList_New(*$1);
        for (i=0; i<*$1; i++) {
            PyList_SetItem($result, i, PyFloat_FromDouble(result[i]));
        }
    }
    else {
        PyErr_SetString(PyExc_ValueError,"got a null pointer");
        return NULL;
    }
}

%define ARRAY(type)
%inline %{
        type* type ## _array_alloc(size_t length) { return malloc(length*sizeof(type)); }
        type  type ## _array_getitem(type* self, size_t index) { return self[index]; }
        void  type ## _array_setitem(type* self, size_t index, type value) { self[index] = value; }
/*
        type* list2 ## type ## _array(type* pylist, size_t length) { return pylist; }
        type* type ## _array2list(type* array, size_t length, size_t* type ## _array_lentgh) {
            *type ## _array_lentgh  = length;
            return array;
        }
*/
%}
%enddef

ARRAY(int)
ARRAY(long)
ARRAY(double)

%inline %{
    int *list2int_array(int *pylist, size_t length) { return pylist; }
    int *int_array2list(int *array, size_t length, size_t *int_array_length) {
        *int_array_length = length;
        return array;
    }
    double *list2double_array(double *pylist, size_t length) { return pylist; }
    double *double_array2list(double *array, size_t length, size_t *double_array_length) {
        *double_array_length = length;
        return array;
    }
%}


%define MATRIX(type)
%inline %{
        type** type ## _matrix_alloc(size_t rows, size_t cols)
            {
                int i;
                type** mat = malloc(rows * sizeof(type*));
                for (i=0; i<rows; ++i) mat[i] = malloc(cols * sizeof(type));
                return mat;
            }
        type** type ## _matrix_alloc_row(size_t rows) { return malloc(rows * sizeof(type*)); }
        void   type ## _matrix_free(type** mat, size_t rows)
            {
                int i;
                for (i=0; i<rows; ++i) free(mat[i]);
                free(mat);
            }
        type*  type ## _matrix_get_col(type** self, size_t index) { return self[index]; }
        void   type ## _matrix_set_col(type** self, size_t index, type* col) { self[index] = col; }
        type   type ## _matrix_getitem(type** self, size_t row, size_t col) { return self[row][col]; }
        void   type ## _matrix_setitem(type** self, size_t row, size_t col, type value) { self[row][col] = value; }
%}
%enddef
MATRIX(int)
MATRIX(double)