//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Descriptor/AxisProperty.cpp
//! @brief     Implements class AxisProperty
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2022
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Descriptor/AxisProperty.h"
#include "Base/Axis/Scale.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString Nbins("Nbins");
const QString Min("Min");
const QString Max("Max");
const QString ExpandGroupbox("ExpandGroupbox");

} // namespace Tag
} // namespace

using std::variant;

AxisProperty::AxisProperty() = default;

void AxisProperty::initMin(const QString& label, const QString& tooltip, double value,
                           const variant<QString, Unit>& unit, const RealLimits& limits,
                           uint decimals)
{
    m_min.init(label, tooltip, value, unit, decimals, limits, "min");
}

void AxisProperty::initMax(const QString& label, const QString& tooltip, double value,
                           const variant<QString, Unit>& unit, const RealLimits& limits,
                           uint decimals)
{
    m_max.init(label, tooltip, value, unit, decimals, limits, "max");
}

std::unique_ptr<Scale> AxisProperty::createAxis(double scaleFactor) const
{
    return uniqueEquiDivision(std::string(), m_nbins, m_min * scaleFactor, m_max * scaleFactor);
}

void AxisProperty::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // nbins
    w->writeStartElement(Tag::Nbins);
    XML::writeAttribute(w, XML::Attrib::value, m_nbins);
    w->writeEndElement();

    // min
    w->writeStartElement(Tag::Min);
    m_min.writeTo(w);
    w->writeEndElement();

    // max
    w->writeStartElement(Tag::Max);
    m_max.writeTo(w);
    w->writeEndElement();

    // groupbox: is expanded?
    w->writeStartElement(Tag::ExpandGroupbox);
    XML::writeAttribute(w, XML::Attrib::value, m_expandGroupBox);
    w->writeEndElement();
}

void AxisProperty::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // nbins
        if (tag == Tag::Nbins) {
            XML::readAttribute(r, XML::Attrib::value, &m_nbins);
            XML::gotoEndElementOfTag(r, tag);

            // min
        } else if (tag == Tag::Min) {
            m_min.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // max
        } else if (tag == Tag::Max) {
            m_max.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // groupbox: is expanded?
        } else if (tag == Tag::ExpandGroupbox) {
            XML::readAttribute(r, XML::Attrib::value, &m_expandGroupBox);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}
