# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from gobject import idle_add
from re import compile as re_compile
from os.path import join
from os import walk
import tagpy

from common.config import ConfigLoader
from common.sqlite import SQLite

class MusicDatabase:
    def __init__(self, config, pbar):
        self.config = config
        self.pbar = pbar
        self.confl = ConfigLoader()

        self.stored_result = []

    def do_scan(self):
        """This function scan the music directory."""
        songs = list()
        songs_filename = list()
        songs_artists = list()
        songs_albums = list()
        nb_song = 0
        fileregxp = re_compile('.+\.(flac|ogg|oga|mp3|mpc)$')

        # Walk in the music folder
        if self.config['folder'] is None:
            folder = ''
        else:
            folder = self.config['folder']

        song_files = list()
        for (dir_, rep, files) in walk(folder):
            for file_ in files:
                if fileregxp.match(file_):
                    nb_song += 1
                    song_files.append(join(dir_, file_))

        self.pbar.set_fraction(0)

        id_song = 0
        for song in song_files:
            id_song += 1

            filename = song
            tagpy_file = tagpy.FileRef(filename)
            tags = tagpy_file.tag()
            audio = tagpy_file.audioProperties()

            try:
                title = tags.title
                artist = tags.artist
                album = tags.album
                comment = tags.comment
                genre = tags.genre
                year = tags.year
                track = tags.track
                length = audio.length
            except AttributeError:
                continue

            if id_song % 5:
                self.pbar.set_fraction(float(id_song) / float(nb_song))

            self.pbar.set_text(_('Added %d songs') % id_song)

            songs_filename.append(filename)
            songs_artists.append(artist)
            songs_albums.append(album)

            songs.append((title, artist, album, comment, genre,
                          year, track, length, filename))

        # Serialize songs
        sqlite = SQLite()
        sqlite.execute('delete from songs')
        sqlite.executemany('insert into songs (title, artist, album, '
                           'comment, genre, year, track, length, filename) '
                           'values (?, ?, ?, ?, ?, ?, ?, ?, ?)', songs)

        # Update statistics
        self.pbar.set_text(_('Update statistics…'))

        # Update songs
        cursor = sqlite.execute('select * from stats_songs')
        for song in cursor.fetchall():
            filename = song[0]

            if filename not in songs_filename:
                # Delete this songs from statistics
                sqlite.execute('delete from stats_songs where filename=:val',
                               {'val': filename})

        # Update artists
        cursor = sqlite.execute('select * from stats_artists')
        for artist in cursor.fetchall():
            artist = artist[0]

            if artist not in songs_artists:
                # Delete this artist from statistics
                sqlite.execute('delete from stats_artists where artist=:val',
                               {'val': artist})

        # Update albums
        cursor = sqlite.execute('select * from stats_albums')
        for album in cursor.fetchall():
            album = album[0]

            if album not in songs_albums:
                # Delete this album from statistics
                sqlite.execute('delete from stats_albums where album=:val',
                               {'val': album})

        # The job is ended o/
        sqlite.close()

    def scan(self):
        self.do_scan()

    def load(self, force_reload=False):
        if not force_reload and len(self.stored_result) > 0:
            # Return the already loaded songs
            return self.stored_result
        else:
            # Load the songs
            sqlite = SQLite()
            cursor = sqlite.execute('select * from songs order by '
                                    'artist, album, title')
            songs = cursor.fetchall()
            sqlite.close()

            return songs