#ifndef SCATT_H
#define SCATT_H 1
#include <math.h>
#include <QString>
#include "molecule.h"

// lab,rad,mass,a1,b1,a2,b2,a3,b3,a4,b4,c,f'cu,f''cu,mucu,f'mo,f''mo,mumo,f'ag,f''ag,muag

struct Atomformfactor{
  char symb[3];
  float rad,mass;
  float ab4c[9];
  float fpcu,fppcu,mucu,fpmo,fppmo,mumo,fpag,fppag,muag;
};

Atomformfactor ff[94]= {//copy paste from shelxl.f 97
  {"H",   .32f,   1.008f,{ 0.49300f, 10.51091f,  0.32291f, 26.12573f,  0.14019f,   3.14236f, 0.04081f,  57.79977f,  0.00304f},  0.0000f,  0.0000f,    .655f,   0.000f, 0.000f,  .624f,  0.000f, 0.000f,  .614f },
  {"HE", 1.50f,    4.00f,{ 0.87340f,  9.10371f,  0.63090f,  3.35680f,  0.31120f,  22.92763f, 0.17800f,   0.98210f,  0.00640f},  0.0000f,  0.0000f,    1.94f,   0.000f, 0.000f,  1.34f,  0.000f, 0.000f,  1.28f },
  {"LI", 1.52f,    6.94f,{ 1.12820f,  3.95460f,  0.75080f,  1.05240f,  0.61750f,  85.39058f, 0.46530f, 168.26120f,  0.03770f},  0.0008f,  0.0003f,    5.76f, -0.0003f, 0.0001f,  2.28f, -0.0004f, 0.0000f,  2.06f },
  {"BE", 1.11f,    9.01f,{ 1.59190f, 43.64275f,  1.12780f,  1.86230f,  0.53910f, 103.48310f, 0.70290f,   0.54200f,  0.03850f},  0.0038f,  0.0014f,    16.6f,  0.0005f, 0.0002f,  3.83f,  0.0001f, 0.0001f,  3.13f },
  {"B",  0.82f,   10.81f,{ 2.05450f, 23.21852f,  1.33260f,  1.02100f,  1.09790f,  60.34987f, 0.70680f,   0.14030f, -0.19320f},  0.0090f,  0.0039f,    41.5f,  0.0013f, 0.0007f,  6.61f,  0.0004f, 0.0004f,  4.79f },
  {"C",  0.77f,   12.01f,{ 2.31000f, 20.84392f,  1.02000f, 10.20751f,  1.58860f,   0.56870f, 0.86500f,  51.65125f,  0.21560f},  0.0181f,  0.0091f,    89.9f,  0.0033f, 0.0016f,  11.5f,  0.0015f, 0.0009f,  7.45f },
  {"N",  0.70f,   14.01f,{12.21261f,  0.00570f,  3.13220f,  9.89331f,  2.01250f,  28.99754f, 1.16630f,   0.58260f,-11.52901f},  0.0311f,  0.0180f,    173.f,  0.0061f, 0.0033f,  19.6f,  0.0030f, 0.0019f,  11.7f },
  {"O",  0.66f,   16.00f,{ 3.04850f, 13.27711f,  2.28680f,  5.70111f,  1.54630f,   0.32390f, 0.86700f,  32.90894f,  0.25080f},  0.0492f,  0.0322f,    304.f,  0.0106f, 0.0060f,  32.5f,  0.0056f, 0.0036f,  18.2f },
  {"F",  0.64f,   19.00f,{ 3.53920f, 10.28251f,  2.64120f,  4.29440f,  1.51700f,   0.26150f, 1.02430f,  26.14763f,  0.27760f},  0.0727f,  0.0534f,    498.f,  0.0171f, 0.0103f,  51.5f,  0.0096f, 0.0061f,  27.7f },
  {"NE", 1.50f,   20.18f,{ 3.95530f,  8.40421f,  3.11250f,  3.42620f,  1.45460f,   0.23060f, 1.12510f,  21.71841f,  0.35150f},  0.1019f,  0.0833f,    768.f,  0.0259f, 0.0164f,  78.6f,  0.0152f, 0.0098f,  41.2f },
  {"NA", 1.86f,   22.99f,{ 4.76260f,  3.28500f,  3.17360f,  8.84221f,  1.26740f,   0.31360f, 1.11280f, 129.42410f,  0.67600f},  0.1353f,  0.1239f,   1140.f,  0.0362f, 0.0249f,  116.f,  0.0218f, 0.0150f,  59.6f },
  {"MG", 1.60f,   24.31f,{ 5.42041f,  2.82750f,  2.17350f, 79.26118f,  1.22690f,   0.38080f, 2.30730f,   7.19371f,  0.85840f},  0.1719f,  0.1771f,   1610.f,  0.0486f, 0.0363f,  165.f,  0.0298f, 0.0220f,  84.2f },
  {"AL", 1.25f,   26.98f,{ 6.42021f,  3.03870f,  1.90020f,  0.74260f,  1.59360f,  31.54724f, 1.96460f,  85.08868f,  1.11510f},  0.2130f,  0.2455f,   2220.f,  0.0645f, 0.0514f,  229.f,  0.0406f, 0.0313f,  116.f },
  {"SI", 1.17f,   28.09f,{ 6.29151f,  2.43860f,  3.03530f, 32.33374f,  1.98910f,   0.67850f, 1.54100f,  81.69379f,  1.14070f},  0.2541f,  0.3302f,   2970.f,  0.0817f, 0.0704f,  310.f,  0.0522f, 0.0431f,  156.f },
  {"P",  1.10f,   30.97f,{ 6.43451f,  1.90670f,  4.17910f, 27.15704f,  1.78000f,   0.52600f, 1.49080f,  68.16457f,  1.11490f},  0.2955f,  0.4335f,   3880.f,  0.1023f, 0.0942f,  410.f,  0.0667f, 0.0580f,  206.f },
  {"S",  1.03f,   32.06f,{ 6.90531f,  1.46790f,  5.20341f, 22.21512f,  1.43790f,   0.25360f, 1.58630f,  56.17207f,  0.86690f},  0.3331f,  0.5567f,   4970.f,  0.1246f, 0.1234f,  532.f,  0.0826f, 0.0763f,  267.f },
  {"CL", 0.99f,   35.45f,{11.46041f,  0.01040f,  7.19641f,  1.16620f,  6.25561f,  18.51942f, 1.64550f,  47.77846f, -9.55741f},  0.3639f,  0.7018f,   6240.f,  0.1484f, 0.1585f,  678.f,  0.0998f, 0.0984f,  341.f },
  {"AR", 1.50f,   39.95f,{ 7.48451f,  0.90720f,  6.77231f, 14.84071f,  0.65390f,  43.89835f, 1.64420f,  33.39293f,  1.44450f},  0.3843f,  0.8717f,   7720.f,  0.1743f, 0.2003f,  851.f,  0.1191f, 0.1249f,  429.f },
  {"K",  2.27f,   39.10f,{ 8.21861f, 12.79491f,  7.43981f,  0.77480f,  1.05190f, 213.18720f, 0.86590f,  41.68416f,  1.42280f},  0.3868f,  1.0657f,   9400.f,  0.2009f, 0.2494f, 1050.f,  0.1399f, 0.1562f,  532.f },
  {"CA", 1.97f,   40.08f,{ 8.62661f, 10.44211f,  7.38731f,  0.65990f,  1.58990f,  85.74849f, 1.02110f, 178.43720f,  1.37510f},  0.3641f,  1.2855f,  11300.f,  0.2262f, 0.3064f, 1290.f,  0.1611f, 0.1926f,  652.f },
  {"SC", 1.61f,   44.96f,{ 9.18901f,  9.02131f,  7.36791f,  0.57290f,  1.64090f, 136.10810f, 1.46800f,  51.35315f,  1.33290f},  0.3119f,  1.5331f,  13500.f,  0.2519f, 0.3716f, 1560.f,  0.1829f, 0.2348f,  789.f },
  {"TI", 1.45f,   47.90f,{ 9.75951f,  7.85081f,  7.35581f,  0.50000f,  1.69910f,  35.63383f, 1.90210f, 116.10510f,  1.28070f},  0.2191f,  1.8069f,  15900.f,  0.2776f, 0.4457f, 1860.f,  0.2060f, 0.2830f,  947.f },
  {"V",  1.31f,   50.94f,{10.29711f,  6.86571f,  7.35111f,  0.43850f,  2.07030f,  26.89383f, 2.05710f, 102.47810f,  1.21990f},  0.0687f,  2.1097f,  18500.f,  0.3005f, 0.5294f, 2200.f,  0.2276f, 0.3376f, 1120.f },
  {"CR", 1.24f,   52.00f,{10.64061f,  6.10381f,  7.35371f,  0.39200f,  3.32400f,  20.26262f, 1.49220f,  98.73999f,  1.18320f}, -0.1635f,  2.4439f,  21300.f,  0.3209f, 0.6236f, 2580.f,  0.2496f, 0.3992f, 1330.f },
  {"MN", 1.37f,   54.94f,{11.28191f,  5.34091f,  7.35731f,  0.34320f,  3.01930f,  17.86742f, 2.24410f,  83.75438f,  1.08960f}, -0.5299f,  2.8052f,  24600.f,  0.3368f, 0.7283f, 3020.f,  0.2704f, 0.4681f, 1550.f },
  {"FE", 1.24f,   55.85f,{11.76951f,  4.76111f,  7.35731f,  0.30720f,  3.52220f,  15.35351f, 2.30450f,  76.88058f,  1.03690f}, -1.1336f,  3.1974f,  28000.f,  0.3463f, 0.8444f, 3490.f,  0.2886f, 0.5448f, 1800.f },
  {"CO", 1.25f,   58.93f,{12.28411f,  4.27910f,  7.34091f,  0.27840f,  4.00340f,  13.53591f, 2.34880f,  71.16927f,  1.01180f}, -2.3653f,  3.6143f,  31400.f,  0.3494f, 0.9721f, 4010.f,  0.3050f, 0.6296f, 2070.f },
  {"NI", 1.25f,   58.71f,{12.83761f,  3.87850f,  7.29201f,  0.25650f,  4.44380f,  12.17631f, 2.38000f,  66.34216f,  1.03410f}, -3.0029f,  0.5091f,   4760.f,  0.3393f, 1.1124f, 4570.f,  0.3147f, 0.7232f, 2380.f },
  {"CU", 1.28f,   63.54f,{13.33801f,  3.58280f,  7.16761f,  0.24700f,  5.61581f,  11.39661f, 1.67350f,  64.81267f,  1.19100f}, -1.9646f,  0.5888f,   5470.f,  0.3201f, 1.2651f, 5180.f,  0.3240f, 0.8257f, 2710.f },
  {"ZN", 1.33f,   65.37f,{14.07431f,  3.26550f,  7.03181f,  0.23330f,  5.16521f,  10.31631f, 2.41000f,  58.70976f,  1.30410f}, -1.5491f,  0.6778f,   6290.f,  0.2839f, 1.4301f, 5860.f,  0.3242f, 0.9375f, 3070.f },
  {"GA", 1.26f,   69.72f,{15.23541f,  3.06690f,  6.70061f,  0.24120f,  4.35910f,  10.78051f, 2.96230f,  61.41357f,  1.71890f}, -1.2846f,  0.7763f,   7190.f,  0.2307f, 1.6083f, 6600.f,  0.3179f, 1.0589f, 3460.f },
  {"GE", 1.22f,   72.59f,{16.08162f,  2.85090f,  6.37471f,  0.25160f,  3.70680f,  11.44681f, 3.68300f,  54.76256f,  2.13130f}, -1.0885f,  0.8855f,   8190.f,  0.1547f, 1.8001f, 7380.f,  0.3016f, 1.1903f, 3870.f },
  {"AS", 1.21f,   74.92f,{16.67232f,  2.63450f,  6.07011f,  0.26470f,  3.43130f,  12.94791f, 4.27790f,  47.79726f,  2.53100f}, -0.9300f,  1.0051f,   9290.f,  0.0499f, 2.0058f, 8220.f,  0.2758f, 1.3314f, 4330.f },
  {"SE", 1.17f,   78.96f,{17.00063f,  2.40980f,  5.81961f,  0.27260f,  3.97310f,  15.23721f, 4.35430f,  43.81635f,  2.84090f}, -0.7943f,  1.1372f,  10500.f, -0.0929f, 2.2259f, 9110.f,  0.2367f, 1.4831f, 4820.f },
  {"BR", 1.14f,   79.91f,{17.17892f,  2.17230f,  5.23581f, 16.57962f,  5.63771f,   0.26090f, 3.98510f,  41.43285f,  2.95570f}, -0.6763f,  1.2805f,  11800.f, -0.2901f, 2.4595f,10000.f,  0.1811f, 1.6452f, 5350.f },
  {"KR", 1.50f,   83.80f,{17.35551f,  1.93840f,  6.72861f, 16.56232f,  5.54931f,   0.22610f, 3.53750f,  39.39723f,  2.82500f}, -0.5657f,  1.4385f,  13200.f, -0.5574f, 2.7079f,11000.f,  0.1067f, 1.8192f, 5920.f },
  {"RB", 2.48f,   85.47f,{17.17842f,  1.78880f,  9.64351f, 17.31512f,  5.13990f,   0.27480f, 1.52920f, 164.93420f,  3.48730f}, -0.4688f,  1.6079f,  14800.f, -0.9393f, 2.9676f,12100.f,  0.0068f, 2.0025f, 6520.f },
  {"SR", 2.15f,   87.62f,{17.56631f,  1.55640f,  9.81841f, 14.09881f,  5.42200f,   0.16640f, 2.66940f, 132.37610f,  2.50640f}, -0.3528f,  1.8200f,  16500.f, -1.5307f, 3.2498f,13200.f, -0.1172f, 2.2025f, 7150.f },
  {"Y",  1.78f,   88.91f,{17.77602f,  1.40290f, 10.29461f, 12.80061f,  5.72630f,   0.12560f, 3.26588f, 104.35410f,  1.91213f}, -0.2670f,  2.0244f,  18300.f, -2.7962f, 3.5667f,14300.f, -0.2879f, 2.4099f, 7800.f },
  {"ZR", 1.59f,   91.22f,{17.87653f,  1.27618f, 10.94801f, 11.91601f,  5.41733f,   0.11762f, 3.65721f,  87.66278f,  2.06929f}, -0.1862f,  2.2449f,  20300.f, -2.9673f, 0.5597f, 2470.f, -0.5364f, 2.6141f, 8470.f },
  {"NB", 1.43f,   92.91f,{17.61423f,  1.18865f, 12.01441f, 11.76601f,  4.04183f,   0.20479f, 3.53346f,  69.79576f,  3.75591f}, -0.1121f,  2.4826f,  22300.f, -2.0727f, 0.6215f, 2730.f, -0.8282f, 2.8404f, 9220.f },
  {"MO", 1.36f,   95.94f,{ 3.70250f,  0.27720f, 17.23563f,  1.09580f, 12.88761f,  11.00401f, 3.74290f,  61.65846f,  4.38750f}, -0.0483f,  2.7339f,  24600.f, -1.6832f, 0.6857f, 3000.f, -1.2703f, 3.0978f, 11500.f },
  {"TC", 1.35f,   98.00f,{19.13013f,  0.86413f, 11.09481f,  8.14488f,  4.64902f,  21.57072f, 2.71263f,  86.84727f,  5.40429f},  0.0057f,  3.0049f,  27000.f, -1.4390f, 0.7593f, 3320.f, -2.0087f, 3.3490f, 10700.f },
  {"RU", 1.33f,  101.07f,{19.26743f,  0.80852f, 12.91821f,  8.43468f,  4.86337f,  24.79974f, 1.56756f,  94.29289f,  5.37875f},  0.0552f,  3.2960f,  29500.f, -1.2594f, 0.8363f, 3640.f, -5.3630f, 3.6506f, 1920.f },
  {"RH", 1.35f,  102.91f,{19.29572f,  0.75154f, 14.35011f,  8.21759f,  4.73425f,  25.87494f, 1.28918f,  98.60629f,  5.32800f},  0.0927f,  3.6045f,  32300.f, -1.1178f, 0.9187f, 3990.f, -2.5280f, 0.5964f, 2100.f },
  {"PD", 1.38f,  106.40f,{19.33192f,  0.69866f, 15.50172f,  7.98930f,  5.29537f,  25.20523f, 0.60584f,  76.89868f,  5.26593f},  0.1215f,  3.9337f,  35200.f, -0.9988f, 1.0072f, 4360.f, -1.9556f, 0.6546f, 2300.f },
  {"AG", 1.44f,  107.87f,{19.28082f,  0.64460f, 16.68852f,  7.47261f,  4.80451f,  24.66054f, 1.04630f,  99.81570f,  5.17900f},  0.1306f,  4.2820f,  38200.f, -0.8971f, 1.1015f, 4760.f, -1.6473f, 0.7167f, 2510.f },
  {"CD", 1.49f,  112.40f,{19.22142f,  0.59460f, 17.64442f,  6.90891f,  4.46100f,  24.70084f, 1.60290f,  87.48257f,  5.06941f},  0.1185f,  4.6533f,  41500.f, -0.8075f, 1.2024f, 5180.f, -1.4396f, 0.7832f, 2730.f },
  {"IN", 1.44f,  114.82f,{19.16241f,  0.54760f, 18.55962f,  6.37761f,  4.29480f,  25.84993f, 2.03960f,  92.80299f,  4.93911f},  0.0822f,  5.0449f,  45000.f, -0.7276f, 1.3100f, 5630.f, -1.2843f, 0.8542f, 2970.f },
  {"SN", 1.40f,  118.69f,{19.18892f,  5.83031f, 19.10052f,  0.50310f,  4.45850f,  26.89093f, 2.46630f,  83.95718f,  4.78211f},  0.0259f,  5.4591f,  48600.f, -0.6537f, 1.4246f, 6110.f, -1.1587f, 0.9299f, 3230.f },
  {"SB", 1.41f,  121.75f,{19.64182f,  5.30340f, 19.04552f,  0.46070f,  5.03711f,  27.90744f, 2.68270f,  75.28258f,  4.59091f}, -0.0562f,  5.8946f,  52500.f, -0.5866f, 1.5461f, 6620.f, -1.0547f, 1.0104f, 3500.f },
  {"TE", 1.37f,  127.60f,{19.96442f,  4.81742f, 19.01382f,  0.42089f,  6.14488f,  28.52844f, 2.52390f,  70.84036f,  4.35200f}, -0.1759f,  6.3531f,  56500.f, -0.5308f, 1.6751f, 7160.f, -0.9710f, 1.0960f, 3780.f },
  {"I",  1.33f,  126.90f,{20.14722f,  4.34700f, 18.99492f,  0.38140f,  7.51381f,  27.76604f, 2.27350f,  66.87767f,  4.07120f}, -0.3257f,  6.8362f,  60700.f, -0.4742f, 1.8119f, 7730.f, -0.8919f, 1.1868f, 4090.f },
  {"XE", 1.50f,  131.30f,{20.29332f,  3.92820f, 19.02982f,  0.34400f,  8.97671f,  26.46594f, 1.99000f,  64.26587f,  3.71180f}, -0.5179f,  7.3500f,  65200.f, -0.4205f, 1.9578f, 8340.f, -0.8200f, 1.2838f, 4410.f },
  {"CS", 2.65f,  132.91f,{20.38922f,  3.56900f, 19.10622f,  0.31070f, 10.66201f,  24.38794f, 1.49530f, 213.90420f,  3.33520f}, -0.7457f,  7.9052f,  70000.f, -0.3680f, 2.1192f, 8980.f, -0.7527f, 1.3916f, 4750.f },
  {"BA", 2.17f,  137.34f,{20.33612f,  3.21600f, 19.29703f,  0.27560f, 10.88801f,  20.20732f, 2.69590f, 167.20220f,  2.77310f}, -1.0456f,  8.4617f,  75000.f, -0.3244f, 2.2819f, 9650.f, -0.6940f, 1.5004f, 5110.f },
  {"LA", 1.87f,  138.91f,{20.57802f,  2.94817f, 19.59901f,  0.24448f, 11.37271f,  18.77261f, 3.28719f, 133.12410f,  2.14678f}, -1.4094f,  9.0376f,  80300.f, -0.2871f, 2.4523f,10400.f, -0.6411f, 1.6148f, 5490.f },
  {"CE", 1.83f,  140.12f,{21.16711f,  2.81219f, 19.76952f,  0.22684f, 11.85131f,  17.60832f, 3.33049f, 127.11310f,  1.86264f}, -1.8482f,  9.6596f,  85700.f, -0.2486f, 2.6331f,11100.f, -0.5890f, 1.7358f, 5880.f },
  {"PR", 1.82f,  140.91f,{22.04402f,  2.77393f, 19.66972f,  0.22209f, 12.38561f,  16.76692f, 2.82428f, 143.64410f,  2.05830f}, -2.4164f, 10.2820f,  91200.f, -0.2180f, 2.8214f,11900.f, -0.5424f, 1.8624f, 6300.f },
  {"ND", 1.81f,  144.24f,{22.68452f,  2.66248f, 19.68472f,  0.21063f, 12.77401f,  15.88502f, 2.85137f, 137.90310f,  1.98486f}, -3.1807f, 10.9079f,  96800.f, -0.1943f, 3.0179f,12700.f, -0.5012f, 1.9950f, 6740.f },
  {"PM", 1.81f,  147.00f,{23.34052f,  2.56270f, 19.60953f,  0.20209f, 13.12351f,  15.10091f, 2.87516f, 132.72110f,  2.02876f}, -4.0598f, 11.5523f, 102000.f, -0.1753f, 3.2249f,13500.f, -0.4626f, 2.1347f, 7200.f },
  {"SM", 1.80f,  150.35f,{24.00424f,  2.47274f, 19.42583f,  0.19645f, 13.43961f,  14.39961f, 2.89604f, 128.00710f,  2.20963f}, -5.3236f, 12.2178f, 108000.f, -0.1638f, 3.4418f,14400.f, -0.4287f, 2.2815f, 7680.f },
  {"EU", 2.00f,  151.96f,{24.62744f,  2.38790f, 19.08862f,  0.19420f, 13.76031f,  13.75461f, 2.92270f, 123.17410f,  2.57450f}, -8.9294f, 11.1857f, 110000.f, -0.1578f, 3.6682f,15400.f, -0.3977f, 2.4351f, 8190.f },
  {"GD", 1.79f,  157.25f,{25.07094f,  2.25341f, 19.07982f,  0.18195f, 13.85181f,  12.93311f, 3.54545f, 101.39810f,  2.41960f}, -8.8380f, 11.9157f, 105000.f, -0.1653f, 3.9035f,16300.f, -0.3741f, 2.5954f, 8720.f },
  {"TB", 1.76f,  158.92f,{25.89763f,  2.24256f, 18.21852f,  0.19614f, 14.31671f,  12.66481f, 2.95354f, 115.36210f,  3.58324f}, -9.1472f,  9.1891f,  84700.f, -0.1723f, 4.1537f,17400.f, -0.3496f, 2.7654f, 9270.f },
  {"DY", 1.75f,  162.50f,{26.50703f,  2.18020f, 17.63832f,  0.20217f, 14.55962f,  12.18991f, 2.96577f, 111.87410f,  4.29728f}, -9.8046f,  9.8477f,  97700.f, -0.1892f, 4.4098f,18400.f, -0.3302f, 2.9404f, 9850.f },
  {"HO", 1.74f,  164.93f,{26.90494f,  2.07051f, 17.29402f,  0.19794f, 14.55831f,  11.44071f, 3.63837f,  92.65669f,  4.56797f},-14.9734f,  3.7046f,  34700.f, -0.2175f, 4.6783f,19500.f, -0.3168f, 3.1241f, 10400.f },
  {"ER", 1.73f,  167.26f,{27.65634f,  2.07356f, 16.42853f,  0.22355f, 14.97791f,  11.36041f, 2.98233f, 105.70310f,  5.92047f}, -9.4367f,  3.9380f,  36700.f, -0.2586f, 4.9576f,20700.f, -0.3091f, 3.3158f, 11100.f },
  {"TM", 1.72f,  168.93f,{28.18193f,  2.02859f, 15.88512f,  0.23885f, 15.15421f,  10.99751f, 2.98706f, 102.96110f,  6.75622f}, -8.0393f,  4.1821f,  39300.f, -0.3139f, 5.2483f,21900.f, -0.3084f, 3.5155f, 11700.f },
  {"YB", 1.94f,  173.04f,{28.66414f,  1.98890f, 15.43451f,  0.25712f, 15.30871f,  10.66471f, 2.98963f, 100.41710f,  7.56673f}, -7.2108f,  4.4329f,  41000.f, -0.3850f, 5.5486f,23100.f, -0.3157f, 3.7229f, 12400.f },
  {"LU", 1.72f,  174.97f,{28.94763f,  1.90182f, 15.22081f,  9.98520f, 15.10001f,   0.26103f, 3.71601f,  84.32988f,  7.97629f}, -6.6179f,  4.6937f,  45000.f, -0.4720f, 5.8584f,24400.f, -0.3299f, 3.9377f, 13100.f },
  {"HF", 1.56f,  178.49f,{29.14404f,  1.83262f, 15.17261f,  9.59991f, 14.75861f,   0.27512f, 4.30013f,  72.02908f,  8.58155f}, -6.1794f,  4.9776f,  46000.f, -0.5830f, 6.1852f,25800.f, -0.3548f, 4.1643f, 13900.f },
  {"TA", 1.43f,  180.95f,{29.20244f,  1.77333f, 15.22931f,  9.37047f, 14.51351f,   0.29598f, 4.76492f,  63.36447f,  9.24355f}, -5.7959f,  5.2718f,  48500.f, -0.7052f, 6.5227f,27200.f, -0.3831f, 4.3992f, 14600.f },
  {"W",  1.37f,  183.85f,{29.08183f,  1.72029f, 15.43001f,  9.22591f, 14.43271f,   0.32170f, 5.11983f,  57.05606f,  9.88751f}, -5.4734f,  5.5774f,  51300.f, -0.8490f, 6.8722f,28600.f, -0.4201f, 4.6430f, 15400.f },
  {"RE", 1.37f,  186.20f,{28.76213f,  1.67191f, 15.71892f,  9.09228f, 14.55641f,   0.35050f, 5.44174f,  52.08615f, 10.47201f}, -5.2083f,  5.8923f,  57200.f, -1.0185f, 7.2310f,30100.f, -0.4693f, 4.8944f, 16200.f },
  {"OS", 1.34f,  190.20f,{28.18944f,  1.62903f, 16.15501f,  8.97949f, 14.93051f,   0.38266f, 5.67590f,  48.16475f, 11.00051f}, -4.9801f,  6.2216f,  58000.f, -1.2165f, 7.6030f,31600.f, -0.5280f, 5.1558f, 17100.f },
  {"IR", 1.36f,  192.20f,{27.30493f,  1.59279f, 16.72961f,  8.86554f, 15.61152f,   0.41792f, 5.83378f,  45.00114f, 11.47221f}, -4.7710f,  6.5667f,  62400.f, -1.4442f, 7.9887f,33100.f, -0.5977f, 5.4269f, 18000.f },
  {"PT", 1.37f,  195.09f,{27.00594f,  1.51293f, 17.76392f,  8.81175f, 15.71312f,   0.42459f, 5.78371f,  38.61034f, 11.68831f}, -4.5932f,  6.9264f,  63400.f, -1.7033f, 8.3905f,34800.f, -0.6812f, 5.7081f, 18900.f },
  {"AU", 1.44f,196.9665f,{16.88193f,  0.46110f, 18.59132f,  8.62161f, 25.55824f,   1.48260f, 5.86001f,  36.39563f, 12.06581f}, -4.4197f,  7.2980f,  66900.f, -2.0133f, 8.8022f,36500.f, -0.7638f, 5.9978f, 19900.f },
  {"HG", 1.50f,  200.59f,{20.68092f,  0.54500f, 19.04172f,  8.44841f, 21.65752f,   1.57290f, 5.96761f,  38.32463f, 12.60891f}, -4.2923f,  7.6849f,  66800.f, -2.3894f, 9.2266f,38200.f, -0.8801f, 6.2989f, 20900.f },
  {"TL", 1.64f,  204.37f,{27.54463f,  0.65515f, 19.15842f,  8.70752f, 15.53802f,   1.96347f, 5.52594f,  45.81496f, 13.17461f}, -4.1627f,  8.0900f,  75400.f, -2.8358f, 9.6688f,40100.f, -1.0117f, 6.6090f, 21900.f },
  {"PB", 1.60f,  207.19f,{31.06174f,  0.69020f, 13.06371f,  2.35760f, 18.44202f,   8.61801f, 5.96961f,  47.25795f, 13.41181f}, -4.0753f,  8.5060f,  79800.f, -3.3944f,10.1111f,41900.f, -1.1676f, 6.9287f, 22900.f },
  {"BI", 1.60f,  208.98f,{33.36894f,  0.70400f, 12.95101f,  2.92380f, 16.58772f,   8.79371f, 6.46921f,  48.00935f, 13.57821f}, -4.0111f,  8.9310f,  84300.f, -4.1077f,10.2566f,43800.f, -1.3494f, 7.2566f, 24000.f },
  {"PO", 1.60f,  210.00f,{34.67264f,  0.70100f, 15.47331f,  3.55078f, 13.11381f,   9.55643f, 7.02589f,  47.00455f, 13.67701f}, -3.9670f,  9.3834f,  88100.f, -5.1210f,11.0496f,45800.f, -1.5613f, 7.5986f, 25100.f },
  {"AT", 1.60f,  210.00f,{35.31633f,  0.68587f, 19.02112f,  3.97458f,  9.49888f,  11.38241f, 7.42519f,  45.47156f, 13.71081f}, -3.9588f,  9.8433f,  86500.f, -7.9122f, 9.9777f,40700.f, -1.8039f, 7.9509f, 26200.f },
  {"RN", 1.80f,  222.00f,{35.56314f,  0.66310f, 21.28162f,  4.06910f,  8.00371f,  14.04221f, 7.44331f,  44.24734f, 13.69051f}, -3.9487f, 10.3181f,  97200.f, -8.0659f,10.4580f,39800.f, -2.0847f, 8.3112f, 27300.f },
  {"FR", 2.80f,  223.00f,{35.92993f,  0.64645f, 23.05472f,  4.17619f, 12.14391f,  23.10522f, 2.11253f, 150.64510f, 13.72471f}, -3.9689f, 10.8038f, 102000.f, -7.2224f, 7.7847f,32200.f, -2.4129f, 8.6839f, 28500.f },
  {"RA", 2.20f,  226.00f,{35.76303f,  0.61634f, 22.90642f,  3.87135f, 12.47391f,  19.98872f, 3.21097f, 142.32510f, 13.62111f}, -4.0088f, 11.2969f, 102000.f, -6.7704f, 8.1435f,33000.f, -2.8081f, 9.0614f, 29800.f },
  {"AC", 1.90f,  227.00f,{35.65973f,  0.58909f, 23.10323f,  3.65155f, 12.59771f,  18.59901f, 4.08655f, 117.02010f, 13.52661f}, -4.0794f, 11.7994f, 143000.f, -6.8494f, 8.5178f,54000.f, -3.2784f, 9.4502f, 31100.f },
  {"TH", 1.85f,  232.04f,{35.56453f,  0.56336f, 23.42192f,  3.46204f, 12.74731f,  17.83092f, 4.80704f,  99.17230f, 13.43141f}, -4.1491f, 12.3296f, 118000.f, -7.2400f, 8.8979f,37000.f, -3.8533f, 9.8403f, 32300.f },
  {"PA", 1.80f,  231.00f,{35.88474f,  0.54775f, 23.29482f,  3.41519f, 14.18911f,  16.92352f, 4.17287f, 105.25110f, 13.42871f}, -4.2473f, 12.8681f, 106000.f, -8.0334f, 9.2807f,38700.f, -4.6067f,10.2413f, 34200.f },
  {"U",  1.80f,  238.03f,{36.02284f,  0.52930f, 23.41283f,  3.32530f, 14.94911f,  16.09273f, 4.18800f, 100.61310f, 13.39661f}, -4.3638f, 13.4090f, 112000.f, -9.6767f, 9.6646f,40300.f, -5.7225f,10.6428f, 35000.f },
  {"NP", 1.80f,  237.00f,{36.18744f,  0.51193f, 23.59642f,  3.25396f, 15.64022f,  15.36222f, 4.18550f,  97.49089f, 13.35731f}, -4.5053f, 13.9666f, 123000.f,-11.4937f, 4.1493f,25700.f, -6.9995f, 9.5876f, 29900.f },
  {"PU", 1.80f,  242.00f,{36.52544f,  0.49938f, 23.80832f,  3.26371f, 16.77072f,  14.94551f, 3.47947f, 105.98010f, 13.38121f}, -4.6563f, 14.3729f, 113000.f, -9.4100f, 4.3056f,16200.f,-13.5905f, 6.9468f, 22700.f} 	
};
#define SCATTCU 3
#define SCATTAG 2
#define SCATTMO 1
class Scatt{
  typedef struct {
    int  ih,//!< h
         ik,//!< k
         il;//!< l 
    float fo,//!< F observed
          so,//!< \f$\sigma(observed)\f$
          si,
          fc,//!< F calculated
          phi, //!< \f$\varphi\f$ 
          dis; 
  } Rec;
  public: 
  Scatt(Molecule *m) { 
    mol=m; 
    f000=0.0;
    Fmax=0.0;
    resd=-1.0f;
    //printf("Scatt(Molecule *m) %f\n",resd);
    //   qDebug()<<"12345"<<atomformfactor(12,0.0);
  }

  void setSCAT(QStringList tok){
    if (tok.size()<16)return;
    int an = mol->getOZ(tok.at(1));
    if (an<0)return;
    for (int i=0; i<9; i++) {
      ff[an].ab4c[i]=tok.at(2+i).toFloat();
      //    qDebug()<<i+2<<ff[an].ab4c[i];
    }
    ff[an].fpcu=ff[an].fpmo=ff[an].fpag    = tok.at(11).toFloat();
    ff[an].fppcu=ff[an].fppmo=ff[an].fppag = tok.at(12).toFloat();
    ff[an].mucu=ff[an].mumo=ff[an].muag    = tok.at(13).toFloat();
    ff[an].rad=tok.at(14).toFloat();
    ff[an].mass=tok.at(15).toFloat();
  }

  void setDISP(QStringList tok){
    if (tok.size()<4)return;
    int an = mol->getOZ(tok.at(1));
    ff[an].fpcu=ff[an].fpmo=ff[an].fpag    = tok.at(2).toFloat();
    if (tok.size()<4)return;
    ff[an].fppcu=ff[an].fppmo=ff[an].fppag = tok.at(3).toFloat();
    if (tok.size()<5)return;
    ff[an].mucu=ff[an].mumo=ff[an].muag    = tok.at(4).toFloat();

  }

  float fprime(int an,int radiation){
    switch (radiation){
      case SCATTCU: return ff[an].fpcu;
      case SCATTMO: return ff[an].fpmo;
      case SCATTAG: return ff[an].fpag;
      default: return ff[an].fpcu; 
    } 
  }
  float fdprime(int an,int radiation){
    switch (radiation){
      case SCATTCU: return ff[an].fppcu;
      case SCATTMO: return ff[an].fppmo;
      case SCATTAG: return ff[an].fppag;
      default: return ff[an].fppcu; 
    } 
  }
  float atomformfactor(int an, float s){
    float erg=0.0;
    for (int i=0; i<4;i++){
      erg+=ff[an].ab4c[2*i]*exp(-ff[an].ab4c[2*i+1]*s);
    }
    erg+=ff[an].ab4c[8];
    return erg; 
  }
  float FF(V3 hkl,int radiation,float &phang,float &I){
    const float twopi=M_PIf*2.0f;
    const float twopi2=M_PIf*M_PIf*2.0f;
    phang=0;
    float A=0.f,Aa=0.f;
    float B=0.f,Ba=0.f;
    float s2=sintl2(hkl)*0.25f;
    //printf("s2 %f %f\n",s2,resd);
    if ((s2< mol->hklShellLow)||(s2> mol->hklShellHig)){
      phang=0.0f;

      I=-666.0f; 
      return 0.0f;
    }
    if (s2>0.001)resd=fmax(s2,resd);
    QList<float> f0;
    f000=0.0;
    for (int i=0; i<isfac.size(); i++) f0.append(atomformfactor(isfac.at(i),s2));
    for (int i=0; i<xyz.size(); i++){
      float sof = sofs.at(i);
      int an= isfac.at(type.at(i));
      f000+=sof*(an+1);
      float p=(float)(twopi*(xyz.at(i)*hkl));
      float fp=fprime(an,radiation);
      float fpp=fdprime(an,radiation);
      //T = 8*(pi**2)*Uiso*sin(theta/lambda)**2    
      if (isos.at(i)) sof*=expf(bet.at(i).m11*s2);
      else {
        sof*=expf(-twopi2* ((hkl*bet.at(i))*hkl));
      }//  */
      float cs=cosf(p),si=sinf(p);
      //float cs=1,si=0.0;
      float a=(f0.at(type.at(i))+fp); 
      A+= sof*(a*cs-fpp*si);
      B+= sof*(a*si+fpp*cs);
      Aa+=sof*(a*cs);
      Ba+=sof*(a*si);
    }
    I=A*A+B*B;
    float Ia=Aa*Aa+Ba*Ba;
    phang=atan2f(Ba,Aa)*180.0f/M_PIf;
    phang=fmod(720.01f+phang,360)-0.01f;
    float fc=sqrtf(Ia);
    if (s2>0.001) Fmax=fmax(Fmax,fc);
    return fc;
  }

  float F(int h, int k, int l,int radiation,float &phang,float &disp){
    const float twopi=M_PIf*2.0f;
    const float twopi2=M_PIf*M_PIf*2.0f;
    phang=0;
    float A=0,Aa=0;
    f000=0.0;
    float B=0,Ba=0;
    Matrix N=Matrix(mol->cell.as,0,0,0,mol->cell.bs,0,0,0,mol->cell.cs),beta_,beta;//,ucc;
    float s2=sintl2(h,k,l)*0.25f;
    //printf("Fs2 %f %f\n",s2,resd);
    if (s2>0.001)resd=fmin(0.5/sqrt(s2),resd);
    int ns = mol->cell.symmops.size();
    int na = mol->asymm.size();
    for (int s=0; s<ns; s++){
      for (int i=0; i<na; i++){
        int an = mol->asymm.at(i).an;
        if (an<0) continue;
        bool iso=(mol->asymm.at(i).isIso);
        if (!iso){
          beta=(N*mol->asymm.at(i).uf*N);
          mol->Usym(beta,mol->cell.symmops.at(s),beta_);
        }
        float sof = mol->asymm.at(i).sof;
        f000+=sof*(an+1);
        V3 uvw = mol->cell.symmops.at(s)*mol->asymm[i].frac+mol->cell.trans.at(s);

        V3 hkl(h,k,l);
        float p=twopi*uvw*hkl;
        float f0=atomformfactor(an,s2);
        float fp=fprime(an,radiation);
        float fpp=fdprime(an,radiation);
        //T = 8*(pi**2)*Uiso*sin(theta/lambda)**2
        if (iso) sof*=exp(-4*twopi2*mol->asymm.at(i).uf.m11*s2);
        else {
          sof*=exp(-twopi2* ((hkl*beta_)*hkl));

        }
        float cs=cos(p),si=sin(p);
        A+=sof*((f0+fp)*cs-fpp*si);
        B+=sof*((f0+fp)*si+fpp*cs);
        Aa+=sof*((f0+fp)*cs);
        Ba+=sof*((f0+fp)*si);
      }
    }
    float I=A*A+B*B;
    float Ia=Aa*Aa+Ba*Ba;
    phang=atan2(Ba,Aa)*180.0/M_PI;
    phang=fmod(720.01+phang,360)-0.01;
    disp=I;
    float fc=sqrt(Ia);
    if (s2>0.001) Fmax=fmax(Fmax,fc);
    return fc;
  }
  /*float F(int h, int k, int l,int radiation,float &phang,float &disp){
    const float twopi=M_PI*2.0;
    const float twopi2=M_PI*M_PI*2.0;
    phang=0;
    float A=0,Aa=0;
    f000=0.0;
    Matrix N=Matrix(mol->cell.as,0,0,0,mol->cell.bs,0,0,0,mol->cell.cs),beta_,beta,ucc;
    float s2=sintl2(h,k,l)*0.25;
    if (s2>0.001)resd=fmin(0.5/sqrt(s2),resd);
    int ns = mol->cell.symmops.size();
    int na = mol->asymm.size();
    float sofsum=0;
    float sofsum2=0;
    for (int s=0; s<ns; s++){
    for (int i=0; i<na; i++){
    int an = mol->asymm.at(i).an;
    if (an<0) continue;
    bool iso=(mol->asymm.at(i).isIso);
    if (!iso){
    beta=(N*mol->asymm.at(i).uf*N);
    mol->Usym(beta,mol->cell.symmops.at(s),beta_);
    Matrix uff;
    mol->Usym(mol->asymm.at(i).uf,mol->cell.symmops.at(s),uff);
    mol->Uf2Uo(uff,ucc);
    }
    float sof = mol->asymm.at(i).sof;
    f000+=sof*(an+1);
    sofsum+=sof;
    V3 uvw = mol->cell.symmops.at(s)*mol->asymm[i].frac+mol->cell.trans.at(s);
    V3 hkl(h,k,l);
    float p=twopi*uvw*hkl;
    float f0=atomformfactor(an,s2);
    float fp=fdprime(an,radiation);
    float fpp=fprime(an,radiation);
    if (iso) sof*=exp(-4*twopi2*mol->asymm.at(i).uf.m11*s2);
    else {
    sof*=exp(-twopi2* ((hkl*beta_)*hkl));
    }
    sofsum2+=sof;
    float cs=cos(p),si=sin(p);
    A+=sof*((f0+fp)*cs+fpp*si);
    B+=sof*((f0+fp)*si+fpp*cs);
    Aa+=sof*((f0+fp)*cs);
    Ba+=sof*((f0+fp)*si);
    }
    }
    float I=A*A+B*B;
    float Ia=Aa*Aa+Ba*Ba;
    phang=atan2(Ba,Aa)*180.0/M_PI;
    phang=fmod(720+phang,360);
    disp= (I-Ia)/Ia;
    if ((h==0)&&(k==0)&&(l==0)){ 
    if (I>0.001){
    printf("%4d%4d%4d %12.6f %12.6f %10.6f s2= %f s= %f d= %f ph%f %f %f%%\n",h,k,l,I,A,B,s2,sqrt(s2),0.5/sqrt(s2) ,phang,I,100*I/Ia);
    }
    else printf("%4d%4d%4d absent d=%f\n",h,k,l,0.5/sqrt(s2));

    }
    if ((h==0)&&(k==0)&&(l==0)) printf("%f %f\n",sofsum, sofsum2);
    disp=I;
    float fc=sqrt(Ia);
    if (s2>0.001) Fmax=fmax(Fmax,fc);
    return fc;
    }*/
  QString myfcf(QString fn){
    //  qDebug()<<mol->hklf;
    //  qDebug()<<mol->hklScale<<mol->hklSigmaScale;
    if ((mol->hklf<3)||(mol->hklf>4)) return "";
    //  const float twopi=M_PI*2.0;
    const float twopi2=M_PIf*M_PIf*2.0f;
    QTime zeit; zeit.start();

    Rec *lr=NULL,ob;
    QString base=fn, mfcf=fn, hklf=fn;
    hklf=hklf.replace(QRegExp("(.res$)|(.ins$)"),".hkl");
    mfcf=mfcf.replace(QRegExp("(.res$)|(.ins$)"),".fcf6");
    base=base.replace(QRegExp("(.res$)|(.ins$)"),"");
    base=base.section('/', -1);
    FILE *hkl=fopen(hklf.toStdString().c_str(),"rt");
    if (hkl==NULL) {fprintf(stderr,"can't open hkl\n"); return "";}
    int nr=0;
    char line[128];
    int wl=0;
    if (fabs(mol->cell.wave-0.71073)<0.01) wl=SCATTMO;
    else if (fabs(mol->cell.wave-1.5418)<0.01) wl=SCATTCU;
    else if (fabs(mol->cell.wave-0.56083)<0.01) wl=SCATTAG;
    else wl=-1;
    int nnr=0;
    mol->hklShellLow = pow(0.5/mol->hklShellLow,2);
    mol->hklShellHig = pow(0.5/mol->hklShellHig,2);
    //  qDebug()<<"@@"<<mol->hklShellLow<<mol->hklShellHig;
    while (!feof(hkl)){
      if (fgets(line,120,hkl)) {;}
      nnr++;
    }
    rewind(hkl);
    lr=(Rec*) malloc(sizeof(Rec)*nnr);
    FILE *fc6=fopen(mfcf.toStdString().c_str(),"wt");
    if (fc6==NULL) {
      fprintf(stderr,"can't open %s\n",mfcf.toStdString().c_str()); 
      qDebug()<<"can't open " << mfcf;
      return "";
    }

    //FILE *tst=NULL;//:qfopen("scale_test_123.456","wt");
    if (1) {
      Matrix N=Matrix(mol->cell.as,0,0,0,mol->cell.bs,0,0,0,mol->cell.cs),beta_,beta;//,ucc;
      type.clear();
      bet.clear();
      sofs.clear();
      xyz.clear();
      isos.clear();
      int ns = mol->cell.symmops.size();
      int na = mol->asymm.size();
      for (int s=0; s<ns; s++){
        for (int i=0; i<na; i++){
          int an = mol->asymm.at(i).an;
          if (an<0) continue;
          if (!isfac.contains(an)) isfac.append(an);
          type.append(isfac.indexOf(an));
          bool iso=(mol->asymm.at(i).isIso);
          if (!iso){
            beta=(N*mol->asymm.at(i).uf*N);
            mol->Usym(beta,mol->cell.symmops.at(s),beta_);
          }else{beta_.m11=-4*twopi2*mol->asymm.at(i).uf.m11;}
          isos.append(iso);
          bet.append(beta_);
          float sof = mol->asymm.at(i).sof;
          sofs.append(sof);
          xyz.append(mol->cell.symmops.at(s)*mol->asymm[i].frac+mol->cell.trans.at(s));
        }
      }
    }




    //  printf("wave:%f %f %f %f %d %d wl=%d\n",mol->cell.wave,fabs(mol->cell.wave-0.71073),fabs(mol->cell.wave-1.5418),fabs(mol->cell.wave-0.56083), mol->cell.ns0, mol->cell.symmops.size(),wl);

    /*printf("hklf N=%d s:%g \n%g %g %g\n%g %g %g\n%g %g %g\nsm: %g \n",mol->hklf,mol->hklScale ,
        mol->hklMat.m11,mol->hklMat.m12,mol->hklMat.m13,
        mol->hklMat.m21,mol->hklMat.m22,mol->hklMat.m23,
        mol->hklMat.m31,mol->hklMat.m32,mol->hklMat.m33,
        mol->hklSigmaScale);*/
    while (!feof(hkl)){    
      if (fgets(line,120,hkl)) {;}
      if (feof(hkl))continue;
      if (strlen(line)<28) continue;
      char chkl[13],cisig[17];
      for (int ci=0; ci<12; ci++) chkl[ci]=line[ci];
      chkl[12]='\0';
      for (int ci=0; ci<16; ci++) cisig[ci]=line[ci+12];
      cisig[16]='\0';
      sscanf(chkl,"%4d%4d%4d", &ob.ih, &ob.ik, &ob.il);
      sscanf(cisig,"%8f%8f", &ob.fo, &ob.si);
      if ((ob.ih==0)&&(ob.il==0)&&(ob.ik==0)) {
        //      printf("#[%d,%d,%d,%d]%s",nr,ob.ih,ob.ik,ob.il,line);

        break; 
      }
      V3 hkl=V3(ob.ih,ob.ik,ob.il)*mol->hklMat; 
      ob.ih=hkl.x;
      ob.ik=hkl.y;
      ob.il=hkl.z;
      ob.fo*=mol->hklScale;
      ob.si=(mol->hklScale*mol->hklSigmaScale*ob.si);
      if (mol->hklf==3){
        double xxx=fmax(0.01,ob.si);
        ob.si=2.0*xxx * fabs(fmax(fabs(ob.fo),xxx));
        ob.fo*=ob.fo;
      }
      //ob.fc=FF(hkl,wl,ob.phi,ob.dis);
      //    if ((ob.ih==1)&&(ob.ik==0)&&(ob.il==0)){ printf("%4d%4d%4d %18.9f %18.9f\n",ob.ih,ob.ik,ob.il,ob.fo,ob.si); }
      //   ob.fc=F(ob.ih,ob.ik,ob.il,wl,ob.phi,ob.dis);

      lr[nr]=ob;
      nr++;
    }
    //  qDebug()<<resd <<0.5/sqrt(resd);
    //resd=0.5f/sqrtf(resd);
    //printf("resd %f\n",resd);
    //  for (int iw=0; iw<xyz.size(); iw++){ qDebug()<<type.at(iw)<<sofs.at(iw)<<xyz.at(iw).x<<xyz.at(iw).y<<xyz.at(iw).z; }
    //if (tst!=NULL) fclose(tst);

    float myosf=0.0;
    for (int i=0; i<nr; i++){
      V3 uvw=V3(lr[i].ih,lr[i].ik,lr[i].il);
      V3 m=uvw;
      float p,q=lr[i].phi;
      lr[i].phi=fmod(720.0+q,360.0);
      for (int s=0; s<mol->cell.ns0;s++){
        float t=1.0;
        V3 nhkl=uvw*mol->cell.symmops.at(s);
        //      if((nl<0)||((nl==0)&&(nk<0))||((nl==0)&&(nk==0)&&(nh<0)))
        //if((nl<0)||((nl==0)&&(nk<0))||((nl==0)&&(nk==0)&&(nh<0)))
        if((nhkl.z<0)||((nhkl.z==0)&&(nhkl.y<0))||((nhkl.z==0)&&(nhkl.y==0)&&(nhkl.x<0))) {
          nhkl.x*=-1;
          nhkl.y*=-1;
          nhkl.z*=-1;
          t=-1.0;
        }
        if ((nhkl.z<m.z)||((nhkl.z==m.z)&&(nhkl.y<m.y))||((nhkl.z==m.z)&&(nhkl.y==m.y)&&(nhkl.x<=m.x))) continue;
        m=nhkl;
        //p=u*sy[9][k]+v*sy[10][k]+w*sy[11][k];
        p=uvw*mol->cell.trans.at(s);
        lr[i].phi=fmod(719.99+t*fmod(q-360.0*p,360.0),360.0)+0.01;
        /*      if (m==V3(1,1,0)) {printf("?%4d%4d%4d sym=%d %g \n %4g%4g%4g fo%12.6f si%12.6f fc%12.6f phi%12.3f %d %f %f (%g %g %g)\n",lr[i].ih,lr[i].ik,lr[i].il,s,t,
                nhkl.x,nhkl.y,nhkl.z,
                lr[i].fo,lr[i].si,lr[i].fc,lr[i].phi,i,q,p, mol->cell.trans.at(s).x,mol->cell.trans.at(s).y,mol->cell.trans.at(s).z);}*/
      }
      lr[i].ih=m.x;
      lr[i].ik=m.y;
      lr[i].il=m.z;
    }
    sorthkl(nr,lr);
    int n=-1;
    //  float sw=0.,swxy=0.0,swx=0.0,swy=0.0,swx2=0.0,sfo=0,sfoc=0;
    float swxy=0.0,swx2=0.0,sfo=0,sfoc=0;;
    {int i=0;
      while(i<nr){
        float t=0.;
        float u=0.;
        float v=0.;
        float v2=0.;
        float ws=0;
        int m;
        int k=i;
        while ((i<nr)&&(lr[i].ih==lr[k].ih)&&(lr[i].ik==lr[k].ik)&&(lr[i].il==lr[k].il)) {
          t=t+1.;
          float w=fmax(4.0,lr[i].fo/lr[i].si)/lr[i].si;//1.0;//
          ws+=w; 
          u+=lr[i].fo*w;//*lr[i].fc*lr[i].fc/lr[i].dis;
          v+=1./(lr[i].si*lr[i].si);
          v2+=lr[i].si;
          // p=lr[i].phi;
          // ddi+=lr[i].dis;
          //         if (t>1) {printf("!%4d%4d%4d fo%12.6f sig%12.6f fc%12.6f phi%12.3f i%d  k%dw=%f dis%f\n",lr[i].ih,lr[i].ik,lr[i].il,lr[i].fo,lr[i].si,lr[i].fc,lr[i].phi,i,k,w,lr[i].dis);}
          i++;
        }
        m=n+1;
        float yy=(u/ws);
        lr[m].fo = sqrt(fmax(0.0f,yy));
        lr[m].so=sqrt(lr[m].fo*lr[m].fo+sqrt(1./v))-lr[m].fo;
        lr[m].si=1.0/sqrt(v);
        n=m;
        lr[n].ih=lr[k].ih;
        lr[n].ik=lr[k].ik;
        lr[n].il=lr[k].il;
      }
      printf("%d %d %d \n",nr,n,i);
    }
    n++;
    nr=n;
    for (int i=0; i<nr; i++){
      V3 chkl=V3(lr[i].ih,lr[i].ik,lr[i].il);
      lr[i].fc=0.0;
      lr[i].fc=FF(chkl,wl,lr[i].phi,lr[i].dis);
      lr[i].fo =lr[i].fo * (lr[i].fc*lr[i].fc/lr[i].dis);//correct for dispersion
      if (lr[i].dis<=-665.0f) {
        lr[i].fc=0.0;
        continue;
      }
      if ((mol->exti!=-666.0)&&(mol->swat==-666.0)){
        lr[i].fc*=pow(1 + 0.001 * mol->exti * lr[i].fc * lr[i].fc  *mol->cell.wave * mol->cell.wave /sqrt(sintl2(chkl)),-0.25);
      }
      float xx=lr[i].fc*lr[i].fc;
      float yy=lr[i].fo*lr[i].fo;
      float v=1.0/(lr[i].si*lr[i].si);
      if (xx>0.001){
        swx2+=v*xx*xx;
        swxy+=v*yy*xx;
      }
    }
    float osf2=(mol->osf<=0.0)?swxy/swx2:mol->osf*mol->osf;//myosf;
    myosf=sqrt(osf2);
    // qDebug() << mol->osf << myosf << osf2 << swxy/swx2<<swxy <<swx2;
    printf("%g %g %g\n",myosf,osf2,swxy/swx2);
    type.clear();
    bet.clear();
    sofs.clear();
    xyz.clear();
    isos.clear();  
    //  myosf=mol->osf;
    resd=0.5f/sqrtf(resd);
    printf("???? %f \n",resd);
    fprintf(fc6,
        "#\n# h,k,l, Fo-squared, sigma(Fo-squared), Fc and phi(calc)\n#\ndata_%s\n_shelx_title '%s'\n_shelx_refln_list_code          6\n"
        "_shelx_F_calc_maximum      %6.2f\n_exptl_crystal_F_000       %6.2f\n_reflns_d_resolution_high  %6.4f\n\nloop_\n_space_group_symop_operation_xyz\n"
        ,base.toStdString().c_str()
        ,mol->titl.toStdString().c_str()
        ,Fmax
        ,f000
        ,resd
        );
    for (int s=0; s<mol->cell.symmops.size();s++) 
      fprintf(fc6,"'%s'\n",mol->symmcode2human(s).trimmed().toStdString().c_str());
    fprintf(fc6,
        "\n_cell_length_a   %8.4f\n_cell_length_b   %8.4f\n_cell_length_c   %8.4f\n_cell_angle_alpha %7.3f\n_cell_angle_beta  %7.3f\n_cell_angle_gamma %7.3f\n\n"
        "loop_\n_refln_index_h\n_refln_index_k\n_refln_index_l\n_refln_F_squared_meas\n_refln_F_squared_sigma\n_refln_F_calc\n_refln_phase_calc\n",
        mol->cell.a,mol->cell.b,mol->cell.c,mol->cell.al,mol->cell.be,mol->cell.ga
        );
    int igut=0;
    for (int i=0; i<nr; i++){
      /*
         fprintf(fc6,"%d %d %d %.2f %.2f %2f %.2f %.1f %f\n"
         ,lr[i].ih
         ,lr[i].ik
         ,lr[i].il
      //        ,(pow(lr[i].fo/mol->osf,2.0))/(1.0+lr[i].dis)
      ,pow(lr[i].fo/mol->osf,2.0)
      ,lr[i].si/(mol->osf*mol->osf)
      ,sqrt(lr[i].dis)
      ,lr[i].fc
      ,lr[i].phi
      ,lr[i].fc*lr[i].fc
      // ,lr[i].dis
      );// */

      float iosig = powf(lr[i].fo/myosf,2.0f)/(lr[i].si/(osf2));
      if ((lr[i].fc>0.001)&& (iosig>-1.0)){
        igut++;
        sfo+=lr[i].fo/myosf;
        sfoc+=fabs(lr[i].fo/myosf-lr[i].fc);
      }
      if (lr[i].fc>0.001f)
        fprintf(fc6,"%d %d %d %.2f %.2f %.2f %.1f\n"
            ,lr[i].ih
            ,lr[i].ik
            ,lr[i].il
            //        ,(pow(lr[i].fo/mol->osf,2.0))/(1.0+lr[i].dis)
            ,powf(lr[i].fo/myosf,2.0f)
            ,lr[i].si/(osf2)
            //,sqrt(lr[i].dis)
            ,lr[i].fc
            ,lr[i].phi
            // ,lr[i].fc*lr[i].fc
            // ,lr[i].dis
            );//  */
    }
    fclose(hkl);
    fclose(fc6);
    printf("R1=%f %d %d %g %g \n",sfoc/sfo,igut,nr,sfoc,sfo);

    printf("=>=>%d %d\n",zeit.elapsed(),xyz.size());
    return mfcf;
  }
  private:
  Molecule *mol;
  float Fmax,f000,resd;
  QList<Matrix> bet;
  QList<V3> xyz;
  QList<float> sofs;
  QList<bool> isos;
  QList<int> type,isfac;

  float sintl2(int h,int k, int l){
    V3 H=V3(h,k,l);
    float s=H*mol->cell.Gi*H;
    return s;
  }
  float sintl2(V3 H){
    float s=H*mol->cell.Gi*H;
    return s;
  }
  void sorthkl(int nr, Rec r[]){
    /*! sorts the reflection list
    */
    Rec *hilf= (Rec*) malloc(sizeof(Rec)*nr);
    if (hilf==NULL)return ; 
    int i,j,k,nj,ni,spalte;int index[4096];
    for (spalte=0; spalte<3; spalte++){
      j=-999999;
      k=999999;
      switch (spalte) {
        case 0: for (i=0; i<nr; i++){ j=(j<r[i].ih)?r[i].ih:j; k=(k>r[i].ih)?r[i].ih:k; } break;
        case 1: for (i=0; i<nr; i++){ j=(j<r[i].ik)?r[i].ik:j; k=(k>r[i].ik)?r[i].ik:k; } break;
        case 2: for (i=0; i<nr; i++){ j=(j<r[i].il)?r[i].il:j; k=(k>r[i].il)?r[i].il:k; } break;
      }
      nj=-k;
      ni=(nj+j+1);
      for (i=0; i<=ni; i++) index[i]=0;
      for (i=0; i<nr; i++){
        switch (spalte){
          case 0: j=r[i].ih+nj; break;
          case 1: j=r[i].ik+nj; break;
          case 2: j=r[i].il+nj; break;
        }
        index[j]++;/*brauch ich das? -->JA!*/
        hilf[i].ih=r[i].ih;
        hilf[i].ik=r[i].ik;
        hilf[i].il=r[i].il;
        hilf[i].fo=r[i].fo;
        hilf[i].so=r[i].so;
        hilf[i].si=r[i].si;
        hilf[i].fc=r[i].fc;
        hilf[i].dis=r[i].dis;
        hilf[i].phi=r[i].phi;
      }/*/4*/
      j=0;
      for (i=0; i<ni; i++){
        k=j;
        j+=index[i];
        index[i]=k;     
      }/*/5*/
      for (i=0; i<nr;i++){
        switch (spalte) {
          case 0: j=hilf[i].ih +nj;break;
          case 1: j=hilf[i].ik +nj;break;
          case 2: j=hilf[i].il +nj;break;
        }     
        index[j]++;   
        j=index[j]-1;
        r[j].ih=hilf[i].ih;
        r[j].ik=hilf[i].ik;
        r[j].il=hilf[i].il;
        r[j].fo=hilf[i].fo;
        r[j].so=hilf[i].so;
        r[j].si=hilf[i].si;
        r[j].fc=hilf[i].fc;
        r[j].dis=hilf[i].dis;
        r[j].phi=hilf[i].phi;
      }/*/6*/
    }/*/spalten*/
    free(hilf);
  }
};
#endif
