use std::path::Path;
use std::path::PathBuf;
use std::str;
use std::sync::Arc;

use anyhow::Context;

use sequoia_openpgp as openpgp;
use openpgp::Fingerprint;
use openpgp::KeyHandle;
use openpgp::Result;
use openpgp::cert::raw::RawCertParser;
use openpgp::cert::prelude::*;
use openpgp::packet::UserID;
use openpgp::parse::Parse;

use openpgp_cert_d as cert_d;

use crate::LazyCert;
use crate::store::Certs;
use crate::store::MergeCerts;
use crate::store::Store;
use crate::store::StoreUpdate;
use crate::store::UserIDQueryParams;

use crate::TRACE;

pub struct CertD<'a> {
    certd: cert_d::CertD,

    certs: Certs<'a>,
}

impl<'a> CertD<'a> {
    /// Returns the canonicalized path.
    ///
    /// If path is `None`, then returns the default location.
    fn path(path: Option<&Path>) -> Result<PathBuf>
    {
        if let Some(path) = path {
            Ok(path.to_owned())
        } else {
            Ok(cert_d::CertD::user_configured_store_path()?)
        }
    }

    /// Opens the default cert-d for reading and writing.
    pub fn open_default() -> Result<Self>
    {
        let path = Self::path(None)?;
        Self::open(path)
    }

    /// Opens a cert-d for reading and writing.
    pub fn open<P>(path: P) -> Result<Self>
        where P: AsRef<Path>,
    {
        tracer!(TRACE, "CertD::open");

        let path = path.as_ref();
        t!("loading cert-d {:?}", path);

        let certd = cert_d::CertD::with_base_dir(&path)
            .map_err(|err| {
                t!("While opening the certd {:?}: {}", path, err);
                let err = anyhow::Error::from(err)
                    .context(format!("While opening the certd {:?}", path));
                err
            })?;

        let mut certd = Self {
            certd,
            certs: Certs::empty(),
        };

        certd.initialize(true)?;
        Ok(certd)
    }

    /// Returns a reference to the certd, if there is one.
    pub fn certd(&self) -> &cert_d::CertD {
        &self.certd
    }

    /// Returns a mutable reference to the certd, if there
    /// is one.
    pub fn certd_mut(&mut self) -> &mut cert_d::CertD {
        &mut self.certd
    }

    // Initialize a certd by reading the entries and populating the
    // index.
    fn initialize(&mut self, lazy: bool) -> Result<()>
    {
        use rayon::prelude::*;

        tracer!(TRACE, "CertD::initialize");

        let open = |fpr: &str| -> Option<( _, _)> {
            match self.certd.get_file(&fpr) {
                Ok(Some(file)) => {
                    match cert_d::Tag::try_from(&file) {
                        Ok(tag) => Some((tag, file)),
                        Err(err) => {
                            t!("Reading tag for {}: {}", fpr, err);
                            None
                        }
                    }
                }
                Ok(None) => {
                    t!("{} disappeared", fpr);
                    None
                }
                Err(err) => {
                    t!("Failed to read {}: {}", fpr, err);
                    None
                }
            }
        };

        let items = self.certd.fingerprints()
            .filter_map(|fpr| fpr.ok());

        let result: Vec<(String, cert_d::Tag, LazyCert)> = if lazy {
            items.collect::<Vec<_>>().into_par_iter()
                .filter_map(|fp| {
                    // XXX: Once we have a cached tag, avoid the
                    // work if tags match.
                    t!("loading {} from overlay", fp);

                    let (tag, file) = open(&fp)?;

                    let mut parser = match RawCertParser::from_reader(file) {
                        Ok(parser) => parser,
                        Err(err) => {
                            t!("While reading {:?} from the certd {:?}: {}",
                               fp, self.certd.base_dir(), err);
                            return None;
                        }
                    };

                    match parser.next() {
                        Some(Ok(cert)) => Some((fp, tag, LazyCert::from(cert))),
                        Some(Err(err)) => {
                            t!("While parsing {:?} from the certd {:?}: {}",
                                fp, self.certd.base_dir(), err);
                            None
                        }
                        None => {
                            t!("While parsing {:?} from the certd {:?}: empty file",
                                fp, self.certd.base_dir());
                            None
                        }
                    }
                })
                .collect()
        } else {
            // For performance reasons, we read, parse, and
            // canonicalize certs in parallel.
            items.collect::<Vec<_>>().into_par_iter()
                .filter_map(|fp| {
                    let (tag, file) = open(&fp)?;

                    // XXX: Once we have a cached tag and
                    // presumably a Sync index, avoid the work if
                    // tags match.
                    t!("loading {} from overlay", fp);
                    match Cert::from_reader(file) {
                        Ok(cert) => Some((fp, tag, LazyCert::from(cert))),
                        Err(err) => {
                            t!("While parsing {:?} from the certd {:?}: {}",
                               fp, self.certd.base_dir(), err);
                            None
                        }
                    }
                })
                .collect()
        };

        for (fp, _tag, cert) in result {
            if let Err(err) = self.certs.update(Arc::new(cert)) {
                // This is an in-memory index and updates doesn't
                // fail.  Nevertheless, we don't panic.
                t!("Error inserting {} into the in-memory index: {}",
                   fp, err);
            }
        }

        Ok(())
    }
}

impl<'a> Store<'a> for CertD<'a> {
    fn lookup_by_cert(&self, kh: &KeyHandle) -> Result<Vec<Arc<LazyCert<'a>>>> {
        self.certs.lookup_by_cert(kh)
    }

    fn lookup_by_cert_fpr(&self, fingerprint: &Fingerprint)
        -> Result<Arc<LazyCert<'a>>>
    {
        self.certs.lookup_by_cert_fpr(fingerprint)
    }

    fn lookup_by_cert_or_subkey(&self, kh: &KeyHandle) -> Result<Vec<Arc<LazyCert<'a>>>> {
        self.certs.lookup_by_cert_or_subkey(kh)
    }

    fn select_userid(&self, query: &UserIDQueryParams, pattern: &str)
        -> Result<Vec<Arc<LazyCert<'a>>>>
    {
        self.certs.select_userid(query, pattern)
    }

    fn lookup_by_userid(&self, userid: &UserID) -> Result<Vec<Arc<LazyCert<'a>>>> {
        self.certs.lookup_by_userid(userid)
    }

    fn grep_userid(&self, pattern: &str) -> Result<Vec<Arc<LazyCert<'a>>>> {
        self.certs.grep_userid(pattern)
    }

    fn lookup_by_email(&self, email: &str) -> Result<Vec<Arc<LazyCert<'a>>>> {
        self.certs.lookup_by_email(email)
    }

    fn grep_email(&self, pattern: &str) -> Result<Vec<Arc<LazyCert<'a>>>> {
        self.certs.grep_email(pattern)
    }

    fn lookup_by_email_domain(&self, domain: &str) -> Result<Vec<Arc<LazyCert<'a>>>> {
        self.certs.lookup_by_email_domain(domain)
    }

    fn fingerprints<'b>(&'b self) -> Box<dyn Iterator<Item=Fingerprint> + 'b> {
        self.certs.fingerprints()
    }

    fn certs<'b>(&'b self)
        -> Box<dyn Iterator<Item=Arc<LazyCert<'a>>> + 'b>
        where 'a: 'b
    {
        self.certs.certs()
    }

    fn prefetch_all(&mut self) {
        self.certs.prefetch_all()
    }

    fn prefetch_some(&mut self, certs: &[KeyHandle]) {
        self.certs.prefetch_some(certs)
    }
}

impl<'a> StoreUpdate<'a> for CertD<'a> {
    fn update_by(&mut self, cert: Arc<LazyCert<'a>>,
                 merge_strategy: &mut dyn MergeCerts<'a>)
        -> Result<Arc<LazyCert<'a>>>
    {
        tracer!(TRACE, "CertD::update_by");
        t!("Inserting {}", cert.fingerprint());

        // This is slightly annoying: cert-d expects bytes.  But
        // serializing cert is a complete waste if we have to merge
        // the certificate with another one.  cert-d actually only
        // needs the primary key, which it uses to derive the
        // fingerprint, so, we only serialize that.
        let fpr = cert.fingerprint();
        let fpr_str = format!("{:x}", fpr);

        let mut merged = None;
        self.certd.insert(&fpr_str, (), false, |(), disk_bytes| {
            let disk: Option<Arc<LazyCert>>
                = if let Some(disk_bytes) = disk_bytes
            {
                let mut parser = RawCertParser::from_bytes(disk_bytes)
                    .with_context(|| {
                        format!("Parsing {} as returned from the cert directory",
                                fpr)
                    })
                    .map_err(|err| {
                        t!("Reading disk version: {}", err);
                        err
                    })?;
                let disk = parser.next().transpose()
                    .with_context(|| {
                        format!("Parsing {} as returned from the cert directory",
                                fpr)
                    })
                    .map_err(|err| {
                        t!("Parsing disk version: {}", err);
                        err
                    })?;
                disk.map(|disk| Arc::new(LazyCert::from(disk)))
            } else {
                t!("No disk version");
                None
            };

            let merged_ = merge_strategy.merge_public(cert, disk)
                .with_context(|| {
                    format!("Merging versions of {}", fpr)
                })
                .map_err(|err| {
                    t!("Merging: {}", err);
                    err
                })?;
            let bytes = merged_.to_vec()?;
            merged = Some(merged_);
            Ok(cert_d::MergeResult::Data(bytes))
        })?;

        let merged = merged.expect("set");
        // Inserting into the in-memory index is infallible.
        if let Err(err) = self.certs.update(merged) {
            t!("Inserting {} into in-memory index: {}", fpr, err);
        }
        // Annoyingly, there is no easy way to get index to return a
        // reference to what it just inserted.
        Ok(self.certs.lookup_by_cert_fpr(&fpr).expect("just set"))
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    use anyhow::Context;

    use openpgp::packet::UserID;
    use openpgp::serialize::Serialize;

    use crate::store::StoreError;
    use crate::tests::print_error_chain;

    // Make sure that we can read a huge cert-d.  Specifically, the
    // typical file descriptor limit is 1024.  Make sure we can
    // initialize and iterate over a cert-d with a few more entries
    // than that.
    #[test]
    fn huge_cert_d() -> Result<()> {
        let path = tempfile::tempdir()?;
        let certd = cert_d::CertD::with_base_dir(&path)
            .map_err(|err| {
                let err = anyhow::Error::from(err)
                    .context(format!("While opening the certd {:?}", path));
                print_error_chain(&err);
                err
            })?;

        // Generate some certificates and write them to a cert-d using
        // the low-level interface.
        const N: usize = 1050;

        let mut certs = Vec::new();
        let mut certs_fpr = Vec::new();
        let mut subkeys_fpr = Vec::new();
        let mut userids = Vec::new();

        for i in 0..N {
            let userid = format!("<{}@example.org>", i);

            let (cert, _rev) = CertBuilder::new()
                .set_cipher_suite(CipherSuite::Cv25519)
                .add_userid(UserID::from(&userid[..]))
                .add_storage_encryption_subkey()
                .generate()
                .expect("ok");

            certs_fpr.push(cert.fingerprint());
            subkeys_fpr.extend(cert.keys().subkeys().map(|ka| ka.fingerprint()));
            userids.push(userid);

            let mut bytes = Vec::new();
            cert.serialize(&mut bytes).expect("can serialize to a vec");
            certd
                .insert_data(&bytes, false, |new, disk| {
                    assert!(disk.is_none());

                    Ok(cert_d::MergeResult::DataRef(new))
                })
                .with_context(|| {
                    format!("{:?} ({})", path, cert.fingerprint())
                })
                .expect("can insert");

            certs.push(cert);
        }

        // One subkey per certificate.
        assert_eq!(certs_fpr.len(), subkeys_fpr.len());

        certs_fpr.sort();

        // Open the cert-d and make sure we can read what we wrote via
        // the low-level interface.
        let certd = CertD::open(&path).expect("exists");

        // Test Store::iter.  In particular, make sure we get
        // everything back.
        let mut certs_read = certd.certs().collect::<Vec<_>>();
        assert_eq!(
            certs_read.len(), certs.len(),
            "Looks like you're exhausting the available file descriptors");

        certs_read.sort_by_key(|c| c.fingerprint());
        let certs_read_fpr
            = certs_read.iter().map(|c| c.fingerprint()).collect::<Vec<_>>();
        assert_eq!(certs_fpr, certs_read_fpr);

        // Test Store::by_cert.
        for cert in certs.iter() {
            let certs_read = certd.lookup_by_cert(&cert.key_handle()).expect("present");
            // We expect exactly one cert.
            assert_eq!(certs_read.len(), 1);
            let cert_read = certs_read.iter().next().expect("have one")
                .to_cert().expect("valid");
            assert_eq!(cert_read, cert);
        }

        for subkey in subkeys_fpr.iter() {
            let kh = KeyHandle::from(subkey.clone());
            match certd.lookup_by_cert(&kh) {
                Ok(certs) => panic!("Expected nothing, got {} certs", certs.len()),
                Err(err) => {
                    if let Some(&StoreError::NotFound(ref got))
                        = err.downcast_ref::<StoreError>()
                    {
                        assert_eq!(&kh, got);
                    } else {
                        panic!("Expected NotFound, got: {}", err);
                    }
                }
            }
        }

        // Test Store::lookup_by_cert_or_subkey.
        for fpr in certs.iter().map(|cert| cert.fingerprint())
            .chain(subkeys_fpr.iter().cloned())
        {
            let certs_read
                = certd.lookup_by_cert_or_subkey(&KeyHandle::from(fpr.clone())).expect("present");
            // We expect exactly one cert.
            assert_eq!(certs_read.len(), 1);
            let cert_read = certs_read.iter().next().expect("have one")
                .to_cert().expect("valid");

            assert!(cert_read.keys().any(|k| k.fingerprint() == fpr));
        }

        // Test Store::lookup_by_userid.
        for userid in userids.iter() {
            let userid = UserID::from(&userid[..]);

            let certs_read
                = certd.lookup_by_userid(&userid).expect("present");
            // We expect exactly one cert.
            assert_eq!(certs_read.len(), 1);
            let cert_read = certs_read.iter().next().expect("have one")
                .to_cert().expect("valid");

            assert!(cert_read.userids().any(|u| u.userid() == &userid));
        }

        Ok(())
    }
}
