#'Build a SAV project
#' 
#'Constructor to build a \link{savProject-class} object and populate it. A SAV 
#'project consists of binary files generated by an Illumina sequencing run
#'and placed in a folder named "InterOp". This folder contains a number
#'of ".bin" files that contain statistics about the run.  Creating
#'this object parses all of the files and makes the data available for analysis.
#' 
#'@param object String Path to Flowcell data
#'@export
#'@examples
#'fc <- savR(system.file("extdata", "MiSeq", package="savR"))
#'fc
#'
#'@rdname savR
setGeneric("savR", function(object) standardGeneric("savR"))

#'Get Flowcell folder location
#'
#'Accessor to obtain the path to data for a particular SAV project.
#'
#'@param project SAV project
#'@return normalized path to Illumina run data.
#'@export
#'@rdname location
#'@examples
#'example(savR)
#'location(fc)
setGeneric("location", function(project) standardGeneric("location"))

#'Get reads
#'
#'Accessor to obtain information about the reads of a particular Illumina
#'sequencing run.
#'
#'@param project SAV project
#'@return List of \link{illuminaRead-class} objects
#'@export
#'@rdname reads
#'@examples
#'example(savR)
#'reads(fc)
setGeneric("reads", function(project) standardGeneric("reads"))

#'Get flowcell layout
#'
#'Accessor to obtain information about the characteristics of the flowcell
#'from an Illumina sequencing run.
#'
#'@param project SAV project
#'@return \link{illuminaFlowCellLayout-class} object
#'@export
#'@rdname flowcellLayout
#'@examples
#'example(savR)
#'flowcellLayout(fc)
setGeneric("flowcellLayout", function(project) standardGeneric("flowcellLayout"))

#'Get the Run ID
#'
#'Accessor to obtain the string identifier of an Illumina sequencing run.
#'
#'@param project SAV project
#'@return parsed Illumina run id
#'@export
#'@rdname run
#'@examples
#'example(savR)
#'run(fc)
setGeneric("run", function(project) standardGeneric("run"))

#'Get the total number of cycles
#'
#'Accessor to obtain the total number of cycles sequenced in an Illumina sequencing run.
#'
#'@param project SAV project
#'@return total number of cycles in run, including all sequencing and index reads.
#'@export
#'@rdname cycles
#'@examples
#'example(savR)
#'cycles(fc)
setGeneric("cycles", function(project) standardGeneric("cycles"))

#'Get the number of sequence reads
#'
#'Returns the number of sequencing reads (excluding index reads).
#'
#'@param project SAV project
#'@return number of reads
#'@export
#'@rdname directions
#'@examples
#'example(savR)
#'directions(fc)
setGeneric("directions", function(project) standardGeneric("directions"))

#'Get Corrected Intensity data
#'
#'Returns a data frame of corrected intensity data.
#'
#' \describe{
#'  \item{\code{lane}:}{Lane number} \cr
#'  \item{\code{tile}:}{Tile ID} \cr
#'  \item{\code{cycle}:}{Cycle number} \cr
#'  \item{\code{avg_intensity}:}{Average intensity} \cr
#'  \item{\code{avg_cor_[ACGT]}:}{Average corrected intensity of channel A, C, G, or T} \cr
#'  \item{\code{avg_cor_called_[ACGT]}:}{Average corrected intensity for called clusters in channel A, C, G, or T} \cr
#'  \item{\code{num_\{none|[ACGT]\}}:}{Number of called bases for no-call, A, C, G, or T} \cr
#'  \item{\code{sig_noise}:}{Signal to noise ratio} \cr
#' }
#'
#'
#'@param project SAV project
#'@return sorted data.frame of CI data.
#'@export
#'@rdname correctedIntensities
#'@examples
#'example(savR)
#'colnames(correctedIntensities(fc))
setGeneric("correctedIntensities", function(project) standardGeneric("correctedIntensities"))

#'Get Quality Metrics data
#'
#'Quality metric by lane, tile and cycle.
#'
#' \describe{
#'  \item{\code{lane}:}{Lane number}
#'  \item{\code{tile}:}{Tile ID}
#'  \item{\code{cycle}:}{Cycle number}
#'  \item{\code{Q1-Q50}:}{Number of clusters with quality of indicated column}
#' }
#'
#'@param project SAV project
#'@return sorted data.frame of quality data
#'@export
#'@rdname qualityMetrics
#'@examples
#'example(savR)
#'colnames(qualityMetrics(fc))
setGeneric("qualityMetrics", function(project) standardGeneric("qualityMetrics"))

#'Get Tile Metrics
#'
#'Returns the Tile Metrics SAV data.  
#'
#'Metrics for each tile are encoded in the following format:
#'\tabular{ll}{
#'cluster density: \tab 100 \cr
#'PF cluster density: \tab 101 \cr
#'number of clusters: \tab 102 \cr
#'number of PF clusters: \tab 103 \cr
#'phasing for read N: \tab (200 + (N - 1) * 2) \cr
#'prephasing for read N: \tab (201 + (N - 1) * 2) \cr
#'percent aligned for read N: \tab (300 + N - 1) \cr
#'control lane: \tab 400 \cr
#'}
#'
#' \describe{
#'  \item{\code{lane}:}{Lane number}
#'  \item{\code{tile}:}{Tile ID}
#'  \item{\code{code}:}{Code described above}
#'  \item{\code{value}:}{Value for code key}
#' }
#'
#' 
#'@references
#'Codes for Tile Metrics were obtained from the Python Illuminate package: \cr
#'\url{https://bitbucket.org/invitae/illuminate}
#' 
#'
#'@param project SAV project
#'@return sorted data.frame of tile metrics
#'@export
#'@rdname tileMetrics
#'@examples
#'example(savR)
#'colnames(tileMetrics(fc))
setGeneric("tileMetrics", function(project) standardGeneric("tileMetrics"))

#'Get Extraction Metrics
#'
#'Extraction (intensity and FWHM) metrics for lane, tile, and cycle.
#'
#' \describe{
#'  \item{\code{lane}:}{Lane number}
#'  \item{\code{tile}:}{Tile ID}
#'  \item{\code{cycle}:}{Cycle number}
#'  \item{\code{FWHM_[ACGT]}:}{Full width at half maximum for A, C, G, or T}
#'  \item{\code{int_[ACGT]}:}{Intensity of channel A, C, G, or T}
#'  \item{\code{datestamp}:}{Time/date stamp}
#' }
#'
#'
#'@param project SAV project
#'@return sorted data.frame of Extraction metrics
#'@export
#'@rdname extractionMetrics
#'@examples
#'example(savR)
#'colnames(extractionMetrics(fc))
setGeneric("extractionMetrics", function(project) standardGeneric("extractionMetrics"))


#'Get Error Metrics
#'
#'Error metrics for lane, tile, and cycle.
#'
#' \describe{
#'  \item{\code{lane}:}{Lane number}
#'  \item{\code{tile}:}{Tile ID}
#'  \item{\code{cycle}:}{Cycle number}
#'  \item{\code{errorrate}:}{Error rate}
#'  \item{\code{nPerfect}:}{number of perfect reads}
#'  \item{\code{n[1-4]Error}:}{Number of reads with 1, 2, 3 and 4 errors}
#' }
#'
#'
#'@param project SAV project
#'@return sorted data.frame of Error metrics
#'@export
#'@rdname errorMetrics
#'@examples
#'example(savR)
#'colnames(extractionMetrics(fc))
setGeneric("errorMetrics", function(project) standardGeneric("errorMetrics"))

#'Plot flowcell intensity by base and cycle
#' 
#'Draws a representation of a flowcell, showing the average corrected called intensity values.
#' 
#'@param project A \link{savProject-class} object
#'@param cycle integer cycle number
#'@param base character for nucleotide
#' 
#'@export
#'@docType methods
#'@rdname plotIntensity
setGeneric("plotIntensity", function(project, cycle, base) standardGeneric("plotIntensity"))

#'Generate FWHM plots
#'
#'Plots the average full width of clusters at half maximum (FWHM) of each tile
#'for a given cycle and base.
#'
#'@param project SAV project
#'@param cycle sequence cycle
#'@param base nucleotide base (ACGT)
#'@export
#'@docType methods
#'@rdname plotFWHM
setGeneric("plotFWHM", function(project, cycle, base) standardGeneric("plotFWHM"))

#'Plot Quality > 30 for a flowcell
#'
#'Generate a plot for a given cycle of the percentage of clusters in each tile
#'that are >= Q30.
#'
#'@param project SAV project
#'@param cycle sequence cycle
#'@export
#'@docType methods
#'@rdname plotQGT30
setGeneric("plotQGT30", function(project, cycle) standardGeneric("plotQGT30"))

#'PF Boxplot
#'
#'Generate a boxplot of the numbers of clusters and the number of
#'Illumina pass-filter clusters per tile and lane
#'
#'@param project SAV project
#'@export 
#'@docType methods
#'@rdname pfBoxplot
setGeneric("pfBoxplot", function(project) standardGeneric("pfBoxplot"))

#'Generate a heatmap of qualities
#'
#'Plots a heatmap of quality vs cycle for a given lane for 1 or more sequence reads.  Read qualities include sequence + index.
#'
#'@param project SAV project
#'@param lane integer lane specification
#'@param read integer vector of sequence reads to include (not including index reads)
#'@param collapse whether or not to collapse index reads into the preceeding read (# reads = directions), default TRUE
#'@export
#'@docType methods
#'@rdname qualityHeatmap
setGeneric("qualityHeatmap", function(project, lane, read, collapse) standardGeneric("qualityHeatmap") )

#'Generate Illumina reports folder
#'
#'Generate a folder of images that approximates the format of the folder that 
#'was superceded by InterOp. Requires the Cairo package.
#'
#'@param project SAV project
#'@param destination path to save reports folder
#'@export
#'@docType methods
#'@rdname buildReports
#'@examples
#'\dontrun{
#'example(savR)
#'buildReports(fc, "reports")
#'}
setGeneric("buildReports", function(project, destination) standardGeneric("buildReports"))

#'Get number of clusters per lane
#'
#'Sum the total number of clusters for all tiles in the lane.
#'
#'@param project SAV project
#'@param lane lane(s) number
#'@export
#'@docType methods
#'@rdname clusters
#'@examples
#'\dontrun{
#'example(savR)
#'clusters(fc, 1L)
#'} 
setGeneric("clusters", function(project, lane) standardGeneric("clusters"))

#'Get number of PF clusters per lane
#'
#'Sum the total pass filter number of clusters for all tiles in the lane.
#'
#'@param project SAV project
#'@param lane lane(s) number
#'@export
#'@docType methods
#'@rdname pfClusters
#'@examples
#'\dontrun{
#'example(savR)
#'pfClusters(fc, 1L)
#'} 
setGeneric("pfClusters", function(project, lane) standardGeneric("pfClusters"))

#'Get the proportion of clusters over a specified quality threshold 
#'
#'Return the ratio of clusters with a quality score less than or equal to 
#'a specified value (n) for the requested lanes and cycles.
#'
#'@param project SAV project
#'@param lane lane(s) number
#'@param cycle cycle(s) number
#'@param n quality threshold
#'@export
#'@docType methods
#'@rdname clusterQualityGtN
#'@examples
#'\dontrun{
#'example(savR)
#'clusterQualityGtN(fc, 1L, 25L, 30L)
#'}
setGeneric("clusterQualityGtN", function(project, lane, cycle, n) standardGeneric("clusterQualityGtN"))
