#### PATTERN | WEB | LOCALE ########################################################################
# -*- coding: utf-8 -*-
# Copyright (c) 2010 University of Antwerp, Belgium
# Author: Tom De Smedt <tom@organisms.be>
# License: BSD (see LICENSE.txt for details).
# http://www.clips.ua.ac.be/pages/pattern

#### LANGUAGE & REGION #############################################################################
# IETF BCP 47 language-region code => (language, region, ISO-639 language code, ISO-3166 region code).
# Note: the list is incomplete (especially for African languages).
# Please help out by correcting errors and omissions.

LANGUAGE_REGION = {
    u'aa-ET': (u'Afar', u'Ethiopia', u'aa', u'ET'),
    u'af-ZA': (u'Afrikaans', u'South Africa', u'af', u'ZA'),
    u'ar-AE': (u'Arabic', u'United Arab Emirates', u'ar', u'AE'),
    u'ar-BH': (u'Arabic', u'Bahrain', u'ar', u'BH'),
    u'ar-DZ': (u'Arabic', u'Algeria', u'ar', u'DZ'),
    u'ar-EG': (u'Arabic', u'Egypt', u'ar', u'EG'),
    u'ar-IQ': (u'Arabic', u'Iraq', u'ar', u'IQ'),
    u'ar-JO': (u'Arabic', u'Jordan', u'ar', u'JO'),
    u'ar-KW': (u'Arabic', u'Kuwait', u'ar', u'KW'),
    u'ar-LB': (u'Arabic', u'Lebanon', u'ar', u'LB'),
    u'ar-LY': (u'Arabic', u'Libya', u'ar', u'LY'),
    u'ar-MA': (u'Arabic', u'Morocco', u'ar', u'MA'),
    u'ar-OM': (u'Arabic', u'Oman', u'ar', u'OM'),
    u'ar-QA': (u'Arabic', u'Qatar', u'ar', u'QA'),
    u'ar-SA': (u'Arabic', u'Saudi Arabia', u'ar', u'SA'),
    u'ar-SD': (u'Arabic', u'Sudan', u'ar', u'SD'),
    u'ar-SY': (u'Arabic', u'Syria', u'ar', u'SY'),
    u'ar-TN': (u'Arabic', u'Tunisia', u'ar', u'TN'),
    u'ar-YE': (u'Arabic', u'Yemen', u'ar', u'YE'),
    u'be-BY': (u'Belarusian', u'Belarus', u'be', u'BY'),
    u'bg-BG': (u'Bulgarian', u'Bulgaria', u'bg', u'BG'),
    u'ca-AD': (u'Catalan', u'Andorra', u'ca', u'AD'),
    u'cs-CZ': (u'Czech', u'Czech Republic', u'cs', u'CZ'),
    u'da-DK': (u'Danish', u'Denmark', u'da', u'DK'),
    u'de-DE': (u'German', u'Germany', u'de', u'DE'),
    u'de-AT': (u'German', u'Austria', u'de', u'AT'),
    u'de-CH': (u'German', u'Switzerland', u'de', u'CH'),
    u'de-LI': (u'German', u'Liechtenstein', u'de', u'LI'),
    u'de-LU': (u'German', u'Luxembourg', u'de', u'LU'),
    u'el-GR': (u'Greek', u'Greece', u'el', u'GR'),
    u'en-AU': (u'English', u'Australia', u'en', u'AU'),
    u'en-BZ': (u'English', u'Belize', u'en', u'BZ'),
    u'en-CA': (u'English', u'Canada', u'en', u'CA'),
    u'en-GB': (u'English', u'United Kingdom', u'en', u'GB'),
    u'en-IE': (u'English', u'Ireland', u'en', u'IE'),
    u'en-JM': (u'English', u'Jamaica', u'en', u'JM'),
    u'en-KE': (u'English', u'Kenya', u'en', u'KE'),
    u'en-NZ': (u'English', u'New Zealand', u'en', u'NZ'),
    u'en-TT': (u'English', u'Trinidad', u'en', u'TT'),
    u'en-US': (u'English', u'United States', u'en', u'US'),
    u'en-ZA': (u'English', u'South Africa', u'en', u'ZA'),
    u'es-ES': (u'Spanish', u'Spain', u'es', u'ES'),
    u'es-AR': (u'Spanish', u'Argentina', u'es', u'AQ'),
    u'es-BO': (u'Spanish', u'Bolivia', u'es', u'BO'),
    u'es-CL': (u'Spanish', u'Chile', u'es', u'CL'),
    u'es-CO': (u'Spanish', u'Colombia', u'es', u'CO'),
    u'es-CR': (u'Spanish', u'Costa Rica', u'es', u'CR'),
    u'es-DO': (u'Spanish', u'Dominican Republic', u'es', u'DO'),
    u'es-EC': (u'Spanish', u'Ecuador', u'es', u'EC'),
    u'es-GT': (u'Spanish', u'Guatemala', u'es', u'GT'),
    u'es-HN': (u'Spanish', u'Honduras', u'es', u'HN'),
    u'es-MX': (u'Spanish', u'Mexico', u'es', u'MX'),
    u'es-NI': (u'Spanish', u'Nicaragua', u'es', u'NI'),
    u'es-PA': (u'Spanish', u'Panama', u'es', u'PA'),
    u'es-PE': (u'Spanish', u'Peru', u'es', u'PE'),
    u'es-PR': (u'Spanish', u'Puerto Rico', u'es', u'PR'),
    u'es-PY': (u'Spanish', u'Paraguay', u'es', u'PY'),
    u'es-SV': (u'Spanish', u'El Salvador', u'es', u'SV'),
    u'es-UY': (u'Spanish', u'Uruguay', u'es', u'UY'),
    u'es-VE': (u'Spanish', u'Venezuela', u'es', u'VE'),
    u'et-EE': (u'Estonian', u'Estonia', u'et', u'EE'),
    u'eu-PV': (u'Basque', u'Basque Country', u'eu', u'PV'),
    u'fa-IR': (u'Farsi', u'Iran', u'fa', u'IR'),
    u'fi-FI': (u'Finnish', u'Finland', u'fi', u'FI'),
    u'fo-FO': (u'Faeroese', u'Faroe Islands', u'fo', u'FO'),
    u'fr-CG': (u'French', u'Congo', u'fr', u'CG'),
    u'fr-FR': (u'French', u'France', u'fr', u'FR'),
    u'fr-BE': (u'French', u'Belgium', u'fr', u'BE'),
    u'fr-CA': (u'French', u'Canada', u'fr', u'CA'),
    u'fr-CH': (u'French', u'Switzerland', u'fr', u'CH'),
    u'fr-LU': (u'French', u'Luxembourg', u'fr', u'LU'),
    u'ga-IE': (u'Irish' , u'Ireland', u'ga', u'IE'),
    u'gd-UK': (u'Gaelic', u'Scotland', u'gd', u'UK'),
    u'he-IL': (u'Hebrew', 'Israel', u'he', u'IL'),
    u'hi-IN': (u'Hindi', u'India', u'hi', u'IN'),
    u'hr-HR': (u'Croatian', u'Croatia', u'hr', u'HR'),
    u'hu-HU': (u'Hungarian', u'Hungary', u'hu', u'HU'),
    u'id-ID': (u'Indonesian', u'Indonesia', u'id', u'ID'),
    u'is-IS': (u'Icelandic', u'Iceland', u'is', u'IS'),
    u'it-IT': (u'Italian', u'Italy', u'it', u'IT'),
    u'it-CH': (u'Italian', u'Switzerland', u'it', u'CH'),
    u'ja-JA': (u'Japanese', u'Japan', u'ja', u'JA'),
    u'ka-GE': (u'Georgian', u'Georgia', u'ka', u'GE'),
    u'kg-CG': (u'Kongo', u'Congo', u'kg', u'CG'),
    u'kl-GL': (u'Kalaallisut', u'Greenland', u'kl', u'GL'),
    u'ko-KP': (u'Korean', u'Johab', u'ko', u'KP'),
    u'ln-CG': (u'Lingala', u'Congo', u'ln', u'CG'),
    u'lo-LA': (u'Lao', u'Lao', u'lo', u'LA'),
    u'lt-LT': (u'Lithuanian', u'Lithuania', u'lt', u'LT'),
    u'lv-LV': (u'Latvian', u'Latvia', u'lv', u'LV'),
    u'mk-ML': (u'Macedonian', u'Macedonia', u'mk', u'MK'),
    u'ms-MY': (u'Malaysian', u'Malaysia', u'ms', u'MY'),
    u'mt-MT': (u'Maltese', u'Malta', u'mt', u'MT'),
    u'nd-ZW': (u'Ndebele', u'Zimbabwe', u'nd', u'ZW'),
    u'nl-NL': (u'Dutch', u'Netherlands', u'nl', u'NL'),
    u'nl-BE': (u'Dutch', u'Belgium', u'nl', u'BE'),
    u'no-NO': (u'Norwegian', u'Nynorsk', u'no', u'NO'),
    u'om-ET': (u'Oromo', u'Ethiopia', u'om', u'ET'),
    u'om-KE': (u'Oromo', u'Kenya', u'om', u'KE'),
    u'pl-PL': (u'Polish', u'Poland', u'pl', u'PL'),
    u'pt-MZ': (u'Portuguese', u'Mozambique', u'pt', u'PT'),
    u'pt-PT': (u'Portuguese', u'Portugal', u'pt', u'PT'),
    u'pt-BR': (u'Portuguese', u'Brazil', u'pt', u'BR'),
    u'rm-IT': (u'Rhaeto-Romanic', u'Italy', u'rm', u'IT'),
    u'ro-RO': (u'Romanian', u'Romania', u'ro', u'RO'),
    u'ro-MO': (u'Romanian', u'Republic of Moldova', u'ro', u'MO'),
    u'ru-RU': (u'Russian', u'Russia', u'ru', u'RU'),
    u'rw-RW': (u'Kinyarwanda', u'Rwanda', u'rw', u'RW'),
    u'sk-SK': (u'Slovak', u'Slovakia', u'sk', u'SK'),
    u'sl-SI': (u'Slovenian', u'Slovenia', u'sl', u'SI'),
    u'sm-SM': (u'Samoan', u'Samoa', u'sm', 'SM'),
    u'so-KE': (u'Somali', u'Kenya', u'so', u'KE'),
    u'so-SO': (u'Somali', u'Somalia', u'so', u'SO'),
    u'sq-AL': (u'Albanian', u'Albania', u'sq', u'AL'),
    u'sr-RS': (u'Serbian', u'Serbia', u'sr', u'RS'),
    u'sv-SE': (u'Swedish', u'Sweden', u'sv', u'SE'),
    u'sw-SW': (u'Swahili', u'Kenya', u'sw', u'KE'),
    u'sw-TZ': (u'Swahili', u'Tanzania', u'sw', u'TZ'),
    u'sv-FI': (u'Swedish', u'Finland', u'sv', u'FI'),
    u'sx-ZA': (u'Sotho', u'South Africa', u'sx', u'ZA'),
    u'sz-FI': (u'Sami', u'Sapmi', u'sz', u'FI'),
    u'th-TH': (u'Thai', u'Thailand', u'th', u'TH'),
    u'tn-BW': (u'Tswana', u'Botswana', u'tn', u'BW'),
    u'to-TO': (u'Tonga', u'Tonga', u'to', u'TO'),
    u'tr-TR': (u'Turkish', u'Turkey', u'tr', u'TR'),
    u'ts-ZA': (u'Tsonga', u'South Africa', u'ts', u'ZA'),
    u'uk-UA': (u'Ukrainian', u'Ukraine', u'uk', u'UA'),
    u'ur-PK': (u'Urdu', u'Pakistan', u'ur', u'PK'),
    u'uz-UZ': (u'Uzbek', u'Uzbekistan', u'uz', u'UZ'),
    u've-ZA': (u'Venda', u'South Africa', u've', u'ZA'),
    u'vi-VN': (u'Vietnamese', u'Vietnam', u'vi', u'VN'),
    u'xh-ZA': (u'Xhosa', u'South Africa', u'xh', u'ZA'),
    u'zh-CN': (u'Chinese', u'China', u'zh', u'CN'),
    u'zh-HK': (u'Chinese', u'Hong Kong', u'zh', u'HK'),
    u'zh-SG': (u'Chinese', u'Singapore', u'zh', u'SG'),
    u'zh-TW': (u'Chinese', u'Taiwan', u'zh', u'TW'),
    u'zu-ZA': (u'Zulu', u'South Africa', u'zu', u'ZA'),
    u'zu-ZW': (u'Zulu', u'Zimbabwe', u'zu', u'ZW')
}

def encode_language(name):
    """ Returns the language code for the given language name.
        For example: encode_language("dutch") => "nl".
    """
    for tag, (language, region, iso639, iso3166) in LANGUAGE_REGION.items():
        if language == name.capitalize():
            return iso639

def decode_language(code):
    """ Returns the language name for the given language code.
        For example: decode_language("nl") => "Dutch".
    """
    for tag, (language, region, iso639, iso3166) in LANGUAGE_REGION.items():
        if iso639 == code.lower(): 
            return language

def encode_region(name):
    """ Returns the region code for the given region name.
        For example: encode_region("belgium") => "BE".
    """
    for tag, (language, region, iso639, iso3166) in LANGUAGE_REGION.items():
        if region == name.capitalize():
            return iso3166

def decode_region(code):
    """ Returns the region name for the given region code.
        For example: decode_region("be") => "Belgium".
    """
    for tag, (language, region, iso639, iso3166) in LANGUAGE_REGION.items():
        if iso3166 == code.upper():
            return region
        
def languages(region):
    """ Returns a list of language codes for the given region code.
        For example: languages(encode_region("belgium")) => ["fr", "nl"]
    """
    v, a = region.upper(), []
    for tag, (language, region, iso639, iso3166) in LANGUAGE_REGION.items():
        if iso3166 == v:
            a.append(iso639)
    return sorted(a)

def regions(language):
    """ Returns a list of region codes for the given language code.
        For example: regions(encode_language("dutch")) => ["NL", "BE"]
    """
    x, a = language.lower(), []
    for tag, (language, region, iso639, iso3166) in LANGUAGE_REGION.items():
        if iso639 == x: 
            a.append(iso3166)
    return sorted(a, key=lambda tag: tag.lower() != x and tag or "")

def regionalize(language):
    """ Returns a list of RFC-5646 language-region codes for the given language code.
        For example: regionalize("nl") => ["nl-nl", "nl-be"]
    """
    if not isinstance(language, basestring):
        return []
    if "-" in language:
        language, region = language.split("-")
        return [language.lower() + "-" + region.upper()]  # nl-nl => nl-NL
    main = lambda tag: tag in ("ar-AE", "en-US", "zh-CN") or tag[:2] == tag[3:].lower() # nl-NL
    a = [language+"-"+r for r in regions(language.lower())]
    a = sorted(a, key=main, reverse=True)
    return a

def market(language):
    """ Returns the first item from regionalize(language).
    """
    a = regionalize(language)
    a = len(a) > 0 and a[0] or None
    return a

#print(encode_language("dutch")) # nl
#print(decode_language("nl"))    # Dutch
#print(encode_region("belgium")) # BE
#print(decode_region("be"))      # Belgium
#print(languages("be"))          # ["fr", "nl"]
#print(regions("nl"))            # ["NL", "BE"]
#print(regionalize("nl"))        # ["nl-NL", "nl-BE"]

### GEOCODE ########################################################################################
# capital => (latitude, longitude, ISO-639 language code, region)

GEOCODE = {
         u'Abu Dhabi': ( 24.467,  54.367, u"ar", u"United Arab Emirates"),
             u'Abuja': (  9.083,   7.533, u"en", u"Nigeria"),
             u'Accra': (  5.550,  -0.217, u"en", u"Ghana"),
           u'Algiers': ( 36.750,   3.050, u"ar", u"Algeria"),
             u'Amman': ( 31.950,  35.933, u"ar", u"Jordan"),
         u'Amsterdam': ( 52.383,   4.900, u"nl", u"Netherlands"),
            u'Ankara': ( 39.933,  32.867, u"tr", u"Turkey"),
            u'Astana': ( 51.167,  71.417, u"ru", u"Kazakhstan"),
          u'Asuncion': (-25.267, -57.667, u"es", u"Paraguay"),
            u'Athens': ( 37.983,  23.733, u"el", u"Greece"),
           u'Baghdad': ( 33.333,  44.383, u"ar", u"Iraq"),
            u'Bamako': ( 12.650,  -8.000, u"fr", u"Mali"),
           u'Bangkok': ( 13.750, 100.517, u"th", u"Thailand"),
            u'Bangui': (  4.367,  18.583, u"fr", u"Central African Republic"),
           u'Beijing': ( 39.917, 116.383, u"zh", u"China"),
            u'Beirut': ( 33.867,  35.500, u"ar", u"Lebanon"),
          u'Belgrade': ( 44.833,  20.500, u"sr", u"Serbia"),
            u'Berlin': ( 52.517,  13.400, u"de", u"Germany"),
              u'Bern': ( 46.950,   7.433, u"de", u"Switzerland"),
            u'Bissau': ( 11.850, -15.583, u"pt", u"Guinea"),
            u'Bogota': (  4.600, -74.083, u"es", u"Colombia"),
          u'Brasilia': (-15.783, -47.917, u"pt", u"Brazil"),
        u'Bratislava': ( 48.150,  17.117, u"sk", u"Slovakia"),
       u'Brazzaville': ( -4.250,  15.283, u"fr", u"Congo"),
          u'Brussels': ( 50.833,   4.333, u"nl", u"Belgium"),
         u'Bucharest': ( 44.433,  26.100, u"ro", u"Romania"),
          u'Budapest': ( 47.500,  19.083, u"hu", u"Hungary"),
      u'Buenos Aires': (-34.600, -58.667, u"es", u"Argentina"),
         u'Bujumbura': ( -3.367,  29.350, u"rn", u"Burundi"),
             u'Cairo': ( 30.050,  31.250, u"ar", u"Egypt"),
          u'Canberra': (-35.283, 149.217, u"en", u"Australia"),
           u'Caracas': ( 10.500, -66.933, u"es", u"Venezuela"),
          u'Chisinau': ( 47.000,  28.850, u"ro", u"Moldova"),
           u'Colombo': (  6.933,  79.850, u"si", u"Sri Lanka"),
           u'Conakry': (  9.550, -13.700, u"fr", u"Guinea"),
        u'Copenhagen': ( 55.667,  12.583, u"da", u"Denmark"),
             u'Dakar': ( 24.633,  46.717, u"fr", u"Senegal"),
          u'Damascus': ( 33.500,  36.300, u"ar", u"Syria"),
     u'Dar es Salaam': ( -6.800,  39.283, u"sw", u"Tanzania"),
             u'Dhaka': ( 23.717,  90.400, u"bn", u"Bangladesh"),
            u'Dublin': ( 53.317,  -6.233, u"en", u"Ireland"),
          u'Freetown': (  8.500, -13.250, u"en", u"Sierra Leone"),
       u'George Town': ( 19.300, -81.383, u"en", u"Malaysia"),
        u'Georgetown': (  6.800, -58.167, u"en", u"Guyana"),
    u'Guatemala City': ( 14.617, -90.517, u"es", u"Guatemala"),
             u'Hanoi': ( 21.033, 105.850, u"vi", u"Vietnam"),
            u'Harare': (-17.833,  31.050, u"en", u"Zimbabwe"),
            u'Havana': ( 23.117, -82.350, u"es", u"Cuba"),
          u'Helsinki': ( 60.167,  24.933, u"fi", u"Finland"),
         u'Islamabad': ( 33.700,  73.167, u"ur", u"Pakistan"),
           u'Jakarta': ( -6.167, 106.817, u"ms", u"Indonesia"),
         u'Jerusalem': ( 31.767,  35.233, u"he", u"Israel"),
              u'Juba': (  4.850,  31.617, u"en", u"Sudan"),
             u'Kabul': ( 34.517,  69.183, u"fa", u"Afghanistan"),
           u'Kampala': (  0.317,  32.417, u"en", u"Uganda"),
         u'Kathmandu': ( 27.717,  85.317, u"ne", u"Nepal"),
          u'Khartoum': ( 15.600,  32.533, u"ar", u"Sudan"),
              u'Kiev': ( 50.433,  30.517, u"rw", u"Ukraine"),
            u'Kigali': ( -1.950,  30.067, u"en", u"Rwanda"),
          u'Kingston': ( 18.000, -76.800, u"fr", u"Jamaica"),
          u'Kinshasa': ( -4.317,  15.300, u"ms", u"Congo"),
      u'Kuala Lumpur': (  3.167, 101.700, u"ar", u"Malaysia"),
       u'Kuwait City': ( 29.367,  47.967, u"uk", u"Kuwait"),
            u'La Paz': (-16.500, -68.150, u"es", u"Bolivia"),
              u'Lima': (-12.050, -77.050, u"es", u"Peru"),
            u'Lisbon': ( 38.717,  -9.133, u"pt", u"Portugal"),
         u'Ljubljana': ( 46.050,  14.517, u"sl", u"Slovenia"),
              u'Lome': (  6.133,   1.217, u"fr", u"Togo"),
            u'London': ( 51.500,  -0.167, u"en", u"United Kingdom"),
            u'Luanda': ( -8.833,  13.233, u"pt", u"Angola"),
            u'Lusaka': (-15.417,  28.283, u"en", u"Zambia"),
        u'Luxembourg': ( 49.600,   6.117, u"cd", u"Luxembourg"),
            u'Madrid': ( 40.400,  -3.683, u"es", u"Spain"),
           u'Managua': ( 12.150, -86.283, u"es", u"Nicaragua"),
            u'Manila': ( 14.583, 121.000, u"tl", u"Philippines"),
            u'Maputo': (-25.950,  32.583, u"pt", u"Mozambique"),
       u'Mexico City': ( 19.433, -99.133, u"es", u"Mexico"),
             u'Minsk': ( 53.900,  27.567, u"be", u"Belarus"),
         u'Mogadishu': (  2.067,  45.367, u"so", u"Somalia"),
            u'Monaco': ( 43.733,   7.417, u"fr", u"Monaco"),
          u'Monrovia': (  6.300, -10.800, u"en", u"Liberia"),
        u'Montevideo': (-34.883, -56.183, u"es", u"Uruguay"),
            u'Moscow': ( 55.750,  37.583, u"ru", u"Russia"),
            u'Muscat': ( 23.617,  58.583, u"ar", u"Oman"),
           u'Nairobi': ( -1.283,  36.817, u"en", u"Kenya"),
            u'Nassau': ( 25.083, -77.350, u"en", u"Bahamas"),
         u'New Delhi': ( 28.600,  77.200, u"hi", u"India"),
          u'New York': ( 40.756, -73.987, u"en", u"United States"),
            u'Niamey': ( 13.517,   2.117, u"fr", u"Niger"),
              u'Oslo': ( 59.917,  10.750, u"no", u"Norway"),
            u'Ottawa': ( 45.417, -75.700, u"en", u"Canada"),
       u'Panama City': (  8.967, -79.533, u"es", u"Panama"),
             u'Paris': ( 48.867,   2.333, u"fr", u"France"),
       u'Philipsburg': ( 18.017, -63.033, u"en", u"Sint Maarten"),
        u'Phnom Penh': ( 11.550, 104.917, u"km", u"Cambodia"),
        u'Port Louis': (-20.150,  57.483, u"en", u"Mauritius"),
    u'Port-au-Prince': ( 18.533, -72.333, u"fr", u"Haiti"),
        u'Porto-Novo': (  6.483,   2.617, u"fr", u"Benin"),
            u'Prague': ( 50.083,  14.467, u"cs", u"Czech Republic"),
          u'Pretoria': (-25.700,  28.217, u"xh", u"South Africa"),
         u'Pyongyang': ( 39.017, 125.750, u"ko", u"North Korea"),
             u'Quito': ( -0.217, -78.500, u"es", u"Ecuador"),
             u'Rabat': ( 34.017,  -6.817, u"ar", u"Morocco"),
           u'Rangoon': ( 16.800,  96.150, u"my", u"Myanmar"),
         u'Reykjavik': ( 64.150, -21.950, u"is", u"Iceland"),
              u'Riga': ( 56.950,  24.100, u"lv", u"Latvia"),
            u'Riyadh': ( 24.633,  46.717, u"ar", u"Saudi Arabia"),
              u'Rome': ( 41.900,  12.483, u"it", u"Italy"),
            u'Saipan': ( 15.200, 145.750, u"en", u"Saipan"),
          u'San Jose': (  9.933, -84.083, u"es", u"Costa Rica"),
          u'San Juan': ( 18.467, -66.117, u"es", u"Puerto Rico"),
        u'San Marino': ( 43.933,  12.417, u"it", u"San Marino"),
      u'San Salvador': ( 13.700, -89.200, u"es", u"El Salvador"),
             u'Sanaa': ( 15.350,  44.200, u"ar", u"Yemen"),
          u'Santiago': (-33.450, -70.667, u"es", u"Chile"),
     u'Santo Domingo': ( 18.467, -69.900, u"es", u"Domenican Republic"),
          u'Sarajevo': ( 43.867,  18.417, u"bo", u"Bosnia and Herzegovina"),
             u'Seoul': ( 37.550, 126.983, u"ko", u"South Korea"),
         u'Singapore': (  1.283, 103.850, u"en", u"Singapore"),
            u'Skopje': ( 42.000,  21.433, u"mk", u"Macedonia"),
             u'Sofia': ( 42.683,  23.317, u"bg", u"Bulgaria"),
         u'Stockholm': ( 59.333,  18.050, u"sv", u"Sweden"),
            u'Taipei': ( 25.050, 121.500, u"zh", u"China"),
           u'Tallinn': ( 59.433,  24.717, u"et", u"Estonia"),
          u'Tashkent': ( 41.333,  69.300, u"uz", u"Uzbekistan"),
       u'Tegucigalpa': ( 14.100, -87.217, u"es", u"Honduras"),
            u'Tehran': ( 35.667,  51.417, u"fa", u"Iran"),
            u'Tirana': ( 41.317,  19.817, u"sq", u"Albania"),
             u'Tokyo': ( 35.683, 139.750, u"ja", u"Japan"),
          u'Torshavn': ( 62.017,  -6.767, u"fo", u"Faroe Islands"),
           u'Tripoli': ( 32.883,  13.167, u"ar", u"Libya"),
             u'Tunis': ( 36.800,  10.183, u"ar", u"Tunis"),
             u'Vaduz': ( 47.133,   9.517, u"de", u"Liechtenstein"),
      u'Vatican City': ( 41.900,  12.450, u"it", u"Vatican City"),
            u'Vienna': ( 48.200,  16.367, u"de", u"Austria"),
         u'Vientiane': ( 17.967, 102.600, u"lo", u"Laos"),
           u'Vilnius': ( 54.683,  25.317, u"lt", u"Lithuania"),
            u'Warsaw': ( 52.250,  21.000, u"pl", u"Poland"),
       u'Washington.': ( 38.883, -77.033, u"en", u"United States"),
        u'Wellington': (-41.467, 174.850, u"en", u"New Zealand"),
      u'Yamoussoukro': (  6.817,  -5.283, u"fr", u"Côte d'Ivoire"),
           u'Yaounde': (  3.867,  11.517, u"en", u"Cameroon"),
            u'Zagreb': ( 45.800,  16.000, u"hr", u"Croatia")
}

def geocode(location):
    """ Returns a (latitude, longitude, language code, region)-tuple 
        for the given city (mostly capitals).
    """
    if location in GEOCODE:
        return GEOCODE[location]
    for k, v in GEOCODE.items():
        if location.lower() == k.lower():
            return v
