"""
The following contains a database of small molecules

Data for the G2/97 database are from
Raghavachari, Redfern, and Pople, J. Chem. Phys. Vol. 106, 1063 (1997).
See http://www.cse.anl.gov/Catalysis_and_Energy_Conversion/Computational_Thermochemistry.shtml for the original files.

All numbers are experimental values, except for coordinates, which are
MP2(full)/6-31G(d) optimized geometries (from http://www.cse.anl.gov/OldCHMwebsiteContent/compmat/G2-97.htm)

Atomic species:
ref: Curtiss et al. JCP 106, 1063 (1997).
'Enthalpy' is the experimental enthalpies of formation at 0K
'thermal correction' is the thermal corrections H(298)-H(0)

Molecular species:
ref: Staroverov et al. JCP 119, 12129 (2003)
'Enthalpy' is the experimental enthalpies of formation at 298K
'ZPE' is the zero-point energies
'thermal correction' is the thermal enthalpy corrections H(298K) - H_exp(0K)
ZPE and thermal corrections are estimated from B3LYP geometries and vibrations.

For details about G2-1 and G2-2 sets see doi:10.1063/1.477422.

Experimental ionization potentials are from http://srdata.nist.gov/cccbdb/
Information presented on these pages is considered public information
and may be distributed or copied http://www.nist.gov/public_affairs/disclaimer.cfm
"""

from ase.atoms import Atoms, string2symbols

atom_names = ['H','B','C','N','O','F','Al','Si','S','Cl']

molecule_names = ['BF3','BCl3','AlF3','AlCl3','CF4','CCl4','OCS','CS2','COF2','SiF4','SiCl4','N2O','ClNO','NF3','PF3','O3','F2O','ClF3','C2F4','C2Cl4','CF3CN','C3H4_C3v','C3H4_D2d','C3H4_C2v','C3H6_Cs','C3H6_D3h','C3H8','butadiene','2-butyne','methylenecyclopropane','bicyclobutane','cyclobutene','cyclobutane','isobutene','trans-butane','isobutane','C5H8','C6H6','H2CF2','HCF3','H2CCl2','HCCl3','H3CNH2','CH3CN','CH3NO2','CH3ONO','CH3SiH3','HCOOH','HCOOCH3','CH3CONH2','CH2NHCH2','NCCN','C2H6NH','CH3CH2NH2','H2CCO','CH2OCH2','CH3CHO','OCHCHO','CH3CH2OH','CH3OCH3','CH2SCH2','C2H6SO','CH3CH2SH','CH3SCH3','H2CCHF','CH3CH2Cl','H2CCHCl','H2CCHCN','CH3COCH3','CH3COOH','CH3COF','CH3COCl','C3H7Cl','C2H6CHOH','CH3CH2OCH3','C3H9N','C4H4O','C4H4S','C4H4NH','C5H5N','H2','SH','CCH','C2H3','CH3CO','H2COH','CH3O','CH3CH2O','CH3S','C2H5','C3H7','C3H9C','NO2']

data = {
'B': {
    'name': 'Boron',
    'database': 'G2-2',
    'symbols': 'B',
    'magmoms': [1.],
    'charges': None,
    'enthalpy': 136.20,
    'thermal correction': 0.29,
    'ionization energy': 8.30,
    'positions': [[ 0.  ,  0.  ,  0.]],
    },
'Al': {
    'name': 'Aluminium',
    'database': 'G2-2',
    'symbols': 'Al',
    'magmoms': [1.],
    'charges': None,
    'enthalpy': 78.23,
    'thermal correction': 1.08,
    'ionization energy': 5.99,
    'positions': [[ 0.  ,  0.  ,  0.]],
    },
'BF3': {
    'description': "BF3, Planar D3h symm.",
    'name': "BF_3",
    'database': 'G2-2',
    'enthalpy': -271.4,
    'ZPE': 7.8257,
    'thermal correction': 2.7893,
    'symbols': 'BFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 0.      ,  1.32176 ,  0.      ],
                  [ 1.144678, -0.66088 ,  0.      ],
                  [-1.144678, -0.66088 ,  0.      ]]},
'BCl3': {
    'description': "BCl3, Planar D3h symm.",
    'name': "BCl_3",
    'database': 'G2-2',
    'enthalpy': -96.3,
    'ZPE': 4.6536,
    'thermal correction': 3.3729,
    'symbols': 'BClClCl',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 0.      ,  1.735352,  0.      ],
                  [ 1.502859, -0.867676,  0.      ],
                  [-1.502859, -0.867676,  0.      ]]},
'AlF3': {
    'description': "AlF3, Planar D3h symm.",
    'name': "AlF_3",
    'database': 'G2-2',
    'enthalpy': -289.0,
    'ZPE': 4.8645,
    'thermal correction': 3.3986,
    'symbols': 'AlFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 0.      ,  1.64472 ,  0.      ],
                  [ 1.424369, -0.82236 ,  0.      ],
                  [-1.424369, -0.82236 ,  0.      ]]},
'AlCl3': {
    'description': "AlCl3, Planar D3h symm.",
    'name': "AlCl_3",
    'database': 'G2-2',
    'enthalpy': -139.7,
    'ZPE': 2.9687,
    'thermal correction': 3.9464,
    'symbols': 'AlClClCl',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 0.      ,  2.069041,  0.      ],
                  [ 1.791842, -1.03452 ,  0.      ],
                  [-1.791842, -1.03452 ,  0.      ]]},
'CF4': {
    'description': "CF4, Td symm.",
    'name': "CF_4",
    'database': 'G2-2',
    'enthalpy': -223.0,
    'ZPE': 10.5999,
    'thermal correction': 3.0717,
    'symbols': 'CFFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 0.767436,  0.767436,  0.767436],
                  [-0.767436, -0.767436,  0.767436],
                  [-0.767436,  0.767436, -0.767436],
                  [ 0.767436, -0.767436, -0.767436]]},
'CCl4': {
    'description': "CCl4, Td symm.",
    'name': "CCl_4",
    'database': 'G2-2',
    'enthalpy': -22.9,
    'ZPE': 5.7455,
    'thermal correction': 4.1754,
    'symbols': 'CClClClCl',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.     ,  0.     ,  0.     ],
                  [ 1.02134,  1.02134,  1.02134],
                  [-1.02134, -1.02134,  1.02134],
                  [-1.02134,  1.02134, -1.02134],
                  [ 1.02134, -1.02134, -1.02134]]},
'OCS': {
    'description': "O=C=S, Linear, C*v symm.",
    'name': "COS",
    'database': 'G2-2',
    'enthalpy': -33.1,
    'ZPE': 5.7706,
    'thermal correction': 2.3663,
    'symbols': 'OCS',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      , -1.699243],
                  [ 0.      ,  0.      , -0.520492],
                  [ 0.      ,  0.      ,  1.044806]]},
'CS2': {
    'description': "CS2, Linear, D*h symm.",
    'name': "CS_2",
    'database': 'G2-2',
    'enthalpy': 28.0,
    'ZPE': 4.3380,
    'thermal correction': 2.5326,
    'symbols': 'SCS',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  1.561117],
                  [ 0.      ,  0.      ,  0.      ],
                  [ 0.      ,  0.      , -1.561117]]},
'COF2': {
    'description': "COF2, C2v symm.",
    'name': "COF_2",
    'database': 'G2-2',
    'enthalpy': -149.1,
    'ZPE': 8.8215,
    'thermal correction': 2.6619,
    'symbols': 'OCFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  1.330715],
                  [ 0.      ,  0.      ,  0.144358],
                  [ 0.      ,  1.06949 , -0.639548],
                  [ 0.      , -1.06949 , -0.639548]]},
'SiF4': {
    'description': "SiF4, Td symm.",
    'name': "SiF_4",
    'database': 'G2-2',
    'enthalpy': -386.0,
    'ZPE': 7.8771,
    'thermal correction': 3.7054,
    'symbols': 'SiFFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 0.912806,  0.912806,  0.912806],
                  [-0.912806, -0.912806,  0.912806],
                  [-0.912806,  0.912806, -0.912806],
                  [ 0.912806, -0.912806, -0.912806]]},
'SiCl4': {
    'description': "SiCl4, Td symm.",
    'name': "SiCl_4",
    'database': 'G2-2',
    'enthalpy': -158.4,
    'ZPE': 4.4396,
    'thermal correction': 4.7182,
    'symbols': 'SiClClClCl',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 1.169349,  1.169349,  1.169349],
                  [-1.169349, -1.169349,  1.169349],
                  [ 1.169349, -1.169349, -1.169349],
                  [-1.169349,  1.169349, -1.169349]]},
'N2O': {
    'description': "N2O, Cs symm.",
    'name': "N_2O",
    'database': 'G2-2',
    'enthalpy': 19.6,
    'ZPE': 6.9748,
    'thermal correction': 2.2710,
    'symbols': 'NNO',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      , -1.231969],
                  [ 0.      ,  0.      , -0.060851],
                  [ 0.      ,  0.      ,  1.131218]]},
'ClNO': {
    'description': "ClNO, Cs symm.",
    'name': "ClNO",
    'database': 'G2-2',
    'enthalpy': 12.4,
    'ZPE': 4.0619,
    'thermal correction': 2.7039,
    'symbols': 'ClNO',
    'magmoms': None,
    'charges': None,
    'positions': [[-0.537724, -0.961291,  0.      ],
                  [ 0.      ,  0.997037,  0.      ],
                  [ 1.142664,  1.170335,  0.      ]]},
'NF3': {
    'description': "NF3, C3v symm.",
    'name': "NF_3",
    'database': 'G2-2',
    'enthalpy': -31.6,
    'ZPE': 6.4477,
    'thermal correction': 2.8301,
    'symbols': 'NFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.489672],
                  [ 0.      ,  1.238218, -0.126952],
                  [ 1.072328, -0.619109, -0.126952],
                  [-1.072328, -0.619109, -0.126952]]},
'PF3': {
    'description': "PF3, C3v symm.",
    'name': "PF_3",
    'database': 'G2-2',
    'enthalpy': -229.1,
    'ZPE': 5.2981,
    'thermal correction': 3.1288,
    'symbols': 'PFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.506767],
                  [ 0.      ,  1.383861, -0.281537],
                  [ 1.198459, -0.691931, -0.281537],
                  [-1.198459, -0.691931, -0.281537]]},
'O3': {
    'description': "O3 (Ozone), C2v symm.",
    'name': "O_3",
    'database': 'G2-2',
    'enthalpy': 34.1,
    'ZPE': 4.6178,
    'thermal correction': 2.4479,
    'symbols': 'OOO',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  1.10381 , -0.228542],
                  [ 0.      ,  0.      ,  0.457084],
                  [ 0.      , -1.10381 , -0.228542]]},
'F2O': {
    'description': "F2O, C2v symm.",
    'name': "F_2O",
    'database': 'G2-2',
    'enthalpy': 5.9,
    'ZPE': 3.4362,
    'thermal correction': 2.5747,
    'symbols': 'FOF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  1.110576, -0.273729],
                  [ 0.      ,  0.      ,  0.61589 ],
                  [ 0.      , -1.110576, -0.273729]]},
'ClF3': {
    'description': "ClF3, C2v symm.",
    'name': "ClF_3",
    'database': 'G2-2',
    'enthalpy': -38.0,
    'ZPE': 4.2922,
    'thermal correction': 3.3289,
    'symbols': 'ClFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.376796],
                  [ 0.      ,  0.      , -1.258346],
                  [ 0.      ,  1.714544,  0.27331 ],
                  [ 0.      , -1.714544,  0.27331 ]]},
'C2F4': {
    'description': "C2F4 (F2C=CF2), D2H symm.",
    'name': "C_2F_4",
    'database': 'G2-2',
    'enthalpy': -157.4,
    'ZPE': 13.4118,
    'thermal correction': 3.9037,
    'symbols': 'CCFFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.66323 ],
                  [ 0.      ,  0.      , -0.66323 ],
                  [ 0.      ,  1.112665,  1.385652],
                  [ 0.      , -1.112665,  1.385652],
                  [ 0.      ,  1.112665, -1.385652],
                  [ 0.      , -1.112665, -1.385652]]},
'C2Cl4': {
    'description': "C2Cl4 (Cl2C=CCl2), D2h symm.",
    'name': "C_2Cl_4",
    'database': 'G2-2',
    'enthalpy': -3.0,
    'ZPE': 9.4628,
    'thermal correction': 4.7132,
    'symbols': 'CCClClClCl',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.675402],
                  [ 0.      ,  0.      , -0.675402],
                  [ 0.      ,  1.448939,  1.589701],
                  [ 0.      , -1.448939,  1.589701],
                  [ 0.      , -1.448939, -1.589701],
                  [ 0.      ,  1.448939, -1.589701]]},
'CF3CN': {
    'description': "CF3CN, C3v symm.",
    'name': "CF_3CN",
    'database': 'G2-2',
    'enthalpy': -118.4,
    'ZPE': 14.1020,
    'thermal correction': 3.7996,
    'symbols': 'CCFFFN',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      , -0.32635 ],
                  [ 0.      ,  0.      ,  1.15083 ],
                  [ 0.      ,  1.257579, -0.787225],
                  [ 1.089096, -0.62879 , -0.787225],
                  [-1.089096, -0.62879 , -0.787225],
                  [ 0.      ,  0.      ,  2.329741]]},
'C3H4_C3v': {
    'description': "Propyne (C3H4), C3v symm.",
    'name': "CH_3CCH (propyne)",
    'database': 'G2-2',
    'enthalpy': 44.2,
    'ZPE': 34.2614,
    'thermal correction': 3.1193,
    'symbols': 'CCCHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.214947],
                  [ 0.      ,  0.      ,  1.43313 ],
                  [ 0.      ,  0.      , -1.246476],
                  [ 0.      ,  0.      ,  2.498887],
                  [ 0.      ,  1.021145, -1.636167],
                  [ 0.884337, -0.510572, -1.636167],
                  [-0.884337, -0.510572, -1.636167]]},
'C3H4_D2d': {
    'description': "Allene (C3H4), D2d symm.",
    'name': "CH_2=C=CH_2 (allene)",
    'database': 'G2-2',
    'enthalpy': 45.5,
    'ZPE': 34.1189,
    'thermal correction': 2.9744,
    'symbols': 'CCCHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 0.      ,  0.      ,  1.31119 ],
                  [ 0.      ,  0.      , -1.31119 ],
                  [ 0.      ,  0.926778,  1.876642],
                  [ 0.      , -0.926778,  1.876642],
                  [ 0.926778,  0.      , -1.876642],
                  [-0.926778,  0.      , -1.876642]]},
'C3H4_C2v': {
    'description': "Cyclopropene (C3H4), C2v symm.",
    'name': "C_3H_4 (cyclopropene)",
    'database': 'G2-2',
    'enthalpy': 66.2,
    'ZPE': 34.7603,
    'thermal correction': 2.6763,
    'symbols': 'CCCHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.858299],
                  [ 0.      , -0.650545, -0.498802],
                  [ 0.      ,  0.650545, -0.498802],
                  [ 0.912438,  0.      ,  1.456387],
                  [-0.912438,  0.      ,  1.456387],
                  [ 0.      , -1.584098, -1.038469],
                  [ 0.      ,  1.584098, -1.038469]]},
'C3H6_Cs': {
    'description': "Propene (C3H6), Cs symm.",
    'name': "CH_3CH=CH_2 (propylene)",
    'database': 'G2-2',
    'enthalpy': 4.8,
    'ZPE': 49.1836,
    'thermal correction': 3.1727,
    'symbols': 'CCHHHCHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 1.29129 ,  0.133682,  0.      ],
                  [ 0.      ,  0.479159,  0.      ],
                  [ 1.60116 , -0.90742 ,  0.      ],
                  [ 2.0808  ,  0.877337,  0.      ],
                  [-0.263221,  1.536098,  0.      ],
                  [-1.139757, -0.492341,  0.      ],
                  [-0.776859, -1.523291,  0.      ],
                  [-1.77554 , -0.352861,  0.88042 ],
                  [-1.77554 , -0.352861, -0.88042 ]]},
'C3H6_D3h': {
    'description': "Cyclopropane (C3H6), D3h symm.",
    'name': "C_3H_6 (cyclopropane)",
    'database': 'G2-2',
    'enthalpy': 12.7,
    'ZPE': 50.2121,
    'thermal correction': 2.7272,
    'symbols': 'CCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.866998,  0.      ],
                  [ 0.750842, -0.433499,  0.      ],
                  [-0.750842, -0.433499,  0.      ],
                  [ 0.      ,  1.455762,  0.910526],
                  [ 0.      ,  1.455762, -0.910526],
                  [ 1.260727, -0.727881, -0.910526],
                  [ 1.260727, -0.727881,  0.910526],
                  [-1.260727, -0.727881,  0.910526],
                  [-1.260727, -0.727881, -0.910526]]},
'C3H8': {
    'description': "Propane (C3H8), C2v symm.",
    'name': "C_3H_8 (propane)",
    'database': 'G2-2',
    'enthalpy': -25.0,
    'ZPE': 63.8008,
    'thermal correction': 3.4632,
    'symbols': 'CCCHHHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.587716],
                  [ 0.      ,  1.266857, -0.260186],
                  [ 0.      , -1.266857, -0.260186],
                  [-0.876898,  0.      ,  1.244713],
                  [ 0.876898,  0.      ,  1.244713],
                  [ 0.      ,  2.16615 ,  0.362066],
                  [ 0.      , -2.16615 ,  0.362066],
                  [ 0.883619,  1.304234, -0.904405],
                  [-0.883619,  1.304234, -0.904405],
                  [-0.883619, -1.304234, -0.904405],
                  [ 0.883619, -1.304234, -0.904405]]},
'butadiene': {
    'description': "Trans-1,3-butadiene (C4H6), C2h symm.",
    'name': "CH_2CHCHCH_2 (butadiene)",
    'database': 'G2-2',
    'enthalpy': 26.3,
    'ZPE': 52.6273,
    'thermal correction': 3.5341,
    'symbols': 'CCCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.605711,  1.74655 ,  0.      ],
                  [ 0.605711,  0.404083,  0.      ],
                  [-0.605711, -0.404083,  0.      ],
                  [-0.605711, -1.74655 ,  0.      ],
                  [ 1.527617,  2.317443,  0.      ],
                  [-0.321132,  2.313116,  0.      ],
                  [ 1.553503, -0.13364 ,  0.      ],
                  [-1.553503,  0.13364 ,  0.      ],
                  [ 0.321132, -2.313116,  0.      ],
                  [-1.527617, -2.317443,  0.      ]]},
'2-butyne': {
    'description': "Dimethylacetylene (2-butyne, C4H6), D3h symm (eclipsed).",
    'name': "C_4H_6 (2-butyne)",
    'database': 'G2-2',
    'enthalpy': 34.8,
    'ZPE': 51.8731,
    'thermal correction': 4.2344,
    'symbols': 'CCCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  2.071955],
                  [ 0.      ,  0.      ,  0.60997 ],
                  [ 0.      ,  0.      , -0.60997 ],
                  [ 0.      ,  0.      , -2.071955],
                  [ 0.      ,  1.020696,  2.464562],
                  [-0.883949, -0.510348,  2.464562],
                  [ 0.883949, -0.510348,  2.464562],
                  [ 0.      ,  1.020696, -2.464562],
                  [ 0.883949, -0.510348, -2.464562],
                  [-0.883949, -0.510348, -2.464562]]},
'methylenecyclopropane': {
    'description': "Methylenecyclopropane (C4H6), C2v symm.",
    'name': "C_4H_6 (methylene cyclopropane)",
    'database': 'G2-2',
    'enthalpy': 47.9,
    'ZPE': 52.6230,
    'thermal correction': 3.2881,
    'symbols': 'CCCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.315026],
                  [ 0.      , -0.76792 , -0.932032],
                  [ 0.      ,  0.76792 , -0.932032],
                  [ 0.      ,  0.      ,  1.640027],
                  [-0.912794, -1.271789, -1.239303],
                  [ 0.912794, -1.271789, -1.239303],
                  [ 0.912794,  1.271789, -1.239303],
                  [-0.912794,  1.271789, -1.239303],
                  [ 0.      , -0.926908,  2.20564 ],
                  [ 0.      ,  0.926908,  2.20564 ]]},
'bicyclobutane': {
    'description': "Bicyclo[1.1.0]butane (C4H6), C2v symm.",
    'name': "C_4H_6 (bicyclobutane)",
    'database': 'G2-2',
    'enthalpy': 51.9,
    'ZPE': 53.3527,
    'thermal correction': 2.9637,
    'symbols': 'CCCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  1.131343,  0.310424],
                  [ 0.      , -1.131343,  0.310424],
                  [ 0.747952,  0.      , -0.311812],
                  [-0.747952,  0.      , -0.311812],
                  [ 0.      ,  1.237033,  1.397617],
                  [ 0.      ,  2.077375, -0.227668],
                  [ 0.      , -1.237033,  1.397617],
                  [ 0.      , -2.077375, -0.227668],
                  [ 1.41441 ,  0.      , -1.161626],
                  [-1.41441 ,  0.      , -1.161626]]},
'cyclobutene': {
    'description': "Cyclobutene (C4H6), C2v symm.",
    'name': "C_4H_6 (cyclobutene)",
    'database': 'G2-2',
    'enthalpy': 37.4,
    'ZPE': 53.4105,
    'thermal correction': 3.0108,
    'symbols': 'CCCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      , -0.672762,  0.811217],
                  [ 0.      ,  0.672762,  0.811217],
                  [ 0.      , -0.78198 , -0.696648],
                  [ 0.      ,  0.78198 , -0.696648],
                  [ 0.      , -1.422393,  1.597763],
                  [ 0.      ,  1.422393,  1.597763],
                  [-0.88931 , -1.239242, -1.142591],
                  [ 0.88931 , -1.239242, -1.142591],
                  [ 0.88931 ,  1.239242, -1.142591],
                  [-0.88931 ,  1.239242, -1.142591]]},
'cyclobutane': {
    'description': "Cyclobutane (C4H8), D2d symm.",
    'name': "C_4H_8 (cyclobutane)",
    'database': 'G2-2',
    'enthalpy': 6.8,
    'ZPE': 68.3314,
    'thermal correction': 3.2310,
    'symbols': 'CCCCHHHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  1.071142,  0.147626],
                  [ 0.      , -1.071142,  0.147626],
                  [-1.071142,  0.      , -0.147626],
                  [ 1.071142,  0.      , -0.147626],
                  [ 0.      ,  1.986858, -0.450077],
                  [ 0.      ,  1.342921,  1.20752 ],
                  [ 0.      , -1.986858, -0.450077],
                  [ 0.      , -1.342921,  1.20752 ],
                  [-1.986858,  0.      ,  0.450077],
                  [-1.342921,  0.      , -1.20752 ],
                  [ 1.986858,  0.      ,  0.450077],
                  [ 1.342921,  0.      , -1.20752 ]]},
'isobutene': {
    'description': "Isobutene (C4H8), Single bonds trans, C2v symm.",
    'name': "C_4H_8 (isobutene)",
    'database': 'G2-2',
    'enthalpy': -4.0,
    'ZPE': 66.5693,
    'thermal correction': 3.9495,
    'symbols': 'CCHHCHHHCHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  1.458807],
                  [ 0.      ,  0.      ,  0.119588],
                  [ 0.      ,  0.924302,  2.028409],
                  [ 0.      , -0.924302,  2.028409],
                  [ 0.      ,  1.272683, -0.678803],
                  [ 0.      ,  2.153042, -0.031588],
                  [ 0.880211,  1.323542, -1.329592],
                  [-0.880211,  1.323542, -1.329592],
                  [ 0.      , -1.272683, -0.678803],
                  [ 0.      , -2.153042, -0.031588],
                  [-0.880211, -1.323542, -1.329592],
                  [ 0.880211, -1.323542, -1.329592]]},
'trans-butane': {
    'description': "Trans-butane (C4H10), C2h symm.",
    'name': "C_4H_{10} (trans butane)",
    'database': 'G2-2',
    'enthalpy': -30.0,
    'ZPE': 81.3980,
    'thermal correction': 4.2633,
    'symbols': 'CCCCHHHHHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.702581,  1.820873,  0.      ],
                  [ 0.702581,  0.296325,  0.      ],
                  [-0.702581, -0.296325,  0.      ],
                  [-0.702581, -1.820873,  0.      ],
                  [ 1.719809,  2.22234 ,  0.      ],
                  [-1.719809, -2.22234 ,  0.      ],
                  [ 0.188154,  2.210362,  0.883614],
                  [ 0.188154,  2.210362, -0.883614],
                  [-0.188154, -2.210362,  0.883614],
                  [-0.188154, -2.210362, -0.883614],
                  [ 1.247707, -0.07266 , -0.877569],
                  [ 1.247707, -0.07266 ,  0.877569],
                  [-1.247707,  0.07266 , -0.877569],
                  [-1.247707,  0.07266 ,  0.877569]]},
'isobutane': {
    'description': "Isobutane (C4H10), C3v symm.",
    'name': "C_4H_{10} (isobutane)",
    'database': 'G2-2',
    'enthalpy': -32.1,
    'ZPE': 81.1050,
    'thermal correction': 4.2282,
    'symbols': 'CHCHHHCHHHCHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.376949],
                  [ 0.      ,  0.      ,  1.475269],
                  [ 0.      ,  1.45029 , -0.096234],
                  [ 0.      ,  1.493997, -1.190847],
                  [-0.885482,  1.984695,  0.261297],
                  [ 0.885482,  1.984695,  0.261297],
                  [ 1.255988, -0.725145, -0.096234],
                  [ 1.293839, -0.746998, -1.190847],
                  [ 2.161537, -0.225498,  0.261297],
                  [ 1.276055, -1.759198,  0.261297],
                  [-1.255988, -0.725145, -0.096234],
                  [-1.293839, -0.746998, -1.190847],
                  [-1.276055, -1.759198,  0.261297],
                  [-2.161537, -0.225498,  0.261297]]},
'C5H8': {
    'description': "Spiropentane (C5H8), D2d symm.",
    'name': "C_5H_8 (spiropentane)",
    'database': 'G2-2',
    'enthalpy': 44.3,
    'ZPE': 70.9964,
    'thermal correction': 3.7149,
    'symbols': 'CCCCCHHHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.      ],
                  [ 0.      ,  0.762014,  1.265752],
                  [ 0.      , -0.762014,  1.265752],
                  [ 0.762014,  0.      , -1.265752],
                  [-0.762014,  0.      , -1.265752],
                  [-0.914023,  1.265075,  1.56809 ],
                  [ 0.914023,  1.265075,  1.56809 ],
                  [-0.914023, -1.265075,  1.56809 ],
                  [ 0.914023, -1.265075,  1.56809 ],
                  [ 1.265075, -0.914023, -1.56809 ],
                  [ 1.265075,  0.914023, -1.56809 ],
                  [-1.265075, -0.914023, -1.56809 ],
                  [-1.265075,  0.914023, -1.56809 ]]},
'C6H6': {
    'description': "Benzene (C6H6), D6h symm.",
    'name': "C_6H_6 (benzene)",
    'database': 'G2-2',
    'enthalpy': 19.7,
    'ZPE': 61.9252,
    'thermal correction': 3.3886,
    'ionization energy': 9.24,
    'vertical ionization energy': 9.25,
    'symbols': 'CCCCCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  1.395248,  0.      ],
                  [ 1.20832 ,  0.697624,  0.      ],
                  [ 1.20832 , -0.697624,  0.      ],
                  [ 0.      , -1.395248,  0.      ],
                  [-1.20832 , -0.697624,  0.      ],
                  [-1.20832 ,  0.697624,  0.      ],
                  [ 0.      ,  2.48236 ,  0.      ],
                  [ 2.149787,  1.24118 ,  0.      ],
                  [ 2.149787, -1.24118 ,  0.      ],
                  [ 0.      , -2.48236 ,  0.      ],
                  [-2.149787, -1.24118 ,  0.      ],
                  [-2.149787,  1.24118 ,  0.      ]]},
'H2CF2': {
    'description': "Difluoromethane (H2CF2), C2v symm.",
    'name': "CH_2F_2",
    'database': 'G2-2',
    'enthalpy': -107.7,
    'ZPE': 20.2767,
    'thermal correction': 2.5552,
    'symbols': 'CFFHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.502903],
                  [ 0.      ,  1.109716, -0.290601],
                  [ 0.      , -1.109716, -0.290601],
                  [-0.908369,  0.      ,  1.106699],
                  [ 0.908369,  0.      ,  1.106699]]},
'HCF3': {
    'description': "Trifluoromethane (HCF3), C3v symm.",
    'name': "CHF_3",
    'database': 'G2-2',
    'enthalpy': -166.6,
    'ZPE': 15.7072,
    'thermal correction': 2.7717,
    'symbols': 'CHFFF',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.341023],
                  [ 0.      ,  0.      ,  1.429485],
                  [ 0.      ,  1.2582  , -0.128727],
                  [ 1.089633, -0.6291  , -0.128727],
                  [-1.089633, -0.6291  , -0.128727]]},
'H2CCl2': {
    'description': "Dichloromethane (H2CCl2), C2v symm.",
    'name': "CH_2Cl_2",
    'database': 'G2-2',
    'enthalpy': -22.8,
    'ZPE': 18.0930,
    'thermal correction': 2.8527,
    'symbols': 'CClClHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.759945],
                  [ 0.      ,  1.4742  , -0.215115],
                  [ 0.      , -1.4742  , -0.215115],
                  [-0.894585,  0.      ,  1.377127],
                  [ 0.894585,  0.      ,  1.377127]]},
'HCCl3': {
    'description': "Chloroform (HCCl3), C3v symm.",
    'name': "CHCl_3",
    'database': 'G2-2',
    'enthalpy': -24.7,
    'ZPE': 12.1975,
    'thermal correction': 3.4262,
    'symbols': 'CHClClCl',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.451679],
                  [ 0.      ,  0.      ,  1.537586],
                  [ 0.      ,  1.681723, -0.083287],
                  [ 1.456415, -0.840862, -0.083287],
                  [-1.456415, -0.840862, -0.083287]]},
'H3CNH2': {
    'description': "Methylamine (H3C-NH2), Cs symm.",
    'name': "CH_3NH_2 (methylamine)",
    'database': 'G2-2',
    'enthalpy': -5.5,
    'ZPE': 39.5595,
    'thermal correction': 2.7428,
    'symbols': 'CNHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.051736,  0.704422,  0.      ],
                  [ 0.051736, -0.759616,  0.      ],
                  [-0.941735,  1.176192,  0.      ],
                  [-0.458181, -1.099433,  0.81237 ],
                  [-0.458181, -1.099433, -0.81237 ],
                  [ 0.592763,  1.056727,  0.88067 ],
                  [ 0.592763,  1.056727, -0.88067 ]]},
'CH3CN': {
    'description': "Acetonitrile (CH3-CN), C3v symm.",
    'name': "CH_3CN (methyl cyanide)",
    'database': 'G2-2',
    'enthalpy': 18.0,
    'ZPE': 28.0001,
    'thermal correction': 2.8552,
    'symbols': 'CCNHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      , -1.18693 ],
                  [ 0.      ,  0.      ,  0.273874],
                  [ 0.      ,  0.      ,  1.452206],
                  [ 0.      ,  1.024986, -1.56237 ],
                  [ 0.887664, -0.512493, -1.56237 ],
                  [-0.887664, -0.512493, -1.56237 ]]},
'CH3NO2': {
    'description': "Nitromethane (CH3-NO2), Cs symm.",
    'name': "CH_3NO_2 (nitromethane)",
    'database': 'G2-2',
    'enthalpy': -17.8,
    'ZPE': 30.7568,
    'thermal correction': 2.7887,
    'symbols': 'CNHHHOO',
    'magmoms': None,
    'charges': None,
    'positions': [[-0.114282, -1.314565,  0.      ],
                  [ 0.      ,  0.16648 ,  0.      ],
                  [ 0.899565, -1.715256,  0.      ],
                  [-0.640921, -1.607212,  0.904956],
                  [-0.640921, -1.607212, -0.904956],
                  [ 0.066748,  0.728232, -1.103775],
                  [ 0.066748,  0.728232,  1.103775]]},
'CH3ONO': {
    'description': "Methylnitrite (CH3-O-N=O), NOCH trans, ONOC cis, Cs symm.",
    'name': "CH_3ONO (methyl nitrite)",
    'database': 'G2-2',
    'enthalpy': -15.9,
    'ZPE': 29.9523,
    'thermal correction': 3.3641,
    'symbols': 'COHHHNO',
    'magmoms': None,
    'charges': None,
    'positions': [[-1.316208,  0.309247,  0.      ],
                  [ 0.      ,  0.896852,  0.      ],
                  [-1.985538,  1.166013,  0.      ],
                  [-1.464336, -0.304637,  0.890672],
                  [-1.464336, -0.304637, -0.890672],
                  [ 1.045334, -0.022815,  0.      ],
                  [ 0.686764, -1.178416,  0.      ]]},
'CH3SiH3': {
    'description': "Methylsilane (CH3-SiH3), C3v symm.",
    'name': "CH_3SiH_3 (methyl silane)",
    'database': 'G2-2',
    'enthalpy': -7.0,
    'ZPE': 37.6606,
    'thermal correction': 3.2486,
    'symbols': 'CSiHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      , -1.244466],
                  [ 0.      ,  0.      ,  0.635703],
                  [ 0.      , -1.019762, -1.636363],
                  [-0.88314 ,  0.509881, -1.636363],
                  [ 0.88314 ,  0.509881, -1.636363],
                  [ 0.      ,  1.391234,  1.158682],
                  [-1.204844, -0.695617,  1.158682],
                  [ 1.204844, -0.695617,  1.158682]]},
'HCOOH': {
    'description': "Formic Acid (HCOOH), HOCO cis, Cs symm.",
    'name': "HCOOH (formic acid)",
    'database': 'G2-2',
    'enthalpy': -90.5,
    'ZPE': 20.9525,
    'thermal correction': 2.5853,
    'symbols': 'OCOHH',
    'magmoms': None,
    'charges': None,
    'positions': [[-1.040945, -0.436432,  0.      ],
                  [ 0.      ,  0.423949,  0.      ],
                  [ 1.169372,  0.103741,  0.      ],
                  [-0.64957 , -1.335134,  0.      ],
                  [-0.377847,  1.452967,  0.      ]]},
'HCOOCH3': {
    'description': "Methyl formate (HCOOCH3), Cs symm.",
    'name': "HCOOCH_3 (methyl formate)",
    'database': 'G2-2',
    'enthalpy': -85.0,
    'ZPE': 38.3026,
    'thermal correction': 3.4726,
    'symbols': 'COOHCHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[-0.931209, -0.083866,  0.      ],
                  [-0.711019, -1.278209,  0.      ],
                  [ 0.      ,  0.886841,  0.      ],
                  [-1.92836 ,  0.374598,  0.      ],
                  [ 1.356899,  0.397287,  0.      ],
                  [ 1.980134,  1.288164,  0.      ],
                  [ 1.541121, -0.206172,  0.889397],
                  [ 1.541121, -0.206172, -0.889397]]},
'CH3CONH2': {
    'description': "Acetamide (CH3CONH2), C1 symm.",
    'name': "CH_3CONH_2 (acetamide)",
    'database': 'G2-2',
    'enthalpy': -57.0,
    'ZPE': 45.2566,
    'thermal correction': 3.9313,
    'symbols': 'OCNCHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[  4.24546000e-01,   1.32702400e+00,   8.03400000e-03],
                  [  7.71580000e-02,   1.49789000e-01,  -4.24900000e-03],
                  [  9.85518000e-01,  -8.78537000e-01,  -4.89100000e-02],
                  [ -1.37147500e+00,  -2.88665000e-01,  -1.44000000e-04],
                  [  7.07952000e-01,  -1.82424900e+00,   1.69942000e-01],
                  [ -1.99722900e+00,   5.84922000e-01,  -1.75477000e-01],
                  [ -1.56084200e+00,  -1.03927000e+00,  -7.71686000e-01],
                  [ -1.63211300e+00,  -7.23007000e-01,   9.69814000e-01],
                  [  1.95313300e+00,  -6.31574000e-01,   1.11866000e-01]]},
'CH2NHCH2': {
    'description': "Aziridine (cyclic CH2-NH-CH2 ring), C2v symm.",
    'name': "C_2H_4NH (aziridine)",
    'database': 'G2-2',
    'enthalpy': 30.2,
    'ZPE': 43.3728,
    'thermal correction': 2.6399,
    'symbols': 'CNCHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[-0.03845 , -0.397326,  0.739421],
                  [-0.03845 ,  0.875189,  0.      ],
                  [-0.03845 , -0.397326, -0.739421],
                  [ 0.903052,  1.268239,  0.      ],
                  [-0.955661, -0.604926,  1.280047],
                  [-0.955661, -0.604926, -1.280047],
                  [ 0.869409, -0.708399,  1.249033],
                  [ 0.869409, -0.708399, -1.249033]]},
'NCCN': {
    'description': "Cyanogen (NCCN). D*h symm.",
    'name': "NCCN (cyanogen)",
    'database': 'G2-2',
    'enthalpy': 73.3,
    'ZPE': 10.2315,
    'thermal correction': 2.9336,
    'symbols': 'NCCN',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  1.875875],
                  [ 0.      ,  0.      ,  0.690573],
                  [ 0.      ,  0.      , -0.690573],
                  [ 0.      ,  0.      , -1.875875]]},
'C2H6NH': {
    'description': "Dimethylamine, (CH3)2NH, Cs symm.",
    'name': "(CH_3)_2NH (dimethylamine)",
    'database': 'G2-2',
    'enthalpy': -4.4,
    'ZPE': 57.0287,
    'thermal correction': 3.3760,
    'symbols': 'CNCHHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[-0.02753 , -0.224702,  1.20488 ],
                  [-0.02753 ,  0.59247 ,  0.      ],
                  [-0.02753 , -0.224702, -1.20488 ],
                  [ 0.791501, -0.962742,  1.248506],
                  [ 0.039598,  0.421182,  2.083405],
                  [-0.97222 , -0.772987,  1.26175 ],
                  [ 0.805303,  1.17822 ,  0.      ],
                  [ 0.791501, -0.962742, -1.248506],
                  [ 0.039598,  0.421182, -2.083405],
                  [-0.97222 , -0.772987, -1.26175 ]]},
'CH3CH2NH2': {
    'description': "Trans-Ethylamine (CH3-CH2-NH2), Cs symm.",
    'name': "CH_3CH_2NH_2 (trans ethylamine)",
    'database': 'G2-2',
    'enthalpy': -11.3,
    'ZPE': 57.2420,
    'thermal correction': 3.3678,
    'symbols': 'CCNHHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 1.210014, -0.353598,  0.      ],
                  [ 0.      ,  0.575951,  0.      ],
                  [-1.305351, -0.087478,  0.      ],
                  [ 2.14931 ,  0.208498,  0.      ],
                  [ 1.201796, -0.99776 ,  0.884909],
                  [ 1.201796, -0.99776 , -0.884909],
                  [ 0.034561,  1.230963, -0.876478],
                  [ 0.034561,  1.230963,  0.876478],
                  [-1.372326, -0.69834 ,  0.813132],
                  [-1.372326, -0.69834 , -0.813132]]},
'H2CCO': {
    'description': "Ketene (H2C=C=O), C2v symm.",
    'name': "CH_2CO (ketene)",
    'database': 'G2-2',
    'enthalpy': -11.4,
    'ZPE': 19.5984,
    'thermal correction': 2.8075,
    'symbols': 'CCHHO',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      , -1.21934 ],
                  [ 0.      ,  0.      ,  0.09892 ],
                  [ 0.      ,  0.938847, -1.753224],
                  [ 0.      , -0.938847, -1.753224],
                  [ 0.      ,  0.      ,  1.27862 ]]},
'CH2OCH2': {
    'description': "Oxirane (cyclic CH2-O-CH2 ring), C2v symm.",
    'name': "C_2H_4O (oxirane)",
    'database': 'G2-2',
    'enthalpy': -12.6,
    'ZPE': 35.4204,
    'thermal correction': 2.5816,
    'symbols': 'COCHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.73158 , -0.375674],
                  [ 0.      ,  0.      ,  0.86095 ],
                  [ 0.      , -0.73158 , -0.375674],
                  [ 0.919568,  1.268821, -0.594878],
                  [-0.919568,  1.268821, -0.594878],
                  [-0.919568, -1.268821, -0.594878],
                  [ 0.919568, -1.268821, -0.594878]]},
'CH3CHO': {
    'description': "Acetaldehyde (CH3CHO), Cs symm.",
    'name': "CH_3CHO (acetaldehyde)",
    'database': 'G2-2',
    'enthalpy': -39.7,
    'ZPE': 34.2288,
    'thermal correction': 3.0428,
    'symbols': 'OCHCHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 1.218055,  0.36124 ,  0.      ],
                  [ 0.      ,  0.464133,  0.      ],
                  [-0.477241,  1.465295,  0.      ],
                  [-0.948102, -0.700138,  0.      ],
                  [-0.385946, -1.634236,  0.      ],
                  [-1.596321, -0.652475,  0.880946],
                  [-1.596321, -0.652475, -0.880946]]},
'OCHCHO': {
    'description': "Glyoxal (O=CH-CH=O). Trans, C2h symm.",
    'name': "HCOCOH (glyoxal)",
    'database': 'G2-2',
    'enthalpy': -50.7,
    'ZPE': 22.8426,
    'thermal correction': 3.2518,
    'symbols': 'CCOHOH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.75643 ,  0.      ],
                  [ 0.      , -0.75643 ,  0.      ],
                  [ 1.04609 ,  1.389916,  0.      ],
                  [-0.99994 ,  1.228191,  0.      ],
                  [-1.04609 , -1.389916,  0.      ],
                  [ 0.99994 , -1.228191,  0.      ]]},
'CH3CH2OH': {
    'description': "Ethanol (trans, CH3CH2OH), Cs symm.",
    'name': "CH_3CH_2OH (ethanol)",
    'database': 'G2-2',
    'enthalpy': -56.2,
    'ZPE': 49.3072,
    'thermal correction': 3.3252,
    'symbols': 'CCOHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 1.168181, -0.400382,  0.      ],
                  [ 0.      ,  0.559462,  0.      ],
                  [-1.190083, -0.227669,  0.      ],
                  [-1.946623,  0.381525,  0.      ],
                  [ 0.042557,  1.207508,  0.886933],
                  [ 0.042557,  1.207508, -0.886933],
                  [ 2.115891,  0.1448  ,  0.      ],
                  [ 1.128599, -1.037234,  0.885881],
                  [ 1.128599, -1.037234, -0.885881]]},
'CH3OCH3': {
    'description': "DimethylEther (CH3-O-CH3), C2v symm.",
    'name': "CH_3OCH_3 (dimethylether)",
    'database': 'G2-2',
    'enthalpy': -44.0,
    'ZPE': 49.1911,
    'thermal correction': 3.3139,
    'symbols': 'COCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  1.165725, -0.19995 ],
                  [ 0.      ,  0.      ,  0.60011 ],
                  [ 0.      , -1.165725, -0.19995 ],
                  [ 0.      ,  2.017769,  0.480203],
                  [ 0.891784,  1.21432 , -0.840474],
                  [-0.891784,  1.21432 , -0.840474],
                  [ 0.      , -2.017769,  0.480203],
                  [-0.891784, -1.21432 , -0.840474],
                  [ 0.891784, -1.21432 , -0.840474]]},
'CH2SCH2': {
    'description': "Thiooxirane (cyclic CH2-S-CH2 ring), C2v symm.",
    'name': "C_2H_4S (thiirane)",
    'database': 'G2-2',
    'enthalpy': 19.6,
    'ZPE': 33.9483,
    'thermal correction': 2.7290,
    'symbols': 'CSCHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      , -0.739719, -0.792334],
                  [ 0.      ,  0.      ,  0.863474],
                  [ 0.      ,  0.739719, -0.792334],
                  [-0.91394 , -1.250142, -1.076894],
                  [ 0.91394 , -1.250142, -1.076894],
                  [ 0.91394 ,  1.250142, -1.076894],
                  [-0.91394 ,  1.250142, -1.076894]]},
'C2H6SO': {
    'description': "Dimethylsulfoxide (CH3)2SO, Cs symm.",
    'name': "(CH_3)_2SO (dimethyl sulfoxide)",
    'database': 'G2-2',
    'enthalpy': -36.2,
    'ZPE': 48.8479,
    'thermal correction': 4.1905,
    'symbols': 'SOCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[  2.00000000e-06,   2.31838000e-01,  -4.38643000e-01],
                  [  2.00000000e-05,   1.50074200e+00,   3.79819000e-01],
                  [  1.33952800e+00,  -8.09022000e-01,   1.80717000e-01],
                  [ -1.33954800e+00,  -8.08992000e-01,   1.80718000e-01],
                  [  1.25583500e+00,  -8.96385000e-01,   1.26682500e+00],
                  [ -2.27940400e+00,  -3.13924000e-01,  -6.86740000e-02],
                  [  1.30440700e+00,  -1.79332700e+00,  -2.92589000e-01],
                  [  2.27939500e+00,  -3.13974000e-01,  -6.86740000e-02],
                  [ -1.30444700e+00,  -1.79329800e+00,  -2.92587000e-01],
                  [ -1.25585700e+00,  -8.96355000e-01,   1.26682600e+00]]},
'CH3CH2SH': {
    'description': "ThioEthanol (CH3-CH2-SH), Cs symm.",
    'name': "C_2H_5SH (ethanethiol)",
    'database': 'G2-2',
    'enthalpy': -11.1,
    'ZPE': 46.1583,
    'thermal correction': 3.5900,
    'symbols': 'CCSHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 1.514343,  0.679412,  0.      ],
                  [ 0.      ,  0.826412,  0.      ],
                  [-0.756068, -0.831284,  0.      ],
                  [-2.035346, -0.427738,  0.      ],
                  [-0.32497 ,  1.376482,  0.885793],
                  [-0.32497 ,  1.376482, -0.885793],
                  [ 1.986503,  1.665082,  0.      ],
                  [ 1.854904,  0.137645,  0.885494],
                  [ 1.854904,  0.137645, -0.885494]]},
'CH3SCH3': {
    'description': "Dimethyl ThioEther (CH3-S-CH3), C2v symm.",
    'name': "CH_3SCH_3 (dimethyl sulfide)",
    'database': 'G2-2',
    'enthalpy': -8.9,
    'ZPE': 46.6760,
    'thermal correction': 3.6929,
    'symbols': 'CSCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  1.366668, -0.513713],
                  [ 0.      ,  0.      ,  0.664273],
                  [ 0.      , -1.366668, -0.513713],
                  [ 0.      ,  2.296687,  0.057284],
                  [ 0.891644,  1.34568 , -1.144596],
                  [-0.891644,  1.34568 , -1.144596],
                  [ 0.      , -2.296687,  0.057284],
                  [-0.891644, -1.34568 , -1.144596],
                  [ 0.891644, -1.34568 , -1.144596]]},
'H2CCHF': {
    'description': "Vinyl fluoride (H2C=CHF), Cs symm.",
    'name': "CH_2=CHF (vinyl fluoride)",
    'database': 'G2-2',
    'enthalpy': -33.2,
    'ZPE': 27.2785,
    'thermal correction': 2.7039,
    'symbols': 'CCFHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.437714,  0.      ],
                  [ 1.191923, -0.145087,  0.      ],
                  [-1.148929, -0.278332,  0.      ],
                  [-0.186445,  1.505778,  0.      ],
                  [ 1.291348, -1.222833,  0.      ],
                  [ 2.083924,  0.466279,  0.      ]]},
'CH3CH2Cl': {
    'description': "Ethyl chloride (CH3-CH2-Cl), Cs symm.",
    'name': "C_2H_5Cl (ethyl chloride)",
    'database': 'G2-2',
    'enthalpy': -26.8,
    'ZPE': 41.0686,
    'thermal correction': 3.1488,
    'symbols': 'CCClHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.807636,  0.      ],
                  [ 1.505827,  0.647832,  0.      ],
                  [-0.823553, -0.77997 ,  0.      ],
                  [-0.344979,  1.341649,  0.885248],
                  [-0.344979,  1.341649, -0.885248],
                  [ 1.976903,  1.634877,  0.      ],
                  [ 1.839246,  0.10425 ,  0.885398],
                  [ 1.839246,  0.10425 , -0.885398]]},
'H2CCHCl': {
    'description': "Vinyl chloride, H2C=CHCl, Cs symm.",
    'name': "CH_2=CHCl (vinyl chloride)",
    'database': 'G2-2',
    'enthalpy': 8.9,
    'ZPE': 26.3554,
    'thermal correction': 2.8269,
    'symbols': 'CCClHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.756016,  0.      ],
                  [ 1.303223,  1.028507,  0.      ],
                  [-0.631555, -0.85498 ,  0.      ],
                  [-0.771098,  1.516963,  0.      ],
                  [ 2.056095,  0.249427,  0.      ],
                  [ 1.632096,  2.061125,  0.      ]]},
'H2CCHCN': {
    'description': "CyanoEthylene (H2C=CHCN), Cs symm.",
    'name': "CH_2=CHCN (acrylonitrile)",
    'database': 'G2-2',
    'enthalpy': 43.2,
    'ZPE': 31.4081,
    'thermal correction': 3.2034,
    'symbols': 'CCCHHHN',
    'magmoms': None,
    'charges': None,
    'positions': [[-0.161594, -1.638625,  0.      ],
                  [ 0.584957, -0.524961,  0.      ],
                  [ 0.      ,  0.782253,  0.      ],
                  [-1.245203, -1.598169,  0.      ],
                  [ 0.305973, -2.616405,  0.      ],
                  [ 1.669863, -0.572107,  0.      ],
                  [-0.467259,  1.867811,  0.      ]]},
'CH3COCH3': {
    'description': "Acetone (CH3-CO-CH3), C2v symm.",
    'name': "CH_3COCH_3 (acetone)",
    'database': 'G2-2',
    'enthalpy': -51.9,
    'ZPE': 51.5587,
    'thermal correction': 3.9878,
    'symbols': 'OCCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  1.405591],
                  [ 0.      ,  0.      ,  0.17906 ],
                  [ 0.      ,  1.28549 , -0.616342],
                  [ 0.      , -1.28549 , -0.616342],
                  [ 0.      ,  2.134917,  0.066535],
                  [ 0.      , -2.134917,  0.066535],
                  [-0.881086,  1.331548, -1.264013],
                  [ 0.881086,  1.331548, -1.264013],
                  [ 0.881086, -1.331548, -1.264013],
                  [-0.881086, -1.331548, -1.264013]]},
'CH3COOH': {
    'description': "Acetic Acid (CH3COOH), Single bonds trans, Cs symm.",
    'name': "CH_3COOH (acetic acid)",
    'database': 'G2-2',
    'enthalpy': -103.4,
    'ZPE': 38.1670,
    'thermal correction': 3.4770,
    'symbols': 'COOHCHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.15456 ,  0.      ],
                  [ 0.166384,  1.360084,  0.      ],
                  [-1.236449, -0.415036,  0.      ],
                  [-1.867646,  0.333582,  0.      ],
                  [ 1.073776, -0.892748,  0.      ],
                  [ 2.048189, -0.408135,  0.      ],
                  [ 0.968661, -1.528353,  0.881747],
                  [ 0.968661, -1.528353, -0.881747]]},
'CH3COF': {
    'description': "Acetyl fluoride (CH3COF), HCCO cis, Cs symm.",
    'name': "CH_3COF (acetyl fluoride)",
    'database': 'G2-2',
    'enthalpy': -105.7,
    'ZPE': 30.2742,
    'thermal correction': 3.3126,
    'symbols': 'COFCHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.186396,  0.      ],
                  [ 0.126651,  1.377199,  0.      ],
                  [-1.24395 , -0.382745,  0.      ],
                  [ 1.049454, -0.876224,  0.      ],
                  [ 2.035883, -0.417099,  0.      ],
                  [ 0.924869, -1.508407,  0.881549],
                  [ 0.924869, -1.508407, -0.881549]]},
'CH3COCl': {
    'description': "Acetyl,Chloride (CH3COCl), HCCO cis, Cs symm.",
    'name': "CH_3COCl (acetyl chloride)",
    'database': 'G2-2',
    'enthalpy': -58.0,
    'ZPE': 29.1855,
    'thermal correction': 3.5235,
    'symbols': 'CCClOHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.523878,  0.      ],
                  [ 1.486075,  0.716377,  0.      ],
                  [-0.452286, -1.217999,  0.      ],
                  [-0.845539,  1.37494 ,  0.      ],
                  [ 1.701027,  1.784793,  0.      ],
                  [ 1.917847,  0.240067,  0.882679],
                  [ 1.917847,  0.240067, -0.882679]]},
'C3H7Cl': {
    'description': "Propyl chloride (CH3CH2CH2Cl), Cs symm.",
    'name': "CH_3CH_2CH_2Cl (propyl chloride)",
    'database': 'G2-2',
    'enthalpy': -31.5,
    'ZPE': 58.6696,
    'thermal correction': 3.9885,
    'symbols': 'CCCHHHClHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.892629, -0.642344,  0.      ],
                  [ 2.365587, -0.245168,  0.      ],
                  [ 0.      ,  0.582921,  0.      ],
                  [ 0.663731, -1.252117,  0.879201],
                  [ 0.663731, -1.252117, -0.879201],
                  [ 3.005476, -1.130924,  0.      ],
                  [-1.73281 ,  0.139743,  0.      ],
                  [ 2.614882,  0.347704, -0.88473 ],
                  [ 2.614882,  0.347704,  0.88473 ],
                  [ 0.172881,  1.195836,  0.88646 ],
                  [ 0.172881,  1.195836, -0.88646 ]]},
'C2H6CHOH': {
    'description': "Isopropyl alcohol, (CH3)2CH-OH, Gauche isomer, C1 symm.",
    'name': "(CH_3)_2CHOH (isopropanol)",
    'database': 'G2-2',
    'enthalpy': -65.2,
    'ZPE': 66.5612,
    'thermal correction': 4.0732,
    'symbols': 'OCHHCCHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[  2.71910000e-02,   1.36369100e+00,  -1.67516000e-01],
                  [ -9.26000000e-04,   3.64590000e-02,   3.70128000e-01],
                  [  8.59465000e-01,   1.77564700e+00,   1.21307000e-01],
                  [  7.37100000e-03,   8.21450000e-02,   1.47050600e+00],
                  [ -1.31327500e+00,  -5.63514000e-01,  -8.89790000e-02],
                  [  1.20072100e+00,  -7.64480000e-01,  -1.04920000e-01],
                  [ -1.33400500e+00,  -6.07253000e-01,  -1.18100900e+00],
                  [  1.20284300e+00,  -8.07817000e-01,  -1.19718900e+00],
                  [ -2.14781200e+00,   5.49930000e-02,   2.47676000e-01],
                  [  2.13646200e+00,  -2.99324000e-01,   2.23164000e-01],
                  [ -1.43870900e+00,  -1.57427500e+00,   3.08340000e-01],
                  [  1.17773600e+00,  -1.78443600e+00,   2.89967000e-01]]},
'CH3CH2OCH3': {
    'description': "Methyl ethyl ether (CH3-CH2-O-CH3), Trans, Cs symm.",
    'name': "C_2H_5OCH_3 (methyl ethyl ether)",
    'database': 'G2-2',
    'enthalpy': -51.7,
    'ZPE': 66.6936,
    'thermal correction': 4.1058,
    'symbols': 'OCCCHHHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.006429, -0.712741,  0.      ],
                  [ 0.      ,  0.705845,  0.      ],
                  [ 1.324518, -1.226029,  0.      ],
                  [-1.442169,  1.160325,  0.      ],
                  [ 0.530962,  1.086484,  0.886881],
                  [ 0.530962,  1.086484, -0.886881],
                  [ 1.241648, -2.313325,  0.      ],
                  [ 1.881329, -0.905925, -0.89171 ],
                  [ 1.881329, -0.905925,  0.89171 ],
                  [-1.954863,  0.780605, -0.885855],
                  [-1.954863,  0.780605,  0.885855],
                  [-1.502025,  2.252083,  0.      ]]},
'C3H9N': {
    'description': "Trimethyl Amine, (CH3)3N, C3v symm.",
    'name': "(CH_3)_3N (trimethylamine)",
    'database': 'G2-2',
    'enthalpy': -5.7,
    'ZPE': 74.1584,
    'thermal correction': 4.0631,
    'symbols': 'NCCCHHHHHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.395846],
                  [ 0.      ,  1.378021, -0.065175],
                  [ 1.193401, -0.689011, -0.065175],
                  [-1.193401, -0.689011, -0.065175],
                  [ 0.      ,  1.461142, -1.167899],
                  [ 0.886156,  1.891052,  0.317655],
                  [-0.886156,  1.891052,  0.317655],
                  [ 1.265386, -0.730571, -1.167899],
                  [ 1.194621, -1.71296 ,  0.317655],
                  [ 2.080777, -0.178092,  0.317655],
                  [-1.265386, -0.730571, -1.167899],
                  [-2.080777, -0.178092,  0.317655],
                  [-1.194621, -1.71296 ,  0.317655]]},
'C4H4O': {
    'description': "Furan (cyclic C4H4O), C2v symm.",
    'name': "C_4H_4O (furan)",
    'database': 'G2-2',
    'enthalpy': -8.3,
    'ZPE': 43.2116,
    'thermal correction': 2.9480,
    'symbols': 'OCCCCHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  1.163339],
                  [ 0.      ,  1.0947  ,  0.348039],
                  [ 0.      , -1.0947  ,  0.348039],
                  [ 0.      ,  0.7132  , -0.962161],
                  [ 0.      , -0.7132  , -0.962161],
                  [ 0.      ,  2.049359,  0.851113],
                  [ 0.      , -2.049359,  0.851113],
                  [ 0.      ,  1.370828, -1.819738],
                  [ 0.      , -1.370828, -1.819738]]},
'C4H4S': {
    'description': "Thiophene (cyclic C4H4S), C2v symm.",
    'name': "C_4H_4S (thiophene)",
    'database': 'G2-2',
    'enthalpy': 27.5,
    'ZPE': 41.2029,
    'thermal correction': 3.1702,
    'symbols': 'SCCCCHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[  0.00000000e+00,   0.00000000e+00,   1.18975300e+00],
                  [  0.00000000e+00,   1.23387600e+00,  -1.47400000e-03],
                  [  0.00000000e+00,  -1.23387600e+00,  -1.47400000e-03],
                  [  0.00000000e+00,   7.09173000e-01,  -1.27232200e+00],
                  [  0.00000000e+00,  -7.09173000e-01,  -1.27232200e+00],
                  [  0.00000000e+00,   2.27534300e+00,   2.91984000e-01],
                  [  0.00000000e+00,  -2.27534300e+00,   2.91984000e-01],
                  [  0.00000000e+00,   1.32193400e+00,  -2.16723100e+00],
                  [  0.00000000e+00,  -1.32193400e+00,  -2.16723100e+00]]},
'C4H4NH': {
    'description': "Pyrrole (Planar cyclic C4H4NH), C2v symm.",
    'name': "C_4H_5N (pyrrole)",
    'database': 'G2-2',
    'enthalpy': 25.9,
    'ZPE': 50.9688,
    'thermal correction': 3.1156,
    'symbols': 'HNCCCCHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  2.129296],
                  [ 0.      ,  0.      ,  1.118684],
                  [ 0.      ,  1.124516,  0.333565],
                  [ 0.      , -1.124516,  0.333565],
                  [ 0.      ,  0.708407, -0.983807],
                  [ 0.      , -0.708407, -0.983807],
                  [ 0.      ,  2.112872,  0.770496],
                  [ 0.      , -2.112872,  0.770496],
                  [ 0.      ,  1.357252, -1.849085],
                  [ 0.      , -1.357252, -1.849085]]},
'C5H5N': {
    'description': "Pyridine (cyclic C5H5N), C2v symm.",
    'name': "C_5H_5N (pyridine)",
    'database': 'G2-2',
    'enthalpy': 33.6,
    'ZPE': 54.8230,
    'thermal correction': 3.3007,
    'symbols': 'NCCCCCHHHHH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  1.424672],
                  [ 0.      ,  0.      , -1.386178],
                  [ 0.      ,  1.144277,  0.720306],
                  [ 0.      , -1.144277,  0.720306],
                  [ 0.      , -1.196404, -0.672917],
                  [ 0.      ,  1.196404, -0.672917],
                  [ 0.      ,  0.      , -2.473052],
                  [ 0.      ,  2.060723,  1.307477],
                  [ 0.      , -2.060723,  1.307477],
                  [ 0.      , -2.155293, -1.183103],
                  [ 0.      ,  2.155293, -1.183103]]},
'H2': {
    'description': "H2. D*h symm.",
    'name': "H_2",
    'database': 'G2-2',
    'enthalpy': 0.0,
    'ZPE': 6.2908,
    'thermal correction': 2.0739,
    'ionization energy': 15.43,
    'symbols': 'HH',
    'magmoms': None,
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.368583],
                  [ 0.      ,  0.      , -0.368583]]},
'SH': {
    'description': "SH radical, C*v symm.",
    'name': "HS",
    'database': 'G2-2',
    'enthalpy': 34.2,
    'ZPE': 3.7625,
    'thermal correction': 2.0739,
    'symbols': 'SH',
    'magmoms': [ 1.,  0.],
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.079083],
                  [ 0.      ,  0.      , -1.26533 ]]},
'CCH': {
    'description': "CCH radical, C*v symm.",
    'name': "CCH",
    'database': 'G2-2',
    'enthalpy': 135.1,
    'ZPE': 7.8533,
    'thermal correction': 2.7830,
    'symbols': 'CCH',
    'magmoms': [ 0.,  1.,  0.],
    'charges': None,
    'positions': [[ 0.      ,  0.      , -0.462628],
                  [ 0.      ,  0.      ,  0.717162],
                  [ 0.      ,  0.      , -1.527198]]},
'C2H3': {
    'description': "C2H3 radical, Cs symm, 2-A'.",
    'name': "C_2H_3 (2A')",
    'database': 'G2-2',
    'enthalpy': 71.6,
    'ZPE': 22.5747,
    'thermal correction': 2.5483,
    'symbols': 'CCHHH',
    'magmoms': [ 0.,  1.,  0.,  0.,  0.],
    'charges': None,
    'positions': [[ 0.049798, -0.576272,  0.      ],
                  [ 0.049798,  0.710988,  0.      ],
                  [-0.87675 , -1.151844,  0.      ],
                  [ 0.969183, -1.154639,  0.      ],
                  [-0.690013,  1.498185,  0.      ]]},
'CH3CO': {
    'description': "CH3CO radical, HCCO cis, Cs symm, 2-A'.",
    'name': "CH_3CO (2A')",
    'database': 'G2-2',
    'enthalpy': -2.4,
    'ZPE': 26.6070,
    'thermal correction': 3.0842,
    'symbols': 'CCHHHO',
    'magmoms': [ 0.1,  0.6,  0. ,  0. ,  0. ,  0.3],
    'charges': None,
    'positions': [[-0.978291, -0.647814,  0.      ],
                  [ 0.      ,  0.506283,  0.      ],
                  [-0.455551, -1.607837,  0.      ],
                  [-1.617626, -0.563271,  0.881061],
                  [-1.617626, -0.563271, -0.881061],
                  [ 1.195069,  0.447945,  0.      ]]},
'H2COH': {
    'description': "H2COH radical, C1 symm.",
    'name': "H_2COH (2A)",
    'database': 'G2-2',
    'enthalpy': -4.1,
    'ZPE': 23.1294,
    'thermal correction': 2.6726,
    'symbols': 'COHHH',
    'magmoms': [ 0.7,  0.3,  0. ,  0. ,  0. ],
    'charges': None,
    'positions': [[ 0.687448,  0.029626, -0.082014],
                  [-0.672094, -0.125648,  0.030405],
                  [-1.09185 ,  0.740282, -0.095167],
                  [ 1.122783,  0.975263,  0.225993],
                  [ 1.221131, -0.888116,  0.118015]]},
'CH3O': {
    'description': "CH3O radical, Cs symm, 2-A'.",
    'name': "CH_3O CS (2A')",
    'database': 'G2-2',
    'enthalpy': 4.1,
    'ZPE': 22.4215,
    'thermal correction': 2.4969,
    'symbols': 'COHHH',
    'magmoms': [ 0.,  1.,  0.,  0.,  0.],
    'charges': None,
    'positions': [[-0.008618, -0.586475,  0.      ],
                  [-0.008618,  0.799541,  0.      ],
                  [ 1.055363, -0.868756,  0.      ],
                  [-0.467358, -1.004363,  0.903279],
                  [-0.467358, -1.004363, -0.903279]]},
'CH3CH2O': {
    'description': "CH3CH2O radical, Cs symm, 2-A''.",
    'name': "CH_3CH_2O (2A'')",
    'database': 'G2-2',
    'enthalpy': -3.7,
    'ZPE': 39.4440,
    'thermal correction': 3.0158,
    'symbols': 'CCOHHHHH',
    'magmoms': [ 0.,  0.,  1.,  0.,  0.,  0.,  0.,  0.],
    'charges': None,
    'positions': [[  1.00475700e+00,  -5.68263000e-01,   0.00000000e+00],
                  [  0.00000000e+00,   5.88691000e-01,   0.00000000e+00],
                  [ -1.26006200e+00,   7.29000000e-04,   0.00000000e+00],
                  [  1.46956000e-01,   1.20468100e+00,   8.96529000e-01],
                  [  1.46956000e-01,   1.20468100e+00,  -8.96529000e-01],
                  [  2.01936300e+00,  -1.64100000e-01,   0.00000000e+00],
                  [  8.69340000e-01,  -1.18683200e+00,   8.88071000e-01],
                  [  8.69340000e-01,  -1.18683200e+00,  -8.88071000e-01]]},
'CH3S': {
    'description': "CH3S radical, Cs symm, 2-A'.",
    'name': "CH_3S (2A')",
    'database': 'G2-2',
    'enthalpy': 29.8,
    'ZPE': 21.9415,
    'thermal correction': 2.6054,
    'symbols': 'CSHHH',
    'magmoms': [ 0.,  1.,  0.,  0.,  0.],
    'charges': None,
    'positions': [[-0.003856,  1.106222,  0.      ],
                  [-0.003856, -0.692579,  0.      ],
                  [ 1.043269,  1.427057,  0.      ],
                  [-0.479217,  1.508437,  0.895197],
                  [-0.479217,  1.508437, -0.895197]]},
'C2H5': {
    'description': "C2H5 radical, Staggered, Cs symm, 2-A'.",
    'name': "C_2H_5 (2A')",
    'database': 'G2-2',
    'enthalpy': 28.9,
    'ZPE': 36.5675,
    'thermal correction': 3.0942,
    'symbols': 'CCHHHHH',
    'magmoms': [ 0.,  1.,  0.,  0.,  0.,  0.,  0.],
    'charges': None,
    'positions': [[-0.014359, -0.694617,  0.      ],
                  [-0.014359,  0.794473,  0.      ],
                  [ 1.006101, -1.104042,  0.      ],
                  [-0.517037, -1.093613,  0.884839],
                  [-0.517037, -1.093613, -0.884839],
                  [ 0.100137,  1.346065,  0.923705],
                  [ 0.100137,  1.346065, -0.923705]]},
'C3H7': {
    'description': "(CH3)2CH radical, Cs symm, 2-A'.",
    'name': "(CH_3)_2CH (2A')",
    'database': 'G2-2',
    'enthalpy': 21.5,
    'ZPE': 54.2928,
    'thermal correction': 3.8435,
    'symbols': 'CCCHHHHHHH',
    'magmoms': [ 1.,  0.,  0.,  0.,  0.,  0.,  0.,  0.,  0.,  0.],
    'charges': None,
    'positions': [[ 0.014223,  0.54385 ,  0.      ],
                  [ 0.014223, -0.199742,  1.291572],
                  [ 0.014223, -0.199742, -1.291572],
                  [-0.32289 ,  1.575329,  0.      ],
                  [ 0.221417,  0.459174,  2.138477],
                  [ 0.221417,  0.459174, -2.138477],
                  [-0.955157, -0.684629,  1.484633],
                  [ 0.767181, -0.995308,  1.286239],
                  [ 0.767181, -0.995308, -1.286239],
                  [-0.955157, -0.684629, -1.484633]]},
'C3H9C': {
    'description': "t-Butyl radical, (CH3)3C, C3v symm.",
    'name': "(CH_3)_3C (t-butyl radical)",
    'database': 'G2-2',
    'enthalpy': 12.3,
    'ZPE': 71.7833,
    'thermal correction': 4.6662,
    'symbols': 'CCCCHHHHHHHHH',
    'magmoms': [1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.],
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.191929],
                  [ 0.      ,  1.478187, -0.020866],
                  [ 1.280147, -0.739093, -0.020866],
                  [-1.280147, -0.739093, -0.020866],
                  [ 0.      ,  1.731496, -1.093792],
                  [-0.887043,  1.945769,  0.417565],
                  [ 0.887043,  1.945769,  0.417565],
                  [ 1.49952 , -0.865748, -1.093792],
                  [ 2.128607, -0.204683,  0.417565],
                  [ 1.241564, -1.741086,  0.417565],
                  [-1.49952 , -0.865748, -1.093792],
                  [-1.241564, -1.741086,  0.417565],
                  [-2.128607, -0.204683,  0.417565]]},
'NO2': {
    'description': "NO2 radical, C2v symm, 2-A1.",
    'name': "NO_2",
    'database': 'G2-2',
    'enthalpy': 7.9,
    'ZPE': 5.4631,
    'thermal correction': 2.4366,
    'symbols': 'NOO',
    'magmoms': [ 1.,  0.,  0.],
    'charges': None,
    'positions': [[ 0.      ,  0.      ,  0.332273],
                  [ 0.      ,  1.118122, -0.14537 ],
                  [ 0.      , -1.118122, -0.14537 ]]},
}


# all constituent atoms
atoms_g22 = []
for f in data.keys():
    s = Atoms(symbols=data[f]['symbols'], positions=data[f]['positions'])
    for a in s:
        atoms_g22.append(a.symbol)
# unique atoms
atoms_g22 = list(set(atoms_g22))

# add remaining atoms from G2_1
from ase.structure import molecule

from ase.data.g2_1 import data as data1

for a in atoms_g22:
    if not a in data.keys():
        data[a] = data1[a]

from ase.data.g2_1 import get_ionization_energy
from ase.data.g2_1 import get_atomization_energy
