/*
mediastreamer2 library - modular sound and video processing and streaming
Copyright (C) 2006  Simon MORLAT (simon.morlat@linphone.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


#ifdef HAVE_CONFIG_H
#include "mediastreamer-config.h"
#endif

#include "mediastreamer2/mscommon.h"
#include "mediastreamer2/msvideo.h"
#include "mediastreamer2/msfilter.h"
#include "mediastreamer2/msticker.h"
#include "mediastreamer2/mswebcam.h"


#ifndef NO_FFMPEG
#include "ffmpeg-priv.h"
#else
#define FF_INPUT_BUFFER_PADDING_SIZE 32
#endif

#if TARGET_OS_IPHONE
#include <CoreGraphics/CGDataProvider.h>
#include <CoreGraphics/CGImage.h>
#include <CoreGraphics/CGContext.h>
#include <CoreGraphics/CGBitmapContext.h>
#endif

#include <sys/stat.h>

#ifdef WIN32
#include <fcntl.h>
#include <sys/types.h>
#include <io.h>
#include <stdio.h>
#include <malloc.h>
#endif

static mblk_t *jpeg2yuv(uint8_t *jpgbuf, int bufsize, MSVideoSize *reqsize){
#ifndef NO_FFMPEG
	AVCodecContext av_context;
	int got_picture=0;
	AVFrame *orig;
	mblk_t *ret;
	struct SwsContext *sws_ctx;
	AVPacket pkt;
	MSPicture dest;
	AVCodec *codec=avcodec_find_decoder(AV_CODEC_ID_MJPEG);

	if (codec==NULL){
		ms_error("Could not find MJPEG decoder in ffmpeg.");
		return NULL;
	}

	orig = av_frame_alloc();
	if (!orig)
		return NULL;

	avcodec_get_context_defaults(&av_context);
	if (avcodec_open(&av_context,codec)<0){
		av_frame_free(&orig);
		ms_error("jpeg2yuv: avcodec_open failed");
		return NULL;
	}
	av_init_packet(&pkt);
	pkt.data=jpgbuf;
	pkt.size=bufsize;

	if (avcodec_decode_video2(&av_context,&orig,&got_picture,&pkt) < 0) {
		ms_error("jpeg2yuv: avcodec_decode_video failed");
		av_frame_free(&orig);
		avcodec_close(&av_context);
		return NULL;
	}
	ret=ms_yuv_buf_alloc(&dest, reqsize->width,reqsize->height);
	/* not using SWS_FAST_BILINEAR because it doesn't play well with
	 * av_context.pix_fmt set to PIX_FMT_YUVJ420P by jpeg decoder */
	sws_ctx=sws_getContext(av_context.width,av_context.height,av_context.pix_fmt,
		reqsize->width,reqsize->height,PIX_FMT_YUV420P,SWS_BILINEAR,
                NULL, NULL, NULL);
	if (sws_ctx==NULL) {
		ms_error("jpeg2yuv: ms_sws_getContext() failed.");
		av_frame_free(&orig);
		avcodec_close(&av_context);
		freemsg(ret);
		return NULL;
	}

#if LIBSWSCALE_VERSION_INT >= AV_VERSION_INT(0,9,0)	
	if (sws_scale(sws_ctx,(const uint8_t* const *)orig->data,orig->linesize,0,av_context.height,dest.planes,dest.strides)<0){
#else
	if (sws_scale(sws_ctx,(uint8_t**)orig->data,orig->linesize,0,av_context.height,dest.planes,dest.strides)<0){
#endif
		ms_error("jpeg2yuv: ms_sws_scale() failed.");
		sws_freeContext(sws_ctx);
		avcodec_close(&av_context);
		av_frame_free(&orig);
		freemsg(ret);
		return NULL;
	}
	sws_freeContext(sws_ctx);
	avcodec_close(&av_context);
	av_frame_free(&orig);
	return ret;
#elif TARGET_OS_IPHONE
	MSPicture dest;
	CGDataProviderRef dataProvider = CGDataProviderCreateWithData(NULL, jpgbuf, bufsize, NULL);
	// use the data provider to get a CGImage; release the data provider
	CGImageRef image = CGImageCreateWithJPEGDataProvider(dataProvider, NULL, FALSE, 
	                                                    kCGRenderingIntentDefault);
	                                                    CGDataProviderRelease(dataProvider);
	reqsize->width = CGImageGetWidth(image);
	reqsize->height = CGImageGetHeight(image);
    
	uint8_t* tmp = (uint8_t*) malloc(reqsize->width * reqsize->height * 4);
	mblk_t* ret=ms_yuv_buf_alloc(&dest, reqsize->width, reqsize->height);
	CGColorSpaceRef colourSpace = CGColorSpaceCreateDeviceRGB();
	CGContextRef imageContext =
	    CGBitmapContextCreate(tmp, reqsize->width, reqsize->height, 8, reqsize->width*4, colourSpace, kCGImageAlphaNoneSkipLast);
	CGColorSpaceRelease(colourSpace);
	// draw the image to the context, release it
	CGContextDrawImage(imageContext, CGRectMake(0, 0, reqsize->width, reqsize->height), image);
	CGImageRelease(image);
	
	/* convert tmp/RGB -> ret/YUV */
	for(int y=0; y<reqsize->height; y++) {
        for(int x=0; x<reqsize->width; x++) {
            uint8_t r = tmp[y * reqsize->width * 4 + x * 4 + 0];
            uint8_t g = tmp[y * reqsize->width * 4 + x * 4 + 1];
            uint8_t b = tmp[y * reqsize->width * 4 + x * 4 + 2];

            // Y
            *dest.planes[0]++ = (uint8_t)((0.257 * r) + (0.504 * g) + (0.098 * b) + 16);		
		
            // U/V subsampling
            if ((y % 2==0) && (x%2==0)) {
                uint32_t r32=0, g32=0, b32=0;
                for(int i=0; i<2; i++) {
                    for(int j=0; j<2; j++) {
                        r32 += tmp[(y+i) * reqsize->width * 4 + (x+j) * 4 + 0];
                        g32 += tmp[(y+i) * reqsize->width * 4 + (x+j) * 4 + 1];
                        b32 += tmp[(y+i) * reqsize->width * 4 + (x+j) * 4 + 2];                    
                    }
                }
                r32 = (uint32_t)(r32 * 0.25f); g32 = (uint32_t)(g32 * 0.25f); b32 = (uint32_t) (b32 * 0.25f);
            
                // U
                *dest.planes[1]++ = (uint8_t)(-(0.148 * r32) - (0.291 * g32) + (0.439 * b32) + 128);
                // V
                *dest.planes[2]++ = (uint8_t)((0.439 * r32) - (0.368 * g32) - (0.071 * b32) + 128);
            }
        }
	}
    free(tmp);
	return ret;
#else
	return NULL;
#endif
		}

#ifndef MS2_MINIMAL_SIZE
unsigned char def_mire[] = /* 22092 */
{0xFF,0xD8,0xFF,0xE0,0x00,0x10,0x4A,0x46,0x49,0x46,0x00,0x01,0x01,0x01,0x00
,0x59,0x00,0x59,0x00,0x00,0xFF,0xDB,0x00,0x43,0x00,0x05,0x03,0x04,0x04,0x04
,0x03,0x05,0x04,0x04,0x04,0x05,0x05,0x05,0x06,0x07,0x0C,0x08,0x07,0x07,0x07
,0x07,0x0F,0x0B,0x0B,0x09,0x0C,0x11,0x0F,0x12,0x12,0x11,0x0F,0x11,0x11,0x13
,0x16,0x1C,0x17,0x13,0x14,0x1A,0x15,0x11,0x11,0x18,0x21,0x18,0x1A,0x1D,0x1D
,0x1F,0x1F,0x1F,0x13,0x17,0x22,0x24,0x22,0x1E,0x24,0x1C,0x1E,0x1F,0x1E,0xFF
,0xDB,0x00,0x43,0x01,0x05,0x05,0x05,0x07,0x06,0x07,0x0E,0x08,0x08,0x0E,0x1E
,0x14,0x11,0x14,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E
,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E
,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E
,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0xFF,0xC0,0x00,0x11,0x08,0x01,0x20
,0x01,0x60,0x03,0x01,0x22,0x00,0x02,0x11,0x01,0x03,0x11,0x01,0xFF,0xC4,0x00
,0x1D,0x00,0x01,0x00,0x02,0x03,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x06,0x07,0x04,0x05,0x08,0x03,0x02,0x01,0x09,0xFF,0xC4
,0x00,0x4A,0x10,0x00,0x02,0x01,0x03,0x03,0x02,0x04,0x03,0x06,0x03,0x04,0x05
,0x0A,0x06,0x03,0x00,0x01,0x02,0x03,0x04,0x05,0x11,0x00,0x06,0x12,0x21,0x31
,0x07,0x13,0x22,0x41,0x14,0x51,0x61,0x08,0x23,0x32,0x52,0x71,0x81,0x15,0x42
,0x91,0x16,0x33,0x62,0xA1,0x43,0x53,0x72,0x82,0xB1,0x24,0x34,0x44,0x63,0x73
,0x83,0x92,0xA2,0xC1,0xC2,0x17,0x26,0x45,0xA3,0xD1,0xE1,0x54,0x93,0xA4,0xFF
,0xC4,0x00,0x19,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x03,0x04,0x05,0xFF,0xC4,0x00,0x22
,0x11,0x01,0x00,0x03,0x00,0x02,0x01,0x04,0x03,0x01,0x00,0x00,0x00,0x00,0x00
,0x00,0x00,0x00,0x01,0x02,0x11,0x21,0x31,0x03,0x04,0x05,0x12,0x41,0x13,0x51
,0x71,0xA1,0xFF,0xDA,0x00,0x0C,0x03,0x01,0x00,0x02,0x11,0x03,0x11,0x00,0x3F
,0x00,0xE3,0x2D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D
,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34
,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D
,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34
,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3
,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40
,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3
,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D
,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x4D,0x76,0xBF,0x87,0x17,0xDB,0xCC,0xC6
,0x39,0x20,0xA9,0x86,0x40,0xE6,0x21,0x4B,0x0C,0x1E,0x75,0x49,0x93,0xD6,0x02
,0x32,0x65,0x56,0x23,0xC9,0x14,0x15,0x95,0xD1,0x8A,0xB8,0x64,0x57,0xED,0xA0
,0x85,0x69,0xAE,0xB3,0xD9,0x5F,0x65,0xBA,0xFB,0x95,0x3A,0xD6,0x6E,0x2A,0x2B
,0x3E,0xDA,0xB7,0x42,0xAE,0xEA,0x2A,0x66,0xA8,0xAB,0xAC,0x91,0x49,0x2C,0x0C
,0xC5,0x64,0x86,0x35,0xC0,0x20,0x65,0x55,0x0E,0x17,0xAA,0xE7,0x2C,0x6A,0x9D
,0xF5,0xE1,0x85,0xBA,0x1B,0xED,0x6D,0xB3,0x65,0x54,0x7F,0x69,0xBE,0xF0,0xB8
,0xB9,0x50,0xCC,0x60,0xB7,0xD3,0xFA,0x8E,0x61,0x51,0x20,0x73,0x26,0x06,0x00
,0x65,0x99,0xF1,0xEF,0x93,0x9D,0x05,0x45,0xA6,0xA7,0x56,0xEF,0x0D,0xAE,0x95
,0xD7,0xC9,0xAD,0xD4,0xD2,0xC9,0x53,0xC3,0x80,0x54,0xA6,0xA7,0xE7,0x3F,0x27
,0x56,0x2A,0x8E,0x85,0x82,0x46,0xC1,0x94,0x2B,0x2B,0xB8,0x27,0x3C,0xA3,0xF3
,0x57,0x04,0xDC,0x16,0xEF,0xB2,0xAE,0xE4,0x9E,0xDF,0x15,0x6D,0x4D,0x9A,0xB6
,0x01,0xE5,0xAB,0x4B,0x1F,0xF1,0x54,0x2D,0x8C,0x75,0x25,0x7E,0x1F,0xCC,0x5E
,0xB9,0xF4,0xF9,0x6C,0x7F,0xDA,0xD0,0x73,0x2E,0x9A,0xB9,0xF7,0x77,0x80,0x1B
,0x96,0xC9,0x4F,0x5D,0x50,0x12,0x5E,0x4B,0x21,0x34,0xB0,0x33,0x46,0xAB,0xE5
,0x96,0xE8,0x1D,0xE7,0x68,0x25,0x24,0x2E,0x7A,0xAC,0x1D,0x48,0xEC,0xB9,0xE9
,0xA8,0xDA,0x9E,0x0D,0xEE,0x3B,0xD7,0x04,0x16,0xCB,0xD4,0xB5,0x2B,0x52,0x04
,0xA9,0x4F,0x42,0x5E,0x9F,0xC8,0xE9,0xEA,0xF3,0x50,0xBC,0x81,0x8F,0xA8,0x60
,0x40,0xE0,0x60,0x1C,0x9C,0x90,0x02,0xAF,0xD3,0x5D,0xE7,0xE1,0x2F,0xD9,0x89
,0x6D,0x96,0x23,0x25,0xD6,0xAE,0x4A,0x3A,0xCA,0x88,0x84,0x72,0xAC,0x66,0x48
,0x54,0xA9,0x09,0xC8,0x37,0x95,0x22,0xCB,0x20,0x62,0x8A,0xD8,0x79,0x15,0x03
,0x03,0xF7,0x23,0x52,0x4B,0x67,0xD9,0x73,0x69,0xDB,0x9E,0xA6,0x58,0xAB,0x26
,0x92,0x49,0xA4,0x67,0x0F,0x14,0x92,0xD3,0xCA,0x80,0x9C,0xF1,0x59,0x43,0xB9
,0x50,0x3E,0x8B,0x9F,0x99,0xD0,0x7F,0x3A,0x34,0xD7,0x73,0x6F,0xCF,0xB2,0xD5
,0x75,0xC9,0x6B,0x27,0xA7,0xBC,0xC9,0x53,0x51,0x53,0xC3,0x9B,0xD4,0xD3,0xA5
,0x64,0x8C,0xA9,0xC7,0x88,0x6A,0x99,0x9C,0x4E,0x30,0x11,0x57,0xEE,0xF1,0xE9
,0x18,0xED,0xD3,0x55,0x65,0xF7,0xEC,0xED,0x57,0x6F,0x90,0xCF,0x74,0xA3,0xB4
,0x22,0x45,0x10,0x48,0xA9,0xE8,0x6E,0x73,0x5A,0xBE,0x24,0x82,0x49,0x66,0x92
,0xE3,0x1B,0x0E,0x67,0x38,0xC2,0x12,0xB8,0x51,0xD3,0x39,0x24,0x39,0xB3,0x4D
,0x5A,0x97,0x3F,0x03,0x37,0xD5,0xB2,0xCA,0xF5,0x55,0x7B,0x63,0x74,0xBD,0x56
,0x14,0xC5,0xF0,0x56,0xA4,0xAC,0xA4,0x61,0xC8,0x72,0x2D,0x3C,0x13,0x39,0x00
,0x0C,0x9C,0x84,0x6C,0x90,0x07,0x4C,0xE4,0x44,0xF7,0xB6,0xC3,0xDC,0x7B,0x41
,0x61,0x7B,0xC5,0x05,0x4C,0x29,0x22,0xA9,0xE5,0x2D,0x1D,0x45,0x3F,0x16,0x39
,0xF4,0x71,0x9E,0x34,0x66,0x20,0x00,0x4B,0x28,0x65,0x1C,0xD7,0xD5,0x93,0x8D
,0x04,0x5F,0x4D,0x34,0xD0,0x34,0xD3,0x4D,0x03,0x4D,0x34,0xD0,0x34,0xD3,0x4D
,0x03,0x4D,0x34,0xD0,0x34,0xD3,0x4D,0x03,0x4D,0x34,0xD0,0x34,0xD3,0x4D,0x03
,0x4D,0x34,0xD0,0x34,0xD3,0x4D,0x03,0x4D,0x34,0xD0,0x34,0xD3,0x4D,0x03,0x4D
,0x34,0xD0,0x34,0xD6,0xCE,0xC3,0x61,0xBB,0xDF,0x65,0x68,0xED,0x74,0x4F,0x38
,0x56,0x0A,0xF2,0x33,0x2C,0x71,0xAB,0x15,0x66,0x55,0x2E,0xC4,0x28,0x66,0x08
,0xC1,0x54,0x9C,0xB1,0x18,0x50,0x4F,0x4D,0x5C,0x3E,0x1D,0x7D,0x9E,0xF7,0x25
,0xE2,0xE6,0x45,0x5D,0x11,0xA9,0x58,0x64,0x19,0x8C,0xAB,0xC5,0x1F,0x42,0x8D
,0xF7,0x81,0xB8,0x48,0x32,0x39,0x83,0x1B,0xF9,0x0D,0x8C,0x32,0x39,0xED,0xA0
,0xA4,0x68,0xE9,0x2A,0xAB,0x6A,0x05,0x3D,0x1D,0x34,0xD5,0x33,0x15,0x66,0x11
,0xC4,0x85,0xDB,0x8A,0xA9,0x66,0x38,0x1D,0x70,0x14,0x12,0x7E,0x40,0x13,0xAB
,0x1F,0x65,0xF8,0x39,0xB8,0xEF,0xF7,0x1A,0x58,0x63,0x8D,0x6A,0xD6,0x41,0xCA
,0x48,0xE8,0x9C,0xBE,0x00,0x93,0x8E,0x3C,0xD0,0xAC,0x85,0x59,0x41,0x22,0x58
,0x84,0xCA,0xB9,0x5E,0x58,0xEB,0x8E,0xA3,0xA0,0xF0,0x87,0xC3,0x8F,0x0E,0xE8
,0x21,0xB9,0xF8,0x87,0x79,0xB7,0xC4,0x59,0x63,0x48,0xE8,0xE0,0x41,0x1A,0xCC
,0xD1,0x82,0x10,0x2A,0x22,0x87,0x99,0xC6,0x72,0x1C,0x2F,0x9B,0x92,0x72,0xED
,0x92,0x4E,0x45,0xCB,0xC5,0x4B,0x9B,0x52,0xB5,0xBB,0xC3,0xBD,0xB5,0x4D,0xB6
,0xED,0xEC,0x72,0x6E,0x17,0x18,0x41,0x9E,0x43,0x8F,0xC6,0xB4,0xEA,0x7B,0x9F
,0xCD,0x2B,0x67,0xE6,0x9A,0x26,0xB4,0x3B,0x5F,0xC0,0x3D,0xB3,0xB3,0xEC,0x70
,0xDE,0x37,0xED,0xCE,0xD5,0x63,0xA2,0xA6,0xF5,0x33,0xB1,0x44,0x90,0xB3,0x46
,0x11,0xB9,0x48,0xC5,0x89,0x2C,0x01,0xCA,0x73,0x92,0x26,0xE4,0xD8,0x89,0x39
,0x11,0xA9,0x0A,0x78,0x93,0x64,0xB2,0x53,0x0A,0x0F,0x0B,0x36,0x7C,0x7C,0x11
,0x3C,0xB5,0xBA,0xDD,0x23,0x68,0x21,0x0B,0x9E,0xD1,0xC7,0xD2,0x57,0x5F,0x70
,0xB8,0x89,0x3E,0x47,0x55,0xFD,0x54,0x51,0x55,0x5E,0x96,0xE3,0x7D,0xB9,0x55
,0xDE,0xEF,0x03,0xAA,0xD5,0x57,0xC9,0xE6,0xBC,0x40,0xE7,0xAA,0x28,0x1C,0x21
,0x5E,0xFF,0x00,0x85,0x54,0x6B,0x16,0xDF,0xB8,0x26,0xBB,0xD5,0x79,0x1B,0x7B
,0x6F,0xDD,0xEF,0x0A,0x7F,0x0C,0xF1,0x24,0x71,0x42,0xFD,0xFA,0xAB,0xCA,0xE8
,0x18,0x74,0x3D,0x46,0x41,0xF6,0x3A,0xA3,0x6B,0xB9,0x2B,0x6E,0x7B,0x82,0x41
,0x2E,0xF2,0xBE,0xD5,0x5E,0x8B,0x1E,0x49,0x46,0xC3,0x85,0x28,0x23,0xF2,0x53
,0x27,0xA5,0xB1,0xF3,0x6E,0x6C,0x3E,0x7A,0xFC,0x8F,0xCE,0x70,0x15,0x63,0x14
,0xF1,0x81,0x80,0x0E,0x0B,0x63,0xF4,0x1D,0x07,0xB7,0xCF,0xF4,0x1A,0xC3,0xB8
,0xD6,0xD6,0xD9,0x6A,0x24,0xFE,0x37,0xB7,0x2E,0x36,0xE8,0x55,0x62,0x69,0xAA
,0xC9,0x8A,0x58,0x93,0xCC,0x62,0x89,0xCD,0x91,0xC9,0x1E,0xA0,0x47,0x6C,0x0E
,0x99,0xC0,0x23,0x59,0x95,0x43,0xD2,0x5A,0x6A,0x8F,0x26,0x11,0xDF,0x07,0x8E
,0x7D,0xBA,0xB7,0xB7,0xB7,0x6C,0x1F,0xAE,0x83,0x15,0xA9,0x8D,0x0D,0xC6,0x7B
,0x9D,0x8E,0xE3,0x5F,0x67,0xBE,0x4B,0x01,0x84,0xD6,0xDB,0xE5,0xE1,0x3B,0x21
,0xC7,0xA5,0xBD,0x8A,0xE5,0x47,0xE2,0xE8,0x30,0x31,0x83,0x83,0xA9,0x7E,0xCC
,0xF1,0x2F,0xC4,0x8D,0xB7,0x6F,0x68,0x2A,0x2A,0x36,0xA5,0x64,0x5C,0x8B,0xC9
,0x23,0xDB,0x24,0x8A,0x69,0x4F,0xE7,0x92,0x45,0x9B,0x05,0x88,0x03,0x2C,0x54
,0x9F,0xA9,0xD4,0x5E,0x36,0x72,0x85,0x28,0xE9,0xC4,0x4A,0x72,0x7C,0xC9,0x14
,0x81,0x9E,0xBD,0x78,0xF7,0x27,0x3D,0xF3,0x8C,0xFC,0xF5,0xAF,0xB8,0xDC,0x6D
,0x74,0x35,0x4B,0x0D,0x54,0xD2,0x57,0x57,0xFE,0x38,0xE9,0x62,0x43,0x2C,0x83
,0xAF,0x42,0x23,0x5F,0xC2,0x3D,0xB9,0x9C,0x7D,0x5B,0x41,0x6F,0x51,0xF8,0xFD
,0xBC,0x0B,0xE2,0x4D,0x8F,0x63,0xAA,0x8F,0xFD,0x72,0x5E,0xA5,0x84,0x1F,0xD0
,0x1A,0x76,0xFF,0x00,0x89,0x1F,0x5D,0x6B,0x77,0x87,0xDA,0x53,0x73,0xDA,0x95
,0x63,0xA4,0xD9,0xB6,0x01,0x55,0x2E,0x7E,0x1E,0x9D,0xAF,0x33,0x4F,0x34,0xC7
,0xFC,0x31,0xA5,0x3A,0x9F,0xDF,0x38,0x1E,0xE4,0x6A,0xB5,0x96,0x2B,0xBD,0xD6
,0x03,0x1C,0x93,0xC9,0x64,0x8D,0xC7,0x41,0x09,0x59,0x2A,0x40,0x23,0x23,0x2D
,0x86,0x44,0xEC,0xC3,0xA7,0x2F,0x6C,0x30,0x3D,0x35,0xE1,0x6A,0xA6,0xA1,0xB4
,0x5C,0x3F,0x86,0x5A,0x68,0x1E,0xE7,0x77,0xA8,0x19,0xF2,0x29,0x11,0xA7,0xAE
,0xA8,0x3E,0xC5,0xF2,0x49,0x23,0xFC,0x6E,0xC1,0x47,0xD3,0x4C,0x16,0xE6,0xCF
,0xF1,0xF7,0x7C,0x08,0x5E,0xA3,0x77,0xEC,0x7B,0x61,0x8D,0x90,0x32,0x43,0x6A
,0xAF,0x3F,0x10,0x87,0xF2,0x95,0x93,0xD0,0xDF,0xFF,0x00,0x60,0xFD,0xF5,0x38
,0xDB,0x9F,0x68,0x6F,0x0A,0xEE,0x91,0xBC,0x77,0x0D,0xC0,0x36,0xE5,0x7C,0x47
,0x13,0xD0,0xDE,0x93,0xE1,0xA5,0x88,0xFB,0x64,0xE4,0xA1,0x04,0x10,0x41,0x56
,0x3D,0xF5,0x00,0xDA,0x7E,0x0E,0xEE,0x7B,0xD2,0xA5,0x4E,0xEF,0xB8,0xAD,0x82
,0x8D,0xBA,0xFF,0x00,0x0E,0xB7,0x48,0x24,0xAA,0x61,0xF2,0x92,0x72,0x38,0xA7
,0xD4,0x46,0x09,0xF9,0x3E,0xAD,0xED,0xB1,0xB2,0xB6,0xA6,0xDA,0xB3,0xC9,0x69
,0xB2,0xD8,0x68,0xA9,0xA9,0x26,0xEB,0x50,0xA6,0x3E,0x6D,0x50,0x7D,0xDA,0x56
,0x6C,0xB4,0x87,0xEA,0xC4,0x9D,0x43,0x5A,0xE9,0xBC,0x69,0xF0,0xD6,0xA2,0x75
,0xA7,0xA1,0xF1,0x13,0x68,0xC2,0x0B,0x61,0xA7,0xA8,0xB8,0xC7,0xC7,0xB8,0xE8
,0x17,0x90,0x07,0x23,0x23,0x25,0x86,0x0E,0x0E,0x1B,0x53,0x8B,0x05,0xC2,0xD9
,0x74,0xB7,0xAD,0x4D,0xAE,0xEF,0x4F,0x77,0x81,0xBB,0xD4,0x43,0x3A,0x4A,0xAF
,0xFB,0xA7,0xA7,0xFA,0x63,0x5A,0xC8,0xAD,0x56,0xB8,0xAD,0xE6,0xDD,0x15,0xB6
,0x8D,0x28,0x8F,0x43,0x4E,0xB0,0x28,0x8C,0xFF,0x00,0xBB,0x8C,0x6A,0x23,0x5D
,0xE1,0x2F,0x86,0xD2,0x55,0x49,0x71,0x4D,0xAD,0x43,0x6C,0xA9,0xC1,0x2F,0x53
,0x6D,0x77,0xA0,0x71,0xEE,0x4F,0x28,0x19,0x0F,0xEF,0xA1,0xAD,0xE5,0xF2,0xD5
,0xB3,0x85,0xC2,0x4A,0x7A,0x1D,0xB8,0xB5,0x37,0x92,0x73,0x27,0xF0,0x85,0xF8
,0x69,0xD3,0x23,0x20,0xC9,0x3C,0x65,0x3C,0xBE,0x9D,0x47,0x27,0x04,0xFF,0x00
,0x28,0x3A,0x8E,0x78,0x8D,0xE0,0xE5,0xAF,0x78,0xD8,0xA9,0x21,0xBC,0x6E,0x9B
,0xF5,0xB9,0x68,0xA5,0xF8,0x88,0x91,0xAB,0x52,0xB2,0x18,0xA4,0x2A,0xC8,0x43
,0x35,0x4C,0x6C,0xF2,0x21,0x57,0x2A,0x55,0x9B,0x0C,0x0F,0x50,0x75,0x85,0x49
,0x68,0xDE,0xBB,0x01,0xCD,0x46,0xC5,0xA9,0x6D,0xC7,0xB6,0x90,0x97,0x9A,0xC1
,0x72,0x6C,0x54,0x82,0x58,0x97,0x7A,0x6A,0xA6,0xF5,0x3B,0x13,0xD7,0x8C,0xD9
,0x0D,0x93,0x87,0x19,0xE9,0x62,0xED,0x3B,0xC5,0x97,0x75,0x59,0x69,0xF7,0x2D
,0x9F,0x35,0x7C,0xF9,0xAA,0x0A,0x80,0x52,0x4A,0x79,0x14,0x95,0x78,0xD9,0x48
,0xCC,0x4E,0xA7,0x2A,0xC0,0x0C,0xF4,0xEB,0x9D,0x15,0xCB,0xD7,0xCF,0xB2,0x05
,0x61,0x6A,0xD9,0xAD,0xB7,0x3D,0xB5,0x73,0x35,0x62,0x46,0xF3,0x6A,0xED,0xF5
,0x34,0x73,0x42,0xCF,0x9E,0xB1,0xC7,0x0C,0xE2,0x10,0x41,0x39,0x55,0x28,0x10
,0x60,0x0E,0x38,0xE9,0xAA,0xDF,0xC4,0x2F,0xB2,0xC6,0xE6,0xDB,0x56,0xF8,0xA5
,0x85,0x84,0xDC,0x41,0x32,0xD5,0x43,0x50,0x6A,0xA3,0x20,0x95,0xEA,0xD1,0x2C
,0x4B,0x2A,0x04,0x51,0x23,0x1F,0x2C,0x54,0x13,0xE9,0xE8,0xA0,0x16,0xD7,0x74
,0xBD,0x63,0xDC,0x04,0x91,0x53,0x2A,0xDD,0x58,0x16,0x46,0x8E,0x17,0xF2,0xE9
,0x23,0x61,0xC8,0x15,0x92,0x5E,0xA5,0xF0,0x54,0xA3,0x05,0x0D,0x8E,0x99,0x41
,0xAC,0xDB,0x52,0xAB,0xC2,0xD4,0xCF,0x34,0x15,0x11,0xA0,0xF2,0xC4,0x74,0xD0
,0xF0,0x82,0x25,0x04,0x80,0xA3,0xBE,0x58,0x60,0x03,0xD7,0xD8,0x7A,0x57,0x41
,0xFC,0x84,0xB9,0x50,0xD5,0x5B,0xAB,0xA5,0xA2,0xAC,0x8C,0x47,0x34,0x64,0x64
,0x06,0x0C,0xA4,0x11,0x90,0xCA,0xCA,0x48,0x65,0x20,0x82,0x18,0x12,0x08,0x20
,0x82,0x41,0x07,0x58,0xDA,0xEA,0xFF,0x00,0xB6,0xD7,0x86,0x62,0xC5,0x34,0x77
,0xEA,0x4A,0x28,0x96,0x9A,0xB6,0x79,0x4A,0x4E,0x10,0x2F,0x09,0x95,0x5A,0x57
,0x50,0x4B,0x80,0x16,0x54,0x2E,0xE4,0x10,0x4F,0x9B,0x03,0x94,0x55,0xF3,0xDB
,0x3C,0xA1,0xA0,0x69,0xA6,0x9A,0x06,0x9A,0x69,0xA0,0x69,0xA6,0x9A,0x06,0x9A
,0x69,0xA0,0x69,0xA6,0x9A,0x06,0x9A,0x69,0xA0,0x69,0xA6,0x9A,0x06,0x9A,0x69
,0xA0,0x69,0xA6,0xBD,0xA8,0xE9,0xAA,0x6B,0x6B,0x21,0xA3,0xA3,0xA7,0x96,0xA6
,0xA6,0x79,0x16,0x28,0x61,0x89,0x0B,0xBC,0x8E,0xC7,0x0A,0xAA,0xA3,0xA9,0x24
,0x90,0x00,0x1D,0xF4,0x1E,0x3A,0x6A,0x6B,0xB3,0x3C,0x38,0xBE,0xEE,0x26,0x32
,0x34,0x33,0x53,0x53,0x87,0x31,0x16,0x11,0x02,0xC2,0x4F,0x58,0x08,0x4B,0xB2
,0x46,0x8D,0xC9,0x00,0x2A,0xEE,0xAF,0x86,0x05,0x55,0xFB,0x6B,0xA2,0x3C,0x2C
,0xFB,0x35,0x5D,0x12,0x68,0xEB,0xE7,0x0D,0x6B,0xF5,0x07,0x49,0xE4,0xCF,0x9F
,0x1F,0x54,0x60,0x51,0xDD,0x03,0xAB,0x2B,0xA6,0x41,0x58,0xA1,0x71,0x92,0x04
,0xC4,0x1D,0x07,0x2F,0xED,0xFD,0xAD,0x72,0xBC,0x3C,0x25,0x4C,0x54,0x71,0x4E
,0x0F,0x91,0x24,0xE1,0xC9,0x9C,0xFA,0xF0,0x22,0x8D,0x15,0xA5,0x97,0x2D,0x19
,0x42,0x63,0x46,0x0A,0xC4,0x72,0x2A,0x3A,0xEA,0xDF,0xF0,0xBB,0xEC,0xFF,0x00
,0x7D,0xDC,0xD4,0xB9,0x36,0x67,0xE3,0x3A,0x71,0x7A,0xBA,0xC7,0x2A,0xB0,0x64
,0x26,0x4A,0x24,0x6C,0x00,0x75,0x60,0xEA,0x4B,0xBB,0x1E,0x2D,0x93,0x00,0x23
,0x5D,0x49,0x2E,0xCF,0xF0,0xB7,0xC2,0x6B,0x0C,0xB7,0xAD,0xC2,0x69,0xCF,0x9C
,0xDC,0x58,0xCC,0xA5,0x9A,0xB2,0x53,0xD7,0xCB,0x09,0x92,0xF3,0xB1,0x3D,0x40
,0x91,0xA4,0x23,0xBE,0x40,0xEB,0xA8,0x6E,0xEB,0xF1,0x0B,0x75,0xEE,0x88,0xDA
,0x9E,0x9E,0x69,0xB6,0xBD,0xA0,0xFA,0x52,0x92,0x8E,0x40,0x2A,0x9D,0x3F,0xEB
,0x26,0x5F,0xC1,0x9E,0x9E,0x98,0xB0,0x47,0xE7,0x6D,0x13,0x5B,0x0B,0x36,0xDC
,0xF0,0xDB,0xC2,0x08,0x84,0x17,0xDB,0xEC,0x95,0xD7,0x99,0x23,0x25,0x6D,0xB4
,0x20,0xBC,0xE5,0x59,0x8B,0x15,0x58,0xE2,0x00,0xA4,0x65,0x99,0xB2,0x14,0x45
,0x13,0x67,0x2C,0xBE,0xFA,0xC3,0xBD,0xF8,0x9F,0xBB,0xAE,0x94,0xC2,0xDF,0xB6
,0x6D,0xF4,0xBB,0x2E,0xD0,0xA3,0x8C,0x64,0x24,0x73,0x56,0xF1,0xFF,0x00,0x0A
,0x8C,0xC3,0x0F,0xFF,0x00,0x73,0xF5,0x1A,0x87,0xD0,0x50,0x51,0x5B,0xA2,0x75
,0xA2,0xA6,0x8E,0x1E,0x67,0x9C,0x84,0x0F,0x54,0x8D,0xEE,0xCC,0xDD,0xD8,0xFD
,0x4E,0x4E,0xA3,0xD4,0xFB,0x86,0x6B,0xBA,0x2D,0x55,0x2C,0x96,0xAA,0x1B,0x6B
,0x4B,0xE5,0x25,0x4D,0xD6,0xE5,0xF0,0x68,0xE4,0xA0,0x90,0x11,0xC5,0x5D,0xB1
,0xC4,0x83,0xEA,0x29,0xDF,0x04,0x6A,0x8D,0xDA,0x53,0xD2,0x43,0x71,0x96,0xB5
,0xCD,0x45,0xC6,0xEB,0x28,0xC4,0xD5,0x55,0x12,0xB4,0xF5,0x2F,0xEF,0x86,0x91
,0xC9,0x2A,0x3E,0x43,0x21,0x7E,0x43,0x59,0x5C,0x26,0x94,0x1F,0x35,0xFC,0xB5
,0x3F,0xCB,0x19,0xEB,0xFB,0xB7,0x7F,0xE9,0x8C,0x7C,0xF5,0x1D,0xB9,0xEE,0x6A
,0x5B,0x2D,0x3C,0x32,0x2D,0xD7,0x68,0xDD,0x29,0x89,0x60,0xD1,0xDA,0x6F,0x0A
,0xF3,0x21,0x08,0xCF,0x85,0x8C,0xA8,0x0C,0x4F,0x1E,0x20,0x72,0x04,0xB1,0x03
,0xDF,0x52,0x52,0x8B,0x28,0x0E,0xC1,0x88,0x23,0x3C,0x0F,0x41,0xFB,0x8F,0xFF
,0x00,0x3A,0x0C,0x77,0x4A,0x59,0x69,0xE5,0xA3,0x8E,0x01,0x2C,0x52,0x86,0x59
,0x02,0x8C,0x29,0x0D,0x90,0xD9,0x6F,0x9F,0xCF,0x19,0x3A,0x86,0x5C,0x76,0x17
,0x08,0x22,0xA6,0x8B,0x7A,0xEE,0x4A,0x5A,0x28,0xD3,0xCB,0x86,0x91,0x6A,0x89
,0x55,0x51,0x92,0x15,0x14,0x01,0x9C,0x75,0xF6,0x27,0x53,0xB7,0x59,0x18,0x63
,0xCC,0xF2,0x97,0xFC,0x23,0x27,0xFA,0x9E,0x9F,0xE5,0xAD,0x30,0xBC,0x51,0x99
,0x25,0x86,0xC7,0x4E,0xD7,0x6A,0xA5,0x3C,0x64,0x30,0xB8,0x28,0xA4,0x7B,0x49
,0x33,0x74,0xE9,0xF2,0x05,0x98,0x7E,0x5D,0x06,0x97,0x6D,0x6C,0x2B,0x75,0xAE
,0xAF,0xE2,0x92,0x5B,0x8D,0x4B,0xF3,0x57,0xE7,0x5F,0x52,0x5F,0x2C,0xA7,0xD2
,0x7C,0xB5,0xC2,0x9C,0x67,0x20,0xB6,0x48,0x3D,0x71,0xAD,0xD5,0xD6,0xF5,0x68
,0xB3,0xCE,0x23,0x9E,0x69,0x2A,0xEE,0x1C,0x72,0xB0,0x46,0x3C,0xC9,0xB0,0x7A
,0x67,0x1D,0xA3,0x53,0xF3,0x3C,0x57,0xEB,0xAF,0x19,0x2D,0xBB,0x8A,0xE6,0x71
,0x72,0xBA,0x47,0x6F,0xA7,0x3D,0xE9,0xED,0xFC,0x83,0x11,0xF2,0x32,0x9C,0x31
,0xFD,0x54,0x27,0xE8,0x75,0x91,0xB7,0xA8,0xAD,0xEB,0x5A,0xD6,0x7D,0x9D,0x60
,0xAA,0xBE,0x5D,0x15,0xBE,0xF5,0x28,0x63,0x12,0x18,0xDC,0xF4,0x2D,0x2C,0xA7
,0x11,0xC4,0x4F,0xB9,0x66,0x04,0xFD,0x74,0x1A,0xD9,0xE1,0xDD,0x97,0xF5,0x2A
,0xD3,0xFF,0x00,0x67,0xE8,0x98,0x75,0x48,0x4F,0x2A,0x96,0x18,0x3D,0xDC,0xF4
,0x4C,0xE0,0x8E,0x80,0x75,0xEC,0xE7,0x59,0x56,0x68,0x2C,0x7B,0x65,0xD6,0xC5
,0x67,0x82,0xAA,0xE5,0x75,0xA8,0x25,0xBE,0x16,0x99,0x1A,0xA6,0xB2,0xA5,0x8E
,0x7D,0x6C,0xAB,0xD7,0xAE,0x41,0xE4,0x70,0x01,0x1D,0xF5,0x6C,0x6D,0xAF,0x04
,0xB7,0x25,0xE4,0xAD,0x46,0xF6,0xBF,0x2D,0xA2,0x95,0xBD,0x5F,0xC2,0xEC,0xAF
,0xF7,0x87,0x3D,0x71,0x25,0x4B,0x00,0x47,0x5C,0xE4,0x46,0xAB,0xDF,0xF1,0x6A
,0xE0,0xD9,0x9B,0x3B,0x6B,0xEC,0xDB,0x79,0xA1,0xDB,0x36,0x4A,0x4B,0x6C,0x4E
,0x73,0x23,0x44,0x99,0x92,0x53,0xF3,0x77,0x39,0x67,0x3F,0x56,0x24,0xE8,0x29
,0x4D,0xAF,0xE1,0x0E,0xF0,0xDC,0xC1,0x6A,0x37,0x55,0x61,0xDA,0xB6,0xD7,0x24
,0x9A,0x2A,0x29,0x16,0x4A,0xF9,0x41,0x24,0xFA,0xA5,0x19,0x48,0x73,0x9E,0xC9
,0xCD,0xBF,0xC4,0x35,0x75,0x6C,0x9D,0x9B,0xB6,0x76,0x5D,0xB4,0xDB,0xF6,0xD5
,0x9E,0x9E,0xDF,0x13,0x9E,0x52,0xBA,0x02,0xD2,0xCC,0xDF,0x9A,0x49,0x0E,0x59
,0xDB,0xEA,0xC4,0xEB,0x7F,0xAF,0x0A,0xDA,0xB8,0x68,0xD1,0x64,0xA8,0xE4,0x91
,0x13,0x83,0x27,0x12,0x55,0x3F,0xDA,0xC7,0x61,0xF5,0x3D,0x07,0xB9,0x1A,0x83
,0xDF,0x5F,0x8C,0xC1,0x54,0xB3,0x10,0x14,0x0C,0x92,0x4F,0x41,0xAF,0xDD,0x63
,0xD6,0xCB,0xE5,0x45,0x9E,0x50,0xC7,0xD4,0x00,0xF2,0x9F,0x48,0x27,0xA0,0xE9
,0xEE,0x73,0x8E,0x9D,0x33,0xF3,0xD1,0x1E,0x33,0x56,0x4A,0xEA,0xDF,0x09,0x10
,0x08,0xA0,0xF2,0xA8,0x9F,0xD1,0x1A,0x8F,0x72,0x07,0x76,0xF9,0xFB,0x29,0x1F
,0xCC,0x35,0xAF,0x8E,0xBA,0x39,0x97,0xE2,0x61,0x3F,0x1E,0xA9,0x87,0x35,0x93
,0x37,0x95,0x48,0x83,0xA1,0xCA,0x1F,0xE6,0x18,0x62,0x43,0x28,0x6E,0xC4,0x17
,0x1A,0xFB,0xF8,0x39,0x2B,0xA4,0x59,0xA4,0x89,0xA7,0xC1,0x0C,0x92,0x56,0xA9
,0x08,0x87,0x21,0x94,0xA4,0x03,0x1D,0x41,0xCE,0x0B,0xE1,0xC7,0xCC,0xEB,0x32
,0xA4,0x50,0x5B,0xE1,0x6B,0x85,0xCE,0xAD,0x15,0x61,0xF5,0x1A,0x9A,0xB9,0x15
,0x56,0x3E,0xE3,0x23,0x38,0x55,0xEF,0x8C,0x80,0x33,0xEF,0x9D,0x06,0x2C,0x0D
,0x75,0x95,0x8F,0x90,0xE5,0xDC,0x90,0x1E,0xA6,0xA2,0x33,0x1C,0x4B,0xEC,0x44
,0x71,0x03,0xC9,0xBA,0xAF,0x76,0x3D,0x9B,0x21,0x98,0x0C,0x6A,0x1F,0x12,0x43
,0xB5,0x3C,0x6E,0xA3,0xB6,0xC8,0xCD,0x25,0x9F,0x7C,0x52,0xC9,0xE7,0x44,0xC7
,0x8A,0x9B,0x95,0x2A,0x03,0xE6,0x10,0xB8,0x5F,0xBD,0x87,0x21,0x86,0x30,0xC6
,0x25,0xE9,0xD7,0x52,0xE5,0xBA,0xD7,0xDC,0xBA,0x59,0x6D,0xEC,0xB0,0x9F,0xFA
,0x65,0x72,0x34,0x71,0xFE,0xA9,0x1F,0x47,0x7F,0xDF,0x82,0x91,0xD9,0x8E,0xA0
,0x3E,0x36,0x59,0xDE,0x8A,0x1D,0xA1,0xBA,0xE7,0xB8,0xD5,0x54,0xD7,0x59,0xF7
,0x55,0xB6,0x42,0xE5,0xB8,0x27,0x95,0x2C,0xEB,0x04,0x91,0xAC,0x6B,0xE9,0x00
,0x89,0x7B,0x9C,0xB7,0x4C,0x16,0x3A,0x2A,0xDE,0xAE,0x82,0x94,0x85,0x82,0xB1
,0xDA,0x48,0x88,0xC2,0x51,0xC4,0xA4,0x2B,0x0E,0xD8,0x2A,0x3A,0xB0,0xEB,0x83
,0x9F,0x46,0x08,0xC8,0xF7,0xD7,0xA2,0xC1,0x55,0x53,0x12,0xA3,0x93,0x43,0x4F
,0x80,0x04,0x30,0x91,0xE6,0x63,0x1D,0x8B,0x0E,0x8B,0xF2,0xC2,0xF5,0x18,0x04
,0x36,0xB3,0x95,0x55,0x49,0x21,0x40,0x27,0xB9,0xC7,0x7D,0x7E,0xE8,0xAA,0xAB
,0xED,0x47,0x40,0x5F,0xC1,0x4B,0x9D,0x6D,0x28,0x3F,0x15,0x63,0x68,0x2E,0xD4
,0x6E,0x49,0x66,0x49,0x29,0x64,0x59,0x32,0x09,0xEA,0x4F,0x05,0x90,0x67,0xEA
,0x75,0xFC,0xC7,0xBC,0xDB,0xAB,0x2C,0xF7,0x8A,0xDB,0x45,0xC6,0x11,0x0D,0x6D
,0x0D,0x44,0x94,0xD5,0x11,0x87,0x57,0xE1,0x22,0x31,0x56,0x1C,0x94,0x90,0x70
,0x41,0xEA,0x09,0x07,0xDB,0x5F,0xD6,0x3F,0x16,0xA9,0x05,0x77,0x87,0x57,0xDA
,0x52,0x03,0x09,0xAD,0xF5,0x30,0xF1,0x23,0x21,0x8C,0x90,0x3C,0x60,0x7E,0xE5
,0xC6,0xBF,0x94,0xDB,0xBE,0x7A,0xAA,0xCD,0xD1,0x71,0xB8,0x56,0x53,0xCD,0x04
,0xB5,0xF5,0x0D,0x5A,0x16,0x54,0x2A,0xC5,0x66,0x3E,0x6A,0x36,0x0F,0xB3,0x2B
,0xAB,0x03,0xEE,0x08,0x23,0xBE,0x83,0x55,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81
,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6
,0x9A,0xCD,0xB7,0x5A,0x6E,0x57,0x08,0x64,0x9E,0x92,0x8E,0x59,0x29,0xE2,0x92
,0x38,0xA6,0xA8,0x23,0x8C,0x30,0xB4,0x84,0x84,0xF3,0x24,0x38,0x48,0xF3,0xC5
,0xB0,0x58,0x81,0xE9,0x3F,0x23,0xA0,0xC2,0xD7,0xAD,0x35,0x3C,0xF5,0x32,0xF9
,0x54,0xD0,0xC9,0x34,0x9C,0x59,0xF8,0xC6,0xA5,0x8F,0x15,0x05,0x98,0xE0,0x7B
,0x00,0x09,0x27,0xD8,0x02,0x75,0x6B,0x6C,0x6F,0x04,0x6F,0x5B,0x88,0xD3,0xA2
,0xAD,0x6C,0xB3,0xC8,0x39,0xBC,0x10,0xD3,0x14,0x0A,0x8D,0x18,0x28,0x43,0x3F
,0xAB,0x21,0xC9,0x04,0x98,0xD6,0x13,0xC7,0xA4,0xDE,0xAC,0x8E,0x9D,0xF0,0xE7
,0xEC,0xDB,0x6C,0xB7,0xA2,0x4B,0x7D,0x10,0x41,0x0F,0x20,0xFF,0x00,0x07,0x4E
,0xA1,0xC9,0x21,0x99,0x97,0x91,0x6E,0x4A,0x08,0xE6,0xC0,0x72,0xF3,0x64,0x5C
,0xFA,0x66,0x03,0x41,0xC9,0xDB,0x07,0xC2,0x8B,0xDE,0xE3,0xAC,0x8A,0x26,0xA6
,0x99,0xD6,0x60,0x7C,0xB1,0x09,0xE8,0xE3,0xD6,0xBC,0x95,0x95,0x5C,0xC8,0xA1
,0x82,0x1E,0x51,0x24,0x88,0x43,0x10,0x5E,0x3E,0xE3,0xAB,0x7C,0x3A,0xFB,0x3A
,0x52,0xD3,0x44,0x93,0x5E,0xA9,0xE8,0xE8,0x63,0xF2,0xDA,0x3F,0x86,0x86,0x04
,0x2D,0xE5,0xB3,0xF3,0x28,0xC3,0x93,0x87,0x01,0x80,0xE9,0x3C,0x95,0x23,0x01
,0x70,0x17,0x00,0x0B,0xEE,0xC3,0x62,0xB4,0x58,0xA9,0x9A,0x0B,0x4D,0x0C,0x54
,0xCA,0xE4,0x19,0x5C,0x65,0xA4,0x95,0x80,0xC7,0x27,0x73,0x96,0x76,0xC7,0xBB
,0x12,0x75,0xB1,0xD1,0x35,0xA5,0xDB,0xBB,0x56,0xC3,0x60,0x58,0xCD,0xBA,0x82
,0x35,0x96,0x34,0xF2,0xD6,0x79,0x3D,0x72,0x2A,0x7E,0x55,0x63,0xF8,0x57,0xFC
,0x0B,0x85,0x1E,0xC0,0x6B,0x6F,0x3C,0x86,0x28,0x24,0x94,0x21,0x72,0x8A,0x5B
,0x88,0xEE,0x70,0x3B,0x6B,0xEF,0x4D,0x11,0xC1,0xD7,0xCB,0x9D,0xF3,0xC4,0x7B
,0xEA,0x6E,0xDD,0xCF,0x53,0x1C,0x15,0x15,0x9E,0x45,0x45,0x91,0x38,0x16,0x86
,0x91,0x51,0xF9,0x88,0x11,0x4E,0x0B,0x13,0x85,0xE4,0xDE,0x90,0xC5,0xBA,0x13
,0x81,0x8D,0xE4,0x3B,0x8E,0xA6,0x96,0x95,0x64,0xDC,0x36,0x3A,0xEB,0x6B,0x01
,0xEA,0x92,0x25,0xF8,0xA8,0x7F,0xAC,0x79,0x60,0x3E,0xAC,0xA0,0x7D,0x4E,0xAD
,0x6D,0xE9,0xE0,0xC5,0xE6,0xD1,0x72,0xAD,0xB8,0xEC,0x58,0xA8,0xEE,0x36,0xEA
,0xB9,0x1A,0x57,0xB4,0x54,0xCD,0xE4,0xC9,0x01,0x63,0x96,0x58,0x24,0x20,0xA9
,0x42,0x49,0x22,0x36,0xE2,0x14,0x93,0x86,0xC1,0xC0,0x83,0x5D,0x96,0xF1,0x67
,0x59,0x05,0xDF,0x6A,0x6E,0x3A,0x13,0x1A,0x72,0x25,0xAD,0xD2,0x49,0x19,0x20
,0x67,0x88,0x96,0x30,0xD1,0xE7,0xDB,0xF1,0x7E,0x99,0xD5,0x56,0x99,0x77,0x7E
,0xD3,0x9E,0x3E,0x9B,0x96,0xD4,0x80,0xF7,0xCD,0x62,0x21,0xFF,0x00,0x32,0x08
,0xD6,0x9E,0xB0,0x78,0x57,0x52,0x59,0xEA,0x24,0xDA,0x6E,0xCC,0x72,0xCC,0x24
,0x80,0x31,0x3F,0x52,0x0E,0x4E,0xB7,0xB6,0x9B,0x8D,0x96,0x5A,0x00,0xB4,0xF5
,0x74,0xD5,0x01,0x59,0xBC,0xC6,0x41,0x90,0x64,0x24,0x97,0x24,0x7B,0x12,0xC4
,0x92,0x35,0x9B,0x4D,0x51,0x4D,0x33,0x71,0xA5,0xA5,0xA9,0x94,0xFF,0x00,0xD4
,0xD1,0x48,0xDF,0xF0,0x5D,0x04,0x4E,0xD1,0x55,0xE1,0xE5,0x15,0x57,0x99,0x62
,0xB7,0xD3,0x54,0xD4,0xC7,0xDA,0x4B,0x7D,0xBD,0xEA,0x19,0x7F,0xDF,0x45,0x38
,0xFE,0xA3,0x5B,0xB7,0xBA,0xDF,0x2A,0x63,0x73,0x6E,0xDB,0xB2,0x46,0x40,0xCA
,0xB5,0xC2,0xA5,0x21,0x56,0xFD,0x02,0x73,0x6F,0xD9,0x82,0xEB,0x6D,0x47,0x4B
,0x7D,0x96,0x6A,0x91,0x47,0xB4,0xF7,0x45,0x61,0x69,0x01,0x45,0x8E,0xCD,0x51
,0x1A,0xFE,0x15,0x18,0xE7,0x22,0xAA,0x77,0x07,0xAF,0x2C,0x6A,0x4F,0x41,0xE1
,0xA7,0x89,0xB7,0x54,0x46,0x8E,0xD7,0x68,0xB1,0x46,0xE0,0x12,0xD7,0x3A,0xC3
,0x2C,0xA9,0xFF,0x00,0x75,0x00,0x65,0x27,0xFE,0xF0,0x68,0x20,0x8B,0x68,0x86
,0xE3,0x1F,0x9B,0x79,0xAB,0x6B,0x88,0xEC,0xD0,0x11,0xE5,0xD3,0x29,0xF7,0x06
,0x2C,0xFA,0x87,0xD2,0x42,0xD8,0xFA,0x6B,0x61,0xB7,0x7E,0x37,0x71,0x54,0x9B
,0x66,0xC9,0xB3,0x4D,0x7B,0x78,0x4F,0x96,0xF2,0x53,0xE2,0x3A,0x3A,0x7F,0xA3
,0xCE,0x7D,0x03,0x1F,0x95,0x79,0x37,0xF8,0x75,0x6D,0x6D,0xEF,0x00,0x6C,0x7E
,0x62,0xD4,0xEF,0x4B,0xCD,0x76,0xE8,0x90,0x1C,0xFC,0x21,0x1F,0x0B,0x42,0x0F
,0xB6,0x61,0x43,0x97,0xFF,0x00,0xBC,0x76,0x1F,0x4D,0x5B,0x96,0xEA,0x1A,0x2B
,0x6D,0x0C,0x54,0x36,0xEA,0x4A,0x7A,0x3A,0x48,0x57,0x8C,0x50,0x41,0x18,0x48
,0xD1,0x7E,0x41,0x47,0x40,0x3F,0x4D,0x34,0x54,0x1B,0x5B,0xC0,0xF4,0x99,0x56
,0xA7,0x7E,0x5D,0xDA,0xE4,0xC7,0xA9,0xB6,0x5B,0xD9,0xA0,0xA4,0x5F,0xA3,0xBF
,0x49,0x25,0xFD,0xCA,0xA9,0xF7,0x4D,0x5B,0x76,0x5B,0x55,0xB2,0xCB,0x6E,0x8A
,0xDB,0x67,0xB7,0x52,0x5B,0xE8,0xA2,0x18,0x8E,0x9E,0x9A,0x15,0x8E,0x35,0xFD
,0x15,0x40,0x1A,0xCC,0xD7,0x9D,0x44,0xF1,0x40,0x9C,0xA5,0x7C,0x67,0xB0,0x03
,0x25,0x8E,0x33,0x80,0x07,0x52,0x7E,0x83,0x51,0x1E,0x9A,0xF2,0xA8,0xA8,0x86
,0x9C,0x2F,0x9B,0x20,0x52,0xC7,0x08,0xA3,0xAB,0x31,0xC6,0x70,0x00,0xEA,0x4E
,0x01,0xE8,0x35,0x85,0x59,0x3D,0x5B,0xC3,0x24,0x86,0x58,0xED,0x74,0x8A,0x0F
,0x3A,0x89,0x8A,0x99,0x00,0xEA,0x32,0x01,0xF4,0xA7,0xF2,0x90,0x5B,0x97,0xB8
,0x2A,0x35,0x8D,0xE6,0xAD,0x34,0x13,0xD5,0x52,0xC6,0x94,0xD1,0xAA,0x96,0x9E
,0xE3,0x71,0x24,0x00,0xA3,0x91,0xCE,0x18,0x86,0x2A,0x0E,0x7A,0x12,0x8A,0x03
,0x65,0x4E,0x3A,0x68,0x36,0xB3,0x54,0xC5,0x04,0x0D,0x3D,0x4B,0xA5,0x3C,0x4B
,0xDD,0xA5,0x70,0xA0,0x75,0xC0,0xC9,0xED,0xF2,0xD7,0xEC,0x12,0xC7,0x53,0x4E
,0x24,0x54,0x7E,0x0E,0x3F,0x0C,0x91,0xB2,0x1C,0x7D,0x55,0x80,0x23,0xF7,0x1A
,0x8D,0xC7,0x2D,0x6D,0x6D,0x4F,0x99,0x67,0xA3,0x7A,0x99,0x47,0x20,0x2E,0xB7
,0x45,0x2B,0x14,0x5F,0x88,0x66,0x28,0x87,0x16,0x7F,0x6F,0xC2,0x23,0x57,0x52
,0x0F,0x98,0xC4,0x6A,0x41,0x45,0x0C,0x94,0x94,0xA4,0x55,0x56,0xC9,0x54,0xF9
,0x67,0x79,0xA5,0x0A,0xBD,0xFA,0xE0,0x05,0x00,0x05,0x03,0xA0,0xFA,0x0E,0xA4
,0x9C,0x92,0x1F,0xB6,0xF6,0x0D,0x4A,0x02,0xE7,0x08,0xEF,0x18,0xC9,0x24,0xE1
,0x58,0xAF,0x73,0xDF,0xB6,0xBE,0x6E,0x15,0x16,0xFB,0x75,0x3C,0x97,0x1B,0x85
,0x45,0x3D,0x2C,0x31,0x2E,0x5E,0xA2,0x77,0x0A,0xA8,0x0E,0x3F,0x98,0xF4,0x03
,0xB6,0xBC,0xAC,0x42,0x43,0x66,0x82,0x56,0x8E,0x48,0x25,0x9C,0x1A,0x86,0x8E
,0x61,0xEA,0x89,0xA4,0x25,0xCA,0x9C,0x7E,0x52,0xD8,0xFD,0xB5,0xF0,0x96,0x6A
,0x46,0xAE,0x4A,0xFA,0xD0,0xD5,0xB5,0x71,0x9E,0x51,0x49,0x3F,0x51,0x09,0xFF
,0x00,0xAB,0x5F,0xC2,0x87,0xDB,0x20,0x72,0x23,0xB9,0x3A,0x0D,0x68,0xBB,0xDE
,0xAF,0x3E,0x9D,0xBF,0x6F,0xF8,0x4A,0x53,0xFF,0x00,0xD4,0x6E,0x71,0x32,0x02
,0x3E,0x71,0xC1,0xD1,0xDF,0xF5,0x73,0x18,0xEC,0x41,0x61,0xAC,0xAB,0x76,0xDB
,0xA2,0x82,0xAA,0x3A,0xFA,0xE9,0xAA,0x2E,0xD7,0x04,0x39,0x4A,0xAA,0xD6,0x0E
,0x63,0x3F,0xF5,0x68,0x00,0x48,0xFA,0x74,0xF4,0x28,0x27,0xDC,0x9D,0x6E,0xB4
,0xD0,0x35,0x5C,0xF8,0xBA,0x7F,0x8D,0x6E,0x9D,0x89,0xB2,0x60,0x1C,0xE6,0xAE
,0xBE,0x45,0x73,0xAA,0x5F,0x64,0xA4,0xA2,0xFB,0xE7,0x63,0xF2,0xCC,0x82,0x14
,0x1F,0x32,0xDA,0x9C,0xDF,0x6E,0xB6,0xFB,0x1D,0x9A,0xB2,0xF1,0x75,0xAA,0x8E
,0x96,0x86,0x8A,0x16,0x9E,0xA2,0x67,0x3D,0x11,0x14,0x64,0x9F,0xFF,0x00,0x5E
,0xFA,0x88,0x78,0x21,0x6A,0xB9,0x5E,0x6E,0x37,0x2F,0x15,0x37,0x1D,0x24,0x94
,0x95,0xF7,0xD8,0x96,0x0B,0x4D,0x14,0xC3,0x0F,0x41,0x6C,0x53,0xCA,0x34,0x61
,0xEC,0xF2,0x13,0xE6,0xB8,0xFA,0xA8,0xF6,0xD1,0x61,0x6A,0x69,0xA6,0x9A,0x2B
,0x02,0xFF,0x00,0x0A,0xCF,0x6D,0x28,0xF8,0xE0,0xB2,0xC5,0x23,0xE7,0xB7,0x15
,0x91,0x58,0xFF,0x00,0x90,0x3A,0xFE,0x4D,0xF8,0x93,0x04,0x90,0xEE,0x45,0xFB
,0xB9,0x84,0x1F,0x05,0x4D,0x14,0x12,0x48,0xA4,0x09,0x56,0x18,0x56,0x06,0x2A
,0x4F,0xE2,0x01,0xE2,0x74,0xC8,0xE9,0x94,0x23,0xD8,0xEB,0xFA,0xBB,0xBC,0xFC
,0xDF,0xEC,0x7D,0xEB,0xC8,0x24,0x4B,0xFC,0x3E,0x7E,0x18,0xEF,0xCB,0xCB,0x6C
,0x7F,0x9E,0xB8,0x3F,0xED,0x6D,0xE1,0x45,0x16,0xC6,0xBE,0x05,0xB6,0x45,0x51
,0x73,0x86,0xED,0x1D,0x65,0x75,0x23,0xC4,0x84,0xCB,0x4B,0x32,0x4F,0xE7,0x4D
,0x1B,0x63,0x39,0x85,0x69,0xE4,0x0D,0x83,0x8C,0x18,0xF9,0x82,0xA0,0xB8,0x70
,0xE6,0xDD,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34
,0x0D,0x34,0xD4,0xBB,0x62,0x78,0x7D,0xB8,0x77,0x6D,0xD2,0x2A,0x2A,0x1A,0x1A
,0xB6,0x2F,0xC5,0x8C,0x70,0x40,0x65,0x9C,0xA1,0xE0,0x79,0x08,0xC1,0x18,0x05
,0x1F,0x92,0xBC,0x8D,0x1C,0x6D,0xC4,0x8E,0x60,0xE8,0x22,0x3A,0x92,0xED,0x1D
,0x8F,0xB8,0x77,0x3C,0xF0,0xC7,0x6F,0xA4,0x75,0x59,0xD4,0xB4,0x2C,0xF1,0xBB
,0x34,0xC3,0xD6,0x33,0x1C,0x71,0xAB,0x49,0x20,0xE4,0x85,0x0B,0x22,0xB2,0xAB
,0x11,0xCC,0xA8,0xEB,0xAE,0xAD,0xF0,0xCF,0xEC,0x9D,0x44,0x20,0xA5,0xA8,0xDE
,0x22,0x3A,0x5E,0x1E,0xA7,0x82,0x29,0xBE,0x22,0xA6,0x52,0xD1,0x85,0x65,0x77
,0x23,0xC9,0x8D,0x33,0xC9,0x82,0xA2,0x3B,0xA9,0x6F,0xEF,0x8E,0x01,0x1D,0x2D
,0xB5,0x76,0xBD,0x83,0x6B,0xD2,0x3D,0x35,0x8A,0xD7,0x05,0x1A,0xCA,0xDC,0xE6
,0x90,0x65,0xA5,0x9D,0xBF,0x34,0x92,0x36,0x5E,0x46,0xFF,0x00,0x13,0x12,0x74
,0x4D,0x71,0xBF,0x87,0x3F,0x65,0x1D,0xCB,0x71,0xA1,0x13,0x6E,0x4A,0x78,0x2D
,0x9E,0x7C,0x78,0x26,0xAA,0x52,0xF3,0x45,0x90,0x87,0x29,0x14,0x4E,0x15,0x58
,0x10,0xE3,0x9B,0xC8,0xD9,0x56,0xEB,0x08,0x23,0x5D,0x11,0xB2,0x3C,0x09,0xDA
,0x1B,0x78,0xD3,0x4D,0x50,0x92,0xD7,0xD4,0xD3,0xF2,0x30,0xBC,0x84,0x0F,0x27
,0x93,0x17,0x22,0x3E,0x20,0x08,0x47,0x22,0x4F,0xDC,0xAC,0x40,0xE4,0xE4,0x1C
,0xEA,0xD8,0xD3,0x41,0x8F,0x6E,0xA0,0xA2,0xB7,0x53,0x0A,0x6B,0x7D,0x24,0x14
,0xB0,0x82,0x5B,0x84,0x31,0x84,0x5C,0x9E,0xE7,0x03,0xDC,0xFC,0xF5,0x91,0xA6
,0xBC,0xEA,0x6A,0x20,0xA6,0x88,0xCB,0x51,0x2A,0x44,0x99,0x03,0x93,0x9C,0x0C
,0x93,0x80,0x3F,0x52,0x7A,0x63,0x44,0x7A,0x69,0xAC,0x44,0xB8,0x53,0x3C,0x51
,0x38,0xF3,0x43,0xCC,0xA1,0xD2,0x16,0x89,0x84,0xA5,0x49,0xC0,0x25,0x08,0xE4
,0xA3,0xA8,0xCE,0x40,0xC6,0x7A,0xE3,0x58,0x93,0xDC,0x1D,0x99,0x97,0xAA,0x70
,0xEB,0x24,0x71,0x90,0x5A,0x31,0xE9,0x27,0xCC,0x73,0xE8,0x8F,0x00,0x9C,0x8C
,0x92,0x47,0x55,0x3A,0x0D,0xB6,0xB1,0xE5,0xAC,0x86,0x36,0x64,0x0C,0x64,0x74
,0xFC,0x41,0x3A,0xF1,0xE9,0x9F,0x51,0xEC,0xBD,0x3A,0xF5,0x23,0x58,0xD4,0xA9
,0x2C,0xD4,0x2B,0x2C,0x52,0xC6,0xC1,0x86,0x51,0x55,0x9B,0xCB,0x3D,0xBA,0x96
,0xFC,0x4F,0xEF,0xD7,0xA0,0x60,0x7B,0x7B,0xEB,0x14,0xED,0xBA,0x4A,0xA6,0x56
,0xBC,0xB9,0xB9,0x05,0xFC,0x14,0xEE,0xBC,0x69,0x63,0xFD,0x21,0x1E,0x93,0x83
,0xD4,0x17,0xE4,0x47,0xB1,0x1A,0x0F,0x1F,0xED,0x3A,0xD7,0x66,0x3D,0xBB,0x43
,0x25,0xE1,0xBB,0x7C,0x42,0x3F,0x97,0x46,0xBF,0xAC,0xE4,0x61,0x87,0xB1,0xF2
,0xC4,0x84,0x1E,0xE0,0x6B,0xC1,0x6D,0x97,0x8B,0x9C,0xA4,0xDE,0x2A,0xBC,0xE8
,0xBF,0xFE,0x3C,0x6A,0x61,0xA5,0x1D,0xFB,0xAE,0x7C,0xC9,0x88,0x3D,0x08,0x72
,0xA8,0xC0,0xE4,0x2E,0xA5,0x00,0x00,0x00,0x00,0x00,0x3B,0x01,0xA6,0x83,0x1E
,0x96,0x92,0x28,0x23,0x8D,0x10,0x61,0x63,0x1C,0x51,0x10,0x71,0x45,0x1F,0x20
,0xA3,0xA6,0x07,0xB6,0xB2,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x35,0xE7,0x51
,0x3C,0x34,0xF1,0xF9,0x93,0xC8,0xB1,0xAE,0x40,0x05,0x8F,0x72,0x7A,0x00,0x3E
,0x64,0x9E,0xC3,0x5A,0xFA,0x9A,0xAA,0xB9,0xE9,0xDE,0x55,0x75,0xB5,0xD1,0xAA
,0x96,0x7A,0x9A,0x80,0x04,0x9C,0x71,0xDC,0x2B,0x74,0x4E,0xF9,0xCB,0xF6,0x20
,0x82,0x9E,0xFA,0x0D,0x85,0x45,0x44,0x14,0xE8,0x1E,0x79,0x52,0x35,0x66,0x0A
,0xBC,0x8E,0x39,0x31,0xEC,0x07,0xCC,0x9F,0x96,0xB0,0xC4,0xB5,0x33,0x92,0xF4
,0x74,0xCB,0x4E,0xAC,0x06,0x67,0xA9,0x53,0x92,0x3A,0x11,0x84,0xE8,0xDE,0xE4
,0x7A,0x8A,0x91,0xF2,0x3A,0xF8,0xA2,0x8E,0x33,0x29,0x96,0x92,0x9D,0xD9,0xC8
,0x2A,0x6A,0xEA,0x72,0x58,0x8C,0x9C,0x85,0xCF,0xA8,0xAE,0x40,0x38,0x1C,0x57
,0x07,0x2B,0xAD,0x80,0x0A,0x8A,0x5D,0xDF,0x38,0xEA,0x59,0x8F,0x61,0xFF,0x00
,0x01,0xA0,0xC5,0x8A,0x82,0x35,0x91,0x66,0x91,0x9E,0xA6,0xA1,0x7A,0xAC,0xD5
,0x1E,0xAE,0x07,0x04,0x65,0x54,0x60,0x2F,0x43,0x83,0xC4,0x0C,0x8E,0xF9,0xD7
,0xDB,0x50,0xD3,0x3C,0xCB,0x51,0x52,0xBE,0x7C,0x91,0xB7,0x24,0x69,0x7A,0x88
,0xC8,0xCF,0x55,0x1D,0x94,0xE0,0x91,0x90,0x33,0x8E,0xE4,0xEB,0xD1,0x67,0x32
,0xFF,0x00,0xCD,0xD7,0x92,0xFF,0x00,0xAC,0x6E,0x8B,0xFB,0x7C,0xFF,0x00,0x6E
,0x87,0xE7,0xAF,0xB5,0x8F,0xB1,0x73,0xCD,0xBB,0xF5,0xEC,0x3F,0x41,0xA0,0xF9
,0x32,0x3B,0x74,0x85,0x39,0x0F,0xCC,0xC7,0x03,0xF6,0xF9,0xFF,0x00,0xC3,0xEB
,0xAF,0xA9,0xE2,0x8E,0x78,0x5E,0x19,0x90,0x3C,0x72,0x29,0x57,0x52,0x32,0x18
,0x1E,0x84,0x1D,0x7D,0xE9,0xA0,0x69,0xA6,0xBC,0x6A,0x2A,0x62,0x85,0x84,0x67
,0x93,0xCA,0x46,0x44,0x68,0x32,0xC7,0xEB,0x8F,0x61,0x9E,0x99,0x38,0x1F,0x5D
,0x07,0xB6,0xBE,0x65,0x74,0x8A,0x36,0x92,0x47,0x54,0x44,0x05,0x99,0x98,0xE0
,0x28,0x1D,0xC9,0x3A,0x8F,0xCD,0xBA,0xA9,0xD6,0x9A,0x69,0xA1,0x8A,0x29,0x96
,0x22,0x79,0xCD,0xF1,0x0A,0xB4,0xD0,0x8E,0xDF,0x7B,0x39,0xF4,0x02,0x0F,0x42
,0xA8,0x5C,0x8F,0x91,0xD5,0x4F,0x4D,0xBA,0xAE,0x5E,0x2E,0xF8,0xC5,0x49,0xB3
,0xF6,0xE5,0xE2,0x1A,0xBD,0x9D,0x66,0x41,0x57,0xB9,0xEA,0x28,0x94,0x1A,0x5A
,0xB2,0x7F,0xBB,0xA4,0x59,0x0E,0x4C,0x8A,0xC4,0x10,0xC4,0x10,0xAE,0xBC,0x86
,0x3A,0x1C,0x95,0x3A,0xB5,0xD2,0x0F,0x16,0xE5,0xA4,0xB9,0x56,0xD1,0xCF,0x0E
,0xC7,0xA4,0xA8,0x13,0xD2,0xC1,0x50,0xBC,0x4D,0xEE,0x44,0x39,0x8E,0x56,0x5F
,0x6A,0x65,0x23,0x92,0xAB,0x75,0x90,0xF1,0x62,0x02,0x80,0x1A,0xD8,0xD7,0xE2
,0xAA,0xA2,0x85,0x55,0x0A,0xA0,0x60,0x00,0x30,0x00,0xD7,0xEE,0x8A,0x69,0xA6
,0x9A,0x0F,0x2A,0xC8,0x84,0xD4,0x73,0x42,0x46,0x44,0x91,0xB2,0x9F,0xDC,0x63
,0x5C,0xB5,0xF6,0xC9,0xA9,0x32,0x6D,0x8B,0x55,0xC4,0x60,0x03,0x4F,0x71,0x99
,0x5F,0xDF,0x8C,0xB6,0xF8,0xA1,0xC7,0xE8,0x7C,0xE3,0xFD,0x35,0xD5,0x5A,0xE4
,0x9F,0xB6,0x2B,0x2A,0x78,0x4D,0x67,0x03,0xF9,0x6C,0x88,0x7E,0x7F,0x8D,0xA9
,0x97,0xFF,0x00,0x6E,0x83,0x87,0x74,0xD3,0x4D,0x03,0x4D,0x34,0xD0,0x34,0xD6
,0x55,0xB6,0xDF,0x59,0x71,0xA8,0xF2,0x68,0xE1,0xE6,0x46,0x0B,0xB1,0x60,0xA9
,0x12,0x96,0x0B,0xC9,0xDD,0x88,0x54,0x5C,0xB2,0x82,0xCC,0x40,0x19,0x19,0x23
,0x57,0x27,0x87,0x9F,0x67,0x5D,0xD5,0xBA,0xA1,0xA5,0x9F,0x84,0xD1,0x45,0x30
,0xE4,0xF2,0x4B,0x13,0x41,0x0A,0xA3,0x20,0xE2,0x55,0xDC,0x19,0x24,0x60,0xE4
,0x83,0xC6,0x23,0x11,0x0B,0xD2,0x5F,0x56,0x40,0x52,0xB4,0xD0,0x4D,0x53,0x51
,0x15,0x35,0x34,0x32,0x4D,0x3C,0xAE,0x12,0x38,0xE3,0x52,0xCC,0xEC,0x4E,0x02
,0x80,0x3A,0x92,0x4F,0x4C,0x6A,0xC0,0xD8,0x9E,0x12,0x6E,0x2D,0xD9,0x01,0x7A
,0x44,0x94,0xBC,0x8A,0x44,0x0B,0x4F,0x10,0x95,0x79,0xF1,0x56,0x02,0x49,0x0B
,0x2C,0x6B,0xE9,0x2F,0x94,0x56,0x79,0x95,0x90,0x83,0x17,0x5C,0xEB,0xB3,0x7C
,0x3E,0xFB,0x3C,0x6D,0x9B,0x1D,0x0F,0x93,0x75,0xE3,0x59,0x13,0x90,0xEF,0x46
,0x91,0x84,0xA6,0x62,0x18,0xB2,0xF3,0x8B,0x2D,0xE6,0xF1,0x2E,0xE0,0x79,0xED
,0x39,0x01,0xB0,0xA5,0x47,0x4D,0x5C,0x96,0xCB,0x65,0x05,0xB2,0x05,0x86,0x82
,0x92,0x28,0x11,0x50,0x20,0xE2,0x3A,0xF1,0x1D,0x97,0x3D,0xF0,0x33,0xD0,0x7B
,0x7B,0x68,0x9A,0xE6,0xEF,0x0A,0x3E,0xCC,0xB1,0x59,0xCA,0x55,0xDD,0xAA,0x7E
,0x1D,0xFA,0xE4,0x53,0xBB,0x2C,0x8C,0x0F,0x21,0x83,0x30,0xE3,0x29,0x05,0x24
,0x74,0x6F,0x2C,0xD3,0xE4,0x01,0xCD,0x1B,0x5D,0x13,0xB6,0xB6,0xFD,0x8F,0x6D
,0x5A,0xD2,0xD9,0xB7,0xED,0x34,0x76,0xCA,0x34,0xEA,0x21,0xA6,0x88,0x22,0x93
,0xF3,0x38,0xEE,0x7E,0xA7,0xA9,0xD6,0xCF,0x5F,0x8C,0x42,0xA9,0x66,0x20,0x00
,0x32,0x49,0xEC,0x34,0x1F,0xBA,0x6B,0x0A,0x6B,0x8C,0x2B,0x18,0x91,0x0A,0x98
,0xD8,0x80,0x25,0x76,0xE2,0x8C,0x4E,0x30,0x14,0xF7,0x62,0x73,0xD3,0x88,0x20
,0xF6,0xCE,0xB1,0x9E,0x7A,0xA9,0x42,0x30,0xA7,0x94,0xAB,0x90,0x14,0xC9,0x11
,0xC1,0xE8,0x0F,0x48,0xFD,0x87,0x43,0xD6,0x42,0xA4,0x1F,0x6E,0xBA,0x23,0x69
,0x23,0xA4,0x68,0x64,0x91,0xD5,0x10,0x75,0x2C,0xC7,0x00,0x6B,0x0A,0xB6,0xE1
,0xE5,0x28,0x11,0xA8,0x52,0xD9,0x0A,0xD2,0x82,0x0B,0x63,0x39,0xE2,0x83,0xD4
,0xE4,0x63,0x38,0x00,0x64,0x76,0x3A,0xF7,0x14,0x80,0x61,0x8C,0xB2,0x3C,0xA3
,0xFD,0x2B,0xE0,0xB0,0xFD,0x06,0x38,0xAF,0xEC,0x3A,0xE9,0x15,0x15,0x3C,0x65
,0xD9,0x14,0x86,0x7C,0x73,0x7E,0x47,0x9B,0x63,0xB6,0x5B,0xF1,0x1C,0x64,0xE3
,0xAE,0x83,0x01,0xEE,0x15,0x6D,0x88,0x12,0x31,0xE7,0x28,0x1C,0xC2,0x27,0x29
,0x18,0xF7,0xE8,0x9C,0xB1,0x18,0x60,0x0F,0x12,0xED,0xDF,0xA1,0x1A,0xFD,0x86
,0xDD,0x55,0x2C,0xA2,0x5A,0x99,0xBC,0xA2,0x06,0x32,0xA7,0x9C,0xAC,0x3D,0xFD
,0x64,0x00,0x81,0x80,0x19,0x54,0x51,0x82,0x32,0x1B,0x5B,0x48,0xA3,0x8E,0x24
,0x11,0xC5,0x1A,0xC6,0x83,0xB2,0xA8,0xC0,0x1F,0xB6,0xBE,0xB4,0x1E,0x10,0x52
,0x53,0xC1,0x11,0x8A,0x28,0xF8,0x2B,0x1E,0x4C,0x41,0x3C,0x9C,0xF4,0xEA,0xC7
,0xBB,0x1E,0x83,0x24,0xE7,0x3E,0xFA,0xF5,0x8A,0x38,0xE2,0x89,0x62,0x8A,0x34
,0x48,0xD4,0x61,0x55,0x46,0x00,0x1F,0x41,0xAF,0xAD,0x34,0x0D,0x34,0xD3,0x40
,0xD3,0x4D,0x62,0x8A,0xCF,0x38,0xA0,0xA3,0x88,0xD4,0x2B,0x11,0x99,0x41,0xC4
,0x60,0x7C,0xF9,0x7F,0x37,0x4C,0xFE,0x10,0x7A,0x8C,0x1C,0x68,0x32,0xB4,0xD3
,0x58,0xF5,0x49,0x51,0x27,0xA6,0x3A,0x81,0x4F,0x1F,0xBB,0xAA,0x86,0x7F,0xDB
,0x3D,0x07,0xEE,0x0E,0x7E,0x9A,0x0F,0x49,0xE7,0x86,0x00,0xA6,0x59,0x15,0x79
,0x1C,0x28,0xF7,0x63,0x8C,0xE0,0x0E,0xE4,0xFD,0x06,0xBC,0x2A,0x2A,0x59,0x20
,0x79,0xE4,0x68,0xE8,0xE9,0xD0,0x72,0x79,0xA7,0x20,0x61,0x47,0x73,0x8C,0xF4
,0x18,0xCF,0x52,0x7A,0x7B,0x8D,0x7E,0x43,0x4A,0x90,0x17,0x34,0x90,0xAA,0xC8
,0xE3,0x0D,0x3C,0xC4,0xBB,0x1E,0xF8,0xC9,0x27,0x93,0x00,0x4F,0x62,0x40,0x00
,0xF4,0xD7,0xEF,0xC1,0x53,0xAC,0xA2,0xAA,0xA5,0x8C,0xF2,0xC6,0x79,0x2C,0x93
,0x10,0x44,0x7D,0xFA,0xA8,0xEC,0xA7,0x04,0x8C,0x81,0x92,0x3B,0x93,0xA0,0xD7
,0xA3,0xD4,0xCF,0x21,0x6B,0x65,0x29,0x69,0x30,0x54,0xD7,0xD7,0xA9,0x00,0x77
,0x07,0x82,0x74,0x66,0x19,0x55,0x38,0x1C,0x10,0x83,0x90,0xDA,0xCB,0x86,0xDB
,0x0C,0x72,0x0A,0xBA,0xE9,0xDE,0xB2,0x74,0x3C,0xD6,0x49,0xC8,0xE3,0x17,0x7F
,0xC0,0xA3,0xD2,0xB8,0x0C,0x47,0x2C,0x72,0xC7,0x42,0xC7,0x5E,0xC9,0x54,0x6A
,0x30,0x68,0xD3,0xCC,0x43,0xFE,0x99,0xBA,0x26,0x3A,0x75,0x1E,0xED,0xD0,0xE4
,0x63,0xA1,0xF9,0xEB,0xD1,0x69,0xC1,0x60,0xF3,0x31,0x99,0xC1,0xC8,0xCF,0xE1
,0x53,0xF4,0x1F,0xFA,0xF5,0x3F,0x5D,0x07,0xC9,0x9E,0x49,0x7A,0x52,0xC7,0xC8
,0x7F,0xAC,0x7E,0x89,0xFB,0x7B,0xB7,0xCF,0xA7,0x43,0xF3,0xD1,0x29,0x14,0xB8
,0x92,0xA5,0xCC,0xF2,0x03,0x91,0xCB,0xA2,0x29,0xFA,0x2F,0x6E,0x87,0xB1,0x39
,0x3F,0x5D,0x64,0x6B,0xC2,0xBA,0xB2,0x92,0x86,0x9C,0xD4,0x56,0x54,0xC5,0x4F
,0x10,0x20,0x73,0x91,0xC2,0x8C,0x9E,0xC3,0xAF,0xB9,0xF9,0x68,0x3D,0xF4,0xD4
,0x6E,0xE9,0xBA,0xE9,0xA9,0x64,0x8E,0x10,0x92,0x47,0x2C,0xC3,0x94,0x31,0x3C
,0x2E,0xF5,0x13,0x0F,0x9A,0x53,0xAF,0xDE,0x10,0x0F,0x42,0x58,0x20,0x1D,0xF3
,0x8D,0x6B,0x9E,0xE1,0x7E,0xAD,0x32,0x16,0x63,0x68,0x82,0x3C,0x99,0x70,0xD1
,0xCD,0x56,0x80,0x63,0x25,0xD8,0xFF,0x00,0xC9,0xE9,0xF0,0x08,0x24,0x7D,0xE1
,0x2B,0xD4,0x60,0xE8,0xA9,0x45,0xC6,0xE5,0x05,0x0C,0xB1,0xAD,0x47,0xA1,0x64
,0xCF,0x16,0x24,0x12,0xC7,0xF2,0xAA,0x8C,0xB3,0xB7,0xD0,0x0F,0xFF,0x00,0x1A
,0xD7,0xCB,0x78,0xAA,0x96,0x57,0x58,0x69,0xC5,0x3A,0x46,0x39,0x37,0x99,0x86
,0x98,0x8C,0x13,0xD5,0x01,0x0B,0x18,0x23,0xA8,0x69,0x18,0x76,0xC1,0x5D,0x47
,0xAD,0x6B,0x51,0x56,0x59,0xAC,0x54,0x6F,0x52,0x65,0x00,0x49,0x5C,0xF3,0xC8
,0x90,0xB0,0xE8,0x7A,0xD4,0xB7,0xDF,0x54,0x00,0x7A,0xAF,0x96,0x16,0x32,0x09
,0x52,0x46,0xB7,0x54,0x9B,0x4A,0x96,0x44,0x43,0x7C,0x98,0x5D,0x78,0x1E,0x49
,0x4B,0xE5,0x08,0xA8,0xA3,0x39,0xCF,0xA6,0x01,0x95,0x3D,0x7A,0x83,0x21,0x76
,0x07,0xB1,0x1A,0x0C,0x2A,0xDD,0xD8,0x27,0xA6,0x79,0xAD,0xC6,0x2A,0x8A,0x54
,0xE8,0xD5,0x86,0x71,0x05,0x18,0xEA,0x40,0xCD,0x4B,0x0F,0x56,0x7B,0x62,0x15
,0x62,0x18,0x63,0x23,0x58,0x30,0xD0,0xDE,0x6F,0xAA,0x43,0x45,0xE6,0xD3,0x31
,0xCB,0x49,0x5B,0x03,0xD3,0xD1,0x9F,0x99,0x14,0xB9,0xF3,0x67,0x3D,0x3A,0xF9
,0xCC,0xAA,0x7A,0x32,0xFC,0xB5,0x38,0xA9,0x4A,0x38,0xF1,0x59,0x52,0x90,0x27
,0xC3,0xA9,0x61,0x34,0x80,0x0F,0x29,0x71,0xD4,0xF2,0x3F,0x84,0x63,0x39,0xD7
,0x25,0x78,0xE1,0xE3,0x75,0xDF,0xC4,0x5D,0xCF,0x17,0x86,0x1E,0x13,0xD2,0xD7
,0xDC,0x68,0xEA,0x65,0x30,0xD7,0x55,0xD0,0xFA,0x65,0xAE,0x51,0xD5,0xE3,0x89
,0xFF,0x00,0x92,0x2C,0x67,0x94,0x87,0xB8,0xC9,0xFC,0x23,0xD4,0x12,0xFD,0xDF
,0x51,0x2F,0x88,0x37,0xD8,0xF6,0x86,0xCE,0x77,0xBC,0xC6,0x8E,0xD0,0xCD,0x73
,0x9D,0xF1,0x1C,0xA5,0x3D,0x32,0x2C,0x01,0x00,0x58,0x29,0xA3,0xCF,0x19,0x26
,0x8C,0x07,0x90,0xE2,0x24,0x62,0x4B,0x30,0xBE,0x7C,0x2A,0xD8,0x56,0x6F,0x0E
,0xF6,0xA4,0x76,0x3B,0x42,0x07,0x76,0x73,0x35,0x65,0x51,0x8D,0x51,0xAA,0x66
,0x3F,0x89,0xC8,0x1D,0x14,0x74,0x01,0x54,0x74,0x55,0x00,0x0E,0xDA,0xC7,0xF0
,0x7B,0x61,0xD1,0xEC,0x2D,0xA7,0x4F,0x6F,0x55,0x81,0xEE,0x32,0x44,0x9F,0x1B
,0x3C,0x49,0xC5,0x0B,0x28,0xC2,0xC7,0x18,0x3D,0x56,0x24,0xC9,0x0A,0xBF,0xAB
,0x1C,0xB3,0x33,0x19,0xA6,0x8A,0x69,0xAC,0x2A,0xDB,0xA5,0x15,0x25,0x42,0xD2
,0xC9,0x37,0x3A,0xA7,0x5E,0x49,0x4F,0x10,0xE7,0x2B,0x0E,0xD9,0xE2,0x3A,0x85
,0xCF,0x4E,0x47,0x00,0x7B,0x91,0xAC,0x8A,0x77,0x9A,0x45,0xE7,0x2C,0x5E,0x4E
,0x7B,0x21,0x60,0x58,0x7E,0xB8,0xE9,0xFD,0x33,0xFA,0xE8,0x3D,0x74,0xD3,0x4D
,0x03,0x5C,0x7B,0xF6,0xD7,0x91,0xA9,0x7C,0x3B,0xB3,0xC2,0xE4,0x65,0xEC,0xF4
,0x90,0x9F,0xD7,0x91,0x6E,0xFF,0x00,0xF7,0x7A,0xEC,0x2D,0x71,0xC7,0xDB,0xCD
,0xA0,0x5D,0x93,0x64,0x88,0xC8,0x03,0xFC,0x2D,0x1F,0x10,0x48,0xF5,0x7F,0x7B
,0xFF,0x00,0xA6,0x7F,0xA6,0x83,0x8B,0xB4,0xD4,0xAB,0x60,0xEC,0x3D,0xC3,0xBC
,0x6B,0xA9,0xA3,0xB5,0xDB,0xAA,0xE6,0xA6,0x9A,0xA7,0xE1,0xDA,0x48,0x22,0x12
,0x48,0x48,0x5E,0x4F,0xE5,0xA1,0x65,0xE6,0x55,0x4A,0xE7,0xA8,0x55,0x2F,0x1F
,0x36,0x40,0xE0,0xEA,0xF3,0xB6,0xFD,0x94,0xFC,0x41,0x48,0x92,0xAA,0x96,0x2D
,0xB5,0x42,0xC6,0x34,0x0E,0x97,0x6A,0xE3,0x51,0x37,0x2E,0x05,0x64,0xE2,0xA9
,0x03,0x42,0xA1,0xB9,0x12,0x01,0xF3,0x0A,0x10,0xB8,0x72,0x47,0x22,0x1C,0xEF
,0x65,0xB0,0x5C,0x6E,0xB3,0x53,0xAC,0x51,0x88,0x61,0x9D,0xC2,0x24,0xF3,0x06
,0x08,0xC7,0x92,0xA9,0xE0,0x00,0x2D,0x23,0x0E,0x60,0x94,0x8C,0x33,0xE3,0x24
,0x29,0xC6,0xAF,0xAF,0x0C,0x3E,0xCC,0x37,0xDB,0xED,0x42,0x54,0xDD,0x83,0xD1
,0xD1,0x14,0x5E,0x7F,0x1B,0x09,0x49,0x43,0x94,0x22,0x45,0x10,0xA3,0xF2,0xE8
,0xCC,0xA5,0x5A,0x49,0x22,0x6F,0x4E,0x5A,0x32,0x09,0x51,0xD0,0x7B,0x17,0x6E
,0x0D,0x87,0xCE,0xAE,0xE9,0xE1,0x4D,0xCA,0xAE,0xE5,0x2F,0xFC,0xF2,0xF7,0x45
,0x73,0x4B,0xB5,0x45,0x41,0xC6,0x09,0x91,0xA5,0xF2,0xE6,0x20,0xFB,0xAA,0xA1
,0x1F,0x4D,0x4E,0xEC,0xFE,0x26,0xEC,0x1A,0xBA,0xA4,0xB6,0x0B,0xFD,0x35,0xAE
,0xB4,0x00,0xAB,0x41,0x73,0x8D,0xA8,0x27,0x1E,0xD8,0x11,0x4E,0x11,0x8F,0xEC
,0x0E,0x88,0xD6,0xEC,0x8F,0x07,0xF6,0x96,0xDA,0x14,0xB3,0x35,0x20,0xB8,0x56
,0x52,0xF2,0x30,0x4D,0x52,0xAA,0xC2,0x9C,0xB3,0x16,0x6F,0x22,0x35,0x55,0x8A
,0x00,0x58,0x96,0xC4,0x48,0x9D,0x49,0xC9,0x3D,0xF5,0x61,0xC7,0x1A,0x46,0x08
,0x45,0x0A,0x09,0xC9,0xC7,0xB9,0xF9,0x9F,0x99,0xD7,0xEA,0x32,0xBA,0x86,0x46
,0x0C,0xA4,0x64,0x10,0x72,0x08,0xD7,0xEE,0x88,0x6B,0xF2,0x47,0x48,0xD0,0xBC
,0x8C,0xA8,0xAA,0x32,0x59,0x8E,0x00,0xD7,0xC4,0xC2,0x63,0x81,0x13,0x22,0x7C
,0xD9,0x86,0x7F,0x60,0x3A,0x7F,0x5F,0xF2,0x3A,0xF8,0x86,0x99,0x11,0x83,0xBB
,0x3C,0xD2,0x8F,0xE7,0x90,0xE4,0xE7,0x18,0xC8,0x1D,0x97,0xF6,0x03,0x41,0xF1
,0x24,0xF3,0xC9,0x95,0xA5,0x87,0xA7,0x6F,0x36,0x50,0x42,0x8E,0xFD,0x97,0xBB
,0x7B,0x7C,0x81,0x1D,0x8E,0xB1,0x9E,0x86,0xA2,0x79,0x03,0x4B,0x2F,0x22,0x0E
,0x79,0xC8,0x01,0x0B,0xD7,0xF9,0x23,0xFC,0x20,0x8E,0xE1,0x9B,0x93,0x0E,0xC7
,0x3A,0xD9,0xE9,0xA0,0xC6,0xA6,0xA2,0x82,0x09,0x3C,0xEC,0x34,0xB3,0x91,0x83
,0x34,0x87,0x93,0x90,0x71,0x90,0x0F,0xB0,0x24,0x67,0x88,0xC0,0xFA,0x6B,0x27
,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x35,0xE5,0x53,0x51,0x05
,0x34,0x62,0x4A,0x89,0x92,0x25,0x2C,0x15,0x4B,0x1C,0x72,0x63,0xD8,0x0F,0x99
,0x3E,0xC0,0x75,0x3A,0xC6,0x15,0x35,0xB5,0x43,0xFE,0x49,0x4F,0xE4,0x21,0x19
,0x13,0x55,0x29,0x07,0xB0,0x20,0x88,0xFA,0x37,0xD0,0x86,0x28,0x47,0xD7,0x41
,0x9D,0xA6,0xB5,0xEF,0x34,0x14,0xB3,0x14,0x69,0x26,0xAD,0xAB,0xC6,0x44,0x6B
,0x86,0x71,0x9E,0x58,0xE8,0x30,0xA8,0x0F,0x51,0xC8,0xE0,0x1C,0x75,0x3A,0xFA
,0x61,0x57,0x22,0xB4,0x95,0x53,0x47,0x45,0x02,0xE4,0x95,0x8D,0xB2,0xD8,0x1F
,0x99,0xCF,0x40,0x31,0xDC,0x01,0xD3,0xD9,0xB4,0x1F,0x35,0x31,0x52,0x1A,0x94
,0x5A,0xE9,0xFC,0xF9,0x1F,0xAC,0x74,0xED,0xD5,0x7A,0x15,0xEA,0x10,0x7E,0x20
,0x0E,0x0E,0x4E,0x78,0xE7,0xB8,0xD6,0xC3,0x58,0x96,0xE8,0xA9,0x62,0x0F,0xF0
,0xB0,0x14,0x0D,0x82,0xD2,0x90,0x73,0x29,0x1D,0x32,0x49,0xF5,0x31,0xC0,0x1E
,0xA3,0xDF,0xA7,0x53,0xAC,0xBD,0x03,0x43,0xA6,0x9A,0x0F,0x09,0x26,0x91,0x88
,0x5A,0x68,0xBC,0xCF,0x9B,0xB1,0xE2,0x83,0xB7,0xBF,0x73,0xD0,0xFB,0x0C,0x74
,0xC6,0x46,0xBC,0xD6,0x85,0x64,0x65,0x92,0xB6,0x4F,0x8A,0x90,0x10,0x40,0x61
,0x88,0xD4,0x8C,0x1C,0xAA,0x76,0xEE,0x32,0x09,0xC9,0x1F,0x3D,0x65,0xEB,0x1A
,0xB2,0xBE,0x9A,0x95,0x84,0x72,0x33,0x3C,0xCC,0x32,0xB0,0xC6,0xA5,0xE4,0x61
,0xF3,0x0A,0x3A,0xE3,0xEB,0xD8,0x7B,0x9D,0x06,0x4E,0xB0,0xAE,0xF7,0x6B,0x75
,0xA6,0x34,0x7A,0xFA,0xA4,0x88,0xC8,0x4A,0xC5,0x18,0x05,0xA4,0x94,0xFE,0x54
,0x45,0xCB,0x39,0xFA,0x28,0x27,0x58,0x15,0x52,0x5E,0xEB,0x3D,0x28,0x1A,0xDB
,0x11,0xEC,0x90,0x84,0x96,0xA4,0xFE,0xAC,0xDF,0x75,0x1F,0xFE,0x7C,0x8E,0xC4
,0x1D,0x78,0x50,0xED,0xA0,0x92,0x49,0x2C,0x92,0x9A,0x66,0x94,0x71,0x95,0xA0
,0x95,0x9E,0xA2,0x51,0xF2,0x7A,0x96,0xFB,0xC2,0x3E,0x41,0x78,0xF1,0xEC,0x0E
,0x34,0x56,0x15,0xC7,0x72,0xDC,0x27,0x9C,0x52,0x52,0x53,0x35,0x03,0xBA,0x86
,0x58,0x9A,0x3F,0x3E,0xB9,0xD4,0xF6,0x61,0x02,0x9E,0x31,0x29,0xF6,0x92,0x56
,0x00,0x1E,0x8C,0xBA,0xF0,0xA5,0xB1,0x5E,0x2A,0xAB,0xE3,0x9A,0xAD,0xEA,0x28
,0x4B,0x83,0xCA,0x75,0x74,0x9E,0xB0,0x29,0x1D,0x41,0x95,0xBD,0x10,0x86,0xEC
,0x63,0x81,0x0E,0x0E,0x08,0x61,0xDF,0x52,0xEB,0x75,0x05,0x15,0xBA,0x03,0x05
,0x0D,0x2C,0x54,0xF1,0x96,0x2C,0xC1,0x17,0x1C,0x98,0xF7,0x66,0x3D,0xCB,0x1F
,0x72,0x7A,0x9D,0x64,0xE8,0x8D,0x4D,0xBF,0x6F,0x5B,0x2D,0xB0,0x49,0x15,0xBE
,0x9C,0x40,0x65,0x21,0xA6,0x7E,0x6C,0xD2,0x4E,0xC3,0xB3,0x4B,0x21,0x3C,0xE4
,0x3F,0xED,0x37,0x51,0xD0,0xF4,0xD7,0xE2,0x58,0x28,0x9D,0xA2,0x6A,0xE5,0x15
,0x82,0x12,0x0C,0x30,0xBA,0x81,0x04,0x38,0xED,0xC2,0x21,0xE9,0xC8,0xF6,0x27
,0x2C,0x3E,0x7A,0xDB,0xE9,0xA0,0x6B,0x5B,0xB9,0xAF,0xD6,0x7D,0xB5,0x65,0xA8
,0xBC,0xDF,0x6E,0x10,0x50,0x50,0x53,0x8C,0xC9,0x34,0xA7,0x03,0xE8,0x00,0xEE
,0xCC,0x4F,0x40,0xA3,0x24,0x9E,0x80,0x6A,0x3D,0xBF,0x7C,0x43,0xB5,0xED,0xAA
,0x9F,0xE1,0x14,0x91,0x7F,0x17,0xDC,0x0F,0x1F,0x98,0x96,0xE8,0xA6,0x58,0xC4
,0x49,0xDB,0xCD,0xA8,0x95,0xBD,0x10,0x45,0x92,0x3D,0x6E,0x7A,0xF6,0x50,0xC7
,0xA6,0xAA,0x6F,0x10,0xE2,0x6B,0x2D,0xA5,0xBC,0x55,0xF1,0x2A,0xA6,0x5B,0xE5
,0x55,0xB7,0xD7,0x6B,0xA2,0x33,0x4B,0x43,0x6B,0xA6,0x9C,0x8C,0x22,0x53,0x45
,0x8F,0x3E,0xA6,0x42,0x4F,0xF7,0x8E,0x15,0x4A,0xE4,0xE4,0x01,0xD0,0xAD,0x1F
,0xDA,0x03,0xC5,0x01,0x35,0xB8,0x45,0x7E,0x49,0x29,0x92,0xA8,0xA8,0xB3,0xED
,0x0E,0xF5,0x55,0x8C,0x7F,0x05,0x4D,0xC5,0x57,0xAC,0x70,0x8E,0x8C,0xB4,0xDF
,0x89,0xCE,0x39,0x74,0xC8,0x16,0xD7,0xD9,0x93,0xC2,0x95,0xD9,0x16,0x56,0xDC
,0x17,0x8A,0x31,0x16,0xE0,0xB9,0x47,0xEA,0x47,0xC1,0x92,0x96,0x26,0x3C,0x8A
,0xB9,0x1D,0xE5,0x76,0xC3,0x48,0x47,0x4C,0x85,0x41,0xE9,0x8D,0x75,0x52,0x7D
,0x98,0xB6,0xF7,0xF6,0xC7,0x75,0x4F,0xE3,0x16,0xF1,0xB4,0x41,0x5D,0x76,0x92
,0x76,0x7B,0x25,0xA2,0xDD,0x4C,0x04,0x74,0x8B,0xED,0x23,0x64,0xF0,0x89,0x40
,0xCF,0x96,0x1D,0x94,0x9F,0x53,0xFA,0x98,0xA9,0xD7,0x49,0xC3,0x72,0x37,0x84
,0x96,0x5A,0xEB,0xD4,0x34,0x96,0xF8,0xCE,0x25,0xFE,0x1F,0x29,0x11,0x03,0x9C
,0x71,0x6A,0xB2,0x06,0x4E,0x7A,0x62,0x3E,0x2C,0xA7,0xA1,0x27,0x45,0x6E,0x6E
,0xB7,0xFB,0x7D,0x05,0x4F,0xC1,0x83,0x2D,0x65,0xC0,0xA8,0x65,0xA2,0xA5,0x4F
,0x32,0x62,0x0F,0x62,0x40,0xE8,0x8A,0x4F,0x4E,0x4E,0x55,0x7E,0xBA,0xC4,0x14
,0xDB,0x86,0xEF,0xD6,0xBA,0xA4,0x59,0x68,0xCF,0xFD,0x1A,0x91,0xC3,0xD4,0xB8
,0xF9,0x3C,0xA4,0x61,0x32,0x3A,0x11,0x18,0x24,0x1E,0xA2,0x4D,0x6C,0xAC,0xF4
,0xB6,0xDA,0x1A,0x45,0x8A,0xDD,0x4D,0x1D,0x2C,0x52,0x31,0x70,0xA2,0x3E,0x05
,0xDB,0xDD,0x8E,0x7A,0x96,0x3E,0xE4,0xF5,0x3A,0xCC,0x57,0x46,0x66,0x55,0x75
,0x62,0x87,0x8B,0x00,0x73,0xC4,0xE0,0x1C,0x1F,0x97,0x42,0x0F,0xEF,0xA0,0xC5
,0xB4,0xDA,0xED,0xF6,0xA8,0x1A,0x1B,0x7D,0x2A,0x40,0x1D,0xB9,0xC8,0xC3,0x25
,0xE5,0x6F,0xCC,0xEC,0x72,0xCE,0xDF,0xE2,0x62,0x49,0xF9,0xEB,0x33,0x42,0x40
,0x19,0x24,0x01,0xF5,0xD6,0x3D,0x5D,0x4A,0xC2,0xA3,0xD4,0xAA,0xCD,0x90,0xB9
,0x04,0x92,0x71,0xEC,0xA3,0xAB,0x1F,0xA0,0xD0,0x64,0x12,0x07,0x73,0x8D,0x35
,0xA5,0x49,0x25,0x92,0xEA,0x8A,0x30,0xF2,0x23,0x8E,0x61,0xC7,0x98,0xE8,0x3A
,0xFB,0x29,0x09,0x17,0x46,0x04,0x31,0x25,0x98,0x64,0x60,0x9D,0x6E,0xB4,0x0D
,0x71,0x7F,0xDB,0xE2,0x82,0xBA,0x74,0xDA,0xB0,0x2C,0x58,0xA6,0x96,0xDC,0x65
,0x8D,0x81,0xEA,0x5A,0x9A,0x29,0xE4,0x90,0x63,0xE8,0xAE,0x9F,0xD7,0x5D,0x9E
,0xEC,0x15,0x0B,0x1E,0xC0,0x64,0xEB,0xF9,0xD1,0xF6,0xB2,0xF1,0x06,0xF5,0x71
,0xDF,0xB7,0x2B,0x54,0xD5,0x55,0x51,0xB5,0xBE,0x6A,0x8B,0x6D,0xBC,0xC2,0xFC
,0x22,0x8A,0x8D,0x59,0xE1,0x9C,0x63,0xA9,0x32,0x4A,0xE8,0x51,0x98,0x70,0xFB
,0xB8,0xF8,0x9E,0x62,0x42,0x14,0x2D,0x9F,0x0A,0x21,0xB3,0xF8,0x51,0xE0,0x7D
,0xCF,0x7F,0x59,0x2D,0xD0,0x9B,0xB5,0xF2,0xB9,0xAD,0xB6,0x38,0xEA,0x5C,0x1C
,0xAF,0x9E,0xD0,0xC3,0xE6,0x3F,0x4C,0x96,0x2A,0xD3,0x48,0xC3,0x88,0x20,0x00
,0x02,0xAA,0xAA,0xAE,0xCE,0xD3,0xE1,0x3C,0x7B,0x9A,0x35,0xB8,0xEF,0x59,0xAE
,0x5B,0xAE,0xEB,0x27,0xAA,0x4A,0xFB,0xDC,0xB2,0xC7,0x4D,0x13,0x1E,0xE2,0x9E
,0x8D,0x59,0x48,0x41,0xDB,0x0F,0xC3,0xE6,0x3A,0x74,0xD7,0xCE,0xCF,0x99,0x9B
,0xC3,0x5F,0x01,0xE0,0x1D,0x10,0xDC,0xE5,0x32,0x63,0xB7,0x34,0xA5,0xAA,0x1F
,0xF1,0x2D,0xAB,0xBA,0xA2,0x68,0x69,0xE0,0x79,0xEA,0x25,0x8E,0x18,0x90,0x65
,0x9D,0xD8,0x2A,0xA8,0xF9,0x92,0x7B,0x6A,0xC2,0x2B,0x9B,0x4F,0x84,0x34,0x16
,0x1A,0x74,0x1B,0x5F,0x77,0x6E,0xDB,0x0C,0xEB,0x92,0x5A,0x92,0xE1,0x98,0x5C
,0xE7,0x3D,0x60,0x75,0x31,0x60,0x76,0xC0,0x51,0xD3,0x59,0x57,0x8A,0x5F,0x10
,0x69,0xED,0x72,0xC1,0x7D,0x93,0x65,0xEF,0xBB,0x54,0x6A,0x5E,0x48,0xEF,0x54
,0x5F,0x03,0x27,0x10,0x3A,0xE5,0xD4,0x49,0x11,0xE9,0xEF,0xE5,0xA8,0xD6,0xFA
,0xE9,0xB8,0xCC,0x71,0x2B,0x52,0x44,0x91,0x44,0xED,0xC1,0x2A,0xAB,0x32,0x88
,0xED,0xF2,0x8A,0x31,0xF7,0x93,0x37,0xB8,0x00,0x00,0xC3,0xB3,0x6B,0x4B,0x50
,0x95,0xB5,0x93,0x24,0xB5,0x34,0xD5,0x75,0xF3,0x02,0x1A,0x21,0x53,0x00,0x3C
,0x4F,0x70,0x63,0xA6,0x04,0x24,0x7F,0x35,0x79,0xDB,0x9A,0x9E,0x9D,0x74,0x15
,0xED,0xBE,0xEB,0xB3,0x2D,0xCE,0x95,0x95,0x3B,0x4F,0x7C,0xF8,0x63,0x45,0x2B
,0x94,0x4B,0xA5,0x8A,0xBE,0x59,0x68,0x24,0x7C,0xE1,0x42,0xC5,0x18,0x64,0x65
,0x6C,0xFA,0x5B,0xC9,0xE2,0x7F,0xA6,0xA6,0x76,0xCD,0xE1,0x7D,0x37,0x2B,0x6D
,0xBB,0x67,0x78,0xA9,0xB6,0x77,0x7C,0xD5,0x75,0x0D,0x4E,0x68,0x2F,0x54,0xAB
,0x05,0x64,0x78,0x86,0x49,0x46,0x4C,0x1C,0x0A,0x93,0xE5,0xF1,0xF5,0x45,0xDD
,0x86,0xB7,0xF4,0xDB,0x7A,0xE3,0x59,0x3F,0xC5,0x57,0x55,0xB5,0x13,0xB0,0x2A
,0x64,0x8D,0xC4,0xB5,0x6C,0xA7,0xBA,0xF9,0xA4,0x05,0x88,0x11,0xDD,0x62,0x51
,0x83,0xD4,0x36,0xBD,0x2E,0xFE,0x1E,0xEC,0x8B,0xBD,0xBD,0x68,0x2E,0x7B,0x5E
,0xD9,0x59,0x0A,0x92,0xC1,0xA5,0x80,0x34,0x9C,0x8F,0x76,0xF3,0x3F,0x1F,0x23
,0xEE,0xD9,0xC9,0xF7,0x3A,0x60,0xC1,0x87,0xC5,0xEB,0xBD,0xB6,0x38,0x5F,0x75
,0xF8,0x7B,0x74,0xA5,0x86,0x5A,0x68,0xEA,0x16,0xAE,0xD3,0x55,0x15,0x74,0x3C
,0x1E,0x17,0x9B,0x24,0x13,0x1C,0x83,0x09,0x14,0x84,0xFA,0x0E,0x38,0x1E,0xFD
,0x09,0x91,0x5B,0x7C,0x5A,0xF0,0xF6,0xB2,0xB0,0xD0,0xCD,0xB9,0x20,0xB5,0x57
,0x2B,0xF0,0x6A,0x4B,0xBC,0x6F,0x41,0x32,0xBE,0x01,0xE3,0xC2,0x75,0x43,0x9E
,0xA3,0xA0,0xF9,0x8D,0x42,0xAA,0x7C,0x20,0xB5,0x45,0x0F,0x95,0x61,0xDC,0xDB
,0xA6,0xC9,0x1A,0xA1,0x48,0xE1,0x8A,0xE0,0x6A,0x61,0x45,0xF2,0xE4,0x8B,0x88
,0x4A,0x81,0x26,0x17,0x84,0xB2,0x28,0x00,0x8C,0x07,0x38,0xD6,0x9B,0xC4,0x1B
,0x4E,0xEA,0xB3,0xD8,0xAE,0xF5,0xFB,0x83,0x77,0xED,0x0B,0xBD,0x15,0x5A,0xCC
,0x66,0x7B,0xF5,0x01,0xA5,0x3C,0xE4,0xA6,0x8A,0x05,0xF5,0x46,0x59,0x58,0x8F
,0x22,0x26,0x00,0x47,0x92,0x41,0x1F,0xCD,0xD2,0x60,0xE8,0x38,0x26,0x86,0xA2
,0x15,0x9A,0x09,0x52,0x58,0x9C,0x65,0x5D,0x18,0x32,0xB0,0xF9,0x82,0x3B,0xEB
,0xEF,0x5C,0x8B,0x60,0xDF,0xFE,0x10,0x54,0xFC,0x45,0x49,0xDA,0x35,0xBB,0x12
,0xA2,0x19,0x9E,0x39,0x26,0xB1,0x6E,0x48,0x28,0x59,0x19,0x4E,0x09,0x34,0xFE
,0x74,0x32,0x11,0xD3,0xB1,0x84,0x9F,0xA6,0xA5,0x34,0x7E,0x20,0xD6,0x43,0xB8
,0xAD,0xB4,0x3B,0x67,0xC6,0x15,0x36,0xFA,0xF8,0xA1,0x7A,0x64,0xDE,0x16,0x80
,0xA5,0xF9,0xA4,0xEF,0x91,0x27,0x18,0x19,0x93,0xEE,0xE2,0x00,0xE5,0xB3,0xE6
,0xAE,0x09,0xC8,0xD0,0xC7,0x48,0xE9,0xAA,0xEE,0x2B,0xF7,0x8A,0xF4,0x11,0xAB
,0xD6,0xEC,0xBB,0x06,0xE2,0x81,0x87,0x25,0x9E,0xC9,0x78,0xF2,0x9D,0x87,0xD2
,0x3A,0x84,0x0B,0xFF,0x00,0xDC,0x3A,0xF9,0x9B,0xC6,0x0D,0xBF,0x6C,0x60,0xBB
,0xB6,0xC9,0xB9,0xF6,0xA6,0x4E,0x0C,0xB7,0x3B,0x5B,0x9A,0x70,0x7F,0xED,0xE1
,0xF3,0x23,0xC7,0xD4,0xB0,0xD0,0x58,0xDA,0x6B,0x12,0xCF,0x74,0xB6,0xDE,0x6D
,0xF1,0x5C,0x6D,0x17,0x0A,0x5B,0x85,0x1C,0xA3,0x31,0xCF,0x4D,0x32,0xC9,0x1B
,0x8F,0xA3,0x29,0x20,0xEB,0x2F,0x44,0x35,0x8D,0x52,0x2B,0x24,0x6F,0x2E,0x06
,0x48,0x13,0xF9,0xA5,0x61,0xC9,0xBF,0xDD,0x5E,0xDF,0x3E,0xA7,0xB1,0x1F,0x84
,0xEB,0xF2,0xA2,0xBE,0x18,0xE5,0x30,0x44,0xB2,0x54,0xD4,0x2F,0x78,0xA1,0x00
,0x95,0xF7,0xF5,0x13,0x80,0xBD,0x3A,0x8E,0x44,0x67,0xDB,0x3A,0xC1,0xB8,0xD0
,0x56,0x5C,0x22,0x22,0xBE,0xE9,0x2D,0x05,0x28,0xEB,0x24,0x54,0x52,0x79,0x6C
,0xEB,0x82,0x08,0x69,0x88,0x0E,0x07,0x50,0x41,0x8F,0xCB,0x60,0x47,0xE2,0x23
,0x41,0xFA,0x24,0xB7,0x51,0x56,0xB4,0x71,0xF9,0xB7,0x0B,0xA0,0x4F,0x58,0x52
,0x24,0x98,0x02,0x33,0xD4,0x9C,0x2C,0x4A,0xDC,0x49,0x00,0xF1,0x52,0x41,0xC7
,0x5D,0x64,0x48,0x95,0x12,0x44,0xF2,0xD7,0xD4,0xA5,0x2C,0x0A,0x09,0x64,0x86
,0x42,0x30,0x3A,0xF5,0x69,0x0E,0x0F,0x6C,0x1F,0x4F,0x1C,0x1C,0xF5,0x23,0x5F
,0x94,0x50,0x45,0x49,0x4D,0xF0,0xB6,0x9A,0x28,0x69,0x61,0x05,0x9B,0x91,0x4E
,0x2B,0xC8,0x92,0x4B,0x71,0x18,0x2C,0x4B,0x75,0x24,0xE3,0x39,0xCE,0x4E,0xBD
,0x5A,0x9E,0x08,0xB3,0x55,0x59,0x2F,0x98,0x63,0xCB,0xF9,0x93,0x10,0x16,0x3C
,0x64,0xE4,0x0E,0xCB,0x80,0x4F,0x5E,0xF8,0xEE,0x4E,0x83,0xCE,0x27,0x6E,0x06
,0x1B,0x65,0x22,0x45,0x1E,0x49,0xF3,0x5D,0x38,0x26,0x49,0x39,0x21,0x7B,0xB9
,0xCE,0x0F,0xB0,0x39,0xCF,0x2D,0x7B,0x25,0x22,0x06,0x13,0x54,0xC8,0xD3,0xC8
,0xA7,0x90,0x67,0xFC,0x29,0xDF,0xF0,0xAF,0x61,0x8C,0x9E,0xBD,0xF1,0xDC,0x9D
,0x79,0x2D,0x74,0x95,0x5D,0x2D,0xF0,0xF3,0x43,0xFF,0x00,0x48,0x93,0x2B,0x1F
,0xEA,0xBE,0xEF,0xEC,0x46,0x30,0xA4,0x7F,0x36,0xBD,0xA1,0xA6,0xC1,0x12,0x54
,0x4A,0xD3,0xC9,0xDF,0x2C,0x30,0xAA,0x7F,0xC2,0xBD,0x87,0xD0,0x9C,0x9F,0xA9
,0xD0,0x7B,0xAB,0x06,0x19,0x1D,0x47,0xCF,0x5F,0xBA,0x69,0xA0,0x69,0xA6,0x9A
,0x0F,0x39,0x51,0xA4,0xE9,0xE6,0x32,0x2F,0xBF,0x1E,0x84,0xFE,0xFE,0xDF,0xB7
,0x5F,0xAE,0x94,0xF0,0x43,0x02,0x91,0x0C,0x6A,0x9C,0x8E,0x58,0x8E,0xEC,0x7E
,0x64,0xF7,0x27,0xEA,0x75,0xE9,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x68,0x48,0x00
,0x92,0x40,0x03,0xB9,0x3A,0xAA,0xF7,0x37,0x8D,0x36,0x51,0x7D,0x6D,0xAB,0xB0
,0xE8,0xE4,0xDE,0x7B,0x97,0xB3,0x41,0x40,0x4B,0x53,0x53,0x7B,0x72,0x9A,0x65
,0x0C,0x00,0x1E,0xE1,0x43,0x1E,0x98,0x20,0x68,0xAB,0x2E,0xEB,0x71,0xA0,0xB4
,0xDB,0xA7,0xB8,0xDD,0x2B,0x69,0xE8,0xA8,0xE9,0xD0,0xBC,0xD3,0xCF,0x20,0x48
,0xE3,0x5F,0x99,0x63,0xD0,0x6A,0xAE,0xAE,0xDE,0xD7,0x8D,0xEB,0x17,0x0D,0xAF
,0x59,0x2E,0xD6,0xDB,0x73,0xE5,0x22,0xBF,0xD4,0xD1,0x99,0x2B,0x2E,0x27,0xDD
,0x6D,0xF4,0x84,0x73,0x71,0x8F,0xF4,0xAC,0xA7,0xFC,0x2A,0x71,0x91,0xAE,0x8B
,0x6A,0x5C,0xB7,0x15,0xE1,0x6B,0xF7,0xED,0xC5,0xB7,0x05,0xD6,0x8D,0xFC,0xC5
,0xB5,0x51,0x47,0x1B,0xD2,0x5B,0x1C,0x0C,0xF5,0x46,0x26,0x9E,0x27,0x5F,0xCD
,0x3B,0x4A,0xEC,0xBD,0x55,0x54,0xF4,0xD4,0xB2,0x1F,0x22,0x2A,0x79,0x2B,0xD2
,0x68,0x44,0x53,0x62,0x39,0x2A,0x21,0xAD,0x78,0xA1,0x9B,0xF2,0xA3,0xD7,0xB8
,0xF3,0xEA,0x3B,0x7A,0x56,0x05,0x00,0x1F,0x49,0x18,0xD0,0xC4,0x6B,0x69,0x6D
,0x3D,0xB9,0xB2,0x25,0xAC,0x9E,0xD5,0x41,0x57,0x1D,0xC3,0x26,0xAE,0xBA,0xE3
,0x73,0x95,0x6E,0x17,0x87,0x3C,0x49,0x33,0x3F,0x23,0xE4,0x52,0x65,0x72,0x44
,0x92,0x11,0x91,0x90,0x50,0x1D,0x52,0xD4,0x94,0x75,0xFF,0x00,0x68,0x1F,0x16
,0xA0,0xDC,0x97,0x1B,0x1C,0xA9,0xB1,0xB6,0xEA,0xF2,0x90,0xD7,0xD6,0xAA,0x07
,0x24,0x07,0x02,0xA2,0x77,0xC9,0x1E,0x66,0x03,0x15,0x45,0x21,0x10,0xA8,0x00
,0x72,0x2D,0xAD,0xDF,0x8A,0xFB,0x9E,0xA7,0x7A,0x6E,0x6A,0x4F,0x0B,0xF6,0x9D
,0x7D,0x54,0x70,0x25,0x53,0xC3,0x5B,0x14,0x14,0x4F,0x47,0x46,0xF2,0x85,0xCB
,0xC7,0xF0,0xE9,0x99,0x24,0x8D,0x01,0xE7,0x28,0x93,0x93,0x33,0xF0,0x45,0x55
,0xE4,0x4E,0xAF,0x1F,0x0B,0xFC,0x34,0x7D,0xB3,0xB6,0xE8,0x6D,0x14,0x86,0x60
,0x94,0xCE,0xD3,0x1B,0x85,0xD1,0x52,0x5A,0x97,0x99,0x98,0xB3,0xCB,0x1D,0x3A
,0xFD,0xDC,0x4E,0x49,0x24,0x48,0xE5,0xDC,0x0F,0x4B,0x02,0x00,0xD1,0x5F,0xB5
,0xBE,0x40,0xB5,0xD3,0x53,0xB8,0xA3,0x86,0xD0,0xB8,0x8A,0x96,0x29,0x29,0x5A
,0x9A,0xDA,0x7E,0x49,0x05,0x1A,0x7D,0xED,0x61,0xFF,0x00,0x0B,0x10,0x8D,0xD1
,0x93,0xE5,0xAD,0xB5,0xB2,0xD9,0x5F,0x51,0x55,0x15,0x49,0x8E,0xA9,0x27,0x8D
,0x7E,0xEA,0xA2,0xB5,0x23,0x92,0xB1,0x57,0x07,0x1E,0x4C,0x03,0x10,0xD2,0x29
,0x03,0x01,0x98,0x72,0x20,0x61,0x97,0x3D,0x75,0x26,0xA0,0xDB,0x54,0x94,0x75
,0xAD,0x5A,0x92,0xCD,0x35,0x53,0xA7,0x19,0x2A,0x66,0x6E,0x55,0x12,0x83,0xDD
,0x4C,0x87,0x25,0x53,0x38,0x21,0x23,0xE0,0xA0,0xE7,0x03,0xAE,0x35,0xF5,0x74
,0xA4,0x81,0x61,0xE1,0x55,0x34,0x6E,0xB2,0x72,0x09,0x4E,0x63,0x26,0x36,0x38
,0x62,0x7E,0xE9,0x7D,0x52,0x92,0x0E,0x48,0x24,0xF6,0xC8,0x03,0x41,0xAE,0x88
,0xD2,0xC5,0x4D,0x34,0xBF,0x15,0x98,0x87,0x49,0xEA,0x45,0x51,0x54,0xEE,0x47
,0xAE,0xA9,0xBA,0x9E,0xA3,0xF0,0xC7,0x8E,0x27,0xA1,0xE9,0xAC,0xDA,0x39,0xA3
,0xA7,0x82,0x38,0x69,0x69,0xE3,0xA5,0x85,0x99,0x8C,0x7C,0xA9,0xCC,0x41,0xDB
,0x24,0xB7,0x08,0x07,0xAD,0x8E,0x72,0x4F,0x2C,0x7E,0x61,0x91,0xAC,0x49,0x69
,0xCF,0xC6,0x07,0x3E,0x7C,0x95,0xA8,0x09,0x8C,0xB0,0x49,0xAA,0x63,0x04,0x38
,0x1C,0x17,0xFB,0x98,0x32,0x06,0x03,0x1F,0xC4,0x3A,0x37,0x5D,0x6C,0x28,0xED
,0x12,0x73,0x91,0xE6,0x7F,0x87,0x12,0x74,0x71,0x0C,0xAC,0xD2,0xC8,0x32,0xD8
,0x0F,0x31,0xF5,0x63,0xD5,0x90,0x17,0x8F,0x13,0x90,0x09,0x1A,0x04,0xB3,0xD4
,0x34,0xC6,0x14,0x69,0x16,0x50,0x3A,0x85,0xE3,0x24,0xE3,0x20,0xE3,0xA7,0xF7
,0x71,0x76,0xE8,0x4E,0x43,0x76,0xC6,0x75,0x91,0x05,0xBD,0x89,0x66,0x95,0xCC
,0x41,0xBF,0x12,0xC5,0x23,0x17,0x6E,0xA7,0x1C,0xA4,0x3E,0xA2,0x3A,0xE4,0x01
,0x8C,0x75,0x1D,0x46,0xB3,0xA9,0xE1,0x86,0x9E,0x21,0x14,0x11,0x24,0x48,0x09
,0x3C,0x51,0x70,0x32,0x4E,0x49,0xFE,0xBA,0xFB,0xD0,0x7C,0x43,0x14,0x50,0x44
,0x22,0x86,0x34,0x8E,0x35,0xEC,0xA8,0xA0,0x01,0xFB,0x0D,0x7D,0xE9,0xA6,0x83
,0xCA,0xB7,0xAD,0x1C,0xC0,0xFF,0x00,0xAB,0x6F,0xF8,0x6B,0xF9,0x59,0xE3,0xBD
,0xD2,0x86,0xED,0xBD,0x5E,0xA2,0x96,0x7F,0x36,0x71,0x2D,0x79,0xAC,0x1C,0x18
,0x70,0x95,0xEE,0x55,0x92,0x81,0x92,0x30,0x73,0x1C,0x91,0xB6,0x46,0x47,0xAB
,0x1D,0xC1,0x03,0xFA,0x97,0x7C,0xAA,0x8E,0x8E,0xD9,0x24,0xF3,0x7F,0x74,0x0A
,0x2B,0xFD,0x15,0x98,0x29,0x3F,0xD0,0xEB,0xF9,0x27,0xBF,0xE2,0x78,0xB7,0x21
,0x79,0x57,0x0F,0x53,0x45,0x47,0x56,0xDF,0xAC,0xD4,0xB1,0x4B,0xFF,0x00,0xBF
,0x41,0xD7,0x3B,0x5A,0xAD,0x6B,0xBC,0x20,0xD8,0x95,0x7B,0x7E,0xBA,0x98,0x2D
,0xB7,0x7E,0x54,0xD3,0x51,0xD5,0xD4,0x53,0xBB,0x46,0x62,0x9D,0xAA,0xC4,0x4C
,0x63,0xCA,0x36,0x08,0x9E,0x3C,0x02,0x54,0xF5,0x19,0xC6,0xAF,0x0A,0x1B,0x3C
,0x71,0x32,0x4F,0x5D,0x53,0x35,0xCA,0xB1,0x7A,0x89,0xEA,0x31,0xE8,0x3F,0x34
,0x40,0x02,0xA7,0xEA,0x06,0x7E,0x64,0xEB,0x93,0xFC,0x32,0xBE,0x54,0xD5,0xF8
,0x15,0x7C,0xF2,0x56,0xE3,0x4C,0x96,0x49,0xED,0x97,0xC8,0x29,0x62,0xA5,0xF8
,0x88,0x99,0x69,0xCC,0x71,0xC9,0x29,0x75,0x4E,0x51,0x96,0x6A,0x39,0x5B,0xA9
,0xE2,0x01,0xC7,0xF8,0x8F,0x61,0x46,0xE9,0x24,0x6B,0x24,0x6C,0x19,0x18,0x06
,0x56,0x07,0xA1,0x07,0xB1,0xD5,0x84,0x96,0x35,0x3D,0xBA,0x8E,0x0A,0xA7,0xAB
,0x48,0x15,0xAA,0x9C,0x71,0x69,0xDF,0xD5,0x21,0x1D,0xF8,0xF2,0x3D,0x42,0xE7
,0xF9,0x47,0x4F,0xA6,0xB2,0xB4,0xD6,0xA7,0x72,0xEE,0x4B,0x1E,0xDC,0xA6,0x49
,0xEF,0x57,0x28,0x69,0x3C,0xD6,0xE3,0x0C,0x67,0x2D,0x2C,0xCD,0xF9,0x63,0x8D
,0x72,0xCE,0xDF,0x45,0x04,0xEA,0xA3,0x6D,0xAD,0x5E,0xE7,0xDC,0x56,0x2D,0xB3
,0x6D,0x6B,0x8D,0xFE,0xEB,0x4B,0x6E,0xA5,0x5E,0x81,0xE7,0x90,0x2F,0x23,0xF9
,0x54,0x77,0x66,0xFA,0x00,0x4E,0xB9,0xCF,0xC4,0xAF,0xB4,0xDD,0x33,0x43,0x53
,0x4D,0xB4,0xAA,0x63,0xA2,0x44,0xF4,0xAC,0xED,0x00,0xA8,0xAA,0x98,0xB4,0x65
,0x95,0x91,0x7F,0xB9,0x8D,0x33,0xC5,0x4B,0x3B,0x3B,0xA9,0x6F,0xEE,0x89,0x04
,0x0E,0x6C,0xDE,0x1B,0xE2,0xF5,0xB8,0xEE,0x53,0x55,0xCF,0x55,0x55,0xCA,0x4E
,0x4A,0x65,0x9A,0xA0,0xCD,0x52,0xE8,0x79,0x8E,0x2D,0x29,0xC6,0x01,0x47,0xE2
,0xC9,0x18,0x8E,0x36,0xC0,0x3C,0x01,0xD4,0xD5,0xC7,0x4A,0x78,0xB3,0xF6,0x9D
,0x6A,0x43,0x3D,0xBB,0x67,0xD2,0x0A,0x49,0x17,0x92,0x1A,0xAA,0xC8,0xB9,0xCF
,0xC8,0x73,0x1D,0x20,0xCE,0x23,0xC3,0xA0,0x07,0xCD,0x21,0x80,0x60,0x7C,0xB2
,0x35,0xCE,0xDB,0xAF,0xC4,0x9D,0xC9,0xB8,0x2A,0x26,0x9A,0xA6,0xB2,0x67,0x9A
,0x50,0xC8,0x6A,0xAA,0x24,0xF3,0xAA,0x42,0x12,0x0F,0x15,0x72,0x00,0x84,0x74
,0xC7,0xDC,0xAC,0x7C,0x95,0x88,0x6E,0x40,0xEA,0x19,0xA6,0xA2,0x9A,0xCC,0xB5
,0xDD,0x2E,0x76,0xB7,0x99,0xED,0x97,0x1A,0xBA,0x16,0x9E,0x23,0x0C,0xCD,0x4F
,0x33,0x46,0x64,0x8C,0x90,0x4A,0x37,0x12,0x32,0xA4,0xA8,0x38,0x3D,0x3A,0x0F
,0x96,0xBE,0xAD,0x76,0xAA,0xEB,0x9F,0x33,0x4B,0x12,0xF9,0x51,0xF4,0x92,0x69
,0xA5,0x58,0x61,0x8C,0xF1,0x66,0x01,0xA4,0x72,0x11,0x49,0x08,0xDC,0x41,0x39
,0x62,0x30,0x32,0x7A,0x6A,0x43,0x41,0xB2,0xA5,0xA9,0xA7,0x9C,0x7C,0x45,0x64
,0xD5,0x6B,0x49,0x15,0x54,0x69,0x4D,0x42,0x7C,0xA5,0x57,0x04,0x91,0x2C,0x93
,0x34,0x5C,0x71,0x8C,0x02,0x82,0x45,0x6C,0x36,0x0E,0x00,0x24,0x3A,0x87,0xEC
,0x9D,0xB2,0xA8,0x77,0x33,0x4D,0x71,0xB8,0xD5,0x5D,0x68,0x1E,0xB6,0x29,0x67
,0x41,0x68,0xB8,0x4D,0x6F,0x44,0x61,0x0D,0x0B,0x16,0x58,0xE0,0x64,0x40,0x0B
,0xCF,0x37,0xA7,0x8F,0x1F,0x90,0xC0,0x1A,0xF7,0xDF,0xDB,0xC3,0x7D,0xEC,0xDF
,0x15,0x2E,0xDB,0x73,0x6A,0x6F,0xCB,0x85,0x75,0x9E,0xD6,0x90,0x47,0x32,0x5E
,0xE9,0xA1,0xAD,0xE7,0x50,0xE9,0xE6,0x3A,0x07,0x55,0x47,0xE2,0xA8,0xD1,0x8E
,0xAC,0x4E,0x49,0xEB,0xF2,0x84,0xF8,0x47,0xE2,0xB6,0xFA,0xD9,0x16,0xAF,0x80
,0xB5,0x6C,0x8A,0x5A,0x1A,0x34,0x76,0xF8,0x5A,0xBB,0xAD,0x2D,0x55,0x54,0xD1
,0xC0,0x52,0x24,0xF2,0xC8,0x80,0x27,0x23,0xF7,0x2A,0xC5,0x8A,0xA8,0xEF,0x80
,0x3B,0x6B,0x41,0x59,0x45,0x79,0xDC,0xF7,0x9B,0xE5,0xE6,0xA6,0xE3,0x43,0x7C
,0xA7,0xB9,0xD6,0x4B,0x55,0x55,0x4F,0x6A,0xAB,0x34,0xB2,0x24,0xB2,0x31,0x2C
,0x84,0x30,0x27,0x00,0x61,0x42,0xB3,0xAF,0x41,0xD7,0x3A,0x0B,0x23,0x63,0xF8
,0x99,0x63,0xBA,0x6E,0xF1,0x44,0x86,0x93,0xC3,0xFD,0xD2,0xF5,0x1E,0x5A,0x5D
,0x6D,0x71,0x73,0xB3,0x5D,0x25,0x20,0x7D,0xCD,0x54,0x24,0x80,0x1F,0xA8,0x19
,0xE5,0xC9,0x4F,0x69,0x07,0x63,0xD1,0xDB,0x23,0x76,0x25,0xF6,0xA2,0xE3,0x65
,0xB9,0x53,0x2D,0xBB,0x71,0x5A,0x5D,0x63,0xB8,0xD0,0x89,0x39,0x28,0x0C,0x32
,0x93,0x44,0xD8,0x05,0xE2,0x71,0xD5,0x5B,0x00,0xF7,0x04,0x02,0x08,0xD7,0x21
,0x59,0xD3,0x6C,0xCF,0x47,0xFD,0x99,0x9A,0x8E,0x4B,0x7C,0x6D,0x1F,0x94,0xB6
,0xCA,0xD8,0xCC,0x6F,0x22,0x8E,0xBD,0x1C,0x93,0xE6,0xE0,0x8C,0xE5,0x18,0x81
,0xAB,0x22,0xC5,0xB8,0x2A,0x68,0x36,0xBA,0x6F,0x19,0x1E,0x4A,0x9D,0xC5,0xB0
,0x26,0x8A,0x96,0xE3,0x39,0x39,0x96,0xE5,0x62,0x98,0xFF,0x00,0xA4,0xFC,0xC5
,0x07,0x26,0xC9,0xFE,0x7A,0x76,0x3F,0xCC,0x74,0x47,0x48,0x29,0x9C,0x01,0x05
,0x05,0x2C,0x54,0xF0,0xAE,0x7E,0xF2,0x45,0xC2,0x8F,0xF6,0x50,0x60,0x9E,0xB9
,0xCE,0x4A,0xFC,0xC6,0x75,0xEE,0xB0,0xC7,0x18,0xF3,0x66,0x90,0xC8,0x57,0xAF
,0x39,0x08,0xF4,0xF7,0xEB,0xEC,0x07,0x43,0xDC,0x0F,0xD7,0x5E,0x91,0xBA,0xC9
,0x1A,0xC9,0x1B,0x2B,0x23,0x00,0x55,0x81,0xC8,0x20,0xFB,0xEB,0xE5,0xA1,0x47
,0x60,0xD2,0x8F,0x30,0x83,0x90,0x1B,0xB0,0x3F,0x30,0x3E,0x7F,0x5E,0xFA,0x23
,0x19,0xAA,0xA6,0x9F,0xD3,0x41,0x08,0x70,0x7F,0xD3,0xCB,0x95,0x8C,0x7D,0x47
,0xBB,0xF7,0xC8,0xC6,0x01,0xFC,0xC3,0x5F,0x31,0xDB,0x23,0x79,0x92,0xA2,0xBE
,0x46,0xAD,0x99,0x08,0x64,0xF3,0x07,0xDD,0xC6,0x47,0x50,0x51,0x3B,0x02,0x0E
,0x70,0xC7,0x2C,0x33,0x8E,0x47,0x59,0xFA,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A
,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x8A,0x6F,0x6F,0x10,0x36,0xDE,0xD4,0xA8
,0x86,0xDF,0x59,0x3C,0xF5,0xD7,0x9A,0x81,0xFF,0x00,0x25,0xB4,0x5B,0xA2,0x35
,0x15,0xB5,0x1F,0xEC,0xC4,0xBD,0x40,0xFF,0x00,0x13,0x61,0x7E,0xBA,0x09,0x5E
,0xA0,0x5B,0xCB,0xC5,0x0B,0x3D,0x9A,0xEC,0x76,0xDD,0x86,0x8A,0xB3,0x76,0x6E
,0x92,0x3D,0x36,0x7B,0x50,0x0E,0xF1,0xFD,0x67,0x93,0xF0,0x40,0xBD,0xB2,0x5C
,0xE7,0xAF,0x63,0xAC,0x48,0xF6,0xCF,0x88,0xFE,0x20,0x7A,0xF7,0x55,0xC2,0x4D
,0x8D,0xB7,0x9F,0xFF,0x00,0xA4,0x5A,0xAA,0x03,0xDC,0x6A,0x17,0xE5,0x3D,0x48
,0xF4,0xC4,0x0F,0xBA,0x45,0x93,0x82,0x41,0x7D,0x58,0x7B,0x3F,0x67,0xED,0x9D
,0xA1,0x65,0x16,0x7D,0xB5,0x65,0xA5,0xB6,0x51,0x67,0x2E,0x90,0xAE,0x1A,0x53
,0xF9,0x9D,0x8F,0xA9,0xDB,0xE6,0x58,0x92,0x74,0x5C,0x52,0xF5,0xFB,0x0F,0x7B
,0x6F,0xE9,0x59,0xFC,0x4C,0xBE,0x4C,0x68,0x7B,0xFF,0x00,0x65,0x36,0xDC,0xE6
,0x1A,0x54,0x1F,0x96,0xB2,0xB0,0xE0,0xBE,0x3A,0x72,0x55,0xEB,0xEE,0xA0,0xEA
,0x57,0x66,0xDB,0xD4,0x56,0x3B,0x28,0xB5,0xD8,0xED,0x54,0x54,0x16,0xA5,0xEF
,0x4B,0x6D,0xE5,0x49,0x40,0x4F,0xB7,0x39,0x80,0xF3,0xEB,0x1C,0xF5,0x1E,0x80
,0x15,0xFB,0x30,0x07,0x56,0x25,0xDA,0xE7,0x6B,0xA0,0x31,0xD1,0x3C,0x66,0xA6
,0xA4,0xA8,0x31,0x50,0xD3,0xC5,0xE6,0x48,0xC3,0xB0,0x3C,0x07,0x45,0x5C,0xF4
,0xE4,0xD8,0x51,0xEE,0x46,0xB1,0xC5,0x25,0xEE,0xEC,0x43,0xDC,0x26,0xFE,0x11
,0x4A,0x7B,0x53,0x52,0xBF,0x2A,0x86,0x07,0xD9,0xE6,0xEC,0x99,0x1D,0xC4,0x7D
,0x41,0x19,0x12,0x1D,0x15,0x03,0x96,0x9A,0x69,0x9E,0x1B,0x6C,0x90,0x8C,0x46
,0xA1,0xA9,0xE8,0x85,0x1A,0xFA,0x57,0x3D,0x1A,0x1A,0x05,0x3C,0x22,0x5C,0xF5
,0x59,0x6A,0x98,0x95,0x6E,0xE0,0x67,0x55,0xF7,0xDA,0x77,0x75,0xDD,0x36,0x3E
,0xD1,0x14,0xB6,0xEA,0xA9,0xE1,0xDD,0xB7,0xA8,0xDA,0x3A,0x4F,0x2A,0x61,0x3D
,0x7C,0x70,0x92,0x03,0xCA,0xD2,0x91,0x88,0x81,0x2C,0x11,0x62,0x85,0x54,0x79
,0x8C,0x38,0xB1,0x00,0x8D,0x5F,0x1B,0x8E,0xE9,0xB6,0xBC,0x3B,0xD9,0x97,0x2B
,0xFD,0x70,0x8A,0xDF,0x6B,0xA1,0x89,0xAA,0x2A,0x19,0x47,0xAA,0x46,0xED,0xDC
,0xF5,0x79,0x18,0xE0,0x64,0x9C,0xB1,0x23,0xAE,0xB9,0xD3,0xEC,0xD1,0xB7,0x6F
,0x1E,0x2E,0x78,0x91,0x71,0xF1,0xDF,0x7C,0xD2,0x81,0x46,0x2A,0x4A,0x6D,0xEA
,0x39,0x17,0x92,0xA7,0x97,0x95,0x56,0x04,0x8F,0xC3,0x1E,0x48,0x18,0xE8,0xD2
,0x16,0x6E,0xEA,0x34,0x16,0x57,0xD9,0x67,0xC2,0x6F,0xFE,0x1D,0x6C,0xF5,0xB8
,0xDE,0xA2,0x56,0xDD,0x17,0x38,0x94,0xD5,0x92,0x79,0x1A,0x58,0xB3,0xC9,0x69
,0xD4,0xFD,0x09,0x2C,0xE7,0xF9,0x9C,0x93,0xD7,0x03,0x57,0x26,0x9A,0x68,0x3C
,0xEA,0x18,0x84,0x21,0x4B,0x86,0x23,0xA7,0x00,0x0B,0x7E,0xD9,0xE9,0xFD,0x7A
,0x6B,0x0D,0x2D,0xE5,0xDD,0xD9,0xDD,0xA1,0x0E,0x30,0xFE,0x5B,0x9F,0x31,0xC7
,0x5E,0x8D,0x27,0x70,0x3A,0xE4,0x05,0xC6,0x0F,0x63,0x8D,0x6C,0x34,0xD0,0x79
,0xD3,0xC1,0x0D,0x3C,0x7E,0x5C,0x11,0x24,0x69,0x92,0xD8,0x51,0x80,0x49,0x39
,0x27,0xF5,0x27,0xAE,0x75,0xE9,0xA6,0x9A,0x06,0x9A,0x69,0xA0,0x69,0xAF,0x99
,0xA4,0x8E,0x18,0xDA,0x59,0x64,0x58,0xD1,0x7A,0xB3,0x31,0xC0,0x1F,0xBE,0xB4
,0x97,0x4B,0xBB,0x07,0x4A,0x68,0x5A,0x58,0x1E,0x60,0x7C,0xA4,0x58,0xF9,0x55
,0x4A,0x31,0xDD,0x23,0x3D,0x10,0x02,0x46,0x5E,0x4C,0x00,0x47,0x51,0x82,0x0E
,0x83,0x55,0xE3,0x75,0xCE,0x1B,0x47,0x84,0x7B,0xAE,0xE7,0x2C,0xAA,0x8D,0x49
,0x6A,0x9A,0x64,0xC9,0xEF,0x20,0x52,0x63,0x1F,0xBB,0x00,0x35,0xFC,0xB0,0xDE
,0x94,0xF3,0xD1,0x6E,0xCB,0xA5,0xBA,0xA2,0xA2,0x4A,0x86,0xA1,0xAA,0x7A,0x35
,0x77,0x90,0xB9,0xE1,0x09,0xF2,0xD1,0x41,0x3E,0xC1,0x51,0x40,0x1D,0x80,0x00
,0x0E,0x9A,0xEC,0x5F,0xB6,0x8F,0x88,0x14,0x56,0xD8,0xAD,0xFB,0x16,0xDF,0x2D
,0x14,0x13,0x89,0x3F,0x8B,0xDE,0x7C,0xE6,0x69,0xC3,0xF9,0x68,0xC6,0x9E,0x96
,0x66,0x53,0xC9,0xDA,0x56,0x00,0x10,0x0F,0xA5,0x42,0x9C,0x84,0xEA,0x38,0x8F
,0x41,0x36,0xD8,0x7B,0xC2,0x3B,0x60,0x86,0xDD,0x5C,0xAA,0x91,0xC9,0x1B,0xD0
,0x3D,0x5C,0x87,0x9A,0xC7,0x45,0x31,0x63,0x24,0x25,0x4A,0x48,0x54,0x73,0x6E
,0x62,0x48,0xD7,0x9A,0xF3,0x97,0x2B,0x2E,0x55,0x47,0x4C,0x78,0x55,0xE3,0xFD
,0x82,0xC3,0xB1,0x6D,0x96,0x7D,0xED,0x52,0xBF,0x1D,0x49,0x4E,0x91,0xC1,0x51
,0x45,0x53,0x15,0x52,0xD4,0xC2,0x17,0xEE,0xD9,0xB8,0xBF,0x28,0xE4,0xE3,0x80
,0xCB,0x20,0x52,0x08,0xEA,0x01,0x24,0x0E,0x31,0xD3,0x41,0xD6,0x3E,0x24,0x7D
,0xA6,0x65,0x9C,0xD4,0xD2,0x6D,0xC9,0x92,0xCF,0x4F,0x1F,0xA4,0x3A,0x2A,0x55
,0xD6,0xCE,0x4C,0x65,0x94,0xA1,0x04,0xD3,0xC4,0x99,0xC2,0xB3,0x17,0x91,0xD4
,0xB6,0x44,0x64,0x82,0x07,0x36,0xEE,0x6D,0xDD,0x7D,0xDC,0x15,0x33,0xCB,0x5D
,0x5F,0x50,0xE2,0xA0,0x71,0x98,0xBC,0xAC,0xEF,0x30,0xF4,0x1C,0x49,0x21,0xF5
,0x38,0xE4,0x8A,0xC1,0x49,0xE2,0xAD,0x92,0xAA,0xA0,0xE3,0x5A,0x0D,0x34,0x0D
,0x34,0xD6,0x5D,0x9E,0x82,0x5B,0x9D,0xC1,0x28,0xE1,0x96,0x9E,0x16,0x65,0x77
,0x32,0x4F,0x28,0x8D,0x11,0x51,0x4B,0x31,0x24,0xFC,0x95,0x4F,0x41,0x96,0x27
,0x01,0x41,0x24,0x02,0x18,0x9A,0xDB,0xD1,0x53,0x53,0xC6,0x0D,0x33,0xDB,0xAA
,0x6B,0xEE,0x6E,0xE5,0x44,0x0A,0x4F,0x08,0x97,0x89,0x07,0x2A,0xBE,0xA6,0x93
,0x24,0x10,0x32,0x02,0x94,0xC3,0x07,0xE4,0x55,0x7A,0x3B,0xC1,0x0F,0x06,0xAD
,0xED,0xB7,0xE8,0x2F,0x97,0x6E,0x48,0x6A,0xE9,0xD9,0xC3,0xC1,0x33,0xAC,0xD5
,0x31,0x4C,0xA3,0x2A,0x58,0x1F,0xBA,0x8C,0xAF,0x40,0x17,0x12,0x30,0x67,0xE6
,0xC1,0x5C,0xC4,0xB6,0x6C,0xFB,0x83,0xC3,0xDD,0x87,0x24,0x36,0x1A,0x53,0x43
,0x45,0x57,0x21,0x0B,0x15,0xB6,0xDB,0x4A,0x65,0xA8,0x91,0x8F,0x61,0xE5,0x44
,0xA5,0x89,0x3F,0x32,0x3A,0xEA,0xE2,0x6B,0x9D,0x28,0xAD,0xB6,0xB9,0xA9,0x54
,0xD6,0x4F,0x43,0x6A,0xA9,0x55,0x3C,0x16,0xE1,0x6A,0x96,0x61,0x4C,0xAC,0x4C
,0x84,0x62,0x64,0x48,0x00,0xEA,0xCD,0xC5,0x23,0x41,0xDC,0xE4,0xF7,0xD4,0xA6
,0xD9,0xB4,0x2C,0x97,0x0A,0x21,0x35,0x0D,0xCE,0xCF,0x71,0xC0,0xC1,0x78,0xED
,0x34,0x26,0x3C,0xFF,0x00,0xDD,0x46,0xAC,0x3F,0x67,0xCF,0xD7,0x56,0x4E,0xE0
,0xB9,0x5C,0x2A,0xEB,0x65,0x75,0xD9,0x5B,0xC2,0x44,0x9C,0x3C,0xFE,0x59,0xB4
,0xB8,0x72,0x9F,0x0D,0x24,0x27,0xD3,0xF8,0xBF,0x13,0xAF,0xB7,0x4E,0x43,0x51
,0x8B,0xC0,0xD9,0x3B,0x93,0x72,0x47,0x4C,0xD1,0x49,0x6B,0xBE,0x49,0x53,0x27
,0x49,0x21,0x92,0xDF,0x5D,0x87,0xAA,0xA7,0x19,0x05,0x82,0xB3,0x1F,0x2F,0xCC
,0xE9,0xD7,0x03,0x3A,0x08,0xE0,0xB6,0xDC,0x76,0xE4,0xCB,0xF0,0xAB,0x52,0x88
,0x3B,0x2D,0xAA,0x67,0x9D,0x71,0x9E,0xED,0x47,0x3B,0x31,0xC6,0x7B,0x98,0xE6
,0x53,0xF4,0xD7,0xCD,0x6C,0xF4,0xD7,0x99,0x1A,0xAA,0xBA,0x1A,0x58,0x61,0x85
,0x42,0x35,0xFE,0xDB,0x33,0xA4,0xF4,0xAF,0x9C,0x70,0x99,0x19,0x44,0x94,0xF9
,0xEB,0xFD,0xE6,0x53,0xDB,0x27,0xBE,0xA5,0x54,0xB6,0x9D,0xD3,0x62,0xB0,0xD1
,0x5D,0xEA,0x17,0xFB,0x41,0x6A,0xAA,0xA6,0x8A,0xA2,0x59,0xE9,0xA0,0xE3,0x59
,0x4F,0xC9,0x01,0x25,0xE3,0x5E,0x92,0xA8,0xC8,0xF5,0x27,0xAB,0x03,0xF0,0x9E
,0xFA,0xFC,0x9A,0x8A,0xDF,0x78,0x5A,0x7B,0xC5,0xBE,0xAC,0xC3,0x54,0x13,0x34
,0xD7,0x0A,0x37,0x01,0xC2,0x9F,0x6C,0xF5,0x0E,0x87,0xDD,0x18,0x15,0x3E,0xE3
,0x41,0xA9,0xAF,0xA4,0x96,0x9A,0x85,0xA3,0xDC,0x47,0xFB,0x43,0x67,0x3E,0xAF
,0x3D,0xA2,0x1E,0x64,0x03,0x1D,0x19,0xD1,0x06,0x1C,0x0E,0xFC,0xD4,0x72,0x5E
,0xF8,0xC6,0x58,0x6F,0xFC,0x22,0xA0,0xA5,0xB7,0x78,0x9D,0x0D,0xAA,0x6B,0xAD
,0x4D,0xC3,0x6E,0xEF,0x4B,0x45,0x45,0x99,0xD2,0x79,0xBC,0xD0,0x9E,0x86,0x96
,0x3E,0x32,0x1E,0xAC,0x0A,0xF9,0xAA,0xB9,0xCF,0xE2,0xEE,0x72,0x31,0xA1,0xB5
,0xAD,0xDE,0xD5,0x53,0x34,0x0B,0x47,0x1B,0x70,0x06,0x46,0xA2,0xA5,0x07,0xCB
,0xA8,0x8C,0x75,0x69,0x69,0x54,0xF5,0x47,0x5E,0xEF,0x4F,0x92,0x31,0xEA,0x8F
,0xDD,0x4E,0x25,0xD6,0x2A,0xAB,0x5D,0x3D,0xBB,0x75,0xEC,0xF6,0x8E,0xAE,0x96
,0x86,0xBE,0x0B,0xB2,0xD2,0x29,0xC8,0x06,0x37,0xE4,0xCD,0x0F,0xCB,0x92,0xF2
,0x0C,0x9D,0x8F,0x23,0x8C,0x1C,0x86,0x0E,0xAD,0xF0,0x0E,0xF9,0x35,0xE7,0xC3
,0x4A,0x1A,0x6A,0xE0,0x56,0xE9,0x64,0x77,0xB3,0x5C,0x94,0xF7,0x15,0x14,0xC7
,0xCA,0x63,0xFE,0xF0,0x0A,0xFF,0x00,0xEF,0x6A,0x7B,0xAA,0xAF,0xC3,0xCB,0x85
,0x1D,0x1F,0x8D,0x1B,0x9E,0xDB,0x44,0x57,0xF8,0x76,0xE5,0xB6,0x51,0xEE,0x5B
,0x7B,0xA7,0xE0,0x94,0x95,0xF2,0x67,0x61,0xF5,0x3C,0x60,0x63,0xFE,0xD6,0x7D
,0xF5,0x6A,0x6A,0x06,0x9A,0x69,0xA2,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0xD3,0x6E
,0xED,0xD5,0xB7,0xB6,0x9D,0xB8,0x57,0xEE,0x1B,0xAD,0x3D,0x04,0x2E,0xDC,0x22
,0x0E,0x73,0x24,0xCD,0xEC,0x91,0xA0,0xCB,0x3B,0x7F,0x85,0x41,0x3A,0x0D,0xCE
,0xA3,0xFB,0xCF,0x79,0xED,0xCD,0xA3,0x04,0x4F,0x7B,0xB8,0x2C,0x73,0xD4,0x1E
,0x34,0xB4,0x70,0xA3,0x4B,0x53,0x54,0xDF,0x96,0x28,0x50,0x17,0x73,0xFA,0x0E
,0x9E,0xF8,0xD4,0x62,0x7B,0x9F,0x88,0x1B,0xB6,0x15,0x96,0xDD,0x4A,0xFB,0x0B
,0x6F,0xCB,0x2C,0x71,0x0B,0x8D,0xCE,0x9C,0x4B,0x73,0x9C,0xC8,0xE1,0x17,0xCA
,0xA6,0xEA,0xB0,0xE5,0x98,0x0E,0x52,0xE4,0x8C,0xE7,0x80,0xD4,0xAF,0xC3,0x8D
,0x97,0xB3,0x2C,0x5F,0x11,0x75,0xB1,0x11,0x76,0xB9,0xCB,0x24,0x90,0x56,0x5E
,0x6A,0xEA,0x3E,0x2A,0xB2,0x69,0x23,0x72,0x8E,0x8F,0x31,0xC9,0x1C,0x59,0x4A
,0x94,0x5E,0x2A,0x08,0x23,0x03,0x1A,0x2E,0x23,0x11,0xD2,0x78,0x99,0xBF,0x3A
,0xB9,0x93,0xC3,0xCD,0xBC,0xFF,0x00,0xEC,0x4D,0x79,0xA8,0x4F,0xF3,0x8E,0x97
,0x23,0xFD,0xB7,0x1F,0xE1,0xD4,0xD7,0x62,0xEC,0x3D,0xAD,0xB2,0xA9,0xE5,0x4B
,0x05,0xB1,0x22,0xA9,0xA8,0x3C,0xAA,0xAB,0x66,0x73,0x35,0x55,0x53,0x7B,0xB4
,0xB3,0x39,0x2E,0xE7,0x3D,0x7A,0x9C,0x0F,0x60,0x35,0x26,0xD3,0x45,0x34,0xD3
,0x4D,0x07,0x95,0x35,0x35,0x3D,0x30,0x7F,0x87,0x82,0x38,0xBC,0xC6,0xE4,0xE5
,0x54,0x02,0xED,0xF3,0x27,0xDC,0xFD,0x4E,0xBD,0x74,0xD5,0x23,0xF6,0xB0,0xF1
,0x2A,0xB3,0x69,0xED,0x51,0xB5,0x76,0xC4,0x92,0x36,0xE9,0xBD,0xC7,0xE5,0xC3
,0xE4,0xF5,0x92,0x9A,0x16,0x61,0x1F,0x35,0x03,0xAF,0x98,0xEC,0xC2,0x38,0xC7
,0xE6,0x24,0xF6,0x43,0xA0,0x83,0xF8,0xB9,0x57,0x55,0xE3,0xCF,0x8A,0x94,0x9E
,0x1A,0x58,0xAA,0x9D,0x76,0xCD,0xB2,0x66,0x92,0xE1,0x53,0x11,0xF4,0xBF,0x96
,0xC5,0x26,0x9B,0x3D,0x88,0x53,0x98,0x62,0x1D,0x41,0x73,0x23,0xF6,0x8C,0x1D
,0x74,0xD5,0x8E,0xD5,0x6F,0xB1,0xD9,0xA8,0xEC,0xF6,0x9A,0x58,0xE9,0x28,0x28
,0xA1,0x58,0x29,0xE1,0x8C,0x61,0x63,0x45,0x18,0x00,0x7E,0xDA,0xAE,0xBE,0xCD
,0x1E,0x16,0xC3,0xE1,0x6F,0x87,0x70,0x50,0x55,0x71,0x9A,0xFD,0x5C,0x12,0x7B
,0xAD,0x46,0x79,0x66,0x40,0xB8,0x58,0x81,0xFC,0x91,0x8F,0x48,0xF6,0xCF,0x23
,0xFC,0xDA,0xB4,0x74,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40
,0xD6,0x0D,0xC6,0xE7,0x4F,0x47,0x4F,0x24,0xF2,0x4B,0x0A,0x43,0x17,0xF7,0xB3
,0xCD,0x20,0x48,0xA3,0xEB,0x8E,0xAC,0x7D,0xF3,0xD3,0x03,0x3D,0x70,0x0E,0x33
,0xA8,0xE5,0xE7,0x72,0xA1,0x79,0x16,0x92,0x48,0xEB,0x15,0x64,0x31,0x72,0x0E
,0x56,0x95,0x1F,0xF2,0xB3,0x80,0x5A,0x69,0x3D,0x2D,0xF7,0x51,0x83,0xD7,0xD2
,0xC0,0x74,0x6D,0x51,0xFE,0x28,0x78,0xD9,0x41,0x64,0xAA,0x10,0x5A,0x02,0xEE
,0x3B,0xEC,0x78,0xE0,0xED,0x3C,0x70,0x52,0xD0,0x86,0x21,0x03,0xB3,0x96,0xF2
,0xE9,0x53,0x93,0x05,0x0D,0x96,0x73,0xCB,0x89,0x91,0x08,0xC6,0x82,0xDD,0xDC
,0xDB,0xAD,0x28,0xA9,0x65,0xAF,0xA9,0xAB,0xFE,0x1F,0x4F,0x04,0x7E,0x6B,0xD6
,0x55,0x2A,0xC7,0x22,0x2E,0x08,0xE5,0x1C,0x72,0x10,0x90,0x29,0xCB,0x62,0x49
,0xCE,0x4E,0x0A,0x85,0x7C,0x8D,0x73,0xE6,0xEE,0xFB,0x44,0x5B,0x28,0x61,0xA8
,0x7B,0x05,0x3B,0x1A,0x1A,0x86,0x92,0x35,0xBC,0x54,0xA4,0xAD,0x4F,0x51,0x32
,0x29,0x24,0x20,0x60,0x26,0xAD,0x75,0x04,0x2F,0xDE,0x18,0xE2,0x05,0xE3,0x56
,0x54,0x52,0x1B,0x5C,0xF1,0xE2,0x37,0x89,0x77,0x5D,0xC9,0x72,0xF3,0xAF,0x37
,0x28,0x77,0x0B,0x28,0x12,0xC3,0x12,0xAC,0x91,0x5B,0xA9,0x64,0x74,0x6C,0x95
,0x85,0x82,0xB4,0xB2,0x29,0x31,0xFD,0xE4,0xB9,0xC9,0x56,0x0C,0xB2,0xA9,0x0C
,0x6B,0xAA,0xAA,0x9A,0x8A,0xB9,0xBC,0xEA,0xAA,0x89,0x67,0x93,0x82,0xA7,0x39
,0x1C,0xB3,0x71,0x55,0x0A,0xA3,0x27,0xD8,0x28,0x00,0x0F,0x60,0x00,0xD0,0x6E
,0x77,0x36,0xEC,0xBD,0x5F,0xCD,0x52,0x56,0xD6,0xCC,0xF4,0xF5,0x35,0xAD,0x5D
,0x2A,0xB9,0x52,0xF2,0xCC,0x57,0x88,0x79,0x1C,0x00,0x64,0x21,0x72,0x17,0x3D
,0x13,0x9B,0xF0,0x08,0x18,0x83,0xA1,0xD3,0x4D,0x03,0x4D,0x34,0xD0,0x34,0xD3
,0x4D,0x03,0x57,0x4F,0x80,0xFB,0x3D,0x2B,0xAB,0x29,0xA5,0xAB,0x05,0x0D,0x35
,0x55,0x0D,0x7D,0x64,0x6D,0x1E,0x0B,0x23,0xCE,0xAB,0x4D,0x11,0xE4,0x81,0x80
,0x60,0x5E,0x66,0xC3,0x32,0x3A,0x9A,0x73,0xDD,0x75,0x52,0x58,0x28,0xE9,0xEE
,0x17,0xDB,0x7D,0x05,0x65,0x67,0xC1,0x53,0x54,0xD5,0x47,0x0C,0xD5,0x3E,0x5F
,0x3F,0x25,0x19,0x80,0x67,0xE3,0x91,0xCB,0x00,0x93,0x8C,0x8C,0xE3,0xB8,0xD7
,0x4A,0xD7,0x4F,0x57,0xB5,0xFC,0x29,0x37,0x2B,0x75,0x1D,0x3D,0x3D,0xEA,0xE8
,0x94,0x37,0x54,0x82,0x28,0x82,0x20,0x96,0x4A,0xE5,0x74,0x89,0x50,0x60,0x05
,0x54,0xE1,0x18,0x51,0x80,0x02,0x81,0xD3,0x41,0xF5,0xE2,0x3F,0x89,0xD5,0xD6
,0xBB,0x1D,0x16,0xD3,0xDB,0x50,0xD5,0x24,0xE9,0x0C,0x54,0x75,0x15,0x34,0xC0
,0x07,0x5A,0x82,0xB8,0x5A,0x68,0x89,0xEC,0xC3,0xF9,0x98,0x02,0x57,0xB0,0xC3
,0x1C,0x89,0x7F,0xD8,0xD3,0xC3,0x04,0xB5,0xC3,0x71,0xDF,0xF7,0xB6,0x4A,0xDB
,0x9D,0x4D,0x44,0xB4,0xB4,0x33,0x93,0xCC,0x04,0x56,0x2B,0x2C,0xAA,0x4F,0x72
,0xEE,0x19,0x43,0x77,0x2A,0xB9,0xFE,0x63,0xAA,0x36,0x9E,0x91,0xE8,0xB7,0x3C
,0xD3,0xC4,0x56,0x67,0xDB,0xB6,0x59,0xAE,0x1C,0xF0,0x5D,0xDA,0xBA,0x52,0x23
,0x46,0x77,0x3D,0x59,0xBC,0xD9,0x12,0x41,0xDB,0xB7,0x6D,0x77,0x7E,0xCB,0xB2
,0x41,0xB6,0xF6,0x8D,0xA3,0x6F,0xD3,0x00,0x22,0xB7,0x51,0xC5,0x4C,0x31,0xEF
,0xC1,0x40,0x27,0xF5,0x24,0x13,0xFB,0xEB,0xAF,0x8E,0xBF,0x6C,0x5A,0x59,0x97
,0x4A,0x36,0xAA,0x85,0x5A,0x19,0x04,0x35,0x50,0xB7,0x38,0x25,0xC6,0x42,0xB7
,0xC8,0x8F,0x75,0x23,0xA1,0x1F,0x23,0xD3,0x07,0x04,0x6B,0x2E,0x16,0xCB,0x0E
,0xF2,0xB2,0xC9,0x6F,0xDC,0x36,0x5A,0x3A,0xD8,0xD1,0x8C,0x75,0x14,0x95,0x51
,0x89,0x3C,0xA9,0x30,0x33,0x82,0x47,0x4E,0x84,0x10,0xC3,0x19,0x04,0x11,0xDF
,0x5B,0xED,0x46,0xAF,0xD5,0x35,0x16,0xFD,0xC1,0x05,0x4C,0x34,0xE4,0xC9,0x2C
,0x7C,0x10,0x29,0xC0,0xAB,0x55,0xCB,0x34,0x3D,0x7A,0x09,0x54,0x65,0xD3,0x3F
,0x8B,0xD6,0xBD,0x3A,0xB6,0xBA,0xCB,0x10,0x84,0x5D,0x36,0x45,0xFF,0x00,0x63
,0xC2,0xB3,0xEC,0xA3,0x5B,0x7E,0xB2,0xC6,0x02,0x1B,0x0D,0x44,0xEA,0x66,0xA6
,0x41,0xEF,0x4D,0x33,0x9C,0x90,0xA0,0x00,0x22,0x72,0x7A,0x76,0x61,0x8C,0x18
,0x45,0xCE,0xCF,0x05,0xC6,0x8A,0x4D,0xDF,0xB2,0x69,0xE5,0x12,0x48,0xEC,0x6E
,0x56,0x77,0x4F,0x29,0xA4,0x91,0x4E,0x1F,0xD0,0x7F,0xBB,0xA8,0x53,0xDC,0x1C
,0x07,0xEC,0x7A,0x90,0xC3,0xA3,0xA8,0xAA,0x60,0xAC,0xA4,0x8A,0xAA,0x96,0x41
,0x2C,0x32,0xA8,0x64,0x61,0xEE,0x0E,0xAB,0xEF,0x14,0x6C,0x95,0xF6,0xB9,0x9F
,0x7B,0x6D,0xAA,0x37,0xA9,0xA8,0x8C,0x0F,0xE3,0x16,0xF8,0x87,0xAA,0xBA,0x05
,0x18,0xF3,0x10,0x7F,0xAF,0x8C,0x76,0xFC,0xEA,0x0A,0x9C,0xE1,0x31,0xCE,0xD4
,0xFB,0x86,0xA2,0xDF,0xB5,0x37,0x4D,0x3D,0x15,0xF2,0xD7,0x1D,0x4D,0x2C,0xCE
,0x63,0x73,0xCA,0x39,0x50,0x94,0x92,0x27,0x53,0xDC,0x7B,0xA3,0xA9,0x1F,0xA8
,0x23,0x51,0x7A,0x7B,0xA5,0x65,0x93,0x70,0x56,0x2D,0xCE,0x38,0x92,0x26,0x61
,0x2D,0x69,0x8D,0x78,0xC6,0xEA,0xC7,0x88,0xAD,0x45,0xEC,0xB9,0x62,0x16,0x64
,0x1D,0x03,0x15,0x71,0xD1,0x8E,0xA6,0x5B,0xC6,0x9E,0x86,0xDF,0x24,0x1B,0xF6
,0xC3,0x2C,0x73,0x6D,0xFB,0xC1,0x46,0xB8,0xF9,0x67,0x09,0x1B,0xBE,0x02,0x55
,0x01,0xEC,0x0F,0x45,0x93,0xB7,0xF2,0xB1,0xEC,0xD9,0xD6,0x6E,0x9B,0x63,0xD7
,0x52,0x25,0x55,0x2C,0x31,0x4B,0x5F,0x46,0x5A,0x4A,0x75,0x90,0x7A,0x25,0x05
,0x4A,0xC9,0x0B,0xFF,0x00,0x82,0x44,0x25,0x0F,0xEA,0x0F,0xB0,0xD7,0x26,0xDB
,0xBD,0xAD,0xBC,0x69,0xAC,0x34,0x96,0x6F,0x8A,0x8A,0xA8,0xDC,0xB6,0x25,0x71
,0xEB,0x0D,0x34,0x92,0x09,0xAC,0x15,0x47,0xCB,0x62,0x4A,0x82,0x00,0x84,0x94
,0xFC,0x44,0x7F,0xCD,0xC6,0x3B,0xEB,0xAB,0xA2,0x74,0x96,0x35,0x96,0x37,0x57
,0x47,0x01,0x95,0x94,0xE4,0x10,0x7B,0x10,0x75,0xC3,0x7B,0x17,0x72,0x4F,0xB6
,0xAE,0xF4,0x57,0x8B,0x69,0xF8,0xAA,0x7A,0x1A,0x6C,0x3C,0x53,0xC5,0xE6,0x9A
,0xDB,0x24,0xE7,0x8C,0xB1,0x3A,0x02,0x0B,0xC9,0x0B,0x21,0x52,0x33,0xF8,0xA2
,0xF7,0xE5,0x83,0x7F,0xFD,0x97,0x37,0x7C,0x37,0x0B,0x45,0xCF,0x63,0x3D,0x7C
,0x55,0xB2,0x6D,0xC9,0x15,0x68,0x2A,0x12,0x4E,0x62,0xA2,0xDF,0x26,0x4C,0x0C
,0x18,0xFE,0x2E,0x00,0x18,0xC9,0xFF,0x00,0x0A,0xFC,0xF5,0x05,0xCF,0xA6,0x9A
,0xC6,0xBA,0xDC,0x68,0x2D,0x36,0xE9,0xEE,0x37,0x4A,0xDA,0x7A,0x2A,0x3A,0x74
,0x2F,0x34,0xF5,0x12,0x08,0xE3,0x8D,0x7E,0x65,0x8F,0x40,0x34,0x46,0x4E,0xB5
,0xBB,0x96,0xFF,0x00,0x65,0xDB,0x56,0xA9,0x2E,0xB7,0xFB,0xA5,0x25,0xB6,0x8A
,0x3E,0x8D,0x35,0x44,0x81,0x06,0x7D,0x80,0xCF,0x72,0x7D,0x80,0xC9,0x3E,0xDA
,0xA3,0xFC,0x47,0xFB,0x41,0xD4,0x9B,0x7D,0x44,0x7E,0x1A,0x59,0x45,0xC9,0xC4
,0x6C,0x63,0xB9,0xDC,0x15,0xE3,0x82,0x4C,0x0F,0xF4,0x10,0x81,0xE6,0x4D,0xFE
,0xD6,0x15,0x3D,0xF2,0x46,0xA9,0x5D,0xD0,0xF5,0xD7,0xFB,0x55,0xB7,0x7E,0xDE
,0xAF,0x75,0xD7,0x5A,0xE8,0x1E,0x9A,0xBC,0x4D,0x52,0xE0,0x84,0x88,0xB2,0x99
,0x15,0x50,0x00,0x91,0xAF,0x06,0x63,0xC5,0x14,0x75,0x50,0x49,0x3A,0x2A,0xEF
,0xDE,0xDE,0x36,0x5D,0xEE,0x25,0xA8,0xB6,0x55,0x22,0x5A,0xA0,0x70,0x71,0x73
,0xB9,0xC2,0x5A,0x77,0x1F,0x9A,0x1A,0x5E,0xFF,0x00,0xBC,0x98,0x23,0xF2,0x1D
,0x6B,0x3E,0xCE,0x2D,0x1C,0x5E,0x3A,0x54,0xB5,0xFE,0xA2,0xAA,0xF5,0x73,0xBA
,0xD9,0xDE,0x5A,0x4B,0x8D,0xD3,0x8C,0xB3,0xC7,0x24,0x32,0x0E,0x69,0x11,0xC7
,0xDD,0x2B,0x24,0xC3,0x2A,0xA1,0x46,0x23,0x1A,0x8D,0x80,0x06,0x70,0x07,0x5D
,0x64,0xED,0x8A,0xC6,0xB5,0x78,0x9D,0xB2,0x6E,0xC8,0x78,0xF9,0x77,0xA8,0xE9
,0x64,0x3F,0x34,0xA9,0x47,0xA7,0xC7,0xFE,0x29,0x50,0xFE,0xC3,0x57,0x07,0x52
,0xEF,0xB7,0xE1,0x64,0xA7,0x6F,0x9D,0xDA,0xDC,0x3F,0xAD,0x6C,0x03,0xFF,0x00
,0x5D,0x47,0xBC,0x0F,0x1E,0x5D,0x9B,0x72,0x53,0x7B,0x43,0xBB,0x2E,0xE0,0x0F
,0x90,0x7A,0xB9,0x24,0xFF,0x00,0xDF,0xAD,0xCF,0x89,0x2F,0xC6,0xC3,0x44,0xBF
,0x9A,0xF7,0x6B,0x1F,0xFF,0x00,0xBA,0x03,0xFF,0x00,0xA6,0xB4,0xFE,0x0C,0xE4
,0x7F,0x6D,0x10,0xFF,0x00,0x2E,0xED,0xAE,0xC7,0xEF,0xC0,0xFF,0x00,0xEB,0xA8
,0xA9,0xF6,0x9A,0x69,0xA0,0x69,0xA6,0x9A,0x0D,0x2E,0xFA,0xDC,0xF6,0xAD,0x9B
,0xB4,0xEE,0x3B,0x96,0xF3,0x29,0x4A,0x3A,0x18,0x8C,0x8C,0x14,0x65,0xE4,0x6E
,0xCB,0x1A,0x8F,0x76,0x66,0x21,0x40,0xF7,0x24,0x6A,0x8B,0xFB,0x3A,0xED,0x8B
,0x8E,0xFB,0xDD,0xB5,0x5E,0x31,0xEF,0x18,0x95,0xDE,0x5A,0x86,0x7B,0x64,0x24
,0xF2,0x4F,0x30,0x02,0x81,0xD7,0xD8,0xC7,0x0A,0xE6,0x28,0xCF,0x5C,0xB1,0x9A
,0x4F,0xE7,0x53,0xA8,0xCF,0x8C,0xB7,0x69,0xFC,0x75,0xFB,0x42,0xDA,0xBC,0x1D
,0xB2,0x54,0xCB,0xFD,0x97,0xB0,0x4E,0x6A,0xEF,0xF5,0x10,0x36,0x04,0x8F,0x1F
,0x47,0x19,0x1D,0xB8,0xE7,0xCA,0x07,0xF3,0xBB,0x1F,0x61,0xAE,0xAC,0xB6,0xD1
,0x52,0x5B,0x6D,0xF4,0xD6,0xFA,0x0A,0x78,0xE9,0xA9,0x29,0xA2,0x58,0x60,0x86
,0x35,0xE2,0xB1,0xA2,0x8C,0x2A,0x81,0xEC,0x00,0x00,0x68,0x32,0x34,0xD3,0x4D
,0x03,0x4D,0x34,0xD0,0x34,0xD6,0x05,0x6D,0xD2,0x9E,0x9E,0x39,0x1F,0xCC,0x8C
,0x47,0x17,0xF7,0xD3,0xC8,0xE1,0x22,0x8B,0xAE,0x0E,0x58,0xFB,0xF4,0x23,0x03
,0x27,0x38,0xCE,0x33,0x9D,0x41,0xF7,0xBF,0x88,0xF4,0xB6,0xB8,0xA5,0x85,0x27
,0x7A,0x42,0x8A,0x0B,0x7A,0x07,0xC4,0x75,0x04,0x83,0xC5,0xFD,0x30,0x82,0x01
,0x20,0xC8,0x0B,0xB0,0x56,0x0B,0x1B,0x1C,0x1D,0x04,0xD6,0xF1,0x7A,0xA2,0xB6
,0x32,0xC3,0x27,0x99,0x51,0x56,0xEA,0x5E,0x3A,0x58,0x00,0x69,0x5C,0x0F,0xE6
,0xC1,0x20,0x2A,0xFB,0x17,0x62,0x14,0x7B,0x91,0xAA,0x67,0xC4,0xCF,0x16,0xAC
,0xF4,0x1C,0x29,0xEE,0xB7,0x2A,0x79,0x9A,0x7C,0x0A,0x7B,0x65,0x23,0x34,0x89
,0x36,0x7A,0x0C,0x85,0xC4,0x95,0x59,0xF6,0x55,0xE1,0x13,0x02,0xC3,0x32,0x95
,0xC6,0xA9,0x6F,0x14,0x7C,0x6D,0x8C,0x50,0x4A,0x94,0x13,0x45,0x15,0x14,0xF2
,0xA1,0x79,0x01,0xF3,0x7C,0xE0,0xCC,0x57,0xCC,0x44,0x77,0x12,0x56,0x15,0xE0
,0xE3,0xCD,0x76,0x11,0x8E,0x0B,0xC5,0xA2,0x70,0x10,0xF3,0x7E,0xE1,0xDE,0x57
,0x0B,0x85,0xC6,0xB2,0xAE,0x85,0xEA,0x28,0x9E,0xBE,0x30,0xB5,0xB3,0xB4,0xC2
,0x4A,0xAA,0x96,0x28,0xCB,0x27,0x39,0x82,0xA9,0xE0,0xFC,0xD8,0x18,0xD4,0x05
,0x2A,0x13,0x90,0x76,0x5E,0x64,0x2C,0xEF,0x19,0xBC,0x64,0xBA,0x6E,0x44,0x14
,0xC9,0x75,0xF2,0xAD,0xEE,0x8D,0x08,0xB6,0xD0,0x54,0x00,0xE6,0x13,0x18,0x29
,0xE7,0x4C,0x8B,0xC0,0x46,0x4B,0x20,0xF2,0x22,0xC0,0x3C,0x18,0x38,0x8D,0xC0
,0x26,0x9C,0xBD,0xDE,0x2A,0x2E,0x86,0x24,0x68,0x69,0x69,0x29,0x29,0xF9,0x8A
,0x6A,0x4A,0x58,0x82,0x45,0x08,0x66,0xC9,0x1E,0xEC,0xE7,0xB0,0xE6,0xE5,0x9C
,0x85,0x50,0x58,0xF1,0x1A,0xD6,0xE9,0xA0,0x69,0xA6,0x9A,0x06,0x9A,0x69,0xA0
,0x69,0xA6,0x9A,0x06,0x9A,0x69,0xA0,0xDD,0x6C,0xBF,0x24,0xDF,0x19,0x26,0xA7
,0x86,0x71,0x25,0x0D,0x64,0x68,0xB2,0xA0,0x65,0x0E,0xD4,0xD2,0xAA,0x36,0x0F
,0xBA,0xB1,0x56,0x07,0xB8,0x20,0x11,0xD4,0x6B,0xAB,0xFC,0x76,0xB6,0xD2,0xC6
,0xBB,0x6A,0x15,0xE5,0xC5,0xAA,0x56,0x22,0x8A,0xC5,0x71,0x1C,0x31,0x48,0xEA
,0x72,0x3A,0xF4,0x7F,0x2F,0xFA,0x6B,0x93,0x36,0x97,0xA6,0xFB,0x14,0x98,0x24
,0x45,0x14,0xD2,0xB6,0x3E,0x49,0x13,0xB1,0xFF,0x00,0x20,0x75,0xD6,0xFE,0x3C
,0xBB,0x0B,0xFE,0xD5,0x5C,0xFA,0x19,0x2B,0x0F,0xEF,0x88,0xB1,0xFE,0x59,0xD5
,0x49,0x54,0x9B,0x7A,0xD6,0xB7,0x0B,0xCE,0xF2,0xA9,0xA7,0x86,0x11,0x55,0x1D
,0xC6,0xCB,0x47,0x4E,0x59,0xCC,0x68,0x0C,0xB5,0x08,0x08,0x62,0xA0,0xFA,0x4F
,0x01,0x9E,0x87,0xB0,0x38,0xC8,0x1A,0xEE,0x9B,0x2D,0xC6,0x3B,0x9D,0x17,0x9C
,0x22,0x92,0x9E,0x64,0x6F,0x2E,0xA2,0x9E,0x4C,0x73,0x82,0x40,0x06,0x51,0xB1
,0xD3,0x3D,0x41,0x04,0x74,0x20,0x82,0x09,0x04,0x1D,0x71,0x2E,0xD8,0x93,0xE1
,0xE9,0x7C,0x4E,0xE2,0x40,0x96,0x92,0x5B,0x45,0xCD,0x47,0xBF,0x18,0x65,0x12
,0x39,0xFE,0x9F,0xF1,0xD7,0x66,0x5F,0x69,0xEA,0xE9,0xE6,0x5B,0xED,0xA2,0x2F
,0x3E,0xA6,0x34,0x0B,0x51,0x4C,0xA4,0x0F,0x8B,0x87,0xBF,0x10,0x7B,0x79,0x8B
,0x92,0x50,0x9E,0x9D,0x4A,0x9C,0x06,0xC8,0xED,0x4E,0x21,0x89,0x6E,0xF5,0x8D
,0x74,0xA1,0xA7,0xB9,0x50,0x4B,0x45,0x54,0xAC,0x63,0x93,0x1D,0x55,0xB8,0xB2
,0x30,0x39,0x56,0x52,0x3A,0x86,0x04,0x02,0x08,0xEA,0x08,0x07,0x4A,0x0A,0xDA
,0x7B,0x85,0xBE,0x2A,0xEA,0x29,0x04,0xD0,0xCA,0x9C,0x90,0xE0,0x8C,0xFD,0x08
,0x3D,0x41,0xF6,0x20,0xF5,0x07,0xA1,0xD7,0xB5,0x3C,0xA9,0x3C,0x42,0x48,0xCE
,0x41,0x24,0x1F,0x98,0x20,0xE0,0x83,0xF5,0x07,0xA6,0xBA,0x32,0x83,0xD9,0xEF
,0x12,0xD8,0xAF,0x95,0x96,0xFB,0xA3,0xAA,0xF9,0x64,0x49,0x59,0x85,0xE2,0xBC
,0x5C,0xE1,0x6B,0x50,0x7B,0x46,0xE7,0x22,0x50,0x3A,0x23,0x82,0xDF,0x84,0x97
,0x33,0xBD,0x6A,0x77,0x1D,0x9A,0x3B,0xA4,0x70,0xCF,0x13,0x24,0x37,0x0A,0x52
,0xCD,0x4B,0x39,0x5E,0x41,0x79,0x0C,0x32,0x30,0xFE,0x68,0xD8,0x74,0x65,0xF7
,0xE8,0x46,0x18,0x29,0x1A,0x7D,0xAD,0x73,0x6B,0x7D,0x4C,0x76,0x5A,0xC4,0x68
,0xA9,0xDD,0xCC,0x34,0x9C,0xDB,0x91,0xA7,0x94,0x0E,0x46,0x95,0xCF,0xBE,0x17
,0xD5,0x1B,0x7F,0x32,0x74,0x3D,0x40,0x2D,0x98,0xE1,0x7B,0x40,0x37,0x7D,0xBA
,0x9F,0x6B,0xF8,0x81,0x25,0x8E,0xB2,0x9A,0x37,0xDA,0x9B,0xC4,0x4B,0xE4,0xC6
,0xC3,0xEE,0xE9,0xEB,0x8A,0x93,0x34,0x38,0xEC,0x16,0x65,0xCB,0x81,0xF9,0xC3
,0xE3,0xBE,0xAB,0x8B,0x6D,0x15,0x45,0x0D,0x75,0xC7,0x6C,0x5D,0x67,0x79,0xE4
,0xB5,0x48,0xB1,0x47,0xCB,0xA7,0x9F,0x4C,0xC3,0x30,0xC8,0xC7,0xF9,0x89,0x19
,0x56,0xF6,0x2C,0xAC,0x31,0xAE,0x86,0xF1,0x7F,0x6A,0xB6,0xF1,0xD8,0x17,0x0B
,0x4D,0x33,0x79,0x57,0x14,0x02,0xAA,0xDB,0x30,0xE8,0x61,0xAA,0x88,0xF3,0x89
,0x81,0xF6,0xF5,0x0C,0x1F,0xA1,0x3A,0xA1,0x77,0x55,0xC1,0x2E,0xB6,0xCD,0x9F
,0xE2,0x34,0x30,0x18,0x4D,0x5A,0x8B,0x65,0xCE,0x30,0x3A,0xC7,0xE6,0x36,0x00
,0x3F,0xF6,0x75,0x0A,0x53,0xE9,0xCD,0xB5,0xCA,0xF1,0x92,0xD5,0x67,0x61,0x09
,0xBD,0x51,0x43,0x60,0xBB,0x19,0x11,0x4A,0xD2,0x53,0x4A,0xD7,0x28,0x23,0x52
,0x42,0xFC,0x3B,0x95,0x4A,0xD8,0x70,0x0F,0x55,0x19,0x8A,0x70,0xBD,0xBD,0x2F
,0xD3,0x59,0x95,0xD6,0x9A,0xEB,0x2F,0x88,0x1B,0x6E,0xEF,0xB5,0xEF,0xB5,0x36
,0x17,0x89,0x65,0x4A,0x51,0x1E,0x64,0xA7,0x49,0x06,0x24,0xF2,0xC4,0x5C,0x82
,0x84,0x75,0x12,0x16,0x51,0x80,0x78,0xF6,0xC9,0xD6,0xEB,0x7A,0xAC,0x31,0x5B
,0x62,0xBA,0x4F,0x17,0x9B,0x15,0xBE,0x61,0x34,0xE9,0xF9,0xE9,0xD8,0x18,0xEA
,0x17,0xF7,0x85,0xE4,0xFD,0xF1,0xAD,0x19,0x33,0xA6,0xC2,0x91,0x67,0x93,0xCD
,0xAF,0xDB,0x35,0x4C,0xB2,0x30,0xEA,0x58,0xD2,0xC9,0xD7,0xF7,0x78,0x87,0xEE
,0x1F,0x58,0x69,0x69,0xC5,0xE3,0x97,0x89,0x6D,0x74,0xA8,0xB1,0xFF,0x00,0x08
,0xDA,0x5F,0x15,0x05,0x3C,0x73,0xFC,0x59,0x35,0x1C,0x5D,0x5C,0xBA,0x82,0x21
,0x07,0x39,0x05,0x0E,0x47,0x3F,0x71,0xD7,0xAE,0xA3,0x3B,0x86,0x5D,0xCD,0xBC
,0x2A,0x52,0x7D,0xE7,0x7E,0x96,0xE5,0x14,0x6E,0x1E,0x2A,0x34,0x89,0x62,0xA5
,0x8D,0x86,0x70,0x56,0x11,0x95,0x24,0x7E,0x69,0x0C,0x87,0xE5,0xC7,0x5A,0xBB
,0x99,0x58,0x37,0x8D,0x92,0xB9,0x08,0x29,0x57,0x0C,0xF4,0x44,0x8F,0xE6,0x25
,0x44,0xC8,0x7F,0x61,0x13,0xFF,0x00,0xE2,0xD4,0x83,0x41,0xE5,0x4F,0x4F,0x0C
,0x0A,0x44,0x49,0x82,0xDF,0x89,0x89,0xCB,0x37,0xB7,0x52,0x7A,0x9F,0xDF,0x51
,0x0A,0x08,0x04,0xBE,0x16,0x5D,0x2D,0x61,0xD4,0x0A,0x78,0x6B,0xA8,0x54,0xB1
,0xE8,0xA2,0x37,0x92,0x35,0xCF,0xEC,0x17,0x52,0xDF,0x88,0xE6,0x40,0xA7,0x4F
,0x3B,0xAF,0x57,0x07,0x08,0x3F,0x7F,0x7F,0xDB,0x3F,0x5C,0x6A,0x23,0x4B,0x1C
,0x14,0xBB,0x43,0x74,0x55,0x54,0x81,0x32,0xAD,0x45,0xC2,0x47,0x5E,0xBC,0x7F
,0x9B,0x20,0x29,0x27,0xE4,0x7F,0xA9,0xD0,0x49,0xAD,0x75,0xCB,0x51,0x6B,0xA2
,0xA8,0x2B,0x23,0x35,0x45,0x3A,0x49,0xE9,0x42,0x47,0x55,0x07,0xBE,0x30,0x3B
,0xEB,0x17,0x72,0x54,0xD4,0x41,0x45,0x4B,0x57,0x14,0x2F,0x1B,0x52,0xDC,0xA8
,0x6A,0x39,0x12,0x32,0x38,0x55,0x44,0xD9,0x18,0xCF,0xCB,0x59,0x3B,0x66,0x9D
,0xE9,0x36,0xDD,0xB2,0x96,0x40,0x43,0xC3,0x47,0x14,0x6C,0x0F,0x7C,0x84,0x00
,0xFF,0x00,0xC3,0x58,0xFB,0xC0,0xB7,0xF0,0x4F,0x2D,0x33,0xCE,0x6A,0xAA,0x68
,0x57,0x1F,0x37,0x9E,0x35,0x1F,0xF1,0xD0,0x75,0xE7,0x88,0xA3,0x95,0xBE,0xD1
,0x1F,0xE6,0xBE,0x50,0x7F,0xE5,0xA8,0x46,0xFF,0x00,0xDB,0xAD,0x3F,0x83,0x0D
,0xC9,0xB7,0xB1,0xCE,0x7F,0xF9,0xB6,0xB8,0x7F,0x41,0x18,0xD6,0xEF,0x7D,0x02
,0xF2,0x6D,0xE8,0xC7,0xF3,0x5E,0xA0,0x3F,0xF8,0x55,0xDB,0xFF,0x00,0x6E,0xA3
,0xFE,0x06,0x12,0xF4,0xDB,0xCE,0x5F,0xCF,0xBC,0x6E,0x7F,0xF9,0x64,0x09,0xFF
,0x00,0xB7,0x51,0x56,0x26,0x9A,0x69,0xA0,0x6A,0xA6,0xFB,0x50,0x78,0x9B,0x1F
,0x87,0x5B,0x04,0xC7,0x45,0x57,0x1C,0x17,0xEB,0xC1,0x7A,0x6B,0x7B,0x1E,0xBE
,0x40,0x0B,0x99,0x6A,0x08,0xF7,0x11,0xAF,0x5C,0x7B,0xB1,0x51,0x83,0x9D,0x5A
,0x75,0xD5,0x54,0xD4,0x34,0x53,0xD6,0xD6,0x4F,0x1C,0x14,0xD4,0xF1,0xB4,0xB3
,0x4B,0x23,0x61,0x51,0x14,0x65,0x98,0x9F,0x60,0x00,0x27,0x5C,0x87,0xB1,0xAD
,0x95,0x1F,0x68,0x5F,0xB4,0x28,0xDF,0x97,0x18,0x24,0x7D,0xA7,0x60,0x94,0x35
,0x3C,0x72,0x7F,0x77,0xC1,0x0F,0x2A,0x68,0x31,0xD8,0xBB,0x9F,0xBF,0x90,0x75
,0xC0,0x28,0xA7,0x1D,0xB4,0x16,0xE7,0xD9,0x3F,0xC3,0x38,0xF6,0x2E,0xC8,0x92
,0xF1,0x5D,0x44,0xD4,0xF7,0xCB,0xFF,0x00,0x0A,0x8A,0x95,0x97,0xAC,0xB0,0x42
,0x07,0xDD,0x44,0xC7,0xF3,0xE0,0xF3,0x7F,0x9B,0xBB,0x7C,0x86,0xAE,0x6D,0x34
,0xD0,0x34,0xD7,0xCC,0xB2,0x24,0x51,0xB4,0x92,0x3A,0xA2,0x28,0xCB,0x33,0x1C
,0x00,0x3E,0xA7,0x51,0xFD,0xC5,0xB8,0x23,0xA2,0x89,0x50,0xCE,0x68,0xFC,0xD0
,0xDE,0x59,0x31,0x17,0xA9,0x97,0x18,0xCF,0x95,0x0F,0x7E,0x99,0x39,0x77,0x1C
,0x57,0x00,0x90,0x54,0xE7,0x41,0xB8,0xAF,0xAF,0xA7,0xA3,0x28,0x92,0x33,0x3C
,0xD2,0x7F,0x75,0x0C,0x6B,0xCA,0x49,0x3A,0x81,0xD1,0x47,0xB0,0x24,0x64,0xF6
,0x19,0xC9,0x20,0x75,0xD4,0x4B,0x73,0xEF,0x2A,0x3B,0x6A,0xB8,0xAD,0x96,0x33
,0x2A,0xB8,0x43,0x47,0x14,0x84,0x84,0x7F,0xC4,0x12,0x57,0x50,0x4B,0x48,0x46
,0x0F,0x93,0x18,0x66,0x39,0x3D,0x1D,0x7A,0x8A,0xE3,0x79,0xF8,0x8E,0xB4,0x94
,0x52,0xCD,0x4B,0x50,0x96,0xEA,0x49,0xD8,0x2B,0x54,0x99,0x5A,0x49,0x6A,0x18
,0xE4,0xAA,0xF9,0x88,0x79,0xCC,0x4F,0x2F,0x4C,0x70,0x10,0xB8,0x07,0x84,0xAF
,0x83,0x1E,0xB9,0x77,0xC4,0xDF,0x17,0x23,0xA8,0x89,0xA0,0xB4,0x55,0x65,0x5E
,0x36,0x8D,0x56,0x36,0x53,0x28,0x46,0x50,0xEB,0x96,0x41,0xE5,0x45,0x19,0x66
,0x04,0xC7,0x1E,0x49,0x31,0x92,0x56,0x09,0x3A,0x90,0xBB,0xBC,0x4A,0xF1,0xB2
,0x43,0x23,0xD3,0xDA,0xDB,0xCC,0x9A,0x9D,0x03,0x06,0x47,0x48,0x96,0x9D,0x09
,0x08,0xAC,0x5F,0x97,0x97,0x4C,0x84,0xB0,0x50,0xC1,0xCB,0xB0,0x7C,0x09,0x91
,0x87,0x97,0xAE,0x5F,0xDE,0xBE,0x24,0xDC,0x2E,0x55,0x68,0x29,0xAA,0x22,0xAB
,0x54,0x66,0x62,0xB2,0xD3,0x03,0x4C,0xAF,0xE6,0x82,0x19,0x23,0x6E,0x93,0x06
,0x08,0x0B,0x34,0xA9,0x86,0x2E,0x49,0x8C,0x30,0xE6,0xD0,0xDD,0xC1,0x7C,0xAD
,0xBD,0x4C,0xAD,0x52,0x22,0x8A,0x08,0xDD,0xDA,0x1A,0x78,0x57,0x8A,0x47,0xCD
,0xB2,0xC7,0x27,0x2C,0xEC,0x7A,0x02,0xEE,0x59,0xC8,0x55,0x0C,0xC7,0x03,0x5A
,0xBD,0x07,0xDC,0xF2,0xCB,0x51,0x3C,0x93,0xCF,0x2B,0xCB,0x2C,0x8C,0x5D,0xDD
,0xD8,0xB3,0x33,0x13,0x92,0x49,0x3D,0xC9,0x3E,0xFA,0xF8,0xD3,0x4D,0x03,0x4D
,0x34,0xD0,0x34,0xD3,0x4D,0x03,0x4D,0x34,0xD0,0x34,0xD3,0x4D,0x03,0x4D,0x34
,0xD0,0x6E,0x76,0x5B,0x56,0x1D,0xCB,0x4B,0x49,0x6F,0x82,0x19,0xEA,0xAE,0x02
,0x4B,0x74,0x69,0x2A,0x92,0x33,0x53,0x1B,0x41,0x90,0x01,0x1E,0xA0,0x24,0xC8
,0xFA,0x81,0x90,0x47,0x4D,0x75,0x07,0x89,0x75,0xF1,0xDD,0xB6,0x5F,0x87,0x5B
,0x98,0xB6,0x12,0x79,0x51,0x18,0xE7,0xB3,0x4D,0x4A,0xFD,0x0F,0xFB,0xEA,0x07
,0xEB,0xAE,0x46,0xD7,0x56,0xED,0xDA,0xAA,0x0D,0xC7,0xE0,0x3E,0xE1,0xA1,0xA5
,0xA7,0x89,0x64,0xB3,0x16,0xB9,0xDB,0x60,0x86,0x61,0x50,0x60,0x0C,0x3E,0x2E
,0x9E,0x30,0xC0,0x0E,0x45,0x19,0x9A,0x13,0xD0,0x1C,0xC6,0xC3,0x00,0xE4,0x00
,0x84,0xD8,0xA9,0xA7,0x1E,0x33,0x4F,0x6C,0x83,0xEF,0x13,0x71,0xD9,0x27,0xA5
,0x9A,0x06,0xE8,0x27,0x1E,0x5B,0x03,0x18,0x63,0xD1,0x5B,0x11,0x82,0x0F,0xCF
,0x00,0xE3,0x39,0xD7,0x60,0xF8,0x2F,0x7F,0xFE,0xD2,0xF8,0x5D,0x60,0xBA,0x49
,0x91,0x55,0xF0,0x8B,0x05,0x62,0x37,0x46,0x4A,0x88,0xBE,0xEE,0x55,0x23,0xD8
,0xF3,0x56,0xE8,0x75,0xC6,0x1B,0xAA,0xBE,0xAA,0x96,0x2D,0xBD,0xBE,0xED,0xB8
,0x26,0xD5,0x51,0x1D,0x58,0x8C,0x37,0xE2,0x8D,0xCA,0x9C,0x13,0xF5,0xC2,0xA9
,0x18,0xFE,0x66,0xD5,0xED,0xE1,0x5E,0xE3,0x8F,0x68,0x6F,0xFB,0xC3,0x0A,0xE6
,0x9B,0x69,0xEE,0x88,0x96,0xFF,0x00,0x45,0x94,0xFE,0xE4,0xBE,0x05,0x44,0x80
,0xFC,0x94,0xB2,0x33,0xAE,0x3A,0x23,0x72,0xC8,0xE0,0xD9,0xED,0x49,0xE1,0xCE
,0xD0,0xB9,0xEF,0x33,0xA6,0xD8,0xA9,0x6B,0xBE,0x0A,0xDA,0x27,0x90,0x7F,0x10
,0x00,0x74,0xA6,0x63,0xD3,0xE2,0x3E,0x8B,0x9C,0x07,0xF6,0x03,0x0F,0xD3,0x0E
,0x4E,0x45,0xE5,0xE5,0xB5,0x4C,0x6F,0x34,0xF1,0xB4,0xD4,0xA7,0x1F,0x1F,0x0C
,0x63,0x2D,0xC0,0x0C,0x79,0xCA,0x07,0x76,0x51,0xDC,0x0E,0xAC,0xA3,0xA6,0x4A
,0xA8,0x3B,0x82,0x12,0x58,0x88,0x21,0x5E,0x37,0x5E,0xA3,0xB8,0x60,0x7F,0xE2
,0x35,0x1F,0xB4,0xA9,0xDB,0xB5,0xB1,0x58,0xE4,0x27,0xF8,0x5C,0xC7,0x8D,0xB6
,0x42,0x73,0xE4,0x9C,0x67,0xE1,0x9B,0xE8,0x00,0x25,0x0F,0xC8,0x15,0xE8,0x54
,0x72,0xE8,0xCA,0x41,0x0C,0x91,0xCD,0x0A,0x4D,0x0C,0x89,0x24,0x6E,0xA1,0x91
,0xD4,0xE4,0x30,0x3D,0x41,0x07,0xDC,0x6B,0x47,0xBB,0x2D,0x0B,0x59,0x03,0xD5
,0x43,0x4E,0x67,0x90,0x20,0x49,0xE0,0x56,0xE2,0xD5,0x11,0x83,0xC8,0x05,0x6C
,0x8E,0x32,0x23,0x7A,0xA3,0x6C,0x8C,0x37,0xBA,0xF2,0x24,0x79,0xC1,0xFF,0x00
,0xCB,0x57,0x24,0xA4,0x63,0x8B,0x2D,0x6C,0xBC,0x69,0xC9,0xED,0x47,0x33,0x1F
,0xEE,0xBE,0x91,0xB9,0x3E,0x9F,0xCA,0xC7,0x8F,0x66,0x40,0x24,0x5A,0x76,0x35
,0x3B,0x62,0xBA,0x4A,0xBB,0x74,0x62,0x69,0xC5,0x4B,0xAA,0xE5,0x2A,0x02,0xF1
,0xF3,0xD3,0xD9,0xC8,0xE9,0xC5,0xBD,0x99,0x70,0x30,0xC0,0xF4,0x00,0x8D,0x52
,0x77,0x9D,0xB8,0x2A,0x26,0xF1,0x2B,0x61,0x21,0xE2,0xB2,0xD4,0x1B,0x85,0x07
,0xB7,0x95,0xF1,0x51,0xF9,0xAA,0x47,0xC8,0x2D,0x42,0x4A,0x47,0xE9,0xAB,0x66
,0xF3,0x23,0x6D,0x5B,0x84,0x97,0xB4,0x04,0xD9,0x2A,0x1B,0x37,0x34,0x03,0x3F
,0x08,0xE7,0xFE,0x92,0xBF,0x24,0xFF,0x00,0x58,0x3F,0x47,0xE9,0x87,0xE5,0x12
,0xDD,0xBC,0x21,0xF1,0xBE,0x9E,0x54,0x20,0x8B,0x8E,0xDA,0x3D,0x57,0xB3,0x79
,0x15,0x3D,0x0F,0xF4,0xA9,0xD6,0x2F,0xD3,0x55,0xED,0x50,0xD2,0xD4,0x41,0x7D
,0xD9,0xB1,0x4B,0x59,0xC6,0x28,0xEE,0x14,0x41,0x65,0x56,0x38,0xE2,0x64,0x4C
,0x32,0xFE,0xB9,0x24,0x6B,0x4D,0xB3,0xA7,0x92,0xE1,0x47,0x4E,0x56,0x25,0xE3
,0x74,0xB3,0xD2,0xD4,0xD5,0xCA,0xFD,0x73,0x3A,0x87,0xA5,0x94,0x71,0xF9,0x9F
,0x87,0x04,0xFD,0x4E,0x7A,0xF6,0xD6,0xE7,0x6D,0xD2,0x45,0x4F,0x3D,0xE2,0x3E
,0x39,0x78,0x2F,0x35,0xF1,0xA9,0x62,0x4F,0x15,0xF8,0x87,0x20,0x2E,0x7F,0x08
,0xC1,0x1D,0x06,0xB5,0x5B,0x32,0x55,0xF8,0xA5,0xA7,0x51,0x81,0x1C,0xF7,0x48
,0x40,0xF9,0x04,0xAD,0xE4,0x07,0xFF,0x00,0x78,0xEB,0x8B,0x6C,0x6F,0x22,0xAA
,0x0D,0xB5,0xB6,0x67,0xA8,0xAC,0xF3,0x64,0xB7,0x57,0xD3,0xC2,0x78,0xA6,0x39
,0xE5,0x8D,0x31,0xE5,0x92,0x49,0x3E,0xB3,0xD7,0x23,0xF4,0xD4,0xC6,0x48,0x16
,0x56,0xCC,0xCC,0x5D,0x3D,0xA3,0xFE,0x5F,0xDC,0x7B,0xFE,0xFD,0x3E,0x9A,0x85
,0x54,0x54,0xA7,0xF6,0x56,0x9A,0x19,0x25,0x41,0x24,0xBB,0x97,0x84,0x6A,0x58
,0x65,0xB8,0xDC,0xC9,0xC0,0x1E,0xF8,0x0B,0xFD,0x06,0xA6,0x72,0x35,0x44,0x8C
,0x56,0x30,0x21,0x8C,0x77,0x91,0xFA,0x9F,0xD8,0x7F,0x5E,0xA7,0xE5,0xD8,0xE8
,0x3F,0x6B,0x6A,0xE0,0xA2,0x80,0xCB,0x3B,0x30,0x00,0x12,0x15,0x10,0xB3,0x36
,0x3E,0x4A,0x32,0x4F,0xED,0xA8,0x4D,0x18,0x9E,0xB3,0x65,0x50,0x42,0x1A,0x34
,0x17,0xCA,0xE3,0x2F,0x96,0xCB,0x96,0x92,0x39,0xAA,0x1A,0x67,0x1F,0x4C,0x43
,0xCB,0xE7,0xFB,0x7B,0xE4,0x78,0x87,0x7A,0x8A,0xDB,0x61,0x6A,0x3A,0x44,0x92
,0x4A,0x9B,0xA1,0x34,0x91,0xD4,0x16,0x03,0x8F,0x25,0x3C,0xA4,0xC9,0x20,0x90
,0xA0,0x96,0xC8,0xF4,0x8F,0x98,0xD7,0xEE,0xCE,0x81,0x6E,0xF7,0x35,0xBD,0xC4
,0x85,0x68,0x68,0x29,0xCD,0x0D,0xB1,0x9D,0x83,0x2C,0x9D,0x83,0xCA,0x14,0x7A
,0x40,0xF4,0x84,0x1C,0x7B,0x80,0xD9,0x27,0xA6,0x82,0x61,0x22,0xC8,0x4E,0x63
,0x97,0x89,0xC7,0x40,0xCB,0x95,0xFF,0x00,0xD0,0xFF,0x00,0x9E,0xB1,0xE9,0x69
,0xAA,0x6E,0x3B,0xE7,0x65,0x59,0x08,0x8A,0x6F,0x8E,0xDC,0x54,0x8D,0x22,0xAA
,0x95,0x3E,0x5C,0x2F,0xE7,0xB9,0xC6,0x4E,0x46,0x22,0xC7,0xEF,0xAF,0x51,0x2C
,0xA0,0xB2,0xF1,0x8E,0x66,0x41,0xEA,0x08,0xD8,0x6C,0xFE,0x87,0xB7,0xEE,0x75
,0x2F,0xFB,0x33,0x5B,0x63,0xDC,0xFE,0x2E,0xDC,0x37,0x14,0xB4,0x52,0xFC,0x36
,0xD6,0xA4,0x34,0xB4,0xF2,0xB1,0x1C,0x45,0x65,0x47,0xF7,0x80,0x60,0x90,0xC5
,0x62,0x50,0x3B,0xF4,0xF3,0x0E,0x7B,0x8D,0x24,0x74,0x46,0xE5,0x8C,0x49,0x76
,0xDB,0x40,0xF6,0x5B,0xA3,0x37,0xF4,0xA4,0xA8,0xC7,0xF9,0xE3,0x51,0x3F,0xB3
,0xDB,0xF9,0xFB,0x4E,0xF9,0x58,0x3B,0x54,0xEE,0xBB,0xCC,0xA3,0xF4,0x35,0xD3
,0x01,0xFF,0x00,0x0D,0x4C,0x6F,0x8B,0xCA,0xED,0x61,0x03,0xA9,0x5A,0xE7,0x73
,0xFA,0x7C,0x34,0xE3,0x3F,0xD5,0x87,0xF5,0xD4,0x27,0xEC,0xC2,0x44,0x9E,0x0C
,0x5B,0x2A,0xBD,0xEA,0xEB,0x6E,0x35,0x24,0xFC,0xFC,0xCA,0xE9,0xD8,0x7F,0x91
,0x1A,0x8A,0xB3,0x34,0xD3,0x58,0x3B,0x86,0xEF,0x41,0x60,0xB0,0xD7,0xDF,0x2E
,0xB3,0x88,0x28,0x68,0x29,0xDE,0xA6,0xA2,0x43,0xFC,0xA8,0x8A,0x58,0x9F,0xAF
,0x41,0xDB,0x41,0xCE,0x5F,0x6D,0x9D,0xF7,0x71,0xA9,0x8E,0xD3,0xE0,0xBE,0xCF
,0x26,0x6B,0xFE,0xE7,0x92,0x35,0xAB,0x58,0xCF,0x58,0xE0,0x66,0xC2,0xC6,0x4F
,0xB7,0x36,0x19,0x27,0xD9,0x14,0xE7,0xA3,0x6A,0xF0,0xF0,0x93,0x64,0x5B,0x3C
,0x3B,0xF0,0xFE,0xD5,0xB4,0xED,0x81,0x59,0x28,0xE2,0x1E,0x7C,0xC1,0x70,0x6A
,0x26,0x3D,0x64,0x90,0xFD,0x59,0xB3,0xFA,0x0C,0x0F,0x6D,0x73,0xE7,0xD9,0x17
,0x6E,0x5C,0x77,0xFF,0x00,0x89,0x3B,0x8F,0xC7,0xFD,0xD3,0x4E,0xCA,0xD5,0xB5
,0x12,0xC1,0x64,0x8E,0x4E,0xBC,0x17,0xF0,0x33,0x2F,0xD1,0x10,0x08,0x81,0xF7
,0xF5,0xFC,0xB5,0xD4,0x75,0xB7,0x2A,0x2A,0x3A,0x88,0x69,0xA7,0xA8,0x51,0x51
,0x36,0x7C,0xA8,0x17,0x2D,0x23,0x81,0xDC,0x85,0x1D,0x48,0x19,0xEA,0x71,0x81
,0xEF,0xA0,0xCB,0xD7,0x85,0x7D,0x65,0x25,0x05,0x1C,0x95,0x95,0xD5,0x30,0xD2
,0xD3,0x44,0x39,0x49,0x2C,0xAE,0x11,0x14,0x7C,0xC9,0x3D,0x06,0xA3,0xDB,0x87
,0x79,0x51,0x50,0x72,0x82,0x80,0x2D,0x6D,0x48,0x63,0x1B,0x1E,0x47,0xCA,0x8D
,0xC7,0xE2,0x52,0xCA,0x09,0x77,0x00,0x1C,0xA4,0x61,0x98,0x7F,0x30,0x51,0xEA
,0xD5,0x0D,0xE2,0x67,0x8B,0x54,0x74,0x15,0x98,0x96,0xBC,0x5C,0xEE,0xC8,0x15
,0xE3,0x45,0x95,0x23,0x8E,0x98,0x31,0x0A,0xAF,0xCB,0x98,0x8E,0x9D,0x09,0x70
,0x3C,0xCF,0x33,0x91,0x0E,0x31,0x39,0xEB,0x1E,0x82,0xD4,0xDE,0x1E,0x20,0x14
,0x2F,0x1D,0xBA,0x36,0xA4,0x58,0xC8,0xCD,0x45,0x44,0x7F,0x7C,0x32,0xA4,0x8E
,0x11,0x30,0xFB,0xAC,0x8C,0x90,0x64,0x1C,0xC8,0x07,0x84,0x52,0x8D,0x73,0xA7
,0x88,0xFE,0x30,0xDA,0x2D,0xF4,0xF3,0x4D,0x1D,0x42,0x57,0x99,0x9E,0x31,0x24
,0x8C,0xE2,0x65,0x98,0x17,0x60,0x1C,0x29,0x70,0xF5,0x61,0x4A,0xC9,0x86,0xE4
,0xB1,0x29,0x51,0xC1,0xA0,0x7C,0x46,0x69,0x0D,0xFD,0xE2,0xD5,0xCE,0xFF,0x00
,0xCE,0x34,0x72,0xDC,0x64,0x06,0x34,0xE2,0xBF,0x08,0x8A,0x24,0x66,0x60,0x62
,0x65,0xFB,0xEE,0x5C,0x63,0x62,0x5C,0x2A,0x12,0x5F,0x31,0x16,0x21,0xF5,0x5A
,0x55,0xD4,0xD4,0xD6,0x54,0x35,0x45,0x5D,0x44,0xB5,0x13,0x36,0x39,0x49,0x2B
,0x96,0x63,0x81,0x81,0x92,0x7A,0xF6,0x00,0x68,0x25,0xDB,0xDB,0xC4,0x6D,0xC1
,0xB9,0x2E,0x52,0xD5,0x0A,0xDA,0xCA,0x65,0x96,0x1F,0x2A,0x42,0xD3,0x86,0x95
,0x83,0x23,0x2C,0xAB,0xCD,0x55,0x78,0xC6,0xFC,0xDF,0x31,0xA0,0x55,0x23,0x8F
,0x30,0xEC,0x0B,0x98,0x5E,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A
,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69
,0xA6,0x81,0xAB,0x73,0xC0,0x9D,0xD1,0x2D,0xB1,0x25,0xA6,0x56,0xA8,0x75,0x0E
,0x94,0xF5,0x80,0xC8,0xEE,0xAB,0x4B,0x24,0xC8,0x23,0x70,0xB8,0xE3,0x1A,0xC7
,0x3B,0x10,0x7D,0x5E,0xA3,0x59,0x90,0x3D,0x24,0x9A,0x8F,0x59,0x96,0x6B,0x8C
,0xD6,0xBB,0x82,0x56,0x41,0x1C,0x12,0x30,0x47,0x8D,0x92,0x68,0xC3,0xA3,0xA3
,0xA1,0x47,0x52,0x3D,0xB2,0xAC,0x46,0x41,0x0C,0x33,0x95,0x20,0x80,0x40,0x5F
,0x77,0x4B,0x61,0xA1,0xBE,0xDE,0x36,0xD5,0xCA,0x38,0x9E,0x82,0xAE,0x7A,0xB9
,0xE8,0x23,0x88,0x30,0x26,0x9F,0xE2,0xA6,0x46,0x42,0x47,0x62,0xA5,0x47,0xCB
,0xA3,0x2F,0xE9,0xAF,0x6F,0x08,0xEB,0x6E,0x69,0x4E,0x76,0x04,0x95,0x10,0xC7
,0xB9,0x6C,0x13,0xFF,0x00,0x15,0xDB,0x13,0xC8,0x4F,0x09,0xC6,0x09,0x92,0x9C
,0x92,0x06,0x51,0x94,0xBA,0x91,0xF2,0x66,0xFC,0x83,0x5B,0x7D,0xA9,0x46,0x9B
,0xE3,0x6D,0x1A,0x5A,0x79,0x9A,0x7D,0xCD,0x6B,0xA4,0x4A,0xFB,0x75,0x6D,0x41
,0x4C,0x56,0x45,0x2C,0xD3,0xBF,0xA8,0xA9,0x60,0x39,0x07,0xF2,0xE4,0x00,0x9E
,0x12,0xA9,0x07,0xAA,0x6A,0x21,0x7D,0xA3,0xAA,0xBA,0x25,0x2D,0xF2,0xC5,0x23
,0xD0,0xEE,0x0B,0x4C,0xE5,0xA1,0xE6,0x38,0x49,0x1C,0xA8,0x7D,0x50,0xC8,0x0F
,0x62,0x08,0xEC,0x7A,0x7E,0xCC,0x4E,0xB7,0x59,0xC9,0x66,0x63,0x5D,0x63,0xE1
,0x8E,0xE4,0xA2,0x96,0xC9,0x6F,0x9A,0x8C,0x3C,0x76,0x2A,0xF3,0xE5,0x52,0xC7
,0x2F,0xE3,0xB6,0x54,0x83,0xC5,0xE8,0x65,0xFC,0xA0,0x30,0x2A,0x99,0xE8,0x3F
,0x07,0x4F,0xBB,0xE5,0x3A,0xB8,0xD1,0xD3,0xDC,0x28,0xA4,0xA3,0xAA,0x8F,0x9C
,0x52,0x0E,0xA0,0x12,0x08,0x20,0xE4,0x10,0x47,0x50,0xC0,0x80,0x41,0x1D,0x41
,0x00,0x8E,0xA3,0x5C,0xDF,0xE1,0xE6,0xF4,0xA2,0xAF,0xA4,0xAA,0xDD,0x74,0x34
,0x6D,0x3D,0x0D,0x6A,0x88,0x77,0x96,0xDF,0x44,0x26,0x5A,0x69,0x80,0xE3,0xF1
,0x51,0xA7,0xE2,0x2C,0x00,0xF5,0x01,0xD5,0x94,0x02,0x3D,0x69,0x83,0x7B,0xED
,0x2B,0xC4,0x55,0x50,0x53,0xD3,0x7F,0x10,0x8E,0xE1,0x14,0xD0,0xF9,0xF6,0xFA
,0xF4,0x70,0xCB,0x5B,0x07,0x4C,0x36,0x47,0x4E,0x6B,0x90,0x1B,0xE7,0x90,0xC3
,0x19,0x2A,0xBD,0xE3,0x8E,0x1C,0xD9,0xD4,0xF0,0x3D,0x55,0x04,0xF6,0x8B,0xD2
,0x25,0x50,0x28,0x63,0x77,0x65,0x1C,0x6A,0x63,0x3D,0x39,0x10,0x3A,0x03,0x8E
,0x8C,0x07,0xBF,0x51,0x80,0x46,0xBC,0x2C,0xB5,0x15,0x14,0x35,0x7F,0xC0,0xEE
,0x52,0xBC,0xB2,0x2A,0x96,0xA3,0xA9,0x73,0x93,0x53,0x10,0xF6,0x63,0xFE,0xB1
,0x7A,0x03,0xF9,0x86,0x18,0x77,0x60,0xBB,0xAD,0x62,0x5D,0xA8,0x22,0xB8,0xD2
,0x79,0x2E,0xEF,0x14,0x8A,0xC2,0x48,0x66,0x8F,0x1C,0xE1,0x90,0x76,0x75,0xCF
,0xB8,0xF9,0x1E,0x84,0x12,0x08,0x20,0x91,0xA6,0x0C,0xA6,0x50,0xCA,0x55,0x80
,0x2A,0x46,0x08,0x23,0xA1,0xD5,0x2F,0x7E,0xB5,0x9B,0x47,0x8C,0xF6,0x2B,0x5D
,0x2C,0xB1,0x9B,0x75,0x36,0xDE,0xAE,0x7A,0x58,0x40,0x3C,0xE9,0xD1,0xAA,0x29
,0x87,0x96,0x4F,0x62,0x80,0x8F,0x40,0xE9,0x80,0x08,0xEC,0x06,0xAD,0x6B,0x1D
,0xCA,0x4A,0x96,0x9A,0x82,0xB9,0x52,0x2B,0x9D,0x2E,0x3C,0xF8,0xD7,0xA2,0xBA
,0x9C,0xF1,0x95,0x33,0xD7,0x83,0x60,0xE3,0xBE,0x08,0x65,0x27,0x2A,0x75,0x58
,0x54,0xCD,0xFC,0x57,0xC7,0x1D,0xCD,0x5C,0xA7,0x94,0x36,0x8B,0x75,0x25,0xA9
,0x08,0xED,0xE6,0x37,0x3A,0x89,0x47,0xEA,0x04,0x91,0x03,0xFF,0x00,0xEB,0x58
,0xF2,0x74,0xD5,0x7B,0x55,0x96,0x86,0xA8,0x6D,0xC1,0xBB,0x42,0xF0,0x08,0x2F
,0xF5,0x22,0x36,0x3D,0x70,0x30,0x99,0xE9,0xFE,0xD7,0x2F,0x96,0xB4,0x1B,0x3E
,0x39,0x85,0xC9,0xA4,0x49,0x11,0x54,0x4D,0x73,0x95,0xE4,0x2B,0xDD,0x5E,0xB0
,0x28,0x20,0x7B,0x67,0xC9,0x6F,0xE9,0xEF,0xAF,0x6A,0x2B,0xD2,0x51,0xED,0xEB
,0xDD,0xFE,0xA1,0xDA,0x38,0x2A,0x2B,0xEB,0x6B,0x94,0xC0,0xBE,0x6C,0xAF,0x1B
,0xCC,0xE5,0x18,0x0C,0x61,0x46,0x31,0xD4,0xE4,0x63,0xBE,0x35,0x1C,0xAD,0xA9
,0x96,0xCB,0x63,0xAD,0xB7,0x7C,0x64,0xB1,0x5C,0xC5,0x1D,0x1D,0xAC,0x04,0x40
,0xCC,0x6A,0xE5,0x0F,0x51,0x2E,0x49,0xF4,0xA8,0xE7,0x52,0xE0,0x93,0x8F,0xC0
,0x70,0x41,0xD7,0x16,0xD9,0xD6,0x99,0x29,0x67,0xB7,0xED,0x3A,0x7A,0x76,0x86
,0x42,0xCD,0x25,0xEA,0xAA,0x49,0xE4,0x0B,0x80,0x03,0x12,0x59,0x80,0xC7,0x2F
,0x32,0x75,0xF9,0x7E,0x12,0x3A,0x6B,0x7D,0xB9,0x37,0x2D,0x0D,0x9A,0x05,0x92
,0xAC,0x4D,0x5B,0x52,0xEC,0x12,0x1A,0x68,0xA2,0x3C,0xA4,0x7F,0x60,0xB1,0xE7
,0x24,0xF6,0x3E,0xAF,0xD4,0x6A,0xBC,0xF2,0x6B,0xAE,0x1B,0xB2,0x9E,0xA4,0xDD
,0x24,0xDB,0xD6,0x54,0xA4,0x4A,0x4B,0x6D,0x7D,0x6C,0x41,0x3E,0x22,0x28,0xC9
,0x56,0xF8,0x77,0xC0,0x8C,0xE5,0xD4,0x8E,0x41,0x89,0x00,0x0C,0x13,0xD4,0x6A
,0xCA,0xDB,0xFB,0x7A,0x86,0xD9,0x05,0x54,0xD6,0xE3,0xF1,0x15,0xAC,0x08,0x8A
,0xB6,0xA5,0x84,0x8E,0xC0,0xA8,0x20,0x72,0x1D,0x93,0x96,0x7D,0x2B,0x81,0xF4
,0xD0,0x42,0xA8,0xB6,0x7D,0xFB,0x76,0xEE,0x53,0x7D,0xDD,0xF2,0xFF,0x00,0x0E
,0xA4,0x95,0x71,0x15,0xB5,0x64,0xE5,0x2F,0x94,0x31,0xE8,0xCF,0xF2,0x03,0xD3
,0x96,0x3A,0x9C,0x90,0x42,0xE7,0x53,0xFB,0xFD,0xCE,0x0D,0xBF,0x6B,0xA7,0x8A
,0x92,0x8B,0xCE,0x9E,0x57,0x5A,0x6A,0x1A,0x38,0x70,0x9C,0xDF,0x1D,0x14,0x7B
,0x2A,0x80,0x09,0x27,0xB0,0x03,0x5F,0xBB,0xAA,0x69,0xA3,0xB1,0xC9,0x35,0x2C
,0xB4,0x34,0xF5,0x88,0x03,0xC1,0x25,0x6C,0xDE,0x54,0x71,0x37,0xE6,0x24,0x03
,0xD8,0x13,0xD3,0xDF,0xA8,0xEC,0x75,0x0A,0xDB,0x97,0xFA,0x33,0x5D,0x3C,0x3B
,0x8A,0xE9,0x44,0xF7,0x39,0x18,0x54,0x25,0xC6,0x29,0xB9,0xD1,0xBC,0x48,0xE1
,0x9A,0x28,0x58,0x80,0x13,0x01,0x7A,0x81,0x9C,0xF7,0x24,0x9E,0xC1,0x95,0xE1
,0xF4,0x75,0xD0,0x50,0xDD,0x16,0xBF,0x9D,0x4D,0xCD,0x6B,0xDA,0xA6,0x46,0x4A
,0x99,0x54,0x4B,0xE6,0x46,0x8E,0x8D,0xC5,0x01,0x1C,0x48,0xC0,0xC1,0xE9,0xE9
,0x23,0xDB,0x5D,0x89,0xF6,0x61,0xB7,0x5B,0xE8,0x3C,0x0B,0xDA,0xF3,0xD0,0x95
,0x79,0x2E,0x54,0x6B,0x71,0xAD,0x98,0x63,0x32,0xD4,0xCD,0xEB,0x95,0x98,0xFB
,0x90,0xC4,0xAF,0xD0,0x28,0x1E,0xDA,0xE3,0x5D,0xC3,0x7F,0xA1,0xA1,0xA8,0xB4
,0xDF,0xE8,0x68,0xAE,0xEB,0x6F,0x84,0x2D,0x35,0x5C,0xE1,0x3C,0xB4,0x92,0x98
,0xE4,0x29,0x61,0x91,0x26,0x54,0x9E,0x4A,0x78,0xE3,0xA9,0x1F,0xCC,0x35,0x7F
,0xFD,0x99,0xB7,0x87,0xF6,0x77,0x72,0x43,0xE1,0xD4,0xCD,0xE6,0x59,0x2E,0xC2
,0x5A,0xBB,0x0C,0xA1,0xB9,0x08,0x25,0x00,0xC9,0x35,0x3E,0x7F,0x23,0x0E,0x52
,0xA7,0xD7,0x98,0xF9,0x6B,0x31,0x68,0xB7,0x52,0xE9,0x7F,0x15,0xFC,0x73,0x11
,0x78,0x98,0xDE,0x79,0xFD,0x2F,0x8B,0xFD,0x52,0xD1,0xDD,0xA0,0xAA,0x97,0x1E
,0x5D,0x35,0xBA,0xB2,0xA4,0xFE,0xA8,0x62,0xEB,0xFD,0x19,0xBF,0xAE,0xA2,0x7F
,0x65,0xD8,0x1A,0x9F,0xEC,0xFB,0xB2,0xD5,0xF3,0xC9,0xED,0x89,0x29,0xCF,0xBF
,0x32,0x5F,0x3F,0xF9,0xB5,0xFB,0xE3,0x45,0xD3,0xE0,0xF6,0xEE,0xEF,0x9D,0x5B
,0x0D,0x6C,0xD9,0xD5,0xD3,0x7F,0xBD,0x2A,0xB0,0x4F,0xF3,0x80,0xEB,0x6D,0xE1
,0x7A,0xD2,0x6D,0x9F,0x06,0x76,0xA4,0x37,0x2A,0x98,0x69,0x21,0xA1,0xB1,0xD1
,0x45,0x2C,0xB3,0x38,0x45,0x0C,0x21,0x40,0x72,0x4F,0xB9,0x3F,0xE6,0x75,0x58
,0x4C,0xB5,0xCE,0xFF,0x00,0x6F,0xFB,0xFD,0x55,0xAF,0xC1,0x18,0x6C,0xB4,0x2E
,0xDE,0x7D,0xF6,0xE9,0x0D,0x23,0xC6,0x87,0xD6,0xF1,0x28,0x69,0x08,0x00,0x75
,0x3E,0xA4,0x8C,0x7F,0xBD,0xF5,0xD5,0xB3,0x71,0xDC,0xB5,0x53,0x79,0x6C,0xBC
,0xAC,0x94,0x13,0x37,0x08,0x67,0xAA,0x84,0xB5,0x6D,0x59,0xF9,0x53,0xD3,0x60
,0xB6,0x71,0xD7,0x2E,0x32,0x30,0x7E,0xEC,0x8E,0xBA,0xE6,0x2F,0xB6,0x9D,0x75
,0x3C,0x9B,0x9B,0x62,0xED,0x7B,0x20,0x92,0x2B,0xC5,0x4C,0xD2,0x56,0xD6,0xCD
,0x2C,0xDE,0x65,0x40,0x8B,0x2A,0xA8,0x64,0x70,0x1F,0x0A,0x38,0xCC,0x70,0x32
,0xAA,0x17,0xA0,0xC7,0x4D,0x05,0xEB,0x6A,0xBB,0x45,0xB4,0x36,0x15,0x97,0x6D
,0x52,0x19,0x2C,0xD4,0xD6,0xFB,0x64,0x51,0x21,0x78,0x54,0xD7,0x4D,0x1C,0x68
,0x14,0xCA,0xB0,0xB7,0xA6,0x05,0x24,0x64,0xCB,0x3E,0x02,0x9C,0xF2,0x51,0x9C
,0xEA,0x25,0xB8,0x37,0xAD,0x0D,0x9A,0xDD,0x3D,0x4D,0x4C,0xF1,0xDA,0xE9,0x2A
,0x19,0x52,0x47,0x92,0x57,0x96,0x7A,0xC6,0x23,0xD2,0xAE,0xE7,0xEF,0x67,0x62
,0x08,0xC2,0x00,0xAA,0x01,0xF4,0x2D,0x42,0x74,0xD5,0x41,0xBE,0xFC,0x57,0xB7
,0x58,0xED,0xA9,0x25,0x34,0xE2,0xB6,0x56,0x92,0x39,0x3E,0x2E,0x47,0xE4,0x1D
,0xCB,0x15,0xF3,0xE1,0x8C,0xC9,0xCE,0xA5,0x97,0x84,0x9F,0x7D,0xCC,0x80,0x53
,0x02,0xA7,0xD4,0x22,0xD7,0x3B,0x6E,0xFD,0xF9,0x7B,0xBF,0xDC,0xA4,0xAB,0xF8
,0xCA,0xC8,0x7C,0xD8,0x7C,0xA7,0x2D,0x32,0xB4,0x84,0x32,0xB0,0x95,0x43,0x2A
,0xA0,0x48,0xDF,0xCC,0x7C,0xC4,0x8A,0xA8,0x47,0x1E,0x41,0x98,0x16,0x21,0x70
,0xF8,0xB7,0xE3,0x4D,0x43,0xAD,0x7D,0x9E,0x85,0xEA,0xAD,0x72,0xC4,0x8D,0x4E
,0x29,0xC4,0x43,0xE2,0x79,0x2F,0x1C,0x24,0xC7,0xA2,0x40,0x8B,0xC8,0x9E,0x08
,0x18,0xAB,0xC4,0x40,0x4A,0x62,0x49,0x14,0x25,0xFA,0xF5,0x59,0x79,0x9D,0x5E
,0xA4,0x45,0x1C,0x68,0xCE,0xD1,0xC5,0x12,0x61,0x57,0x9B,0x16,0x24,0x93,0x96
,0x91,0xBA,0x81,0xCD,0xCB,0x39,0x0A,0xA0,0xB1,0xE2,0x31,0xAD,0xD3,0x40,0xD3
,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D
,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34
,0x0D,0x34,0xD3,0x41,0x3B,0xF0,0x77,0x7D,0xB6,0xC8,0xBE,0x8A,0xA6,0x8F,0x92
,0x3B,0x0C,0x49,0xCD,0xB0,0x80,0xF4,0x74,0x65,0x19,0x06,0x37,0x1C,0x79,0x10
,0x39,0x02,0x91,0xB0,0xCF,0x03,0x1B,0xF4,0x66,0xE1,0xB1,0x51,0xEF,0x7B,0x74
,0x7B,0xFB,0x62,0xB2,0x35,0x7C,0xA8,0x16,0xB6,0x88,0xB0,0x51,0x56,0x14,0x7E
,0x16,0x39,0xC2,0xCC,0xA3,0xF0,0xB6,0x70,0xC0,0x8E,0xA5,0x48,0x61,0xC7,0x1A
,0xB0,0xFC,0x18,0xF1,0x1A,0xE1,0xB2,0xAE,0xE2,0x2F,0x8C,0x31,0x50,0x4C,0x40
,0x6F,0x30,0x33,0x44,0x87,0x3F,0xCE,0xAA,0x09,0x28,0x73,0xD4,0xA8,0x2C,0xBF
,0x89,0x43,0x7A,0xA3,0x90,0x27,0x53,0xC5,0x72,0xA4,0xBB,0x0D,0xD3,0xB4,0xEA
,0x9E,0xD7,0xB8,0x28,0xF3,0x1C,0xF1,0xCA,0xBC,0x04,0xC0,0x63,0x94,0x33,0xA1
,0xEC,0x71,0x8E,0xFF,0x00,0x4F,0xF0,0xB0,0x96,0xEC,0x9F,0x11,0x22,0xAF,0x99
,0xA0,0xB4,0xCF,0x16,0xD4,0xDC,0x4F,0x30,0xA8,0x9E,0xC5,0x72,0x63,0x15,0x24
,0xF5,0x1F,0xEB,0xE9,0x64,0x23,0xEE,0x65,0x6C,0x9C,0x82,0x0A,0xBF,0x22,0x19
,0x5B,0x25,0xB5,0x30,0x9E,0x87,0x69,0xF8,0x96,0x0C,0xAA,0x64,0xB3,0x6E,0x6A
,0x68,0x91,0x9D,0xA2,0x74,0xF3,0xD1,0x08,0xCA,0x37,0x4C,0xA5,0x44,0x0C,0x1B
,0x2A,0xE3,0x92,0xB0,0x6C,0xA9,0x19,0xD4,0x13,0x79,0xEC,0x0B,0xF5,0x2C,0x2D
,0x0D,0xF3,0x6F,0x47,0xB8,0x68,0x17,0xF0,0xD5,0xDB,0xA3,0x2C,0xEA,0x0F,0xB9
,0x87,0x3E,0x62,0x1F,0xFB,0x32,0xFF,0x00,0xB6,0xB7,0x16,0x98,0x66,0x63,0x5D
,0x1B,0xB3,0x7C,0x4C,0xB3,0x5C,0xFE,0x16,0xD5,0xB8,0x9E,0x3D,0xB9,0xB9,0x9A
,0x20,0x66,0xB6,0x56,0xB8,0x4E,0x4D,0xD8,0x98,0x5C,0xFA,0x66,0x42,0x41,0xC1
,0x42,0x4E,0x3B,0xE0,0xE7,0x53,0xCD,0x70,0x13,0xDA,0xAA,0x22,0x9A,0x9E,0x9E
,0x0B,0xE5,0x61,0x82,0x8C,0x9F,0x2E,0xD9,0x7D,0xB6,0x9A,0x94,0x8C,0x30,0x1E
,0x85,0x8E,0x65,0x05,0x54,0x8C,0x74,0x03,0x3D,0x8E,0x73,0x83,0xAD,0x85,0x35
,0x97,0xC5,0xEB,0xC5,0x65,0x3C,0x5B,0x35,0xEB,0xEC,0x34,0xC3,0x22,0x59,0x68
,0x26,0xAC,0xB6,0xD3,0x28,0xF6,0x3C,0x64,0x93,0x04,0x7F,0xD9,0xAE,0xB7,0xF9
,0x19,0xF8,0xBA,0xBF,0xC7,0x2B,0xD5,0x3E,0xD5,0xD9,0xED,0xBA,0xA2,0x94,0x45
,0x79,0xA1,0x6E,0x16,0xB5,0x03,0x91,0xA9,0x95,0xC8,0x1F,0x0E,0xCB,0x90,0x59
,0x1F,0x1E,0xA1,0xEC,0x17,0x98,0xC1,0x40,0x45,0x20,0x95,0xD5,0x3B,0x23,0x67
,0x6E,0x7B,0x95,0x75,0x64,0x55,0x3B,0x92,0xBD,0x0C,0x6D,0x36,0x42,0x7C,0x4D
,0x74,0x95,0x95,0x91,0xAB,0xE0,0x9E,0x80,0x65,0x7F,0x45,0x4F,0x90,0xD6,0x4E
,0xD6,0xDB,0x55,0xFB,0x5A,0xDD,0x78,0xBA,0xEE,0xED,0xCB,0x71,0xDC,0x97,0x6A
,0x3A,0x3A,0xA1,0x25,0x55,0x55,0x54,0xD2,0xA4,0x09,0xF0,0xF1,0x48,0x55,0x03
,0xB1,0xFC,0xC7,0xD5,0x80,0x48,0x38,0xE9,0xAA,0x7B,0x72,0x6E,0xF1,0xB9,0xFC
,0x47,0xAC,0xAA,0xA0,0x72,0xD0,0xD1,0xCD,0x2C,0x16,0xC9,0x02,0xB3,0x21,0xE7
,0x3C,0xAE,0x26,0x08,0x0E,0x64,0x90,0xF9,0x8C,0x16,0x30,0xAD,0x90,0x7D,0xBA
,0x91,0x8B,0x5B,0x5A,0x88,0xC6,0xD2,0xBA,0xB6,0x82,0xC9,0x6D,0xB7,0x5A,0x2B
,0x9C,0x4B,0x43,0x63,0x48,0x66,0xB8,0x41,0x14,0x65,0x05,0x54,0xEB,0x83,0x0C
,0x00,0xE0,0x06,0x32,0x3A,0xF2,0x6C,0x86,0xF4,0xAB,0x36,0x7E,0x7A,0xCD,0x8D
,0x4F,0x0E,0xF8,0xDF,0x16,0xC3,0xB9,0xEB,0xAB,0xED,0x16,0x45,0x4A,0x8B,0x9D
,0x55,0x74,0x14,0xB2,0xB4,0xB5,0x05,0xDF,0x8C,0xAD,0x94,0x0E,0xD1,0x42,0xCE
,0x4C,0x7E,0x6E,0x38,0x2F,0x5F,0xE6,0xC8,0xD4,0x4F,0x72,0x5F,0x29,0x62,0x85
,0x9A,0x86,0x78,0xB9,0xB1,0x61,0x0C,0x00,0x89,0x24,0x06,0x40,0x1E,0x4A,0x99
,0x9D,0x7E,0xEF,0x9C,0x9C,0x94,0x70,0x42,0x4A,0x84,0x28,0xC5,0x42,0x71,0x92
,0x27,0x7A,0xBA,0x57,0x5E,0x6E,0x73,0x5C,0xAE,0x53,0x99,0xEA,0x66,0x23,0x93
,0x71,0x0A,0xA0,0x00,0x15,0x55,0x55,0x40,0x54,0x45,0x50,0x15,0x55,0x40,0x55
,0x50,0x00,0x00,0x00,0x35,0x86,0x9F,0xD2,0xFD,0xB5,0xE2,0x1E,0xD5,0xAE,0xB2
,0xC3,0x66,0xDA,0x34,0x96,0x6B,0x9D,0xAA,0x8E,0x05,0xA7,0x4A,0x1A,0x59,0x44
,0xF1,0xC7,0x1A,0x00,0x00,0x1E,0x40,0x95,0x82,0x80,0x3A,0x79,0x91,0xC7,0xF3
,0x27,0x48,0xAD,0x7E,0x0D,0xEE,0x69,0x1D,0xAE,0x5E,0x1D,0xD8,0xCC,0xC8,0x7E
,0xF2,0x51,0x69,0xA7,0xA9,0x0B,0xF3,0x2D,0x2D,0x3F,0x31,0x18,0xFF,0x00,0xB4
,0x2A,0x7E,0x9A,0xFE,0x6F,0xDA,0xF7,0x45,0xD6,0x8A,0x78,0x64,0x95,0xE3,0xAF
,0x58,0x8C,0x3C,0x56,0xAD,0x79,0xB2,0xA4,0x5F,0x86,0x34,0x94,0x62,0x58,0x93
,0x07,0x04,0x44,0xE8,0x71,0x8E,0xBD,0x01,0x16,0x6E,0xD4,0xF1,0xFB,0x72,0xD2
,0x4D,0x0A,0x5E,0xAA,0xE5,0xAD,0x82,0x39,0x1D,0x90,0x56,0xC0,0xB7,0x24,0x81
,0x38,0x7A,0x12,0x31,0x3B,0x09,0xC1,0xE4,0x3A,0xB0,0xA9,0x18,0x07,0x20,0x12
,0xBD,0x43,0xBC,0xF6,0xDF,0x86,0x7E,0x10,0xC7,0x8A,0xEB,0x16,0xCA,0xDA,0x12
,0xF1,0x6F,0x4C,0xF0,0x50,0xC1,0x2F,0x16,0x1F,0x26,0xC1,0xC1,0xFD,0x35,0xA2
,0xFB,0x56,0xED,0xCA,0x8B,0xDF,0x83,0x15,0x49,0x6E,0xB6,0x0A,0xE7,0xB5,0xD5
,0xD2,0xD7,0xFC,0x2C,0x70,0x87,0x76,0x8A,0x19,0x54,0xC8,0x23,0x1F,0x3F,0x2F
,0x97,0x41,0xDC,0x02,0x3D,0xF5,0x41,0xEC,0xFF,0x00,0xB4,0x85,0x9A,0xE2,0x12
,0x6B,0xB5,0x13,0x52,0x55,0x24,0x2D,0x24,0x92,0xC5,0x5E,0xB2,0xB4,0x61,0x5F
,0x88,0xC8,0xAB,0x64,0x94,0x13,0x90,0xC1,0x20,0xAA,0x63,0x83,0xDB,0xA3,0x01
,0x6C,0x6D,0xAF,0x1A,0xAD,0x15,0x7F,0x0F,0x1D,0x06,0xEC,0xA7,0x59,0x27,0x8D
,0x5E,0x0A,0x6B,0x8A,0xF9,0x6C,0xE8,0xC3,0x21,0x84,0x75,0x46,0x17,0x60,0x41
,0xC8,0x29,0x3C,0xD9,0xF6,0xCE,0xA4,0xC6,0xAD,0x66,0x6B,0x31,0x30,0xE7,0x5F
,0xE2,0xF6,0x8A,0xCB,0x54,0xB5,0x55,0x35,0xB4,0x2D,0x6D,0x91,0x59,0x59,0x9A
,0x45,0xE2,0xCA,0x7A,0x60,0xF5,0xE8,0x71,0x91,0xC7,0xBF,0x6E,0xC7,0xA6,0xB3
,0x7E,0xCF,0xB3,0xD7,0xCF,0x59,0xE1,0x8C,0x4E,0x64,0x6A,0xA5,0xBE,0xA7,0xC3
,0x06,0x8D,0x83,0x18,0x17,0xCD,0x0C,0x73,0xDB,0x02,0x2E,0x5F,0x4E,0x3A,0xE8
,0xBB,0xCE,0xDE,0xF0,0xBF,0x7B,0xAC,0xB6,0xED,0xD5,0xB0,0x36,0xF7,0xC7,0x5C
,0x1B,0x92,0xCF,0x4A,0xAB,0x47,0x5B,0x2B,0x7E,0x70,0x66,0x58,0x25,0x39,0xEF
,0xE8,0x32,0x03,0xF3,0x3A,0xCB,0xD9,0x3E,0x19,0x78,0x7B,0xE1,0xAD,0xDE,0x4D
,0xC3,0x63,0xB2,0x5F,0x2E,0x37,0xCF,0x87,0x68,0x60,0x9E,0xB6,0x36,0x02,0x08
,0xF1,0xD5,0x44,0x8C,0xA9,0x0C,0x43,0x03,0xAB,0x1E,0xA4,0x64,0x02,0x73,0x83
,0xE7,0xF4,0xFE,0x96,0xBE,0x0D,0xF8,0xFD,0xBE,0xB7,0xBA,0xFB,0xC7,0x97,0xDC
,0xA2,0x9F,0x96,0x39,0xAE,0xF3,0xDC,0xF3,0x9F,0xE4,0x67,0x1F,0xD9,0x6A,0xBE
,0xD0,0x35,0x52,0xD4,0xED,0x4F,0x15,0x44,0x1D,0x19,0xE8,0x2D,0x56,0x08,0xFF
,0x00,0xC5,0x34,0xB2,0xB3,0x63,0xF5,0xC5,0x6A,0x7E,0xD8,0xD4,0xFB,0x7D,0xDC
,0xA9,0x2C,0x71,0xD3,0x88,0x23,0x8A,0x39,0xA9,0x95,0x62,0x82,0x56,0x55,0x96
,0x48,0x72,0x38,0x81,0x18,0x63,0xE5,0xC6,0xEC,0x3A,0x06,0x63,0x96,0xC6,0x02
,0x49,0xDB,0x51,0x6B,0xCE,0xFE,0xD8,0x56,0x5B,0x41,0x83,0x70,0x98,0xEF,0x97
,0x1B,0x9D,0x64,0x77,0x01,0x0D,0xB6,0x06,0xA8,0xF8,0xFA,0xE8,0xDA,0x33,0x12
,0x52,0xA7,0xE2,0x9B,0xCB,0x31,0x44,0xA1,0xC7,0xA0,0x79,0x63,0x91,0x04,0x95
,0xD7,0x2B,0x78,0xAD,0xF6,0x88,0xBA,0xDD,0xAE,0x33,0x4F,0x6B,0x91,0x69,0xE5
,0x2E,0x55,0x62,0x82,0x66,0x66,0x54,0xE6,0x43,0xA3,0xD4,0xA1,0x5E,0x01,0x95
,0x7F,0x0D,0x2F,0x10,0xC1,0xD4,0xB4,0xCE,0x54,0x83,0xE9,0x7C,0x85,0xCF,0xBF
,0x77,0xDC,0x1B,0x56,0x9E,0xBE,0xB6,0xB2,0xBA,0x25,0xBC,0x3A,0xF1,0x96,0x39
,0x27,0xCC,0x83,0xD2,0xCE,0x16,0xA2,0x59,0x3A,0xAE,0x55,0x59,0x96,0x16,0x0B
,0xCB,0x04,0x2D,0x1B,0x77,0xD7,0x1E,0xEE,0xED,0xE9,0x51,0xB8,0x6E,0xD5,0xB7
,0x9B,0x80,0x96,0xE9,0x78,0xAE,0x8C,0x47,0x2D,0x7D,0x67,0x4F,0x25,0x78,0xA0
,0x0B,0x0C,0x4A,0x78,0xAF,0x10,0x1D,0x03,0x31,0x7C,0xA9,0x56,0x0B,0x1B,0x0C
,0xEA,0x3D,0x7A,0xBB,0x57,0xDE,0x2A,0xD6,0xAA,0xE1,0x2A,0x49,0x22,0x46,0xB1
,0x20,0x48,0x92,0x24,0x55,0x1E,0xC1,0x10,0x05,0x19,0x39,0x62,0x40,0xF5,0x33
,0x33,0x1C,0xB3,0x12,0x70,0x74,0x1E,0x95,0x33,0xCD,0x53,0x53,0x2D,0x4D,0x4C
,0xD2,0x4D,0x3C,0xAE,0x5E,0x49,0x24,0x62,0xCC,0xEC,0x4E,0x4B,0x12,0x7A,0x92
,0x4F,0x5C,0xEB,0xCF,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34
,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3
,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40,0xD3,0x4D,0x34,0x0D,0x34,0xD3,0x40
,0xD3,0x4D,0x34,0x1B,0xAD,0xB7,0xB9,0xAE,0xB6,0x02,0x3E,0x02,0x45,0x01,0x64
,0x32,0xC7,0x90,0x43,0xC5,0x27,0x02,0xBC,0xD2,0x45,0x21,0xD0,0xF5,0x04,0x85
,0x60,0x1F,0x82,0x07,0x0C,0x14,0x0D,0x5F,0x1B,0x1F,0xED,0x09,0x5C,0x62,0x8E
,0x9E,0xE8,0xB4,0x75,0xAE,0x65,0x48,0x92,0x3A,0xB9,0x45,0x2C,0xE7,0x9B,0xB0
,0x1F,0x7C,0x17,0xC8,0x60,0xA0,0x29,0x67,0x71,0x00,0x1C,0xBD,0xC0,0x24,0x73
,0x6E,0x9A,0x0E,0xC0,0xB2,0x78,0xEB,0xB1,0x69,0xEE,0x17,0x46,0xBB,0xD4,0x54
,0x5B,0xE5,0x9E,0xA1,0x5D,0x50,0x2A,0xD4,0xAE,0x04,0x48,0xBD,0x1E,0x02,0xEA
,0x7A,0xA9,0xF7,0xD6,0x6D,0xC3,0xC7,0xCD,0x94,0xB6,0xF9,0xAA,0xED,0x8E,0xF5
,0xAB,0x16,0x39,0x19,0x25,0x8E,0x00,0x32,0xC1,0x41,0xE0,0xCD,0xE7,0x30,0xC9
,0x19,0xE1,0x1B,0x90,0x32,0x71,0x80,0x75,0xC6,0x5A,0x6A,0xEA,0x62,0xDB,0xF1
,0x3B,0xC5,0x76,0xDD,0x75,0x12,0x51,0xD5,0xD4,0x55,0xC9,0x6A,0x7A,0x87,0xF8
,0x8A,0x2B,0x5B,0x7C,0x2C,0x53,0x71,0xE2,0xB1,0xC9,0xE7,0x48,0xAC,0xF2,0xE4
,0x2F,0x67,0x8A,0x3C,0x71,0x5F,0x4E,0x73,0x8A,0xCA,0xA6,0xE5,0x23,0x42,0xF4
,0xD4,0xB1,0xAD,0x1D,0x2B,0x8C,0x34,0x51,0x77,0x90,0x7A,0x3F,0x1B,0x7E,0x27
,0xEA,0x8A,0xD8,0x27,0x88,0x6C,0x95,0x0B,0x9C,0x6B,0x07,0x4D,0x45,0x34,0xD3
,0x4D,0x03,0x4D,0x34,0xD0,0x35,0xB0,0xB4,0x5E,0x6E,0x96,0xAE,0x4B,0x41,0x59
,0x24,0x70,0xC9,0x24,0x72,0xCD,0x4E,0xD8,0x78,0x27,0x68,0xC9,0x29,0xE6,0xC4
,0xD9,0x49,0x00,0xC9,0xF4,0xB8,0x23,0x04,0x8C,0x60,0x9D,0x6B,0xF4,0xD0,0x4F
,0x36,0x77,0x8A,0x1B,0x83,0x6E,0x8A,0x91,0x4F,0x57,0x5B,0x0A,0xCB,0xC9,0xD6
,0x2A,0x29,0xD6,0x0A,0x76,0x72,0x57,0x02,0x4A,0x66,0x47,0xA7,0x78,0xC0,0x0D
,0xE9,0x11,0xA9,0xCB,0x67,0x96,0x06,0x35,0x67,0xD5,0xFD,0xA4,0xA6,0x8B,0x6E
,0x51,0x45,0x47,0x67,0xB0,0x4F,0x74,0x1C,0x3E,0x26,0x41,0xB5,0x29,0xE9,0x5E
,0x2F,0x49,0xE4,0x63,0x9B,0xCF,0x95,0x0B,0x06,0xC0,0x04,0xD3,0x80,0x46,0x4E
,0x17,0x1C,0x4F,0x3A,0x69,0xA0,0x92,0x6E,0xDD,0xEB,0x7F,0xDC,0xF5,0x32,0x55
,0x5C,0xEA,0xDD,0xAA,0x6A,0x23,0xE1,0x5B,0x32,0xCB,0x21,0x7A,0xD3,0xC8,0x36
,0x65,0x2C,0xC7,0x20,0x15,0x4C,0x20,0xC2,0x2F,0x00,0x55,0x54,0xE4,0x98,0xDE
,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A
,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68
,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A
,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69
,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6
,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81
,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6
,0x9A,0x68,0x1A,0x69,0xA6,0x81,0xA6,0x9A,0x68,0x3F,0xFF,0xD9};
/* end binary data. size = 22092 bytes */
#endif

mblk_t *ms_load_generate_yuv(MSVideoSize *reqsize)
{
	YuvBuf buf;
	mblk_t *m=NULL;
	int ysize;

#ifndef MS2_MINIMAL_SIZE
	m=jpeg2yuv((uint8_t*)def_mire,sizeof(def_mire),reqsize);
	if (m!=NULL)
	{
		return m;
	}
#endif

	m = ms_yuv_buf_alloc(&buf, reqsize->width, reqsize->height);
	ysize=buf.strides[0]*buf.h;
	memset(buf.planes[0],16,ysize);
	memset(buf.planes[1],128,ysize/4);
	memset(buf.planes[2],128,ysize/4);
	buf.planes[3]=NULL;
	return m;
}

mblk_t *ms_load_jpeg_as_yuv(const char *jpgpath, MSVideoSize *reqsize){
#if defined(WIN32)
	mblk_t *m=NULL;
	DWORD st_sizel;
	DWORD st_sizeh;
	uint8_t *jpgbuf;
	DWORD err;
	HANDLE fd;
	
#ifdef UNICODE
	WCHAR wUnicode[1024];
	MultiByteToWideChar(CP_UTF8, 0, jpgpath, -1, wUnicode, 1024);
	fd = CreateFile(wUnicode, GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, 0, NULL);
#else
	fd = CreateFile(jpgpath, GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, 0, NULL);
#endif
	if (fd==INVALID_HANDLE_VALUE){
		ms_error("Failed to open %s",jpgpath);
		m=ms_load_generate_yuv(reqsize);
		return m;
	}
	st_sizel=0;
	st_sizeh=0;
	st_sizel = GetFileSize(fd, &st_sizeh);
	if (st_sizeh>0 || st_sizel<=0)
	{
		CloseHandle(fd);
		ms_error("Can't load file %s",jpgpath);
		m=ms_load_generate_yuv(reqsize);
		return m;
	}
	jpgbuf=(uint8_t*)ms_malloc0(st_sizel);
	if (jpgbuf==NULL)
	{
		CloseHandle(fd);
		ms_error("Cannot allocate buffer for %s",jpgpath);
		m=ms_load_generate_yuv(reqsize);
		return m;
	}
	err=0;
	ReadFile(fd, jpgbuf, st_sizel, &err, NULL) ;            
	
	if (err!=st_sizel){
		  ms_error("Could not read as much as wanted !");
	}
	m=jpeg2yuv(jpgbuf,st_sizel,reqsize);
	ms_free(jpgbuf);
	if (m==NULL)
	{
		CloseHandle(fd);
		ms_error("Cannot load image from buffer for %s",jpgpath);
		m=ms_load_generate_yuv(reqsize);
		return m;
	}
	CloseHandle(fd);
	return m;
#else
	mblk_t *m=NULL;
	struct stat statbuf;
	uint8_t *jpgbuf;
	int err;
	int fd=open(jpgpath,O_RDONLY);

	if (fd!=-1){
		fstat(fd,&statbuf);
		if (statbuf.st_size<=0)
		{
			close(fd);
			ms_error("Cannot load %s",jpgpath);
			m=ms_load_generate_yuv(reqsize);
			return m;
		}
		jpgbuf=(uint8_t*)ms_malloc0(statbuf.st_size + FF_INPUT_BUFFER_PADDING_SIZE);
		if (jpgbuf==NULL)
		{
			close(fd);
			ms_error("Cannot allocate buffer for %s",jpgpath);
			m=ms_load_generate_yuv(reqsize);
			return m;
		}
		err=read(fd,jpgbuf,statbuf.st_size);
		if (err!=statbuf.st_size){
			ms_error("Could not read as much as wanted: %i<>%li !",err,(long)statbuf.st_size);
		}
		m=jpeg2yuv(jpgbuf,statbuf.st_size,reqsize);
		ms_free(jpgbuf);
		if (m==NULL)
		{
			close(fd);
			ms_error("Cannot load image from buffer for %s",jpgpath);
			m=ms_load_generate_yuv(reqsize);
			return m;
		}
	}else{
		m=ms_load_generate_yuv(reqsize);
		ms_error("Cannot load %s",jpgpath);
		return m;
	}
	close(fd);
	return m;
#endif
}



#ifndef PACKAGE_DATA_DIR
#define PACKAGE_DATA_DIR "share"
#endif

#ifndef NOWEBCAM_JPG
#define NOWEBCAM_JPG "nowebcamCIF"
#endif

static char *def_image=NULL;

static const char *def_image_path=PACKAGE_DATA_DIR "/images/" NOWEBCAM_JPG ".jpg";


mblk_t *ms_load_nowebcam(MSVideoSize *reqsize, int idx){
	char tmp[256];
	if (idx<0)
		snprintf(tmp, sizeof(tmp), "%s/images/%s.jpg", PACKAGE_DATA_DIR, NOWEBCAM_JPG);
	else
		snprintf(tmp, sizeof(tmp), "%s/images/%s%i.jpg", PACKAGE_DATA_DIR, NOWEBCAM_JPG, idx);
	return ms_load_jpeg_as_yuv(tmp,reqsize);
}

typedef struct _SIData{
	MSVideoSize vsize;
	char *nowebcamimage;
	uint64_t lasttime;
	float fps;
	mblk_t *pic;
}SIData;

void static_image_init(MSFilter *f){
	SIData *d=(SIData*)ms_new(SIData,1);
	d->vsize.width=MS_VIDEO_SIZE_CIF_W;
	d->vsize.height=MS_VIDEO_SIZE_CIF_H;

	if (def_image==NULL)
		def_image=ms_strdup(def_image_path);

	d->nowebcamimage=ms_strdup(def_image);
	d->lasttime=0;
#ifndef NO_FFMPEG
	d->pic=NULL;
#else
    // no rescaling without ffmpeg -> we need to load jpg image
    // before get_vsize is called
    d->pic=ms_load_jpeg_as_yuv(d->nowebcamimage,&d->vsize);
#endif
	
	d->fps=1;
	f->data=d;
    

}

void static_image_uninit(MSFilter *f){
	SIData *d=(SIData*)f->data;
	ms_free(d->nowebcamimage);
	ms_free(d);
}

void static_image_preprocess(MSFilter *f){
	SIData *d=(SIData*)f->data;
	if (d->pic==NULL){
		d->pic=ms_load_jpeg_as_yuv(d->nowebcamimage,&d->vsize);
	}
}

void static_image_process(MSFilter *f){
	SIData *d=(SIData*)f->data;
	int frame_interval=(int)(1000/d->fps);
	/*output a frame whenever needed, i.e. respect the FPS parameter */
	if ((f->ticker->time - d->lasttime>frame_interval) || d->lasttime==0){
		ms_mutex_lock(&f->lock);
		if (d->pic) {
			mblk_t *o=dupmsg(d->pic);
			/*prevent mirroring at the output*/
			mblk_set_precious_flag(o,1);
			ms_queue_put(f->outputs[0],o);
		}
		ms_filter_unlock(f);
		d->lasttime=f->ticker->time;
	}
}

void static_image_postprocess(MSFilter *f){
	SIData *d=(SIData*)f->data;
	if (d->pic) {
		freemsg(d->pic);
		d->pic=NULL;
	}
}

static int static_image_set_fps(MSFilter *f, void *arg){
	SIData *d=(SIData*)f->data;
	d->fps=*((float*)arg);
	d->lasttime=0;
	return 0;
}

static int static_image_get_fps(MSFilter *f, void *arg){
	SIData *d=(SIData*)f->data;
	*((float*)arg) = d->fps;
	return 0;
}

int static_image_set_vsize(MSFilter *f, void* data){
#ifndef NO_FFMPEG
	SIData *d=(SIData*)f->data;
	d->vsize=*(MSVideoSize*)data;
#else
    // no rescaling without ffmpeg
#endif
	return 0;
}

int static_image_get_vsize(MSFilter *f, void* data){
	SIData *d=(SIData*)f->data;
	*(MSVideoSize*)data=d->vsize;
	return 0;
}

int static_image_get_pix_fmt(MSFilter *f, void *data){
	*(MSPixFmt*)data=MS_YUV420P;
	return 0;
}

static int static_image_set_image(MSFilter *f, void *arg){
	SIData *d=(SIData*)f->data;
	const char *image = (const char *)arg;
	ms_filter_lock(f);
	if (d->nowebcamimage) ms_free(d->nowebcamimage);
	if (image!=NULL && image[0]!='\0')
		d->nowebcamimage=ms_strdup(image);
	else
		d->nowebcamimage = ms_strdup(def_image);

	if (d->pic!=NULL){
		/* Get rid of the old image and force a new preprocess so that the
			 new image is properly read. */
		freemsg(d->pic);
		d->pic=NULL;
		d->lasttime=0;
		static_image_preprocess(f);
	}

	ms_filter_unlock(f);
	return 0;
}

MSFilterMethod static_image_methods[]={
	{	MS_FILTER_SET_FPS,	static_image_set_fps	},
	{	MS_FILTER_GET_FPS,	static_image_get_fps	},
	{	MS_FILTER_SET_VIDEO_SIZE, static_image_set_vsize },
	{	MS_FILTER_GET_VIDEO_SIZE, static_image_get_vsize },
	{	MS_FILTER_GET_PIX_FMT, static_image_get_pix_fmt },
	{	MS_STATIC_IMAGE_SET_IMAGE, static_image_set_image },
	{	0,0 }
};

MSFilterDesc ms_static_image_desc={
	MS_STATIC_IMAGE_ID,
	"MSStaticImage",
	N_("A filter that outputs a static image."),
	MS_FILTER_OTHER,
	NULL,
	0,
	1,
	static_image_init,
	static_image_preprocess,
	static_image_process,
	static_image_postprocess,
	static_image_uninit,
	static_image_methods
};

MS_FILTER_DESC_EXPORT(ms_static_image_desc)

static void static_image_detect(MSWebCamManager *obj);

static void static_image_cam_init(MSWebCam *cam){
	cam->name=ms_strdup("Static picture");
}


static MSFilter *static_image_create_reader(MSWebCam *obj){
	return ms_filter_new_from_desc(&ms_static_image_desc);
}

MSWebCamDesc static_image_desc={
	"StaticImage",
	&static_image_detect,
	&static_image_cam_init,
	&static_image_create_reader,
	NULL
};

static void static_image_detect(MSWebCamManager *obj){
	MSWebCam *cam=ms_web_cam_new(&static_image_desc);
	ms_web_cam_manager_add_cam(obj,cam);
}

void ms_static_image_set_default_image(const char *path){
	if (def_image!=NULL)
		ms_free(def_image);
	def_image=NULL;
	if (path)
		def_image=ms_strdup(path);
}

const char *ms_static_image_get_default_image(){
	return def_image;
}
