/*
 * Copyright © 2011 Canonical Limited
 *
 * This library is free software: you can redistribute it and/or modify it under the terms of version 3 of the
 * GNU Lesser General Public License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this program.  If not,
 * see <http://www.gnu.org/licenses/>.
 *
 * Author: Ryan Lortie <desrt@desrt.ca>
 */

#ifndef _qconf_h_
#define _qconf_h_

#include <QObject>

typedef struct _QConfPrivate QConfPrivate;

class QConf : public QObject {
  private:
    void setSchema(const QString& schema);
    void setPath(const QString& path);
    QConfPrivate *priv;

  public:
    QConf(const QString& schema, const QString& path);
    QConf(const QString& schema);
    ~QConf();

    /* QML wants to construct the QConf with no arguments and set the properties of it after it is constructed.
     * It needs two things for this to work: a constructor that takes zero arguments and a staticMetaObject with
     * the properties that it can set at construct-time.
     *
     * We export the properties "schema" and "path" so that it can do something like:
     *
     *   QConf {
     *     id:     mySettings;
     *     schema: "ca.desrt.MyApp";
     *   }
     *
     * The staticMetaObject is only used at construction time; for future accesses (ie: mySettings.someKey) the
     * dynamic properties from the schema will be used.  This means that the keys "schema" and "path" can even
     * appear in the GSettings schema without causing problems.  The switchover from handling these static
     * properties to the dynamic ones occurs at the first call to metaObject() (ie: the first point at which the
     * dynamic metaobject is exposed).
     *
     * This zero-argument constructor and the staticMetaObject are quite magic and should be considered to be
     * internal API.  Since subclassing with Q_OBJECT involves using the staticMetaObject, this means that you
     * should never subclass QConf unless you do it with vanilla C++.
     */
    static const QMetaObject staticMetaObject;
    QConf();

    /* The primary interface to QConf is through setting and getting QObject properties.  We dynamically
     * generate and expose a QMetaObject based on the contents of the GSettings schema and response to
     * qt_metacall() requests on the basis of that metadata.
     */ 
    virtual int qt_metacall(QMetaObject::Call, int, void **);
    virtual const QMetaObject *metaObject() const;
    virtual void *qt_metacast(const char *);

    void notify(const char *key);
};

#endif /* _qconf_h_ */
