/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/

#include "REAL.H"
#include "CONSTANTS.H"
#include "HGPROJ_F.H"
#include "BCTypes.H"

#define DIMS lo_1,lo_2,lo_3,hi_1,hi_2,hi_3
#define CDIMS loc_1,loc_2,loc_3,hic_1,hic_2,hic_3
#define GDIMS g_lo_1,g_lo_2,g_lo_3,g_hi_1,g_hi_2,g_hi_3
#define PDIMS p_lo_1,p_lo_2,p_lo_3,p_hi_1,p_hi_2,p_hi_3

#if BL_USE_FLOAT
#define sixteenth  .0625e0
#else
#define sixteenth  .0625d0
#endif

c *************************************************************************
c ** MKCOEFF **
c ** Define the coefficients along the axis as averages of sig.
c *************************************************************************

      subroutine FORT_MKCOEFF(coeff,sig,DIMS,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T   sig(lo_1-ng:hi_1  +ng,lo_2-ng:hi_2  +ng,lo_3-ng:hi_3  +ng)
      REAL_T coeff(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng,3)

c     Local variables
      integer i,j,k

      do k = lo_3  ,hi_3+1
      do j = lo_2  ,hi_2+1
      do i = lo_1-1,hi_1+1 
          coeff(i,j,k,1) = ( sig(i,j,k  ) + sig(i,j-1,k  ) +
     $                       sig(i,j,k-1) + sig(i,j-1,k-1) ) * fourth
          coeff(i,j,k,1) = coeff(i,j,k,1)
      enddo
      enddo
      enddo

      do k = lo_3  ,hi_3+1
      do j = lo_2-1,hi_2+1 
      do i = lo_1  ,hi_1+1
          coeff(i,j,k,2) = ( sig(i,j,k  ) + sig(i-1,j,k  ) +
     $                       sig(i,j,k-1) + sig(i-1,j,k-1) ) * fourth
          coeff(i,j,k,2) = coeff(i,j,k,2)
      enddo
      enddo
      enddo

      do k = lo_3-1,hi_3+1 
      do j = lo_2  ,hi_2+1
      do i = lo_1  ,hi_1+1   
          coeff(i,j,k,3) = ( sig(i  ,j,k) + sig(i  ,j-1,k) +
     $                       sig(i-1,j,k) + sig(i-1,j-1,k) ) * fourth
          coeff(i,j,k,3) = coeff(i,j,k,3)
      enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** GRADHG **
c ** Compute the cell-centered gradient of the nodal pressure field
c *************************************************************************

      subroutine FORT_GRADHG(gphi,GDIMS,phi,PDIMS,DIMS,dx)

      implicit none

      integer DIMS
      integer g_lo_1, g_lo_2, g_lo_3
      integer g_hi_1, g_hi_2, g_hi_3
      integer p_lo_1, p_lo_2, p_lo_3
      integer p_hi_1, p_hi_2, p_hi_3
      REAL_T  gphi(g_lo_1:g_hi_1,g_lo_2:g_hi_2,g_lo_3:g_hi_3,3)
      REAL_T   phi(p_lo_1:p_hi_1,p_lo_2:p_hi_2,p_lo_3:p_hi_3)
      REAL_T  dx(3)
      
c     Local variables
      integer i, j, k,n

      do k = lo_3,hi_3
         do j = lo_2,hi_2
            do i = lo_1,hi_1
               
               gphi(i,j,k,1) = fourth*(phi(i+1,j,k  ) + phi(i+1,j+1,k  ) +
     $              phi(i+1,j,k+1) + phi(i+1,j+1,k+1) -
     $              phi(i  ,j,k  ) - phi(i  ,j+1,k  ) -
     $              phi(i  ,j,k+1) - phi(i  ,j+1,k+1) ) /dx(1)
               
               gphi(i,j,k,2) = fourth*(phi(i,j+1,k  ) + phi(i+1,j+1,k  ) +
     $              phi(i,j+1,k+1) + phi(i+1,j+1,k+1) -
     $              phi(i,j  ,k  ) - phi(i+1,j  ,k  ) -
     $              phi(i,j  ,k+1) - phi(i+1,j  ,k+1) ) /dx(2)
               
               gphi(i,j,k,3) = fourth*(phi(i,j  ,k+1) + phi(i+1,j  ,k+1) + 
     $              phi(i,j+1,k+1) + phi(i+1,j+1,k+1) -
     $              phi(i,j  ,k  ) - phi(i+1,j  ,k  ) -
     $              phi(i,j+1,k  ) - phi(i+1,j+1,k  ) ) /dx(3)
            enddo
         enddo
      enddo

      return
      end

c *************************************************************************
c ** RHSHG **
c ** Compute the right-hand-side D(V) for the projection
c *************************************************************************

      subroutine FORT_RHSHG(du,u,divu_src,vol,DIMS,dx,bc,norm,ng,
     $                      is_singular,sum_src,sum_fac)

      implicit none

      integer DIMS
      integer ng
      REAL_T        du(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T         u(lo_1-3:hi_1+3,lo_2-3:hi_2+3,lo_3-3:hi_3+3,3)
      REAL_T  divu_src(lo_1  :hi_1+1,lo_2  :hi_2+1,lo_3  :hi_3+1)
      REAL_T       vol(lo_1-1:hi_1+1,lo_2-1:hi_2+1,lo_3-1:hi_3+1)
      REAL_T  dx(3)
      REAL_T sum_src, sum_fac
      integer is_singular
      integer bc(2,3)

c     Local variables
      REAL_T fac,facx,facy,facz,norm
      REAL_T factor
      REAL_T  vavg
      integer i, j, k, n
      integer is,ie,js,je,ks,ke
      integer istrt,iend
      integer jstrt,jend
      integer kstrt,kend

      is = lo_1
      ie = hi_1
      js = lo_2
      je = hi_2
      ks = lo_3
      ke = hi_3

      istrt = cvmgt(lo_1+1,lo_1  ,BCX_LO .eq. OUTLET)
      iend  = cvmgt(hi_1  ,hi_1+1,BCX_HI .eq. OUTLET)

      jstrt = cvmgt(lo_2+1,lo_2  ,BCY_LO .eq. OUTLET)
      jend  = cvmgt(hi_2  ,hi_2+1,BCY_HI .eq. OUTLET)

      kstrt = cvmgt(lo_3+1,lo_3  ,BCZ_LO .eq. OUTLET)
      kend  = cvmgt(hi_3  ,hi_3+1,BCZ_HI .eq. OUTLET)

      if (BCX_LO .eq. WALL) then
        do n = 1,3
        do k = ks-1,ke+1
        do j = js-1,je+1
          u(is-1,j,k,n) = zero
        enddo
        enddo
        enddo
      endif

      if (BCX_HI .eq. WALL) then
        do n = 1,3
        do k = ks-1,ke+1
        do j = js-1,je+1
          u(ie+1,j,k,n) = zero
        enddo
        enddo
        enddo
      endif

      if (BCY_LO .eq. WALL) then
        do n = 1,3
        do k = ks-1,ke+1
        do i = is-1,ie+1
          u(i,js-1,k,n) = zero
        enddo
        enddo
        enddo
      endif

      if (BCY_HI .eq. WALL) then
        do n = 1,3
        do k = ks-1,ke+1
        do i = is-1,ie+1
          u(i,je+1,k,n) = zero
        enddo
        enddo
        enddo
      endif

      if (BCZ_LO .eq. WALL) then
        do n = 1,3
        do j = js-1,je+1
        do i = is-1,ie+1
          u(i,j,ks-1,n) = zero
        enddo
        enddo
        enddo
      endif

      if (BCZ_HI .eq. WALL) then
        do n = 1,3
        do j = js-1,je+1
        do i = is-1,ie+1
          u(i,j,ke+1,n) = zero
        enddo
        enddo
        enddo
      endif
 
      facx = fourth / dx(1)
      facy = fourth / dx(2)
      facz = fourth / dx(3)

      do k = kstrt,kend 
      do j = jstrt,jend 
      do i = istrt,iend 

          du(i,j,k) = ((vol(i  ,j-1,k  )*u(i  ,j-1,k  ,1) + 
     $                  vol(i  ,j  ,k  )*u(i  ,j  ,k  ,1) +
     $                  vol(i  ,j-1,k-1)*u(i  ,j-1,k-1,1) + 
     $                  vol(i  ,j  ,k-1)*u(i  ,j  ,k-1,1)) - 
     $                 (vol(i-1,j-1,k  )*u(i-1,j-1,k  ,1) + 
     $                  vol(i-1,j  ,k  )*u(i-1,j  ,k  ,1)  +
     $                  vol(i-1,j-1,k-1)*u(i-1,j-1,k-1,1) + 
     $                  vol(i-1,j  ,k-1)*u(i-1,j  ,k-1,1)) ) * facx
     $              + ((vol(i-1,j  ,k  )*u(i-1,j  ,k  ,2) + 
     $                  vol(i  ,j  ,k  )*u(i  ,j  ,k  ,2) +
     $                  vol(i-1,j  ,k-1)*u(i-1,j  ,k-1,2) + 
     $                  vol(i  ,j  ,k-1)*u(i  ,j  ,k-1,2)) - 
     $                 (vol(i-1,j-1,k  )*u(i-1,j-1,k  ,2) + 
     $                  vol(i  ,j-1,k  )*u(i  ,j-1,k  ,2) +
     $                  vol(i-1,j-1,k-1)*u(i-1,j-1,k-1,2) + 
     $                  vol(i  ,j-1,k-1)*u(i  ,j-1,k-1,2)) ) * facy
     $              + ((vol(i-1,j  ,k  )*u(i-1,j  ,k  ,3) + 
     $                  vol(i  ,j  ,k  )*u(i  ,j  ,k  ,3) +
     $                  vol(i-1,j-1,k  )*u(i-1,j-1,k  ,3) + 
     $                  vol(i  ,j-1,k  )*u(i  ,j-1,k  ,3)) - 
     $                 (vol(i-1,j  ,k-1)*u(i-1,j  ,k-1,3) + 
     $                  vol(i  ,j  ,k-1)*u(i  ,j  ,k-1,3) +
     $                  vol(i-1,j-1,k-1)*u(i-1,j-1,k-1,3) + 
     $                  vol(i  ,j-1,k-1)*u(i  ,j-1,k-1,3)) ) * facz
          vavg = 
     $          (vol(i,j,k  ) + vol(i-1,j,k  ) + vol(i,j-1,k  ) + vol(i-1,j-1,k  )
     $          +vol(i,j,k-1) + vol(i-1,j,k-1) + vol(i,j-1,k-1) + vol(i-1,j-1,k-1))
          du(i,j,k) = du(i,j,k) - vavg * divu_src(i,j,k)
        
          fac = one
          fac = cvmgt(two*fac,fac,(i .eq. is  ) .and. 
     $                (BCX_LO .eq. WALL .or. BCX_LO .eq. INLET) )
          fac = cvmgt(two*fac,fac,(i .eq. ie+1) .and. 
     $                (BCX_HI .eq. WALL .or. BCX_HI .eq. INLET) )
          fac = cvmgt(two*fac,fac,(j .eq. js  ) .and. 
     $                (BCY_LO .eq. WALL .or. BCY_LO .eq. INLET) )
          fac = cvmgt(two*fac,fac,(j .eq. je+1) .and. 
     $                (BCY_HI .eq. WALL .or. BCY_HI .eq. INLET) )
          fac = cvmgt(two*fac,fac,(k .eq. ks  ) .and. 
     $                (BCZ_LO .eq. WALL .or. BCZ_LO .eq. INLET) )
          fac = cvmgt(two*fac,fac,(k .eq. ke+1) .and. 
     $                (BCZ_HI .eq. WALL .or. BCZ_HI .eq. INLET) )
 
          du(i,j,k) = du(i,j,k) * fac

          if (i .eq. 5 .and. j .eq. 1 .and. k .eq. 0) then
            print *,'DU ',du(i,j,k)
            print *,'VAVG ',vavg
            print *,'FAC ',fac
          endif

      enddo
      enddo
      enddo

      if (is_singular .eq. 1) then
        sum_src  = zero
        sum_fac  = zero
        do k = kstrt, kend 
        do j = jstrt, jend 
        do i = istrt, iend 
          factor = one
          factor = cvmgt(half*factor,factor,i.eq.is .or. i.eq.ie+1)
          factor = cvmgt(half*factor,factor,j.eq.js .or. j.eq.je+1)
          factor = cvmgt(half*factor,factor,k.eq.ks .or. k.eq.ke+1)
          sum_src = sum_src + factor * du(i,j,k)
          sum_fac = sum_fac + factor
        enddo
        enddo
        enddo
      endif

      norm = zero

      do k = kstrt, kend 
      do j = jstrt, jend 
      do i = istrt, iend 
        norm = max(norm, abs(du(i,j,k)))
      enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** ADJUSTRHS **
c ** Adjust the RHS so it sums to zero if the problem has no outflow boundaries.
c *************************************************************************

      subroutine FORT_ADJUSTRHS(du,DIMS,bc,ng,sum_src)

      implicit none

      integer DIMS
      integer bc(2,3)
      integer ng
      REAL_T  du(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T  sum_src

c     Local variables
      integer i, j, k
      integer istart,iend,jstart,jend,kstart,kend

      istart = cvmgt(lo_1+1,lo_1  ,BCX_LO .eq. OUTLET)
      iend   = cvmgt(hi_1  ,hi_1+1,BCX_HI .eq. OUTLET)
      jstart = cvmgt(lo_2+1,lo_2  ,BCY_LO .eq. OUTLET)
      jend   = cvmgt(hi_2  ,hi_2+1,BCY_HI .eq. OUTLET)
      kstart = cvmgt(lo_3+1,lo_3  ,BCZ_LO .eq. OUTLET)
      kend   = cvmgt(hi_3  ,hi_3+1,BCZ_HI .eq. OUTLET)

      do k = kstart,kend 
      do j = jstart,jend 
         do i = istart,iend 
           du(i,j,k) = du(i,j,k) - sum_src
         enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** PROJUHG **
c ** Define the updated pressure and vector field
c *************************************************************************

      subroutine FORT_PROJUHG(u,pressure,phi,gradphi,rhonph,DIMS,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T         u(lo_1-3:hi_1+3,lo_2-3:hi_2+3,lo_3-3:hi_3+3,3)
      REAL_T  pressure(lo_1  :hi_1+1,lo_2  :hi_2+1,lo_3  :hi_3+1)
      REAL_T       phi(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T   gradphi(lo_1-1:hi_1+1,lo_2-1:hi_2+1,lo_3-1:hi_3+1,3)
      REAL_T    rhonph(lo_1-1:hi_1+1,lo_2-1:hi_2+1,lo_3-1:hi_3+1)

c     Local variables
      integer i, j, k

      do k = lo_3,hi_3 
      do j = lo_2,hi_2 
        do i = lo_1,hi_1 
          u(i,j,k,1) = u(i,j,k,1) - gradphi(i,j,k,1)/rhonph(i,j,k)
          u(i,j,k,2) = u(i,j,k,2) - gradphi(i,j,k,2)/rhonph(i,j,k)
          u(i,j,k,3) = u(i,j,k,3) - gradphi(i,j,k,3)/rhonph(i,j,k)
        enddo
      enddo
      enddo

    
      print *,'UMAX ',u(7,7,7,1)

      do k = lo_3,hi_3+1
      do j = lo_2,hi_2+1
        do i = lo_1,hi_1+1 
c          pressure(i,j,k) = pressure(i,j,k) + phi(i,j,k)
           pressure(i,j,k) = phi(i,j,k)
        enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** RESIDUAL **
c ** Compute the residual R = f - D(sig G(phi))
c *************************************************************************

      subroutine FORT_RESIDUAL(residual,phi,source,coeff,dgphi,
     $                         DIMS,dx,resnorm,bc,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T  residual(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T       phi(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T    source(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T     coeff(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng,3)
      REAL_T     dgphi(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T  dx(3)
      REAL_T  resnorm
      integer bc(2,3)

c     Local variables
      integer i,j,k
      integer istrt,jstrt,kstrt
      integer iend,jend,kend

      resnorm = zero

      istrt = cvmgt(lo_1+1,lo_1  ,BCX_LO .eq. OUTLET)
      iend  = cvmgt(hi_1  ,hi_1+1,BCX_HI .eq. OUTLET)

      jstrt = cvmgt(lo_2+1,lo_2  ,BCY_LO .eq. OUTLET)
      jend  = cvmgt(hi_2  ,hi_2+1,BCY_HI .eq. OUTLET)

      kstrt = cvmgt(lo_3+1,lo_3  ,BCZ_LO .eq. OUTLET)
      kend  = cvmgt(hi_3  ,hi_3+1,BCZ_HI .eq. OUTLET)

      call makedgphi(phi,dgphi,coeff,DIMS,dx,bc,ng,istrt,jstrt,kstrt,iend,jend,kend)

      do k = kstrt,kend
      do j = jstrt,jend
        do i = istrt,iend
          residual(i,j,k) = source(i,j,k) - dgphi(i,j,k)
        enddo
      enddo
      enddo

      do k = kstrt,kend 
      do j = jstrt,jend 
        do i = istrt,iend 
          resnorm = max(resnorm,abs(residual(i,j,k)))
        enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** RELAX **
c ** Gauss-Seidel relaxation
c *************************************************************************

      subroutine FORT_RELAX(phi,source,coeff,dgphi,DIMS,dx,bc,nnrelax,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T     phi(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T  source(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T   coeff(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng,3)
      REAL_T   dgphi(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T dx(3)
      integer bc(2,3)
      integer nnrelax

c     Local variables
      REAL_T  facx, facy, facz
      REAL_T  rfac, mult
      integer i,j,k
      integer iter, ioff, iinc
      integer is,ie,js,je,ks,ke
      integer istrt, jstrt, kstrt
      integer iend, jend, kend
      integer extra_xlo,extra_xhi
      integer extra_ylo,extra_yhi
      integer extra_zlo,extra_zhi

      is = lo_1
      ie = hi_1
      js = lo_2
      je = hi_2
      ks = lo_3
      ke = hi_3

      facx = one/(dx(1)*dx(1))
      facy = one/(dx(2)*dx(2))
      facz = one/(dx(3)*dx(3))

      istrt = cvmgt(lo_1+1,lo_1  ,BCX_LO .eq. OUTLET)
      iend  = cvmgt(hi_1  ,hi_1+1,BCX_HI .eq. OUTLET)

      jstrt = cvmgt(lo_2+1,lo_2  ,BCY_LO .eq. OUTLET)
      jend  = cvmgt(hi_2  ,hi_2+1,BCY_HI .eq. OUTLET)

      kstrt = cvmgt(lo_3+1,lo_3  ,BCZ_LO .eq. OUTLET)
      kend  = cvmgt(hi_3  ,hi_3+1,BCZ_HI .eq. OUTLET)

      if (2*nnrelax .gt. ng) then
        print *,'DONT HAVE ENOUGH BUFFER CELLS IN HG:RELAX '
        print *,'NG = ',ng
        print *,'NRELAX = ',nnrelax
        stop
      endif
 
      do iter = 1, 2*nnrelax 

        extra_xlo = cvmgt(ng-iter,0,BCX_LO .eq. INTERIOR .or. BCX_LO .eq. PERIODIC)
        extra_ylo = cvmgt(ng-iter,0,BCY_LO .eq. INTERIOR .or. BCY_LO .eq. PERIODIC)
        extra_zlo = cvmgt(ng-iter,0,BCZ_LO .eq. INTERIOR .or. BCZ_LO .eq. PERIODIC)

        extra_xhi = cvmgt(ng-iter,0,BCX_HI .eq. INTERIOR .or. BCX_HI .eq. PERIODIC)
        extra_yhi = cvmgt(ng-iter,0,BCY_HI .eq. INTERIOR .or. BCY_HI .eq. PERIODIC)
        extra_zhi = cvmgt(ng-iter,0,BCZ_HI .eq. INTERIOR .or. BCZ_HI .eq. PERIODIC)

        do k = kstrt-extra_zlo,kend+extra_zhi
        do j = jstrt-extra_ylo,jend+extra_yhi
          iinc = mod(j+k+iter+1+extra_xlo+2*ng,2)
          do i = istrt-extra_xlo+iinc,iend+extra_xhi,2

            dgphi(i,j,k) =
     $         (coeff(i  ,j,k,1) * (phi(i+1,j,k) - phi(i,j,k)) +
     $          coeff(i-1,j,k,1) * (phi(i-1,j,k) - phi(i,j,k)) ) * facx +
     $         (coeff(i,j  ,k,2) * (phi(i,j+1,k) - phi(i,j,k)) +
     $          coeff(i,j-1,k,2) * (phi(i,j-1,k) - phi(i,j,k)) ) * facy +
     $         (coeff(i,j,k  ,3) * (phi(i,j,k+1) - phi(i,j,k)) +
     $          coeff(i,j,k-1,3) * (phi(i,j,k-1) - phi(i,j,k)) ) * facz


              rfac =  (coeff(i,j,k,1) + coeff(i-1,j,k,1)) * facx +
     $                (coeff(i,j,k,2) + coeff(i,j-1,k,2)) * facy +
     $                (coeff(i,j,k,3) + coeff(i,j,k-1,3)) * facz 

              mult = one 
              mult = cvmgt(two*mult,mult,
     $                     i.eq.is   .and. (BCX_LO.eq.WALL.or.BCX_LO.eq.INLET))
              mult = cvmgt(two*mult,mult,
     $                     j.eq.js   .and. (BCY_LO.eq.WALL.or.BCY_LO.eq.INLET))
              mult = cvmgt(two*mult,mult,
     $                     k.eq.ks   .and. (BCZ_LO.eq.WALL.or.BCZ_LO.eq.INLET))
              mult = cvmgt(two*mult,mult,
     $                     i.eq.ie+1 .and. (BCX_HI.eq.WALL.or.BCX_HI.eq.INLET))
              mult = cvmgt(two*mult,mult,
     $                     j.eq.je+1 .and. (BCY_HI.eq.WALL.or.BCY_HI.eq.INLET))
              mult = cvmgt(two*mult,mult,
     $                     k.eq.ke+1 .and. (BCZ_HI.eq.WALL.or.BCZ_HI.eq.INLET))

              dgphi(i,j,k) = mult*dgphi(i,j,k)
              rfac = mult * rfac

              rfac = one/rfac

              phi(i,j,k) = phi(i,j,k) + rfac*(dgphi(i,j,k) - source(i,j,k))

        enddo
        enddo
        enddo

      enddo

      return
      end

c *************************************************************************
c ** COARSIG **
c ** Coarsening of the sig coefficients
c *************************************************************************

      subroutine FORT_COARSIG(sig,sigc,DIMS,CDIMS,ng)

      implicit none

      integer DIMS
      integer CDIMS
      integer ng
      REAL_T  sig(lo_1 -ng:hi_1 +ng,lo_2 -ng:hi_2 +ng,lo_3 -ng:hi_3 +ng)
      REAL_T sigc(loc_1-ng:hic_1+ng,loc_2-ng:hic_2+ng,loc_3-ng:hic_3+ng)

c     Local variables
      integer i ,j, k

      do k = lo_3,hi_3,2
      do j = lo_2,hi_2,2
      do i = lo_1,hi_1,2
        sigc(i/2,j/2,k/2) = ( sig(i  ,j,k  ) + sig(i  ,j+1,k  )+ 
     $                        sig(i+1,j,k  ) + sig(i+1,j+1,k  )+
     $                        sig(i  ,j,k+1) + sig(i  ,j+1,k+1)+
     $                        sig(i+1,j,k+1) + sig(i+1,j+1,k+1) ) * eighth

      enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** RESTRICT **
c ** Conservative restriction of the residual
c *************************************************************************

      subroutine FORT_RESTRICT(res,resc,DIMS,CDIMS,bc,ng)

      implicit none

      integer DIMS
      integer CDIMS
      integer ng
      REAL_T   res(lo_1 -ng:hi_1 +1+ng,lo_2 -ng:hi_2 +1+ng,lo_3 -ng:hi_3 +1+ng)
      REAL_T  resc(loc_1-ng:hic_1+1+ng,loc_2-ng:hic_2+1+ng,loc_3-ng:hic_3+1+ng)
      integer bc(2,3)

c     Local variables
      integer i,j,k,ii,jj,kk
      integer istrt,jstrt,kstrt
      integer iend,jend,kend
      REAL_T cen, faces, sides, corners

      istrt = cvmgt((lo_1  )/2+1,(lo_1  )/2,BCX_LO .eq. OUTLET)
      iend  = cvmgt((hi_1+1)/2-1,(hi_1+1)/2,BCX_HI .eq. OUTLET)

      jstrt = cvmgt((lo_2  )/2+1,(lo_2  )/2,BCY_LO .eq. OUTLET)
      jend  = cvmgt((hi_2+1)/2-1,(hi_2+1)/2,BCY_HI .eq. OUTLET)

      kstrt = cvmgt((lo_3  )/2+1,(lo_3  )/2,BCZ_LO .eq. OUTLET)
      kend  = cvmgt((hi_3+1)/2-1,(hi_3+1)/2,BCZ_HI .eq. OUTLET)

      if (BCX_LO .eq. WALL .or. BCX_LO .eq. INLET) then
        do k = lo_3-1,hi_3+2
        do j = lo_2-1,hi_2+2
          res(lo_1-1,j,k) = res(lo_1+1,j,k)
        enddo
        enddo
      endif
      if (BCX_HI .eq. WALL .or. BCX_HI .eq. INLET) then
        do k = lo_3-1,hi_3+2
        do j = lo_2-1,hi_2+2
          res(hi_1+2,j,k) = res(hi_1  ,j,k)
        enddo
        enddo
      endif

      if (BCY_LO .eq. WALL .or. BCY_LO .eq. INLET) then
        do k = lo_3-1,hi_3+2
        do i = lo_1-1,hi_1+2
          res(i,lo_2-1,k) = res(i,lo_2+1,k)
        enddo
        enddo
      endif
      if (BCY_HI .eq. WALL .or. BCY_HI .eq. INLET) then
        do k = lo_3-1,hi_3+2
        do i = lo_1-1,hi_1+2
          res(i,hi_2+2,k) = res(i,hi_2  ,k)
        enddo
        enddo
      endif

      if (BCZ_LO .eq. WALL .or. BCZ_LO .eq. INLET) then
        do j = lo_2-1,hi_2+2
        do i = lo_1-1,hi_1+2
          res(i,j,lo_3-1) = res(i,j,lo_3+1)
        enddo
        enddo
      endif
      if (BCZ_HI .eq. WALL .or. BCZ_HI .eq. INLET) then
        do j = lo_2-1,hi_2+2
        do i = lo_1-1,hi_1+2
          res(i,j,hi_3+2) = res(i,j,hi_3  )
        enddo
        enddo
      endif

      do k = kstrt,kend
      do j = jstrt,jend
      do i = istrt,iend

          ii = 2*i
          jj = 2*j
          kk = 2*k

          cen = res(ii,jj,kk)

          faces = res(ii+1,jj,kk) + res(ii-1,jj,kk) +
     $            res(ii,jj+1,kk) + res(ii,jj-1,kk) +
     $            res(ii,jj,kk+1) + res(ii,jj,kk-1)

          sides = res(ii+1,jj+1,kk) + res(ii-1,jj+1,kk) +
     $            res(ii+1,jj-1,kk) + res(ii-1,jj-1,kk) +
     $            res(ii+1,jj,kk+1) + res(ii-1,jj,kk+1) +
     $            res(ii+1,jj,kk-1) + res(ii-1,jj,kk-1) +
     $            res(ii,jj+1,kk+1) + res(ii,jj-1,kk+1) +
     $            res(ii,jj+1,kk-1) + res(ii,jj-1,kk-1)

          corners = res(ii+1,jj+1,kk+1) + res(ii-1,jj+1,kk+1) +
     $              res(ii+1,jj-1,kk+1) + res(ii-1,jj-1,kk+1) +
     $              res(ii+1,jj+1,kk-1) + res(ii-1,jj+1,kk-1) +
     $              res(ii+1,jj-1,kk-1) + res(ii-1,jj-1,kk-1)

          resc(i,j,k) = eighth * (cen + half   * faces + 
     $                                  fourth * sides + 
     $                                  eighth * corners)

      enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** INTERP **
c ** Simple bilinear interpolation
c *************************************************************************

      subroutine FORT_INTERP(phi,deltac,DIMS,CDIMS,ng)

      implicit none

      integer DIMS
      integer CDIMS
      integer ng
      REAL_T     phi(lo_1 -ng:hi_1 +1+ng,lo_2 -ng:hi_2 +1+ng,lo_3 -ng:hi_3 +1+ng)
      REAL_T  deltac(loc_1-ng:hic_1+1+ng,loc_2-ng:hic_2+1+ng,loc_3-ng:hic_3+1+ng)

c     Local variables
      integer i,j,k

      do k = lo_3,hi_3+1,2
      do j = lo_2,hi_2+1,2
      do i = lo_1,hi_1+1,2
          phi(i,j,k) = phi(i,j,k) + deltac(i/2,j/2,k/2)
      enddo
      enddo
      enddo

      do k = lo_3+1,hi_3  ,2
      do j = lo_2  ,hi_2+1,2
      do i = lo_1  ,hi_1+1,2
        phi(i,j,k) = phi(i,j,k) + half*(deltac(i/2,j/2,k/2)+deltac(i/2,j/2,k/2+1))
      enddo
      enddo
      enddo

      do k = lo_3  ,hi_3+1,2
      do j = lo_2+1,hi_2  ,2
      do i = lo_1  ,hi_1+1,2
        phi(i,j,k) = phi(i,j,k) + half*(deltac(i/2,j/2,k/2)+deltac(i/2,j/2+1,k/2))
      enddo
      enddo
      enddo

      do k = lo_3  ,hi_3+1,2
      do j = lo_2  ,hi_2+1,2
      do i = lo_1+1,hi_1  ,2
        phi(i,j,k) = phi(i,j,k) + half*(deltac(i/2,j/2,k/2)+deltac(i/2+1,j/2,k/2))
      enddo
      enddo
      enddo

      do k = lo_3  ,hi_3+1,2
      do j = lo_2+1,hi_2,2
      do i = lo_1+1,hi_1,2
        phi(i,j,k) = phi(i,j,k) + 
     $     fourth*(deltac(i/2,j/2  ,k/2)+deltac(i/2+1,j/2  ,k/2)
     $            +deltac(i/2,j/2+1,k/2)+deltac(i/2+1,j/2+1,k/2))
      enddo
      enddo
      enddo

      do k = lo_3+1,hi_3  ,2
      do j = lo_2  ,hi_2+1,2
      do i = lo_1+1,hi_1  ,2
        phi(i,j,k) = phi(i,j,k) + 
     $     fourth*(deltac(i/2,j/2,k/2  )+deltac(i/2+1,j/2,k/2  )
     $            +deltac(i/2,j/2,k/2+1)+deltac(i/2+1,j/2,k/2+1))
      enddo
      enddo
      enddo

      do k = lo_3+1,hi_3  ,2
      do j = lo_2+1,hi_2  ,2
      do i = lo_1  ,hi_1+1,2
        phi(i,j,k) = phi(i,j,k) + 
     $     fourth*(deltac(i/2,j/2,k/2  )+deltac(i/2,j/2+1,k/2  )
     $            +deltac(i/2,j/2,k/2+1)+deltac(i/2,j/2+1,k/2+1))
      enddo
      enddo
      enddo

      do k = lo_3+1,hi_3,2
      do j = lo_2+1,hi_2,2
      do i = lo_1+1,hi_1,2
        phi(i,j,k) = phi(i,j,k) + 
     $     eighth*(deltac(i/2  ,j/2,k/2  )+deltac(i/2  ,j/2+1,k/2  )
     $            +deltac(i/2  ,j/2,k/2+1)+deltac(i/2  ,j/2+1,k/2+1)
     $            +deltac(i/2+1,j/2,k/2  )+deltac(i/2+1,j/2+1,k/2  )
     $            +deltac(i/2+1,j/2,k/2+1)+deltac(i/2+1,j/2+1,k/2+1))
      enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** MAKEDGPHI **
c ** Compute D(sig G(phi))
c *************************************************************************

      subroutine makedgphi(phi,dgphi,coeff,DIMS,dx,bc,ng,rlo_1,rlo_2,rlo_3,
     $                                                   rhi_1,rhi_2,rhi_3)

      implicit none

      integer DIMS
      integer ng
      integer rlo_1,rlo_2,rlo_3
      integer rhi_1,rhi_2,rhi_3
      REAL_T    phi(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T  dgphi(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T  coeff(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng,3)
      REAL_T  dx(3)
      integer bc(2,3)

c     Local variables
      REAL_T facx, facy, facz
      REAL_T mult
      integer istart,iend
      integer jstart,jend
      integer kstart,kend
      integer i,j,k

      facx = one / (dx(1)*dx(1))
      facy = one / (dx(2)*dx(2))
      facz = one / (dx(3)*dx(3))

      istart = cvmgt(lo_1+1,rlo_1,BCX_LO .eq. OUTLET)
      jstart = cvmgt(lo_2+1,rlo_2,BCY_LO .eq. OUTLET)
      kstart = cvmgt(lo_3+1,rlo_3,BCZ_LO .eq. OUTLET)
      iend   = cvmgt(hi_1  ,rhi_1,BCX_HI .eq. OUTLET)
      jend   = cvmgt(hi_2  ,rhi_2,BCY_HI .eq. OUTLET)
      kend   = cvmgt(hi_3  ,rhi_3,BCZ_HI .eq. OUTLET)

      do k = kstart,kend
      do j = jstart,jend
      do i = istart,iend

            dgphi(i,j,k) = 
     $         (coeff(i  ,j,k,1) * (phi(i+1,j,k) - phi(i,j,k)) +
     $          coeff(i-1,j,k,1) * (phi(i-1,j,k) - phi(i,j,k)) ) * facx +
     $         (coeff(i,j  ,k,2) * (phi(i,j+1,k) - phi(i,j,k)) +
     $          coeff(i,j-1,k,2) * (phi(i,j-1,k) - phi(i,j,k)) ) * facy +
     $         (coeff(i,j,k  ,3) * (phi(i,j,k+1) - phi(i,j,k)) +
     $          coeff(i,j,k-1,3) * (phi(i,j,k-1) - phi(i,j,k)) ) * facz

            mult = one 
            mult = cvmgt(two*mult,mult,
     $                   i.eq.lo_1   .and. (BCX_LO.eq.WALL.or.BCX_LO.eq.INLET))
            mult = cvmgt(two*mult,mult,
     $                   j.eq.lo_2   .and. (BCY_LO.eq.WALL.or.BCY_LO.eq.INLET))
            mult = cvmgt(two*mult,mult,
     $                   k.eq.lo_3   .and. (BCZ_LO.eq.WALL.or.BCZ_LO.eq.INLET))
            mult = cvmgt(two*mult,mult,
     $                   i.eq.hi_1+1 .and. (BCX_HI.eq.WALL.or.BCX_HI.eq.INLET))
            mult = cvmgt(two*mult,mult,
     $                   j.eq.hi_2+1 .and. (BCY_HI.eq.WALL.or.BCY_HI.eq.INLET))
            mult = cvmgt(two*mult,mult,
     $                   k.eq.hi_3+1 .and. (BCZ_HI.eq.WALL.or.BCZ_HI.eq.INLET))

            dgphi(i,j,k) = mult*dgphi(i,j,k)

      enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** SOLVEHG **
c *************************************************************************

      subroutine FORT_SOLVEHG(dest,dest0,source,coeff,sum,r,w,z,work,
     $                        DIMS,dx,bc,
     $                        maxiter,norm,prob_norm,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T   dest(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T  dest0(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T source(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T  coeff(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng,3)
      REAL_T    sum(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T      r(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T      w(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T      z(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T   work(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T dx(3)
      integer bc(2,3)
      integer maxiter
      REAL_T norm
      REAL_T prob_norm
      REAL_T sum0

c     Local variables
      REAL_T  factor
      REAL_T  alpha
      REAL_T  beta
      REAL_T  rho
      REAL_T  rhol
      REAL_T  local_norm
      REAL_T  tol, tolfac
      integer i,j,k
      integer iter
      integer istrt,iend
      integer jstrt,jend
      integer kstrt,kend

      tolfac = 1.0d-3

      istrt = cvmgt(lo_1+1,lo_1  ,BCX_LO .eq. OUTLET)
      iend  = cvmgt(hi_1  ,hi_1+1,BCX_HI .eq. OUTLET)
      jstrt = cvmgt(lo_2+1,lo_2  ,BCY_LO .eq. OUTLET)
      jend  = cvmgt(hi_2  ,hi_2+1,BCY_HI .eq. OUTLET)
      kstrt = cvmgt(lo_3+1,lo_3  ,BCZ_LO .eq. OUTLET)
      kend  = cvmgt(hi_3  ,hi_3+1,BCZ_HI .eq. OUTLET)

      call setperiodic(dest,DIMS,bc,ng)

      do k = lo_3-1,hi_3+2
      do j = lo_2-1,hi_2+2
         do i = lo_1-1,hi_1+2
            dest0(i,j,k) = dest(i,j,k)
            dest(i,j,k) = zero
         enddo
      enddo
      enddo

  10  call makedgphi(dest0,w,coeff,DIMS,dx,bc,ng,istrt,jstrt,kstrt,iend,jend,kend)

      rho = zero
      norm = zero

      do k = kstrt, kend 
      do j = jstrt, jend 
        do i = istrt, iend 
          r(i,j,k) = source(i,j,k) - w(i,j,k)
        enddo
      enddo
      enddo

      local_norm = zero
      do k = kstrt, kend 
      do j = jstrt, jend 
        do i = istrt, iend 
          factor = one
          factor = cvmgt(half*factor,factor,i.eq.lo_1 .or. i.eq.hi_1+1)
          factor = cvmgt(half*factor,factor,j.eq.lo_2 .or. j.eq.hi_2+1)
          factor = cvmgt(half*factor,factor,k.eq.lo_3 .or. k.eq.hi_3+1)
          local_norm  = max(local_norm, abs(r(i,j,k)))
          z(i,j,k) = r(i,j,k) / sum(i,j,k)
          rho    = rho + z(i,j,k) * r(i,j,k) * factor
          norm   = max(norm,abs(r(i,j,k)))
        enddo
      enddo
      enddo

      tol = Max(tolfac*local_norm,1.0d-15*prob_norm)
      if (norm .le. tol) return

      do k = kstrt, kend 
      do j = jstrt, jend 
        do i = istrt, iend 
          work(i,j,k) = zero
          dest(i,j,k) = z(i,j,k)
        enddo
      enddo
      enddo

      iter = 1
c     write(6,1000) iter, norm/prob_norm

 100  call setperiodic(dest,DIMS,bc,ng)

      call makedgphi(dest,w,coeff,DIMS,dx,bc,ng,istrt,jstrt,kstrt,iend,jend,kend)

      alpha = zero
      do k = kstrt, kend 
      do j = jstrt, jend 
        do i = istrt, iend 
          factor = one
          factor = cvmgt(half*factor,factor,i.eq.lo_1 .or. i.eq.hi_1+1)
          factor = cvmgt(half*factor,factor,j.eq.lo_2 .or. j.eq.hi_2+1)
          factor = cvmgt(half*factor,factor,k.eq.lo_3 .or. k.eq.hi_3+1)
          alpha  = alpha + dest(i,j,k)*w(i,j,k) * factor
        enddo
      enddo
      enddo

      alpha = rho / alpha
      rhol  = rho
      rho   = zero
      norm  = zero

      do k = kstrt, kend 
      do j = jstrt, jend 
        do i = istrt, iend 
          factor = one
          factor = cvmgt(half*factor,factor,i.eq.lo_1 .or. i.eq.hi_1+1)
          factor = cvmgt(half*factor,factor,j.eq.lo_2 .or. j.eq.hi_2+1)
          factor = cvmgt(half*factor,factor,k.eq.lo_3 .or. k.eq.hi_3+1)
          work(i,j,k) = work(i,j,k) + alpha * dest(i,j,k)
          r(i,j,k) = r(i,j,k) - alpha * w(i,j,k)
          z(i,j,k) = r(i,j,k) / sum(i,j,k)
          rho    = rho + z(i,j,k) * r(i,j,k) * factor
          norm   = max(norm,abs(r(i,j,k)))
        enddo
      enddo
      enddo

      iter = iter+1
c     write(6,1000) iter, norm/prob_norm

      if (norm .le. tol) then

        do k = kstrt, kend 
        do j = jstrt, jend 
          do i = istrt, iend 
            dest(i,j,k) = work(i,j,k) + dest0(i,j,k)
          enddo
        enddo
        enddo

      else if (iter .ge. maxiter .or. norm .ge. 100.d0*local_norm) then

        tolfac = 10.d0 * tolfac 
        iter = 1 
        do k = kstrt, kend 
        do j = jstrt, jend 
          do i = istrt, iend 
            dest(i,j,k) = zero
          enddo
        enddo
        enddo
        goto 10

      else

        beta = rho / rhol
        do k = kstrt, kend 
        do j = jstrt, jend 
          do i = istrt, iend 
            dest(i,j,k) = z(i,j,k) + beta * dest(i,j,k)
          enddo
        enddo
        enddo
        goto 100

      endif

1000  format('Res/Res0 in solve: ',i4,2x,e12.5)
c     call flush(6)

      return
      end

c *************************************************************************
c ** MAKESUM **
c *************************************************************************

      subroutine FORT_MAKESUM(sum,coeff,DIMS,dx,bc,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T    sum(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      REAL_T  coeff(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng,3)
      REAL_T  dx(3)
      integer bc(2,3)

c     Local variables
      integer i, j, k
      integer is, ie, js, je, ks, ke
      integer istrt,iend
      integer jstrt,jend
      integer kstrt,kend
      REAL_T  facx, facy, facz
      REAL_T  mult

      facx = one/(dx(1)*dx(1))
      facy = one/(dx(2)*dx(2))
      facz = one/(dx(3)*dx(3))

      is = lo_1
      js = lo_2
      ks = lo_3
      ie = hi_1
      je = hi_2
      ke = hi_3

      istrt = cvmgt(lo_1+1,lo_1  ,BCX_LO .eq. OUTLET)
      iend  = cvmgt(hi_1  ,hi_1+1,BCX_HI .eq. OUTLET)

      jstrt = cvmgt(lo_2+1,lo_2  ,BCY_LO .eq. OUTLET)
      jend  = cvmgt(hi_2  ,hi_2+1,BCY_HI .eq. OUTLET)

      kstrt = cvmgt(lo_3+1,lo_3  ,BCZ_LO .eq. OUTLET)
      kend  = cvmgt(hi_3  ,hi_3+1,BCZ_HI .eq. OUTLET)

      do k = kstrt, kend 
      do j = jstrt, jend 
      do i = istrt, iend 
        sum(i,j,k) = - (
     $              (coeff(i,j,k,1) + coeff(i-1,j,k,1)) * facx +
     $              (coeff(i,j,k,2) + coeff(i,j-1,k,2)) * facy +
     $              (coeff(i,j,k,3) + coeff(i,j,k-1,3)) * facz )
         mult = one 
         mult = cvmgt(two*mult,mult,
     $                i.eq.is   .and. (BCX_LO.eq.WALL.or.BCX_LO.eq.INLET))
         mult = cvmgt(two*mult,mult,
     $                j.eq.js   .and. (BCY_LO.eq.WALL.or.BCY_LO.eq.INLET))
         mult = cvmgt(two*mult,mult,
     $                k.eq.ks   .and. (BCZ_LO.eq.WALL.or.BCZ_LO.eq.INLET))
         mult = cvmgt(two*mult,mult,
     $                i.eq.ie+1 .and. (BCX_HI.eq.WALL.or.BCX_HI.eq.INLET))
         mult = cvmgt(two*mult,mult,
     $                j.eq.je+1 .and. (BCY_HI.eq.WALL.or.BCY_HI.eq.INLET))
         mult = cvmgt(two*mult,mult,
     $                     k.eq.ke+1 .and. (BCZ_HI.eq.WALL.or.BCZ_HI.eq.INLET))
         sum(i,j,k) = mult * sum(i,j,k)

      enddo
      enddo
      enddo

      return
      end

c *************************************************************************
c ** SETPERIODIC **
c  Impose periodic boundary conditions on the single grid data in the
c   conjugate gradient bottom solver.
c *************************************************************************

      subroutine setperiodic(dest,DIMS,bc,ng)

      implicit none

      integer DIMS
      integer ng
      REAL_T   dest(lo_1-ng:hi_1+1+ng,lo_2-ng:hi_2+1+ng,lo_3-ng:hi_3+1+ng)
      integer bc(2,3)

      integer i,j,k

      if (BCX_LO .eq. PERIODIC .and. BCX_HI .eq. PERIODIC) then
        do k = lo_3,hi_3+1 
        do j = lo_2,hi_2+1 
          dest(hi_1+2,j,k) = dest(lo_1+1,j,k)
          dest(lo_1-1,j,k) = dest(hi_1  ,j,k)
        enddo
        enddo
      endif

      if (BCY_LO .eq. PERIODIC .and. BCY_HI .eq. PERIODIC) then
        do k = lo_3,hi_3+1
        do i = lo_1,hi_1+1
          dest(i,hi_2+2,k) = dest(i,lo_2+1,k)
          dest(i,lo_2-1,k) = dest(i,hi_2  ,k)
        enddo
        enddo
      endif

      if (BCZ_LO .eq. PERIODIC .and. BCZ_HI .eq. PERIODIC) then
        do j = lo_2,hi_2+1
        do i = lo_1,hi_1+1
          dest(i,j,hi_3+2) = dest(i,j,lo_3+1)
          dest(i,j,lo_3-1) = dest(i,j,hi_3  )
        enddo
        enddo
      endif

      if (BCX_LO .eq. PERIODIC .and. BCX_HI .eq. PERIODIC .and.
     $    BCY_LO .eq. PERIODIC .and. BCY_HI .eq. PERIODIC) then
        do k = lo_3,hi_3+1
          dest(lo_1-1,lo_2-1,k) = dest(hi_1,hi_2,k)
          dest(lo_1-1,hi_2+2,k) = dest(hi_1,lo_2+1,k)
          dest(hi_1+2,lo_2-1,k) = dest(lo_1+1,hi_2,k)
          dest(hi_1+2,hi_2+2,k) = dest(lo_1+1,lo_2+1,k)
        enddo
      endif

      if (BCX_LO .eq. PERIODIC .and. BCX_HI .eq. PERIODIC .and.
     $    BCZ_LO .eq. PERIODIC .and. BCZ_HI .eq. PERIODIC) then
        do j = lo_2,hi_2+1
          dest(lo_1-1,j,lo_3-1) = dest(hi_1,j,hi_3)
          dest(lo_1-1,j,hi_3+2) = dest(hi_1,j,lo_3+1)
          dest(hi_1+2,j,lo_3-1) = dest(lo_1+1,j,hi_3)
          dest(hi_1+2,j,hi_3+2) = dest(lo_1+1,j,lo_3+1)
        enddo
      endif

      if (BCY_LO .eq. PERIODIC .and. BCY_HI .eq. PERIODIC .and.
     $    BCZ_LO .eq. PERIODIC .and. BCZ_HI .eq. PERIODIC) then
        do i = lo_1,hi_1+1
          dest(i,lo_2-1,lo_3-1) = dest(i,hi_2,hi_3)
          dest(i,lo_2-1,hi_3+2) = dest(i,hi_2,lo_3+1)
          dest(i,hi_2+2,lo_3-1) = dest(i,lo_2+1,hi_3)
          dest(i,hi_2+2,hi_3+2) = dest(i,lo_2+1,lo_3+1)
        enddo
      endif

      if (BCX_LO .eq. PERIODIC .and. BCX_HI .eq. PERIODIC .and.
     &    BCY_LO .eq. PERIODIC .and. BCY_HI .eq. PERIODIC .and.
     &    BCZ_LO .eq. PERIODIC .and. BCZ_HI .eq. PERIODIC) then
          dest(lo_1-1,lo_1-1,lo_3-1) = dest(hi_1  ,hi_2  ,hi_3)
          dest(hi_1+2,lo_1-1,lo_3-1) = dest(lo_1+1,hi_2  ,hi_3)
          dest(lo_1-1,hi_2+2,lo_3-1) = dest(hi_1  ,lo_1+1,hi_3)
          dest(hi_1+2,hi_2+2,lo_3-1) = dest(lo_1+1,lo_1+1,hi_3)
          dest(lo_1-1,lo_1-1,hi_3+2) = dest(hi_1  ,hi_2  ,lo_3+1)
          dest(hi_1+2,lo_1-1,hi_3+2) = dest(lo_1+1,hi_2  ,lo_3+1)
          dest(lo_1-1,hi_2+2,hi_3+2) = dest(hi_1  ,lo_1+1,lo_3+1)
          dest(hi_1+2,hi_2+2,hi_3+2) = dest(lo_1+1,lo_1+1,lo_3+1)
      endif

      return
      end
