/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


#ifndef _MASK_H_
#define _MASK_H_
//
// $Id: Mask.H,v 1.5 2001/08/01 21:51:04 lijewski Exp $
//

#include <BaseFab.H>
#include <FArrayBox.H>

//@Man:
/*@Memo:
  A Mask is a derived Fab class whose elements maintain a logical
  "TRUE" or "FALSE" state.  A Mask can perform several elementary
  logical operations on its data members.
*/        
/*@Doc:
        Implement a logical class Mask to maintain the TRUE or FALSE state
        of elements in a box.  A Mask can perform element-wise logical
        operations, but is otherwise very similar to an FArrayBox object.
        A Mask object can set its components based on comparing (a subregion
        of a) real-valued FArrayBox element magnitudes to a scalar value, or
        by "and", "or" operations with (the a subregion of) other Masks.
        
        This class does NOT provide a copy constructor or assignment operator.
*/

class Mask
    :
    public BaseFab<int>
{
public:
    //
    //@ManDoc: default constructor
    //
    Mask ()
        :
        BaseFab<int>() {}
    //
    //@ManDoc: allocating constructor (nc=number of components)
    //
    Mask (const Box& bx,
          int        nc = 1)
        :
        BaseFab<int>(bx,nc) {}
    //
    //@ManDoc: construct from input stream
    //
    Mask (std::istream& is);
    //
    //@ManDoc: destructor
    //
    virtual ~Mask () {}
    //
    //@ManDoc: Initialize from stream.
    //
    friend std::istream& operator>> (std::istream&, Mask&);
    //
    //@ManDoc: initialize from stream, FAB-style
    //
    void readFrom (std::istream&);
    //
    //@ManDoc: output to stream
    //
    friend std::ostream& operator<< (std::ostream&, const Mask&);
    //
    //@ManDoc: output to stream, FAB-style
    //
    void writeOn (std::ostream&) const;
    //
    //@ManDoc: in-place And operator
    //
    Mask& operator&= (const Mask& src) { return And(src); }
    //
    //@ManDoc: in-place And
    //
    Mask& And (const Mask& src);
    //
    //@ManDoc: as above, but specify source/destination/number of components
    //
    Mask& And (const Mask& src,
               int         srccomp,
               int         destcomp,
               int         numcomp = 1);
    //
    //@ManDoc: as above, and specify subregion
    //
    Mask& And (const Mask& src,
               const Box&  subbox,
               int         srccomp,
               int         destcomp,
               int         numcomp = 1);
    //
    //@ManDoc: in-pace And, over source components to destination components,
    //  and on intersection of source and destination boxes
    //
    Mask& And (const Mask& src,
               const Box&  srcbox,
               const Box&  destbox,
               int         srccomp,
               int         destcomp,
               int         numcomp = 1);
    //
    //@ManDoc: in-place Or operator
    //
    Mask& operator|= (const Mask& src) { return Or(src); }
    //
    //@ManDoc: in-place Or
    //
    Mask& Or (const Mask& src);
    //
    //@ManDoc: as above, but specify source/destination/number of components
    //
    Mask& Or (const Mask& src,
              int         srccomp,
              int         destcomp,
              int         numcomp = 1);
    //
    //@ManDoc: as above, and specify subregion
    //
    Mask& Or (const Mask& src,
              const Box&  subbox,
              int         srccomp,
              int         destcomp,
              int         numcomp = 1);
    //
    //@ManDoc: in-pace Or, over source components to destination components,
    //  and on intersection of source and destination boxes
    //
    Mask& Or (const Mask& src,
              const Box&  srcbox,
              const Box&  destbox,
              int         srccomp,
              int         destcomp,
              int         numcomp = 1);
private:
    //
    // Disallow copy constructor and operator.
    //
    Mask& operator= (const Mask&);
    Mask (const Mask&);
};

#endif /*_MASK_H_*/
