/*
 * Copyright (C) 2009 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <iostream>
#include <sstream>
#include <boost/lexical_cast.hpp>

#include "Wt/Dbo/DbAction"
#include "Wt/Dbo/Exception"
#include "Wt/Dbo/Session"
#include "Wt/Dbo/SqlConnection"
#include "Wt/Dbo/SqlStatement"

namespace Wt {
  namespace Dbo {

InitSchema::InitSchema(Session& session, Session::MappingInfo& mapping)
  : session_(session),
    mapping_(mapping),
    idField_(false)
{ }

bool InitSchema::getsValue() const { return false; }
bool InitSchema::setsValue() const { return false; }
bool InitSchema::isSchema() const { return true; }

DropSchema::DropSchema(Session& session, const char *tableName,
		       std::set<std::string>& tablesDropped)
  : session_(session),
    tableName_(tableName),
    tablesDropped_(tablesDropped)
{
  tablesDropped_.insert(tableName);
}

void DropSchema::drop(const std::string& table)
{
  tablesDropped_.insert(table);
  session_.connection(true)
    ->executeSql("drop table \"" + Impl::quoteSchemaDot(table) + "\"");
}

bool DropSchema::getsValue() const { return false; }
bool DropSchema::setsValue() const { return false; }
bool DropSchema::isSchema() const { return true; }

DboAction::DboAction()
  : dbo_(0),
    mapping_(0),
    setStatementIdx_(0),
    setIdx_(0)
{ }

DboAction::DboAction(MetaDboBase& dbo, Session::MappingInfo& mapping)
  : dbo_(&dbo),
    mapping_(&mapping),
    setStatementIdx_(0),
    setIdx_(0)
{ }

bool DboAction::getsValue() const { return false; }
bool DboAction::setsValue() const { return false; }
bool DboAction::isSchema() const { return false; }

LoadBaseAction::LoadBaseAction(MetaDboBase& dbo, Session::MappingInfo& mapping,
			       SqlStatement *statement, int& column)
  : DboAction(dbo, mapping),
    statement_(statement),
    column_(column)
{ }

void LoadBaseAction::start()
{
  if (mapping().versionFieldName) {
    int version;
    statement_->getResult(column_++, &version);
    dbo().setVersion(version);
  }
}

bool LoadBaseAction::setsValue() const { return true; }

SaveBaseAction::SaveBaseAction(SqlStatement *statement, int column)
  : DboAction(),
    statement_(statement),
    column_(column),
    bindNull_(false)
{
  pass_ = Self;
}

SaveBaseAction::SaveBaseAction(MetaDboBase& dbo, Session::MappingInfo& mapping,
			       SqlStatement *statement, int column)
  : DboAction(dbo, mapping),
    statement_(statement),
    column_(column),
    bindNull_(false)
{
  pass_ = Self;
}

void SaveBaseAction::startDependencyPass()
{
  pass_ = Dependencies;
}

void SaveBaseAction::startSelfPass()
{
  pass_ = Self;
  needSetsPass_ = false;

  statement_->reset();
  column_ = 0;

  if (mapping().versionFieldName)
    statement_->bind(column_++, dbo().version() + 1);
}

void SaveBaseAction::exec()
{
  statement_->execute();

  if (isInsert_ && mapping().surrogateIdFieldName)
    dbo().setAutogeneratedId(statement_->insertedId());

  dbo().setTransactionState(MetaDboBase::SavedInTransaction);
}

void SaveBaseAction::startSetsPass()
{
  pass_ = Sets;
}

bool SaveBaseAction::getsValue() const { return true; }

TransactionDoneAction::TransactionDoneAction(MetaDboBase& dbo,
					     Session::MappingInfo& mapping,
					     bool success)
  : DboAction(dbo, mapping),
    success_(success)
{ }

bool TransactionDoneAction::getsValue() const { return true; }

SessionAddAction::SessionAddAction(MetaDboBase& dbo,
				   Session::MappingInfo& mapping)
  : DboAction(dbo, mapping)
{ }

bool SessionAddAction::getsValue() const { return true; }
bool SessionAddAction::setsValue() const { return false; }
bool SessionAddAction::isSchema() const { return false; }

SetReciproceAction::SetReciproceAction(const std::string& joinName,
				       MetaDboBase *value)
  : joinName_(joinName),
    value_(value)
{ }

bool SetReciproceAction::getsValue() const { return false; }
bool SetReciproceAction::setsValue() const { return true; }
bool SetReciproceAction::isSchema() const { return false; }

ToAnysAction::ToAnysAction(std::vector<boost::any>& result)
  : result_(result)
{ }

bool ToAnysAction::getsValue() const { return true; }
bool ToAnysAction::setsValue() const { return false; }
bool ToAnysAction::isSchema() const { return false; }

FromAnyAction::FromAnyAction(int& index, const boost::any& value)
  : index_(index),
    value_(value)
{ }

bool FromAnyAction::getsValue() const { return false; }
bool FromAnyAction::setsValue() const { return true; }
bool FromAnyAction::isSchema() const { return false; }

  }
}
