/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <unistd.h> /* for getcwd() */

#ifdef USE_DYNAMIC
# ifdef HAVE_DLFCN_H
#  include <dlfcn.h>
# endif
# ifdef HAVE_DL_H
#  include <dl.h>
# endif
#endif

#include "wfoutmodule.h"
#include "module-info.cc" /* file generated by mkmodinfo.sh */
#include "defs.h"


/* portability -- HPUX support untested... RV@@0 */
#ifdef HPUX
# ifndef BIND_IMMEDIATE
#  define BIND_IMMEDIATE 0
# endif
# define DLOPEN(file)		((void *) shl_load((file), BIND_IMMEDIATE, 0))
# define DLCLOSE(handle)		(shl_unload((shl_t) (handle)))
# define DLSYM(handle, name, sym) (shl_findsym((shl_t *) (&(handle)), (name), \
				TYPE_PROCEDURE, (void *) (&(sym))) == 0 && sym != NULL)
# define DLERROR()		(strerror(errno))
#else /* not HPUX */
# ifndef RTLD_NOW
#  define RTLD_NOW 0
# endif
# define DLOPEN(file)		(dlopen((file), RTLD_NOW))
# define DLCLOSE(handle)		(dlclose((handle)))
# define DLSYM(handle, name, sym) (((sym) = dlsym((handle), (name))) != NULL)
# define DLERROR()		(dlerror())
#endif

/* define path correctly RV@@7 */
static string modulepath = "output_modules/ipchains/.libs:output_modules/netfilter/.libs:output_modules/text/.libs:output_modules/html/.libs:output_modules/xml/.libs";

typedef wf_outmodule* (*initfunc)(void);


wf_outmodule*
wf_outmodule_init(const string& name) {
  initfunc moduleinit;

  /* Try to instanciate the module if it is built-in (static). */

  /* The function new_module is defined in module-info.cc
     module-info.cc is generated by mkmodinfo.sh */

  /* first have a look in the static modules list */
  int modulesnbr = sizeof(outmodules) / sizeof(wf_outmodulelist);
  int i;
  for (i = 0; i < modulesnbr; i++) {
    if (outmodules[i].name == name) {
      moduleinit = outmodules[i].initfunc;
      return moduleinit();
    }
  }
  
  /* Wanted module is not built-in. Maybe dynamic (if supported)? */

#ifdef USE_DYNAMIC
  unsigned int pos, oldpos = 0;
  string fname = "libwf" + name + ".so";
  void* handle = NULL;
  do {
    pos = modulepath.find(':', oldpos);
    string path = modulepath.substr(oldpos, pos - oldpos);
    oldpos = pos + 1;

    if (path.empty())
      path = fname; /* lookup in LD_LIBRARY_PATH */
    else {
      if (path[0] != '/') {
	char* str = getcwd(NULL, 0);
	path = string(str) + '/' + path;
	free(str);
      }
      path += '/' + fname;
    }

    // cerr << path << endl;
    handle = DLOPEN(path.c_str());
    if (handle != NULL)
      break;
  } while (pos != string::npos);

  if (handle == NULL) {
    fprintf(stderr, _("Error: cannot load module `%s': %s.\n"),
	    name.c_str(), DLERROR());
    return NULL;
  }

  string symname = "wf_outmodule_" + name + "_init";
  void* symaddr;
  // cerr << "sym: " << symname << endl;
  if (!DLSYM(handle, symname.c_str(), symaddr)) {
    fprintf(stderr, _("Error: cannot load module `%s': %s.\n"), name.c_str(),
	    DLERROR());
    DLCLOSE(handle);
    return NULL;
  }

  moduleinit = (initfunc)symaddr;
  return moduleinit();
#else
  /* Dynamic modules unsupported, and wanted module is not built-in. */
  return NULL;
#endif
}

ostream&
wf_outmodule_available_modules_print(ostream& os) {
  int modulesnbr = sizeof(outmodules) / sizeof(wf_outmodulelist);

  if (modulesnbr) {
    int i;
    os << _("Built-in modules: ");
    for (i = 0; i < modulesnbr; i++)
      os << outmodules[i].name << ' ';
    os << endl;
  }
  else
    os << _("No built-in modules.") << endl;

#ifdef USE_DYNAMIC
  os << _("Dynamic loading of modules supported.") << endl;

#if 0
  os << _("Available dynamic modules: ");
  /*
    Here we should do the following: RV@@4
    Through modulepath directories, try to open files of the desired form
    `libwf*.so'.  If open succeeds, try to load them to see if they are
    functional then unload them.
  */
  os << endl;
#endif

#else
  os << _("Dynamic loading of modules not supported.") << endl;
#endif       

  return os;
}

bool
wf_outmodule::print(const wf_logentries& logentries, ostream& os) {
  list<wf_logentry*>::const_iterator first = logentries.elems.begin(),
    last = logentries.elems.end();
  for (; first != last; ++first) {
    if (print(*first, os) == false)
      return false;
    os << endl;
  }
  return true;
}
