# -*- coding: utf-8 -*-

# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel.backend import (
    DEVICE_TYPE_COMPUTER,
    DEVICE_TYPE_PHONE,
)
from ubuntuone.controlpanel import cache
from ubuntuone.controlpanel.gui import DEVICE_CONFIRM_REMOVE
from ubuntuone.controlpanel.gui.qt import (
    handle_errors,
    icon_from_name,
    pixmap_from_name,
)
from ubuntuone.controlpanel.gui.qt.ui import device_ui
from ubuntuone.controlpanel.logger import setup_logging

COMPUTER_ICON = "computer"
PHONE_ICON = "phone"
DEFAULT_ICON = COMPUTER_ICON

DEVICE_TYPE_TO_ICON_MAP = {
    DEVICE_TYPE_COMPUTER: COMPUTER_ICON,
    DEVICE_TYPE_PHONE: PHONE_ICON,
}

CANCEL = QtGui.QMessageBox.Cancel
NO = QtGui.QMessageBox.No
YES = QtGui.QMessageBox.Yes


logger = setup_logging('qt.device')


def icon_name_from_type(device_type):
    """Get the icon name for the device."""
    icon_name = DEVICE_TYPE_TO_ICON_MAP.get(device_type, DEFAULT_ICON)
    return icon_name


class DeviceWidget(cache.Cache, QtGui.QWidget):
    """The widget for each device in the control panel."""

    removed = QtCore.pyqtSignal()
    removeCanceled = QtCore.pyqtSignal()

    def __init__(self, device_id, *args, **kwargs):
        """Initialize the UI of the widget."""
        super(DeviceWidget, self).__init__(*args, **kwargs)
        self.ui = device_ui.Ui_Form()
        self.ui.setupUi(self)
        self.id = device_id

    def update_device_info(self, device_info):
        """Update the device info."""
        self.ui.device_name_label.setText(device_info["name"])
        icon_name = icon_name_from_type(device_info["type"])
        pixmap = pixmap_from_name(icon_name)
        self.ui.device_icon_label.setPixmap(pixmap)

    @QtCore.pyqtSlot()
    @handle_errors(logger=logger)
    @defer.inlineCallbacks
    def on_remove_device_button_clicked(self):
        """The user wants to remove this device."""
        msg = DEVICE_CONFIRM_REMOVE
        buttons = YES | NO
        response = QtGui.QMessageBox.warning(self, '', msg, buttons, NO)

        if response == YES:
            yield self.backend.remove_device(device_id=self.id)
            self.removed.emit()
        else:
            self.removeCanceled.emit()


def get_device_for_list_widget(device_info):
    """Return a QListWidgetItem representing a device with the proper info."""
    text = device_info["name"]
    icon_name = icon_name_from_type(device_info["type"])
    icon = icon_from_name(icon_name)
    item = QtGui.QListWidgetItem(icon, text)
    return item
