# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for utility code that's common between darwin and windows."""

from twisted.internet import defer

try:
    from configparser import SafeConfigParser
except ImportError:
    from ConfigParser import SafeConfigParser

from ubuntuone.controlpanel import utils
from ubuntuone.controlpanel.tests import TestCase
from ubuntuone.controlpanel.utils import common
assert(common)

UPDATE_KEY = 'Update'
URL_KEY = 'url'
TEST_URL = 'test-url'
VERSION_ID_KEY = 'version_id'
TEST_VERSION_ID = '100'
CHANNEL_KEY = 'channel'
TEST_CHANNEL = 'test-channel'
CHECK_KEY = 'check_for_updates'
TEST_CHECK = '1'


class GetPathsTestCase(TestCase):
    """Test case for get_update_config and get_bin_cmd."""

    TEST_EXE_NAME = 'test-exe'
    TEST_BIN_CMD_NAME = 'test-bin-cmd-name'
    TEST_PYTHON_BIN_NAME = 'test-python-bin-name'

    def _run_get_bin_cmd(self, frozen):
        """Helper func to test get_bin_cmd."""
        self.patch(utils.common, "get_program_path",
                   lambda x, **kwargs: self.TEST_BIN_CMD_NAME)
        self.patch(utils.common, '__file__',
                   "fake/path/to/utils/common.py")
        return utils.common.get_bin_cmd(self.TEST_EXE_NAME)

    def test_get_bin_cmd_frozen(self):
        """Test that we append 'python' from procutils when frozen."""
        self.patch(utils.common.procutils, "which",
                   lambda x: [self.TEST_PYTHON_BIN_NAME])
        rv = self._run_get_bin_cmd(True)
        self.assertEqual(rv, [self.TEST_PYTHON_BIN_NAME,
                              self.TEST_BIN_CMD_NAME])


class GetUpdateConfigTestCase(TestCase):
    """Test reading the config file"""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(GetUpdateConfigTestCase, self).setUp()

        self.blank_conf = SafeConfigParser()
        self.good_conf = SafeConfigParser()
        self.good_conf.add_section(UPDATE_KEY)
        self.good_conf.set(UPDATE_KEY, URL_KEY, TEST_URL)
        self.good_conf.set(UPDATE_KEY, VERSION_ID_KEY,
                           TEST_VERSION_ID)
        self.good_conf.set(UPDATE_KEY, CHANNEL_KEY,
                           TEST_CHANNEL)
        self.good_conf.set(UPDATE_KEY, CHECK_KEY, TEST_CHECK)

        self.patch(utils.common, 'load_config_paths',
                   lambda x: 'test/confpath')

    def test_cant_read(self):
        """Test that we return no_config when we can't read the cfg"""
        rv = utils.common.get_update_config()
        self.assertEqual(rv, (None, None, None, False))

    def test_read_bad_file(self):
        """Test that we return no_config when we read a bad cfg file"""
        self.patch(utils.common, "SafeConfigParser",
                   lambda: self.blank_conf)

        self.patch(self.blank_conf, "read", lambda path: True)
        rv = utils.common.get_update_config()
        self.assertEqual(rv, (None, None, None, False))

    def test_read_good_file(self):
        """test that we correctly read a good cfg file"""
        self.patch(utils.common, "SafeConfigParser",
                   lambda: self.good_conf)

        self.patch(self.good_conf, "read", lambda path: True)
        rv = utils.common.get_update_config()
        self.assertEqual(rv,
                         (TEST_URL, TEST_VERSION_ID, TEST_CHANNEL, 1))


class CallUpdaterTestCase(TestCase):
    """Test calling the updater."""

    UPDATE_BIN_CMD = ['path/to/updater']
    BUILDOUT_UPDATE_BIN_CMD = ['python', 'path/to/updater']

    @defer.inlineCallbacks
    def setUp(self):
        """Set up common data"""
        yield super(CallUpdaterTestCase, self).setUp()

        self.getProcessValueArgs = []
        self.custom_call_args = []

        self.call_retval = 0

        def fake_getProcessValue(exe, args):
            self.getProcessValueArgs.append((exe, args))
            return self.call_retval

        self.patch(utils.common, 'getProcessValue',
                   fake_getProcessValue)

    def fake_custom_call(self, exe, args):
        """Mock custom call"""
        self.custom_call_args.append((exe, args))

    @defer.inlineCallbacks
    def _call_call_updater(self, should_check, custom_call, install, frozen):
        """Helper to test call_udpater"""

        self.patch(utils.common, 'get_update_config',
                   lambda: (TEST_URL, TEST_VERSION_ID, TEST_CHANNEL,
                            should_check))

        self.patch(utils.common, 'get_bin_cmd',
                   lambda x: self.UPDATE_BIN_CMD if frozen
                   else self.BUILDOUT_UPDATE_BIN_CMD)

        rv = yield utils.common.call_updater(custom_call, install)
        defer.returnValue(rv)

    @defer.inlineCallbacks
    def test_do_not_check(self):
        """Test that we return False when config says don't check """

        rv = yield self._call_call_updater(should_check=False,
                                           custom_call=None,
                                           install=False,
                                           frozen=False)
        self.assertEqual(rv, False)

    @defer.inlineCallbacks
    def test_no_updater_cmd(self):
        """Test that we return False when get_bin_cmd fails."""
        def raise_in_get_bin_cmd(name):
            raise OSError()

        self.patch(utils.common, 'get_bin_cmd', raise_in_get_bin_cmd)
        rv = yield utils.common.call_updater()

        self.assertEqual(rv, False)

    @defer.inlineCallbacks
    def test_update_check_frozen_nocustom_noinstall(self):
        """Test update check when frozen."""
        self.call_retval = 1
        rv = yield self._call_call_updater(should_check=True,
                                           custom_call=None,
                                           install=False,
                                           frozen=True)
        self.assertEqual(rv, True)
        self.assertEqual(self.getProcessValueArgs,
                         [(self.UPDATE_BIN_CMD[0],
                          ['--url', TEST_URL,
                          '--current', TEST_VERSION_ID,
                           '--release', TEST_CHANNEL])])

    @defer.inlineCallbacks
    def test_update_check_unfrozen_nocustom_noinstall(self):
        """Test update check when not frozen."""
        self.call_retval = 1
        rv = yield self._call_call_updater(should_check=True,
                                           custom_call=None,
                                           install=False,
                                           frozen=False)
        self.assertEqual(rv, True)
        self.assertEqual(self.getProcessValueArgs,
                         [(self.BUILDOUT_UPDATE_BIN_CMD[0],
                          [self.BUILDOUT_UPDATE_BIN_CMD[1],
                           '--url', TEST_URL,
                          '--current', TEST_VERSION_ID,
                           '--release', TEST_CHANNEL])])

    @defer.inlineCallbacks
    def test_update_check_nocustom_install(self):
        """Test that we send --install when install=True"""

        yield self._call_call_updater(should_check=True,
                                      custom_call=None,
                                      install=True,
                                      frozen=True)
        self.assertEqual(self.getProcessValueArgs,
                         [(self.UPDATE_BIN_CMD[0],
                          ['--url', TEST_URL,
                          '--current', TEST_VERSION_ID,
                           '--release', TEST_CHANNEL,
                           '--install'])])

    @defer.inlineCallbacks
    def test_update_check_custom_install(self):
        """Test that we send --install when install=True to the custom call"""

        yield self._call_call_updater(should_check=True,
                                      custom_call=self.fake_custom_call,
                                      install=True,
                                      frozen=True)
        self.assertEqual(self.custom_call_args,
                         [(self.UPDATE_BIN_CMD[0],
                          ['--url', TEST_URL,
                          '--current', TEST_VERSION_ID,
                           '--release', TEST_CHANNEL,
                           '--install'])])
