# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the Qt UI for the control panel for Ubuntu One."""

import logging

from PyQt4 import QtGui, QtCore
from twisted.internet import defer
from ubuntuone.devtools.handlers import MementoHandler

from ubuntuone.controlpanel import backend, cache
from ubuntuone.controlpanel.tests import TestCase, EXPECTED_ACCOUNT_INFO, TOKEN
from ubuntuone.controlpanel.gui import (
    qt,
    RECENTTRANSFERS,
    UPLOADING,
)
from ubuntuone.controlpanel.gui.tests import (
    FakedObject,
    FAKE_VOLUMES_INFO,
    USER_HOME,
)

# Attribute 'yyy' defined outside __init__, access to a protected member
# pylint: disable=W0201, W0212


SAMPLE_ACCOUNT_INFO = EXPECTED_ACCOUNT_INFO
SAMPLE_NAME = EXPECTED_ACCOUNT_INFO["name"]
SAMPLE_EMAIL = EXPECTED_ACCOUNT_INFO["email"]
SAMPLE_PLAN = EXPECTED_ACCOUNT_INFO["type"]

SAMPLE_COMPUTER_INFO = {
    "type": "Computer",
    "name": "desktop i5",
    "is_local": False,
    "configurable": False,
    "device_id": '1258-6854',
}

SAMPLE_PHONE_INFO = {
    "type": "Phone",
    "name": "nokia 1100",
    "is_local": False,
    "configurable": False,
    "device_id": '987456-2321',
}

SAMPLE_DEVICES_INFO = [
    {
        "type": "Computer",
        "name": "toshiba laptop",
        "is_local": True,
        "configurable": False,
        "device_id": '0000',
    },
    SAMPLE_COMPUTER_INFO,
    SAMPLE_PHONE_INFO,
]

SAMPLE_SETTINGS = {
    backend.AUTOCONNECT_KEY: True,
    backend.SHOW_ALL_NOTIFICATIONS_KEY: True,
    backend.SHARE_AUTOSUBSCRIBE_KEY: True,
    backend.UDF_AUTOSUBSCRIBE_KEY: True,
    backend.DOWNLOAD_KEY: 20480,
    backend.UPLOAD_KEY: 2048,
}

NO_OP = lambda *args: None


class FakeUi(FakedObject):
    """A fake Ui object."""

    exposed_methods = ['setupUi']
    raise_attr_error = False


class FakedControlPanelBackend(FakedObject):
    """Fake a Control Panel Backend."""

    ROOT_TYPE = u'ROOT'
    FOLDER_TYPE = u'UDF'
    SHARE_TYPE = u'SHARE'
    NAME_NOT_SET = u'ENAMENOTSET'
    FREE_BYTES_NOT_AVAILABLE = u'EFREEBYTESNOTAVAILABLE'
    DEFAULT_FILE_SYNC_SETTINGS = {
        backend.AUTOCONNECT_KEY: True,
        backend.SHOW_ALL_NOTIFICATIONS_KEY: True,
        backend.SHARE_AUTOSUBSCRIBE_KEY: False,
        backend.UDF_AUTOSUBSCRIBE_KEY: False,
        backend.DOWNLOAD_KEY: -1,  # no limit
        backend.UPLOAD_KEY: -1,  # no limit
    }

    next_result = []
    exposed_methods = [
        'account_info',
        'add_status_changed_handler',
        'build_signed_iri',
        'change_device_settings',
        'change_file_sync_settings',
        'change_public_access',
        'change_replication_settings',
        'change_volume_settings',
        'connect_files',
        'create_folder',
        'device_names_info',
        'devices_info',
        'disable_files',
        'disconnect_files',
        'enable_files',
        'file_sync_settings_info',
        'file_sync_status',
        'get_public_files',
        'login',
        'register',
        'remove_device',
        'replications_info',
        'restart_files',
        'restore_file_sync_settings',
        'search_files',
        'set_public_access_changed_handler',
        'set_public_access_change_error_handler',
        'set_public_files_list_handler',
        'shutdown',
        'start_files',
        'stop_files',
        'validate_path_for_folder',
        'volumes_info',
    ]
    exposed_results = {
        'account_info': SAMPLE_ACCOUNT_INFO,
        'devices_info': SAMPLE_DEVICES_INFO,
        'volumes_info': FAKE_VOLUMES_INFO,
        'file_sync_settings_info': SAMPLE_SETTINGS,
    }

    def get_credentials(self):
        """Fake credentials retrieval."""
        self._called['get_credentials'] = ((), {})
        return TOKEN

    def get_home_dir(self):
        """Fake home return."""
        return USER_HOME

    def build_signed_iri(self, iri):
        """Fake iri signing."""

    def sync_menu(self):
        """Fake sync_menu."""
        data = {}
        data[RECENTTRANSFERS] = []
        data[UPLOADING] = []
        return data

    def get_shares(self):
        """Fake get_shares."""
        return []


class CrashyBackendException(Exception):
    """A faked backend crash."""


class CrashyBackend(FakedControlPanelBackend):
    """A faked backend that crashes."""

    def __init__(self, *args, **kwargs):
        super(CrashyBackend, self).__init__(*args, **kwargs)
        for i in self.exposed_methods + ['get_credentials']:
            setattr(self, i, self._fail(i))

    def _fail(self, f):
        """Crash boom bang."""

        def inner(*args, **kwargs):
            """Raise a custom exception."""
            raise CrashyBackendException(f)

        return inner


class FakedDialog(QtGui.QMessageBox):
    """Fake a confirmation dialog."""

    properties = response = args = kwargs = None

    def __init__(self, *a, **kw):
        super(FakedDialog, self).__init__(*a, **kw)
        FakedDialog.args = a
        FakedDialog.kwargs = kw
        FakedDialog.properties = {'shown': 0}

    @classmethod
    def reset(cls, *a, **kw):
        """Clear all values."""
        cls.properties = cls.response = cls.args = cls.kwargs = None

    @classmethod
    def warning(cls, *a, **kw):
        """Simulate a warning message."""
        cls.args = a
        cls.kwargs = kw
        return cls.response

    @classmethod
    def question(cls, *a, **kw):
        """Simulate a question message."""
        cls.args = a
        cls.kwargs = kw
        return cls.response

    # Invalid name "setDetailedText", "setDefaultButton"
    # pylint: disable=C0103

    def setDetailedText(self, text):
        """Fake the setDetailedText."""
        super(FakedDialog, self).setDetailedText(text)
        FakedDialog.properties['detailed_text'] = text

    def setDefaultButton(self, button):
        """Fake the setDefaultButton."""
        super(FakedDialog, self).setDefaultButton(button)
        FakedDialog.properties['default_button'] = button

    # pylint: enable=C0103

    def show(self):
        """Fake show."""
        FakedDialog.properties['shown'] += 1
        self.finished.emit(0)

    def exec_(self):
        """Fake exec_."""
        FakedDialog.properties['shown'] += 1


class BaseTestCase(TestCase):
    """Base Test Case."""

    innerclass_ui = None
    innerclass_name = None
    class_ui = None
    kwargs = {}
    logger = None

    @defer.inlineCallbacks
    def setUp(self):
        cache.Cache._shared_objects = {}
        yield super(BaseTestCase, self).setUp()
        self.patch(backend, 'ControlBackend', FakedControlPanelBackend)

        self.ui = None
        if self.class_ui is not None:
            # self.class_ui is not callable
            # pylint: disable=E1102
            self.ui = self.class_ui(**self.kwargs)
            # pylint: enable=E1102
            self.ui.show()
            self.addCleanup(self.ui.hide)
            self.addCleanup(self.ui.deleteLater)
            self.addCleanup(QtCore.QCoreApplication.instance().processEvents)

        if getattr(self.ui, 'backend', None) is not None:
            self.addCleanup(self.ui.backend._called.clear)

        logger = self.logger if self.logger is not None else \
                 getattr(self.ui, 'logger', None)
        self.memento = None
        if logger is not None:
            self.memento = MementoHandler()
            self.memento.setLevel(logging.DEBUG)
            logger.addHandler(self.memento)
            self.addCleanup(logger.removeHandler, self.memento)

        FakedDialog.reset()
        self.patch(QtGui, 'QMessageBox', FakedDialog)

    def get_pixmap_data(self, pixmap):
        """Get the raw data of a QPixmap."""
        byte_array = QtCore.QByteArray()
        array_buffer = QtCore.QBuffer(byte_array)
        pixmap.save(array_buffer, "PNG")
        return byte_array

    # pylint: disable=C0103
    def assertEqualPixmaps(self, pixmap1, pixmap2):
        """Compare two Qt pixmaps."""
        d1 = self.get_pixmap_data(pixmap1)
        d2 = self.get_pixmap_data(pixmap2)
        self.assertEqual(d1, d2)
    # pylint: enable=C0103

    def assert_backend_called(self, method_name, *args, **kwargs):
        """Check that the control panel backend 'method_name' was called."""
        self.assertIn(method_name, self.ui.backend._called)
        self.assertEqual(self.ui.backend._called[method_name], (args, kwargs))

    def assert_uri_hook_called(self, button, url):
        """Check that uri_hook was called with 'url' when clicking 'button'."""
        self.patch(self.ui.backend, 'next_result', url)
        self.patch(qt, 'uri_hook', self._set_called)
        button.click()

        self.assertEqual(len(self._called), 2, 'uri_hook must be called.')
        self.assertEqual(len(self._called[0]), 1, 'uri_hook must be called.')
        actual_url = self._called[0][0]
        self.assertEqual(actual_url, url)

    def test_init_loads_ui(self, expected_setup_ui=None):
        """The __init__ method loads the ui."""
        if self.innerclass_ui is None:
            return
        self.patch(self.innerclass_ui, self.innerclass_name, FakeUi)
        if getattr(self.class_ui, '_setup', None) is not None:
            self.patch(self.class_ui, '_setup', lambda *a: None)
        # pylint: disable=E1102
        self.ui = self.class_ui(**self.kwargs)
        # pylint: disable=E1101

        if expected_setup_ui is None:
            expected_setup_ui = self.ui
        else:
            expected_setup_ui = expected_setup_ui(self.ui)
        self.assertEqual(self.ui.ui._called,
                         {'setupUi': ((expected_setup_ui,), {})})

    def test_backend_is_correct(self):
        """The backend instance is correct."""
        if getattr(self.ui, 'backend', None) is not None:
            self.assertIsInstance(self.ui.backend, FakedControlPanelBackend)


class FakeDesktopService(object):
    """Fake QDesktopService."""

    def __init__(self):
        self.opened_url = None

    # pylint: disable=C0103

    def openUrl(self, url):
        """Fake openUrl."""
        self.opened_url = url
