# -*- coding: utf-8 -*-

# Authors: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

from __future__ import division
import sys

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel import cache
from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.gui import FOLDER_INVALID_PATH
from ubuntuone.controlpanel.gui.qt import handle_errors


logger = setup_logging('qt.addfolder')

CLOSE = QtGui.QMessageBox.Close
# NOTE: this is temporary because of Qt bugs.
# You cannot, in general, use sys.platform in these
# modules. Do not do this. Please. This is an exception, one
# time only, pinky-swear, cross my heart and hope to die if
# I lie.
FILE_CHOOSER_OPTIONS = QtGui.QFileDialog.ShowDirsOnly
if sys.platform in ('win32'):
    FILE_CHOOSER_OPTIONS |= QtGui.QFileDialog.DontUseNativeDialog
# Yes, that is true. If you do that again, you will be hunted
# down and taught a lesson. You will be sorry.

if sys.platform == 'darwin':
    from ubuntuone.controlpanel.gui.qt.main.darwin import (
        DarwinFileDialog)
    FileDialog = DarwinFileDialog
else:
    FileDialog = QtGui.QFileDialog


class AddFolderButton(cache.Cache, QtGui.QPushButton):
    """The AddFolderButton widget"""

    logger = logger

    folderCreated = QtCore.pyqtSignal(unicode)
    folderCreationCanceled = QtCore.pyqtSignal()

    def __init__(self, *args, **kwargs):
        """Initialize the UI of the widget."""
        super(AddFolderButton, self).__init__(*args, **kwargs)
        self.add_folder_func = \
            lambda *a, **kw: defer.fail(NotImplementedError())
        self.cloud_folders = []
        self.clicked.connect(self.on_clicked)

    @QtCore.pyqtSlot()
    @handle_errors(logger=logger)
    @defer.inlineCallbacks
    def on_clicked(self):
        """The 'Sync another folder' button was clicked."""
        # The options argument is because of LP: #835013
        home_dir = yield self.backend.get_home_dir()
        folder = FileDialog.getExistingDirectory(parent=self,
                                                 directory=home_dir,
                                                 options=FILE_CHOOSER_OPTIONS)
        self.window().raise_()
        folder = unicode(QtCore.QDir.toNativeSeparators(folder))
        logger.info('on_add_folder_button_clicked: user requested folder '
                    'creation for path %r.', folder)
        if folder == '':
            self.folderCreationCanceled.emit()
            defer.returnValue(None)

        is_valid = yield self.backend.validate_path_for_folder(folder)
        if not is_valid:
            logger.error('on_add_folder_button_clicked: user requested to '
                         'create a folder for an invalid path %r.', folder)
            text = FOLDER_INVALID_PATH % {'folder_path': folder,
                                          'home_folder': home_dir}
            QtGui.QMessageBox.warning(self, '', text, CLOSE)
            defer.returnValue(None)

        yield self.add_folder_func(folder_path=folder)
        self.folderCreated.emit(folder)
