# -*- coding: utf-8 -*-

# Authors: Natalia B. Bidart <nataliabidart@canonical.com>
# Authors: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Integration tests for the Ubuntu One Control Panel."""

import dbus

from twisted.internet import defer

from ubuntuone.devtools.testcases.dbus import DBusTestCase as TestCase

from ubuntuone.controlpanel import dbus_service


class TestDBusException(Exception):
    """A DBus exception to be used in tests."""


class MockDBusNoMethods(dbus.service.Object):
    """A mock that fails at the DBus layer (because it's got no methods!)."""


class DBusClientTestCase(TestCase):
    """Test for a dbus client."""

    timeout = 5

    @defer.inlineCallbacks
    def setUp(self):
        yield super(DBusClientTestCase, self).setUp()
        self.mock = None
        self._called = False
        dbus_service.init_mainloop()

    def _set_called(self, *args, **kwargs):
        """Keep track of function calls, useful for monkeypatching."""
        self._called = (args, kwargs)

    def register_mockserver(self, bus_name, object_path, object_class,
                            **kwargs):
        """The mock service is registered on the DBus."""
        session_bus = dbus.SessionBus()
        name = session_bus.request_name(bus_name,
                                        dbus.bus.NAME_FLAG_DO_NOT_QUEUE)
        self.assertNotEqual(name, dbus.bus.REQUEST_NAME_REPLY_EXISTS)
        self.mock = object_class(object_path=object_path, conn=session_bus,
                                 **kwargs)
        self.addCleanup(self.mock.remove_from_connection)
