/*
# ***** BEGIN LICENSE BLOCK *****
# Version: MPL 1.1/GPL 2.0/LGPL 2.1
#
# The contents of this file are subject to the Trisquel Public License Version
# 1.1 (the "License"); you may not use this file except in compliance with
# the License. You may obtain a copy of the License at
# http://www.mozilla.org/MPL/
#
# Software distributed under the License is distributed on an "AS IS" basis,
# WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
# for the specific language governing rights and limitations under the
# License.
#
# The Original Code is the Extension Manager.
#
# The Initial Developer of the Original Code is
# the Trisquel Foundation.
# Portions created by the Initial Developer are Copyright (C) 2009
# the Initial Developer. All Rights Reserved.
#
# Contributor(s):
#   Dave Townsend <dtownsend@oxymoronical.com>
#   Chris Coulson <chris.coulson@canonical.com>
#
# Alternatively, the contents of this file may be used under the terms of
# either the GNU General Public License Version 2 or later (the "GPL"), or
# the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
# in which case the provisions of the GPL or the LGPL are applicable instead
# of those above. If you wish to allow use of your version of this file only
# under the terms of either the GPL or the LGPL, and not to allow others to
# use your version of this file under the terms of the MPL, indicate your
# decision by deleting the provisions above and replace them with the notice
# and other provisions required by the GPL or the LGPL. If you do not delete
# the provisions above, a recipient may use your version of this file under
# the terms of any one of the MPL, the GPL or the LGPL.
#
# ***** END LICENSE BLOCK *****
*/

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cu = Components.utils;
const Cr = Components.results;

const nsILocalFile            = Ci.nsILocalFile;
const nsIFile                 = Ci.nsIFile;
const nsIRDFService           = Ci.nsIRDFService;
const nsIRDFLiteral           = Ci.nsIRDFLiteral;
const nsIRDFResource          = Ci.nsIRDFResource;
const nsIRDFInt               = Ci.nsIRDFInt;
const nsIRDFXMLParser         = Ci.nsIRDFXMLParser;
const nsIRDFDataSource        = Ci.nsIRDFDataSource;
const nsIInputStreamChannel   = Ci.nsIInputStreamChannel;
const nsIChannel              = Ci.nsIChannel;
const nsIFileInputStream      = Ci.nsIFileInputStream;
const nsIBufferedInputStream  = Ci.nsIBufferedInputStream;
const nsIZipReader            = Ci.nsIZipReader;

Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/AddonManager.jsm");
Cu.import("resource://gre/modules/NetUtil.jsm");

const PREF_AI_INSTALL_CACHE         = "extensions.ubufox@ubuntu.com.installCache";
const PREF_AI_INSTALL_DIRS          = "extensions.ubufox@ubuntu.com.installDirs";

const FILE_INSTALL_MANIFEST         = "install.rdf";

const PREFIX_NS_EM                  = "http://www.mozilla.org/2004/em-rdf#";
const RDFURI_INSTALL_MANIFEST_ROOT  = "urn:mozilla:install-manifest";

const REGEXP_XPI_FILE = /^.+\.xpi$/;
const REGEXP_VALID_ID = /^(\{[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12}\}|[a-z0-9-\._]*\@[a-z0-9-\._]+)$/;

const PROP_METADATA = [ "id", "version" ];

var EXPORTED_SYMBOLS = [];

/**
 * A helpful wrapper around the prefs service that allows for default values
 * when requested values aren't set.
 */
var Prefs = {
  getCharPref: function(aName, aDefaultValue) {
    try {
      return Services.prefs.getCharPref(aName);
    }
    catch (e) {
    }
    return aDefaultValue;
  },

  getBoolPref: function(aName, aDefaultValue) {
    try {
      return Services.prefs.getBoolPref(aName);
    }
    catch (e) {
    }
    return aDefaultValue;
  }
}

function AddonInternal() {
}

this.__defineGetter__("gRDF", function() {
  delete this.gRDF;
  return this.gRDF = Cc["@mozilla.org/rdf/rdf-service;1"].
                     getService(nsIRDFService);
});

function EM_R(aProperty) {
  return gRDF.GetResource(PREFIX_NS_EM + aProperty);
}

function getRDFValue(aLiteral) {
  if (aLiteral instanceof nsIRDFLiteral)
    return aLiteral.Value;
  if (aLiteral instanceof nsIRDFResource)
    return aLiteral.Value;
  if (aLiteral instanceof nsIRDFInt)
    return aLiteral.Value;
  return null;
}

function getRDFProperty(aDs, aResource, aProperty) {
  return getRDFValue(aDs.GetTarget(aResource, EM_R(aProperty), true));
}

function loadManifestFromRDF(aUri, aStream) {
  let rdfParser = Cc["@mozilla.org/rdf/xml-parser;1"].
                  createInstance(nsIRDFXMLParser)
  let ds = Cc["@mozilla.org/rdf/datasource;1?name=in-memory-datasource"].
           createInstance(nsIRDFDataSource);
  let listener = rdfParser.parseAsync(ds, aUri);
  let channel = Cc["@mozilla.org/network/input-stream-channel;1"].
                createInstance(nsIInputStreamChannel);
  channel.setURI(aUri);
  channel.contentStream = aStream;
  channel.QueryInterface(nsIChannel);
  channel.contentType = "text/xml";

  listener.onStartRequest(channel, null);

  try {
    let pos = 0;
    let count = aStream.available();
    while (count > 0) {
      listener.onDataAvailable(channel, null, aStream, pos, count);
      pos += count;
      count = aStream.available();
    }
    listener.onStopRequest(channel, null, Cr.NS_OK);
  }
  catch (e) {
    listener.onStopRequest(channel, null, e.result);
    throw e;
  }

  let root = gRDF.GetResource(RDFURI_INSTALL_MANIFEST_ROOT);
  let addon = new AddonInternal();
  PROP_METADATA.forEach(function(aProp) {
    addon[aProp] = getRDFProperty(ds, root, aProp);
  });

  return addon;
}

function loadManifestFromDir(aDir) {
  let file = aDir.clone();
  file.append(FILE_INSTALL_MANIFEST);
  if (!file.exists() || !file.isFile())
    throw new Error("Directory " + aDir.path + " does not contain a valid " +
                    "install manifest");

  let fis = Cc["@mozilla.org/network/file-input-stream;1"].
            createInstance(nsIFileInputStream);
  fis.init(file, -1, -1, false);
  let bis = Cc["@mozilla.org/network/buffered-input-stream;1"].
            createInstance(nsIBufferedInputStream);
  bis.init(fis, 4096);

  try {
    let addon = loadManifestFromRDF(Services.io.newFileURI(file), bis);
    return addon;
  }
  finally {
    bis.close();
    fis.close();
  }
}

function loadManifestFromZipReader(aZipReader) {
  let zis = aZipReader.getInputStream(FILE_INSTALL_MANIFEST);
  let bis = Cc["@mozilla.org/network/buffered-input-stream;1"].
            createInstance(nsIBufferedInputStream);
  bis.init(zis, 4096);

  try {
    let uri = buildJarURI(aZipReader.file, FILE_INSTALL_MANIFEST);
    let addon = loadManifestFromRDF(uri, bis);

    return addon;
  }
  finally {
    bis.close();
    zis.close();
  }
}

function loadManifestFromZipFile(aXPIFile) {
  let zipReader = Cc["@mozilla.org/libjar/zip-reader;1"].
                  createInstance(nsIZipReader);
  try {
    zipReader.open(aXPIFile);

    return loadManifestFromZipReader(zipReader);
  }
  finally {
    zipReader.close();
  }
}

function loadManifestFromFile(aFile) {
  if (aFile.isFile())
    return loadManifestFromZipFile(aFile);
  else
    return loadManifestFromDir(aFile);
}

function buildJarURI(aJarfile, aPath) {
  let uri = Services.io.newFileURI(aJarfile);
  uri = "jar:" + uri.spec + "!/" + aPath;
  return NetUtil.newURI(uri);
}

// Borrowed from nsBrowserGlue.js
function getMostRecentBrowserWindow() {
  function isFullBrowserWindow(win) {
    return !win.closed &&
           !win.document.documentElement.getAttribute("chromehidden");
  }

  var win = Services.wm.getMostRecentWindow("navigator:browser");

  // if we're lucky, this isn't a popup, and we can just return this
  if (win && !isFullBrowserWindow(win)) {
    win = null;
    let windowList = Services.wm.getEnumerator("navigator:browser");
    // this is oldest to newest, so this gets a bit ugly
    while (windowList.hasMoreElements()) {
      let nextWin = windowList.getNext();
      if (isFullBrowserWindow(nextWin))
        win = nextWin;
    }
  }
  return win;
}

function DirectoryEntry(file, dirCache) {
  this.file = file;
  this.dirCache = dirCache;
}

var uAddonInstaller = {
  installCache: null,

  startup: function() {
    try {
      this.installCache = JSON.parse(Prefs.getCharPref(PREF_AI_INSTALL_CACHE,
                                                       null));
    } catch(e) {
      Cu.reportError("installCache failed to load: " + e);
      return;
    }

    if (this.installCache == null)
      this.installCache = [];

    try {
      let dirs = Prefs.getCharPref(PREF_AI_INSTALL_DIRS,
                                   "/usr/share/ubufox/extensions").split(",");
      dirs.forEach(function(aDir) {
        try {
          let dirFile = Cc["@mozilla.org/file/local;1"].createInstance(nsILocalFile);
          dirFile.initWithPath(aDir);

          var cache = null;
          uAddonInstaller.installCache.forEach(function(aCache) {
            try {
              if (aCache.location == aDir)
                cache = aCache;
            } catch(e) {
              Cu.reportError("Entry in installCache missing location");
              uAddonInstaller.installCache.splice(uAddonInstaller.installCache.indexOf(aCache), 1);
            }
          });

          if (cache && !("mtime" in cache && "addons" in cache)) {
            Cu.reportError("Entry in installCache for " + aDir + " is missing mtime or addons array");
            uAddonInstaller.installCache.splice(uAddonInstaller.installCache.indexOf(cache), 1);
            cache = null;
          }

          if (dirFile.exists() && dirFile.isDirectory() &&
              (!cache || dirFile.lastModifiedTime != cache.mtime)) {
            if (!cache) {
              cache = JSON.parse("{\"location\": \"" + aDir + "\", \"addons\": []}");
              uAddonInstaller.installCache.push(cache);
            }
            cache.mtime = dirFile.lastModifiedTime;
            uAddonInstaller.processDir(dirFile, cache);
          }
        } catch(e) {
          Cu.reportError("Failed to process directory " + aDir);
        }
      });
    } catch(e) {
      Cu.reportError(e);
      return;
    }

    Services.prefs.setCharPref(PREF_AI_INSTALL_CACHE,
                               JSON.stringify(this.installCache));
  },

  processDir: function(aDir, aCache) {
    let entries = aDir.directoryEntries;
    while (entries.hasMoreElements())
      this.processFile(entries.getNext().QueryInterface(nsIFile), aCache);
  },

  processFile: function(aFile, aCache) {
    var id = aFile.leafName;

    if (aFile.isFile()) {
      if (REGEXP_XPI_FILE.test(id))
        id = id.substring(0, (id.length - 4));
      else
        return;
    } else if (!aFile.isDirectory()) {
      return;
    }

    if (!REGEXP_VALID_ID.test(id))
      return;

    var cachedAddon = null;
    var seenBefore = false;
    aCache.addons.forEach(function(aAddon) {
      try {
        if (aAddon.id == id)
          cachedAddon = aAddon;
      } catch(e) {
        Cu.reportError("Addon in cache without ID");
        aCache.addons.splice(aCache.addons.indexOf(aAddon), 1);
      }
    });

    if (cachedAddon)
      seenBefore = true;

    if (cachedAddon && !("mtime" in cachedAddon)) {
      Cu.reportError("Addon entry for " + id + " in installCache is missing mtime");
      aCache.addons.splice(aCache.addons.indexOf(cachedAddon), 1);
      cachedAddon = null;
    }

    if (cachedAddon && aFile.lastModifiedTime == cachedAddon.mtime)
      return;

    if (!cachedAddon) {
      cachedAddon = JSON.parse("{\"id\": \"" + id + "\"}");
      aCache.addons.push(cachedAddon);
    }

    cachedAddon.mtime = aFile.lastModifiedTime;

    var addon = null;
    try {
      addon = loadManifestFromFile(aFile);
    } catch(e) {
      Cu.reportError("Failed to load extension manifest: " + e);
      return;
    }

    if (addon.id != id)
      return;

    AddonManager.getAddonByID(id, function(aAddon) {
      let shouldInstall = false;
      if (aAddon) {
        // Install the addon to the profile if we already have the same
        // version of the addon or older installed outside of the profile,
        // and we have never processed this addon before.
        if (Services.vc.compare(aAddon.version, addon.version) <= 0 &&
            aAddon.scope != AddonManager.SCOPE_PROFILE && !seenBefore) {
          shouldInstall = true;
        }

        // Install the addon to the profile if it already has an older
        // version installed.
        if (Services.vc.compare(aAddon.version, addon.version) < 0 &
            aAddon.scope == AddonManager.SCOPE_PROFILE) {
          shouldInstall = true;
        }
      } else if (!seenBefore) {
        // Install this addon if we have never processed it before and
        // there isn't any other version installed
        shouldInstall = true;
      }

      if (!shouldInstall)
        return;

      let changedIDs = AddonManager.getStartupChanges(AddonManager.STARTUP_CHANGE_INSTALLED);
      var newAddonTabShown = changedIDs.indexOf(aAddon.id) != 1
                             && aAddon.userDisabled
                             && (aAddon.permissions & AddonManager.PERM_CAN_ENABLE);

      AddonManager.getInstallForFile(aFile, function(aInstall) {
        if (aInstall) {
          aInstall.addListener({
            onInstallEnded: function(aInstall, aAddon) {

              function maybeShowNewAddonTab(aAddon) {
                // Maybe show the new addon tab. This is shown if the addon
                // is disabled but could be enabled, and we have never seen it
                // before. We don't show the tab if it is in the list of startup
                // changes, as nsBrowserGlue will have already shown it
                if (aAddon.userDisabled && !seenBefore &&
                    (aAddon.permissions & AddonManager.PERM_CAN_ENABLE) &&
                    !newAddonTabShown) {
                  let browser = getMostRecentBrowserWindow().gBrowser;
                  browser.selectedTab = browser.addTab("about:newaddon?id=" + aAddon.id);
                }
              }

              // Check if the addon has updates
              aAddon.findUpdates({
                onUpdateAvailable: function(aAddon, aInstall) {
                  if (aInstall) {
                    aInstall.addListener({
                      onInstallEnded: function(aInstall, aAddon) {
                        maybeShowNewAddonTab(aAddon);
                      }
                    });

                    aInstall.install();
                  }
                },

                onNoUpdateAvailable: function(aAddon) {
                  maybeShowNewAddonTab(aAddon);
                }
              }, AddonManager.UPDATE_WHEN_ADDON_INSTALLED);
            }
          });

          aInstall.install();
        }
      });
    });
  },
};

uAddonInstaller.startup();
